<?php

namespace Modules\BusinessManagement\Tests;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use App\User;
use App\Business;
use App\BusinessLocation;
use App\Transaction;
use App\PurchaseLine;
use App\Product;
use App\Variation;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Artisan;

/**
 * Performance Test Suite
 * 
 * Tests query performance with large datasets
 */
class PerformanceTest extends TestCase
{
    use RefreshDatabase;

    protected $business;
    protected $locations = [];
    protected $user;

    protected function setUp(): void
    {
        parent::setUp();

        $this->business = Business::factory()->create();
        
        // Create 5 locations
        for ($i = 1; $i <= 5; $i++) {
            $this->locations[] = BusinessLocation::factory()->create([
                'business_id' => $this->business->id,
                'name' => "Location {$i}"
            ]);
        }

        $this->user = User::factory()->create([
            'business_id' => $this->business->id,
        ]);
        $this->user->permitted_locations = 'all';
        $this->user->save();
    }

    /**
     * Test Purchase Register performance with large dataset
     */
    public function test_purchase_register_performance_with_large_dataset()
    {
        $this->actingAs($this->user);

        // Create 1000 purchases across locations
        $startTime = microtime(true);
        
        Transaction::factory()->count(1000)->create([
            'business_id' => $this->business->id,
            'type' => 'purchase',
        ])->each(function ($transaction) {
            $transaction->location_id = $this->locations[array_rand($this->locations)]->id;
            $transaction->save();
        });

        $creationTime = microtime(true) - $startTime;
        $this->assertLessThan(30, $creationTime, 'Transaction creation should be fast');

        // Test query performance
        $startTime = microtime(true);
        
        $response = $this->getJson(route('businessmanagement.purchase_register.index', [
            'ajax' => true
        ]));

        $queryTime = microtime(true) - $startTime;
        
        $response->assertStatus(200);
        $this->assertLessThan(2, $queryTime, 'Query should complete in under 2 seconds');
    }

    /**
     * Test that indexes are being used
     */
    public function test_indexes_are_being_used()
    {
        // Run migration to create indexes
        Artisan::call('migrate', [
            '--path' => 'Modules/BusinessManagement/Database/Migrations/2024_01_03_000001_add_business_management_indexes.php'
        ]);

        // Create test data
        Transaction::factory()->count(100)->create([
            'business_id' => $this->business->id,
            'type' => 'purchase',
        ]);

        // Enable query log
        DB::enableQueryLog();

        // Execute query
        Transaction::where('business_id', $this->business->id)
            ->whereIn('type', ['purchase', 'opening_stock'])
            ->where('location_id', $this->locations[0]->id)
            ->get();

        $queries = DB::getQueryLog();
        
        // Check that query was executed (indexes should make it fast)
        $this->assertNotEmpty($queries);
        
        // The query should use indexes (check EXPLAIN in actual implementation)
    }

    /**
     * Test date range query performance
     */
    public function test_date_range_query_performance()
    {
        $this->actingAs($this->user);

        // Create purchases over 1 year period
        $startDate = now()->subYear();
        $endDate = now();

        Transaction::factory()->count(500)->create([
            'business_id' => $this->business->id,
            'type' => 'purchase',
        ])->each(function ($transaction, $index) use ($startDate, $endDate) {
            $transaction->location_id = $this->locations[array_rand($this->locations)]->id;
            $transaction->transaction_date = $startDate->copy()->addDays($index % 365);
            $transaction->save();
        });

        $startTime = microtime(true);

        $response = $this->getJson(route('businessmanagement.purchase_register.index', [
            'ajax' => true,
            'start_date' => $startDate->format('Y-m-d'),
            'end_date' => $endDate->format('Y-m-d'),
        ]));

        $queryTime = microtime(true) - $startTime;

        $response->assertStatus(200);
        $this->assertLessThan(2, $queryTime, 'Date range query should be fast');
    }

    /**
     * Test multi-location query performance
     */
    public function test_multi_location_query_performance()
    {
        $this->actingAs($this->user);

        // Create data across all locations
        foreach ($this->locations as $location) {
            Transaction::factory()->count(200)->create([
                'business_id' => $this->business->id,
                'location_id' => $location->id,
                'type' => 'purchase',
            ]);
        }

        $startTime = microtime(true);

        $response = $this->getJson(route('businessmanagement.purchase_register.index', [
            'ajax' => true
        ]));

        $queryTime = microtime(true) - $startTime;

        $response->assertStatus(200);
        $this->assertLessThan(3, $queryTime, 'Multi-location query should be fast');
    }
}

