<?php

namespace Modules\BusinessManagement\Tests;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use App\User;
use App\Business;
use App\BusinessLocation;
use App\Role;
use App\Transaction;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role as SpatieRole;

/**
 * Multi-Location Security Test Suite
 * 
 * Tests location access validation across all BusinessManagement controllers
 */
class MultiLocationSecurityTest extends TestCase
{
    use RefreshDatabase;

    protected $business;
    protected $locationA;
    protected $locationB;
    protected $locationC;
    protected $adminUser;
    protected $restrictedUser;
    protected $noAccessUser;

    protected function setUp(): void
    {
        parent::setUp();

        // Create business
        $this->business = Business::factory()->create();

        // Create locations
        $this->locationA = BusinessLocation::factory()->create([
            'business_id' => $this->business->id,
            'name' => 'Location A'
        ]);

        $this->locationB = BusinessLocation::factory()->create([
            'business_id' => $this->business->id,
            'name' => 'Location B'
        ]);

        $this->locationC = BusinessLocation::factory()->create([
            'business_id' => $this->business->id,
            'name' => 'Location C'
        ]);

        // Create admin user (access to all locations)
        $this->adminUser = User::factory()->create([
            'business_id' => $this->business->id,
        ]);
        $this->adminUser->permitted_locations = 'all';
        $this->adminUser->save();

        // Create restricted user (access to Location A only)
        $this->restrictedUser = User::factory()->create([
            'business_id' => $this->business->id,
        ]);
        $this->restrictedUser->permitted_locations = [$this->locationA->id];
        $this->restrictedUser->save();

        // Create user with no location access
        $this->noAccessUser = User::factory()->create([
            'business_id' => $this->business->id,
        ]);
        $this->noAccessUser->permitted_locations = [];
        $this->noAccessUser->save();

        // Assign permissions
        $this->assignPermissions();
    }

    protected function assignPermissions()
    {
        // Ensure permissions exist
        $permissions = [
            'businessmanagement.view',
            'businessmanagement.purchase_register',
            'businessmanagement.sales_register',
            'businessmanagement.stock_register',
            'businessmanagement.accounts_register',
        ];

        foreach ($permissions as $permission) {
            Permission::firstOrCreate(['name' => $permission, 'guard_name' => 'web']);
        }

        // Assign permissions to users
        $this->adminUser->givePermissionTo($permissions);
        $this->restrictedUser->givePermissionTo($permissions);
        // noAccessUser gets no permissions
    }

    /**
     * Test Purchase Register - Restricted user cannot access Location B
     */
    public function test_purchase_register_restricted_user_cannot_access_unauthorized_location()
    {
        $this->actingAs($this->restrictedUser);

        // Create purchase in Location B
        $purchase = Transaction::factory()->create([
            'business_id' => $this->business->id,
            'location_id' => $this->locationB->id,
            'type' => 'purchase',
        ]);

        // Try to access purchase register with Location B
        $response = $this->get(route('businessmanagement.purchase_register.index', [
            'location_id' => $this->locationB->id
        ]));

        // Should get 403 Forbidden
        $response->assertStatus(403);
    }

    /**
     * Test Purchase Register - Restricted user CAN access Location A
     */
    public function test_purchase_register_restricted_user_can_access_authorized_location()
    {
        $this->actingAs($this->restrictedUser);

        // Create purchase in Location A
        $purchase = Transaction::factory()->create([
            'business_id' => $this->business->id,
            'location_id' => $this->locationA->id,
            'type' => 'purchase',
        ]);

        // Access purchase register with Location A
        $response = $this->get(route('businessmanagement.purchase_register.index', [
            'location_id' => $this->locationA->id
        ]));

        // Should be successful
        $response->assertStatus(200);
    }

    /**
     * Test Purchase Register - Admin user can access all locations
     */
    public function test_purchase_register_admin_can_access_all_locations()
    {
        $this->actingAs($this->adminUser);

        // Create purchases in different locations
        Transaction::factory()->create([
            'business_id' => $this->business->id,
            'location_id' => $this->locationA->id,
            'type' => 'purchase',
        ]);

        Transaction::factory()->create([
            'business_id' => $this->business->id,
            'location_id' => $this->locationB->id,
            'type' => 'purchase',
        ]);

        // Access with Location A
        $response = $this->get(route('businessmanagement.purchase_register.index', [
            'location_id' => $this->locationA->id
        ]));
        $response->assertStatus(200);

        // Access with Location B
        $response = $this->get(route('businessmanagement.purchase_register.index', [
            'location_id' => $this->locationB->id
        ]));
        $response->assertStatus(200);
    }

    /**
     * Test Sales Register - Restricted user cannot access unauthorized location
     */
    public function test_sales_register_restricted_user_cannot_access_unauthorized_location()
    {
        $this->actingAs($this->restrictedUser);

        $response = $this->get(route('businessmanagement.sales_register.index', [
            'location_id' => $this->locationB->id
        ]));

        $response->assertStatus(403);
    }

    /**
     * Test Stock Register - Restricted user cannot access unauthorized location
     */
    public function test_stock_register_restricted_user_cannot_access_unauthorized_location()
    {
        $this->actingAs($this->restrictedUser);

        $response = $this->get(route('businessmanagement.stock_register.index', [
            'location_id' => $this->locationB->id
        ]));

        $response->assertStatus(403);
    }

    /**
     * Test Accounts Register - Restricted user cannot access unauthorized location
     */
    public function test_accounts_register_restricted_user_cannot_access_unauthorized_location()
    {
        $this->actingAs($this->restrictedUser);

        $response = $this->get(route('businessmanagement.accounts_register.index', [
            'location_id' => $this->locationB->id
        ]));

        $response->assertStatus(403);
    }

    /**
     * Test Date Wise Purchase Report - Location validation
     */
    public function test_date_wise_purchase_location_validation()
    {
        $this->actingAs($this->restrictedUser);

        $response = $this->get(route('businessmanagement.purchase_register.date_wise', [
            'location_id' => $this->locationB->id
        ]));

        $response->assertStatus(403);
    }

    /**
     * Test Supplier Wise Purchase Report - Location validation
     */
    public function test_supplier_wise_purchase_location_validation()
    {
        $this->actingAs($this->restrictedUser);

        $response = $this->get(route('businessmanagement.purchase_register.supplier_wise', [
            'location_id' => $this->locationB->id
        ]));

        $response->assertStatus(403);
    }

    /**
     * Test Print Methods - Location validation
     */
    public function test_print_methods_location_validation()
    {
        $this->actingAs($this->restrictedUser);

        // Test print date wise purchase
        $response = $this->get(route('businessmanagement.purchase_register.date_wise.print', [
            'location_id' => $this->locationB->id
        ]));

        $response->assertStatus(403);
    }

    /**
     * Test User without permissions cannot access module
     */
    public function test_user_without_permissions_cannot_access()
    {
        $this->actingAs($this->noAccessUser);

        $response = $this->get(route('businessmanagement.purchase_register.index'));

        $response->assertStatus(403);
    }

    /**
     * Test AJAX requests respect location validation
     */
    public function test_ajax_requests_respect_location_validation()
    {
        $this->actingAs($this->restrictedUser);

        $response = $this->getJson(route('businessmanagement.purchase_register.index', [
            'location_id' => $this->locationB->id,
            'ajax' => true
        ]));

        $response->assertStatus(403);
    }

    /**
     * Test that permitted locations filter works correctly
     */
    public function test_permitted_locations_filter_works()
    {
        $this->actingAs($this->restrictedUser);

        // Create purchases in different locations
        Transaction::factory()->create([
            'business_id' => $this->business->id,
            'location_id' => $this->locationA->id,
            'type' => 'purchase',
        ]);

        Transaction::factory()->create([
            'business_id' => $this->business->id,
            'location_id' => $this->locationB->id,
            'type' => 'purchase',
        ]);

        // Access without location_id - should only see Location A
        $response = $this->getJson(route('businessmanagement.purchase_register.index', [
            'ajax' => true
        ]));

        $response->assertStatus(200);
        $data = $response->json();
        
        // Should only contain Location A purchases
        $this->assertCount(1, $data['data']);
        $this->assertEquals($this->locationA->id, $data['data'][0]['location_id']);
    }
}

