# BusinessManagement Module - Testing Guide

**Version:** 1.0.1  
**Last Updated:** 2024

---

## Overview

This guide provides step-by-step instructions for testing the BusinessManagement module, focusing on multi-location security, performance, and functionality.

---

## Prerequisites

1. **Database Setup**
   - MySQL/PostgreSQL database
   - Test database configured
   - Migrations can be run

2. **Test Data**
   - At least 2 business locations
   - Multiple users with different permission levels
   - Sample transactions (purchases, sales, stock)

3. **Environment**
   - Laravel test environment configured
   - PHPUnit installed
   - Test database seeded

---

## Step 1: Run Database Migrations

### 1.1 Run Module Migrations

```bash
php artisan module:migrate BusinessManagement
```

This will:
- Create permission migrations
- Create index migrations
- Set up all required database structures

### 1.2 Verify Migrations

```bash
php artisan migrate:status
```

Check that all BusinessManagement migrations show as "Ran".

### 1.3 Verify Indexes Created

```sql
-- MySQL
SHOW INDEXES FROM transactions WHERE Key_name LIKE 'idx_%';

-- PostgreSQL
SELECT indexname FROM pg_indexes WHERE tablename = 'transactions' AND indexname LIKE 'idx_%';
```

---

## Step 2: Set Up Test Data

### 2.1 Create Test Locations

```php
// In tinker or seeder
$business = Business::find(1);

$locationA = BusinessLocation::create([
    'business_id' => $business->id,
    'name' => 'Location A - Test',
    'landmark' => 'Test Location A',
    'country' => 'US',
    'state' => 'CA',
    'city' => 'Test City',
    'zip_code' => '12345',
]);

$locationB = BusinessLocation::create([
    'business_id' => $business->id,
    'name' => 'Location B - Test',
    'landmark' => 'Test Location B',
    'country' => 'US',
    'state' => 'CA',
    'city' => 'Test City',
    'zip_code' => '12346',
]);
```

### 2.2 Create Test Users

```php
// Admin user (all locations)
$admin = User::create([
    'business_id' => $business->id,
    'first_name' => 'Admin',
    'last_name' => 'User',
    'username' => 'admin_test',
    'email' => 'admin@test.com',
    'password' => bcrypt('password'),
]);
$admin->permitted_locations = 'all';
$admin->save();
$admin->givePermissionTo([
    'businessmanagement.view',
    'businessmanagement.purchase_register',
    'businessmanagement.sales_register',
    'businessmanagement.stock_register',
    'businessmanagement.accounts_register',
]);

// Restricted user (Location A only)
$restricted = User::create([
    'business_id' => $business->id,
    'first_name' => 'Restricted',
    'last_name' => 'User',
    'username' => 'restricted_test',
    'email' => 'restricted@test.com',
    'password' => bcrypt('password'),
]);
$restricted->permitted_locations = [$locationA->id];
$restricted->save();
$restricted->givePermissionTo([
    'businessmanagement.view',
    'businessmanagement.purchase_register',
    'businessmanagement.sales_register',
    'businessmanagement.stock_register',
    'businessmanagement.accounts_register',
]);
```

### 2.3 Create Test Transactions

```php
// Create purchases in Location A
Transaction::factory()->count(10)->create([
    'business_id' => $business->id,
    'location_id' => $locationA->id,
    'type' => 'purchase',
    'transaction_date' => now(),
]);

// Create purchases in Location B
Transaction::factory()->count(10)->create([
    'business_id' => $business->id,
    'location_id' => $locationB->id,
    'type' => 'purchase',
    'transaction_date' => now(),
]);
```

---

## Step 3: Security Testing

### 3.1 Test Location Access Validation

#### Test 1: Restricted User Cannot Access Unauthorized Location

1. Login as restricted user (Location A only)
2. Navigate to: `/business-management/purchase-register?location_id={locationB_id}`
3. **Expected:** 403 Forbidden error
4. **Result:** ✅ Should block access

#### Test 2: Restricted User CAN Access Authorized Location

1. Login as restricted user
2. Navigate to: `/business-management/purchase-register?location_id={locationA_id}`
3. **Expected:** 200 OK, shows data
4. **Result:** ✅ Should show Location A data only

#### Test 3: Admin User Can Access All Locations

1. Login as admin user
2. Navigate to: `/business-management/purchase-register?location_id={locationA_id}`
3. **Expected:** 200 OK
4. Navigate to: `/business-management/purchase-register?location_id={locationB_id}`
5. **Expected:** 200 OK
6. **Result:** ✅ Should access both locations

### 3.2 Test All Controllers

Repeat tests for:
- ✅ Purchase Register
- ✅ Sales Register
- ✅ Stock Register
- ✅ Accounts Register

### 3.3 Test All Report Methods

Test location validation on:
- Date-wise reports
- Supplier/Customer-wise reports
- Product-wise reports
- Print methods

---

## Step 4: Performance Testing

### 4.1 Test with Large Dataset

```bash
# Create large dataset (1000+ transactions)
php artisan tinker

Transaction::factory()->count(1000)->create([
    'business_id' => 1,
    'type' => 'purchase',
]);
```

### 4.2 Measure Query Performance

1. Enable query logging:
```php
DB::enableQueryLog();
```

2. Execute report query
3. Check query time:
```php
$queries = DB::getQueryLog();
dd($queries);
```

### 4.3 Performance Benchmarks

**Target Performance:**
- Date-wise report (1 year, 1 location): < 1 second
- Supplier-wise report (all suppliers): < 2 seconds
- Stock register (all products): < 1 second
- Multi-location query (5 locations): < 3 seconds

### 4.4 Verify Index Usage

```sql
-- MySQL
EXPLAIN SELECT * FROM transactions 
WHERE business_id = 1 
AND location_id = 1 
AND type = 'purchase';

-- Check if indexes are used (key column should show index name)
```

---

## Step 5: Functional Testing

### 5.1 Test Purchase Register Reports

1. **Date Wise Purchase**
   - Select date range
   - Filter by location
   - Verify data accuracy

2. **Supplier Wise Purchase**
   - Select supplier
   - Filter by location
   - Verify totals

3. **Product Wise Purchase**
   - Select product
   - Filter by location
   - Verify quantities

### 5.2 Test Print Functionality

1. Generate report
2. Click print button
3. Verify print view renders correctly
4. Verify location validation on print

### 5.3 Test DataTables

1. Verify AJAX loading works
2. Test pagination
3. Test sorting
4. Test filtering

---

## Step 6: Automated Testing

### 6.1 Run Security Tests

```bash
php artisan test --filter MultiLocationSecurityTest
```

**Expected:** All tests pass

### 6.2 Run Performance Tests

```bash
php artisan test --filter PerformanceTest
```

**Expected:** All tests pass within time limits

### 6.3 Run All Module Tests

```bash
php artisan test Modules/BusinessManagement/Tests
```

---

## Step 7: Manual Testing Checklist

### Security Checklist

- [ ] Restricted user cannot access unauthorized location
- [ ] Restricted user can access authorized location
- [ ] Admin user can access all locations
- [ ] User without permissions gets 403
- [ ] Location validation works on all controllers
- [ ] Location validation works on all report methods
- [ ] Location validation works on print methods
- [ ] AJAX requests respect location validation

### Functionality Checklist

- [ ] All purchase register reports work
- [ ] All sales register reports work
- [ ] All stock register reports work
- [ ] All accounts register reports work
- [ ] Print views render correctly
- [ ] DataTables pagination works
- [ ] Date filters work correctly
- [ ] Location filters work correctly
- [ ] Supplier/Customer filters work
- [ ] Product filters work

### Performance Checklist

- [ ] Reports load in < 2 seconds (small dataset)
- [ ] Reports load in < 5 seconds (large dataset)
- [ ] Indexes are being used
- [ ] No N+1 query issues
- [ ] Memory usage is reasonable

---

## Step 8: Test Results Documentation

### 8.1 Document Test Results

Create a test results document:

```markdown
# Test Results - [Date]

## Security Tests
- ✅ Location validation: PASS
- ✅ Permission checks: PASS
- ✅ Business ID filtering: PASS

## Performance Tests
- ✅ Small dataset (< 1000 records): PASS
- ✅ Large dataset (> 1000 records): PASS
- ✅ Multi-location queries: PASS

## Functional Tests
- ✅ Purchase Register: PASS
- ✅ Sales Register: PASS
- ✅ Stock Register: PASS
- ✅ Accounts Register: PASS
```

### 8.2 Document Issues Found

If issues are found:
1. Document the issue
2. Steps to reproduce
3. Expected vs actual behavior
4. Screenshots if applicable
5. Priority level

---

## Troubleshooting

### Issue: Tests Fail with Permission Errors

**Solution:**
```bash
# Ensure permissions are created
php artisan module:migrate BusinessManagement

# Assign permissions to test user
php artisan tinker
$user->givePermissionTo('businessmanagement.view');
```

### Issue: Location Validation Not Working

**Check:**
1. User's `permitted_locations` is set correctly
2. Location IDs match
3. Migration ran successfully
4. Code changes are deployed

### Issue: Performance Tests Too Slow

**Solutions:**
1. Verify indexes are created
2. Check query execution plans
3. Review for N+1 queries
4. Consider adding more indexes

---

## Next Steps

After testing:
1. ✅ Fix any issues found
2. ✅ Document test results
3. ✅ Get approval for deployment
4. ✅ Deploy to staging
5. ✅ Perform staging tests
6. ✅ Deploy to production

---

**Testing Completed By:** _______________  
**Date:** _______________  
**Status:** ✅ PASS / ❌ FAIL

