<?php

namespace Modules\BusinessManagement\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Transaction;
use App\BusinessLocation;
use App\Contact;
use App\Category;
use App\TransactionSellLine;
use App\Product;
use App\Brands;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use App\Utils\Util;
use DB;

class SalesRegisterController extends Controller
{
    /**
     * Display a listing of the sales register.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id);

        if ($request->ajax()) {
            $sales = Transaction::where('business_id', $business_id)
                ->where('type', 'sell')
                ->with(['contact', 'location'])
                ->select('transactions.*');

            // Apply location filter based on user permissions
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sales->whereIn('location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $sales->where('location_id', $request->location_id);
            }

            if (!empty($request->start_date) && !empty($request->end_date)) {
                $sales->whereBetween('transaction_date', [$request->start_date, $request->end_date]);
            }

            // Permission check for view own sales only
            if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
                $sales->where('transactions.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();
            
            return DataTables::of($sales)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('contact_name', function ($row) {
                    return $row->contact->name ?? '';
                })
                ->addColumn('location_name', function ($row) {
                    return $row->location->name ?? '';
                })
                ->editColumn('final_total', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->final_total . '</span>';
                })
                ->addColumn('action', function ($row) {
                    $html = '<div class="btn-group">
                        <button type="button" class="btn btn-info btn-xs view-sale" data-href="' . action([\App\Http\Controllers\SellController::class, 'show'], [$row->id]) . '">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>';
                    return $html;
                })
                ->rawColumns(['action', 'final_total'])
                ->make(true);
        }

        return view('businessmanagement::sales_register.index', compact('locations'));
    }

    /**
     * Display Date Wise Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWiseSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register.date_wise_sales') && 
            !auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            $sales = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
                ->leftJoin('users', 'transactions.created_by', '=', 'users.id')
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell')
                ->select(
                    'transactions.id',
                    'transactions.ref_no',
                    'transactions.invoice_no',
                    'transactions.transaction_date',
                    'transactions.discount_amount',
                    'transactions.discount_type',
                    'transactions.total_before_tax',
                    'transactions.tax_amount',
                    'transactions.final_total',
                    'transactions.status',
                    'transactions.payment_status',
                    'transactions.shipping_charges',
                    'transactions.additional_expense_value_1',
                    'transactions.additional_expense_value_2',
                    'transactions.additional_expense_value_3',
                    'transactions.additional_expense_value_4',
                    'contacts.name as customer_name',
                    'contacts.supplier_business_name',
                    'contacts.contact_id',
                    DB::raw("CONCAT(COALESCE(users.surname, ''), ' ', COALESCE(users.first_name, ''), ' ', COALESCE(users.last_name, '')) as entered_by")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sales->whereIn('transactions.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $sales->where('transactions.location_id', $request->location_id);
            }

            // Apply customer filter
            if (!empty($request->customer_id)) {
                $sales->where('contacts.id', $request->customer_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $sales->whereDate('transactions.transaction_date', '>=', $request->start_date)
                    ->whereDate('transactions.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
                $sales->where('transactions.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();

            return DataTables::of($sales)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('trader_name', function ($row) {
                    $customer = $row->customer_name ?? 'Walking Customer';
                    if (!empty($row->supplier_business_name)) {
                        $customer = $row->supplier_business_name . ' - ' . $customer;
                    }
                    return $customer;
                })
                ->editColumn('ref_no', function ($row) {
                    return $row->ref_no ?? '';
                })
                ->addColumn('bill_type', function ($row) {
                    if ($row->payment_status == 'paid') {
                        return '<span class="label label-success">Cash</span>';
                    } elseif ($row->payment_status == 'due') {
                        return '<span class="label label-warning">Credit</span>';
                    } elseif ($row->payment_status == 'partial') {
                        return '<span class="label label-info">Partial</span>';
                    }
                    return '<span class="label label-default">' . ucfirst($row->payment_status) . '</span>';
                })
                ->editColumn('total_before_tax', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->total_before_tax . '</span>';
                })
                ->editColumn('discount_amount', function ($row) {
                    $discount = $row->discount_amount ?? 0;
                    if (!empty($discount) && $row->discount_type == 'percentage') {
                        $discount = $row->total_before_tax * ($discount / 100);
                    }
                    return '<span class="display_currency" data-currency_symbol="true">' . $discount . '</span>';
                })
                ->addColumn('extra_charges', function ($row) {
                    $extra_charges = $row->shipping_charges ?? 0;
                    $extra_charges += $row->additional_expense_value_1 ?? 0;
                    $extra_charges += $row->additional_expense_value_2 ?? 0;
                    $extra_charges += $row->additional_expense_value_3 ?? 0;
                    $extra_charges += $row->additional_expense_value_4 ?? 0;
                    return '<span class="display_currency" data-currency_symbol="true">' . $extra_charges . '</span>';
                })
                ->editColumn('tax_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . ($row->tax_amount ?? 0) . '</span>';
                })
                ->editColumn('final_total', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->final_total . '</span>';
                })
                ->addColumn('link', function ($row) {
                    return '<a href="' . action([\App\Http\Controllers\SellController::class, 'show'], [$row->id]) . '" target="_blank" class="btn btn-xs btn-info"><i class="fas fa-eye"></i></a>';
                })
                ->rawColumns(['bill_type', 'total_before_tax', 'discount_amount', 'tax_amount', 'extra_charges', 'final_total', 'link'])
                ->make(true);
        }

        return view('businessmanagement::sales_register.date_wise_sales', compact('customers', 'locations'));
    }

    /**
     * Print Date Wise Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWiseSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        $sales = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
            ->leftJoin('users', 'transactions.created_by', '=', 'users.id')
            ->where('transactions.business_id', $business_id)
            ->where('transactions.type', 'sell')
            ->select(
                'transactions.id',
                'transactions.ref_no',
                'transactions.invoice_no',
                'transactions.transaction_date',
                'transactions.discount_amount',
                'transactions.discount_type',
                'transactions.total_before_tax',
                'transactions.tax_amount',
                'transactions.final_total',
                'transactions.payment_status',
                'transactions.shipping_charges',
                'transactions.additional_expense_value_1',
                'transactions.additional_expense_value_2',
                'transactions.additional_expense_value_3',
                'transactions.additional_expense_value_4',
                'contacts.name as customer_name',
                'contacts.supplier_business_name',
                DB::raw("CONCAT(COALESCE(users.surname, ''), ' ', COALESCE(users.first_name, ''), ' ', COALESCE(users.last_name, '')) as entered_by")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $sales->whereIn('transactions.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales->where('transactions.location_id', $request->location_id);
        }

        // Apply customer filter
        $customer_name = null;
        $is_single_customer = false;
        if (!empty($request->customer_id)) {
            $sales->where('contacts.id', $request->customer_id);
            $customer = Contact::find($request->customer_id);
            if ($customer) {
                $customer_name = $customer->name;
                if (!empty($customer->supplier_business_name)) {
                    $customer_name = $customer->supplier_business_name . ' - ' . $customer_name;
                }
                $is_single_customer = true;
            }
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) == 2) {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        if ($start_date && $end_date) {
            $sales->whereDate('transactions.transaction_date', '>=', $start_date)
                ->whereDate('transactions.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $sales->where('transactions.created_by', request()->session()->get('user.id'));
        }

        $sales = $sales->orderBy('transactions.transaction_date', 'asc')
            ->orderBy('transactions.invoice_no', 'asc')
            ->get();

        // Group by date and payment status
        $sales_by_date = [];
        foreach ($sales as $sale) {
            $date = \Carbon\Carbon::parse($sale->transaction_date)->format('d/m/Y');
            $payment_status = $sale->payment_status;
            
            if (!isset($sales_by_date[$date])) {
                $sales_by_date[$date] = [];
            }
            
            $bill_type = 'Cash Bill';
            if ($payment_status == 'due') {
                $bill_type = 'Credit Bill';
            } elseif ($payment_status == 'partial') {
                $bill_type = 'Partial Bill';
            }
            
            if (!isset($sales_by_date[$date][$bill_type])) {
                $sales_by_date[$date][$bill_type] = [];
            }
            
            $sales_by_date[$date][$bill_type][] = $sale;
        }

        // Calculate totals
        $total_discount = 0;
        $total_extra_charges = 0;
        $total_tax = 0;
        $total_amount = 0;

        foreach ($sales as $sale) {
            $discount = $sale->discount_amount ?? 0;
            if (!empty($discount) && $sale->discount_type == 'percentage') {
                $discount = $sale->total_before_tax * ($discount / 100);
            }
            $total_discount += $discount;

            $extra_charges = $sale->shipping_charges ?? 0;
            $extra_charges += $sale->additional_expense_value_1 ?? 0;
            $extra_charges += $sale->additional_expense_value_2 ?? 0;
            $extra_charges += $sale->additional_expense_value_3 ?? 0;
            $extra_charges += $sale->additional_expense_value_4 ?? 0;
            $total_extra_charges += $extra_charges;

            $total_tax += $sale->tax_amount ?? 0;
            $total_amount += $sale->final_total;
        }

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::sales_register.print_date_wise_sales', compact(
            'sales',
            'sales_by_date',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'customer_name',
            'is_single_customer',
            'util',
            'currency',
            'total_discount',
            'total_extra_charges',
            'total_tax',
            'total_amount'
        ));
    }

    /**
     * Display Product Wise Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productWiseSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register.product_wise_sales') && 
            !auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, 'product');
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->pluck('name', 'id');

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get sell lines with product details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.ref_no',
                't.transaction_date',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as quantity'),
                'transaction_sell_lines.unit_price_inc_tax',
                DB::raw('(SELECT pt2.invoice_no FROM transaction_sell_lines_purchase_lines tspl2 
                    INNER JOIN purchase_lines pl2 ON tspl2.purchase_line_id = pl2.id 
                    INNER JOIN transactions pt2 ON pl2.transaction_id = pt2.id 
                    WHERE tspl2.sell_line_id = transaction_sell_lines.id 
                    LIMIT 1) as purchase_invoice'),
                DB::raw("CASE 
                    WHEN p.type = 'variable' AND v.name IS NOT NULL THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_name")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('product_name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Group by product
        $grouped_data = $sell_lines->groupBy('product_name');

        // Calculate product-wise totals
        $products_totals = [];
        foreach ($grouped_data as $product_name => $product_lines) {
            $products_totals[$product_name] = [
                'total_qty' => $product_lines->sum('quantity'),
                'total_amount' => $product_lines->sum(function($line) {
                    return $line->quantity * $line->unit_price_inc_tax;
                }),
            ];
        }

        // Calculate sub totals
        $sub_total_qty = $sell_lines->sum('quantity');
        $sub_total_amount = $sell_lines->sum(function($line) {
            return $line->quantity * $line->unit_price_inc_tax;
        });

        // Calculate grand totals (same as sub totals for this report)
        $grand_total_qty = $sub_total_qty;
        $grand_total_amount = $sub_total_amount;

        $util = new Util();

        return view('businessmanagement::sales_register.product_wise_sales', compact(
            'sell_lines',
            'grouped_data',
            'products_totals',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'sub_total_qty',
            'sub_total_amount',
            'grand_total_qty',
            'grand_total_amount',
            'util'
        ));
    }

    /**
     * Print Product Wise Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductWiseSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get sell lines with product details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.ref_no',
                't.transaction_date',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as quantity'),
                'transaction_sell_lines.unit_price_inc_tax',
                DB::raw('(SELECT pt2.invoice_no FROM transaction_sell_lines_purchase_lines tspl2 
                    INNER JOIN purchase_lines pl2 ON tspl2.purchase_line_id = pl2.id 
                    INNER JOIN transactions pt2 ON pl2.transaction_id = pt2.id 
                    WHERE tspl2.sell_line_id = transaction_sell_lines.id 
                    LIMIT 1) as purchase_invoice'),
                DB::raw("CASE 
                    WHEN p.type = 'variable' AND v.name IS NOT NULL THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_name")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('product_name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Group by product
        $grouped_data = $sell_lines->groupBy('product_name');

        // Calculate product-wise totals
        $products_totals = [];
        foreach ($grouped_data as $product_name => $product_lines) {
            $products_totals[$product_name] = [
                'total_qty' => $product_lines->sum('quantity'),
                'total_amount' => $product_lines->sum(function($line) {
                    return $line->quantity * $line->unit_price_inc_tax;
                }),
            ];
        }

        // Calculate sub totals
        $sub_total_qty = $sell_lines->sum('quantity');
        $sub_total_amount = $sell_lines->sum(function($line) {
            return $line->quantity * $line->unit_price_inc_tax;
        });

        // Calculate grand totals (same as sub totals for this report)
        $grand_total_qty = $sub_total_qty;
        $grand_total_amount = $sub_total_amount;

        $util = new Util();

        return view('businessmanagement::sales_register.print_product_wise_sales', compact(
            'sell_lines',
            'grouped_data',
            'products_totals',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'business',
            'logo',
            'sub_total_qty',
            'sub_total_amount',
            'grand_total_qty',
            'grand_total_amount',
            'util'
        ));
    }

    /**
     * Display Customer Wise Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function customerWiseSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            // Get sell lines with customer and product details
            $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->whereNull('transaction_sell_lines.parent_sell_line_id')
                ->select(
                    't.id as transaction_id',
                    't.invoice_no',
                    't.transaction_date',
                    'c.id as customer_id',
                    'c.name as customer_name',
                    'c.supplier_business_name',
                    'transaction_sell_lines.id as sell_line_id',
                    DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as quantity'),
                    'transaction_sell_lines.unit_price_before_discount',
                    'transaction_sell_lines.unit_price_inc_tax',
                    'transaction_sell_lines.line_discount_type',
                    'transaction_sell_lines.line_discount_amount',
                    'u.short_name as unit_name',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description"),
                    DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as line_total")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $request->location_id);
        }

        // Apply customer filter
        if (!empty($request->customer_id)) {
                $query->where('c.id', $request->customer_id);
            }

            // Apply invoice filter
            if (!empty($request->invoice_no)) {
                $query->where('t.invoice_no', 'like', '%' . $request->invoice_no . '%');
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
                $query->where('t.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();
            
            return DataTables::of($query)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('customer_name', function ($row) {
                    $customer = $row->customer_name ?? 'Walking Customer';
                    if (!empty($row->supplier_business_name)) {
                        $customer = $row->supplier_business_name . ' - ' . $customer;
                    }
                    return $customer;
                })
                ->editColumn('quantity', function ($row) use ($util) {
                    return $util->num_f($row->quantity);
                })
                ->addColumn('unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->unit_price_before_discount . '</span>';
                })
                ->addColumn('discount', function ($row) use ($util) {
                    $discount = 0;
                    if (!empty($row->line_discount_type) && !empty($row->line_discount_amount)) {
                        if ($row->line_discount_type == 'fixed') {
                            $discount = $row->line_discount_amount;
                        } elseif ($row->line_discount_type == 'percentage') {
                            $discount = ($row->unit_price_before_discount * $row->line_discount_amount) / 100;
                        }
                    }
                    return '<span class="display_currency" data-currency_symbol="true">' . $discount . '</span>';
                })
                ->addColumn('net_unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->unit_price_inc_tax . '</span>';
                })
                ->editColumn('line_total', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->line_total . '</span>';
                })
                ->rawColumns(['unit_price', 'discount', 'net_unit_price', 'line_total'])
                ->make(true);
        }

        return view('businessmanagement::sales_register.customer_wise_sales', compact('customers', 'locations'));
    }

    /**
     * Print Customer Wise Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printCustomerWiseSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        // Get sell lines with customer and product details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                'c.id as customer_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as quantity'),
                'transaction_sell_lines.unit_price_before_discount',
                'transaction_sell_lines.unit_price_inc_tax',
                'transaction_sell_lines.line_discount_type',
                'transaction_sell_lines.line_discount_amount',
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $request->location_id);
        }

        // Apply customer filter
        if (!empty($request->customer_id)) {
            $query->where('c.id', $request->customer_id);
        }

        // Apply invoice filter
        if (!empty($request->invoice_no)) {
            $query->where('t.invoice_no', 'like', '%' . $request->invoice_no . '%');
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) == 2) {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        if ($start_date && $end_date) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Group by customer, then by date, then by invoice
        $customers_data = [];
        foreach ($sell_lines as $line) {
            $customer_id = $line->customer_id ?? 0;
            $customer_name = $line->customer_name ?? 'Walking Customer';
            if (!empty($line->supplier_business_name)) {
                $customer_name = $line->supplier_business_name . ' - ' . $customer_name;
            }
            
            $date = \Carbon\Carbon::parse($line->transaction_date)->format('d/m/Y');
            $invoice_no = $line->invoice_no ?? '';
            
            if (!isset($customers_data[$customer_id])) {
                $customers_data[$customer_id] = [
                    'customer_name' => $customer_name,
                    'dates' => [],
                    'total_qty' => 0,
                    'total_amount' => 0
                ];
            }
            
            if (!isset($customers_data[$customer_id]['dates'][$date])) {
                $customers_data[$customer_id]['dates'][$date] = [
                    'invoices' => [],
                    'total_qty' => 0,
                    'total_amount' => 0
                ];
            }
            
            if (!isset($customers_data[$customer_id]['dates'][$date]['invoices'][$invoice_no])) {
                $customers_data[$customer_id]['dates'][$date]['invoices'][$invoice_no] = [
                    'lines' => [],
                    'total_qty' => 0,
                    'total_amount' => 0
                ];
            }
            
            $customers_data[$customer_id]['dates'][$date]['invoices'][$invoice_no]['lines'][] = $line;
            $customers_data[$customer_id]['dates'][$date]['invoices'][$invoice_no]['total_qty'] += $line->quantity;
            $customers_data[$customer_id]['dates'][$date]['invoices'][$invoice_no]['total_amount'] += $line->line_total;
            
            $customers_data[$customer_id]['dates'][$date]['total_qty'] += $line->quantity;
            $customers_data[$customer_id]['dates'][$date]['total_amount'] += $line->line_total;
            
            $customers_data[$customer_id]['total_qty'] += $line->quantity;
            $customers_data[$customer_id]['total_amount'] += $line->line_total;
        }

        // Calculate grand totals
        $grand_total_qty = 0;
        $grand_total_amount = 0;
        foreach ($customers_data as $customer_data) {
            $grand_total_qty += $customer_data['total_qty'];
            $grand_total_amount += $customer_data['total_amount'];
        }

        $util = new Util();

        return view('businessmanagement::sales_register.print_customer_wise_sales', compact(
            'customers_data',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'grand_total_qty',
            'grand_total_amount',
            'util'
        ));
    }

    /**
     * Display Date Wise Net Product Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWiseNetProductSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->pluck('name', 'id');

        if ($request->ajax()) {
            // Get sales lines
            $sales_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->whereNull('transaction_sell_lines.parent_sell_line_id')
                ->select(
                    'p.id as product_id',
                    'p.category_id',
                    'cat.name as category_name',
                    'p.brand_id',
                    'b.name as brand_name',
                    DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                    DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as unit_price'),
                    DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount'),
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description")
                )
                ->groupBy('p.id', 'p.category_id', 'cat.name', 'p.brand_id', 'b.name', 'product_description');

            // Get return lines
            $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell_return')
                ->where('t.status', 'final')
                ->whereNull('transaction_sell_lines.parent_sell_line_id')
                ->select(
                    'p.id as product_id',
                    DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                    DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as return_price'),
                    DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount')
                )
                ->groupBy('p.id');

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sales_query->whereIn('t.location_id', $permitted_locations);
                $return_query->whereIn('t.location_id', $permitted_locations);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $sales_query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
                $return_query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Apply category filter
            if (!empty($request->category_id)) {
                $sales_query->where('p.category_id', $request->category_id);
            }

            // Apply product filter
            if (!empty($request->product_id)) {
                $sales_query->where('p.id', $request->product_id);
                $return_query->where('p.id', $request->product_id);
            }

            $sales = $sales_query->get();
            $returns = $return_query->get()->keyBy('product_id');

            // Combine sales and returns
            $result = [];
            foreach ($sales as $sale) {
                $return = $returns->get($sale->product_id);
                $return_qty = $return ? $return->return_qty : 0;
                $return_price = $return ? $return->return_price : 0;
                $return_amount = $return ? $return->return_amount : 0;
                
                $result[] = [
                    'product_id' => $sale->product_id,
                    'category_id' => $sale->category_id,
                    'category_name' => $sale->category_name ?? 'UNCATEGORIZED',
                    'brand_id' => $sale->brand_id,
                    'brand_name' => $sale->brand_name ?? 'NO BRAND',
                    'product_description' => $sale->product_description,
                    'sale_qty' => $sale->sale_qty ?? 0,
                    'unit_price' => $sale->unit_price ?? 0,
                    'sales_amount' => $sale->sales_amount ?? 0,
                    'return_qty' => $return_qty,
                    'return_price' => $return_price,
                    'return_amount' => $return_amount,
                    'net_qty' => ($sale->sale_qty ?? 0) - $return_qty,
                    'net_amount' => ($sale->sales_amount ?? 0) - $return_amount,
                ];
            }

            $util = new Util();
            
            return DataTables::of(collect($result))
                ->editColumn('sale_qty', function ($row) use ($util) {
                    return $util->num_f($row['sale_qty'], true);
                })
                ->editColumn('unit_price', function ($row) use ($util) {
                    return $util->num_f($row['unit_price'], true);
                })
                ->editColumn('sales_amount', function ($row) use ($util) {
                    return $util->num_f($row['sales_amount'], true);
                })
                ->editColumn('return_qty', function ($row) use ($util) {
                    return $util->num_f($row['return_qty'], true);
                })
                ->editColumn('return_price', function ($row) use ($util) {
                    return $util->num_f($row['return_price'], true);
                })
                ->editColumn('return_amount', function ($row) use ($util) {
                    return $util->num_f($row['return_amount'], true);
                })
                ->editColumn('net_qty', function ($row) use ($util) {
                    return $util->num_f($row['net_qty'], true);
                })
                ->editColumn('net_amount', function ($row) use ($util) {
                    return $util->num_f($row['net_amount'], true);
                })
                ->make(true);
        }

        return view('businessmanagement::sales_register.date_wise_net_product_sales', compact('categories', 'products'));
    }

    /**
     * Print Date Wise Net Product Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWiseNetProductSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        // Get sales lines
        $sales_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'p.brand_id',
                'b.name as brand_name',
                DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as unit_price'),
                DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount'),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            )
            ->groupBy('p.id', 'p.category_id', 'cat.name', 'p.brand_id', 'b.name', 'product_description');

        // Get return lines
        $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as return_price'),
                DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount')
            )
            ->groupBy('p.id');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $sales_query->whereIn('t.location_id', $permitted_locations);
            $return_query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) == 2) {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        if ($start_date && $end_date) {
            $sales_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
            $return_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply category filter
        if (!empty($request->category_id)) {
            $sales_query->where('p.category_id', $request->category_id);
        }

        // Apply product filter
        if (!empty($request->product_id)) {
            $sales_query->where('p.id', $request->product_id);
            $return_query->where('p.id', $request->product_id);
        }

        $sales = $sales_query->get();
        $returns = $return_query->get()->keyBy('product_id');

        // Combine and group by category -> brand -> product
        $categories_data = [];
        foreach ($sales as $sale) {
            $category_id = $sale->category_id ?? 0;
            $category_name = $sale->category_name ?? 'UNCATEGORIZED';
            $brand_id = $sale->brand_id ?? 0;
            $brand_name = $sale->brand_name ?? 'NO BRAND';
            
            $return = $returns->get($sale->product_id);
            $return_qty = $return ? $return->return_qty : 0;
            $return_price = $return ? $return->return_price : 0;
            $return_amount = $return ? $return->return_amount : 0;
            
            if (!isset($categories_data[$category_id])) {
                $categories_data[$category_id] = [
                    'category_name' => $category_name,
                    'brands' => [],
                    'total_sale_qty' => 0,
                    'total_sales_amount' => 0,
                    'total_return_qty' => 0,
                    'total_return_amount' => 0,
                    'total_net_qty' => 0,
                    'total_net_amount' => 0
                ];
            }
            
            if (!isset($categories_data[$category_id]['brands'][$brand_id])) {
                $categories_data[$category_id]['brands'][$brand_id] = [
                    'brand_name' => $brand_name,
                    'products' => [],
                    'total_sale_qty' => 0,
                    'total_sales_amount' => 0,
                    'total_return_qty' => 0,
                    'total_return_amount' => 0,
                    'total_net_qty' => 0,
                    'total_net_amount' => 0
                ];
            }
            
            $net_qty = ($sale->sale_qty ?? 0) - $return_qty;
            $net_amount = ($sale->sales_amount ?? 0) - $return_amount;
            
            $categories_data[$category_id]['brands'][$brand_id]['products'][] = [
                'product_description' => $sale->product_description,
                'sale_qty' => $sale->sale_qty ?? 0,
                'unit_price' => $sale->unit_price ?? 0,
                'sales_amount' => $sale->sales_amount ?? 0,
                'return_qty' => $return_qty,
                'return_price' => $return_price,
                'return_amount' => $return_amount,
                'net_qty' => $net_qty,
                'net_amount' => $net_amount,
            ];
            
            $categories_data[$category_id]['brands'][$brand_id]['total_sale_qty'] += $sale->sale_qty ?? 0;
            $categories_data[$category_id]['brands'][$brand_id]['total_sales_amount'] += $sale->sales_amount ?? 0;
            $categories_data[$category_id]['brands'][$brand_id]['total_return_qty'] += $return_qty;
            $categories_data[$category_id]['brands'][$brand_id]['total_return_amount'] += $return_amount;
            $categories_data[$category_id]['brands'][$brand_id]['total_net_qty'] += $net_qty;
            $categories_data[$category_id]['brands'][$brand_id]['total_net_amount'] += $net_amount;
            
            $categories_data[$category_id]['total_sale_qty'] += $sale->sale_qty ?? 0;
            $categories_data[$category_id]['total_sales_amount'] += $sale->sales_amount ?? 0;
            $categories_data[$category_id]['total_return_qty'] += $return_qty;
            $categories_data[$category_id]['total_return_amount'] += $return_amount;
            $categories_data[$category_id]['total_net_qty'] += $net_qty;
            $categories_data[$category_id]['total_net_amount'] += $net_amount;
        }

        $util = new Util();

        return view('businessmanagement::sales_register.print_date_wise_net_product_sales', compact(
            'categories_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'util'
        ));
    }

    /**
     * Display Customer Wise Net Product Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function customerWiseNetProductSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            // Get sales lines
            $sales_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->whereNull('transaction_sell_lines.parent_sell_line_id')
                ->select(
                    'c.id as customer_id',
                    'c.name as customer_name',
                    'c.supplier_business_name',
                    'p.id as product_id',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description"),
                    DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                    DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as unit_price'),
                    DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount')
                )
                ->groupBy('c.id', 'c.name', 'c.supplier_business_name', 'p.id', 'product_description');

            // Get return lines
            $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell_return')
                ->where('t.status', 'final')
                ->whereNull('transaction_sell_lines.parent_sell_line_id')
                ->select(
                    'c.id as customer_id',
                    'p.id as product_id',
                    DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                    DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as return_price'),
                    DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount')
                )
                ->groupBy('c.id', 'p.id');

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sales_query->whereIn('t.location_id', $permitted_locations);
                $return_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $sales_query->where('t.location_id', $request->location_id);
                $return_query->where('t.location_id', $request->location_id);
            }

            // Apply customer filter
            if (!empty($request->customer_id)) {
                $sales_query->where('c.id', $request->customer_id);
                $return_query->where('c.id', $request->customer_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $sales_query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
                $return_query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
                $sales_query->where('t.created_by', request()->session()->get('user.id'));
                $return_query->where('t.created_by', request()->session()->get('user.id'));
            }

            $sales = $sales_query->get();
            $returns = $return_query->get()->keyBy(function($item) {
                return $item->customer_id . '_' . $item->product_id;
            });

            // Combine sales and returns
            $result = [];
            foreach ($sales as $sale) {
                $key = $sale->customer_id . '_' . $sale->product_id;
                $return = $returns->get($key);
                $return_qty = $return ? $return->return_qty : 0;
                $return_price = $return ? $return->return_price : 0;
                $return_amount = $return ? $return->return_amount : 0;
                
                $result[] = [
                    'customer_id' => $sale->customer_id ?? 0,
                    'customer_name' => $sale->customer_name ?? 'Walking Customer',
                    'supplier_business_name' => $sale->supplier_business_name ?? '',
                    'product_id' => $sale->product_id,
                    'product_description' => $sale->product_description,
                    'sale_qty' => $sale->sale_qty ?? 0,
                    'unit_price' => $sale->unit_price ?? 0,
                    'sales_amount' => $sale->sales_amount ?? 0,
                    'return_qty' => $return_qty,
                    'return_price' => $return_price,
                    'return_amount' => $return_amount,
                    'net_qty' => ($sale->sale_qty ?? 0) - $return_qty,
                    'net_amount' => ($sale->sales_amount ?? 0) - $return_amount,
                ];
            }

            $util = new Util();
            
            return DataTables::of(collect($result))
                ->editColumn('customer_name', function ($row) {
                    $customer = $row['customer_name'] ?? 'Walking Customer';
                    if (!empty($row['supplier_business_name'])) {
                        $customer = $row['supplier_business_name'] . ' - ' . $customer;
                    }
                    return $customer;
                })
                ->editColumn('sale_qty', function ($row) use ($util) {
                    return $util->num_f($row['sale_qty'], true);
                })
                ->editColumn('unit_price', function ($row) use ($util) {
                    return $util->num_f($row['unit_price'], true);
                })
                ->editColumn('sales_amount', function ($row) use ($util) {
                    return $util->num_f($row['sales_amount'], true);
                })
                ->editColumn('return_qty', function ($row) use ($util) {
                    return $util->num_f($row['return_qty'], true);
                })
                ->editColumn('return_price', function ($row) use ($util) {
                    return $util->num_f($row['return_price'], true);
                })
                ->editColumn('return_amount', function ($row) use ($util) {
                    return $util->num_f($row['return_amount'], true);
                })
                ->editColumn('net_qty', function ($row) use ($util) {
                    return $util->num_f($row['net_qty'], true);
                })
                ->editColumn('net_amount', function ($row) use ($util) {
                    return $util->num_f($row['net_amount'], true);
                })
                ->make(true);
        }

        return view('businessmanagement::sales_register.customer_wise_net_product_sales', compact('customers', 'locations'));
    }

    /**
     * Print Customer Wise Net Product Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printCustomerWiseNetProductSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        // Get sales lines
        $sales_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'c.id as customer_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as unit_price'),
                DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount')
            )
            ->groupBy('c.id', 'c.name', 'c.supplier_business_name', 'p.id', 'product_description');

        // Get return lines
        $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'c.id as customer_id',
                'p.id as product_id',
                DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as return_price'),
                DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount')
            )
            ->groupBy('c.id', 'p.id');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $sales_query->whereIn('t.location_id', $permitted_locations);
            $return_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_query->where('t.location_id', $request->location_id);
            $return_query->where('t.location_id', $request->location_id);
        }

        // Apply customer filter
        $customer_name = null;
        $is_single_customer = false;
        if (!empty($request->customer_id)) {
            $sales_query->where('c.id', $request->customer_id);
            $return_query->where('c.id', $request->customer_id);
            $customer = Contact::find($request->customer_id);
            if ($customer) {
                $customer_name = $customer->name;
                if (!empty($customer->supplier_business_name)) {
                    $customer_name = $customer->supplier_business_name . ' - ' . $customer_name;
                }
                $is_single_customer = true;
            }
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) == 2) {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        if ($start_date && $end_date) {
            $sales_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
            $return_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $sales_query->where('t.created_by', request()->session()->get('user.id'));
            $return_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales = $sales_query->orderBy('c.name', 'asc')->get();
        $returns = $return_query->get()->keyBy(function($item) {
            return $item->customer_id . '_' . $item->product_id;
        });

        // Combine and group by customer
        $customers_data = [];
        $sl = 1;
        foreach ($sales as $sale) {
            $customer_id = $sale->customer_id ?? 0;
            $customer_name = $sale->customer_name ?? 'Walking Customer';
            if (!empty($sale->supplier_business_name)) {
                $customer_name = $sale->supplier_business_name . ' - ' . $customer_name;
            }
            
            $key = $customer_id . '_' . $sale->product_id;
            $return = $returns->get($key);
            $return_qty = $return ? $return->return_qty : 0;
            $return_price = $return ? $return->return_price : 0;
            $return_amount = $return ? $return->return_amount : 0;
            
            if (!isset($customers_data[$customer_id])) {
                $customers_data[$customer_id] = [
                    'customer_name' => $customer_name,
                    'products' => [],
                    'total_sale_qty' => 0,
                    'total_sales_amount' => 0,
                    'total_return_qty' => 0,
                    'total_return_amount' => 0,
                    'total_net_qty' => 0,
                    'total_net_amount' => 0
                ];
            }
            
            $net_qty = ($sale->sale_qty ?? 0) - $return_qty;
            $net_amount = ($sale->sales_amount ?? 0) - $return_amount;
            
            $customers_data[$customer_id]['products'][] = [
                'sl' => $sl++,
                'product_description' => $sale->product_description,
                'sale_qty' => $sale->sale_qty ?? 0,
                'unit_price' => $sale->unit_price ?? 0,
                'sales_amount' => $sale->sales_amount ?? 0,
                'return_qty' => $return_qty,
                'return_price' => $return_price,
                'return_amount' => $return_amount,
                'net_qty' => $net_qty,
                'net_amount' => $net_amount,
            ];
            
            $customers_data[$customer_id]['total_sale_qty'] += $sale->sale_qty ?? 0;
            $customers_data[$customer_id]['total_sales_amount'] += $sale->sales_amount ?? 0;
            $customers_data[$customer_id]['total_return_qty'] += $return_qty;
            $customers_data[$customer_id]['total_return_amount'] += $return_amount;
            $customers_data[$customer_id]['total_net_qty'] += $net_qty;
            $customers_data[$customer_id]['total_net_amount'] += $net_amount;
        }

        // Calculate grand totals
        $grand_total_sale_qty = 0;
        $grand_total_sales_amount = 0;
        $grand_total_return_qty = 0;
        $grand_total_return_amount = 0;
        $grand_total_net_qty = 0;
        $grand_total_net_amount = 0;
        
        foreach ($customers_data as $customer_data) {
            $grand_total_sale_qty += $customer_data['total_sale_qty'];
            $grand_total_sales_amount += $customer_data['total_sales_amount'];
            $grand_total_return_qty += $customer_data['total_return_qty'];
            $grand_total_return_amount += $customer_data['total_return_amount'];
            $grand_total_net_qty += $customer_data['total_net_qty'];
            $grand_total_net_amount += $customer_data['total_net_amount'];
        }

        $util = new Util();

        return view('businessmanagement::sales_register.print_customer_wise_net_product_sales', compact(
            'customers_data',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'customer_name',
            'is_single_customer',
            'grand_total_sale_qty',
            'grand_total_sales_amount',
            'grand_total_return_qty',
            'grand_total_return_amount',
            'grand_total_net_qty',
            'grand_total_net_amount',
            'util'
        ));
    }

    /**
     * Display Date Wise Gross Profit Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWiseGrossProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            $query = Transaction::leftJoin('contacts as c', 'transactions.contact_id', '=', 'c.id')
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell')
                ->where('transactions.status', 'final')
                ->select(
                    'transactions.id',
                    'transactions.invoice_no',
                    'transactions.transaction_date',
                    'transactions.final_total',
                    'transactions.discount_amount',
                    'transactions.discount_type',
                    'transactions.total_before_tax',
                    'transactions.tax_amount',
                    'transactions.shipping_charges',
                    'transactions.additional_expense_value_1',
                    'transactions.additional_expense_value_2',
                    'transactions.additional_expense_value_3',
                    'transactions.additional_expense_value_4',
                    'c.name as trader_name',
                    'c.supplier_business_name',
                    DB::raw('(SELECT SUM(IF(tspl.id IS NULL, 0, (tspl.quantity - tspl.qty_returned) * pl.purchase_price_inc_tax))
                        FROM transaction_sell_lines tsl2
                        LEFT JOIN transaction_sell_lines_purchase_lines tspl ON tsl2.id = tspl.sell_line_id
                        LEFT JOIN purchase_lines pl ON tspl.purchase_line_id = pl.id
                        WHERE tsl2.transaction_id = transactions.id AND tsl2.parent_sell_line_id IS NULL) as total_cost')
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('transactions.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $query->where('transactions.location_id', $request->location_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $query->whereDate('transactions.transaction_date', '>=', $request->start_date)
                    ->whereDate('transactions.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
                $query->where('transactions.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();
            
            return DataTables::of($query)
                ->editColumn('invoice_no', function ($row) {
                    return $row->invoice_no ?? '';
                })
                ->addColumn('trader_name', function ($row) {
                    $trader = $row->trader_name ?? 'Walking Customer';
                    if (!empty($row->supplier_business_name)) {
                        $trader = $row->supplier_business_name . ' - ' . $trader;
                    }
                    return $trader;
                })
                ->addColumn('total_amount', function ($row) {
                    return $row->total_before_tax ?? 0;
                })
                ->addColumn('discount', function ($row) {
                    $discount = $row->discount_amount ?? 0;
                    if (!empty($discount) && $row->discount_type == 'percentage') {
                        $discount = ($row->total_before_tax ?? 0) * ($discount / 100);
                    }
                    return $discount;
                })
                ->addColumn('extra_charges', function ($row) {
                    $extra = ($row->shipping_charges ?? 0) + 
                             ($row->additional_expense_value_1 ?? 0) + 
                             ($row->additional_expense_value_2 ?? 0) + 
                             ($row->additional_expense_value_3 ?? 0) + 
                             ($row->additional_expense_value_4 ?? 0);
                    return $extra;
                })
                ->addColumn('ait', function ($row) {
                    // AIT calculation - check if there's AIT in tax breakdown
                    return 0; // Default to 0, can be calculated from tax breakdown if available
                })
                ->addColumn('vat', function ($row) {
                    return $row->tax_amount ?? 0;
                })
                ->addColumn('net_payable', function ($row) {
                    return $row->final_total ?? 0;
                })
                ->addColumn('total_cost', function ($row) {
                    return $row->total_cost ?? 0;
                })
                ->addColumn('gross_profit', function ($row) {
                    $net_payable = $row->final_total ?? 0;
                    $total_cost = $row->total_cost ?? 0;
                    return $net_payable - $total_cost;
                })
                ->addColumn('profit_percent', function ($row) {
                    $net_payable = $row->final_total ?? 0;
                    $total_cost = $row->total_cost ?? 0;
                    $gross_profit = $net_payable - $total_cost;
                    if ($net_payable != 0) {
                        return ($gross_profit / $net_payable) * 100;
                    }
                    return 0;
                })
                ->editColumn('total_amount', function ($row) use ($util) {
                    return '<span class="display_currency" data-currency_symbol="true">' . ($row->total_before_tax ?? 0) . '</span>';
                })
                ->editColumn('discount', function ($row) use ($util) {
                    $discount = $row->discount_amount ?? 0;
                    if (!empty($discount) && $row->discount_type == 'percentage') {
                        $discount = ($row->total_before_tax ?? 0) * ($discount / 100);
                    }
                    return '<span class="display_currency" data-currency_symbol="true">' . $discount . '</span>';
                })
                ->editColumn('extra_charges', function ($row) use ($util) {
                    $extra = ($row->shipping_charges ?? 0) + 
                             ($row->additional_expense_value_1 ?? 0) + 
                             ($row->additional_expense_value_2 ?? 0) + 
                             ($row->additional_expense_value_3 ?? 0) + 
                             ($row->additional_expense_value_4 ?? 0);
                    return '<span class="display_currency" data-currency_symbol="true">' . $extra . '</span>';
                })
                ->editColumn('ait', function ($row) use ($util) {
                    return '<span class="display_currency" data-currency_symbol="true">0.00</span>';
                })
                ->editColumn('vat', function ($row) use ($util) {
                    return '<span class="display_currency" data-currency_symbol="true">' . ($row->tax_amount ?? 0) . '</span>';
                })
                ->editColumn('net_payable', function ($row) use ($util) {
                    return '<span class="display_currency" data-currency_symbol="true">' . ($row->final_total ?? 0) . '</span>';
                })
                ->editColumn('total_cost', function ($row) use ($util) {
                    return '<span class="display_currency" data-currency_symbol="true">' . ($row->total_cost ?? 0) . '</span>';
                })
                ->editColumn('gross_profit', function ($row) use ($util) {
                    $net_payable = $row->final_total ?? 0;
                    $total_cost = $row->total_cost ?? 0;
                    $gross_profit = $net_payable - $total_cost;
                    $class = $gross_profit < 0 ? 'text-red' : '';
                    return '<span class="display_currency ' . $class . '" data-currency_symbol="true">' . $gross_profit . '</span>';
                })
                ->editColumn('profit_percent', function ($row) use ($util) {
                    $net_payable = $row->final_total ?? 0;
                    $total_cost = $row->total_cost ?? 0;
                    $gross_profit = $net_payable - $total_cost;
                    $profit_percent = 0;
                    if ($net_payable != 0) {
                        $profit_percent = ($gross_profit / $net_payable) * 100;
                    }
                    $class = $profit_percent < 0 ? 'text-red' : '';
                    return '<span class="' . $class . '">' . $util->num_f($profit_percent, true) . '%</span>';
                })
                ->rawColumns(['total_amount', 'discount', 'extra_charges', 'ait', 'vat', 'net_payable', 'total_cost', 'gross_profit', 'profit_percent'])
                ->make(true);
        }

        return view('businessmanagement::sales_register.date_wise_gross_profit', compact('locations'));
    }

    /**
     * Print Date Wise Gross Profit Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWiseGrossProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        $query = Transaction::leftJoin('contacts as c', 'transactions.contact_id', '=', 'c.id')
            ->where('transactions.business_id', $business_id)
            ->where('transactions.type', 'sell')
            ->where('transactions.status', 'final')
            ->select(
                'transactions.id',
                'transactions.invoice_no',
                'transactions.transaction_date',
                'transactions.final_total',
                'transactions.discount_amount',
                'transactions.discount_type',
                'transactions.total_before_tax',
                'transactions.tax_amount',
                'transactions.shipping_charges',
                'transactions.additional_expense_value_1',
                'transactions.additional_expense_value_2',
                'transactions.additional_expense_value_3',
                'transactions.additional_expense_value_4',
                'c.name as trader_name',
                'c.supplier_business_name',
                DB::raw('(SELECT SUM(IF(tspl.id IS NULL, 0, (tspl.quantity - tspl.qty_returned) * pl.purchase_price_inc_tax))
                    FROM transaction_sell_lines tsl2
                    LEFT JOIN transaction_sell_lines_purchase_lines tspl ON tsl2.id = tspl.sell_line_id
                    LEFT JOIN purchase_lines pl ON tspl.purchase_line_id = pl.id
                    WHERE tsl2.transaction_id = transactions.id AND tsl2.parent_sell_line_id IS NULL) as total_cost')
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('transactions.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('transactions.location_id', $request->location_id);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) == 2) {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        if ($start_date && $end_date) {
            $query->whereDate('transactions.transaction_date', '>=', $start_date)
                ->whereDate('transactions.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('transactions.created_by', request()->session()->get('user.id'));
        }

        $transactions = $query->orderBy('transactions.transaction_date', 'asc')
            ->orderBy('transactions.invoice_no', 'asc')
            ->get();

        // Calculate totals
        $grand_total_amount = 0;
        $grand_total_discount = 0;
        $grand_total_extra_charges = 0;
        $grand_total_ait = 0;
        $grand_total_vat = 0;
        $grand_total_net_payable = 0;
        $grand_total_cost = 0;
        $grand_total_gross_profit = 0;

        foreach ($transactions as $transaction) {
            $discount = $transaction->discount_amount ?? 0;
            if (!empty($discount) && $transaction->discount_type == 'percentage') {
                $discount = ($transaction->total_before_tax ?? 0) * ($discount / 100);
            }

            $extra_charges = ($transaction->shipping_charges ?? 0) + 
                            ($transaction->additional_expense_value_1 ?? 0) + 
                            ($transaction->additional_expense_value_2 ?? 0) + 
                            ($transaction->additional_expense_value_3 ?? 0) + 
                            ($transaction->additional_expense_value_4 ?? 0);

            $grand_total_amount += $transaction->total_before_tax ?? 0;
            $grand_total_discount += $discount;
            $grand_total_extra_charges += $extra_charges;
            $grand_total_vat += $transaction->tax_amount ?? 0;
            $grand_total_net_payable += $transaction->final_total ?? 0;
            $grand_total_cost += $transaction->total_cost ?? 0;
        }

        $grand_total_gross_profit = $grand_total_net_payable - $grand_total_cost;
        $grand_total_profit_percent = 0;
        if ($grand_total_net_payable != 0) {
            $grand_total_profit_percent = ($grand_total_gross_profit / $grand_total_net_payable) * 100;
        }

        $util = new Util();

        return view('businessmanagement::sales_register.print_date_wise_gross_profit', compact(
            'transactions',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'grand_total_amount',
            'grand_total_discount',
            'grand_total_extra_charges',
            'grand_total_ait',
            'grand_total_vat',
            'grand_total_net_payable',
            'grand_total_cost',
            'grand_total_gross_profit',
            'grand_total_profit_percent',
            'util'
        ));
    }

    /**
     * Display Product Wise Gross Profit Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productWiseGrossProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->pluck('name', 'id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
                ->leftJoin('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->whereNull('transaction_sell_lines.parent_sell_line_id')
                ->select(
                    'p.id as product_id',
                    'p.category_id',
                    'cat.name as category_name',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_model"),
                    DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                    DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as unit_price'),
                    DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as total_price'),
                    DB::raw('AVG(IF(tspl.id IS NULL, 0, pl.purchase_price_inc_tax)) as unit_cost'),
                    DB::raw('SUM(IF(tspl.id IS NULL, 0, (tspl.quantity - tspl.qty_returned) * pl.purchase_price_inc_tax)) as total_cost')
                )
                ->groupBy('p.id', 'p.category_id', 'cat.name', 'product_model');

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $query->where('t.location_id', $request->location_id);
            }

            // Apply category filter
            if (!empty($request->category_id)) {
                $query->where('p.category_id', $request->category_id);
            }

            // Apply product filter
            if (!empty($request->product_id)) {
                $query->where('p.id', $request->product_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
                $query->where('t.created_by', request()->session()->get('user.id'));
            }

            $results = $query->get();

            // Calculate gross profit and profit percent
            $data = [];
            foreach ($results as $row) {
                $gross_profit = ($row->total_price ?? 0) - ($row->total_cost ?? 0);
                $profit_percent = 0;
                if (($row->total_price ?? 0) != 0) {
                    $profit_percent = ($gross_profit / ($row->total_price ?? 0)) * 100;
                }

                $data[] = [
                    'product_id' => $row->product_id,
                    'category_id' => $row->category_id,
                    'category_name' => $row->category_name ?? 'UNCATEGORIZED',
                    'product_model' => $row->product_model,
                    'qty' => $row->qty ?? 0,
                    'unit_price' => $row->unit_price ?? 0,
                    'total_price' => $row->total_price ?? 0,
                    'unit_cost' => $row->unit_cost ?? 0,
                    'total_cost' => $row->total_cost ?? 0,
                    'gross_profit' => $gross_profit,
                    'profit_percent' => $profit_percent,
                ];
            }

            $util = new Util();
            
            return DataTables::of(collect($data))
                ->editColumn('qty', function ($row) use ($util) {
                    return $util->num_f($row['qty'], true);
                })
                ->editColumn('unit_price', function ($row) use ($util) {
                    return $util->num_f($row['unit_price'], true);
                })
                ->editColumn('total_price', function ($row) use ($util) {
                    return $util->num_f($row['total_price'], true);
                })
                ->editColumn('unit_cost', function ($row) use ($util) {
                    return $util->num_f($row['unit_cost'], true);
                })
                ->editColumn('total_cost', function ($row) use ($util) {
                    return $util->num_f($row['total_cost'], true);
                })
                ->editColumn('gross_profit', function ($row) use ($util) {
                    $class = $row['gross_profit'] < 0 ? 'text-red' : '';
                    return '<span class="' . $class . '">' . $util->num_f($row['gross_profit'], true) . '</span>';
                })
                ->editColumn('profit_percent', function ($row) use ($util) {
                    $class = $row['profit_percent'] < 0 ? 'text-red' : '';
                    return '<span class="' . $class . '">' . $util->num_f($row['profit_percent'], true) . '%</span>';
                })
                ->rawColumns(['gross_profit', 'profit_percent'])
                ->make(true);
        }

        return view('businessmanagement::sales_register.product_wise_gross_profit', compact('categories', 'products', 'locations'));
    }

    /**
     * Print Product Wise Gross Profit Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductWiseGrossProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            // Validate that user has access to the requested location
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
            ->leftJoin('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_model"),
                DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as unit_price'),
                DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as total_price'),
                DB::raw('AVG(IF(tspl.id IS NULL, 0, pl.purchase_price_inc_tax)) as unit_cost'),
                DB::raw('SUM(IF(tspl.id IS NULL, 0, (tspl.quantity - tspl.qty_returned) * pl.purchase_price_inc_tax)) as total_cost')
            )
            ->groupBy('p.id', 'p.category_id', 'cat.name', 'product_model');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            $query->where('t.location_id', $request->location_id);
        }

        // Apply category filter
        if (!empty($request->category_id)) {
            $query->where('p.category_id', $request->category_id);
        }

        // Apply product filter
        if (!empty($request->product_id)) {
            $query->where('p.id', $request->product_id);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) == 2) {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        if ($start_date && $end_date) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $results = $query->orderBy('cat.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group by category
        $categories_data = [];
        foreach ($results as $row) {
            $category_id = $row->category_id ?? 0;
            $category_name = $row->category_name ?? 'UNCATEGORIZED';
            
            $gross_profit = ($row->total_price ?? 0) - ($row->total_cost ?? 0);
            $profit_percent = 0;
            if (($row->total_price ?? 0) != 0) {
                $profit_percent = ($gross_profit / ($row->total_price ?? 0)) * 100;
            }

            if (!isset($categories_data[$category_id])) {
                $categories_data[$category_id] = [
                    'category_name' => $category_name,
                    'products' => [],
                    'total_qty' => 0,
                    'total_price' => 0,
                    'total_cost' => 0,
                    'total_gross_profit' => 0,
                ];
            }

            $categories_data[$category_id]['products'][] = [
                'product_model' => $row->product_model,
                'qty' => $row->qty ?? 0,
                'unit_price' => $row->unit_price ?? 0,
                'total_price' => $row->total_price ?? 0,
                'unit_cost' => $row->unit_cost ?? 0,
                'total_cost' => $row->total_cost ?? 0,
                'gross_profit' => $gross_profit,
                'profit_percent' => $profit_percent,
            ];

            $categories_data[$category_id]['total_qty'] += $row->qty ?? 0;
            $categories_data[$category_id]['total_price'] += $row->total_price ?? 0;
            $categories_data[$category_id]['total_cost'] += $row->total_cost ?? 0;
            $categories_data[$category_id]['total_gross_profit'] += $gross_profit;
        }

        // Calculate category profit percent
        foreach ($categories_data as $category_id => $category_data) {
            $profit_percent = 0;
            if ($category_data['total_price'] != 0) {
                $profit_percent = ($category_data['total_gross_profit'] / $category_data['total_price']) * 100;
            }
            $categories_data[$category_id]['profit_percent'] = $profit_percent;
        }

        // Calculate grand totals
        $grand_total_qty = 0;
        $grand_total_price = 0;
        $grand_total_cost = 0;
        $grand_total_gross_profit = 0;

        foreach ($categories_data as $category_data) {
            $grand_total_qty += $category_data['total_qty'];
            $grand_total_price += $category_data['total_price'];
            $grand_total_cost += $category_data['total_cost'];
            $grand_total_gross_profit += $category_data['total_gross_profit'];
        }

        $grand_total_profit_percent = 0;
        if ($grand_total_price != 0) {
            $grand_total_profit_percent = ($grand_total_gross_profit / $grand_total_price) * 100;
        }

        $util = new Util();

        return view('businessmanagement::sales_register.print_product_wise_gross_profit', compact(
            'categories_data',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'grand_total_qty',
            'grand_total_price',
            'grand_total_cost',
            'grand_total_gross_profit',
            'grand_total_profit_percent',
            'util'
        ));
    }

    /**
     * Display Date Wise Sales Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWiseSalesReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');

        // Default to current date if not provided
        if (empty($start_date)) {
            $start_date = date('Y-m-d');
        }
        if (empty($end_date)) {
            $end_date = date('Y-m-d');
        }

        $sales_returns = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
            ->leftJoin('business_locations', 'transactions.location_id', '=', 'business_locations.id')
            ->where('transactions.business_id', $business_id)
            ->where('transactions.type', 'sell_return')
            ->where('transactions.status', 'final')
            ->select(
                'transactions.id',
                'transactions.invoice_no',
                'transactions.ref_no',
                'transactions.transaction_date',
                'transactions.final_total',
                'transactions.payment_status',
                'transactions.additional_notes',
                'contacts.name as customer_name',
                'contacts.supplier_business_name',
                'business_locations.name as location_name'
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $sales_returns->whereIn('transactions.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_returns->where('transactions.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $sales_returns->whereDate('transactions.transaction_date', '>=', $start_date)
                ->whereDate('transactions.transaction_date', '<=', $end_date);
        }

        // Permission check for view own sales only
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $sales_returns->where('transactions.created_by', request()->session()->get('user.id'));
        }

        $sales_returns = $sales_returns->orderBy('transactions.transaction_date', 'asc')
            ->orderBy('transactions.invoice_no', 'asc')
            ->get();

        $util = new Util();
        $grand_total = $sales_returns->sum('final_total');

        return view('businessmanagement::sales_register.date_wise_sales_return', compact(
            'sales_returns',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'grand_total',
            'util'
        ));
    }

    /**
     * Print Date Wise Sales Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWiseSalesReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');

        // Default to current date if not provided
        if (empty($start_date)) {
            $start_date = date('Y-m-d');
        }
        if (empty($end_date)) {
            $end_date = date('Y-m-d');
        }

        $sales_returns = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
            ->leftJoin('business_locations', 'transactions.location_id', '=', 'business_locations.id')
            ->where('transactions.business_id', $business_id)
            ->where('transactions.type', 'sell_return')
            ->where('transactions.status', 'final')
            ->select(
                'transactions.id',
                'transactions.invoice_no',
                'transactions.ref_no',
                'transactions.transaction_date',
                'transactions.final_total',
                'transactions.payment_status',
                'transactions.additional_notes',
                'contacts.name as customer_name',
                'contacts.supplier_business_name',
                'business_locations.name as location_name'
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $sales_returns->whereIn('transactions.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_returns->where('transactions.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $sales_returns->whereDate('transactions.transaction_date', '>=', $start_date)
                ->whereDate('transactions.transaction_date', '<=', $end_date);
        }

        // Permission check for view own sales only
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $sales_returns->where('transactions.created_by', request()->session()->get('user.id'));
        }

        $sales_returns = $sales_returns->orderBy('transactions.transaction_date', 'asc')
            ->orderBy('transactions.invoice_no', 'asc')
            ->get();

        $util = new Util();
        $grand_total = $sales_returns->sum('final_total');

        return view('businessmanagement::sales_register.print_date_wise_sales_return', compact(
            'sales_returns',
            'start_date',
            'end_date',
            'location_id',
            'business',
            'logo',
            'grand_total',
            'util'
        ));
    }

    /**
     * Display Product Wise Sales Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productWiseSalesReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Default to current date if not provided
        if (empty($start_date)) {
            $start_date = date('Y-m-d');
        }
        if (empty($end_date)) {
            $end_date = date('Y-m-d');
        }

        // Get sell return lines with product details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'transaction_sell_lines.id as sell_line_id',
                'transaction_sell_lines.quantity',
                'transaction_sell_lines.unit_price_inc_tax',
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $query->where('p.id', $product_id);
        }

        // Permission check for view own sales only
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_return_lines = $query->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        $util = new Util();
        
        // Calculate totals
        $grand_total_qty = $sales_return_lines->sum('quantity');
        $grand_total_price = $sales_return_lines->sum('line_total');

        return view('businessmanagement::sales_register.product_wise_sales_return', compact(
            'sales_return_lines',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'grand_total_qty',
            'grand_total_price',
            'util'
        ));
    }

    /**
     * Print Product Wise Sales Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductWiseSalesReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Default to current date if not provided
        if (empty($start_date)) {
            $start_date = date('Y-m-d');
        }
        if (empty($end_date)) {
            $end_date = date('Y-m-d');
        }

        // Get sell return lines with product details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'transaction_sell_lines.id as sell_line_id',
                'transaction_sell_lines.quantity',
                'transaction_sell_lines.unit_price_inc_tax',
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $query->where('p.id', $product_id);
        }

        // Permission check for view own sales only
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_return_lines = $query->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        $util = new Util();
        
        // Calculate totals
        $grand_total_qty = $sales_return_lines->sum('quantity');
        $grand_total_price = $sales_return_lines->sum('line_total');

        return view('businessmanagement::sales_register.print_product_wise_sales_return', compact(
            'sales_return_lines',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'business',
            'logo',
            'grand_total_qty',
            'grand_total_price',
            'util'
        ));
    }

    /**
     * Display Customer Wise Sales Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function customerWiseSalesReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Default to current date if not provided
        if (empty($start_date)) {
            $start_date = date('Y-m-d');
        }
        if (empty($end_date)) {
            $end_date = date('Y-m-d');
        }

        // Get sell return lines with product and customer details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                'c.id as customer_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'transaction_sell_lines.id as sell_line_id',
                'transaction_sell_lines.quantity',
                'transaction_sell_lines.unit_price_inc_tax',
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('c.id', $customer_id);
        }

        // Permission check for view own sales only
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_return_lines = $query->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        $util = new Util();
        
        // Group by customer for display
        $grouped_data = $sales_return_lines->groupBy('customer_id');
        
        // Calculate totals
        $grand_total_qty = $sales_return_lines->sum('quantity');
        $grand_total_price = $sales_return_lines->sum('line_total');

        return view('businessmanagement::sales_register.customer_wise_sales_return', compact(
            'sales_return_lines',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'locations',
            'customers',
            'grand_total_qty',
            'grand_total_price',
            'util'
        ));
    }

    /**
     * Print Customer Wise Sales Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printCustomerWiseSalesReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Default to current date if not provided
        if (empty($start_date)) {
            $start_date = date('Y-m-d');
        }
        if (empty($end_date)) {
            $end_date = date('Y-m-d');
        }

        // Get sell return lines with product and customer details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                'c.id as customer_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'transaction_sell_lines.id as sell_line_id',
                'transaction_sell_lines.quantity',
                'transaction_sell_lines.unit_price_inc_tax',
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('c.id', $customer_id);
        }

        // Permission check for view own sales only
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_return_lines = $query->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        $util = new Util();
        
        // Group by customer for display
        $grouped_data = $sales_return_lines->groupBy('customer_id');
        
        // Calculate totals
        $grand_total_qty = $sales_return_lines->sum('quantity');
        $grand_total_price = $sales_return_lines->sum('line_total');

        return view('businessmanagement::sales_register.print_customer_wise_sales_return', compact(
            'sales_return_lines',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'business',
            'logo',
            'grand_total_qty',
            'grand_total_price',
            'util'
        ));
    }

    /**
     * Display Explicit Product Wise Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function explicitProductWiseSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');

        // Get sell lines with product details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.ref_no',
                't.transaction_date',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as quantity'),
                'transaction_sell_lines.unit_price_inc_tax',
                'u.short_name as unit_name',
                DB::raw('(SELECT pt2.invoice_no FROM transaction_sell_lines_purchase_lines tspl2 
                    INNER JOIN purchase_lines pl2 ON tspl2.purchase_line_id = pl2.id 
                    INNER JOIN transactions pt2 ON pl2.transaction_id = pt2.id 
                    WHERE tspl2.sell_line_id = transaction_sell_lines.id 
                    LIMIT 1) as purchase_invoice'),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $query->orderBy('p.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        $util = new Util();
        
        // Group by product for display
        $grouped_data = $sales_lines->groupBy('product_id');
        
        // Calculate totals
        $grand_total_qty = $sales_lines->sum('quantity');
        $grand_total_price = $sales_lines->sum('line_total');

        return view('businessmanagement::sales_register.explicit_product_wise_sales', compact(
            'sales_lines',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'locations',
            'categories',
            'grand_total_qty',
            'grand_total_price',
            'util'
        ));
    }

    /**
     * Print Explicit Product Wise Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printExplicitProductWiseSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');

        // Get sell lines with product details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.ref_no',
                't.transaction_date',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as quantity'),
                'transaction_sell_lines.unit_price_inc_tax',
                'u.short_name as unit_name',
                DB::raw('(SELECT pt2.invoice_no FROM transaction_sell_lines_purchase_lines tspl2 
                    INNER JOIN purchase_lines pl2 ON tspl2.purchase_line_id = pl2.id 
                    INNER JOIN transactions pt2 ON pl2.transaction_id = pt2.id 
                    WHERE tspl2.sell_line_id = transaction_sell_lines.id 
                    LIMIT 1) as purchase_invoice'),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $query->orderBy('p.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        $util = new Util();
        
        // Group by product for display
        $grouped_data = $sales_lines->groupBy('product_id');
        
        // Calculate totals
        $grand_total_qty = $sales_lines->sum('quantity');
        $grand_total_price = $sales_lines->sum('line_total');

        return view('businessmanagement::sales_register.print_explicit_product_wise_sales', compact(
            'sales_lines',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'business',
            'logo',
            'grand_total_qty',
            'grand_total_price',
            'util'
        ));
    }

    /**
     * Display Customer Wise Statement Report
     *
     * @return \Illuminate\Http\Response
     */
    public function customerWiseStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get customers list
        $customers_query = Contact::where('business_id', $business_id)
            ->where('type', 'customer');

        if (!empty($customer_id)) {
            $customers_query->where('id', $customer_id);
        }

        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            // Filter customers who have transactions in permitted locations
            $customers_query->whereHas('transactions', function($q) use ($permitted_locations) {
                $q->whereIn('location_id', $permitted_locations);
            });
        }

        $customers_list = $customers_query->get();

        $table_data = [];
        foreach ($customers_list as $customer) {
            // Calculate opening balance (before start_date)
            $opening_balance_query = Transaction::where('business_id', $business_id)
                ->where('contact_id', $customer->id)
                ->whereIn('type', ['sell', 'sell_return', 'opening_balance', 'ledger_discount']);

            if (!empty($location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    continue; // Skip this customer if location is not permitted
                }
                $opening_balance_query->where('location_id', $location_id);
            }

            if ($permitted_locations != 'all') {
                $opening_balance_query->whereIn('location_id', $permitted_locations);
            }

            if (!empty($start_date)) {
                $opening_balance_query->whereDate('transaction_date', '<', $start_date);
            }

            $opening_balance_data = $opening_balance_query->select(
                DB::raw("SUM(IF(type = 'sell' AND status = 'final', final_total, 0)) as total_sales"),
                DB::raw("SUM(IF(type = 'sell_return' AND status = 'final', final_total, 0)) as total_sales_return"),
                DB::raw("SUM(IF(type = 'opening_balance', final_total, 0)) as total_opening_balance"),
                DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
            )->first();

            // Get payments before start_date
            $opening_payments = 0;
            if (!empty($start_date)) {
                $opening_payments_query = \App\TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['sell', 'sell_return', 'opening_balance'])
                    ->whereDate('transaction_payments.paid_on', '<', $start_date);

                if (!empty($location_id)) {
                    $opening_payments_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $opening_payments_query->whereIn('t.location_id', $permitted_locations);
                }

                $opening_payments = $opening_payments_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;
            }

            $opening_balance = ($opening_balance_data->total_sales ?? 0) 
                - ($opening_balance_data->total_sales_return ?? 0)
                + ($opening_balance_data->total_opening_balance ?? 0)
                - ($opening_balance_data->total_ledger_discount ?? 0)
                - $opening_payments;

            // Get transactions in date range
            $period_query = Transaction::where('business_id', $business_id)
                ->where('contact_id', $customer->id)
                ->whereIn('type', ['sell', 'sell_return', 'purchase', 'purchase_return', 'ledger_discount']);

            if (!empty($location_id)) {
                $period_query->where('location_id', $location_id);
            }

            if ($permitted_locations != 'all') {
                $period_query->whereIn('location_id', $permitted_locations);
            }

            if (!empty($start_date) && !empty($end_date)) {
                $period_query->whereDate('transaction_date', '>=', $start_date)
                    ->whereDate('transaction_date', '<=', $end_date);
            }

            $period_data = $period_query->select(
                DB::raw("SUM(IF(type = 'sell' AND status = 'final', final_total, 0)) as total_sales"),
                DB::raw("SUM(IF(type = 'sell_return' AND status = 'final', final_total, 0)) as total_sales_return"),
                DB::raw("SUM(IF(type = 'purchase' AND status = 'received', final_total, 0)) as total_purchase"),
                DB::raw("SUM(IF(type = 'purchase_return' AND status = 'received', final_total, 0)) as total_purchase_return"),
                DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
            )->first();

            $sales = $period_data->total_sales ?? 0;
            $sales_return = $period_data->total_sales_return ?? 0;
            $net_sales = $sales - $sales_return;
            $purchase = $period_data->total_purchase ?? 0;
            $purchase_return = $period_data->total_purchase_return ?? 0;
            $discount = $period_data->total_ledger_discount ?? 0;
            $other_dr = 0; // Other debits (can be extended later)
            $other_cr = 0; // Other credits (can be extended later)

            // Get quantities
            $sales_qty = 0;
            $sales_return_qty = 0;
            $purchase_qty = 0;
            $purchase_return_qty = 0;

            if (!empty($start_date) && !empty($end_date)) {
                // Sales quantity
                $sales_qty_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereNull('transaction_sell_lines.parent_sell_line_id')
                    ->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $sales_qty_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $sales_qty_query->whereIn('t.location_id', $permitted_locations);
                }

                $sales_qty = $sales_qty_query->sum(DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned)')) ?? 0;

                // Sales return quantity
                $sales_return_qty_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereNull('transaction_sell_lines.parent_sell_line_id')
                    ->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $sales_return_qty_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $sales_return_qty_query->whereIn('t.location_id', $permitted_locations);
                }

                $sales_return_qty = $sales_return_qty_query->sum('transaction_sell_lines.quantity') ?? 0;

                // Purchase quantity
                $purchase_qty_query = \App\PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'purchase')
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $purchase_qty_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $purchase_qty_query->whereIn('t.location_id', $permitted_locations);
                }

                $purchase_qty = $purchase_qty_query->sum(DB::raw('(purchase_lines.quantity - purchase_lines.quantity_returned)')) ?? 0;

                // Purchase return quantity
                $purchase_return_qty_query = \App\PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'purchase_return')
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $purchase_return_qty_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $purchase_return_qty_query->whereIn('t.location_id', $permitted_locations);
                }

                $purchase_return_qty = $purchase_return_qty_query->sum('purchase_lines.quantity') ?? 0;
            }

            // Get payments in date range
            $receive = 0; // Payments received from customer
            $payment = 0; // Payments made to customer (if they're also suppliers)
            if (!empty($start_date) && !empty($end_date)) {
                // Payments received (for sales)
                $receive_query = \App\TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['sell', 'sell_return'])
                    ->whereDate('transaction_payments.paid_on', '>=', $start_date)
                    ->whereDate('transaction_payments.paid_on', '<=', $end_date);

                if (!empty($location_id)) {
                    $receive_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $receive_query->whereIn('t.location_id', $permitted_locations);
                }

                $receive = $receive_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;

                // Payments made (for purchases - if customer is also a supplier)
                $payment_query = \App\TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['purchase', 'purchase_return'])
                    ->whereDate('transaction_payments.paid_on', '>=', $start_date)
                    ->whereDate('transaction_payments.paid_on', '<=', $end_date);

                if (!empty($location_id)) {
                    $payment_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $payment_query->whereIn('t.location_id', $permitted_locations);
                }

                $payment = $payment_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;
            }

            // Calculate closing balance (Due)
            $due = $opening_balance + $net_sales + $purchase - $purchase_return - $receive + $payment + $other_cr - $other_dr - $discount;

            $customer_name = $customer->name;
            if (!empty($customer->supplier_business_name)) {
                $customer_name = $customer->supplier_business_name;
            }

            // Get customer group if exists
            $customer_group = 'ALL CUSTOMERS';
            if (!empty($customer->customer_group_id)) {
                $customer_group_obj = \App\CustomerGroup::find($customer->customer_group_id);
                if ($customer_group_obj) {
                    $customer_group = $customer_group_obj->name;
                }
            }

            $table_data[] = [
                'customer_group' => $customer_group,
                'customer_name' => $customer_name,
                'opening_balance' => $opening_balance,
                'sales' => $sales,
                'sales_return' => $sales_return,
                'net_sales' => $net_sales,
                'sales_qty' => $sales_qty,
                'sales_return_qty' => $sales_return_qty,
                'purchase_qty' => $purchase_qty,
                'purchase_return_qty' => $purchase_return_qty,
                'purchase' => $purchase,
                'purchase_return' => $purchase_return,
                'receive' => $receive,
                'payment' => $payment,
                'other_cr' => $other_cr,
                'other_dr' => $other_dr,
                'discount' => $discount,
                'due' => $due,
            ];
        }

        // Group by customer group
        $grouped_data = collect($table_data)->groupBy('customer_group');

        $util = new Util();

        return view('businessmanagement::sales_register.customer_wise_statement', compact(
            'table_data',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'locations',
            'customers',
            'util'
        ));
    }

    /**
     * Print Customer Wise Statement Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printCustomerWiseStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get customers list
        $customers_query = Contact::where('business_id', $business_id)
            ->where('type', 'customer');

        if (!empty($customer_id)) {
            $customers_query->where('id', $customer_id);
        }

        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $customers_query->whereHas('transactions', function($q) use ($permitted_locations) {
                $q->whereIn('location_id', $permitted_locations);
            });
        }

        $customers_list = $customers_query->get();

        $table_data = [];
        foreach ($customers_list as $customer) {
            // Calculate opening balance (before start_date)
            $opening_balance_query = Transaction::where('business_id', $business_id)
                ->where('contact_id', $customer->id)
                ->whereIn('type', ['sell', 'sell_return', 'opening_balance', 'ledger_discount']);

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    continue;
                }
                $opening_balance_query->where('location_id', $location_id);
            }

            if ($permitted_locations != 'all') {
                $opening_balance_query->whereIn('location_id', $permitted_locations);
            }

            if (!empty($start_date)) {
                $opening_balance_query->whereDate('transaction_date', '<', $start_date);
            }

            $opening_balance_data = $opening_balance_query->select(
                DB::raw("SUM(IF(type = 'sell' AND status = 'final', final_total, 0)) as total_sales"),
                DB::raw("SUM(IF(type = 'sell_return' AND status = 'final', final_total, 0)) as total_sales_return"),
                DB::raw("SUM(IF(type = 'opening_balance', final_total, 0)) as total_opening_balance"),
                DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
            )->first();

            // Get payments before start_date
            $opening_payments = 0;
            if (!empty($start_date)) {
                $opening_payments_query = \App\TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['sell', 'sell_return', 'opening_balance'])
                    ->whereDate('transaction_payments.paid_on', '<', $start_date);

                if (!empty($location_id)) {
                    $opening_payments_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $opening_payments_query->whereIn('t.location_id', $permitted_locations);
                }

                $opening_payments = $opening_payments_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;
            }

            $opening_balance = ($opening_balance_data->total_sales ?? 0) 
                - ($opening_balance_data->total_sales_return ?? 0)
                + ($opening_balance_data->total_opening_balance ?? 0)
                - ($opening_balance_data->total_ledger_discount ?? 0)
                - $opening_payments;

            // Get transactions in date range
            $period_query = Transaction::where('business_id', $business_id)
                ->where('contact_id', $customer->id)
                ->whereIn('type', ['sell', 'sell_return', 'purchase', 'purchase_return', 'ledger_discount']);

            if (!empty($location_id)) {
                $period_query->where('location_id', $location_id);
            }

            if ($permitted_locations != 'all') {
                $period_query->whereIn('location_id', $permitted_locations);
            }

            if (!empty($start_date) && !empty($end_date)) {
                $period_query->whereDate('transaction_date', '>=', $start_date)
                    ->whereDate('transaction_date', '<=', $end_date);
            }

            $period_data = $period_query->select(
                DB::raw("SUM(IF(type = 'sell' AND status = 'final', final_total, 0)) as total_sales"),
                DB::raw("SUM(IF(type = 'sell_return' AND status = 'final', final_total, 0)) as total_sales_return"),
                DB::raw("SUM(IF(type = 'purchase' AND status = 'received', final_total, 0)) as total_purchase"),
                DB::raw("SUM(IF(type = 'purchase_return' AND status = 'received', final_total, 0)) as total_purchase_return"),
                DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
            )->first();

            $sales = $period_data->total_sales ?? 0;
            $sales_return = $period_data->total_sales_return ?? 0;
            $net_sales = $sales - $sales_return;
            $purchase = $period_data->total_purchase ?? 0;
            $purchase_return = $period_data->total_purchase_return ?? 0;
            $discount = $period_data->total_ledger_discount ?? 0;
            $other_dr = 0;
            $other_cr = 0;

            // Get quantities
            $sales_qty = 0;
            $sales_return_qty = 0;
            $purchase_qty = 0;
            $purchase_return_qty = 0;

            if (!empty($start_date) && !empty($end_date)) {
                $sales_qty_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereNull('transaction_sell_lines.parent_sell_line_id')
                    ->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $sales_qty_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $sales_qty_query->whereIn('t.location_id', $permitted_locations);
                }

                $sales_qty = $sales_qty_query->sum(DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned)')) ?? 0;

                $sales_return_qty_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereNull('transaction_sell_lines.parent_sell_line_id')
                    ->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $sales_return_qty_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $sales_return_qty_query->whereIn('t.location_id', $permitted_locations);
                }

                $sales_return_qty = $sales_return_qty_query->sum('transaction_sell_lines.quantity') ?? 0;

                $purchase_qty_query = \App\PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'purchase')
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $purchase_qty_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $purchase_qty_query->whereIn('t.location_id', $permitted_locations);
                }

                $purchase_qty = $purchase_qty_query->sum(DB::raw('(purchase_lines.quantity - purchase_lines.quantity_returned)')) ?? 0;

                $purchase_return_qty_query = \App\PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'purchase_return')
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $purchase_return_qty_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $purchase_return_qty_query->whereIn('t.location_id', $permitted_locations);
                }

                $purchase_return_qty = $purchase_return_qty_query->sum('purchase_lines.quantity') ?? 0;
            }

            // Get payments in date range
            $receive = 0;
            $payment = 0;
            if (!empty($start_date) && !empty($end_date)) {
                $receive_query = \App\TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['sell', 'sell_return'])
                    ->whereDate('transaction_payments.paid_on', '>=', $start_date)
                    ->whereDate('transaction_payments.paid_on', '<=', $end_date);

                if (!empty($location_id)) {
                    $receive_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $receive_query->whereIn('t.location_id', $permitted_locations);
                }

                $receive = $receive_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;

                $payment_query = \App\TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['purchase', 'purchase_return'])
                    ->whereDate('transaction_payments.paid_on', '>=', $start_date)
                    ->whereDate('transaction_payments.paid_on', '<=', $end_date);

                if (!empty($location_id)) {
                    $payment_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $payment_query->whereIn('t.location_id', $permitted_locations);
                }

                $payment = $payment_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;
            }

            // Calculate closing balance (Due)
            $due = $opening_balance + $net_sales + $purchase - $purchase_return - $receive + $payment + $other_cr - $other_dr - $discount;

            $customer_name = $customer->name;
            if (!empty($customer->supplier_business_name)) {
                $customer_name = $customer->supplier_business_name;
            }

            // Get customer group if exists
            $customer_group = 'ALL CUSTOMERS';
            if (!empty($customer->customer_group_id)) {
                $customer_group_obj = \App\CustomerGroup::find($customer->customer_group_id);
                if ($customer_group_obj) {
                    $customer_group = $customer_group_obj->name;
                }
            }

            $table_data[] = [
                'customer_group' => $customer_group,
                'customer_name' => $customer_name,
                'opening_balance' => $opening_balance,
                'sales' => $sales,
                'sales_return' => $sales_return,
                'net_sales' => $net_sales,
                'sales_qty' => $sales_qty,
                'sales_return_qty' => $sales_return_qty,
                'purchase_qty' => $purchase_qty,
                'purchase_return_qty' => $purchase_return_qty,
                'purchase' => $purchase,
                'purchase_return' => $purchase_return,
                'receive' => $receive,
                'payment' => $payment,
                'other_cr' => $other_cr,
                'other_dr' => $other_dr,
                'discount' => $discount,
                'due' => $due,
            ];
        }

        // Group by customer group
        $grouped_data = collect($table_data)->groupBy('customer_group');

        $util = new Util();

        return view('businessmanagement::sales_register.print_customer_wise_statement', compact(
            'table_data',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'business',
            'logo',
            'util'
        ));
    }

    /**
     * Display Product Group Wise Net Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productGroupWiseNetSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->pluck('name', 'id');
        $brands = Brands::where('business_id', $business_id)->pluck('name', 'id');

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get sell lines with product, brand, and category details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                'p.name as product_name',
                'p.category_id',
                'cat.name as category_name',
                'p.brand_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as unit_price'),
                DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount')
            )
            ->groupBy('p.id', 'p.name', 'p.category_id', 'cat.name', 'p.brand_id', 'b.name', 'product_description');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $query->get();

        // Get return lines
        $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount')
            )
            ->groupBy('p.id', 'b.name');

        // Apply location filter
        if ($permitted_locations != 'all') {
            $return_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $return_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $return_query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $return_query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $return_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $return_lines = $return_query->get()->keyBy('product_id');

        // Combine sales and returns, group by brand
        $brands_data = [];
        foreach ($sales_lines as $sale) {
            $brand_id = $sale->brand_id ?? 0;
            $brand_name = $sale->brand_name ?? 'NO BRAND';
            
            $return = $return_lines->get($sale->product_id);
            $return_qty = $return ? $return->return_qty : 0;
            $return_amount = $return ? $return->return_amount : 0;
            
            $net_qty = $sale->sale_qty - $return_qty;
            $net_amount = $sale->sales_amount - $return_amount;
            
            if (!isset($brands_data[$brand_id])) {
                $brands_data[$brand_id] = [
                    'brand_name' => $brand_name,
                    'products' => [],
                    'total_sale_qty' => 0,
                    'total_sales_amount' => 0,
                    'total_return_qty' => 0,
                    'total_return_amount' => 0,
                    'total_net_qty' => 0,
                    'total_net_amount' => 0,
                ];
            }
            
            $brands_data[$brand_id]['products'][] = [
                'product_name' => $sale->product_description,
                'sale_qty' => $sale->sale_qty,
                'unit_price' => $sale->unit_price,
                'sales_amount' => $sale->sales_amount,
                'return_qty' => $return_qty,
                'return_amount' => $return_amount,
                'net_qty' => $net_qty,
                'net_amount' => $net_amount,
            ];
            
            $brands_data[$brand_id]['total_sale_qty'] += $sale->sale_qty;
            $brands_data[$brand_id]['total_sales_amount'] += $sale->sales_amount;
            $brands_data[$brand_id]['total_return_qty'] += $return_qty;
            $brands_data[$brand_id]['total_return_amount'] += $return_amount;
            $brands_data[$brand_id]['total_net_qty'] += $net_qty;
            $brands_data[$brand_id]['total_net_amount'] += $net_amount;
        }

        // Calculate grand totals
        $grand_total_sale_qty = collect($brands_data)->sum('total_sale_qty');
        $grand_total_sales_amount = collect($brands_data)->sum('total_sales_amount');
        $grand_total_return_qty = collect($brands_data)->sum('total_return_qty');
        $grand_total_return_amount = collect($brands_data)->sum('total_return_amount');
        $grand_total_net_qty = collect($brands_data)->sum('total_net_qty');
        $grand_total_net_amount = collect($brands_data)->sum('total_net_amount');

        $util = new Util();

        return view('businessmanagement::sales_register.product_group_wise_net_sales', compact(
            'brands_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'grand_total_sale_qty',
            'grand_total_sales_amount',
            'grand_total_return_qty',
            'grand_total_return_amount',
            'grand_total_net_qty',
            'grand_total_net_amount',
            'util'
        ));
    }

    /**
     * Print Product Group Wise Net Sales Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductGroupWiseNetSales(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get sell lines with product, brand, and category details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                'p.name as product_name',
                'p.category_id',
                'cat.name as category_name',
                'p.brand_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as unit_price'),
                DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount')
            )
            ->groupBy('p.id', 'p.name', 'p.category_id', 'cat.name', 'p.brand_id', 'b.name', 'product_description');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $query->get();

        // Get return lines
        $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount')
            )
            ->groupBy('p.id', 'b.name');

        // Apply location filter
        if ($permitted_locations != 'all') {
            $return_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $return_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $return_query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $return_query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $return_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $return_lines = $return_query->get()->keyBy('product_id');

        // Combine sales and returns, group by brand
        $brands_data = [];
        foreach ($sales_lines as $sale) {
            $brand_id = $sale->brand_id ?? 0;
            $brand_name = $sale->brand_name ?? 'NO BRAND';
            
            $return = $return_lines->get($sale->product_id);
            $return_qty = $return ? $return->return_qty : 0;
            $return_amount = $return ? $return->return_amount : 0;
            
            $net_qty = $sale->sale_qty - $return_qty;
            $net_amount = $sale->sales_amount - $return_amount;
            
            if (!isset($brands_data[$brand_id])) {
                $brands_data[$brand_id] = [
                    'brand_name' => $brand_name,
                    'products' => [],
                    'total_sale_qty' => 0,
                    'total_sales_amount' => 0,
                    'total_return_qty' => 0,
                    'total_return_amount' => 0,
                    'total_net_qty' => 0,
                    'total_net_amount' => 0,
                ];
            }
            
            $brands_data[$brand_id]['products'][] = [
                'product_name' => $sale->product_description,
                'sale_qty' => $sale->sale_qty,
                'unit_price' => $sale->unit_price,
                'sales_amount' => $sale->sales_amount,
                'return_qty' => $return_qty,
                'return_amount' => $return_amount,
                'net_qty' => $net_qty,
                'net_amount' => $net_amount,
            ];
            
            $brands_data[$brand_id]['total_sale_qty'] += $sale->sale_qty;
            $brands_data[$brand_id]['total_sales_amount'] += $sale->sales_amount;
            $brands_data[$brand_id]['total_return_qty'] += $return_qty;
            $brands_data[$brand_id]['total_return_amount'] += $return_amount;
            $brands_data[$brand_id]['total_net_qty'] += $net_qty;
            $brands_data[$brand_id]['total_net_amount'] += $net_amount;
        }

        // Calculate grand totals
        $grand_total_sale_qty = collect($brands_data)->sum('total_sale_qty');
        $grand_total_sales_amount = collect($brands_data)->sum('total_sales_amount');
        $grand_total_return_qty = collect($brands_data)->sum('total_return_qty');
        $grand_total_return_amount = collect($brands_data)->sum('total_return_amount');
        $grand_total_net_qty = collect($brands_data)->sum('total_net_qty');
        $grand_total_net_amount = collect($brands_data)->sum('total_net_amount');

        $util = new Util();

        return view('businessmanagement::sales_register.print_product_group_wise_net_sales', compact(
            'brands_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'business',
            'logo',
            'grand_total_sale_qty',
            'grand_total_sales_amount',
            'grand_total_return_qty',
            'grand_total_return_amount',
            'grand_total_net_qty',
            'grand_total_net_amount',
            'util'
        ));
    }

    /**
     * Display Product Wise Sales Summary Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productWiseSalesSummary(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');

        // Get sell lines with product and customer details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                'c.name as customer_name',
                'c.supplier_business_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_model"),
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                'transaction_sell_lines.unit_price_before_discount as unit_price',
                DB::raw('COALESCE(transaction_sell_lines.line_discount_amount, 0) as discount'),
                'transaction_sell_lines.unit_price_inc_tax as net_unit_price',
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as amount")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $query->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        $util = new Util();
        
        // Calculate grand totals
        $grand_total_amount = $sales_lines->sum('amount');

        return view('businessmanagement::sales_register.product_wise_sales_summary', compact(
            'sales_lines',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'grand_total_amount',
            'util'
        ));
    }

    /**
     * Print Product Wise Sales Summary Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductWiseSalesSummary(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');

        // Get sell lines with product and customer details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                'c.name as customer_name',
                'c.supplier_business_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_model"),
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                'transaction_sell_lines.unit_price_before_discount as unit_price',
                DB::raw('COALESCE(transaction_sell_lines.line_discount_amount, 0) as discount'),
                'transaction_sell_lines.unit_price_inc_tax as net_unit_price',
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as amount")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $query->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        $util = new Util();
        
        // Calculate grand totals
        $grand_total_amount = $sales_lines->sum('amount');

        return view('businessmanagement::sales_register.print_product_wise_sales_summary', compact(
            'sales_lines',
            'start_date',
            'end_date',
            'location_id',
            'business',
            'logo',
            'grand_total_amount',
            'util'
        ));
    }

    /**
     * Display Date Wise Sales Summary Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWiseSalesSummary(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get transactions with sell lines
        $query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final')
            ->with(['sell_lines' => function($q) {
                $q->whereNull('parent_sell_line_id')
                  ->with(['product', 'variations.product_variation']);
            }, 'contact']);

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', request()->session()->get('user.id'));
        }

        $transactions = $query->orderBy('transaction_date', 'asc')
            ->orderBy('invoice_no', 'asc')
            ->get();

        // Group by date
        $grouped_data = $transactions->groupBy(function($transaction) {
            return $transaction->transaction_date;
        });

        $util = new Util();

        return view('businessmanagement::sales_register.date_wise_sales_summary', compact(
            'transactions',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'locations',
            'customers',
            'util'
        ));
    }

    /**
     * Print Date Wise Sales Summary Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWiseSalesSummary(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get transactions with sell lines
        $query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final')
            ->with(['sell_lines' => function($q) {
                $q->whereNull('parent_sell_line_id')
                  ->with(['product', 'variations.product_variation']);
            }, 'contact']);

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', request()->session()->get('user.id'));
        }

        $transactions = $query->orderBy('transaction_date', 'asc')
            ->orderBy('invoice_no', 'asc')
            ->get();

        // Group by date
        $grouped_data = $transactions->groupBy(function($transaction) {
            return $transaction->transaction_date;
        });

        $util = new Util();

        return view('businessmanagement::sales_register.print_date_wise_sales_summary', compact(
            'transactions',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'business',
            'logo',
            'util'
        ));
    }

    /**
     * Display Product Group Wise Net Sales (MPO) Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productGroupWiseNetSalesMPO(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');

        // Get sell lines grouped by category
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'cat.id as category_id',
                DB::raw("COALESCE(cat.name, 'UNCATEGORIZED') as category_name"),
                DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as sale_unit_price'),
                DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount')
            )
            ->groupBy('cat.id', 'cat.name');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $query->get();

        // Get return lines grouped by category
        $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'cat.id as category_id',
                DB::raw("COALESCE(cat.name, 'UNCATEGORIZED') as category_name"),
                DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as return_unit_price'),
                DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount')
            )
            ->groupBy('cat.id', 'cat.name');

        // Apply location filter
        if ($permitted_locations != 'all') {
            $return_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $return_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $return_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $return_lines = $return_query->get()->keyBy('category_id');

        // Combine sales and returns, group by category
        $categories_data = [];
        foreach ($sales_lines as $sale) {
            $category_id = $sale->category_id ?? 0;
            $category_name = $sale->category_name ?? 'UNCATEGORIZED';
            
            $return = $return_lines->get($category_id);
            $return_qty = $return ? $return->return_qty : 0;
            $return_unit_price = $return ? $return->return_unit_price : 0;
            $return_amount = $return ? $return->return_amount : 0;
            
            $net_qty = $sale->sale_qty - $return_qty;
            $net_amount = $sale->sales_amount - $return_amount;
            
            $categories_data[$category_id] = [
                'category_name' => $category_name,
                'sale_qty' => $sale->sale_qty,
                'sale_unit_price' => $sale->sale_unit_price,
                'sales_amount' => $sale->sales_amount,
                'return_qty' => $return_qty,
                'return_unit_price' => $return_unit_price,
                'return_amount' => $return_amount,
                'net_qty' => $net_qty,
                'net_amount' => $net_amount,
            ];
        }

        // Add categories that only have returns
        foreach ($return_lines as $return) {
            if (!isset($categories_data[$return->category_id])) {
                $category_id = $return->category_id ?? 0;
                $category_name = $return->category_name ?? 'UNCATEGORIZED';
                
                $categories_data[$category_id] = [
                    'category_name' => $category_name,
                    'sale_qty' => 0,
                    'sale_unit_price' => 0,
                    'sales_amount' => 0,
                    'return_qty' => $return->return_qty,
                    'return_unit_price' => $return->return_unit_price,
                    'return_amount' => $return->return_amount,
                    'net_qty' => -$return->return_qty,
                    'net_amount' => -$return->return_amount,
                ];
            }
        }

        // Calculate grand totals
        $grand_total_sale_qty = collect($categories_data)->sum('sale_qty');
        $grand_total_sales_amount = collect($categories_data)->sum('sales_amount');
        $grand_total_return_qty = collect($categories_data)->sum('return_qty');
        $grand_total_return_amount = collect($categories_data)->sum('return_amount');
        $grand_total_net_qty = collect($categories_data)->sum('net_qty');
        $grand_total_net_amount = collect($categories_data)->sum('net_amount');

        // Get total discount from transactions
        $discount_query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final');

        if ($permitted_locations != 'all') {
            $discount_query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $discount_query->where('location_id', $location_id);
        }

        if (!empty($start_date) && !empty($end_date)) {
            $discount_query->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $discount_query->where('created_by', request()->session()->get('user.id'));
        }

        $total_discount = $discount_query->sum('discount_amount') ?? 0;
        $net_after_discount = $grand_total_net_amount - $total_discount;

        $util = new Util();

        return view('businessmanagement::sales_register.product_group_wise_net_sales_mpo', compact(
            'categories_data',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'grand_total_sale_qty',
            'grand_total_sales_amount',
            'grand_total_return_qty',
            'grand_total_return_amount',
            'grand_total_net_qty',
            'grand_total_net_amount',
            'total_discount',
            'net_after_discount',
            'util'
        ));
    }

    /**
     * Print Product Group Wise Net Sales (MPO) Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductGroupWiseNetSalesMPO(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');

        // Get sell lines grouped by category
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'cat.id as category_id',
                DB::raw("COALESCE(cat.name, 'UNCATEGORIZED') as category_name"),
                DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as sale_unit_price'),
                DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount')
            )
            ->groupBy('cat.id', 'cat.name');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $query->get();

        // Get return lines grouped by category
        $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'cat.id as category_id',
                DB::raw("COALESCE(cat.name, 'UNCATEGORIZED') as category_name"),
                DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as return_unit_price'),
                DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount')
            )
            ->groupBy('cat.id', 'cat.name');

        // Apply location filter
        if ($permitted_locations != 'all') {
            $return_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_query->where('t.location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $return_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $return_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $return_lines = $return_query->get()->keyBy('category_id');

        // Combine sales and returns, group by category
        $categories_data = [];
        foreach ($sales_lines as $sale) {
            $category_id = $sale->category_id ?? 0;
            $category_name = $sale->category_name ?? 'UNCATEGORIZED';
            
            $return = $return_lines->get($category_id);
            $return_qty = $return ? $return->return_qty : 0;
            $return_unit_price = $return ? $return->return_unit_price : 0;
            $return_amount = $return ? $return->return_amount : 0;
            
            $net_qty = $sale->sale_qty - $return_qty;
            $net_amount = $sale->sales_amount - $return_amount;
            
            $categories_data[$category_id] = [
                'category_name' => $category_name,
                'sale_qty' => $sale->sale_qty,
                'sale_unit_price' => $sale->sale_unit_price,
                'sales_amount' => $sale->sales_amount,
                'return_qty' => $return_qty,
                'return_unit_price' => $return_unit_price,
                'return_amount' => $return_amount,
                'net_qty' => $net_qty,
                'net_amount' => $net_amount,
            ];
        }

        // Add categories that only have returns
        foreach ($return_lines as $return) {
            if (!isset($categories_data[$return->category_id])) {
                $category_id = $return->category_id ?? 0;
                $category_name = $return->category_name ?? 'UNCATEGORIZED';
                
                $categories_data[$category_id] = [
                    'category_name' => $category_name,
                    'sale_qty' => 0,
                    'sale_unit_price' => 0,
                    'sales_amount' => 0,
                    'return_qty' => $return->return_qty,
                    'return_unit_price' => $return->return_unit_price,
                    'return_amount' => $return->return_amount,
                    'net_qty' => -$return->return_qty,
                    'net_amount' => -$return->return_amount,
                ];
            }
        }

        // Calculate grand totals
        $grand_total_sale_qty = collect($categories_data)->sum('sale_qty');
        $grand_total_sales_amount = collect($categories_data)->sum('sales_amount');
        $grand_total_return_qty = collect($categories_data)->sum('return_qty');
        $grand_total_return_amount = collect($categories_data)->sum('return_amount');
        $grand_total_net_qty = collect($categories_data)->sum('net_qty');
        $grand_total_net_amount = collect($categories_data)->sum('net_amount');

        // Get total discount from transactions
        $discount_query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final');

        if ($permitted_locations != 'all') {
            $discount_query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $discount_query->where('location_id', $location_id);
        }

        if (!empty($start_date) && !empty($end_date)) {
            $discount_query->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $discount_query->where('created_by', request()->session()->get('user.id'));
        }

        $total_discount = $discount_query->sum('discount_amount') ?? 0;
        $net_after_discount = $grand_total_net_amount - $total_discount;

        $util = new Util();

        return view('businessmanagement::sales_register.print_product_group_wise_net_sales_mpo', compact(
            'categories_data',
            'start_date',
            'end_date',
            'location_id',
            'business',
            'logo',
            'grand_total_sale_qty',
            'grand_total_sales_amount',
            'grand_total_return_qty',
            'grand_total_return_amount',
            'grand_total_net_qty',
            'grand_total_net_amount',
            'total_discount',
            'net_after_discount',
            'util'
        ));
    }

    /**
     * Display Customer Wise Sales Summary Report
     *
     * @return \Illuminate\Http\Response
     */
    public function customerWiseSalesSummary(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get transactions with sell lines
        $query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final')
            ->with(['sell_lines' => function($q) {
                $q->whereNull('parent_sell_line_id')
                  ->with(['product', 'variations.product_variation']);
            }, 'contact']);

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', request()->session()->get('user.id'));
        }

        $transactions = $query->orderBy('contact_id', 'asc')
            ->orderBy('transaction_date', 'asc')
            ->orderBy('invoice_no', 'asc')
            ->get();

        // Group by customer
        $grouped_data = $transactions->groupBy('contact_id');

        // Calculate grand totals
        $grand_total_vat = $transactions->sum('tax_amount') ?? 0;
        $grand_total_ait = 0; // AIT (Advance Income Tax) - can be extended if needed
        $grand_total_discount = $transactions->sum('discount_amount') ?? 0;
        $grand_total_service_charge = $transactions->sum(function($t) {
            return ($t->shipping_charges ?? 0) + ($t->additional_expense_value_1 ?? 0) + ($t->additional_expense_value_2 ?? 0);
        });
        $grand_total = $transactions->sum('final_total') ?? 0;

        $util = new Util();

        return view('businessmanagement::sales_register.customer_wise_sales_summary', compact(
            'transactions',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'locations',
            'customers',
            'grand_total_vat',
            'grand_total_ait',
            'grand_total_discount',
            'grand_total_service_charge',
            'grand_total',
            'util'
        ));
    }

    /**
     * Print Customer Wise Sales Summary Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printCustomerWiseSalesSummary(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get transactions with sell lines
        $query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final')
            ->with(['sell_lines' => function($q) {
                $q->whereNull('parent_sell_line_id')
                  ->with(['product', 'variations.product_variation']);
            }, 'contact']);

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('location_id', $location_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', request()->session()->get('user.id'));
        }

        $transactions = $query->orderBy('contact_id', 'asc')
            ->orderBy('transaction_date', 'asc')
            ->orderBy('invoice_no', 'asc')
            ->get();

        // Group by customer
        $grouped_data = $transactions->groupBy('contact_id');

        // Calculate grand totals
        $grand_total_vat = $transactions->sum('tax_amount') ?? 0;
        $grand_total_ait = 0; // AIT (Advance Income Tax) - can be extended if needed
        $grand_total_discount = $transactions->sum('discount_amount') ?? 0;
        $grand_total_service_charge = $transactions->sum(function($t) {
            return ($t->shipping_charges ?? 0) + ($t->additional_expense_value_1 ?? 0) + ($t->additional_expense_value_2 ?? 0);
        });
        $grand_total = $transactions->sum('final_total') ?? 0;

        $util = new Util();

        return view('businessmanagement::sales_register.print_customer_wise_sales_summary', compact(
            'transactions',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'business',
            'logo',
            'grand_total_vat',
            'grand_total_ait',
            'grand_total_discount',
            'grand_total_service_charge',
            'grand_total',
            'util'
        ));
    }

    /**
     * Display Customer Wise Invoice Status Report
     *
     * @return \Illuminate\Http\Response
     */
    public function customerWiseInvoiceStatus(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get transactions (invoices)
        $query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final')
            ->with('contact');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            // If only end_date is provided, get invoices up to that date
            $query->whereDate('transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            // If only start_date is provided, get invoices from that date onwards
            $query->whereDate('transaction_date', '>=', $start_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', request()->session()->get('user.id'));
        }

        $transactions = $query->orderBy('contact_id', 'asc')
            ->orderBy('transaction_date', 'asc')
            ->orderBy('invoice_no', 'asc')
            ->get();

        // Calculate received amounts and outstanding for each transaction
        $report_date = !empty($end_date) ? $end_date : date('Y-m-d');
        $invoices_data = [];
        
        foreach ($transactions as $transaction) {
            // Get received amount from transaction_payments
            // For date range, only count payments up to end_date
            $payment_query = \App\TransactionPayment::where('transaction_id', $transaction->id);
            if (!empty($end_date)) {
                $payment_query->whereDate('paid_on', '<=', $end_date);
            }
            $received_amount = $payment_query->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)')) ?? 0;
            
            $amount = $transaction->final_total ?? 0;
            $outstanding = $amount - $received_amount;
            
            // Skip if fully paid (outstanding <= 0.01)
            if ($outstanding <= 0.01) {
                continue;
            }
            
            // Calculate age in days
            $transaction_date = $transaction->transaction_date;
            $age_days = \Carbon\Carbon::parse($transaction_date)->diffInDays(\Carbon\Carbon::parse($report_date));
            
            $customer_name = 'WALKING CUSTOMER';
            $mobile_no = '';
            
            if (!empty($transaction->contact)) {
                $customer_name = !empty($transaction->contact->supplier_business_name) 
                    ? $transaction->contact->supplier_business_name 
                    : ($transaction->contact->name ?? 'WALKING CUSTOMER');
                $mobile_no = $transaction->contact->mobile ?? '';
            }
            
            $invoices_data[] = [
                'contact_id' => $transaction->contact_id ?? 0,
                'customer_name' => $customer_name,
                'mobile_no' => $mobile_no,
                'transaction_date' => $transaction_date,
                'invoice_no' => $transaction->invoice_no,
                'ref_no' => $transaction->ref_no ?? '',
                'amount' => $amount,
                'received_amount' => $received_amount,
                'outstanding' => $outstanding,
                'age_days' => $age_days,
            ];
        }

        // Group by customer
        $grouped_data = collect($invoices_data)->groupBy('contact_id');

        // Calculate customer-wise totals
        $customers_totals = [];
        foreach ($grouped_data as $contact_id => $customer_invoices) {
            $customers_totals[$contact_id] = [
                'total_amount' => $customer_invoices->sum('amount'),
                'total_received' => $customer_invoices->sum('received_amount'),
                'total_outstanding' => $customer_invoices->sum('outstanding'),
            ];
        }

        // Calculate sub totals (for visible customers)
        $sub_total_amount = collect($invoices_data)->sum('amount');
        $sub_total_received = collect($invoices_data)->sum('received_amount');
        $sub_total_outstanding = collect($invoices_data)->sum('outstanding');

        // Calculate grand totals (all invoices, not just outstanding)
        $grand_total_query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final');

        if ($permitted_locations != 'all') {
            $grand_total_query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $grand_total_query->where('location_id', $location_id);
        }

        // Apply date filter - date range (same as main query)
        if (!empty($start_date) && !empty($end_date)) {
            $grand_total_query->whereBetween(DB::raw('DATE(transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            // If only end_date is provided, get invoices up to that date
            $grand_total_query->whereDate('transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            // If only start_date is provided, get invoices from that date onwards
            $grand_total_query->whereDate('transaction_date', '>=', $start_date);
        }

        if (!empty($customer_id)) {
            $grand_total_query->where('contact_id', $customer_id);
        }

        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $grand_total_query->where('created_by', request()->session()->get('user.id'));
        }

        $all_transactions = $grand_total_query->get();
        
        $grand_total_amount = $all_transactions->sum('final_total') ?? 0;
        $grand_total_received = 0;
        foreach ($all_transactions as $t) {
            // For date range, only count payments up to end_date
            $payment_query = \App\TransactionPayment::where('transaction_id', $t->id);
            if (!empty($end_date)) {
                $payment_query->whereDate('paid_on', '<=', $end_date);
            }
            $grand_total_received += $payment_query->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)')) ?? 0;
        }
        $grand_total_outstanding = $grand_total_amount - $grand_total_received;

        $util = new Util();

        return view('businessmanagement::sales_register.customer_wise_invoice_status', compact(
            'invoices_data',
            'grouped_data',
            'customers_totals',
            'start_date',
            'end_date',
            'report_date',
            'location_id',
            'customer_id',
            'locations',
            'customers',
            'sub_total_amount',
            'sub_total_received',
            'sub_total_outstanding',
            'grand_total_amount',
            'grand_total_received',
            'grand_total_outstanding',
            'util'
        ));
    }

    /**
     * Print Customer Wise Invoice Status Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printCustomerWiseInvoiceStatus(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get transactions (invoices)
        $query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final')
            ->with('contact');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            // If only end_date is provided, get invoices up to that date
            $query->whereDate('transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            // If only start_date is provided, get invoices from that date onwards
            $query->whereDate('transaction_date', '>=', $start_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', request()->session()->get('user.id'));
        }

        $transactions = $query->orderBy('contact_id', 'asc')
            ->orderBy('transaction_date', 'asc')
            ->orderBy('invoice_no', 'asc')
            ->get();

        // Calculate received amounts and outstanding for each transaction
        $report_date = !empty($end_date) ? $end_date : date('Y-m-d');
        $invoices_data = [];
        
        foreach ($transactions as $transaction) {
            // Get received amount from transaction_payments
            // For date range, only count payments up to end_date
            $payment_query = \App\TransactionPayment::where('transaction_id', $transaction->id);
            if (!empty($end_date)) {
                $payment_query->whereDate('paid_on', '<=', $end_date);
            }
            $received_amount = $payment_query->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)')) ?? 0;
            
            $amount = $transaction->final_total ?? 0;
            $outstanding = $amount - $received_amount;
            
            // Skip if fully paid (outstanding <= 0.01)
            if ($outstanding <= 0.01) {
                continue;
            }
            
            // Calculate age in days
            $transaction_date = $transaction->transaction_date;
            $age_days = \Carbon\Carbon::parse($transaction_date)->diffInDays(\Carbon\Carbon::parse($report_date));
            
            $customer_name = 'WALKING CUSTOMER';
            $mobile_no = '';
            
            if (!empty($transaction->contact)) {
                $customer_name = !empty($transaction->contact->supplier_business_name) 
                    ? $transaction->contact->supplier_business_name 
                    : ($transaction->contact->name ?? 'WALKING CUSTOMER');
                $mobile_no = $transaction->contact->mobile ?? '';
            }
            
            $invoices_data[] = [
                'contact_id' => $transaction->contact_id ?? 0,
                'customer_name' => $customer_name,
                'mobile_no' => $mobile_no,
                'transaction_date' => $transaction_date,
                'invoice_no' => $transaction->invoice_no,
                'ref_no' => $transaction->ref_no ?? '',
                'amount' => $amount,
                'received_amount' => $received_amount,
                'outstanding' => $outstanding,
                'age_days' => $age_days,
            ];
        }

        // Group by customer
        $grouped_data = collect($invoices_data)->groupBy('contact_id');

        // Calculate customer-wise totals
        $customers_totals = [];
        foreach ($grouped_data as $contact_id => $customer_invoices) {
            $customers_totals[$contact_id] = [
                'total_amount' => $customer_invoices->sum('amount'),
                'total_received' => $customer_invoices->sum('received_amount'),
                'total_outstanding' => $customer_invoices->sum('outstanding'),
            ];
        }

        // Calculate sub totals
        $sub_total_amount = collect($invoices_data)->sum('amount');
        $sub_total_received = collect($invoices_data)->sum('received_amount');
        $sub_total_outstanding = collect($invoices_data)->sum('outstanding');

        // Calculate grand totals (all invoices)
        $grand_total_query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final');

        if ($permitted_locations != 'all') {
            $grand_total_query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $grand_total_query->where('location_id', $location_id);
        }

        // Apply date filter - date range (same as main query)
        if (!empty($start_date) && !empty($end_date)) {
            $grand_total_query->whereBetween(DB::raw('DATE(transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $grand_total_query->whereDate('transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            // If only start_date is provided, get invoices from that date onwards
            $grand_total_query->whereDate('transaction_date', '>=', $start_date);
        }

        if (!empty($customer_id)) {
            $grand_total_query->where('contact_id', $customer_id);
        }

        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $grand_total_query->where('created_by', request()->session()->get('user.id'));
        }

        $all_transactions = $grand_total_query->get();
        
        $grand_total_amount = $all_transactions->sum('final_total') ?? 0;
        $grand_total_received = 0;
        foreach ($all_transactions as $t) {
            $payment_query = \App\TransactionPayment::where('transaction_id', $t->id);
            if (!empty($end_date)) {
                $payment_query->whereDate('paid_on', '<=', $end_date);
            }
            $grand_total_received += $payment_query->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)')) ?? 0;
        }
        $grand_total_outstanding = $grand_total_amount - $grand_total_received;

        $util = new Util();

        return view('businessmanagement::sales_register.print_customer_wise_invoice_status', compact(
            'invoices_data',
            'grouped_data',
            'customers_totals',
            'start_date',
            'end_date',
            'report_date',
            'location_id',
            'customer_id',
            'business',
            'logo',
            'sub_total_amount',
            'sub_total_received',
            'sub_total_outstanding',
            'grand_total_amount',
            'grand_total_received',
            'grand_total_outstanding',
            'util'
        ));
    }

    /**
     * Display Date Wise Sales Invoice Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWiseSalesInvoice(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');

        // Get transactions (invoices)
        $query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final')
            ->with(['contact', 'location', 'sell_lines' => function($q) {
                $q->whereNull('parent_sell_line_id')
                  ->with(['product', 'variations', 'variations.product_variation']);
            }, 'sales_person']);

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('transaction_date', '>=', $start_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', request()->session()->get('user.id'));
        }

        $transactions = $query->orderBy('transaction_date', 'asc')
            ->orderBy('invoice_no', 'asc')
            ->get();

        // Group by date
        $grouped_data = $transactions->groupBy(function($transaction) {
            return \Carbon\Carbon::parse($transaction->transaction_date)->format('Y-m-d');
        });

        $util = new Util();

        return view('businessmanagement::sales_register.date_wise_sales_invoice', compact(
            'transactions',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'util'
        ));
    }

    /**
     * Print Date Wise Sales Invoice Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWiseSalesInvoice(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');

        // Get transactions (invoices)
        $query = Transaction::where('business_id', $business_id)
            ->where('type', 'sell')
            ->where('status', 'final')
            ->with(['contact', 'location', 'sell_lines' => function($q) {
                $q->whereNull('parent_sell_line_id')
                  ->with(['product', 'variations', 'variations.product_variation']);
            }, 'sales_person']);

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('transaction_date', '>=', $start_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', request()->session()->get('user.id'));
        }

        $transactions = $query->orderBy('transaction_date', 'asc')
            ->orderBy('invoice_no', 'asc')
            ->get();

        // Group by date
        $grouped_data = $transactions->groupBy(function($transaction) {
            return \Carbon\Carbon::parse($transaction->transaction_date)->format('Y-m-d');
        });

        $util = new Util();

        return view('businessmanagement::sales_register.print_date_wise_sales_invoice', compact(
            'transactions',
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'business',
            'logo',
            'util'
        ));
    }

    /**
     * Display Customer Wise Gross Profit Report
     *
     * @return \Illuminate\Http\Response
     */
    public function customerWiseGrossProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get sell lines with purchase line mappings for cost calculation
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('products as p', 'transaction_sell_lines.product_id', '=', 'p.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
            ->leftJoin('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                't.contact_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.enable_stock',
                DB::raw("CASE 
                    WHEN p.type = 'variable' AND v.name IS NOT NULL THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_name"),
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as quantity'),
                'transaction_sell_lines.unit_price_inc_tax',
                DB::raw('((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount'),
                DB::raw('CASE 
                    WHEN p.enable_stock = 0 THEN ((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax)
                    WHEN tspl.id IS NOT NULL THEN ((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * pl.purchase_price_inc_tax)
                    ELSE 0
                END as cost_amount'),
                DB::raw('CASE 
                    WHEN p.enable_stock = 0 THEN 0
                    WHEN tspl.id IS NOT NULL THEN ((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * (transaction_sell_lines.unit_price_inc_tax - pl.purchase_price_inc_tax))
                    ELSE 0
                END as gross_profit')
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('t.contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('c.supplier_business_name', 'asc')
            ->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Group by customer and transaction
        $customers_data = [];
        foreach ($sell_lines as $line) {
            $contact_id = $line->contact_id ?? 0;
            $customer_name = 'WALKING CUSTOMER';
            if (!empty($line->supplier_business_name)) {
                $customer_name = $line->supplier_business_name;
            } elseif (!empty($line->customer_name)) {
                $customer_name = $line->customer_name;
            }

            if (!isset($customers_data[$contact_id])) {
                $customers_data[$contact_id] = [
                    'customer_name' => $customer_name,
                    'transactions' => [],
                    'total_qty' => 0,
                    'total_sales' => 0,
                    'total_cost' => 0,
                    'total_profit' => 0,
                ];
            }

            $transaction_id = $line->transaction_id;
            if (!isset($customers_data[$contact_id]['transactions'][$transaction_id])) {
                $customers_data[$contact_id]['transactions'][$transaction_id] = [
                    'invoice_no' => $line->invoice_no,
                    'transaction_date' => $line->transaction_date,
                    'products' => [],
                    'total_qty' => 0,
                    'total_sales' => 0,
                    'total_cost' => 0,
                    'total_profit' => 0,
                ];
            }

            $profit_percent = $line->sales_amount > 0 ? ($line->gross_profit / $line->sales_amount) * 100 : 0;

            $customers_data[$contact_id]['transactions'][$transaction_id]['products'][] = [
                'product_name' => $line->product_name,
                'quantity' => $line->quantity,
                'sales_amount' => $line->sales_amount,
                'cost_amount' => $line->cost_amount,
                'gross_profit' => $line->gross_profit,
                'profit_percent' => $profit_percent,
            ];

            $customers_data[$contact_id]['transactions'][$transaction_id]['total_qty'] += $line->quantity;
            $customers_data[$contact_id]['transactions'][$transaction_id]['total_sales'] += $line->sales_amount;
            $customers_data[$contact_id]['transactions'][$transaction_id]['total_cost'] += $line->cost_amount;
            $customers_data[$contact_id]['transactions'][$transaction_id]['total_profit'] += $line->gross_profit;

            $customers_data[$contact_id]['total_qty'] += $line->quantity;
            $customers_data[$contact_id]['total_sales'] += $line->sales_amount;
            $customers_data[$contact_id]['total_cost'] += $line->cost_amount;
            $customers_data[$contact_id]['total_profit'] += $line->gross_profit;
        }

        // Calculate customer-wise profit percentages
        foreach ($customers_data as $contact_id => $customer_data) {
            $customers_data[$contact_id]['profit_percent'] = $customer_data['total_sales'] > 0 
                ? ($customer_data['total_profit'] / $customer_data['total_sales']) * 100 
                : 0;
            
            foreach ($customer_data['transactions'] as $trans_id => $trans_data) {
                $customers_data[$contact_id]['transactions'][$trans_id]['profit_percent'] = $trans_data['total_sales'] > 0 
                    ? ($trans_data['total_profit'] / $trans_data['total_sales']) * 100 
                    : 0;
            }
        }

        // Calculate overall totals
        $overall_total_qty = collect($customers_data)->sum('total_qty');
        $overall_total_sales = collect($customers_data)->sum('total_sales');
        $overall_total_cost = collect($customers_data)->sum('total_cost');
        $overall_total_profit = collect($customers_data)->sum('total_profit');
        $overall_profit_percent = $overall_total_sales > 0 ? ($overall_total_profit / $overall_total_sales) * 100 : 0;

        $util = new Util();

        return view('businessmanagement::sales_register.customer_wise_gross_profit', compact(
            'customers_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'locations',
            'customers',
            'overall_total_qty',
            'overall_total_sales',
            'overall_total_cost',
            'overall_total_profit',
            'overall_profit_percent',
            'util'
        ));
    }

    /**
     * Print Customer Wise Gross Profit Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printCustomerWiseGrossProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $location_id = $request->get('location_id');
        $customer_id = $request->get('customer_id');

        // Get sell lines with purchase line mappings for cost calculation
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('products as p', 'transaction_sell_lines.product_id', '=', 'p.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
            ->leftJoin('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                't.contact_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.enable_stock',
                DB::raw("CASE 
                    WHEN p.type = 'variable' AND v.name IS NOT NULL THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_name"),
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as quantity'),
                'transaction_sell_lines.unit_price_inc_tax',
                DB::raw('((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as sales_amount'),
                DB::raw('CASE 
                    WHEN p.enable_stock = 0 THEN ((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax)
                    WHEN tspl.id IS NOT NULL THEN ((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * pl.purchase_price_inc_tax)
                    ELSE 0
                END as cost_amount'),
                DB::raw('CASE 
                    WHEN p.enable_stock = 0 THEN 0
                    WHEN tspl.id IS NOT NULL THEN ((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * (transaction_sell_lines.unit_price_inc_tax - pl.purchase_price_inc_tax))
                    ELSE 0
                END as gross_profit')
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('t.contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('c.supplier_business_name', 'asc')
            ->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Group by customer and transaction
        $customers_data = [];
        foreach ($sell_lines as $line) {
            $contact_id = $line->contact_id ?? 0;
            $customer_name = 'WALKING CUSTOMER';
            if (!empty($line->supplier_business_name)) {
                $customer_name = $line->supplier_business_name;
            } elseif (!empty($line->customer_name)) {
                $customer_name = $line->customer_name;
            }

            if (!isset($customers_data[$contact_id])) {
                $customers_data[$contact_id] = [
                    'customer_name' => $customer_name,
                    'transactions' => [],
                    'total_qty' => 0,
                    'total_sales' => 0,
                    'total_cost' => 0,
                    'total_profit' => 0,
                ];
            }

            $transaction_id = $line->transaction_id;
            if (!isset($customers_data[$contact_id]['transactions'][$transaction_id])) {
                $customers_data[$contact_id]['transactions'][$transaction_id] = [
                    'invoice_no' => $line->invoice_no,
                    'transaction_date' => $line->transaction_date,
                    'products' => [],
                    'total_qty' => 0,
                    'total_sales' => 0,
                    'total_cost' => 0,
                    'total_profit' => 0,
                ];
            }

            $profit_percent = $line->sales_amount > 0 ? ($line->gross_profit / $line->sales_amount) * 100 : 0;

            $customers_data[$contact_id]['transactions'][$transaction_id]['products'][] = [
                'product_name' => $line->product_name,
                'quantity' => $line->quantity,
                'sales_amount' => $line->sales_amount,
                'cost_amount' => $line->cost_amount,
                'gross_profit' => $line->gross_profit,
                'profit_percent' => $profit_percent,
            ];

            $customers_data[$contact_id]['transactions'][$transaction_id]['total_qty'] += $line->quantity;
            $customers_data[$contact_id]['transactions'][$transaction_id]['total_sales'] += $line->sales_amount;
            $customers_data[$contact_id]['transactions'][$transaction_id]['total_cost'] += $line->cost_amount;
            $customers_data[$contact_id]['transactions'][$transaction_id]['total_profit'] += $line->gross_profit;

            $customers_data[$contact_id]['total_qty'] += $line->quantity;
            $customers_data[$contact_id]['total_sales'] += $line->sales_amount;
            $customers_data[$contact_id]['total_cost'] += $line->cost_amount;
            $customers_data[$contact_id]['total_profit'] += $line->gross_profit;
        }

        // Calculate customer-wise profit percentages
        foreach ($customers_data as $contact_id => $customer_data) {
            $customers_data[$contact_id]['profit_percent'] = $customer_data['total_sales'] > 0 
                ? ($customer_data['total_profit'] / $customer_data['total_sales']) * 100 
                : 0;
            
            foreach ($customer_data['transactions'] as $trans_id => $trans_data) {
                $customers_data[$contact_id]['transactions'][$trans_id]['profit_percent'] = $trans_data['total_sales'] > 0 
                    ? ($trans_data['total_profit'] / $trans_data['total_sales']) * 100 
                    : 0;
            }
        }

        // Calculate overall totals
        $overall_total_qty = collect($customers_data)->sum('total_qty');
        $overall_total_sales = collect($customers_data)->sum('total_sales');
        $overall_total_cost = collect($customers_data)->sum('total_cost');
        $overall_total_profit = collect($customers_data)->sum('total_profit');
        $overall_profit_percent = $overall_total_sales > 0 ? ($overall_total_profit / $overall_total_sales) * 100 : 0;

        $util = new Util();

        return view('businessmanagement::sales_register.print_customer_wise_gross_profit', compact(
            'customers_data',
            'start_date',
            'end_date',
            'location_id',
            'customer_id',
            'business',
            'logo',
            'overall_total_qty',
            'overall_total_sales',
            'overall_total_cost',
            'overall_total_profit',
            'overall_profit_percent',
            'util'
        ));
    }

    /**
     * Display Product Wise Net Profit Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productWiseNetProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->pluck('name', 'id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            // Handle both " ~ " and " - " separators
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    // Try alternative format
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        // Log error but continue
                        \Log::error('Date parsing error in productWiseNetProfit: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }
        
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');
        $location_id = $request->get('location_id');

        // Get sales lines with product, brand, and category details
        $sales_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
            ->leftJoin('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'p.brand_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as net_unit_price'),
                DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as net_sales_amount'),
                DB::raw('SUM(CASE 
                    WHEN p.enable_stock = 0 THEN 0
                    WHEN tspl.id IS NOT NULL THEN ((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * pl.purchase_price_inc_tax)
                    ELSE 0
                END) as total_cost'),
                DB::raw('SUM(CASE 
                    WHEN p.enable_stock = 0 THEN ((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax)
                    WHEN tspl.id IS NOT NULL THEN ((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * (transaction_sell_lines.unit_price_inc_tax - pl.purchase_price_inc_tax))
                    ELSE 0
                END) as total_profit')
            )
            ->groupBy('p.id', 'p.category_id', 'cat.name', 'p.brand_id', 'b.name', 'product_description');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $sales_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $sales_query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $sales_query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $sales_query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $sales_query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $sales_query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $sales_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $sales_query->get();

        // Get return lines
        $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
            ->leftJoin('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as return_price'),
                DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount'),
                DB::raw('SUM(CASE 
                    WHEN p.enable_stock = 0 THEN 0
                    WHEN tspl.id IS NOT NULL THEN (tspl.qty_returned * pl.purchase_price_inc_tax)
                    ELSE 0
                END) as return_cost')
            )
            ->groupBy('p.id');

        // Apply location filter
        if ($permitted_locations != 'all') {
            $return_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $return_query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $return_query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $return_query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $return_query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $return_query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $return_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $return_lines = $return_query->get()->keyBy('product_id');

        // Combine sales and returns, group by category -> brand -> product
        $categories_data = [];
        foreach ($sales_lines as $sale) {
            $category_id_key = $sale->category_id ?? 0;
            $category_name = $sale->category_name ?? 'UNCATEGORIZED';
            $brand_id = $sale->brand_id ?? 0;
            $brand_name = $sale->brand_name ?? 'NO BRAND';
            
            $return = $return_lines->get($sale->product_id);
            $return_qty = $return ? $return->return_qty : 0;
            $return_price = $return ? $return->return_price : 0;
            $return_amount = $return ? $return->return_amount : 0;
            $return_cost = $return ? $return->return_cost : 0;
            
            // Net Total Profit = TotalProfit + Return Amount (based on image pattern)
            $net_total_profit = $sale->total_profit + $return_amount;
            
            if (!isset($categories_data[$category_id_key])) {
                $categories_data[$category_id_key] = [
                    'category_name' => $category_name,
                    'brands' => [],
                    'group_total_sale_qty' => 0,
                    'group_total_net_sales_amount' => 0,
                    'group_total_cost' => 0,
                    'group_total_profit' => 0,
                    'group_total_return_qty' => 0,
                    'group_total_return_amount' => 0,
                    'group_total_return_cost' => 0,
                    'group_total_net_profit' => 0,
                ];
            }
            
            if (!isset($categories_data[$category_id_key]['brands'][$brand_id])) {
                $categories_data[$category_id_key]['brands'][$brand_id] = [
                    'brand_name' => $brand_name,
                    'products' => [],
                    'brand_total_sale_qty' => 0,
                    'brand_total_net_sales_amount' => 0,
                    'brand_total_cost' => 0,
                    'brand_total_profit' => 0,
                    'brand_total_return_qty' => 0,
                    'brand_total_return_amount' => 0,
                    'brand_total_return_cost' => 0,
                    'brand_total_net_profit' => 0,
                ];
            }
            
            $categories_data[$category_id_key]['brands'][$brand_id]['products'][] = [
                'product_description' => $sale->product_description,
                'sale_qty' => $sale->sale_qty,
                'net_unit_price' => $sale->net_unit_price,
                'net_sales_amount' => $sale->net_sales_amount,
                'total_cost' => $sale->total_cost,
                'total_profit' => $sale->total_profit,
                'return_qty' => $return_qty,
                'return_price' => $return_price,
                'return_amount' => $return_amount,
                'return_cost' => $return_cost,
                'net_total_profit' => $net_total_profit,
            ];
            
            // Brand totals
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_sale_qty'] += $sale->sale_qty;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_net_sales_amount'] += $sale->net_sales_amount;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_cost'] += $sale->total_cost;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_profit'] += $sale->total_profit;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_return_qty'] += $return_qty;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_return_amount'] += $return_amount;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_return_cost'] += $return_cost;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_net_profit'] += $net_total_profit;
            
            // Group totals
            $categories_data[$category_id_key]['group_total_sale_qty'] += $sale->sale_qty;
            $categories_data[$category_id_key]['group_total_net_sales_amount'] += $sale->net_sales_amount;
            $categories_data[$category_id_key]['group_total_cost'] += $sale->total_cost;
            $categories_data[$category_id_key]['group_total_profit'] += $sale->total_profit;
            $categories_data[$category_id_key]['group_total_return_qty'] += $return_qty;
            $categories_data[$category_id_key]['group_total_return_amount'] += $return_amount;
            $categories_data[$category_id_key]['group_total_return_cost'] += $return_cost;
            $categories_data[$category_id_key]['group_total_net_profit'] += $net_total_profit;
        }

        // Calculate grand totals
        $grand_total_sale_qty = collect($categories_data)->sum('group_total_sale_qty');
        $grand_total_net_sales_amount = collect($categories_data)->sum('group_total_net_sales_amount');
        $grand_total_cost = collect($categories_data)->sum('group_total_cost');
        $grand_total_profit = collect($categories_data)->sum('group_total_profit');
        $grand_total_return_qty = collect($categories_data)->sum('group_total_return_qty');
        $grand_total_return_amount = collect($categories_data)->sum('group_total_return_amount');
        $grand_total_return_cost = collect($categories_data)->sum('group_total_return_cost');
        $grand_total_net_profit = collect($categories_data)->sum('group_total_net_profit');

        $util = new Util();

        return view('businessmanagement::sales_register.product_wise_net_profit', compact(
            'categories_data',
            'start_date',
            'end_date',
            'category_id',
            'product_id',
            'location_id',
            'categories',
            'products',
            'locations',
            'grand_total_sale_qty',
            'grand_total_net_sales_amount',
            'grand_total_cost',
            'grand_total_profit',
            'grand_total_return_qty',
            'grand_total_return_amount',
            'grand_total_return_cost',
            'grand_total_net_profit',
            'util'
        ));
    }

    /**
     * Print Product Wise Net Profit Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductWiseNetProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            // Handle both " ~ " and " - " separators
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    // Try alternative format
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        // Log error but continue
                        \Log::error('Date parsing error in printProductWiseNetProfit: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }
        
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');
        $location_id = $request->get('location_id');
        $location_id = $request->get('location_id');

        // Get sales lines with product, brand, and category details
        $sales_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
            ->leftJoin('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'p.brand_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw('SUM(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as sale_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as net_unit_price'),
                DB::raw('SUM((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as net_sales_amount'),
                DB::raw('SUM(CASE 
                    WHEN p.enable_stock = 0 THEN 0
                    WHEN tspl.id IS NOT NULL THEN ((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * pl.purchase_price_inc_tax)
                    ELSE 0
                END) as total_cost'),
                DB::raw('SUM(CASE 
                    WHEN p.enable_stock = 0 THEN ((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax)
                    WHEN tspl.id IS NOT NULL THEN ((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * (transaction_sell_lines.unit_price_inc_tax - pl.purchase_price_inc_tax))
                    ELSE 0
                END) as total_profit')
            )
            ->groupBy('p.id', 'p.category_id', 'cat.name', 'p.brand_id', 'b.name', 'product_description');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $sales_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $sales_query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $sales_query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $sales_query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $sales_query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $sales_query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $sales_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sales_lines = $sales_query->get();

        // Get return lines
        $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
            ->leftJoin('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                'p.id as product_id',
                DB::raw('SUM(transaction_sell_lines.quantity) as return_qty'),
                DB::raw('AVG(transaction_sell_lines.unit_price_inc_tax) as return_price'),
                DB::raw('SUM(transaction_sell_lines.quantity * transaction_sell_lines.unit_price_inc_tax) as return_amount'),
                DB::raw('SUM(CASE 
                    WHEN p.enable_stock = 0 THEN 0
                    WHEN tspl.id IS NOT NULL THEN (tspl.qty_returned * pl.purchase_price_inc_tax)
                    ELSE 0
                END) as return_cost')
            )
            ->groupBy('p.id');

        // Apply location filter
        if ($permitted_locations != 'all') {
            $return_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $return_query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $return_query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $return_query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $return_query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $return_query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $return_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $return_lines = $return_query->get()->keyBy('product_id');

        // Combine sales and returns, group by category -> brand -> product
        $categories_data = [];
        foreach ($sales_lines as $sale) {
            $category_id_key = $sale->category_id ?? 0;
            $category_name = $sale->category_name ?? 'UNCATEGORIZED';
            $brand_id = $sale->brand_id ?? 0;
            $brand_name = $sale->brand_name ?? 'NO BRAND';
            
            $return = $return_lines->get($sale->product_id);
            $return_qty = $return ? $return->return_qty : 0;
            $return_price = $return ? $return->return_price : 0;
            $return_amount = $return ? $return->return_amount : 0;
            $return_cost = $return ? $return->return_cost : 0;
            
            // Net Total Profit = TotalProfit + Return Amount (based on image pattern)
            $net_total_profit = $sale->total_profit + $return_amount;
            
            if (!isset($categories_data[$category_id_key])) {
                $categories_data[$category_id_key] = [
                    'category_name' => $category_name,
                    'brands' => [],
                    'group_total_sale_qty' => 0,
                    'group_total_net_sales_amount' => 0,
                    'group_total_cost' => 0,
                    'group_total_profit' => 0,
                    'group_total_return_qty' => 0,
                    'group_total_return_amount' => 0,
                    'group_total_return_cost' => 0,
                    'group_total_net_profit' => 0,
                ];
            }
            
            if (!isset($categories_data[$category_id_key]['brands'][$brand_id])) {
                $categories_data[$category_id_key]['brands'][$brand_id] = [
                    'brand_name' => $brand_name,
                    'products' => [],
                    'brand_total_sale_qty' => 0,
                    'brand_total_net_sales_amount' => 0,
                    'brand_total_cost' => 0,
                    'brand_total_profit' => 0,
                    'brand_total_return_qty' => 0,
                    'brand_total_return_amount' => 0,
                    'brand_total_return_cost' => 0,
                    'brand_total_net_profit' => 0,
                ];
            }
            
            $categories_data[$category_id_key]['brands'][$brand_id]['products'][] = [
                'product_description' => $sale->product_description,
                'sale_qty' => $sale->sale_qty,
                'net_unit_price' => $sale->net_unit_price,
                'net_sales_amount' => $sale->net_sales_amount,
                'total_cost' => $sale->total_cost,
                'total_profit' => $sale->total_profit,
                'return_qty' => $return_qty,
                'return_price' => $return_price,
                'return_amount' => $return_amount,
                'return_cost' => $return_cost,
                'net_total_profit' => $net_total_profit,
            ];
            
            // Brand totals
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_sale_qty'] += $sale->sale_qty;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_net_sales_amount'] += $sale->net_sales_amount;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_cost'] += $sale->total_cost;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_profit'] += $sale->total_profit;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_return_qty'] += $return_qty;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_return_amount'] += $return_amount;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_return_cost'] += $return_cost;
            $categories_data[$category_id_key]['brands'][$brand_id]['brand_total_net_profit'] += $net_total_profit;
            
            // Group totals
            $categories_data[$category_id_key]['group_total_sale_qty'] += $sale->sale_qty;
            $categories_data[$category_id_key]['group_total_net_sales_amount'] += $sale->net_sales_amount;
            $categories_data[$category_id_key]['group_total_cost'] += $sale->total_cost;
            $categories_data[$category_id_key]['group_total_profit'] += $sale->total_profit;
            $categories_data[$category_id_key]['group_total_return_qty'] += $return_qty;
            $categories_data[$category_id_key]['group_total_return_amount'] += $return_amount;
            $categories_data[$category_id_key]['group_total_return_cost'] += $return_cost;
            $categories_data[$category_id_key]['group_total_net_profit'] += $net_total_profit;
        }

        // Calculate grand totals
        $grand_total_sale_qty = collect($categories_data)->sum('group_total_sale_qty');
        $grand_total_net_sales_amount = collect($categories_data)->sum('group_total_net_sales_amount');
        $grand_total_cost = collect($categories_data)->sum('group_total_cost');
        $grand_total_profit = collect($categories_data)->sum('group_total_profit');
        $grand_total_return_qty = collect($categories_data)->sum('group_total_return_qty');
        $grand_total_return_amount = collect($categories_data)->sum('group_total_return_amount');
        $grand_total_return_cost = collect($categories_data)->sum('group_total_return_cost');
        $grand_total_net_profit = collect($categories_data)->sum('group_total_net_profit');

        $util = new Util();

        return view('businessmanagement::sales_register.print_product_wise_net_profit', compact(
            'categories_data',
            'start_date',
            'end_date',
            'category_id',
            'product_id',
            'business',
            'logo',
            'grand_total_sale_qty',
            'grand_total_net_sales_amount',
            'grand_total_cost',
            'grand_total_profit',
            'grand_total_return_qty',
            'grand_total_return_amount',
            'grand_total_return_cost',
            'grand_total_net_profit',
            'util'
        ));
    }

    /**
     * Display Customer Wise Sales with Serial Report
     *
     * @return \Illuminate\Http\Response
     */
    public function customerWiseSalesWithSerial(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $customers = Contact::customersDropdown($business_id, true);

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            // Handle both " ~ " and " - " separators
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in customerWiseSalesWithSerial: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $customer_id = $request->get('customer_id');

        // Get sell lines with customer, product, and serial number details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                't.contact_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'transaction_sell_lines.variation_id',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                'transaction_sell_lines.unit_price_before_discount as unit_price',
                DB::raw('COALESCE(transaction_sell_lines.line_discount_amount, 0) as discount'),
                'transaction_sell_lines.unit_price_inc_tax as net_unit_price',
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as amount"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('t.contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('c.supplier_business_name', 'asc')
            ->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Get serial numbers for each sell line
        $serial_numbers = [];
        if ($sell_lines->count() > 0) {
            $transaction_ids = $sell_lines->pluck('transaction_id')->unique()->toArray();
            $serial_data = DB::table('serial_numbers')
                ->whereIn('transaction_id', $transaction_ids)
                ->where('stock_status', 'out')
                ->select('transaction_id', 'product_id', 'variation_id', 'serial_number')
                ->get();
            
            foreach ($serial_data as $serial) {
                $key = $serial->transaction_id . '_' . $serial->product_id . '_' . ($serial->variation_id ?? 0);
                if (!isset($serial_numbers[$key])) {
                    $serial_numbers[$key] = [];
                }
                $serial_numbers[$key][] = $serial->serial_number;
            }
        }

        // Group by customer -> transaction -> product
        $customers_data = [];
        foreach ($sell_lines as $line) {
            $contact_id = $line->contact_id ?? 0;
            $customer_name = 'WALKING CUSTOMER';
            if (!empty($line->supplier_business_name)) {
                $customer_name = $line->supplier_business_name;
            } elseif (!empty($line->customer_name)) {
                $customer_name = $line->customer_name;
            }

            if (!isset($customers_data[$contact_id])) {
                $customers_data[$contact_id] = [
                    'customer_name' => $customer_name,
                    'transactions' => [],
                    'customer_total_qty' => 0,
                    'customer_total_amount' => 0,
                ];
            }

            $transaction_id = $line->transaction_id;
            if (!isset($customers_data[$contact_id]['transactions'][$transaction_id])) {
                $customers_data[$contact_id]['transactions'][$transaction_id] = [
                    'invoice_no' => $line->invoice_no,
                    'transaction_date' => $line->transaction_date,
                    'products' => [],
                    'invoice_total_qty' => 0,
                    'invoice_total_amount' => 0,
                ];
            }

            // Get serial numbers for this product
            $variation_id = $line->variation_id ?? 0;
            $serial_key = $line->transaction_id . '_' . $line->product_id . '_' . $variation_id;
            $product_serials = $serial_numbers[$serial_key] ?? [];
            $product_description = $line->product_description;
            if (!empty($product_serials)) {
                $product_description .= ' ' . implode(' ', $product_serials);
            }

            $customers_data[$contact_id]['transactions'][$transaction_id]['products'][] = [
                'product_description' => $product_description,
                'qty' => $line->qty,
                'unit_price' => $line->unit_price,
                'discount' => $line->discount,
                'net_unit_price' => $line->net_unit_price,
                'amount' => $line->amount,
            ];

            $customers_data[$contact_id]['transactions'][$transaction_id]['invoice_total_qty'] += $line->qty;
            $customers_data[$contact_id]['transactions'][$transaction_id]['invoice_total_amount'] += $line->amount;

            $customers_data[$contact_id]['customer_total_qty'] += $line->qty;
            $customers_data[$contact_id]['customer_total_amount'] += $line->amount;
        }

        // Calculate grand totals
        $grand_total_qty = collect($customers_data)->sum('customer_total_qty');
        $grand_total_amount = collect($customers_data)->sum('customer_total_amount');

        $util = new Util();

        return view('businessmanagement::sales_register.customer_wise_sales_with_serial', compact(
            'customers_data',
            'start_date',
            'end_date',
            'customer_id',
            'customers',
            'grand_total_qty',
            'grand_total_amount',
            'util'
        ));
    }

    /**
     * Print Customer Wise Sales with Serial Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printCustomerWiseSalesWithSerial(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            // Handle both " ~ " and " - " separators
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in printCustomerWiseSalesWithSerial: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $customer_id = $request->get('customer_id');

        // Get sell lines with customer, product, and serial number details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                't.contact_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'transaction_sell_lines.variation_id',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                'transaction_sell_lines.unit_price_before_discount as unit_price',
                DB::raw('COALESCE(transaction_sell_lines.line_discount_amount, 0) as discount'),
                'transaction_sell_lines.unit_price_inc_tax as net_unit_price',
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as amount"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply customer filter
        if (!empty($customer_id)) {
            $query->where('t.contact_id', $customer_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('c.supplier_business_name', 'asc')
            ->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Get serial numbers for each sell line
        $serial_numbers = [];
        if ($sell_lines->count() > 0) {
            $transaction_ids = $sell_lines->pluck('transaction_id')->unique()->toArray();
            $serial_data = DB::table('serial_numbers')
                ->whereIn('transaction_id', $transaction_ids)
                ->where('stock_status', 'out')
                ->select('transaction_id', 'product_id', 'variation_id', 'serial_number')
                ->get();
            
            foreach ($serial_data as $serial) {
                $key = $serial->transaction_id . '_' . $serial->product_id . '_' . ($serial->variation_id ?? 0);
                if (!isset($serial_numbers[$key])) {
                    $serial_numbers[$key] = [];
                }
                $serial_numbers[$key][] = $serial->serial_number;
            }
        }

        // Group by customer -> transaction -> product
        $customers_data = [];
        foreach ($sell_lines as $line) {
            $contact_id = $line->contact_id ?? 0;
            $customer_name = 'WALKING CUSTOMER';
            if (!empty($line->supplier_business_name)) {
                $customer_name = $line->supplier_business_name;
            } elseif (!empty($line->customer_name)) {
                $customer_name = $line->customer_name;
            }

            if (!isset($customers_data[$contact_id])) {
                $customers_data[$contact_id] = [
                    'customer_name' => $customer_name,
                    'transactions' => [],
                    'customer_total_qty' => 0,
                    'customer_total_amount' => 0,
                ];
            }

            $transaction_id = $line->transaction_id;
            if (!isset($customers_data[$contact_id]['transactions'][$transaction_id])) {
                $customers_data[$contact_id]['transactions'][$transaction_id] = [
                    'invoice_no' => $line->invoice_no,
                    'transaction_date' => $line->transaction_date,
                    'products' => [],
                    'invoice_total_qty' => 0,
                    'invoice_total_amount' => 0,
                ];
            }

            // Get serial numbers for this product
            $variation_id = $line->variation_id ?? 0;
            $serial_key = $line->transaction_id . '_' . $line->product_id . '_' . $variation_id;
            $product_serials = $serial_numbers[$serial_key] ?? [];
            $product_description = $line->product_description;
            if (!empty($product_serials)) {
                $product_description .= ' ' . implode(' ', $product_serials);
            }

            $customers_data[$contact_id]['transactions'][$transaction_id]['products'][] = [
                'product_description' => $product_description,
                'qty' => $line->qty,
                'unit_price' => $line->unit_price,
                'discount' => $line->discount,
                'net_unit_price' => $line->net_unit_price,
                'amount' => $line->amount,
            ];

            $customers_data[$contact_id]['transactions'][$transaction_id]['invoice_total_qty'] += $line->qty;
            $customers_data[$contact_id]['transactions'][$transaction_id]['invoice_total_amount'] += $line->amount;

            $customers_data[$contact_id]['customer_total_qty'] += $line->qty;
            $customers_data[$contact_id]['customer_total_amount'] += $line->amount;
        }

        // Calculate grand totals
        $grand_total_qty = collect($customers_data)->sum('customer_total_qty');
        $grand_total_amount = collect($customers_data)->sum('customer_total_amount');

        $util = new Util();

        return view('businessmanagement::sales_register.print_customer_wise_sales_with_serial', compact(
            'customers_data',
            'start_date',
            'end_date',
            'customer_id',
            'business',
            'logo',
            'grand_total_qty',
            'grand_total_amount',
            'util'
        ));
    }

    /**
     * Display Product Wise Sales with Serial Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productWiseSalesWithSerial(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->pluck('name', 'id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in productWiseSalesWithSerial: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');
        $location_id = $request->get('location_id');

        // Get sell lines with product, customer, and serial number details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                't.ref_no',
                't.contact_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.name as product_name',
                'transaction_sell_lines.variation_id',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                'transaction_sell_lines.unit_price_before_discount as unit_price',
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as amount"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('p.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Get serial numbers for each sell line
        $serial_numbers = [];
        if ($sell_lines->count() > 0) {
            $transaction_ids = $sell_lines->pluck('transaction_id')->unique()->toArray();
            $serial_data = DB::table('serial_numbers')
                ->whereIn('transaction_id', $transaction_ids)
                ->where('stock_status', 'out')
                ->select('transaction_id', 'product_id', 'variation_id', 'serial_number')
                ->get();
            
            foreach ($serial_data as $serial) {
                $key = $serial->transaction_id . '_' . $serial->product_id . '_' . ($serial->variation_id ?? 0);
                if (!isset($serial_numbers[$key])) {
                    $serial_numbers[$key] = [];
                }
                $serial_numbers[$key][] = $serial->serial_number;
            }
        }

        // Group by product
        $products_data = [];
        foreach ($sell_lines as $line) {
            $product_id_key = $line->product_id;
            $product_name = $line->product_name ?? 'N/A';

            if (!isset($products_data[$product_id_key])) {
                $products_data[$product_id_key] = [
                    'product_name' => $product_name,
                    'lines' => [],
                    'product_total_qty' => 0,
                    'product_total_amount' => 0,
                ];
            }

            // Get serial numbers for this product
            $variation_id = $line->variation_id ?? 0;
            $serial_key = $line->transaction_id . '_' . $line->product_id . '_' . $variation_id;
            $product_serials = $serial_numbers[$serial_key] ?? [];
            $product_description = $line->product_description;
            if (!empty($product_serials)) {
                $product_description .= ' ' . implode(' ', $product_serials);
            }

            $customer_name = 'WALKING CUSTOMER';
            if (!empty($line->supplier_business_name)) {
                $customer_name = $line->supplier_business_name;
            } elseif (!empty($line->customer_name)) {
                $customer_name = $line->customer_name;
            }

            $products_data[$product_id_key]['lines'][] = [
                'transaction_date' => $line->transaction_date,
                'invoice_no' => $line->invoice_no,
                'customer_name' => $customer_name,
                'ref_no' => $line->ref_no ?? '',
                'qty' => $line->qty,
                'unit_price' => $line->unit_price,
                'amount' => $line->amount,
                'product_description' => $product_description,
            ];

            $products_data[$product_id_key]['product_total_qty'] += $line->qty;
            $products_data[$product_id_key]['product_total_amount'] += $line->amount;
        }

        // Calculate grand totals
        $grand_total_qty = collect($products_data)->sum('product_total_qty');
        $grand_total_amount = collect($products_data)->sum('product_total_amount');

        $util = new Util();

        return view('businessmanagement::sales_register.product_wise_sales_with_serial', compact(
            'products_data',
            'start_date',
            'end_date',
            'category_id',
            'product_id',
            'location_id',
            'categories',
            'products',
            'locations',
            'grand_total_qty',
            'grand_total_amount',
            'util'
        ));
    }

    /**
     * Print Product Wise Sales with Serial Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductWiseSalesWithSerial(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in printProductWiseSalesWithSerial: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');
        $location_id = $request->get('location_id');

        // Get sell lines with product, customer, and serial number details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                't.ref_no',
                't.contact_id',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.name as product_name',
                'transaction_sell_lines.variation_id',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                'transaction_sell_lines.unit_price_before_discount as unit_price',
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as amount"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $query->where('p.id', $product_id);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('p.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Get serial numbers for each sell line
        $serial_numbers = [];
        if ($sell_lines->count() > 0) {
            $transaction_ids = $sell_lines->pluck('transaction_id')->unique()->toArray();
            $serial_query = DB::table('serial_numbers')
                ->whereIn('transaction_id', $transaction_ids)
                ->where('stock_status', 'out');
            
            // Apply location filter to serial numbers if specified
            if (!empty($location_id)) {
                $serial_query->where('location_id', $location_id);
            } elseif ($permitted_locations != 'all') {
                $serial_query->whereIn('location_id', $permitted_locations);
            }
            
            $serial_data = $serial_query->select('transaction_id', 'product_id', 'variation_id', 'serial_number')
                ->get();
            
            foreach ($serial_data as $serial) {
                $key = $serial->transaction_id . '_' . $serial->product_id . '_' . ($serial->variation_id ?? 0);
                if (!isset($serial_numbers[$key])) {
                    $serial_numbers[$key] = [];
                }
                $serial_numbers[$key][] = $serial->serial_number;
            }
        }

        // Group by product
        $products_data = [];
        foreach ($sell_lines as $line) {
            $product_id_key = $line->product_id;
            $product_name = $line->product_name ?? 'N/A';

            if (!isset($products_data[$product_id_key])) {
                $products_data[$product_id_key] = [
                    'product_name' => $product_name,
                    'lines' => [],
                    'product_total_qty' => 0,
                    'product_total_amount' => 0,
                ];
            }

            // Get serial numbers for this product
            $variation_id = $line->variation_id ?? 0;
            $serial_key = $line->transaction_id . '_' . $line->product_id . '_' . $variation_id;
            $product_serials = $serial_numbers[$serial_key] ?? [];
            $product_description = $line->product_description;
            if (!empty($product_serials)) {
                $product_description .= ' ' . implode(' ', $product_serials);
            }

            $customer_name = 'WALKING CUSTOMER';
            if (!empty($line->supplier_business_name)) {
                $customer_name = $line->supplier_business_name;
            } elseif (!empty($line->customer_name)) {
                $customer_name = $line->customer_name;
            }

            $products_data[$product_id_key]['lines'][] = [
                'transaction_date' => $line->transaction_date,
                'invoice_no' => $line->invoice_no,
                'customer_name' => $customer_name,
                'ref_no' => $line->ref_no ?? '',
                'qty' => $line->qty,
                'unit_price' => $line->unit_price,
                'amount' => $line->amount,
                'product_description' => $product_description,
            ];

            $products_data[$product_id_key]['product_total_qty'] += $line->qty;
            $products_data[$product_id_key]['product_total_amount'] += $line->amount;
        }

        // Calculate grand totals
        $grand_total_qty = collect($products_data)->sum('product_total_qty');
        $grand_total_amount = collect($products_data)->sum('product_total_amount');

        $util = new Util();

        return view('businessmanagement::sales_register.print_product_wise_sales_with_serial', compact(
            'products_data',
            'start_date',
            'end_date',
            'category_id',
            'product_id',
            'location_id',
            'business',
            'logo',
            'grand_total_qty',
            'grand_total_amount',
            'util'
        ));
    }

    /**
     * Display Daily Sales Details Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dailySalesDetails(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in dailySalesDetails: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');

        // Get sell lines with product, customer details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                't.contact_id',
                't.final_total as transaction_total',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                'transaction_sell_lines.unit_price_before_discount as rate',
                'transaction_sell_lines.unit_price_inc_tax as unit_price_inc_tax',
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as line_total"),
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('c.supplier_business_name', 'asc')
            ->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group by customer and calculate balances
        $customers_data = [];
        $contact_ids = $sell_lines->pluck('contact_id')->unique()->toArray();

        foreach ($contact_ids as $contact_id) {
            // Calculate previous balance (before start_date)
            $prev_balance = 0;
            if (!empty($start_date)) {
                $opening_balance_query = Transaction::where('business_id', $business_id)
                    ->where('contact_id', $contact_id)
                    ->whereIn('type', ['sell', 'sell_return', 'opening_balance', 'ledger_discount']);

                if ($permitted_locations != 'all') {
                    $opening_balance_query->whereIn('location_id', $permitted_locations);
                }

                if (!empty($location_id)) {
                    $opening_balance_query->where('location_id', $location_id);
                }

                $opening_balance_query->whereDate('transaction_date', '<', $start_date);

                $opening_balance_data = $opening_balance_query->select(
                    DB::raw("SUM(IF(type = 'sell' AND status = 'final', final_total, 0)) as total_sales"),
                    DB::raw("SUM(IF(type = 'sell_return' AND status = 'final', final_total, 0)) as total_sales_return"),
                    DB::raw("SUM(IF(type = 'opening_balance', final_total, 0)) as total_opening_balance"),
                    DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
                )->first();

                // Get payments before start_date
                $opening_payments_query = \App\TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $contact_id)
                    ->whereIn('t.type', ['sell', 'sell_return', 'opening_balance'])
                    ->whereDate('transaction_payments.paid_on', '<', $start_date);

                if ($permitted_locations != 'all') {
                    $opening_payments_query->whereIn('t.location_id', $permitted_locations);
                }

                if (!empty($location_id)) {
                    $opening_payments_query->where('t.location_id', $location_id);
                }

                $opening_payments = $opening_payments_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;

                $prev_balance = ($opening_balance_data->total_sales ?? 0) 
                    - ($opening_balance_data->total_sales_return ?? 0)
                    + ($opening_balance_data->total_opening_balance ?? 0)
                    - ($opening_balance_data->total_ledger_discount ?? 0)
                    - $opening_payments;
            }

            // Get customer lines
            $customer_lines = $sell_lines->where('contact_id', $contact_id);
            $customer = $customer_lines->first();
            $customer_name = 'WALKING CUSTOMER';
            if (!empty($customer->supplier_business_name)) {
                $customer_name = $customer->supplier_business_name;
            } elseif (!empty($customer->customer_name)) {
                $customer_name = $customer->customer_name;
            }

            // Calculate totals for date range
            $customer_total_qty = $customer_lines->sum('qty');
            $customer_total_amount = $customer_lines->sum('line_total');
            $customer_service_charges = 0; // Service charges would need to be stored separately

            // Get payments received in date range
            $transaction_ids = $customer_lines->pluck('transaction_id')->unique()->toArray();
            $received_with_cheque = 0;
            if (!empty($transaction_ids)) {
                $payments_query = \App\TransactionPayment::whereIn('transaction_id', $transaction_ids);
                if (!empty($start_date) && !empty($end_date)) {
                    $payments_query->whereBetween(DB::raw('DATE(paid_on)'), [$start_date, $end_date]);
                } elseif (!empty($end_date)) {
                    $payments_query->whereDate('paid_on', '<=', $end_date);
                } elseif (!empty($start_date)) {
                    $payments_query->whereDate('paid_on', '>=', $start_date);
                }
                $received_with_cheque = $payments_query->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)')) ?? 0;
            }

            // Calculate current balance
            $current_balance = $prev_balance + $customer_total_amount - $received_with_cheque;

            $customers_data[$contact_id] = [
                'customer_name' => $customer_name,
                'lines' => $customer_lines,
                'customer_total_qty' => $customer_total_qty,
                'customer_total_amount' => $customer_total_amount,
                'customer_service_charges' => $customer_service_charges,
                'received_with_cheque' => $received_with_cheque,
                'prev_balance' => $prev_balance,
                'current_balance' => $current_balance,
            ];
        }

        // Calculate grand totals
        $grand_total_qty = collect($customers_data)->sum('customer_total_qty');
        $grand_total_amount = collect($customers_data)->sum('customer_total_amount');
        $grand_service_charges = collect($customers_data)->sum('customer_service_charges');
        $grand_received = collect($customers_data)->sum('received_with_cheque');

        $util = new Util();

        return view('businessmanagement::sales_register.daily_sales_details', compact(
            'customers_data',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'grand_total_qty',
            'grand_total_amount',
            'grand_service_charges',
            'grand_received',
            'util'
        ));
    }

    /**
     * Print Daily Sales Details Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDailySalesDetails(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.sales_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get business logo
        $logo = null;
        if (!empty($business->logo)) {
            $logo = asset('uploads/business_logos/' . $business->logo);
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in printDailySalesDetails: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');

        // Get sell lines with product, customer details
        $query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereNull('transaction_sell_lines.parent_sell_line_id')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.transaction_date',
                't.contact_id',
                't.final_total as transaction_total',
                'c.name as customer_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'transaction_sell_lines.id as sell_line_id',
                DB::raw('(transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) as qty'),
                'transaction_sell_lines.unit_price_before_discount as rate',
                'transaction_sell_lines.unit_price_inc_tax as unit_price_inc_tax',
                DB::raw("((transaction_sell_lines.quantity - transaction_sell_lines.quantity_returned) * transaction_sell_lines.unit_price_inc_tax) as line_total"),
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply date filter - date range
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
        } elseif (!empty($end_date)) {
            $query->whereDate('t.transaction_date', '<=', $end_date);
        } elseif (!empty($start_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date);
        }

        // Permission check
        if (!auth()->user()->can('sell.view') && auth()->user()->can('view_own_sell_only')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $sell_lines = $query->orderBy('c.supplier_business_name', 'asc')
            ->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group by customer and calculate balances
        $customers_data = [];
        $contact_ids = $sell_lines->pluck('contact_id')->unique()->toArray();

        foreach ($contact_ids as $contact_id) {
            // Calculate previous balance (before start_date)
            $prev_balance = 0;
            if (!empty($start_date)) {
                $opening_balance_query = Transaction::where('business_id', $business_id)
                    ->where('contact_id', $contact_id)
                    ->whereIn('type', ['sell', 'sell_return', 'opening_balance', 'ledger_discount']);

                if ($permitted_locations != 'all') {
                    $opening_balance_query->whereIn('location_id', $permitted_locations);
                }

                if (!empty($location_id)) {
                    $opening_balance_query->where('location_id', $location_id);
                }

                $opening_balance_query->whereDate('transaction_date', '<', $start_date);

                $opening_balance_data = $opening_balance_query->select(
                    DB::raw("SUM(IF(type = 'sell' AND status = 'final', final_total, 0)) as total_sales"),
                    DB::raw("SUM(IF(type = 'sell_return' AND status = 'final', final_total, 0)) as total_sales_return"),
                    DB::raw("SUM(IF(type = 'opening_balance', final_total, 0)) as total_opening_balance"),
                    DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
                )->first();

                // Get payments before start_date
                $opening_payments_query = \App\TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $contact_id)
                    ->whereIn('t.type', ['sell', 'sell_return', 'opening_balance'])
                    ->whereDate('transaction_payments.paid_on', '<', $start_date);

                if ($permitted_locations != 'all') {
                    $opening_payments_query->whereIn('t.location_id', $permitted_locations);
                }

                if (!empty($location_id)) {
                    $opening_payments_query->where('t.location_id', $location_id);
                }

                $opening_payments = $opening_payments_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;

                $prev_balance = ($opening_balance_data->total_sales ?? 0) 
                    - ($opening_balance_data->total_sales_return ?? 0)
                    + ($opening_balance_data->total_opening_balance ?? 0)
                    - ($opening_balance_data->total_ledger_discount ?? 0)
                    - $opening_payments;
            }

            // Get customer lines
            $customer_lines = $sell_lines->where('contact_id', $contact_id);
            $customer = $customer_lines->first();
            $customer_name = 'WALKING CUSTOMER';
            if (!empty($customer->supplier_business_name)) {
                $customer_name = $customer->supplier_business_name;
            } elseif (!empty($customer->customer_name)) {
                $customer_name = $customer->customer_name;
            }

            // Calculate totals for date range
            $customer_total_qty = $customer_lines->sum('qty');
            $customer_total_amount = $customer_lines->sum('line_total');
            $customer_service_charges = 0; // Service charges would need to be stored separately

            // Get payments received in date range
            $transaction_ids = $customer_lines->pluck('transaction_id')->unique()->toArray();
            $received_with_cheque = 0;
            if (!empty($transaction_ids)) {
                $payments_query = \App\TransactionPayment::whereIn('transaction_id', $transaction_ids);
                if (!empty($start_date) && !empty($end_date)) {
                    $payments_query->whereBetween(DB::raw('DATE(paid_on)'), [$start_date, $end_date]);
                } elseif (!empty($end_date)) {
                    $payments_query->whereDate('paid_on', '<=', $end_date);
                } elseif (!empty($start_date)) {
                    $payments_query->whereDate('paid_on', '>=', $start_date);
                }
                $received_with_cheque = $payments_query->sum(DB::raw('IF(is_return = 1, -1 * amount, transaction_payments.amount)')) ?? 0;
            }

            // Calculate current balance
            $current_balance = $prev_balance + $customer_total_amount - $received_with_cheque;

            $customers_data[$contact_id] = [
                'customer_name' => $customer_name,
                'lines' => $customer_lines,
                'customer_total_qty' => $customer_total_qty,
                'customer_total_amount' => $customer_total_amount,
                'customer_service_charges' => $customer_service_charges,
                'received_with_cheque' => $received_with_cheque,
                'prev_balance' => $prev_balance,
                'current_balance' => $current_balance,
            ];
        }

        // Calculate grand totals
        $grand_total_qty = collect($customers_data)->sum('customer_total_qty');
        $grand_total_amount = collect($customers_data)->sum('customer_total_amount');
        $grand_service_charges = collect($customers_data)->sum('customer_service_charges');
        $grand_received = collect($customers_data)->sum('received_with_cheque');

        $util = new Util();

        return view('businessmanagement::sales_register.print_daily_sales_details', compact(
            'customers_data',
            'start_date',
            'end_date',
            'business',
            'logo',
            'grand_total_qty',
            'grand_total_amount',
            'grand_service_charges',
            'grand_received',
            'util'
        ));
    }
}

