<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

class AddBusinessManagementIndexes extends Migration
{
    /**
     * Run the migrations.
     * Note: DDL statements like CREATE INDEX cannot be rolled back in transactions.
     *
     * @return void
     */
    public function up()
    {
        // Transactions indexes
        $this->createIndexSafely('transactions', 'idx_transactions_business_location_type', 
            'business_id, location_id, type');
        
        $this->createIndexSafely('transactions', 'idx_transactions_business_date', 
            'business_id, transaction_date');
        
        $this->createIndexSafely('transactions', 'idx_transactions_location_date', 
            'location_id, transaction_date');
        
        $this->createIndexSafely('transactions', 'idx_transactions_contact_type', 
            'contact_id, type');
        
        $this->createIndexSafely('transactions', 'idx_transactions_type_date', 
            'type, transaction_date');
        
        // Purchase Lines indexes
        $this->createIndexSafely('purchase_lines', 'idx_purchase_lines_transaction', 
            'transaction_id');
        
        $this->createIndexSafely('purchase_lines', 'idx_purchase_lines_variation', 
            'variation_id');
        
        $this->createIndexSafely('purchase_lines', 'idx_purchase_lines_transaction_variation', 
            'transaction_id, variation_id');
        
        // Transaction Sell Lines indexes
        $this->createIndexSafely('transaction_sell_lines', 'idx_transaction_sell_lines_transaction', 
            'transaction_id');
        
        $this->createIndexSafely('transaction_sell_lines', 'idx_transaction_sell_lines_variation', 
            'variation_id');
        
        $this->createIndexSafely('transaction_sell_lines', 'idx_transaction_sell_lines_transaction_variation', 
            'transaction_id, variation_id');
        
        // Variation Location Details indexes
        $this->createIndexSafely('variation_location_details', 'idx_variation_location_details_location', 
            'location_id');
        
        $this->createIndexSafely('variation_location_details', 'idx_variation_location_details_variation_location', 
            'variation_id, location_id');
        
        $this->createIndexSafely('variation_location_details', 'idx_variation_location_details_location_variation', 
            'location_id, variation_id');
        
        // Account Transactions indexes
        $this->createIndexSafely('account_transactions', 'idx_account_transactions_account_date', 
            'account_id, operation_date');
        
        $this->createIndexSafely('account_transactions', 'idx_account_transactions_operation_date', 
            'operation_date');
        
        // Products indexes (for product-wise reports)
        $this->createIndexSafely('products', 'idx_products_business_category', 
            'business_id, category_id');
        
        // Contacts indexes (for supplier/customer-wise reports)
        $this->createIndexSafely('contacts', 'idx_contacts_business_type', 
            'business_id, type');
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        // Drop indexes
        $indexes = [
            'transactions' => [
                'idx_transactions_business_location_type',
                'idx_transactions_business_date',
                'idx_transactions_location_date',
                'idx_transactions_contact_type',
                'idx_transactions_type_date',
            ],
            'purchase_lines' => [
                'idx_purchase_lines_transaction',
                'idx_purchase_lines_variation',
                'idx_purchase_lines_transaction_variation',
            ],
            'transaction_sell_lines' => [
                'idx_transaction_sell_lines_transaction',
                'idx_transaction_sell_lines_variation',
                'idx_transaction_sell_lines_transaction_variation',
            ],
            'variation_location_details' => [
                'idx_variation_location_details_location',
                'idx_variation_location_details_variation_location',
                'idx_variation_location_details_location_variation',
            ],
            'account_transactions' => [
                'idx_account_transactions_account_date',
                'idx_account_transactions_operation_date',
            ],
            'products' => [
                'idx_products_business_category',
            ],
            'contacts' => [
                'idx_contacts_business_type',
            ],
        ];

        foreach ($indexes as $table => $tableIndexes) {
            foreach ($tableIndexes as $index) {
                try {
                    $this->dropIndexIfExists($table, $index);
                } catch (\Exception $e) {
                    // Index might not exist, continue
                    \Log::warning("Could not drop index {$index} on table {$table}: " . $e->getMessage());
                }
            }
        }
    }

    /**
     * Create index safely (handles if index already exists)
     */
    private function createIndexSafely($table, $indexName, $columns)
    {
        $connection = DB::connection()->getDriverName();
        
        try {
            if ($connection === 'mysql') {
                // MySQL - try to create, ignore if exists
                try {
                    DB::unprepared("CREATE INDEX `{$indexName}` ON `{$table}` ({$columns})");
                } catch (\Exception $e) {
                    // Check if error is about duplicate index
                    $errorMessage = $e->getMessage();
                    if (strpos($errorMessage, 'Duplicate key name') === false && 
                        strpos($errorMessage, 'already exists') === false &&
                        strpos($errorMessage, 'Duplicate index') === false) {
                        // Re-throw if it's a different error
                        throw $e;
                    }
                    // Index already exists, which is fine - continue silently
                }
            } elseif ($connection === 'pgsql') {
                // PostgreSQL - use IF NOT EXISTS
                DB::unprepared("CREATE INDEX IF NOT EXISTS {$indexName} ON {$table} ({$columns})");
            } else {
                // For other databases, try to create
                try {
                    DB::unprepared("CREATE INDEX {$indexName} ON {$table} ({$columns})");
                } catch (\Exception $e) {
                    // Index might already exist, which is fine
                    $errorMessage = $e->getMessage();
                    if (strpos($errorMessage, 'already exists') === false && 
                        strpos($errorMessage, 'Duplicate') === false) {
                        // Re-throw if it's a different error
                        throw $e;
                    }
                }
            }
        } catch (\Exception $e) {
            // Log only if it's not a "duplicate" error
            $errorMessage = $e->getMessage();
            if (strpos($errorMessage, 'already exists') === false && 
                strpos($errorMessage, 'Duplicate') === false) {
                \Log::warning("Could not create index {$indexName} on table {$table}: " . $errorMessage);
            }
        }
    }

    /**
     * Drop index if it exists
     */
    private function dropIndexIfExists($table, $indexName)
    {
        $connection = DB::connection()->getDriverName();
        
        try {
            if ($connection === 'mysql') {
                DB::unprepared("DROP INDEX `{$indexName}` ON `{$table}`");
            } elseif ($connection === 'pgsql') {
                DB::unprepared("DROP INDEX IF EXISTS {$indexName}");
            } else {
                DB::unprepared("DROP INDEX {$indexName} ON {$table}");
            }
        } catch (\Exception $e) {
            // Index might not exist, which is fine
            if (strpos($e->getMessage(), "doesn't exist") === false &&
                strpos($e->getMessage(), 'Unknown key') === false) {
                \Log::warning("Could not drop index {$indexName} on table {$table}: " . $e->getMessage());
            }
        }
    }
}

