# BusinessManagement Module - Developer Guide

**Version:** 1.0.1  
**Last Updated:** 2024

---

## Table of Contents

1. [Module Overview](#module-overview)
2. [Architecture](#architecture)
3. [Security Patterns](#security-patterns)
4. [Adding New Reports](#adding-new-reports)
5. [Best Practices](#best-practices)
6. [Common Patterns](#common-patterns)
7. [Troubleshooting](#troubleshooting)

---

## Module Overview

The BusinessManagement module provides comprehensive reporting functionality for UltimatePOS:

- **Purchase Register:** 20+ purchase-related reports
- **Sales Register:** 25+ sales-related reports
- **Stock Register:** 11 stock/inventory reports
- **Accounts Register:** 20+ financial/accounting reports

---

## Architecture

### Module Structure

```
BusinessManagement/
├── Http/Controllers/
│   ├── PurchaseRegisterController.php
│   ├── SalesRegisterController.php
│   ├── StockRegisterController.php
│   ├── AccountsRegisterController.php
│   ├── DataController.php (permissions & menu)
│   └── InstallController.php
├── Resources/views/
│   ├── purchase_register/
│   ├── sales_register/
│   ├── stock_register/
│   └── accounts_register/
├── Routes/web.php
└── Database/Migrations/
```

### Key Components

1. **Controllers:** Handle report logic and data processing
2. **Views:** Blade templates for displaying reports
3. **Routes:** Web routes with middleware protection
4. **Permissions:** Granular permission system

---

## Security Patterns

### Location Access Validation

**CRITICAL:** Always validate location access when using `location_id` parameter.

```php
// Get permitted locations
$permitted_locations = auth()->user()->permitted_locations();

// Filter by permitted locations
if ($permitted_locations != 'all') {
    $query->whereIn('location_id', $permitted_locations);
}

// Validate requested location
if (!empty($request->location_id)) {
    // SECURITY: Always validate
    if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
        abort(403, 'Unauthorized location access.');
    }
    $query->where('location_id', $request->location_id);
}
```

### Permission Checks

**Always check permissions at method start:**

```php
public function someMethod(Request $request)
{
    if (!auth()->user()->can('businessmanagement.purchase_register')) {
        abort(403, 'Unauthorized action.');
    }
    // ... rest of method
}
```

### Business ID Filtering

**Always filter by business_id from session:**

```php
$business_id = request()->session()->get('user.business_id');
$query->where('business_id', $business_id);
```

---

## Adding New Reports

### Step 1: Add Route

In `Routes/web.php`:

```php
Route::get('purchase-register/new-report', [
    PurchaseRegisterController::class, 
    'newReport'
])->name('purchase_register.new_report');

Route::get('purchase-register/new-report/print', [
    PurchaseRegisterController::class, 
    'printNewReport'
])->name('purchase_register.new_report.print');
```

### Step 2: Add Controller Method

```php
public function newReport(Request $request)
{
    // Permission check
    if (!auth()->user()->can('businessmanagement.purchase_register')) {
        abort(403, 'Unauthorized action.');
    }

    $business_id = request()->session()->get('user.business_id');
    $locations = BusinessLocation::forDropdown($business_id, true);

    if ($request->ajax()) {
        $query = Transaction::where('business_id', $business_id)
            ->whereIn('type', ['purchase', 'opening_stock']);

        // Apply location filter with validation
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('location_id', $request->location_id);
        }

        // Date filter
        if (!empty($request->start_date) && !empty($request->end_date)) {
            $query->whereBetween('transaction_date', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }

        // View own permission
        if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
            $query->where('created_by', request()->session()->get('user.id'));
        }

        return DataTables::of($query)
            ->editColumn('transaction_date', function ($row) {
                return \Carbon::parse($row->transaction_date)->format('d/m/Y');
            })
            ->make(true);
    }

    return view('businessmanagement::purchase_register.new_report', compact('locations'));
}
```

### Step 3: Add Print Method

```php
public function printNewReport(Request $request)
{
    if (!auth()->user()->can('businessmanagement.purchase_register')) {
        abort(403, 'Unauthorized action.');
    }

    $business_id = request()->session()->get('user.business_id');
    $business = \App\Business::find($business_id);
    $location = null;

    if (!empty($request->location_id)) {
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
            abort(403, 'Unauthorized location access.');
        }
        $location = BusinessLocation::find($request->location_id);
    }

    // Build query (similar to main method)
    // Get data
    // Return print view
}
```

### Step 4: Add View

Create `Resources/views/purchase_register/new_report.blade.php`:

```blade
@extends('layouts.app')

@section('title', __('businessmanagement::lang.new_report'))

@section('content')
<!-- Report content -->
@endsection
```

### Step 5: Add Permission

In `DataController::user_permissions()`:

```php
[
    'value' => 'businessmanagement.purchase_register.new_report',
    'label' => __('businessmanagement::lang.new_report'),
    'default' => false
],
```

### Step 6: Add Migration

Create migration to register permission:

```php
DB::table('permissions')->insert([
    'name' => 'businessmanagement.purchase_register.new_report',
    'guard_name' => 'web',
    'created_at' => now(),
    'updated_at' => now()
]);
```

---

## Best Practices

### 1. Always Validate Location Access

```php
// ✅ GOOD
if (!empty($request->location_id)) {
    if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
        abort(403, 'Unauthorized location access.');
    }
}

// ❌ BAD
if (!empty($request->location_id)) {
    $query->where('location_id', $request->location_id);
}
```

### 2. Use Eager Loading

```php
// ✅ GOOD
$transactions = Transaction::with(['contact', 'location'])->get();

// ❌ BAD
$transactions = Transaction::get();
// Then accessing $transaction->contact in loop (N+1 problem)
```

### 3. Select Specific Columns

```php
// ✅ GOOD
->select('id', 'ref_no', 'transaction_date', 'final_total')

// ❌ BAD
->select('*')
```

### 4. Use Parameter Binding

```php
// ✅ GOOD (Laravel does this automatically)
->where('location_id', $request->location_id)

// ❌ BAD (Never do this)
->whereRaw("location_id = {$request->location_id}")
```

### 5. Filter by Business ID

```php
// ✅ GOOD
$business_id = request()->session()->get('user.business_id');
$query->where('business_id', $business_id);

// ❌ BAD
$query->where('business_id', $request->business_id) // Never trust user input
```

---

## Common Patterns

### DataTables Pattern

```php
if ($request->ajax()) {
    $query = Model::where(...);
    
    // Apply filters
    // Apply location filter
    // Apply date filter
    
    return DataTables::of($query)
        ->editColumn('date', function ($row) {
            return format_date($row->date);
        })
        ->addColumn('custom_column', function ($row) {
            return custom_logic($row);
        })
        ->rawColumns(['html_column'])
        ->make(true);
}
```

### Print Report Pattern

```php
public function printReport(Request $request)
{
    // Permission check
    // Location validation
    // Build query
    // Get data
    // Format data
    // Return view with data
}
```

### Date Range Filter Pattern

```php
if (!empty($request->start_date) && !empty($request->end_date)) {
    $query->whereBetween('transaction_date', [
        $request->start_date . ' 00:00:00',
        $request->end_date . ' 23:59:59'
    ]);
}
```

---

## Troubleshooting

### Issue: Slow Query Performance

**Solutions:**
1. Check if indexes exist on filtered columns
2. Use `EXPLAIN` to analyze query
3. Consider adding eager loading
4. Review complex joins

### Issue: 403 Unauthorized Errors

**Check:**
1. User has module permission
2. User has report-specific permission
3. Location access validation is correct
4. Business ID matches session

### Issue: Missing Data in Reports

**Check:**
1. Location filter is correct
2. Date range is correct
3. Business ID filter is applied
4. View own permission is handled correctly

### Issue: Print View Not Rendering

**Check:**
1. View file exists
2. Data is passed to view
3. Location validation passed
4. Business/logo data is available

---

## Testing Checklist

When adding new reports:

- [ ] Permission check implemented
- [ ] Location validation implemented
- [ ] Business ID filtering implemented
- [ ] Date range filtering works
- [ ] View own permission handled
- [ ] Print method works
- [ ] DataTables AJAX works
- [ ] No N+1 query issues
- [ ] Proper error handling
- [ ] Translation keys added

---

## Resources

- **Security Review:** `SECURITY_REVIEW.md`
- **Performance Guide:** `PERFORMANCE_OPTIMIZATION.md`
- **Module Review:** `MODULE_REVIEW.md`
- **Sales Register Review:** `SALES_REGISTER_REVIEW.md`

---

**Questions?** Contact the development team or refer to UltimatePOS documentation.

