<?php

namespace Modules\BusinessManagement\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Account;
use App\AccountTransaction;
use App\BusinessLocation;
use App\Contact;
use App\Category;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use App\Utils\Util;
use App\Utils\TransactionUtil;
use App\Utils\BusinessUtil;
use Illuminate\Support\Facades\DB;

class AccountsRegisterController extends Controller
{
    /**
     * Display a listing of the accounts register.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id);
        $accounts = Account::forDropdown($business_id, true, false, false);

        if ($request->ajax()) {
            $account_transactions = AccountTransaction::join('accounts as a', 'account_transactions.account_id', '=', 'a.id')
                ->where('a.business_id', $business_id)
                ->with(['account', 'transaction'])
                ->select('account_transactions.*');

            // Apply location filter based on user permissions
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $account_transactions->whereHas('transaction', function ($q) use ($permitted_locations) {
                    $q->whereIn('location_id', $permitted_locations);
                });
            }

            if (!empty($request->account_id)) {
                $account_transactions->where('account_transactions.account_id', $request->account_id);
            }

            if (!empty($request->location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $account_transactions->whereHas('transaction', function ($q) use ($request) {
                    $q->where('location_id', $request->location_id);
                });
            }

            if (!empty($request->start_date) && !empty($request->end_date)) {
                $account_transactions->whereBetween('account_transactions.operation_date', [$request->start_date, $request->end_date]);
            }

            $util = new Util();
            
            return DataTables::of($account_transactions)
                ->addColumn('account_name', function ($row) {
                    return $row->account->name ?? '';
                })
                ->addColumn('type', function ($row) {
                    return ucfirst(str_replace('_', ' ', $row->type));
                })
                ->editColumn('operation_date', function ($row) use ($util) {
                    return $util->format_date($row->operation_date, true);
                })
                ->editColumn('amount', function ($row) {
                    $class = $row->type == 'credit' ? 'text-success' : 'text-danger';
                    $sign = $row->type == 'credit' ? '+' : '-';
                    return '<span class="' . $class . '">' . $sign . '</span> <span class="display_currency" data-currency_symbol="true">' . $row->amount . '</span>';
                })
                ->rawColumns(['amount'])
                ->make(true);
        }

        return view('businessmanagement::accounts_register.index', compact('locations', 'accounts'));
    }

    /**
     * Due List Report
     */
    public function dueList(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filter
        $date = !empty($request->date) ? $request->date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Get contact type filter
        $contact_type_filter = $request->get('contact_type_filter', 'all_customer');
        $contact_id = $request->get('contact_id');

        // Get contacts for dropdown
        $customers = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select('id', 'name', 'supplier_business_name')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function($contact) {
                return [$contact->id => ($contact->supplier_business_name ?? $contact->name)];
            })
            ->prepend(__('lang_v1.all'), 'all');

        $suppliers = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->select('id', 'name', 'supplier_business_name')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function($contact) {
                return [$contact->id => ($contact->supplier_business_name ?? $contact->name)];
            })
            ->prepend(__('lang_v1.all'), 'all');

        // Get categories for grouping
        $categories = Category::where('business_id', $business_id)
            ->where('parent_id', 0)
            ->orderBy('name', 'asc')
            ->pluck('name', 'id');

        // Build query to get contacts with outstanding balances
        $query = Contact::where('business_id', $business_id)
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.type',
                DB::raw("'NO GROUP' as category_name"),
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name")
            );

        // Apply contact type filter
        if ($contact_type_filter == 'all_customer') {
            $query->where('contacts.type', 'customer');
        } elseif ($contact_type_filter == 'all_supplier') {
            $query->where('contacts.type', 'supplier');
        } elseif ($contact_type_filter == 'single_customer') {
            $query->where('contacts.type', 'customer');
            if (!empty($contact_id) && $contact_id != 'all') {
                $query->where('contacts.id', $contact_id);
            }
        } elseif ($contact_type_filter == 'single_supplier') {
            $query->where('contacts.type', 'supplier');
            if (!empty($contact_id) && $contact_id != 'all') {
                $query->where('contacts.id', $contact_id);
            }
        }

        $contacts = $query->get();

        // Calculate outstanding balance for each contact as of the date
        $due_data = [];
        foreach ($contacts as $contact) {
            // Get all transactions for this contact up to the date
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $contact->id)
                ->whereIn('t.type', ['sell', 'purchase', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'sell')->where('t.status', 'final');
                    })->orWhere(function($q2) {
                        $q2->where('t.type', 'purchase')->where('t.status', 'received');
                    })->orWhere('t.type', 'opening_balance');
                })
                ->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                // Add invoice amount
                if ($transaction->type == 'sell' || $transaction->type == 'purchase' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                // Calculate payments for this transaction
                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate due amount
            // For customers: due = invoices - payments (positive means customer owes us)
            // For suppliers: due = invoices - payments (positive means we owe supplier)
            $due_amount = $total_invoice - $total_paid;

            // Only include contacts with non-zero balance
            if (abs($due_amount) > 0.01) {
                $due_data[] = (object)[
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->display_name,
                    'category_name' => $contact->category_name,
                    'type' => $contact->type,
                    'due_amount' => $due_amount,
                ];
            }
        }

        // Group data by category
        $grouped_data = [];
        foreach ($due_data as $item) {
            $category_name = $item->category_name ?? 'NO GROUP';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Sort each category by contact name
        foreach ($grouped_data as $category => $items) {
            usort($grouped_data[$category], function($a, $b) {
                return strcmp($a->contact_name, $b->contact_name);
            });
        }

        // Calculate totals
        $grand_total = collect($due_data)->sum('due_amount');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.due_list', compact(
            'grouped_data',
            'date',
            'contact_type_filter',
            'contact_id',
            'customers',
            'suppliers',
            'grand_total',
            'util',
            'business'
        ));
    }

    /**
     * Print Due List Report
     */
    public function printDueList(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filter
        $date = !empty($request->date) ? $request->date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Get contact type filter
        $contact_type_filter = $request->get('contact_type_filter', 'all_customer');
        $contact_id = $request->get('contact_id');

        // Build query to get contacts with outstanding balances (same as web view)
        $query = Contact::where('business_id', $business_id)
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.type',
                DB::raw("'NO GROUP' as category_name"),
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name")
            );

        // Apply contact type filter
        if ($contact_type_filter == 'all_customer') {
            $query->where('contacts.type', 'customer');
        } elseif ($contact_type_filter == 'all_supplier') {
            $query->where('contacts.type', 'supplier');
        } elseif ($contact_type_filter == 'single_customer') {
            $query->where('contacts.type', 'customer');
            if (!empty($contact_id) && $contact_id != 'all') {
                $query->where('contacts.id', $contact_id);
            }
        } elseif ($contact_type_filter == 'single_supplier') {
            $query->where('contacts.type', 'supplier');
            if (!empty($contact_id) && $contact_id != 'all') {
                $query->where('contacts.id', $contact_id);
            }
        }

        $contacts = $query->get();

        // Calculate outstanding balance for each contact (same logic as web view)
        $due_data = [];
        foreach ($contacts as $contact) {
            // Get all transactions for this contact up to the date
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $contact->id)
                ->whereIn('t.type', ['sell', 'purchase', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'sell')->where('t.status', 'final');
                    })->orWhere(function($q2) {
                        $q2->where('t.type', 'purchase')->where('t.status', 'received');
                    })->orWhere('t.type', 'opening_balance');
                })
                ->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                // Add invoice amount
                if ($transaction->type == 'sell' || $transaction->type == 'purchase' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                // Calculate payments for this transaction
                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate due amount
            $due_amount = $total_invoice - $total_paid;

            if (abs($due_amount) > 0.01) {
                $due_data[] = (object)[
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->display_name,
                    'category_name' => $contact->category_name,
                    'type' => $contact->type,
                    'due_amount' => $due_amount,
                ];
            }
        }

        // Group data by category
        $grouped_data = [];
        foreach ($due_data as $item) {
            $category_name = $item->category_name ?? 'NO GROUP';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Sort each category by contact name
        foreach ($grouped_data as $category => $items) {
            usort($grouped_data[$category], function($a, $b) {
                return strcmp($a->contact_name, $b->contact_name);
            });
        }

        // Calculate totals
        $grand_total = collect($due_data)->sum('due_amount');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_due_list', compact(
            'grouped_data',
            'date',
            'contact_type_filter',
            'contact_id',
            'grand_total',
            'util',
            'business'
        ));
    }

    /**
     * All Customers Due List Report
     */
    public function allCustomersDueList(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filter - handle both formats (dd/mm/yyyy and Y-m-d)
        $date = null;
        if (!empty($request->date)) {
            // Try to parse as dd/mm/yyyy first
            try {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            } catch (\Exception $e) {
                // If that fails, try Y-m-d format
                try {
                    $date = \Carbon\Carbon::parse($request->date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        } else {
            $date = \Carbon\Carbon::now()->format('Y-m-d');
        }
        
        // Get customer filter
        $customer_filter = $request->get('customer_filter', 'all');
        $customer_id = $request->get('customer_id');

        // Get customers for dropdown
        $customers = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select('id', 'name', 'supplier_business_name', 'mobile', 'address_line_1', 'address_line_2', 'city', 'state', 'country')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function($contact) {
                return [$contact->id => ($contact->supplier_business_name ?? $contact->name)];
            })
            ->prepend(__('lang_v1.all'), 'all');

        // Build query to get customers with outstanding balances
        $query = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.mobile',
                'contacts.address_line_1',
                'contacts.address_line_2',
                'contacts.city',
                'contacts.state',
                'contacts.country',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                DB::raw("CONCAT(
                    COALESCE(contacts.address_line_1, ''),
                    CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.address_line_2, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.city, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.state, '')
                ) as full_address")
            );

        // Apply customer filter
        if ($customer_filter == 'single' && !empty($customer_id) && $customer_id != 'all') {
            $query->where('contacts.id', $customer_id);
        }

        $contacts = $query->orderBy('display_name', 'asc')->get();

        // Calculate outstanding balance for each customer as of the date
        $due_data = [];
        foreach ($contacts as $contact) {
            // Get all transactions for this customer up to the date
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $contact->id)
                ->whereIn('t.type', ['sell', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'sell')->where('t.status', 'final');
                    })->orWhere('t.type', 'opening_balance');
                })
                ->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                // Add invoice amount
                if ($transaction->type == 'sell' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                // Calculate payments for this transaction
                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate due amount (receivable)
            $due_amount = $total_invoice - $total_paid;

            // Only include customers with non-zero balance
            if (abs($due_amount) > 0.01) {
                $due_data[] = (object)[
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->display_name,
                    'address' => $contact->full_address ?? '',
                    'mobile' => $contact->mobile ?? '',
                    'due_amount' => $due_amount,
                ];
            }
        }

        // Calculate totals
        $grand_total = collect($due_data)->sum('due_amount');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.all_customers_due_list', compact(
            'due_data',
            'date',
            'customer_filter',
            'customer_id',
            'customers',
            'grand_total',
            'util',
            'business'
        ));
    }

    /**
     * Print All Customers Due List Report
     */
    public function printAllCustomersDueList(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filter
        $date = !empty($request->date) ? $request->date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Get customer filter
        $customer_filter = $request->get('customer_filter', 'all');
        $customer_id = $request->get('customer_id');

        // Build query to get customers with outstanding balances (same as web view)
        $query = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.mobile',
                'contacts.address_line_1',
                'contacts.address_line_2',
                'contacts.city',
                'contacts.state',
                'contacts.country',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                DB::raw("CONCAT(
                    COALESCE(contacts.address_line_1, ''),
                    CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.address_line_2, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.city, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.state, '')
                ) as full_address")
            );

        // Apply customer filter
        if ($customer_filter == 'single' && !empty($customer_id) && $customer_id != 'all') {
            $query->where('contacts.id', $customer_id);
        }

        $contacts = $query->orderBy('display_name', 'asc')->get();

        // Calculate outstanding balance for each customer (same logic as web view)
        $due_data = [];
        foreach ($contacts as $contact) {
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $contact->id)
                ->whereIn('t.type', ['sell', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'sell')->where('t.status', 'final');
                    })->orWhere('t.type', 'opening_balance');
                })
                ->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                if ($transaction->type == 'sell' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            $due_amount = $total_invoice - $total_paid;

            if (abs($due_amount) > 0.01) {
                $due_data[] = (object)[
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->display_name,
                    'address' => $contact->full_address ?? '',
                    'mobile' => $contact->mobile ?? '',
                    'due_amount' => $due_amount,
                ];
            }
        }

        // Calculate totals
        $grand_total = collect($due_data)->sum('due_amount');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_all_customers_due_list', compact(
            'due_data',
            'date',
            'customer_filter',
            'customer_id',
            'grand_total',
            'util',
            'business'
        ));
    }

    /**
     * Supplier Due List Report
     */
    public function supplierDueList(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filter - handle both formats (dd/mm/yyyy and Y-m-d)
        $date = null;
        if (!empty($request->date)) {
            // Try to parse as dd/mm/yyyy first
            try {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            } catch (\Exception $e) {
                // If that fails, try Y-m-d format
                try {
                    $date = \Carbon\Carbon::parse($request->date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        } else {
            $date = \Carbon\Carbon::now()->format('Y-m-d');
        }
        
        // Get supplier filter
        $supplier_filter = $request->get('supplier_filter', 'all');
        $supplier_id = $request->get('supplier_id');

        // Get suppliers for dropdown
        $suppliers = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->select('id', 'name', 'supplier_business_name', 'mobile', 'address_line_1', 'address_line_2', 'city', 'state', 'country')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function($contact) {
                return [$contact->id => ($contact->supplier_business_name ?? $contact->name)];
            })
            ->prepend(__('lang_v1.all'), 'all');

        // Build query to get suppliers with outstanding balances
        $query = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.mobile',
                'contacts.address_line_1',
                'contacts.address_line_2',
                'contacts.city',
                'contacts.state',
                'contacts.country',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                DB::raw("CONCAT(
                    COALESCE(contacts.address_line_1, ''),
                    CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.address_line_2, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.city, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.state, '')
                ) as full_address")
            );

        // Apply supplier filter
        if ($supplier_filter == 'single' && !empty($supplier_id) && $supplier_id != 'all') {
            $query->where('contacts.id', $supplier_id);
        }

        $contacts = $query->orderBy('display_name', 'asc')->get();

        // Calculate outstanding balance for each supplier as of the date
        $due_data = [];
        foreach ($contacts as $contact) {
            // Get all transactions for this supplier up to the date
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $contact->id)
                ->whereIn('t.type', ['purchase', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'purchase')->where('t.status', 'received');
                    })->orWhere('t.type', 'opening_balance');
                })
                ->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                // Add invoice amount
                if ($transaction->type == 'purchase' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                // Calculate payments for this transaction
                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate due amount (payable - what we owe supplier)
            $due_amount = $total_invoice - $total_paid;

            // Only include suppliers with non-zero balance
            if (abs($due_amount) > 0.01) {
                $due_data[] = (object)[
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->display_name,
                    'address' => $contact->full_address ?? '',
                    'mobile' => $contact->mobile ?? '',
                    'due_amount' => $due_amount,
                ];
            }
        }

        // Calculate totals
        $grand_total = collect($due_data)->sum('due_amount');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.supplier_due_list', compact(
            'due_data',
            'date',
            'supplier_filter',
            'supplier_id',
            'suppliers',
            'grand_total',
            'util',
            'business'
        ));
    }

    /**
     * Print Supplier Due List Report
     */
    public function printSupplierDueList(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filter - handle both formats (dd/mm/yyyy and Y-m-d)
        $date = null;
        if (!empty($request->date)) {
            // Try to parse as dd/mm/yyyy first
            try {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            } catch (\Exception $e) {
                // If that fails, try Y-m-d format
                try {
                    $date = \Carbon\Carbon::parse($request->date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        } else {
            $date = \Carbon\Carbon::now()->format('Y-m-d');
        }
        
        // Get supplier filter
        $supplier_filter = $request->get('supplier_filter', 'all');
        $supplier_id = $request->get('supplier_id');

        // Build query to get suppliers with outstanding balances (same as web view)
        $query = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.mobile',
                'contacts.address_line_1',
                'contacts.address_line_2',
                'contacts.city',
                'contacts.state',
                'contacts.country',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                DB::raw("CONCAT(
                    COALESCE(contacts.address_line_1, ''),
                    CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.address_line_2, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.city, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.state, '')
                ) as full_address")
            );

        // Apply supplier filter
        if ($supplier_filter == 'single' && !empty($supplier_id) && $supplier_id != 'all') {
            $query->where('contacts.id', $supplier_id);
        }

        $contacts = $query->orderBy('display_name', 'asc')->get();

        // Calculate outstanding balance for each supplier (same logic as web view)
        $due_data = [];
        foreach ($contacts as $contact) {
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $contact->id)
                ->whereIn('t.type', ['purchase', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'purchase')->where('t.status', 'received');
                    })->orWhere('t.type', 'opening_balance');
                })
                ->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                if ($transaction->type == 'purchase' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            $due_amount = $total_invoice - $total_paid;

            if (abs($due_amount) > 0.01) {
                $due_data[] = (object)[
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->display_name,
                    'address' => $contact->full_address ?? '',
                    'mobile' => $contact->mobile ?? '',
                    'due_amount' => $due_amount,
                ];
            }
        }

        // Calculate totals
        $grand_total = collect($due_data)->sum('due_amount');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_supplier_due_list', compact(
            'due_data',
            'date',
            'supplier_filter',
            'supplier_id',
            'grand_total',
            'util',
            'business'
        ));
    }

    /**
     * Accounts Ledger Report
     */
    public function customerAccountsLedger(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        // Get contact type filter
        $contact_type = $request->get('contact_type', 'customer');
        $customer_id = $request->get('customer_id');
        $supplier_id = $request->get('supplier_id');
        $location_id = $request->get('location_id');

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id, true);

        // Get customers for dropdown
        $customers = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select('id', 'name', 'supplier_business_name')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function($contact) {
                return [$contact->id => ($contact->supplier_business_name ?? $contact->name)];
            })
            ->prepend(__('lang_v1.select_customer'), '');

        // Get suppliers for dropdown
        $suppliers = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->select('id', 'name', 'supplier_business_name')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function($contact) {
                return [$contact->id => ($contact->supplier_business_name ?? $contact->name)];
            })
            ->prepend(__('lang_v1.select_supplier'), '');

        // Get contact details
        $contact_id = ($contact_type == 'supplier') ? $supplier_id : $customer_id;
        $customer = null;
        $ledger_data = [];
        $opening_balance = 0;
        $running_balance = 0;

        if (!empty($contact_id)) {
            $customer = Contact::where('business_id', $business_id)
                ->where('type', $contact_type)
                ->where('id', $contact_id)
                ->first();

            if ($customer) {
                // Calculate opening balance (balance before start_date)
                $transaction_types = ($contact_type == 'supplier') ? ['purchase', 'opening_balance'] : ['sell', 'opening_balance'];
                $status_condition = ($contact_type == 'supplier') ? 'received' : 'final';
                
                // Apply location filter
                $permitted_locations = auth()->user()->permitted_locations();
                if ($permitted_locations != 'all') {
                    $opening_transactions_query = DB::table('transactions as t')
                        ->whereIn('t.location_id', $permitted_locations);
                } else {
                    $opening_transactions_query = DB::table('transactions as t');
                }

                if (!empty($location_id)) {
                    // Validate that user has access to the requested location
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        abort(403, 'Unauthorized location access.');
                    }
                }

                $opening_transactions = $opening_transactions_query
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $contact_id)
                    ->whereIn('t.type', $transaction_types)
                    ->whereDate('t.transaction_date', '<', $start_date)
                    ->where(function($q) use ($contact_type, $status_condition) {
                        if ($contact_type == 'supplier') {
                            $q->where(function($q2) use ($status_condition) {
                                $q2->where('t.type', 'purchase')->where('t.status', $status_condition);
                            })->orWhere('t.type', 'opening_balance');
                        } else {
                            $q->where(function($q2) use ($status_condition) {
                                $q2->where('t.type', 'sell')->where('t.status', $status_condition);
                            })->orWhere('t.type', 'opening_balance');
                        }
                    });

                if (!empty($location_id)) {
                    $opening_transactions->where('t.location_id', $location_id);
                }

                $opening_transactions = $opening_transactions
                    ->select('t.id', 't.type', 't.final_total')
                    ->get();

                $opening_invoice = 0;
                $opening_paid = 0;

                foreach ($opening_transactions as $transaction) {
                    if (in_array($transaction->type, $transaction_types)) {
                        $opening_invoice += $transaction->final_total ?? 0;
                    }

                    $payments = DB::table('transaction_payments')
                        ->where('transaction_id', $transaction->id)
                        ->whereDate('paid_on', '<', $start_date)
                        ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                    $opening_paid += $payments ?? 0;
                }

                $opening_balance = $opening_invoice - $opening_paid;
                $running_balance = $opening_balance;

                // Add opening balance row
                if ($opening_balance != 0) {
                    if ($contact_type == 'supplier') {
                        // For suppliers: positive = we owe (credit), negative = they owe us (debit)
                        $ledger_data[] = (object)[
                            'date' => $start_date,
                            'type' => 'Balance Forward',
                            'transaction_mode' => 'Balance Forward',
                            'ac_name' => '',
                            'ref_no' => '',
                            'voucher_no' => '',
                            'debit' => $opening_balance < 0 ? abs($opening_balance) : 0,
                            'credit' => $opening_balance > 0 ? abs($opening_balance) : 0,
                            'balance' => $running_balance,
                        ];
                    } else {
                        // For customers: positive = they owe us (debit), negative = we owe them (credit)
                        $ledger_data[] = (object)[
                            'date' => $start_date,
                            'type' => 'Balance Forward',
                            'transaction_mode' => 'Balance Forward',
                            'ac_name' => '',
                            'ref_no' => '',
                            'voucher_no' => '',
                            'debit' => $opening_balance > 0 ? abs($opening_balance) : 0,
                            'credit' => $opening_balance < 0 ? abs($opening_balance) : 0,
                            'balance' => $running_balance,
                        ];
                    }
                }

                // Get all transactions in date range
                // Get all transactions in date range
                $transaction_types_range = ($contact_type == 'supplier') 
                    ? ['purchase', 'opening_balance', 'purchase_return'] 
                    : ['sell', 'opening_balance', 'sell_return'];
                
                $transactions = DB::table('transactions as t')
                    ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $contact_id)
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where(function($q) use ($contact_type, $status_condition) {
                        if ($contact_type == 'supplier') {
                            $q->where(function($q2) use ($status_condition) {
                                $q2->where('t.type', 'purchase')->where('t.status', $status_condition);
                            })->orWhere('t.type', 'opening_balance')
                            ->orWhere('t.type', 'purchase_return');
                        } else {
                            $q->where(function($q2) use ($status_condition) {
                                $q2->where('t.type', 'sell')->where('t.status', $status_condition);
                            })->orWhere('t.type', 'opening_balance')
                            ->orWhere('t.type', 'sell_return');
                        }
                    })
                    ->select(
                        't.id',
                        't.transaction_date',
                        't.type',
                        't.status',
                        't.invoice_no',
                        't.ref_no',
                        't.final_total',
                        't.additional_notes',
                        DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as contact_name")
                    )
                    ->orderBy('t.transaction_date', 'asc')
                    ->orderBy('t.id', 'asc')
                    ->get();

                // Process transactions
                foreach ($transactions as $transaction) {
                    if ($contact_type == 'supplier') {
                        if ($transaction->type == 'purchase' && $transaction->status == $status_condition) {
                            // Purchase transaction - credit (we owe supplier)
                            $running_balance += $transaction->final_total;
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Purchase',
                                'transaction_mode' => 'Cash Purchase',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => 0,
                                'credit' => $transaction->final_total,
                                'balance' => $running_balance,
                            ];
                        } elseif ($transaction->type == 'purchase_return') {
                            // Purchase return - debit (reduces what we owe)
                            $running_balance -= abs($transaction->final_total);
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Purchase Return',
                                'transaction_mode' => 'Purchase Return',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => abs($transaction->final_total),
                                'credit' => 0,
                                'balance' => $running_balance,
                            ];
                        } elseif ($transaction->type == 'opening_balance') {
                            // Opening balance
                            $running_balance += $transaction->final_total;
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Opening Balance',
                                'transaction_mode' => 'Opening Balance',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => 0,
                                'credit' => $transaction->final_total > 0 ? $transaction->final_total : 0,
                                'balance' => $running_balance,
                            ];
                        }
                    } else {
                        if ($transaction->type == 'sell' && $transaction->status == $status_condition) {
                            // Sales transaction - debit (customer owes)
                            $running_balance += $transaction->final_total;
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Sales',
                                'transaction_mode' => 'Cash Sales',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => $transaction->final_total,
                                'credit' => 0,
                                'balance' => $running_balance,
                            ];
                        } elseif ($transaction->type == 'sell_return') {
                            // Sales return - credit (reduces customer debt)
                            $running_balance -= abs($transaction->final_total);
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Sales Return',
                                'transaction_mode' => 'Sales Return',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => 0,
                                'credit' => abs($transaction->final_total),
                                'balance' => $running_balance,
                            ];
                        } elseif ($transaction->type == 'opening_balance') {
                            // Opening balance
                            $running_balance += $transaction->final_total;
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Opening Balance',
                                'transaction_mode' => 'Opening Balance',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => $transaction->final_total > 0 ? $transaction->final_total : 0,
                                'credit' => $transaction->final_total < 0 ? abs($transaction->final_total) : 0,
                                'balance' => $running_balance,
                            ];
                        }
                    }
                }

                // Get all payments in date range
                // Apply location filter
                if ($permitted_locations != 'all') {
                    $payments_query = DB::table('transaction_payments as tp')
                        ->leftJoin('transactions as t', 'tp.transaction_id', '=', 't.id')
                        ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                        ->whereIn('t.location_id', $permitted_locations);
                } else {
                    $payments_query = DB::table('transaction_payments as tp')
                        ->leftJoin('transactions as t', 'tp.transaction_id', '=', 't.id')
                        ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id');
                }

                $payments = $payments_query
                    ->where('tp.business_id', $business_id)
                    ->where('tp.payment_for', $contact_id)
                    ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date])
                    ->whereNull('tp.parent_id'); // Exclude child payments to avoid duplicates

                if (!empty($location_id)) {
                    $payments->where('t.location_id', $location_id);
                }

                $payments = $payments
                    ->select(
                        'tp.id',
                        'tp.paid_on as transaction_date',
                        'tp.method',
                        'tp.amount',
                        'tp.is_return',
                        'tp.payment_ref_no',
                        't.invoice_no',
                        't.ref_no',
                        't.type as transaction_type',
                        DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as contact_name")
                    )
                    ->groupBy('tp.id', 'tp.paid_on', 'tp.method', 'tp.amount', 'tp.is_return', 'tp.payment_ref_no', 't.invoice_no', 't.ref_no', 't.type', 'c.supplier_business_name', 'c.name') // Ensure unique payments
                    ->orderBy('tp.paid_on', 'asc')
                    ->orderBy('tp.id', 'asc')
                    ->get();

                // Process payments
                foreach ($payments as $payment) {
                    $payment_amount = $payment->is_return == 1 ? -$payment->amount : $payment->amount;
                    $running_balance -= $payment_amount;

                    $payment_method = ucfirst(str_replace('_', ' ', $payment->method ?? 'cash'));
                    if ($payment->is_return == 1) {
                        $payment_method = 'Payment Return';
                    }

                    if ($contact_type == 'supplier') {
                        // For suppliers: payments reduce what we owe (debit)
                        $ledger_data[] = (object)[
                            'date' => $payment->transaction_date,
                            'type' => $payment->is_return == 1 ? 'Payment Return' : 'Payment',
                            'transaction_mode' => $payment->is_return == 1 ? 'Payment Return' : 'Paid By Cash',
                            'ac_name' => $payment->contact_name,
                            'ref_no' => $payment->payment_ref_no ?? '',
                            'voucher_no' => $payment->invoice_no ?? $payment->ref_no ?? '',
                            'debit' => abs($payment_amount),
                            'credit' => 0,
                            'balance' => $running_balance,
                        ];
                    } else {
                        // For customers: payments reduce what they owe (credit)
                        $ledger_data[] = (object)[
                            'date' => $payment->transaction_date,
                            'type' => $payment->is_return == 1 ? 'Payment Return' : 'Money Receipt',
                            'transaction_mode' => $payment->is_return == 1 ? 'Payment Return' : 'Received By Cash',
                            'ac_name' => $payment->contact_name,
                            'ref_no' => $payment->payment_ref_no ?? '',
                            'voucher_no' => $payment->invoice_no ?? $payment->ref_no ?? '',
                            'debit' => 0,
                            'credit' => abs($payment_amount),
                            'balance' => $running_balance,
                        ];
                    }
                }

                // Sort all entries - Balance Forward/Opening Balance first, then by date
                usort($ledger_data, function($a, $b) {
                    // Priority: Balance Forward and Opening Balance always first
                    $aIsOpening = ($a->type == 'Balance Forward' || $a->type == 'Opening Balance');
                    $bIsOpening = ($b->type == 'Balance Forward' || $b->type == 'Opening Balance');
                    
                    if ($aIsOpening && !$bIsOpening) {
                        return -1; // $a comes first
                    }
                    if (!$aIsOpening && $bIsOpening) {
                        return 1; // $b comes first
                    }
                    
                    // If both are opening or both are not, sort by date
                    $dateA = strtotime($a->date);
                    $dateB = strtotime($b->date);
                    if ($dateA == $dateB) {
                        return 0;
                    }
                    return ($dateA < $dateB) ? -1 : 1;
                });
            }
        }

        // Calculate grand totals
        $grand_total_debit = collect($ledger_data)->sum('debit');
        $grand_total_credit = collect($ledger_data)->sum('credit');
        $closing_balance = $running_balance;

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.customer_accounts_ledger', compact(
            'ledger_data',
            'start_date',
            'end_date',
            'contact_type',
            'customer_id',
            'supplier_id',
            'location_id',
            'locations',
            'customers',
            'suppliers',
            'customer',
            'opening_balance',
            'running_balance',
            'grand_total_debit',
            'grand_total_credit',
            'closing_balance',
            'util',
            'business'
        ));
    }

    /**
     * Print Accounts Ledger Report
     */
    public function printCustomerAccountsLedger(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        // Get contact type filter
        $contact_type = $request->get('contact_type', 'customer');
        $customer_id = $request->get('customer_id');
        $supplier_id = $request->get('supplier_id');
        $location_id = $request->get('location_id');

        // Get contact details
        $contact_id = ($contact_type == 'supplier') ? $supplier_id : $customer_id;
        $customer = null;
        $ledger_data = [];
        $opening_balance = 0;
        $running_balance = 0;

        if (!empty($contact_id)) {
            $customer = Contact::where('business_id', $business_id)
                ->where('type', $contact_type)
                ->where('id', $contact_id)
                ->first();

            if ($customer) {
                // Calculate opening balance (balance before start_date)
                $transaction_types = ($contact_type == 'supplier') ? ['purchase', 'opening_balance'] : ['sell', 'opening_balance'];
                $status_condition = ($contact_type == 'supplier') ? 'received' : 'final';
                
                // Apply location filter
                $permitted_locations = auth()->user()->permitted_locations();
                if ($permitted_locations != 'all') {
                    $opening_transactions_query = DB::table('transactions as t')
                        ->whereIn('t.location_id', $permitted_locations);
                } else {
                    $opening_transactions_query = DB::table('transactions as t');
                }

                if (!empty($location_id)) {
                    // Validate that user has access to the requested location
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        abort(403, 'Unauthorized location access.');
                    }
                }

                $opening_transactions = $opening_transactions_query
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $contact_id)
                    ->whereIn('t.type', $transaction_types)
                    ->whereDate('t.transaction_date', '<', $start_date)
                    ->where(function($q) use ($contact_type, $status_condition) {
                        if ($contact_type == 'supplier') {
                            $q->where(function($q2) use ($status_condition) {
                                $q2->where('t.type', 'purchase')->where('t.status', $status_condition);
                            })->orWhere('t.type', 'opening_balance');
                        } else {
                            $q->where(function($q2) use ($status_condition) {
                                $q2->where('t.type', 'sell')->where('t.status', $status_condition);
                            })->orWhere('t.type', 'opening_balance');
                        }
                    });

                if (!empty($location_id)) {
                    $opening_transactions->where('t.location_id', $location_id);
                }

                $opening_transactions = $opening_transactions
                    ->select('t.id', 't.type', 't.final_total')
                    ->get();

                $opening_invoice = 0;
                $opening_paid = 0;

                foreach ($opening_transactions as $transaction) {
                    if (in_array($transaction->type, $transaction_types)) {
                        $opening_invoice += $transaction->final_total ?? 0;
                    }

                    $payments = DB::table('transaction_payments')
                        ->where('transaction_id', $transaction->id)
                        ->whereDate('paid_on', '<', $start_date)
                        ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                    $opening_paid += $payments ?? 0;
                }

                $opening_balance = $opening_invoice - $opening_paid;
                $running_balance = $opening_balance;

                // Add opening balance row
                if ($opening_balance != 0) {
                    if ($contact_type == 'supplier') {
                        // For suppliers: positive = we owe (credit), negative = they owe us (debit)
                        $ledger_data[] = (object)[
                            'date' => $start_date,
                            'type' => 'Balance Forward',
                            'transaction_mode' => 'Balance Forward',
                            'ac_name' => '',
                            'ref_no' => '',
                            'voucher_no' => '',
                            'debit' => $opening_balance < 0 ? abs($opening_balance) : 0,
                            'credit' => $opening_balance > 0 ? abs($opening_balance) : 0,
                            'balance' => $running_balance,
                        ];
                    } else {
                        // For customers: positive = they owe us (debit), negative = we owe them (credit)
                        $ledger_data[] = (object)[
                            'date' => $start_date,
                            'type' => 'Balance Forward',
                            'transaction_mode' => 'Balance Forward',
                            'ac_name' => '',
                            'ref_no' => '',
                            'voucher_no' => '',
                            'debit' => $opening_balance > 0 ? abs($opening_balance) : 0,
                            'credit' => $opening_balance < 0 ? abs($opening_balance) : 0,
                            'balance' => $running_balance,
                        ];
                    }
                }

                // Get all transactions in date range
                // Get all transactions in date range
                $transaction_types_range = ($contact_type == 'supplier') 
                    ? ['purchase', 'opening_balance', 'purchase_return'] 
                    : ['sell', 'opening_balance', 'sell_return'];
                
                $transactions = DB::table('transactions as t')
                    ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $contact_id)
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where(function($q) use ($contact_type, $status_condition) {
                        if ($contact_type == 'supplier') {
                            $q->where(function($q2) use ($status_condition) {
                                $q2->where('t.type', 'purchase')->where('t.status', $status_condition);
                            })->orWhere('t.type', 'opening_balance')
                            ->orWhere('t.type', 'purchase_return');
                        } else {
                            $q->where(function($q2) use ($status_condition) {
                                $q2->where('t.type', 'sell')->where('t.status', $status_condition);
                            })->orWhere('t.type', 'opening_balance')
                            ->orWhere('t.type', 'sell_return');
                        }
                    })
                    ->select(
                        't.id',
                        't.transaction_date',
                        't.type',
                        't.status',
                        't.invoice_no',
                        't.ref_no',
                        't.final_total',
                        't.additional_notes',
                        DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as contact_name")
                    )
                    ->orderBy('t.transaction_date', 'asc')
                    ->orderBy('t.id', 'asc')
                    ->get();

                // Process transactions
                foreach ($transactions as $transaction) {
                    if ($contact_type == 'supplier') {
                        if ($transaction->type == 'purchase' && $transaction->status == $status_condition) {
                            // Purchase transaction - credit (we owe supplier)
                            $running_balance += $transaction->final_total;
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Purchase',
                                'transaction_mode' => 'Cash Purchase',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => 0,
                                'credit' => $transaction->final_total,
                                'balance' => $running_balance,
                            ];
                        } elseif ($transaction->type == 'purchase_return') {
                            // Purchase return - debit (reduces what we owe)
                            $running_balance -= abs($transaction->final_total);
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Purchase Return',
                                'transaction_mode' => 'Purchase Return',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => abs($transaction->final_total),
                                'credit' => 0,
                                'balance' => $running_balance,
                            ];
                        } elseif ($transaction->type == 'opening_balance') {
                            // Opening balance
                            $running_balance += $transaction->final_total;
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Opening Balance',
                                'transaction_mode' => 'Opening Balance',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => 0,
                                'credit' => $transaction->final_total > 0 ? $transaction->final_total : 0,
                                'balance' => $running_balance,
                            ];
                        }
                    } else {
                        if ($transaction->type == 'sell' && $transaction->status == $status_condition) {
                            // Sales transaction - debit (customer owes)
                            $running_balance += $transaction->final_total;
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Sales',
                                'transaction_mode' => 'Cash Sales',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => $transaction->final_total,
                                'credit' => 0,
                                'balance' => $running_balance,
                            ];
                        } elseif ($transaction->type == 'sell_return') {
                            // Sales return - credit (reduces customer debt)
                            $running_balance -= abs($transaction->final_total);
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Sales Return',
                                'transaction_mode' => 'Sales Return',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => 0,
                                'credit' => abs($transaction->final_total),
                                'balance' => $running_balance,
                            ];
                        } elseif ($transaction->type == 'opening_balance') {
                            // Opening balance
                            $running_balance += $transaction->final_total;
                            $ledger_data[] = (object)[
                                'date' => $transaction->transaction_date,
                                'type' => 'Opening Balance',
                                'transaction_mode' => 'Opening Balance',
                                'ac_name' => $transaction->contact_name,
                                'ref_no' => $transaction->ref_no ?? '',
                                'voucher_no' => $transaction->invoice_no ?? '',
                                'debit' => $transaction->final_total > 0 ? $transaction->final_total : 0,
                                'credit' => $transaction->final_total < 0 ? abs($transaction->final_total) : 0,
                                'balance' => $running_balance,
                            ];
                        }
                    }
                }

                // Get all payments in date range
                $payments = DB::table('transaction_payments as tp')
                    ->leftJoin('transactions as t', 'tp.transaction_id', '=', 't.id')
                    ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                    ->where('tp.business_id', $business_id)
                    ->where('tp.payment_for', $contact_id)
                    ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date])
                    ->whereNull('tp.parent_id') // Exclude child payments to avoid duplicates
                    ->select(
                        'tp.id',
                        'tp.paid_on as transaction_date',
                        'tp.method',
                        'tp.amount',
                        'tp.is_return',
                        'tp.payment_ref_no',
                        't.invoice_no',
                        't.ref_no',
                        't.type as transaction_type',
                        DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as contact_name")
                    )
                    ->groupBy('tp.id', 'tp.paid_on', 'tp.method', 'tp.amount', 'tp.is_return', 'tp.payment_ref_no', 't.invoice_no', 't.ref_no', 't.type', 'c.supplier_business_name', 'c.name') // Ensure unique payments
                    ->orderBy('tp.paid_on', 'asc')
                    ->orderBy('tp.id', 'asc')
                    ->get();

                // Process payments
                foreach ($payments as $payment) {
                    $payment_amount = $payment->is_return == 1 ? -$payment->amount : $payment->amount;
                    $running_balance -= $payment_amount;

                    $payment_method = ucfirst(str_replace('_', ' ', $payment->method ?? 'cash'));
                    if ($payment->is_return == 1) {
                        $payment_method = 'Payment Return';
                    }

                    if ($contact_type == 'supplier') {
                        // For suppliers: payments reduce what we owe (debit)
                        $ledger_data[] = (object)[
                            'date' => $payment->transaction_date,
                            'type' => $payment->is_return == 1 ? 'Payment Return' : 'Payment',
                            'transaction_mode' => $payment->is_return == 1 ? 'Payment Return' : 'Paid By Cash',
                            'ac_name' => $payment->contact_name,
                            'ref_no' => $payment->payment_ref_no ?? '',
                            'voucher_no' => $payment->invoice_no ?? $payment->ref_no ?? '',
                            'debit' => abs($payment_amount),
                            'credit' => 0,
                            'balance' => $running_balance,
                        ];
                    } else {
                        // For customers: payments reduce what they owe (credit)
                        $ledger_data[] = (object)[
                            'date' => $payment->transaction_date,
                            'type' => $payment->is_return == 1 ? 'Payment Return' : 'Money Receipt',
                            'transaction_mode' => $payment->is_return == 1 ? 'Payment Return' : 'Received By Cash',
                            'ac_name' => $payment->contact_name,
                            'ref_no' => $payment->payment_ref_no ?? '',
                            'voucher_no' => $payment->invoice_no ?? $payment->ref_no ?? '',
                            'debit' => 0,
                            'credit' => abs($payment_amount),
                            'balance' => $running_balance,
                        ];
                    }
                }

                // Sort all entries - Balance Forward/Opening Balance first, then by date
                usort($ledger_data, function($a, $b) {
                    // Priority: Balance Forward and Opening Balance always first
                    $aIsOpening = ($a->type == 'Balance Forward' || $a->type == 'Opening Balance');
                    $bIsOpening = ($b->type == 'Balance Forward' || $b->type == 'Opening Balance');
                    
                    if ($aIsOpening && !$bIsOpening) {
                        return -1; // $a comes first
                    }
                    if (!$aIsOpening && $bIsOpening) {
                        return 1; // $b comes first
                    }
                    
                    // If both are opening or both are not, sort by date
                    $dateA = strtotime($a->date);
                    $dateB = strtotime($b->date);
                    if ($dateA == $dateB) {
                        return 0;
                    }
                    return ($dateA < $dateB) ? -1 : 1;
                });
            }
        }

        // Calculate grand totals
        $grand_total_debit = collect($ledger_data)->sum('debit');
        $grand_total_credit = collect($ledger_data)->sum('credit');
        $closing_balance = $running_balance;

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_customer_accounts_ledger', compact(
            'ledger_data',
            'start_date',
            'end_date',
            'contact_type',
            'customer_id',
            'supplier_id',
            'customer',
            'opening_balance',
            'running_balance',
            'grand_total_debit',
            'grand_total_credit',
            'closing_balance',
            'util',
            'business'
        ));
    }

    /**
     * Cash Book Report
     */
    public function cashBook(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Check if Account module is enabled
        $util = new Util();
        $account_module_enabled = $util->isModuleEnabled('account', $business_id);
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        // Get filters
        $account_id = $request->get('account_id');
        $location_id = $request->get('location_id');

        // Get accounts for dropdown
        $accounts = Account::forDropdown($business_id, true, false, false);
        // Add "All" option
        $accounts = ['all' => __('lang_v1.all')] + $accounts;

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id, true);

        // Get account details
        $account = null;
        $ledger_data = [];
        $opening_balance = 0;
        $running_balance = 0;
        $show_all_accounts = ($account_id == 'all' || empty($account_id));

        if ($show_all_accounts) {
            // Show all accounts combined
            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            
            if (!empty($location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
            }

            // Calculate opening balance for all accounts (balance before start_date)
            $opening_transactions_query = DB::table('account_transactions as at')
                ->whereNull('at.deleted_at')
                ->whereDate('at.operation_date', '<', $start_date);

            // Apply location filter to opening balance through transactions
            if ($permitted_locations != 'all') {
                $opening_transactions_query->leftJoin('transactions as t', function($join) {
                    $join->on('t.id', '=', 'at.transaction_id');
                })
                ->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                $opening_transactions_query->leftJoin('transactions as t', function($join) {
                    $join->on('t.id', '=', 'at.transaction_id');
                })
                ->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            // In Cash Book: Debit = money in, Credit = money out
            // Account transactions: debit = money out, credit = money in
            $opening_transactions = $opening_transactions_query
                ->select(DB::raw("SUM(IF(at.type='credit', at.amount, -1 * at.amount)) as balance"))
                ->first();

            $opening_balance = $opening_transactions->balance ?? 0;
            $running_balance = $opening_balance;

            // Add opening balance row
            if ($opening_balance != 0) {
                $ledger_data[] = (object)[
                    'date' => $start_date,
                    'type' => 'Balance Forward',
                    'transaction_mode' => 'Balance Forward',
                    'ac_name' => 'All Accounts',
                    'narration' => '',
                    'ref_no' => '',
                    'voucher_no' => '',
                    'debit' => $opening_balance > 0 ? abs($opening_balance) : 0,
                    'credit' => $opening_balance < 0 ? abs($opening_balance) : 0,
                    'balance' => $running_balance,
                ];
            }

            // Get all account transactions in date range for all accounts
            $transactions_query = DB::table('account_transactions as at')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->whereNull('at.deleted_at')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            // Apply location filter
            if ($permitted_locations != 'all') {
                $transactions_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                $transactions_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $transactions = $transactions_query
                ->select(
                    'at.id',
                    'at.account_id',
                    'at.operation_date',
                    'at.type as transaction_type',
                    'at.sub_type',
                    'at.amount',
                    'at.reff_no',
                    'at.note',
                    't.id as transaction_id',
                    't.type as main_transaction_type',
                    't.invoice_no',
                    't.ref_no',
                    't.status',
                    'tp.method as payment_method',
                    'tp.is_return as payment_is_return',
                    'a.name as account_name',
                    DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as contact_name")
                )
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();

            // Process transactions
            foreach ($transactions as $transaction) {
                $transaction_type = '';
                $transaction_mode = '';
                $ac_name = '';
                $voucher_no = '';

                // Determine transaction type and mode based on sub_type and linked transaction
                if ($transaction->sub_type == 'opening_balance') {
                    $transaction_type = 'Opening Balance';
                    $transaction_mode = 'Opening Balance';
                    $ac_name = $transaction->account_name ?? 'All Accounts';
                } elseif ($transaction->sub_type == 'deposit') {
                    $transaction_type = 'Deposit';
                    $transaction_mode = 'Deposit';
                    $ac_name = $transaction->account_name ?? 'All Accounts';
                } elseif ($transaction->sub_type == 'fund_transfer') {
                    $transaction_type = 'Fund Transfer';
                    $transaction_mode = 'Fund Transfer';
                    $ac_name = $transaction->account_name ?? 'All Accounts';
                } elseif (!empty($transaction->transaction_id)) {
                    // Linked to a transaction
                    if ($transaction->main_transaction_type == 'sell') {
                        if ($transaction->payment_is_return == 1) {
                            $transaction_type = 'Sales Return';
                            $transaction_mode = 'Sales Return';
                        } else {
                            $transaction_type = 'Sales';
                            $transaction_mode = 'Cash Sales';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = ucfirst(str_replace('_', ' ', $transaction->payment_method)) . ' Sales';
                            }
                        }
                    } elseif ($transaction->main_transaction_type == 'purchase') {
                        $transaction_type = 'Purchase';
                        $transaction_mode = 'Paid By Cash';
                        if (!empty($transaction->payment_method)) {
                            $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                        }
                    } elseif ($transaction->main_transaction_type == 'expense') {
                        $transaction_type = 'Expense';
                        $transaction_mode = 'Paid By Cash';
                        if (!empty($transaction->payment_method)) {
                            $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                        }
                    } elseif ($transaction->main_transaction_type == 'sell_return') {
                        $transaction_type = 'Sales Return';
                        $transaction_mode = 'Sales Return';
                    } elseif ($transaction->main_transaction_type == 'purchase_return') {
                        $transaction_type = 'Purchase Return';
                        $transaction_mode = 'Purchase Return';
                    }

                    $ac_name = ($transaction->contact_name ?? $transaction->account_name) ?? 'All Accounts';
                    $voucher_no = $transaction->invoice_no ?? $transaction->ref_no ?? '';
                } elseif (!empty($transaction->payment_method)) {
                    // Direct payment transaction
                    // In Cash Book: credit = money in (debit), debit = money out (credit)
                    if ($transaction->transaction_type == 'credit') {
                        // Account credit = money coming in = Cash Book Debit = Money Receipt
                        $transaction_type = 'Money Receipt';
                        $transaction_mode = 'Received By Cash';
                        if (!empty($transaction->payment_method)) {
                            $transaction_mode = 'Received By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                        }
                    } else {
                        // Account debit = money going out = Cash Book Credit = Payment
                        $transaction_type = 'Payment';
                        $transaction_mode = 'Paid By Cash';
                        if (!empty($transaction->payment_method)) {
                            $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                        }
                    }
                    $ac_name = ($transaction->contact_name ?? $transaction->account_name) ?? 'All Accounts';
                } else {
                    // Generic transaction
                    $transaction_type = $transaction->transaction_type == 'debit' ? 'Debit' : 'Credit';
                    $transaction_mode = $transaction->transaction_type == 'debit' ? 'Debit Entry' : 'Credit Entry';
                    $ac_name = $transaction->account_name ?? 'All Accounts';
                }

                // Update running balance
                // In Cash Book: Debit = money in (increases cash), Credit = money out (decreases cash)
                // But account_transactions: debit = money out, credit = money in
                // So we need to reverse the logic
                if ($transaction->transaction_type == 'credit') {
                    // Account credit = money coming in = Cash Book Debit
                    $running_balance += $transaction->amount;
                } else {
                    // Account debit = money going out = Cash Book Credit
                    $running_balance -= $transaction->amount;
                }

                $ledger_data[] = (object)[
                    'date' => $transaction->operation_date,
                    'type' => $transaction_type,
                    'transaction_mode' => $transaction_mode,
                    'ac_name' => $ac_name,
                    'narration' => $transaction->note ?? '',
                    'ref_no' => $transaction->reff_no ?? '',
                    'voucher_no' => $voucher_no,
                    'debit' => $transaction->transaction_type == 'credit' ? $transaction->amount : 0,
                    'credit' => $transaction->transaction_type == 'debit' ? $transaction->amount : 0,
                    'balance' => $running_balance,
                ];
            }

            // Sort all entries - Balance Forward first, then by date
            usort($ledger_data, function($a, $b) {
                if ($a->type == 'Balance Forward' && $b->type != 'Balance Forward') {
                    return -1;
                }
                if ($a->type != 'Balance Forward' && $b->type == 'Balance Forward') {
                    return 1;
                }
                $dateA = strtotime($a->date);
                $dateB = strtotime($b->date);
                if ($dateA == $dateB) {
                    return 0;
                }
                return ($dateA < $dateB) ? -1 : 1;
            });
        } elseif (!empty($account_id)) {
            $account = Account::where('business_id', $business_id)
                ->where('id', $account_id)
                ->first();

            if ($account) {
                // Apply location filter
                $permitted_locations = auth()->user()->permitted_locations();
                
                if (!empty($location_id)) {
                    // Validate that user has access to the requested location
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        abort(403, 'Unauthorized location access.');
                    }
                }

                // Calculate opening balance (balance before start_date)
                $opening_transactions_query = DB::table('account_transactions as at')
                    ->where('at.account_id', $account_id)
                    ->whereNull('at.deleted_at')
                    ->whereDate('at.operation_date', '<', $start_date);

                // Apply location filter to opening balance through transactions
                if ($permitted_locations != 'all') {
                    $opening_transactions_query->leftJoin('transactions as t', function($join) {
                        $join->on('t.id', '=', 'at.transaction_id');
                    })
                    ->where(function($q) use ($permitted_locations) {
                        $q->whereIn('t.location_id', $permitted_locations)
                          ->orWhereNull('t.id');
                    });
                }

                if (!empty($location_id)) {
                    $opening_transactions_query->leftJoin('transactions as t', function($join) {
                        $join->on('t.id', '=', 'at.transaction_id');
                    })
                    ->where(function($q) use ($location_id) {
                        $q->where('t.location_id', $location_id)
                          ->orWhereNull('t.id');
                    });
                }

                // In Cash Book: Debit = money in, Credit = money out
                // Account transactions: debit = money out, credit = money in
                // So for opening balance: credit increases cash (debit in cash book), debit decreases cash (credit in cash book)
                $opening_transactions = $opening_transactions_query
                    ->select(DB::raw("SUM(IF(at.type='credit', at.amount, -1 * at.amount)) as balance"))
                    ->first();

                $opening_balance = $opening_transactions->balance ?? 0;
                $running_balance = $opening_balance;

                // Add opening balance row
                if ($opening_balance != 0) {
                    $ledger_data[] = (object)[
                        'date' => $start_date,
                        'type' => 'Balance Forward',
                        'transaction_mode' => 'Balance Forward',
                        'ac_name' => '',
                        'narration' => '',
                        'ref_no' => '',
                        'voucher_no' => '',
                        'debit' => $opening_balance > 0 ? abs($opening_balance) : 0,
                        'credit' => $opening_balance < 0 ? abs($opening_balance) : 0,
                        'balance' => $running_balance,
                    ];
                }

                // Get all account transactions in date range
                $transactions_query = DB::table('account_transactions as at')
                    ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
                    ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                    ->where('at.account_id', $account_id)
                    ->whereNull('at.deleted_at')
                    ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

                // Apply location filter
                if ($permitted_locations != 'all') {
                    $transactions_query->where(function($q) use ($permitted_locations) {
                        $q->whereIn('t.location_id', $permitted_locations)
                          ->orWhereNull('t.id');
                    });
                }

                if (!empty($location_id)) {
                    $transactions_query->where(function($q) use ($location_id) {
                        $q->where('t.location_id', $location_id)
                          ->orWhereNull('t.id');
                    });
                }

                $transactions = $transactions_query
                    ->select(
                        'at.id',
                        'at.operation_date',
                        'at.type as transaction_type',
                        'at.sub_type',
                        'at.amount',
                        'at.reff_no',
                        'at.note',
                        't.id as transaction_id',
                        't.type as main_transaction_type',
                        't.invoice_no',
                        't.ref_no',
                        't.status',
                        'tp.method as payment_method',
                        'tp.is_return as payment_is_return',
                        DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as contact_name")
                    )
                    ->orderBy('at.operation_date', 'asc')
                    ->orderBy('at.id', 'asc')
                    ->get();

                // Process transactions
                foreach ($transactions as $transaction) {
                    $transaction_type = '';
                    $transaction_mode = '';
                    $ac_name = '';
                    $voucher_no = '';

                    // Determine transaction type and mode based on sub_type and linked transaction
                    if ($transaction->sub_type == 'opening_balance') {
                        $transaction_type = 'Opening Balance';
                        $transaction_mode = 'Opening Balance';
                        $ac_name = $account->name;
                    } elseif ($transaction->sub_type == 'deposit') {
                        $transaction_type = 'Deposit';
                        $transaction_mode = 'Deposit';
                        $ac_name = $account->name;
                    } elseif ($transaction->sub_type == 'fund_transfer') {
                        $transaction_type = 'Fund Transfer';
                        $transaction_mode = 'Fund Transfer';
                        $ac_name = $account->name;
                    } elseif (!empty($transaction->transaction_id)) {
                        // Linked to a transaction
                        if ($transaction->main_transaction_type == 'sell') {
                            if ($transaction->payment_is_return == 1) {
                                $transaction_type = 'Sales Return';
                                $transaction_mode = 'Sales Return';
                            } else {
                                $transaction_type = 'Sales';
                                $transaction_mode = 'Cash Sales';
                                if (!empty($transaction->payment_method)) {
                                    $transaction_mode = ucfirst(str_replace('_', ' ', $transaction->payment_method)) . ' Sales';
                                }
                            }
                        } elseif ($transaction->main_transaction_type == 'purchase') {
                            $transaction_type = 'Purchase';
                            $transaction_mode = 'Paid By Cash';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                            }
                        } elseif ($transaction->main_transaction_type == 'expense') {
                            $transaction_type = 'Expense';
                            $transaction_mode = 'Paid By Cash';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                            }
                        } elseif ($transaction->main_transaction_type == 'sell_return') {
                            $transaction_type = 'Sales Return';
                            $transaction_mode = 'Sales Return';
                        } elseif ($transaction->main_transaction_type == 'purchase_return') {
                            $transaction_type = 'Purchase Return';
                            $transaction_mode = 'Purchase Return';
                        }

                        $ac_name = $transaction->contact_name ?? $account->name;
                        $voucher_no = $transaction->invoice_no ?? $transaction->ref_no ?? '';
                    } elseif (!empty($transaction->payment_method)) {
                        // Direct payment transaction
                        // In Cash Book: credit = money in (debit), debit = money out (credit)
                        if ($transaction->transaction_type == 'credit') {
                            // Account credit = money coming in = Cash Book Debit = Money Receipt
                            $transaction_type = 'Money Receipt';
                            $transaction_mode = 'Received By Cash';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = 'Received By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                            }
                        } else {
                            // Account debit = money going out = Cash Book Credit = Payment
                            $transaction_type = 'Payment';
                            $transaction_mode = 'Paid By Cash';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                            }
                        }
                        $ac_name = $transaction->contact_name ?? $account->name;
                    } else {
                        // Generic transaction
                        $transaction_type = $transaction->transaction_type == 'debit' ? 'Debit' : 'Credit';
                        $transaction_mode = $transaction->transaction_type == 'debit' ? 'Debit Entry' : 'Credit Entry';
                        $ac_name = $account->name;
                    }

                    // Update running balance
                    // In Cash Book: Debit = money in (increases cash), Credit = money out (decreases cash)
                    // But account_transactions: debit = money out, credit = money in
                    // So we need to reverse the logic
                    if ($transaction->transaction_type == 'credit') {
                        // Account credit = money coming in = Cash Book Debit
                        $running_balance += $transaction->amount;
                    } else {
                        // Account debit = money going out = Cash Book Credit
                        $running_balance -= $transaction->amount;
                    }

                    $ledger_data[] = (object)[
                        'date' => $transaction->operation_date,
                        'type' => $transaction_type,
                        'transaction_mode' => $transaction_mode,
                        'ac_name' => $ac_name,
                        'narration' => $transaction->note ?? '',
                        'ref_no' => $transaction->reff_no ?? '',
                        'voucher_no' => $voucher_no,
                        'debit' => $transaction->transaction_type == 'credit' ? $transaction->amount : 0,
                        'credit' => $transaction->transaction_type == 'debit' ? $transaction->amount : 0,
                        'balance' => $running_balance,
                    ];
                }

                // Sort all entries - Balance Forward first, then by date
                usort($ledger_data, function($a, $b) {
                    if ($a->type == 'Balance Forward' && $b->type != 'Balance Forward') {
                        return -1;
                    }
                    if ($a->type != 'Balance Forward' && $b->type == 'Balance Forward') {
                        return 1;
                    }
                    $dateA = strtotime($a->date);
                    $dateB = strtotime($b->date);
                    if ($dateA == $dateB) {
                        return 0;
                    }
                    return ($dateA < $dateB) ? -1 : 1;
                });
            }
        }

        // Calculate grand totals
        $grand_total_debit = collect($ledger_data)->sum('debit');
        $grand_total_credit = collect($ledger_data)->sum('credit');
        $closing_balance = $running_balance;

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.cash_book', compact(
            'ledger_data',
            'start_date',
            'end_date',
            'account_id',
            'location_id',
            'accounts',
            'locations',
            'account',
            'opening_balance',
            'running_balance',
            'grand_total_debit',
            'grand_total_credit',
            'closing_balance',
            'util',
            'business',
            'account_module_enabled',
            'show_all_accounts'
        ));
    }

    /**
     * Print Cash Book Report
     */
    public function printCashBook(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Check if Account module is enabled
        $util = new Util();
        $account_module_enabled = $util->isModuleEnabled('account', $business_id);
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        // Get filters
        $account_id = $request->get('account_id');
        $location_id = $request->get('location_id');

        // Get account details
        $account = null;
        $ledger_data = [];
        $opening_balance = 0;
        $running_balance = 0;
        $show_all_accounts = ($account_id == 'all' || empty($account_id));

        if ($show_all_accounts) {
            // Show all accounts combined - same logic as cashBook method
            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            
            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
            }

            // Calculate opening balance for all accounts
            $opening_transactions_query = DB::table('account_transactions as at')
                ->whereNull('at.deleted_at')
                ->whereDate('at.operation_date', '<', $start_date);

            if ($permitted_locations != 'all') {
                $opening_transactions_query->leftJoin('transactions as t', function($join) {
                    $join->on('t.id', '=', 'at.transaction_id');
                })
                ->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                $opening_transactions_query->leftJoin('transactions as t', function($join) {
                    $join->on('t.id', '=', 'at.transaction_id');
                })
                ->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $opening_transactions = $opening_transactions_query
                ->select(DB::raw("SUM(IF(at.type='credit', at.amount, -1 * at.amount)) as balance"))
                ->first();

            $opening_balance = $opening_transactions->balance ?? 0;
            $running_balance = $opening_balance;

            // Add opening balance row
            if ($opening_balance != 0) {
                $ledger_data[] = (object)[
                    'date' => $start_date,
                    'type' => 'Balance Forward',
                    'transaction_mode' => 'Balance Forward',
                    'ac_name' => 'All Accounts',
                    'narration' => '',
                    'ref_no' => '',
                    'voucher_no' => '',
                    'debit' => $opening_balance > 0 ? abs($opening_balance) : 0,
                    'credit' => $opening_balance < 0 ? abs($opening_balance) : 0,
                    'balance' => $running_balance,
                ];
            }

            // Get all account transactions in date range for all accounts
            $transactions_query = DB::table('account_transactions as at')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->whereNull('at.deleted_at')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $transactions_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                $transactions_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $transactions = $transactions_query
                ->select(
                    'at.id',
                    'at.account_id',
                    'at.operation_date',
                    'at.type as transaction_type',
                    'at.sub_type',
                    'at.amount',
                    'at.reff_no',
                    'at.note',
                    't.id as transaction_id',
                    't.type as main_transaction_type',
                    't.invoice_no',
                    't.ref_no',
                    't.status',
                    'tp.method as payment_method',
                    'tp.is_return as payment_is_return',
                    'a.name as account_name',
                    DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as contact_name")
                )
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();

            // Process transactions - same logic as cashBook method
            foreach ($transactions as $transaction) {
                $transaction_type = '';
                $transaction_mode = '';
                $ac_name = '';
                $voucher_no = '';

                if ($transaction->sub_type == 'opening_balance') {
                    $transaction_type = 'Opening Balance';
                    $transaction_mode = 'Opening Balance';
                    $ac_name = $transaction->account_name ?? 'All Accounts';
                } elseif ($transaction->sub_type == 'deposit') {
                    $transaction_type = 'Deposit';
                    $transaction_mode = 'Deposit';
                    $ac_name = $transaction->account_name ?? 'All Accounts';
                } elseif ($transaction->sub_type == 'fund_transfer') {
                    $transaction_type = 'Fund Transfer';
                    $transaction_mode = 'Fund Transfer';
                    $ac_name = $transaction->account_name ?? 'All Accounts';
                } elseif (!empty($transaction->transaction_id)) {
                    if ($transaction->main_transaction_type == 'sell') {
                        if ($transaction->payment_is_return == 1) {
                            $transaction_type = 'Sales Return';
                            $transaction_mode = 'Sales Return';
                        } else {
                            $transaction_type = 'Sales';
                            $transaction_mode = 'Cash Sales';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = ucfirst(str_replace('_', ' ', $transaction->payment_method)) . ' Sales';
                            }
                        }
                    } elseif ($transaction->main_transaction_type == 'purchase') {
                        $transaction_type = 'Purchase';
                        $transaction_mode = 'Paid By Cash';
                        if (!empty($transaction->payment_method)) {
                            $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                        }
                    } elseif ($transaction->main_transaction_type == 'expense') {
                        $transaction_type = 'Expense';
                        $transaction_mode = 'Paid By Cash';
                        if (!empty($transaction->payment_method)) {
                            $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                        }
                    } elseif ($transaction->main_transaction_type == 'sell_return') {
                        $transaction_type = 'Sales Return';
                        $transaction_mode = 'Sales Return';
                    } elseif ($transaction->main_transaction_type == 'purchase_return') {
                        $transaction_type = 'Purchase Return';
                        $transaction_mode = 'Purchase Return';
                    }

                    $ac_name = ($transaction->contact_name ?? $transaction->account_name) ?? 'All Accounts';
                    $voucher_no = $transaction->invoice_no ?? $transaction->ref_no ?? '';
                } elseif (!empty($transaction->payment_method)) {
                    if ($transaction->transaction_type == 'credit') {
                        $transaction_type = 'Money Receipt';
                        $transaction_mode = 'Received By Cash';
                        if (!empty($transaction->payment_method)) {
                            $transaction_mode = 'Received By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                        }
                    } else {
                        $transaction_type = 'Payment';
                        $transaction_mode = 'Paid By Cash';
                        if (!empty($transaction->payment_method)) {
                            $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                        }
                    }
                    $ac_name = ($transaction->contact_name ?? $transaction->account_name) ?? 'All Accounts';
                } else {
                    $transaction_type = $transaction->transaction_type == 'debit' ? 'Debit' : 'Credit';
                    $transaction_mode = $transaction->transaction_type == 'debit' ? 'Debit Entry' : 'Credit Entry';
                    $ac_name = $transaction->account_name ?? 'All Accounts';
                }

                if ($transaction->transaction_type == 'credit') {
                    $running_balance += $transaction->amount;
                } else {
                    $running_balance -= $transaction->amount;
                }

                $ledger_data[] = (object)[
                    'date' => $transaction->operation_date,
                    'type' => $transaction_type,
                    'transaction_mode' => $transaction_mode,
                    'ac_name' => $ac_name,
                    'narration' => $transaction->note ?? '',
                    'ref_no' => $transaction->reff_no ?? '',
                    'voucher_no' => $voucher_no,
                    'debit' => $transaction->transaction_type == 'credit' ? $transaction->amount : 0,
                    'credit' => $transaction->transaction_type == 'debit' ? $transaction->amount : 0,
                    'balance' => $running_balance,
                ];
            }

            usort($ledger_data, function($a, $b) {
                if ($a->type == 'Balance Forward' && $b->type != 'Balance Forward') {
                    return -1;
                }
                if ($a->type != 'Balance Forward' && $b->type == 'Balance Forward') {
                    return 1;
                }
                $dateA = strtotime($a->date);
                $dateB = strtotime($b->date);
                if ($dateA == $dateB) {
                    return 0;
                }
                return ($dateA < $dateB) ? -1 : 1;
            });
        } elseif (!empty($account_id)) {
            $account = Account::where('business_id', $business_id)
                ->where('id', $account_id)
                ->first();

            if ($account) {
                // Apply location filter
                $permitted_locations = auth()->user()->permitted_locations();
                
                if (!empty($location_id)) {
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        abort(403, 'Unauthorized location access.');
                    }
                }

                // Calculate opening balance (balance before start_date)
                $opening_transactions_query = DB::table('account_transactions as at')
                    ->where('at.account_id', $account_id)
                    ->whereNull('at.deleted_at')
                    ->whereDate('at.operation_date', '<', $start_date);

                if ($permitted_locations != 'all') {
                    $opening_transactions_query->leftJoin('transactions as t', function($join) {
                        $join->on('t.id', '=', 'at.transaction_id');
                    })
                    ->where(function($q) use ($permitted_locations) {
                        $q->whereIn('t.location_id', $permitted_locations)
                          ->orWhereNull('t.id');
                    });
                }

                if (!empty($location_id)) {
                    $opening_transactions_query->leftJoin('transactions as t', function($join) {
                        $join->on('t.id', '=', 'at.transaction_id');
                    })
                    ->where(function($q) use ($location_id) {
                        $q->where('t.location_id', $location_id)
                          ->orWhereNull('t.id');
                    });
                }

                // In Cash Book: Debit = money in, Credit = money out
                // Account transactions: debit = money out, credit = money in
                // So for opening balance: credit increases cash (debit in cash book), debit decreases cash (credit in cash book)
                $opening_transactions = $opening_transactions_query
                    ->select(DB::raw("SUM(IF(at.type='credit', at.amount, -1 * at.amount)) as balance"))
                    ->first();

                $opening_balance = $opening_transactions->balance ?? 0;
                $running_balance = $opening_balance;

                // Add opening balance row
                if ($opening_balance != 0) {
                    $ledger_data[] = (object)[
                        'date' => $start_date,
                        'type' => 'Balance Forward',
                        'transaction_mode' => 'Balance Forward',
                        'ac_name' => '',
                        'narration' => '',
                        'ref_no' => '',
                        'voucher_no' => '',
                        'debit' => $opening_balance > 0 ? abs($opening_balance) : 0,
                        'credit' => $opening_balance < 0 ? abs($opening_balance) : 0,
                        'balance' => $running_balance,
                    ];
                }

                // Get all account transactions in date range
                $transactions_query = DB::table('account_transactions as at')
                    ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
                    ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                    ->where('at.account_id', $account_id)
                    ->whereNull('at.deleted_at')
                    ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

                if ($permitted_locations != 'all') {
                    $transactions_query->where(function($q) use ($permitted_locations) {
                        $q->whereIn('t.location_id', $permitted_locations)
                          ->orWhereNull('t.id');
                    });
                }

                if (!empty($location_id)) {
                    $transactions_query->where(function($q) use ($location_id) {
                        $q->where('t.location_id', $location_id)
                          ->orWhereNull('t.id');
                    });
                }

                $transactions = $transactions_query
                    ->select(
                        'at.id',
                        'at.operation_date',
                        'at.type as transaction_type',
                        'at.sub_type',
                        'at.amount',
                        'at.reff_no',
                        'at.note',
                        't.id as transaction_id',
                        't.type as main_transaction_type',
                        't.invoice_no',
                        't.ref_no',
                        't.status',
                        'tp.method as payment_method',
                        'tp.is_return as payment_is_return',
                        DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as contact_name")
                    )
                    ->orderBy('at.operation_date', 'asc')
                    ->orderBy('at.id', 'asc')
                    ->get();

                // Process transactions
                foreach ($transactions as $transaction) {
                    $transaction_type = '';
                    $transaction_mode = '';
                    $ac_name = '';
                    $voucher_no = '';

                    if ($transaction->sub_type == 'opening_balance') {
                        $transaction_type = 'Opening Balance';
                        $transaction_mode = 'Opening Balance';
                        $ac_name = $account->name;
                    } elseif ($transaction->sub_type == 'deposit') {
                        $transaction_type = 'Deposit';
                        $transaction_mode = 'Deposit';
                        $ac_name = $account->name;
                    } elseif ($transaction->sub_type == 'fund_transfer') {
                        $transaction_type = 'Fund Transfer';
                        $transaction_mode = 'Fund Transfer';
                        $ac_name = $account->name;
                    } elseif (!empty($transaction->transaction_id)) {
                        if ($transaction->main_transaction_type == 'sell') {
                            if ($transaction->payment_is_return == 1) {
                                $transaction_type = 'Sales Return';
                                $transaction_mode = 'Sales Return';
                            } else {
                                $transaction_type = 'Sales';
                                $transaction_mode = 'Cash Sales';
                                if (!empty($transaction->payment_method)) {
                                    $transaction_mode = ucfirst(str_replace('_', ' ', $transaction->payment_method)) . ' Sales';
                                }
                            }
                        } elseif ($transaction->main_transaction_type == 'purchase') {
                            $transaction_type = 'Purchase';
                            $transaction_mode = 'Paid By Cash';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                            }
                        } elseif ($transaction->main_transaction_type == 'expense') {
                            $transaction_type = 'Expense';
                            $transaction_mode = 'Paid By Cash';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                            }
                        } elseif ($transaction->main_transaction_type == 'sell_return') {
                            $transaction_type = 'Sales Return';
                            $transaction_mode = 'Sales Return';
                        } elseif ($transaction->main_transaction_type == 'purchase_return') {
                            $transaction_type = 'Purchase Return';
                            $transaction_mode = 'Purchase Return';
                        }

                        $ac_name = $transaction->contact_name ?? $account->name;
                        $voucher_no = $transaction->invoice_no ?? $transaction->ref_no ?? '';
                    } elseif (!empty($transaction->payment_method)) {
                        if ($transaction->transaction_type == 'debit') {
                            $transaction_type = 'Money Receipt';
                            $transaction_mode = 'Received By Cash';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = 'Received By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                            }
                        } else {
                            $transaction_type = 'Payment';
                            $transaction_mode = 'Paid By Cash';
                            if (!empty($transaction->payment_method)) {
                                $transaction_mode = 'Paid By ' . ucfirst(str_replace('_', ' ', $transaction->payment_method));
                            }
                        }
                        $ac_name = $transaction->contact_name ?? $account->name;
                    } else {
                        $transaction_type = $transaction->transaction_type == 'debit' ? 'Debit' : 'Credit';
                        $transaction_mode = $transaction->transaction_type == 'debit' ? 'Debit Entry' : 'Credit Entry';
                        $ac_name = $account->name;
                    }

                    if ($transaction->transaction_type == 'debit') {
                        $running_balance += $transaction->amount;
                    } else {
                        $running_balance -= $transaction->amount;
                    }

                    $ledger_data[] = (object)[
                        'date' => $transaction->operation_date,
                        'type' => $transaction_type,
                        'transaction_mode' => $transaction_mode,
                        'ac_name' => $ac_name,
                        'narration' => $transaction->note ?? '',
                        'ref_no' => $transaction->reff_no ?? '',
                        'voucher_no' => $voucher_no,
                        'debit' => $transaction->transaction_type == 'debit' ? $transaction->amount : 0,
                        'credit' => $transaction->transaction_type == 'credit' ? $transaction->amount : 0,
                        'balance' => $running_balance,
                    ];
                }

                usort($ledger_data, function($a, $b) {
                    if ($a->type == 'Balance Forward' && $b->type != 'Balance Forward') {
                        return -1;
                    }
                    if ($a->type != 'Balance Forward' && $b->type == 'Balance Forward') {
                        return 1;
                    }
                    $dateA = strtotime($a->date);
                    $dateB = strtotime($b->date);
                    if ($dateA == $dateB) {
                        return 0;
                    }
                    return ($dateA < $dateB) ? -1 : 1;
                });
            }
        }

        // Calculate grand totals
        $grand_total_debit = collect($ledger_data)->sum('debit');
        $grand_total_credit = collect($ledger_data)->sum('credit');
        $closing_balance = $running_balance;

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_cash_book', compact(
            'ledger_data',
            'start_date',
            'end_date',
            'account_id',
            'location_id',
            'account',
            'opening_balance',
            'running_balance',
            'grand_total_debit',
            'grand_total_credit',
            'closing_balance',
            'util',
            'business',
            'account_module_enabled',
            'show_all_accounts'
        ));
    }

    /**
     * Date Wise Credit Voucher Report
     */
    public function dateWiseCreditVoucher(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }

        // Check if Account module is enabled
        $util = new Util();
        $account_module_enabled = $util->isModuleEnabled('account', $business_id);

        // Get location filter
        $location_id = $request->get('location_id');

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id, true);

        // Get credit vouchers (money receipts) - account_transactions with type='credit'
        $vouchers = [];
        $total_amount = 0;

        if ($account_module_enabled) {
            $permitted_locations = auth()->user()->permitted_locations();
            
            $vouchers_query = DB::table('account_transactions as at')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
                ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
                ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
                ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'credit') // Credit = money received
                ->whereNull('at.deleted_at')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            // Apply location filter
            if ($permitted_locations != 'all') {
                $vouchers_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $vouchers_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $vouchers = $vouchers_query
                ->select(
                    'at.id',
                    'at.operation_date',
                    'at.amount',
                    'at.reff_no',
                    'at.note as narration',
                    'a.name as account_name',
                    'a.account_number',
                    'c_t.name as contact_name_t',
                    'c_t.supplier_business_name as supplier_business_name_t',
                    'c_p.name as contact_name_p',
                    'c_p.supplier_business_name as supplier_business_name_p',
                    't.invoice_no',
                    't.ref_no',
                    'tp.method as payment_method',
                    'tp.cheque_number',
                    'tp.card_number',
                    'tp.bank_account_number',
                    'tp.payment_ref_no',
                    DB::raw("NULL as cheque_date"),
                    DB::raw("COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name, 'Walk-In Customer') as particulars"),
                    DB::raw("TRIM(CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, ''))) as entered_by"),
                    DB::raw("CASE 
                        WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                        WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                        WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                        ELSE CONCAT('CRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                    END as voucher_id"),
                    DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                    DB::raw("CASE 
                        WHEN a.name IS NOT NULL AND a.account_number IS NOT NULL THEN CONCAT(a.name, ' & ', a.account_number)
                        WHEN a.name IS NOT NULL THEN a.name
                        ELSE ''
                    END as bank_name_account")
                )
                ->groupBy('at.id', 'at.operation_date', 'at.amount', 'at.reff_no', 'at.note', 'a.name', 'a.account_number', 'c_t.name', 'c_t.supplier_business_name', 'c_p.name', 'c_p.supplier_business_name', 't.invoice_no', 't.ref_no', 'tp.method', 'tp.cheque_number', 'tp.card_number', 'tp.bank_account_number', 'tp.payment_ref_no', 'u.surname', 'u.first_name', 'u.last_name')
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();

            $total_amount = $vouchers->sum('amount');
        }

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.date_wise_credit_voucher', compact(
            'vouchers',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'total_amount',
            'util',
            'business',
            'account_module_enabled'
        ));
    }

    /**
     * Print Date Wise Credit Voucher Report
     */
    public function printDateWiseCreditVoucher(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }

        // Check if Account module is enabled
        $util = new Util();
        $account_module_enabled = $util->isModuleEnabled('account', $business_id);

        // Get location filter
        $location_id = $request->get('location_id');

        // Get credit vouchers (money receipts)
        $vouchers = [];
        $total_amount = 0;

        if ($account_module_enabled) {
            $permitted_locations = auth()->user()->permitted_locations();
            
            $vouchers_query = DB::table('account_transactions as at')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
                ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
                ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
                ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'credit')
                ->whereNull('at.deleted_at')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $vouchers_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $vouchers_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $vouchers = $vouchers_query
                ->select(
                    'at.id',
                    'at.operation_date',
                    'at.amount',
                    'at.reff_no',
                    'at.note as narration',
                    'a.name as account_name',
                    'a.account_number',
                    'c_t.name as contact_name_t',
                    'c_t.supplier_business_name as supplier_business_name_t',
                    'c_p.name as contact_name_p',
                    'c_p.supplier_business_name as supplier_business_name_p',
                    't.invoice_no',
                    't.ref_no',
                    'tp.method as payment_method',
                    'tp.cheque_number',
                    'tp.card_number',
                    'tp.bank_account_number',
                    'tp.payment_ref_no',
                    DB::raw("NULL as cheque_date"),
                    DB::raw("COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name, 'Walk-In Customer') as particulars"),
                    DB::raw("TRIM(CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, ''))) as entered_by"),
                    DB::raw("CASE 
                        WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                        WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                        WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                        ELSE CONCAT('CRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                    END as voucher_id"),
                    DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                    DB::raw("CASE 
                        WHEN a.name IS NOT NULL AND a.account_number IS NOT NULL THEN CONCAT(a.name, ' & ', a.account_number)
                        WHEN a.name IS NOT NULL THEN a.name
                        ELSE ''
                    END as bank_name_account")
                )
                ->groupBy('at.id', 'at.operation_date', 'at.amount', 'at.reff_no', 'at.note', 'a.name', 'a.account_number', 'c_t.name', 'c_t.supplier_business_name', 'c_p.name', 'c_p.supplier_business_name', 't.invoice_no', 't.ref_no', 'tp.method', 'tp.cheque_number', 'tp.card_number', 'tp.bank_account_number', 'tp.payment_ref_no', 'u.surname', 'u.first_name', 'u.last_name')
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();

            $total_amount = $vouchers->sum('amount');
        }

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_date_wise_credit_voucher', compact(
            'vouchers',
            'start_date',
            'end_date',
            'location_id',
            'total_amount',
            'util',
            'business',
            'account_module_enabled'
        ));
    }

    /**
     * Date Wise Debit Voucher Report
     */
    public function dateWiseDebitVoucher(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }

        // Check if Account module is enabled
        $util = new Util();
        $account_module_enabled = $util->isModuleEnabled('account', $business_id);

        // Get location filter
        $location_id = $request->get('location_id');

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id, true);

        // Get debit vouchers (money payments) - account_transactions with type='debit'
        $supplier_customer_payments = [];
        $general_ac_payments = [];
        $total_supplier_customer_amount = 0;
        $total_general_ac_amount = 0;

        if ($account_module_enabled) {
            $permitted_locations = auth()->user()->permitted_locations();
            
            $vouchers_query = DB::table('account_transactions as at')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
                ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
                ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
                ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
                ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'debit') // Debit = money paid out
                ->whereNull('at.deleted_at')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            // Apply location filter
            if ($permitted_locations != 'all') {
                $vouchers_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $vouchers_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $vouchers = $vouchers_query
                ->select(
                    'at.id',
                    'at.operation_date',
                    'at.amount',
                    'at.reff_no',
                    'at.note as narration',
                    'at.sub_type',
                    'a.name as account_name',
                    'a.account_number',
                    'c_t.name as contact_name_t',
                    'c_t.supplier_business_name as supplier_business_name_t',
                    'c_p.name as contact_name_p',
                    'c_p.supplier_business_name as supplier_business_name_p',
                    't.id as transaction_id',
                    't.type as transaction_type',
                    't.invoice_no',
                    't.ref_no',
                    'tp.method as payment_method',
                    'tp.cheque_number',
                    'tp.card_number',
                    'tp.bank_account_number',
                    'tp.payment_ref_no',
                    DB::raw("NULL as cheque_date"),
                    DB::raw("COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name, 'Walk-In Customer') as particulars"),
                    DB::raw("TRIM(CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, ''))) as prepared_by"),
                    DB::raw("CASE 
                        WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                        WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                        WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                        ELSE CONCAT('DRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                    END as voucher_id"),
                    DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                    DB::raw("CASE 
                        WHEN a.name IS NOT NULL AND a.account_number IS NOT NULL THEN CONCAT(a.name, ' & ', a.account_number)
                        WHEN a.name IS NOT NULL THEN a.name
                        ELSE ''
                    END as bank_name_account")
                )
                ->groupBy('at.id', 'at.operation_date', 'at.amount', 'at.reff_no', 'at.note', 'at.sub_type', 'a.name', 'a.account_number', 'c_t.name', 'c_t.supplier_business_name', 'c_p.name', 'c_p.supplier_business_name', 't.id', 't.type', 't.invoice_no', 't.ref_no', 'tp.method', 'tp.cheque_number', 'tp.card_number', 'tp.bank_account_number', 'tp.payment_ref_no', 'ec.name', 'u.surname', 'u.first_name', 'u.last_name')
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();

            // Separate into Supplier/Customer Payment and General A/C Payment
            foreach ($vouchers as $voucher) {
                // Determine if it's Supplier/Customer Payment or General A/C Payment
                $is_supplier_customer = false;
                
                if (!empty($voucher->transaction_id)) {
                    // Linked to a transaction
                    // Supplier/Customer Payment: purchase, purchase_return, sell, sell_return
                    // General A/C Payment: expense, payroll, expense_refund, fund_transfer, deposit, opening_balance, and others
                    if (in_array($voucher->transaction_type, ['purchase', 'purchase_return', 'sell', 'sell_return'])) {
                        // Supplier/Customer Payment
                        $is_supplier_customer = true;
                    } else {
                        // All other transaction types (expense, payroll, etc.) go to General A/C Payment
                        $is_supplier_customer = false;
                    }
                } elseif (!empty($voucher->sub_type)) {
                    // Account transaction sub_type (fund_transfer, deposit, opening_balance)
                    // These go to General A/C Payment
                    $is_supplier_customer = false;
                } elseif (!empty($voucher->contact_name_t) || !empty($voucher->contact_name_p)) {
                    // Has contact but no transaction - could be direct payment to supplier/customer
                    $is_supplier_customer = true;
                } else {
                    // No transaction, no contact - General A/C Payment
                    $is_supplier_customer = false;
                }

                if ($is_supplier_customer) {
                    $supplier_customer_payments[] = $voucher;
                    $total_supplier_customer_amount += $voucher->amount;
                } else {
                    $general_ac_payments[] = $voucher;
                    $total_general_ac_amount += $voucher->amount;
                }
            }
        }

        $total_amount = $total_supplier_customer_amount + $total_general_ac_amount;
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.date_wise_debit_voucher', compact(
            'supplier_customer_payments',
            'general_ac_payments',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'total_supplier_customer_amount',
            'total_general_ac_amount',
            'total_amount',
            'util',
            'business',
            'account_module_enabled'
        ));
    }

    /**
     * Print Date Wise Debit Voucher Report
     */
    public function printDateWiseDebitVoucher(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }

        // Check if Account module is enabled
        $util = new Util();
        $account_module_enabled = $util->isModuleEnabled('account', $business_id);

        // Get location filter
        $location_id = $request->get('location_id');

        // Get debit vouchers (money payments)
        $supplier_customer_payments = [];
        $general_ac_payments = [];
        $total_supplier_customer_amount = 0;
        $total_general_ac_amount = 0;

        if ($account_module_enabled) {
            $permitted_locations = auth()->user()->permitted_locations();
            
            $vouchers_query = DB::table('account_transactions as at')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
                ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
                ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
                ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
                ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'debit')
                ->whereNull('at.deleted_at')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $vouchers_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $vouchers_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $vouchers = $vouchers_query
                ->select(
                    'at.id',
                    'at.operation_date',
                    'at.amount',
                    'at.reff_no',
                    'at.note as narration',
                    'at.sub_type',
                    'a.name as account_name',
                    'a.account_number',
                    'c_t.name as contact_name_t',
                    'c_t.supplier_business_name as supplier_business_name_t',
                    'c_p.name as contact_name_p',
                    'c_p.supplier_business_name as supplier_business_name_p',
                    't.id as transaction_id',
                    't.type as transaction_type',
                    't.invoice_no',
                    't.ref_no',
                    'tp.method as payment_method',
                    'tp.cheque_number',
                    'tp.card_number',
                    'tp.bank_account_number',
                    'tp.payment_ref_no',
                    DB::raw("NULL as cheque_date"),
                    'ec.name as expense_category_name',
                    DB::raw("CASE 
                        WHEN t.type = 'expense' AND ec.name IS NOT NULL AND ec.name != '' THEN ec.name
                        WHEN t.type = 'expense' AND c_t.supplier_business_name IS NOT NULL AND c_t.supplier_business_name != '' THEN c_t.supplier_business_name
                        WHEN t.type = 'expense' AND c_t.name IS NOT NULL AND c_t.name != '' THEN c_t.name
                        WHEN COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name) IS NOT NULL 
                            AND COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name) != ''
                            THEN COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name)
                        WHEN at.sub_type = 'fund_transfer' THEN 'Fund Transfer'
                        WHEN at.sub_type = 'deposit' THEN 'Deposit'
                        WHEN at.sub_type = 'opening_balance' THEN 'Opening Balance'
                        ELSE 'Walk-In Customer'
                    END as particulars"),
                    DB::raw("TRIM(CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, ''))) as prepared_by"),
                    DB::raw("CASE 
                        WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                        WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                        WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                        ELSE CONCAT('DRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                    END as voucher_id"),
                    DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                    DB::raw("CASE 
                        WHEN a.name IS NOT NULL AND a.account_number IS NOT NULL THEN CONCAT(a.name, ' & ', a.account_number)
                        WHEN a.name IS NOT NULL THEN a.name
                        ELSE ''
                    END as bank_name_account")
                )
                ->groupBy('at.id', 'at.operation_date', 'at.amount', 'at.reff_no', 'at.note', 'at.sub_type', 'a.name', 'a.account_number', 'c_t.name', 'c_t.supplier_business_name', 'c_p.name', 'c_p.supplier_business_name', 't.id', 't.type', 't.invoice_no', 't.ref_no', 'tp.method', 'tp.cheque_number', 'tp.card_number', 'tp.bank_account_number', 'tp.payment_ref_no', 'ec.name', 'u.surname', 'u.first_name', 'u.last_name')
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();

            // Separate into Supplier/Customer Payment and General A/C Payment
            foreach ($vouchers as $voucher) {
                $is_supplier_customer = false;
                
                if (!empty($voucher->transaction_id)) {
                    if (in_array($voucher->transaction_type, ['purchase', 'purchase_return', 'sell', 'sell_return'])) {
                        $is_supplier_customer = true;
                    }
                } elseif (!empty($voucher->contact_name_t) || !empty($voucher->contact_name_p)) {
                    $is_supplier_customer = true;
                }

                if ($is_supplier_customer) {
                    $supplier_customer_payments[] = $voucher;
                    $total_supplier_customer_amount += $voucher->amount;
                } else {
                    $general_ac_payments[] = $voucher;
                    $total_general_ac_amount += $voucher->amount;
                }
            }
        }

        $total_amount = $total_supplier_customer_amount + $total_general_ac_amount;
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_date_wise_debit_voucher', compact(
            'supplier_customer_payments',
            'general_ac_payments',
            'start_date',
            'end_date',
            'location_id',
            'total_supplier_customer_amount',
            'total_general_ac_amount',
            'total_amount',
            'util',
            'business',
            'account_module_enabled'
        ));
    }

    /**
     * Income Statement Report
     */
    public function incomeStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }

        // Get location filter
        $location_id = $request->get('location_id');

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id, true);

        // Get profit/loss data using TransactionUtil
        $transactionUtil = new \App\Utils\TransactionUtil();
        $permitted_locations = auth()->user()->permitted_locations();
        
        $pl_data = $transactionUtil->getProfitLossDetails(
            $business_id,
            $location_id,
            $start_date,
            $end_date,
            null,
            $permitted_locations
        );

        // Extract data
        $sales_accounts = (float) ($pl_data['total_sell'] ?? 0);
        $opening_stock = (float) ($pl_data['opening_stock'] ?? 0);
        $total_purchase = (float) ($pl_data['total_purchase'] ?? 0);
        $closing_stock = (float) ($pl_data['closing_stock'] ?? 0);
        $cost_of_goods_sold = $opening_stock + $total_purchase - $closing_stock;
        $sales_return = (float) ($pl_data['total_sell_return'] ?? 0);
        $purchase_return = (float) ($pl_data['total_purchase_return'] ?? 0);
        $cost_of_goods_sales_return = $purchase_return;
        $extracharges_on_sales = (float) ($pl_data['total_sell_additional_expense'] ?? 0);
        $extracharges_on_purchase = (float) ($pl_data['total_purchase_additional_expense'] ?? 0);
        $discount_on_sales = (float) ($pl_data['total_sell_discount'] ?? 0);
        
        // Calculate Gross Profit
        $gross_profit = $sales_accounts - $cost_of_goods_sold - $sales_return + $cost_of_goods_sales_return + $extracharges_on_sales - $extracharges_on_purchase - $discount_on_sales;

        // Get expenses grouped by category
        $expenses_by_category = [];
        $total_expense = 0;

        $expenses_query = DB::table('transactions as t')
            ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'expense')
            ->where('t.status', 'final')
            ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

        // Apply location filter
        if ($permitted_locations != 'all') {
            $expenses_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $expenses_query->where('t.location_id', $location_id);
        }

        $expenses = $expenses_query
            ->select(
                'ec.id as category_id',
                'ec.name as category_name',
                't.id',
                't.final_total',
                't.ref_no',
                't.additional_notes',
                DB::raw("COALESCE(ec.name, 'Uncategorized') as expense_category")
            )
            ->orderBy('ec.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->get();

        // Group expenses by category
        foreach ($expenses as $expense) {
            $category_name = $expense->expense_category;
            
            if (!isset($expenses_by_category[$category_name])) {
                $expenses_by_category[$category_name] = [
                    'category_name' => $category_name,
                    'items' => [],
                    'sub_total' => 0
                ];
            }
            
            // Use ref_no, additional_notes, or category name as item name
            $item_name = !empty($expense->ref_no) ? $expense->ref_no : (!empty($expense->additional_notes) ? $expense->additional_notes : $expense->category_name);
            
            $expenses_by_category[$category_name]['items'][] = [
                'name' => $item_name,
                'amount' => (float) $expense->final_total
            ];
            
            $expenses_by_category[$category_name]['sub_total'] += (float) $expense->final_total;
            $total_expense += (float) $expense->final_total;
        }
        
        // If a category has only one item and the item name is the same as category name, use a more descriptive name
        foreach ($expenses_by_category as $category_key => $category) {
            if (count($category['items']) == 1 && $category['items'][0]['name'] == $category['category_name']) {
                // Keep the category name as item name (matches the image design)
                $expenses_by_category[$category_key]['items'][0]['name'] = $category['category_name'];
            }
        }

        // Calculate Net Profit
        $net_profit = $gross_profit - $total_expense;

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.income_statement', compact(
            'sales_accounts',
            'cost_of_goods_sold',
            'sales_return',
            'cost_of_goods_sales_return',
            'extracharges_on_sales',
            'extracharges_on_purchase',
            'discount_on_sales',
            'gross_profit',
            'expenses_by_category',
            'total_expense',
            'net_profit',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'util',
            'business'
        ));
    }

    /**
     * Print Income Statement Report
     */
    public function printIncomeStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }

        // Get location filter
        $location_id = $request->get('location_id');

        // Get profit/loss data using TransactionUtil
        $transactionUtil = new \App\Utils\TransactionUtil();
        $permitted_locations = auth()->user()->permitted_locations();
        
        $pl_data = $transactionUtil->getProfitLossDetails(
            $business_id,
            $location_id,
            $start_date,
            $end_date,
            null,
            $permitted_locations
        );

        // Extract data
        $sales_accounts = (float) ($pl_data['total_sell'] ?? 0);
        $opening_stock = (float) ($pl_data['opening_stock'] ?? 0);
        $total_purchase = (float) ($pl_data['total_purchase'] ?? 0);
        $closing_stock = (float) ($pl_data['closing_stock'] ?? 0);
        $cost_of_goods_sold = $opening_stock + $total_purchase - $closing_stock;
        $sales_return = (float) ($pl_data['total_sell_return'] ?? 0);
        $purchase_return = (float) ($pl_data['total_purchase_return'] ?? 0);
        $cost_of_goods_sales_return = $purchase_return;
        $extracharges_on_sales = (float) ($pl_data['total_sell_additional_expense'] ?? 0);
        $extracharges_on_purchase = (float) ($pl_data['total_purchase_additional_expense'] ?? 0);
        $discount_on_sales = (float) ($pl_data['total_sell_discount'] ?? 0);
        
        // Calculate Gross Profit
        $gross_profit = $sales_accounts - $cost_of_goods_sold - $sales_return + $cost_of_goods_sales_return + $extracharges_on_sales - $extracharges_on_purchase - $discount_on_sales;

        // Get expenses grouped by category
        $expenses_by_category = [];
        $total_expense = 0;

        $expenses_query = DB::table('transactions as t')
            ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'expense')
            ->where('t.status', 'final')
            ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

        if ($permitted_locations != 'all') {
            $expenses_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $expenses_query->where('t.location_id', $location_id);
        }

        $expenses = $expenses_query
            ->select(
                'ec.id as category_id',
                'ec.name as category_name',
                't.id',
                't.final_total',
                't.ref_no',
                't.additional_notes',
                DB::raw("COALESCE(ec.name, 'Uncategorized') as expense_category")
            )
            ->orderBy('ec.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->get();

        // Group expenses by category
        foreach ($expenses as $expense) {
            $category_name = $expense->expense_category;
            
            if (!isset($expenses_by_category[$category_name])) {
                $expenses_by_category[$category_name] = [
                    'category_name' => $category_name,
                    'items' => [],
                    'sub_total' => 0
                ];
            }
            
            // Use ref_no, additional_notes, or category name as item name
            $item_name = !empty($expense->ref_no) ? $expense->ref_no : (!empty($expense->additional_notes) ? $expense->additional_notes : $expense->category_name);
            
            $expenses_by_category[$category_name]['items'][] = [
                'name' => $item_name,
                'amount' => (float) $expense->final_total
            ];
            
            $expenses_by_category[$category_name]['sub_total'] += (float) $expense->final_total;
            $total_expense += (float) $expense->final_total;
        }
        
        // If a category has only one item and the item name is the same as category name, use a more descriptive name
        foreach ($expenses_by_category as $category_key => $category) {
            if (count($category['items']) == 1 && $category['items'][0]['name'] == $category['category_name']) {
                // Keep the category name as item name (matches the image design)
                $expenses_by_category[$category_key]['items'][0]['name'] = $category['category_name'];
            }
        }

        // Calculate Net Profit
        $net_profit = $gross_profit - $total_expense;

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_income_statement', compact(
            'sales_accounts',
            'cost_of_goods_sold',
            'sales_return',
            'cost_of_goods_sales_return',
            'extracharges_on_sales',
            'extracharges_on_purchase',
            'discount_on_sales',
            'gross_profit',
            'expenses_by_category',
            'total_expense',
            'net_profit',
            'start_date',
            'end_date',
            'location_id',
            'util',
            'business'
        ));
    }

    /**
     * Day Book Report
     */
    public function dayBook(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }

        // Get location filter
        $location_id = $request->get('location_id');
        
        // Get transaction type filter (sales, purchase, expense, everything, all)
        $transaction_type = $request->get('transaction_type', 'all');

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id, true);
        
        $permitted_locations = auth()->user()->permitted_locations();

        // Initialize data arrays
        $credit_vouchers = [];
        $debit_vouchers = [];
        $purchases = [];
        $sales = [];
        $sales_returns = [];
        $grand_total_debit = 0;
        $grand_total_credit = 0;

        // 1. Credit Vouchers (Receipt)
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'credit_voucher') {
            $credit_query = DB::table('account_transactions as at')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
                ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
                ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'credit')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $credit_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $credit_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $credit_vouchers = $credit_query
                ->select(
                    DB::raw("CASE 
                        WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                        WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                        WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                        ELSE CONCAT('CRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                    END as voucher_id"),
                    DB::raw("COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name, 'Walk-In Customer') as trader_name"),
                    DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                    DB::raw("COALESCE(at.note, '') as narration"),
                    DB::raw("0 as debit"),
                    'at.amount as credit',
                    'tp.method as payment_method'
                )
                ->groupBy('at.id', 'at.operation_date', 'at.amount', 'at.reff_no', 'at.note', 't.invoice_no', 't.ref_no', 'tp.payment_ref_no', 'c_p.supplier_business_name', 'c_p.name', 'c_t.supplier_business_name', 'c_t.name', 'tp.method')
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();
        }

        // 2. Debit Vouchers (Payment and General Payment)
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'debit_voucher') {
            $debit_query = DB::table('account_transactions as at')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
                ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
                ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
                ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'debit')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $debit_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $debit_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $debit_vouchers = $debit_query
                ->select(
                    'at.id',
                    'at.operation_date',
                    'at.amount',
                    'at.reff_no',
                    'at.note',
                    'at.sub_type',
                    't.id as transaction_id',
                    't.type as transaction_type',
                    't.invoice_no',
                    't.ref_no',
                    'tp.payment_ref_no',
                    'tp.method as payment_method',
                    DB::raw("CASE 
                        WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                        WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                        WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                        ELSE CONCAT('DRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                    END as voucher_id"),
                    DB::raw("CASE 
                        WHEN t.type = 'expense' AND ec.name IS NOT NULL AND ec.name != '' THEN ec.name
                        WHEN COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name) IS NOT NULL 
                            AND COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name) != ''
                            THEN COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name)
                        WHEN at.sub_type = 'fund_transfer' THEN 'Fund Transfer'
                        WHEN at.sub_type = 'deposit' THEN 'Deposit'
                        WHEN at.sub_type = 'opening_balance' THEN 'Opening Balance'
                        ELSE 'Walk-In Customer'
                    END as trader_name"),
                    DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                    DB::raw("COALESCE(at.note, '') as narration"),
                    'at.amount as debit',
                    DB::raw("0 as credit")
                )
                ->groupBy('at.id', 'at.operation_date', 'at.amount', 'at.reff_no', 'at.note', 'at.sub_type', 't.id', 't.type', 't.invoice_no', 't.ref_no', 'tp.payment_ref_no', 'tp.method', 'c_p.supplier_business_name', 'c_p.name', 'c_t.supplier_business_name', 'c_t.name', 'ec.name')
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();
        }

        // 3. Purchases (Cash and Credit)
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'purchase') {
            $purchase_query = DB::table('transactions as t')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase')
                ->where('t.status', 'final')
                ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $purchase_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $purchase_query->where('t.location_id', $location_id);
            }

            $purchases = $purchase_query
                ->select(
                    't.invoice_no as voucher_id',
                    DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Supplier') as trader_name"),
                    't.ref_no',
                    DB::raw("COALESCE(t.additional_notes, '') as narration"),
                    't.final_total as debit',
                    DB::raw("0 as credit"),
                    'tp.method as payment_method'
                )
                ->groupBy('t.id', 't.invoice_no', 't.ref_no', 't.final_total', 't.additional_notes', 'c.supplier_business_name', 'c.name', 'tp.method')
                ->orderBy('t.transaction_date', 'asc')
                ->orderBy('t.id', 'asc')
                ->get();
        }

        // 4. Sales (Cash and Credit)
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'sales') {
            $sales_query = DB::table('transactions as t')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $sales_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $sales_query->where('t.location_id', $location_id);
            }

            $sales = $sales_query
                ->select(
                    't.invoice_no as voucher_id',
                    DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as trader_name"),
                    't.ref_no',
                    DB::raw("COALESCE(t.additional_notes, '') as narration"),
                    DB::raw("0 as debit"),
                    't.final_total as credit',
                    'tp.method as payment_method'
                )
                ->groupBy('t.id', 't.invoice_no', 't.ref_no', 't.final_total', 't.additional_notes', 'c.supplier_business_name', 'c.name', 'tp.method')
                ->orderBy('t.transaction_date', 'asc')
                ->orderBy('t.id', 'asc')
                ->get();
        }

        // 5. Sales Returns
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'sales_return') {
            $sales_return_query = DB::table('transactions as t')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell_return')
                ->where('t.status', 'final')
                ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $sales_return_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $sales_return_query->where('t.location_id', $location_id);
            }

            $sales_returns = $sales_return_query
                ->select(
                    't.invoice_no as voucher_id',
                    DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as trader_name"),
                    't.ref_no',
                    DB::raw("COALESCE(t.additional_notes, '') as narration"),
                    't.final_total as debit',
                    DB::raw("0 as credit"),
                    'tp.method as payment_method'
                )
                ->groupBy('t.id', 't.invoice_no', 't.ref_no', 't.final_total', 't.additional_notes', 'c.supplier_business_name', 'c.name', 'tp.method')
                ->orderBy('t.transaction_date', 'asc')
                ->orderBy('t.id', 'asc')
                ->get();
        }

        // Calculate totals
        foreach ($credit_vouchers as $voucher) {
            $grand_total_credit += (float) $voucher->credit;
        }
        
        foreach ($debit_vouchers as $voucher) {
            $grand_total_debit += (float) $voucher->debit;
        }
        
        foreach ($purchases as $purchase) {
            $grand_total_debit += (float) $purchase->debit;
        }
        
        foreach ($sales as $sale) {
            $grand_total_credit += (float) $sale->credit;
        }
        
        foreach ($sales_returns as $return) {
            $grand_total_debit += (float) $return->debit;
        }

        // Separate debit vouchers into Payment and General Payment
        $payment_vouchers = [];
        $general_payment_vouchers = [];
        
        foreach ($debit_vouchers as $voucher) {
            $is_payment = false;
            
            if (!empty($voucher->transaction_id)) {
                if (in_array($voucher->transaction_type, ['purchase', 'purchase_return', 'sell', 'sell_return'])) {
                    $is_payment = true;
                } elseif (!empty($voucher->trader_name) && $voucher->trader_name != 'Walk-In Customer' && $voucher->trader_name != 'Fund Transfer' && $voucher->trader_name != 'Deposit' && $voucher->trader_name != 'Opening Balance') {
                    $is_payment = true;
                }
            } elseif (!empty($voucher->trader_name) && $voucher->trader_name != 'Walk-In Customer' && $voucher->trader_name != 'Fund Transfer' && $voucher->trader_name != 'Deposit' && $voucher->trader_name != 'Opening Balance') {
                $is_payment = true;
            }
            
            if ($is_payment) {
                $payment_vouchers[] = $voucher;
            } else {
                $general_payment_vouchers[] = $voucher;
            }
        }

        // Group purchases by payment method
        $cash_purchases = [];
        $credit_purchases = [];
        foreach ($purchases as $purchase) {
            if (strtolower($purchase->payment_method ?? '') == 'cash' || empty($purchase->payment_method)) {
                $cash_purchases[] = $purchase;
            } else {
                $credit_purchases[] = $purchase;
            }
        }

        // Group sales by payment method
        $cash_sales = [];
        $credit_sales = [];
        foreach ($sales as $sale) {
            if (strtolower($sale->payment_method ?? '') == 'cash' || empty($sale->payment_method)) {
                $cash_sales[] = $sale;
            } else {
                $credit_sales[] = $sale;
            }
        }

        // Group sales returns by payment method
        $cash_sales_returns = [];
        $credit_sales_returns = [];
        foreach ($sales_returns as $return) {
            if (strtolower($return->payment_method ?? '') == 'cash' || empty($return->payment_method)) {
                $cash_sales_returns[] = $return;
            } else {
                $credit_sales_returns[] = $return;
            }
        }

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.day_book', compact(
            'credit_vouchers',
            'payment_vouchers',
            'general_payment_vouchers',
            'cash_purchases',
            'credit_purchases',
            'cash_sales',
            'credit_sales',
            'cash_sales_returns',
            'credit_sales_returns',
            'grand_total_debit',
            'grand_total_credit',
            'start_date',
            'end_date',
            'location_id',
            'transaction_type',
            'locations',
            'util',
            'business'
        ));
    }

    /**
     * Print Day Book Report
     */
    public function printDayBook(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }

        // Get location filter
        $location_id = $request->get('location_id');
        
        // Get transaction type filter
        $transaction_type = $request->get('transaction_type', 'all');
        
        $permitted_locations = auth()->user()->permitted_locations();

        // Initialize data arrays
        $credit_vouchers = [];
        $debit_vouchers = [];
        $purchases = [];
        $sales = [];
        $sales_returns = [];
        $grand_total_debit = 0;
        $grand_total_credit = 0;

        // 1. Credit Vouchers (Receipt)
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'credit_voucher') {
            $credit_query = DB::table('account_transactions as at')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
                ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
                ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'credit')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $credit_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $credit_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $credit_vouchers = $credit_query
                ->select(
                    DB::raw("CASE 
                        WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                        WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                        WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                        ELSE CONCAT('CRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                    END as voucher_id"),
                    DB::raw("COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name, 'Walk-In Customer') as trader_name"),
                    DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                    DB::raw("COALESCE(at.note, '') as narration"),
                    DB::raw("0 as debit"),
                    'at.amount as credit',
                    'tp.method as payment_method'
                )
                ->groupBy('at.id', 'at.operation_date', 'at.amount', 'at.reff_no', 'at.note', 't.invoice_no', 't.ref_no', 'tp.payment_ref_no', 'c_p.supplier_business_name', 'c_p.name', 'c_t.supplier_business_name', 'c_t.name', 'tp.method')
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();
        }

        // 2. Debit Vouchers (Payment and General Payment)
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'debit_voucher') {
            $debit_query = DB::table('account_transactions as at')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
                ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
                ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
                ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'debit')
                ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $debit_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.id');
                });
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $debit_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.id');
                });
            }

            $debit_vouchers = $debit_query
                ->select(
                    'at.id',
                    'at.operation_date',
                    'at.amount',
                    'at.reff_no',
                    'at.note',
                    'at.sub_type',
                    't.id as transaction_id',
                    't.type as transaction_type',
                    't.invoice_no',
                    't.ref_no',
                    'tp.payment_ref_no',
                    'tp.method as payment_method',
                    DB::raw("CASE 
                        WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                        WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                        WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                        ELSE CONCAT('DRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                    END as voucher_id"),
                    DB::raw("CASE 
                        WHEN t.type = 'expense' AND ec.name IS NOT NULL AND ec.name != '' THEN ec.name
                        WHEN COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name) IS NOT NULL 
                            AND COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name) != ''
                            THEN COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name)
                        WHEN at.sub_type = 'fund_transfer' THEN 'Fund Transfer'
                        WHEN at.sub_type = 'deposit' THEN 'Deposit'
                        WHEN at.sub_type = 'opening_balance' THEN 'Opening Balance'
                        ELSE 'Walk-In Customer'
                    END as trader_name"),
                    DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                    DB::raw("COALESCE(at.note, '') as narration"),
                    'at.amount as debit',
                    DB::raw("0 as credit")
                )
                ->groupBy('at.id', 'at.operation_date', 'at.amount', 'at.reff_no', 'at.note', 'at.sub_type', 't.id', 't.type', 't.invoice_no', 't.ref_no', 'tp.payment_ref_no', 'tp.method', 'c_p.supplier_business_name', 'c_p.name', 'c_t.supplier_business_name', 'c_t.name', 'ec.name')
                ->orderBy('at.operation_date', 'asc')
                ->orderBy('at.id', 'asc')
                ->get();
        }

        // 3. Purchases (Cash and Credit)
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'purchase') {
            $purchase_query = DB::table('transactions as t')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase')
                ->where('t.status', 'final')
                ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $purchase_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $purchase_query->where('t.location_id', $location_id);
            }

            $purchases = $purchase_query
                ->select(
                    't.invoice_no as voucher_id',
                    DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Supplier') as trader_name"),
                    't.ref_no',
                    DB::raw("COALESCE(t.additional_notes, '') as narration"),
                    't.final_total as debit',
                    DB::raw("0 as credit"),
                    'tp.method as payment_method'
                )
                ->groupBy('t.id', 't.invoice_no', 't.ref_no', 't.final_total', 't.additional_notes', 'c.supplier_business_name', 'c.name', 'tp.method')
                ->orderBy('t.transaction_date', 'asc')
                ->orderBy('t.id', 'asc')
                ->get();
        }

        // 4. Sales (Cash and Credit)
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'sales') {
            $sales_query = DB::table('transactions as t')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $sales_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $sales_query->where('t.location_id', $location_id);
            }

            $sales = $sales_query
                ->select(
                    't.invoice_no as voucher_id',
                    DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as trader_name"),
                    't.ref_no',
                    DB::raw("COALESCE(t.additional_notes, '') as narration"),
                    DB::raw("0 as debit"),
                    't.final_total as credit',
                    'tp.method as payment_method'
                )
                ->groupBy('t.id', 't.invoice_no', 't.ref_no', 't.final_total', 't.additional_notes', 'c.supplier_business_name', 'c.name', 'tp.method')
                ->orderBy('t.transaction_date', 'asc')
                ->orderBy('t.id', 'asc')
                ->get();
        }

        // 5. Sales Returns
        if ($transaction_type == 'all' || $transaction_type == 'everything' || $transaction_type == 'sales_return') {
            $sales_return_query = DB::table('transactions as t')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell_return')
                ->where('t.status', 'final')
                ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

            if ($permitted_locations != 'all') {
                $sales_return_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $sales_return_query->where('t.location_id', $location_id);
            }

            $sales_returns = $sales_return_query
                ->select(
                    't.invoice_no as voucher_id',
                    DB::raw("COALESCE(c.supplier_business_name, c.name, 'Walk-In Customer') as trader_name"),
                    't.ref_no',
                    DB::raw("COALESCE(t.additional_notes, '') as narration"),
                    't.final_total as debit',
                    DB::raw("0 as credit"),
                    'tp.method as payment_method'
                )
                ->groupBy('t.id', 't.invoice_no', 't.ref_no', 't.final_total', 't.additional_notes', 'c.supplier_business_name', 'c.name', 'tp.method')
                ->orderBy('t.transaction_date', 'asc')
                ->orderBy('t.id', 'asc')
                ->get();
        }

        // Calculate totals
        foreach ($credit_vouchers as $voucher) {
            $grand_total_credit += (float) $voucher->credit;
        }
        
        foreach ($debit_vouchers as $voucher) {
            $grand_total_debit += (float) $voucher->debit;
        }
        
        foreach ($purchases as $purchase) {
            $grand_total_debit += (float) $purchase->debit;
        }
        
        foreach ($sales as $sale) {
            $grand_total_credit += (float) $sale->credit;
        }
        
        foreach ($sales_returns as $return) {
            $grand_total_debit += (float) $return->debit;
        }

        // Separate debit vouchers into Payment and General Payment
        $payment_vouchers = [];
        $general_payment_vouchers = [];
        
        foreach ($debit_vouchers as $voucher) {
            $is_payment = false;
            
            if (!empty($voucher->transaction_id)) {
                if (in_array($voucher->transaction_type, ['purchase', 'purchase_return', 'sell', 'sell_return'])) {
                    $is_payment = true;
                } elseif (!empty($voucher->trader_name) && $voucher->trader_name != 'Walk-In Customer' && $voucher->trader_name != 'Fund Transfer' && $voucher->trader_name != 'Deposit' && $voucher->trader_name != 'Opening Balance') {
                    $is_payment = true;
                }
            } elseif (!empty($voucher->trader_name) && $voucher->trader_name != 'Walk-In Customer' && $voucher->trader_name != 'Fund Transfer' && $voucher->trader_name != 'Deposit' && $voucher->trader_name != 'Opening Balance') {
                $is_payment = true;
            }
            
            if ($is_payment) {
                $payment_vouchers[] = $voucher;
            } else {
                $general_payment_vouchers[] = $voucher;
            }
        }

        // Group purchases by payment method
        $cash_purchases = [];
        $credit_purchases = [];
        foreach ($purchases as $purchase) {
            if (strtolower($purchase->payment_method ?? '') == 'cash' || empty($purchase->payment_method)) {
                $cash_purchases[] = $purchase;
            } else {
                $credit_purchases[] = $purchase;
            }
        }

        // Group sales by payment method
        $cash_sales = [];
        $credit_sales = [];
        foreach ($sales as $sale) {
            if (strtolower($sale->payment_method ?? '') == 'cash' || empty($sale->payment_method)) {
                $cash_sales[] = $sale;
            } else {
                $credit_sales[] = $sale;
            }
        }

        // Group sales returns by payment method
        $cash_sales_returns = [];
        $credit_sales_returns = [];
        foreach ($sales_returns as $return) {
            if (strtolower($return->payment_method ?? '') == 'cash' || empty($return->payment_method)) {
                $cash_sales_returns[] = $return;
            } else {
                $credit_sales_returns[] = $return;
            }
        }

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_day_book', compact(
            'credit_vouchers',
            'payment_vouchers',
            'general_payment_vouchers',
            'cash_purchases',
            'credit_purchases',
            'cash_sales',
            'credit_sales',
            'cash_sales_returns',
            'credit_sales_returns',
            'grand_total_debit',
            'grand_total_credit',
            'start_date',
            'end_date',
            'location_id',
            'transaction_type',
            'util',
            'business'
        ));
    }

    /**
     * Due List Without Treatment Report
     */
    public function dueListWithoutTreatment(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filter
        $date = !empty($request->date) ? $request->date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Get filters
        $supplier_filter = $request->get('supplier_filter', 'all');
        $supplier_id = $request->get('supplier_id');
        $customer_filter = $request->get('customer_filter', 'all');
        $customer_id = $request->get('customer_id');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get all customers and suppliers for dropdowns
        $customers = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->pluck('name', 'id');
        
        $suppliers = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->pluck('supplier_business_name', 'id')
            ->map(function($name, $id) {
                $contact = Contact::find($id);
                return $contact && $contact->supplier_business_name ? $contact->supplier_business_name : $contact->name;
            });

        $locations = BusinessLocation::forDropdown($business_id);

        // Get customers with due amounts (Receivable)
        $customer_due_data = [];
        if ($customer_filter == 'all' || ($customer_filter == 'single' && !empty($customer_id))) {
            $customer_query = Contact::where('business_id', $business_id)
                ->where('type', 'customer')
                ->select(
                    'contacts.id',
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.mobile',
                    'contacts.address_line_1',
                    'contacts.address_line_2',
                    'contacts.city',
                    'contacts.state',
                    DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                    DB::raw("CONCAT(
                        COALESCE(contacts.address_line_1, ''),
                        CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.address_line_2, ''),
                        CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.city, ''),
                        CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.state, '')
                    ) as full_address")
                );

            if ($customer_filter == 'single' && !empty($customer_id) && $customer_id != 'all') {
                $customer_query->where('contacts.id', $customer_id);
            }

            $customers_list = $customer_query->orderBy('display_name', 'asc')->get();

            foreach ($customers_list as $customer) {
                // Get all transactions for this customer up to the date
                $transactions = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['sell', 'opening_balance'])
                    ->whereDate('t.transaction_date', '<=', $date)
                    ->where(function($q) {
                        $q->where(function($q2) {
                            $q2->where('t.type', 'sell')->where('t.status', 'final');
                        })->orWhere('t.type', 'opening_balance');
                    });

                // Apply location filter
                if (!empty($location_id)) {
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        continue;
                    }
                    $transactions->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                    $transactions->whereIn('t.location_id', $permitted_locations);
                }

                $transactions = $transactions->select('t.id', 't.type', 't.final_total')
                    ->get();

                $total_invoice = 0;
                $total_paid = 0;

                foreach ($transactions as $transaction) {
                    if ($transaction->type == 'sell' || $transaction->type == 'opening_balance') {
                        $total_invoice += $transaction->final_total ?? 0;
                    }

                    $payments = DB::table('transaction_payments')
                        ->where('transaction_id', $transaction->id)
                        ->whereDate('paid_on', '<=', $date)
                        ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                    $total_paid += $payments ?? 0;
                }

                $due_amount = $total_invoice - $total_paid;

                if (abs($due_amount) > 0.01) {
                    $customer_due_data[] = (object)[
                        'id' => $customer->id,
                        'name' => $customer->display_name,
                        'address' => $customer->full_address ?? '',
                        'mobile' => $customer->mobile ?? '',
                        'due_amount' => $due_amount,
                    ];
                }
            }
        }

        // Get suppliers with due amounts (Payable)
        $supplier_due_data = [];
        if ($supplier_filter == 'all' || ($supplier_filter == 'single' && !empty($supplier_id))) {
            $supplier_query = Contact::where('business_id', $business_id)
                ->where('type', 'supplier')
                ->select(
                    'contacts.id',
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.mobile',
                    'contacts.address_line_1',
                    'contacts.address_line_2',
                    'contacts.city',
                    'contacts.state',
                    DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                    DB::raw("CONCAT(
                        COALESCE(contacts.address_line_1, ''),
                        CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.address_line_2, ''),
                        CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.city, ''),
                        CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.state, '')
                    ) as full_address")
                );

            if ($supplier_filter == 'single' && !empty($supplier_id) && $supplier_id != 'all') {
                $supplier_query->where('contacts.id', $supplier_id);
            }

            $suppliers_list = $supplier_query->orderBy('display_name', 'asc')->get();

            foreach ($suppliers_list as $supplier) {
                // Get all transactions for this supplier up to the date
                $transactions = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $supplier->id)
                    ->whereIn('t.type', ['purchase', 'opening_balance'])
                    ->whereDate('t.transaction_date', '<=', $date)
                    ->where(function($q) {
                        $q->where(function($q2) {
                            $q2->where('t.type', 'purchase')->where('t.status', 'received');
                        })->orWhere('t.type', 'opening_balance');
                    });

                // Apply location filter
                if (!empty($location_id)) {
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        continue;
                    }
                    $transactions->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                    $transactions->whereIn('t.location_id', $permitted_locations);
                }

                $transactions = $transactions->select('t.id', 't.type', 't.final_total')
                    ->get();

                $total_invoice = 0;
                $total_paid = 0;

                foreach ($transactions as $transaction) {
                    if ($transaction->type == 'purchase' || $transaction->type == 'opening_balance') {
                        $total_invoice += $transaction->final_total ?? 0;
                    }

                    $payments = DB::table('transaction_payments')
                        ->where('transaction_id', $transaction->id)
                        ->whereDate('paid_on', '<=', $date)
                        ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                    $total_paid += $payments ?? 0;
                }

                // For suppliers, due = what we owe them (positive means payable)
                $due_amount = $total_invoice - $total_paid;

                if (abs($due_amount) > 0.01) {
                    $supplier_due_data[] = (object)[
                        'id' => $supplier->id,
                        'name' => $supplier->display_name,
                        'address' => $supplier->full_address ?? '',
                        'mobile' => $supplier->mobile ?? '',
                        'due_amount' => -$due_amount, // Negative for payable
                    ];
                }
            }
        }

        // Calculate totals
        $customer_total = collect($customer_due_data)->sum('due_amount');
        $supplier_total = collect($supplier_due_data)->sum('due_amount');
        $grand_total = $customer_total + $supplier_total;

        $util = new Util();

        return view('businessmanagement::accounts_register.due_list_without_treatment', compact(
            'customer_due_data',
            'supplier_due_data',
            'date',
            'supplier_filter',
            'supplier_id',
            'customer_filter',
            'customer_id',
            'location_id',
            'customers',
            'suppliers',
            'locations',
            'customer_total',
            'supplier_total',
            'grand_total',
            'util'
        ));
    }

    /**
     * Print Due List Without Treatment Report
     */
    public function printDueListWithoutTreatment(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filter
        $date = !empty($request->date) ? $request->date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Get filters
        $supplier_filter = $request->get('supplier_filter', 'all');
        $supplier_id = $request->get('supplier_id');
        $customer_filter = $request->get('customer_filter', 'all');
        $customer_id = $request->get('customer_id');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get customers with due amounts (Receivable) - same logic as web view
        $customer_due_data = [];
        if ($customer_filter == 'all' || ($customer_filter == 'single' && !empty($customer_id))) {
            $customer_query = Contact::where('business_id', $business_id)
                ->where('type', 'customer')
                ->select(
                    'contacts.id',
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.mobile',
                    'contacts.address_line_1',
                    'contacts.address_line_2',
                    'contacts.city',
                    'contacts.state',
                    DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                    DB::raw("CONCAT(
                        COALESCE(contacts.address_line_1, ''),
                        CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.address_line_2, ''),
                        CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.city, ''),
                        CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.state, '')
                    ) as full_address")
                );

            if ($customer_filter == 'single' && !empty($customer_id) && $customer_id != 'all') {
                $customer_query->where('contacts.id', $customer_id);
            }

            $customers_list = $customer_query->orderBy('display_name', 'asc')->get();

            foreach ($customers_list as $customer) {
                $transactions = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['sell', 'opening_balance'])
                    ->whereDate('t.transaction_date', '<=', $date)
                    ->where(function($q) {
                        $q->where(function($q2) {
                            $q2->where('t.type', 'sell')->where('t.status', 'final');
                        })->orWhere('t.type', 'opening_balance');
                    });

                if (!empty($location_id)) {
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        continue;
                    }
                    $transactions->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                    $transactions->whereIn('t.location_id', $permitted_locations);
                }

                $transactions = $transactions->select('t.id', 't.type', 't.final_total')
                    ->get();

                $total_invoice = 0;
                $total_paid = 0;

                foreach ($transactions as $transaction) {
                    if ($transaction->type == 'sell' || $transaction->type == 'opening_balance') {
                        $total_invoice += $transaction->final_total ?? 0;
                    }

                    $payments = DB::table('transaction_payments')
                        ->where('transaction_id', $transaction->id)
                        ->whereDate('paid_on', '<=', $date)
                        ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                    $total_paid += $payments ?? 0;
                }

                $due_amount = $total_invoice - $total_paid;

                if (abs($due_amount) > 0.01) {
                    $customer_due_data[] = (object)[
                        'id' => $customer->id,
                        'name' => $customer->display_name,
                        'address' => $customer->full_address ?? '',
                        'mobile' => $customer->mobile ?? '',
                        'due_amount' => $due_amount,
                    ];
                }
            }
        }

        // Get suppliers with due amounts (Payable) - same logic as web view
        $supplier_due_data = [];
        if ($supplier_filter == 'all' || ($supplier_filter == 'single' && !empty($supplier_id))) {
            $supplier_query = Contact::where('business_id', $business_id)
                ->where('type', 'supplier')
                ->select(
                    'contacts.id',
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.mobile',
                    'contacts.address_line_1',
                    'contacts.address_line_2',
                    'contacts.city',
                    'contacts.state',
                    DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                    DB::raw("CONCAT(
                        COALESCE(contacts.address_line_1, ''),
                        CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.address_line_2, ''),
                        CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.city, ''),
                        CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                        COALESCE(contacts.state, '')
                    ) as full_address")
                );

            if ($supplier_filter == 'single' && !empty($supplier_id) && $supplier_id != 'all') {
                $supplier_query->where('contacts.id', $supplier_id);
            }

            $suppliers_list = $supplier_query->orderBy('display_name', 'asc')->get();

            foreach ($suppliers_list as $supplier) {
                $transactions = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $supplier->id)
                    ->whereIn('t.type', ['purchase', 'opening_balance'])
                    ->whereDate('t.transaction_date', '<=', $date)
                    ->where(function($q) {
                        $q->where(function($q2) {
                            $q2->where('t.type', 'purchase')->where('t.status', 'received');
                        })->orWhere('t.type', 'opening_balance');
                    });

                if (!empty($location_id)) {
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        continue;
                    }
                    $transactions->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                    $transactions->whereIn('t.location_id', $permitted_locations);
                }

                $transactions = $transactions->select('t.id', 't.type', 't.final_total')
                    ->get();

                $total_invoice = 0;
                $total_paid = 0;

                foreach ($transactions as $transaction) {
                    if ($transaction->type == 'purchase' || $transaction->type == 'opening_balance') {
                        $total_invoice += $transaction->final_total ?? 0;
                    }

                    $payments = DB::table('transaction_payments')
                        ->where('transaction_id', $transaction->id)
                        ->whereDate('paid_on', '<=', $date)
                        ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                    $total_paid += $payments ?? 0;
                }

                $due_amount = $total_invoice - $total_paid;

                if (abs($due_amount) > 0.01) {
                    $supplier_due_data[] = (object)[
                        'id' => $supplier->id,
                        'name' => $supplier->display_name,
                        'address' => $supplier->full_address ?? '',
                        'mobile' => $supplier->mobile ?? '',
                        'due_amount' => -$due_amount, // Negative for payable
                    ];
                }
            }
        }

        // Calculate totals
        $customer_total = collect($customer_due_data)->sum('due_amount');
        $supplier_total = collect($supplier_due_data)->sum('due_amount');
        $grand_total = $customer_total + $supplier_total;

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_due_list_without_treatment', compact(
            'customer_due_data',
            'supplier_due_data',
            'date',
            'supplier_filter',
            'supplier_id',
            'customer_filter',
            'customer_id',
            'location_id',
            'customer_total',
            'supplier_total',
            'grand_total',
            'util',
            'business'
        ));
    }

    /**
     * Date Wise Sales Received Report
     */
    public function dateWiseSalesReceived(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get walk-in customer ID
        $walk_in_customer = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->where('is_default', 1)
            ->first();
        $walk_in_customer_id = $walk_in_customer ? $walk_in_customer->id : null;

        // Initialize data arrays for each category
        $categories = [
            'Supplier' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
            'Retailer' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
            'Dealer' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
            'Credit Customer' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
            'Cash' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
        ];

        // Process Suppliers
        $suppliers = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->pluck('id');

        if ($suppliers->count() > 0) {
            // Opening Due (before start_date)
            $supplier_opening = $this->calculateOpeningDue($business_id, $suppliers->toArray(), $start_date, $permitted_locations);
            $categories['Supplier']['opening_due'] = $supplier_opening['opening_due'];
            
            // Sales, Discount, Received (within date range)
            $supplier_period = $this->calculatePeriodData($business_id, $suppliers->toArray(), $start_date, $end_date, $permitted_locations, 'supplier');
            $categories['Supplier']['sales'] = $supplier_period['sales'];
            $categories['Supplier']['discount'] = $supplier_period['discount'];
            $categories['Supplier']['received'] = $supplier_period['received'];
            $categories['Supplier']['sales_due'] = $supplier_period['sales'] - $supplier_period['received'];
            $categories['Supplier']['current_due'] = $supplier_opening['opening_due'] + $categories['Supplier']['sales_due'];
        }

        // Process Retailers (customer group contains "Retailer")
        $retailers = Contact::where('contacts.business_id', $business_id)
            ->where('contacts.type', 'customer')
            ->leftJoin('customer_groups as cg', 'contacts.customer_group_id', '=', 'cg.id')
            ->where('cg.name', 'like', '%Retailer%')
            ->pluck('contacts.id');

        if ($retailers->count() > 0) {
            $retailer_opening = $this->calculateOpeningDue($business_id, $retailers->toArray(), $start_date, $permitted_locations);
            $categories['Retailer']['opening_due'] = $retailer_opening['opening_due'];
            
            $retailer_period = $this->calculatePeriodData($business_id, $retailers->toArray(), $start_date, $end_date, $permitted_locations, 'customer');
            $categories['Retailer']['sales'] = $retailer_period['sales'];
            $categories['Retailer']['discount'] = $retailer_period['discount'];
            $categories['Retailer']['received'] = $retailer_period['received'];
            $categories['Retailer']['sales_due'] = $retailer_period['sales'] - $retailer_period['received'];
            $categories['Retailer']['current_due'] = $retailer_opening['opening_due'] + $categories['Retailer']['sales_due'];
        }

        // Process Dealers (customer group contains "Dealer")
        $dealers = Contact::where('contacts.business_id', $business_id)
            ->where('contacts.type', 'customer')
            ->leftJoin('customer_groups as cg', 'contacts.customer_group_id', '=', 'cg.id')
            ->where('cg.name', 'like', '%Dealer%')
            ->pluck('contacts.id');

        if ($dealers->count() > 0) {
            $dealer_opening = $this->calculateOpeningDue($business_id, $dealers->toArray(), $start_date, $permitted_locations);
            $categories['Dealer']['opening_due'] = $dealer_opening['opening_due'];
            
            $dealer_period = $this->calculatePeriodData($business_id, $dealers->toArray(), $start_date, $end_date, $permitted_locations, 'customer');
            $categories['Dealer']['sales'] = $dealer_period['sales'];
            $categories['Dealer']['discount'] = $dealer_period['discount'];
            $categories['Dealer']['received'] = $dealer_period['received'];
            $categories['Dealer']['sales_due'] = $dealer_period['sales'] - $dealer_period['received'];
            $categories['Dealer']['current_due'] = $dealer_opening['opening_due'] + $categories['Dealer']['sales_due'];
        }

        // Process Credit Customers (customers with credit sales, excluding walk-in)
        $credit_customers = Contact::where('contacts.business_id', $business_id)
            ->where('contacts.type', 'customer')
            ->leftJoin('customer_groups as cg', 'contacts.customer_group_id', '=', 'cg.id')
            ->where(function($q) use ($walk_in_customer_id) {
                if ($walk_in_customer_id) {
                    $q->where('contacts.id', '!=', $walk_in_customer_id);
                }
            })
            ->where(function($q) {
                $q->whereNull('cg.name')
                  ->orWhere(function($q2) {
                      $q2->where('cg.name', 'not like', '%Retailer%')
                         ->where('cg.name', 'not like', '%Dealer%');
                  });
            })
            ->pluck('contacts.id');

        if ($credit_customers->count() > 0) {
            $credit_opening = $this->calculateOpeningDue($business_id, $credit_customers->toArray(), $start_date, $permitted_locations);
            $categories['Credit Customer']['opening_due'] = $credit_opening['opening_due'];
            
            $credit_period = $this->calculatePeriodData($business_id, $credit_customers->toArray(), $start_date, $end_date, $permitted_locations, 'customer');
            $categories['Credit Customer']['sales'] = $credit_period['sales'];
            $categories['Credit Customer']['discount'] = $credit_period['discount'];
            $categories['Credit Customer']['received'] = $credit_period['received'];
            $categories['Credit Customer']['sales_due'] = $credit_period['sales'] - $credit_period['received'];
            $categories['Credit Customer']['current_due'] = $credit_opening['opening_due'] + $categories['Credit Customer']['sales_due'];
        }

        // Process Cash (walk-in customer or cash payments)
        if ($walk_in_customer_id) {
            $cash_opening = $this->calculateOpeningDue($business_id, [$walk_in_customer_id], $start_date, $permitted_locations);
            $categories['Cash']['opening_due'] = $cash_opening['opening_due'];
            
            $cash_period = $this->calculatePeriodData($business_id, [$walk_in_customer_id], $start_date, $end_date, $permitted_locations, 'cash');
            $categories['Cash']['sales'] = $cash_period['sales'];
            $categories['Cash']['discount'] = $cash_period['discount'];
            $categories['Cash']['received'] = $cash_period['received'];
            $categories['Cash']['sales_due'] = $cash_period['sales'] - $cash_period['received'];
            $categories['Cash']['current_due'] = $cash_opening['opening_due'] + $categories['Cash']['sales_due'];
        }

        // Calculate Grand Totals
        $grand_totals = [
            'opening_due' => array_sum(array_column($categories, 'opening_due')),
            'sales' => array_sum(array_column($categories, 'sales')),
            'discount' => array_sum(array_column($categories, 'discount')),
            'received' => array_sum(array_column($categories, 'received')),
            'sales_due' => array_sum(array_column($categories, 'sales_due')),
            'current_due' => array_sum(array_column($categories, 'current_due')),
        ];

        $util = new Util();

        return view('businessmanagement::accounts_register.date_wise_sales_received', compact(
            'categories',
            'grand_totals',
            'start_date',
            'end_date',
            'util'
        ));
    }

    /**
     * Calculate Opening Due (before start_date)
     */
    private function calculateOpeningDue($business_id, $contact_ids, $start_date, $permitted_locations)
    {
        $query = DB::table('transactions as t')
            ->where('t.business_id', $business_id)
            ->whereIn('t.contact_id', $contact_ids)
            ->whereIn('t.type', ['sell', 'purchase', 'opening_balance', 'sell_return', 'purchase_return'])
            ->whereDate('t.transaction_date', '<', $start_date);

        if ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        $transactions = $query->select(
            DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', t.final_total, 0)) as total_sales"),
            DB::raw("SUM(IF(t.type = 'purchase' AND t.status = 'received', t.final_total, 0)) as total_purchase"),
            DB::raw("SUM(IF(t.type = 'sell_return' AND t.status = 'final', t.final_total, 0)) as total_sales_return"),
            DB::raw("SUM(IF(t.type = 'purchase_return' AND t.status = 'received', t.final_total, 0)) as total_purchase_return"),
            DB::raw("SUM(IF(t.type = 'opening_balance', t.final_total, 0)) as total_opening_balance")
        )->first();

        $total_invoice = ($transactions->total_sales ?? 0) + ($transactions->total_purchase ?? 0) + ($transactions->total_opening_balance ?? 0);
        $total_return = ($transactions->total_sales_return ?? 0) + ($transactions->total_purchase_return ?? 0);

        // Get payments before start_date
        $payments = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->where('t.business_id', $business_id)
            ->whereIn('t.contact_id', $contact_ids)
            ->whereDate('tp.paid_on', '<', $start_date);

        if ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $payments->whereIn('t.location_id', $permitted_locations);
        }

        $total_paid = $payments->sum(DB::raw('IF(tp.is_return = 1, -1 * tp.amount, tp.amount)')) ?? 0;

        $opening_due = $total_invoice - $total_return - $total_paid;

        return ['opening_due' => $opening_due];
    }

    /**
     * Calculate Period Data (within date range)
     */
    private function calculatePeriodData($business_id, $contact_ids, $start_date, $end_date, $permitted_locations, $type = 'customer')
    {
        $query = DB::table('transactions as t')
            ->where('t.business_id', $business_id)
            ->whereIn('t.contact_id', $contact_ids)
            ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

        if ($type == 'supplier') {
            $query->whereIn('t.type', ['purchase', 'purchase_return']);
        } elseif ($type == 'cash') {
            $query->where('t.type', 'sell')
                  ->where('t.status', 'final')
                  ->join('transaction_payments as tp_cash', function($join) {
                      $join->on('t.id', '=', 'tp_cash.transaction_id')
                           ->where('tp_cash.method', 'cash');
                  });
        } else {
            $query->whereIn('t.type', ['sell', 'sell_return']);
        }

        if ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        $transactions = $query->select(
            DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', t.final_total, 0)) as total_sales"),
            DB::raw("SUM(IF(t.type = 'purchase' AND t.status = 'received', t.final_total, 0)) as total_purchase"),
            DB::raw("SUM(IF(t.discount_type = 'percentage', (t.discount_amount * t.final_total / 100), COALESCE(t.discount_amount, 0))) as total_discount")
        )->first();

        $sales = ($transactions->total_sales ?? 0) + ($transactions->total_purchase ?? 0);
        $discount = $transactions->total_discount ?? 0;

        // Get payments within date range
        $payments = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->where('t.business_id', $business_id)
            ->whereIn('t.contact_id', $contact_ids)
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if ($type == 'cash') {
            $payments->where('tp.method', 'cash');
        }

        if ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $payments->whereIn('t.location_id', $permitted_locations);
        }

        $received = $payments->sum(DB::raw('IF(tp.is_return = 1, -1 * tp.amount, tp.amount)')) ?? 0;

        return [
            'sales' => $sales,
            'discount' => $discount,
            'received' => $received
        ];
    }

    /**
     * Print Date Wise Sales Received Report
     */
    public function printDateWiseSalesReceived(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get walk-in customer ID
        $walk_in_customer = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->where('is_default', 1)
            ->first();
        $walk_in_customer_id = $walk_in_customer ? $walk_in_customer->id : null;

        // Initialize data arrays for each category (same logic as web view)
        $categories = [
            'Supplier' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
            'Retailer' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
            'Dealer' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
            'Credit Customer' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
            'Cash' => ['opening_due' => 0, 'sales' => 0, 'discount' => 0, 'received' => 0, 'sales_due' => 0, 'current_due' => 0],
        ];

        // Process Suppliers
        $suppliers = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->pluck('id');

        if ($suppliers->count() > 0) {
            $supplier_opening = $this->calculateOpeningDue($business_id, $suppliers->toArray(), $start_date, $permitted_locations);
            $categories['Supplier']['opening_due'] = $supplier_opening['opening_due'];
            
            $supplier_period = $this->calculatePeriodData($business_id, $suppliers->toArray(), $start_date, $end_date, $permitted_locations, 'supplier');
            $categories['Supplier']['sales'] = $supplier_period['sales'];
            $categories['Supplier']['discount'] = $supplier_period['discount'];
            $categories['Supplier']['received'] = $supplier_period['received'];
            $categories['Supplier']['sales_due'] = $supplier_period['sales'] - $supplier_period['received'];
            $categories['Supplier']['current_due'] = $supplier_opening['opening_due'] + $categories['Supplier']['sales_due'];
        }

        // Process Retailers
        $retailers = Contact::where('contacts.business_id', $business_id)
            ->where('contacts.type', 'customer')
            ->leftJoin('customer_groups as cg', 'contacts.customer_group_id', '=', 'cg.id')
            ->where('cg.name', 'like', '%Retailer%')
            ->pluck('contacts.id');

        if ($retailers->count() > 0) {
            $retailer_opening = $this->calculateOpeningDue($business_id, $retailers->toArray(), $start_date, $permitted_locations);
            $categories['Retailer']['opening_due'] = $retailer_opening['opening_due'];
            
            $retailer_period = $this->calculatePeriodData($business_id, $retailers->toArray(), $start_date, $end_date, $permitted_locations, 'customer');
            $categories['Retailer']['sales'] = $retailer_period['sales'];
            $categories['Retailer']['discount'] = $retailer_period['discount'];
            $categories['Retailer']['received'] = $retailer_period['received'];
            $categories['Retailer']['sales_due'] = $retailer_period['sales'] - $retailer_period['received'];
            $categories['Retailer']['current_due'] = $retailer_opening['opening_due'] + $categories['Retailer']['sales_due'];
        }

        // Process Dealers
        $dealers = Contact::where('contacts.business_id', $business_id)
            ->where('contacts.type', 'customer')
            ->leftJoin('customer_groups as cg', 'contacts.customer_group_id', '=', 'cg.id')
            ->where('cg.name', 'like', '%Dealer%')
            ->pluck('contacts.id');

        if ($dealers->count() > 0) {
            $dealer_opening = $this->calculateOpeningDue($business_id, $dealers->toArray(), $start_date, $permitted_locations);
            $categories['Dealer']['opening_due'] = $dealer_opening['opening_due'];
            
            $dealer_period = $this->calculatePeriodData($business_id, $dealers->toArray(), $start_date, $end_date, $permitted_locations, 'customer');
            $categories['Dealer']['sales'] = $dealer_period['sales'];
            $categories['Dealer']['discount'] = $dealer_period['discount'];
            $categories['Dealer']['received'] = $dealer_period['received'];
            $categories['Dealer']['sales_due'] = $dealer_period['sales'] - $dealer_period['received'];
            $categories['Dealer']['current_due'] = $dealer_opening['opening_due'] + $categories['Dealer']['sales_due'];
        }

        // Process Credit Customers
        $credit_customers = Contact::where('contacts.business_id', $business_id)
            ->where('contacts.type', 'customer')
            ->leftJoin('customer_groups as cg', 'contacts.customer_group_id', '=', 'cg.id')
            ->where(function($q) use ($walk_in_customer_id) {
                if ($walk_in_customer_id) {
                    $q->where('contacts.id', '!=', $walk_in_customer_id);
                }
            })
            ->where(function($q) {
                $q->whereNull('cg.name')
                  ->orWhere(function($q2) {
                      $q2->where('cg.name', 'not like', '%Retailer%')
                         ->where('cg.name', 'not like', '%Dealer%');
                  });
            })
            ->pluck('contacts.id');

        if ($credit_customers->count() > 0) {
            $credit_opening = $this->calculateOpeningDue($business_id, $credit_customers->toArray(), $start_date, $permitted_locations);
            $categories['Credit Customer']['opening_due'] = $credit_opening['opening_due'];
            
            $credit_period = $this->calculatePeriodData($business_id, $credit_customers->toArray(), $start_date, $end_date, $permitted_locations, 'customer');
            $categories['Credit Customer']['sales'] = $credit_period['sales'];
            $categories['Credit Customer']['discount'] = $credit_period['discount'];
            $categories['Credit Customer']['received'] = $credit_period['received'];
            $categories['Credit Customer']['sales_due'] = $credit_period['sales'] - $credit_period['received'];
            $categories['Credit Customer']['current_due'] = $credit_opening['opening_due'] + $categories['Credit Customer']['sales_due'];
        }

        // Process Cash
        if ($walk_in_customer_id) {
            $cash_opening = $this->calculateOpeningDue($business_id, [$walk_in_customer_id], $start_date, $permitted_locations);
            $categories['Cash']['opening_due'] = $cash_opening['opening_due'];
            
            $cash_period = $this->calculatePeriodData($business_id, [$walk_in_customer_id], $start_date, $end_date, $permitted_locations, 'cash');
            $categories['Cash']['sales'] = $cash_period['sales'];
            $categories['Cash']['discount'] = $cash_period['discount'];
            $categories['Cash']['received'] = $cash_period['received'];
            $categories['Cash']['sales_due'] = $cash_period['sales'] - $cash_period['received'];
            $categories['Cash']['current_due'] = $cash_opening['opening_due'] + $categories['Cash']['sales_due'];
        }

        // Calculate Grand Totals
        $grand_totals = [
            'opening_due' => array_sum(array_column($categories, 'opening_due')),
            'sales' => array_sum(array_column($categories, 'sales')),
            'discount' => array_sum(array_column($categories, 'discount')),
            'received' => array_sum(array_column($categories, 'received')),
            'sales_due' => array_sum(array_column($categories, 'sales_due')),
            'current_due' => array_sum(array_column($categories, 'current_due')),
        ];

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_date_wise_sales_received', compact(
            'categories',
            'grand_totals',
            'start_date',
            'end_date',
            'util',
            'business'
        ));
    }

    /**
     * Balance Sheet Report
     */
    public function balanceSheet(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();
        
        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id);

        // Get account balances as of the date
        $account_balances = $this->getAccountBalancesForBalanceSheet($business_id, $end_date, $permitted_locations, $location_id);

        // Get customer due (Sundry Debtor)
        $customer_due = $this->getCustomerDue($business_id, $end_date, $permitted_locations, $location_id);

        // Get supplier due (Sundry Creditor)
        $supplier_due = $this->getSupplierDue($business_id, $end_date, $permitted_locations, $location_id);

        // Get closing stock
        $transactionUtil = new \App\Utils\TransactionUtil();
        $closing_stock = $transactionUtil->getOpeningClosingStock($business_id, $end_date, $location_id, $permitted_locations);

        // Get profit/loss for the date range
        $pl_data = $transactionUtil->getProfitLossDetails($business_id, $location_id, $start_date, $end_date, null, $permitted_locations);
        $current_period_profit = ($pl_data['net_profit'] ?? 0);
        
        // Get opening profit (previous balance - before start_date)
        $opening_profit = $this->getOpeningProfit($business_id, $start_date, $permitted_locations, $location_id);

        // Build Liabilities
        $liabilities = $this->buildBalanceSheetLiabilities($account_balances, $supplier_due, $current_period_profit, $business_id, $end_date);

        // Build Assets
        $assets = $this->buildBalanceSheetAssets($account_balances, $customer_due, $closing_stock, $opening_profit, $business_id, $end_date);

        // Calculate totals
        $total_liabilities = $this->calculateBalanceSheetTotal($liabilities);
        $total_assets = $this->calculateBalanceSheetTotal($assets);

        $util = new Util();

        return view('businessmanagement::accounts_register.balance_sheet', compact(
            'liabilities',
            'assets',
            'total_liabilities',
            'total_assets',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'util'
        ));
    }

    /**
     * Print Balance Sheet Report
     */
    public function printBalanceSheet(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get account balances as of the date
        $account_balances = $this->getAccountBalancesForBalanceSheet($business_id, $end_date, $permitted_locations, $location_id);

        // Get customer due (Sundry Debtor)
        $customer_due = $this->getCustomerDue($business_id, $end_date, $permitted_locations, $location_id);

        // Get supplier due (Sundry Creditor)
        $supplier_due = $this->getSupplierDue($business_id, $end_date, $permitted_locations, $location_id);

        // Get closing stock
        $transactionUtil = new \App\Utils\TransactionUtil();
        $closing_stock = $transactionUtil->getOpeningClosingStock($business_id, $end_date, $location_id, $permitted_locations);

        // Get profit/loss for the date range
        $pl_data = $transactionUtil->getProfitLossDetails($business_id, $location_id, $start_date, $end_date, null, $permitted_locations);
        $current_period_profit = ($pl_data['net_profit'] ?? 0);
        
        // Get opening profit (previous balance - before start_date)
        $opening_profit = $this->getOpeningProfit($business_id, $start_date, $permitted_locations, $location_id);

        // Build Liabilities
        $liabilities = $this->buildBalanceSheetLiabilities($account_balances, $supplier_due, $current_period_profit, $business_id, $end_date);

        // Build Assets
        $assets = $this->buildBalanceSheetAssets($account_balances, $customer_due, $closing_stock, $opening_profit, $business_id, $end_date);

        // Calculate totals
        $total_liabilities = $this->calculateBalanceSheetTotal($liabilities);
        $total_assets = $this->calculateBalanceSheetTotal($assets);

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_balance_sheet', compact(
            'liabilities',
            'assets',
            'total_liabilities',
            'total_assets',
            'start_date',
            'end_date',
            'location_id',
            'util',
            'business'
        ));
    }

    /**
     * Get Account Balances for Balance Sheet
     */
    private function getAccountBalancesForBalanceSheet($business_id, $end_date, $permitted_locations, $location_id = null)
    {
        $query = Account::leftJoin('account_transactions as AT', function($join) use ($end_date) {
            $join->on('AT.account_id', '=', 'accounts.id')
                 ->whereNull('AT.deleted_at')
                 ->whereDate('AT.operation_date', '<=', $end_date);
        })
        ->leftJoin('account_types as ats', 'accounts.account_type_id', '=', 'ats.id')
        ->leftJoin('account_types as pat', 'ats.parent_account_type_id', '=', 'pat.id')
        ->where('accounts.business_id', $business_id)
        ->where('accounts.is_closed', 0)
        ->select([
            'accounts.id',
            'accounts.name',
            'accounts.account_type_id',
            'ats.name as account_type_name',
            'pat.name as parent_account_type_name',
            DB::raw("COALESCE(SUM(IF(AT.type='credit', AT.amount, -1*AT.amount)), 0) as balance")
        ])
        ->groupBy('accounts.id', 'accounts.name', 'accounts.account_type_id', 'ats.name', 'pat.name');

        // Apply location filter if needed
        if (!empty($location_id)) {
            // Validate that user has access to the requested location
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            
            $location = \App\BusinessLocation::find($location_id);
            if ($location && !empty($location->default_payment_accounts)) {
                $default_accounts = json_decode($location->default_payment_accounts, true);
                $location_account_ids = [];
                foreach ($default_accounts as $key => $account) {
                    if (!empty($account['is_enabled']) && !empty($account['account'])) {
                        $location_account_ids[] = $account['account'];
                    }
                }
                if (!empty($location_account_ids)) {
                    $query->whereIn('accounts.id', array_unique($location_account_ids));
                }
            }
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $location_account_ids = [];
            $locations = \App\BusinessLocation::where('business_id', $business_id)
                ->whereIn('id', $permitted_locations)
                ->get();
            
            foreach ($locations as $location) {
                if (!empty($location->default_payment_accounts)) {
                    $default_accounts = json_decode($location->default_payment_accounts, true);
                    foreach ($default_accounts as $key => $account) {
                        if (!empty($account['is_enabled']) && !empty($account['account'])) {
                            $location_account_ids[] = $account['account'];
                        }
                    }
                }
            }
            
            if (!empty($location_account_ids)) {
                $query->whereIn('accounts.id', array_unique($location_account_ids));
            }
        }

        $accounts = $query->get();

        $balances = [];
        foreach ($accounts as $account) {
            $balance = (float) ($account->balance ?? 0);
            if (abs($balance) > 0.01) {
                $balances[] = [
                    'id' => $account->id,
                    'name' => $account->name,
                    'account_type' => strtolower($account->account_type_name ?? ''),
                    'parent_type' => strtolower($account->parent_account_type_name ?? ''),
                    'balance' => $balance
                ];
            }
        }

        return $balances;
    }

    /**
     * Get Customer Due (Sundry Debtor)
     */
    private function getCustomerDue($business_id, $end_date, $permitted_locations, $location_id = null)
    {
        $query = DB::table('transactions as t')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereDate('t.transaction_date', '<=', $end_date);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                return 0;
            }
            $query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        $total_invoice = $query->sum('t.final_total') ?? 0;

        $payments_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereDate('tp.paid_on', '<=', $end_date);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                return 0;
            }
            $payments_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $payments_query->whereIn('t.location_id', $permitted_locations);
        }

        $total_paid = $payments_query->sum(DB::raw('IF(tp.is_return = 1, -1 * tp.amount, tp.amount)')) ?? 0;

        return $total_invoice - $total_paid;
    }

    /**
     * Get Supplier Due (Sundry Creditor)
     */
    private function getSupplierDue($business_id, $end_date, $permitted_locations, $location_id = null)
    {
        $query = DB::table('transactions as t')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', '<=', $end_date);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                return 0;
            }
            $query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        $total_invoice = $query->sum('t.final_total') ?? 0;

        $payments_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->whereDate('tp.paid_on', '<=', $end_date);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                return 0;
            }
            $payments_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $payments_query->whereIn('t.location_id', $permitted_locations);
        }

        $total_paid = $payments_query->sum(DB::raw('IF(tp.is_return = 1, -1 * tp.amount, tp.amount)')) ?? 0;

        return $total_invoice - $total_paid;
    }

    /**
     * Get Opening Profit
     */
    private function getOpeningProfit($business_id, $end_date, $permitted_locations, $location_id = null)
    {
        // Get profit/loss before the end date
        $transactionUtil = new \App\Utils\TransactionUtil();
        
        // Get financial year start
        $businessUtil = new \App\Utils\BusinessUtil();
        $fy = $businessUtil->getCurrentFinancialYear($business_id);
        $fy_start = $fy['start'] ?? date('Y-01-01');
        
        // Calculate profit from start of FY to day before end_date
        $prev_date = \Carbon\Carbon::parse($end_date)->subDay()->format('Y-m-d');
        if ($prev_date >= $fy_start) {
            $pl_data = $transactionUtil->getProfitLossDetails($business_id, $location_id, $fy_start, $prev_date, null, $permitted_locations);
            return ($pl_data['net_profit'] ?? 0);
        }
        
        return 0;
    }

    /**
     * Build Balance Sheet Liabilities
     */
    private function buildBalanceSheetLiabilities($account_balances, $supplier_due, $current_period_profit, $business_id, $end_date)
    {
        $liabilities = [];

        // 1. Capital Accounts
        $capital_accounts = [];
        $capital_total = 0;
        
        foreach ($account_balances as $account) {
            $name = strtolower($account['name']);
            $balance = $account['balance'];
            
            if ((strpos($name, 'capital') !== false || 
                 strpos($name, 'loan') !== false ||
                 strpos($name, 'equity') !== false ||
                 $account['account_type'] == 'equity' ||
                 $account['parent_type'] == 'equity') && 
                abs($balance) > 0.01) {
                $capital_accounts[] = [
                    'name' => $account['name'],
                    'value' => abs($balance)
                ];
                $capital_total += abs($balance);
            }
        }

        if ($capital_total > 0) {
            $liabilities[] = [
                'label' => 'Capital Accounts',
                'items' => $capital_accounts,
                'total' => $capital_total
            ];
        }

        // 2. Current Liabilities - Sundry Creditor
        if ($supplier_due > 0.01) {
            $liabilities[] = [
                'label' => 'Current Liabilities',
                'items' => [
                    ['name' => 'Sundry Creditor', 'value' => $supplier_due]
                ],
                'total' => $supplier_due
            ];
        }

        // 3. Opening Capital
        $opening_capital_accounts = [];
        $opening_capital_total = 0;
        
        foreach ($account_balances as $account) {
            $name = strtolower($account['name']);
            $balance = $account['balance'];
            
            if (strpos($name, 'opening') !== false && 
                (strpos($name, 'capital') !== false || strpos($name, 'account') !== false) &&
                abs($balance) > 0.01) {
                $opening_capital_accounts[] = [
                    'name' => $account['name'],
                    'value' => abs($balance)
                ];
                $opening_capital_total += abs($balance);
            }
        }

        if ($opening_capital_total > 0) {
            $liabilities[] = [
                'label' => 'Opening Capital',
                'items' => $opening_capital_accounts,
                'total' => $opening_capital_total
            ];
        }

        // 4. RMA Sundry Creditor
        $rma_creditor = 0;
        foreach ($account_balances as $account) {
            $name = strtolower($account['name']);
            if (strpos($name, 'rma') !== false && 
                (strpos($name, 'creditor') !== false || strpos($name, 'sundry') !== false) &&
                $account['balance'] > 0.01) {
                $rma_creditor += abs($account['balance']);
            }
        }

        if ($rma_creditor > 0.01) {
            $liabilities[] = [
                'label' => 'RMA Sundry Creditor',
                'items' => [
                    ['name' => 'RMA Sundry Creditor', 'value' => $rma_creditor]
                ],
                'total' => $rma_creditor
            ];
        }

        // 5. Profit - Current Period
        if (abs($current_period_profit) > 0.01) {
            $liabilities[] = [
                'label' => 'Profit',
                'sub_label' => 'Current Period',
                'items' => [
                    ['name' => 'Profit A/C', 'value' => abs($current_period_profit)]
                ],
                'total' => abs($current_period_profit)
            ];
        }

        return $liabilities;
    }

    /**
     * Build Balance Sheet Assets
     */
    private function buildBalanceSheetAssets($account_balances, $customer_due, $closing_stock, $opening_profit, $business_id, $end_date)
    {
        $assets = [];

        // 1. Current Asset - Opening Cash
        $cash_accounts = [];
        $cash_total = 0;
        
        foreach ($account_balances as $account) {
            $name = strtolower($account['name']);
            $balance = $account['balance'];
            
            if ((strpos($name, 'cash') !== false || 
                 strpos($name, 'petty') !== false) &&
                $balance > 0.01) {
                $cash_accounts[] = [
                    'name' => $account['name'],
                    'value' => abs($balance)
                ];
                $cash_total += abs($balance);
            }
        }

        if ($cash_total > 0) {
            $assets[] = [
                'label' => 'Current Asset',
                'sub_label' => 'Opening Cash',
                'items' => $cash_accounts,
                'total' => $cash_total
            ];
        }

        // 2. Stock Accounts
        if ($closing_stock > 0.01) {
            $assets[] = [
                'label' => 'Stock Accounts',
                'items' => [
                    ['name' => 'Stock Accounts', 'value' => $closing_stock]
                ],
                'total' => $closing_stock
            ];
        }

        // 3. Sundry Debtor
        if ($customer_due > 0.01) {
            $assets[] = [
                'label' => 'Sundry Debtor',
                'items' => [
                    ['name' => 'Sundry Debtor', 'value' => $customer_due]
                ],
                'total' => $customer_due
            ];
        }

        // 4. RMA Stock Accounts
        $rma_stock = 0;
        foreach ($account_balances as $account) {
            $name = strtolower($account['name']);
            if (strpos($name, 'rma') !== false && 
                (strpos($name, 'stock') !== false) &&
                $account['balance'] > 0.01) {
                $rma_stock += abs($account['balance']);
            }
        }

        if ($rma_stock > 0.01) {
            $assets[] = [
                'label' => 'RMA Stock Accounts',
                'items' => [
                    ['name' => 'RMAStock Accounts', 'value' => $rma_stock]
                ],
                'total' => $rma_stock
            ];
        }

        // 5. RMA Sundry Debtor
        $rma_debtor = 0;
        foreach ($account_balances as $account) {
            $name = strtolower($account['name']);
            if (strpos($name, 'rma') !== false && 
                (strpos($name, 'debtor') !== false || strpos($name, 'sundry') !== false) &&
                $account['balance'] > 0.01) {
                $rma_debtor += abs($account['balance']);
            }
        }

        if ($rma_debtor > 0.01) {
            $assets[] = [
                'label' => 'RMA Sundry Debtor',
                'items' => [
                    ['name' => 'RMA Sundry Debtor', 'value' => $rma_debtor]
                ],
                'total' => $rma_debtor
            ];
        }

        // 6. Investment
        $investment_accounts = [];
        $investment_total = 0;
        
        foreach ($account_balances as $account) {
            $name = strtolower($account['name']);
            $balance = $account['balance'];
            
            if ((strpos($name, 'investment') !== false || 
                 strpos($name, 'investor') !== false) &&
                $balance > 0.01) {
                $investment_accounts[] = [
                    'name' => $account['name'],
                    'value' => abs($balance)
                ];
                $investment_total += abs($balance);
            }
        }

        if ($investment_total > 0) {
            $assets[] = [
                'label' => 'Investment',
                'sub_label' => 'INVESTOR\'S',
                'items' => $investment_accounts,
                'total' => $investment_total
            ];
        }

        // 7. Profit - Previous Balance
        if (abs($opening_profit) > 0.01) {
            $assets[] = [
                'label' => 'Profit',
                'sub_label' => 'Previous Balance',
                'items' => [
                    ['name' => 'Loss A/C', 'value' => abs($opening_profit)],
                    ['name' => 'RMA Profit Or Loss', 'value' => 0] // Placeholder, adjust as needed
                ],
                'total' => abs($opening_profit)
            ];
        }

        return $assets;
    }

    /**
     * Calculate Balance Sheet Total
     */
    private function calculateBalanceSheetTotal($sections)
    {
        $total = 0;
        foreach ($sections as $section) {
            $total += $section['total'] ?? 0;
        }
        return $total;
    }

    /**
     * Date Wise General Expenses Report
     */
    public function dateWiseGeneralExpenses(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $category_filter = $request->get('category_filter', 'all');
        $category_id = $request->get('category_id');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get expense categories for dropdown
        $expense_categories = DB::table('expense_categories')
            ->where('business_id', $business_id)
            ->pluck('name', 'id');

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id);

        // Get expenses grouped by category
        $expenses_query = DB::table('transactions as t')
            ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'expense')
            ->where('t.status', 'final')
            ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

        // Apply category filter
        if ($category_filter == 'single' && !empty($category_id) && $category_id != 'all') {
            $expenses_query->where('t.expense_category_id', $category_id);
        }

        // Apply location filter
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $expenses_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $expenses_query->whereIn('t.location_id', $permitted_locations);
        }

        $expenses = $expenses_query
            ->leftJoin('account_transactions as at', function($join) {
                $join->on('at.transaction_id', '=', 't.id')
                     ->where('at.type', '=', 'credit');
            })
            ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
            ->select(
                'ec.id as category_id',
                'ec.name as category_name',
                't.id as transaction_id',
                't.ref_no',
                't.additional_notes',
                't.final_total',
                'a.name as account_name',
                DB::raw("COALESCE(ec.name, 'Uncategorized') as expense_category")
            )
            ->orderBy('ec.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->get();

        // Group expenses by category
        $expenses_by_category = [];
        $total_expenses = 0;
        $serial = 1;

        foreach ($expenses as $expense) {
            $category_name = $expense->expense_category ?? 'Uncategorized';
            
            if (!isset($expenses_by_category[$category_name])) {
                $expenses_by_category[$category_name] = [
                    'category_name' => $category_name,
                    'items' => [],
                    'sub_total' => 0
                ];
            }

            // Get expense item name (account name, additional_notes, ref_no, or category name)
            $item_name = !empty($expense->account_name) ? $expense->account_name : 
                        (!empty($expense->additional_notes) ? $expense->additional_notes : 
                        (!empty($expense->ref_no) ? $expense->ref_no : 
                        ($expense->category_name ?? 'Expense')));

            $expenses_by_category[$category_name]['items'][] = [
                'serial' => $serial++,
                'name' => $item_name,
                'amount' => (float) $expense->final_total
            ];

            $expenses_by_category[$category_name]['sub_total'] += (float) $expense->final_total;
            $total_expenses += (float) $expense->final_total;
        }

        $util = new Util();

        return view('businessmanagement::accounts_register.date_wise_general_expenses', compact(
            'expenses_by_category',
            'total_expenses',
            'start_date',
            'end_date',
            'category_filter',
            'category_id',
            'location_id',
            'expense_categories',
            'locations',
            'util'
        ));
    }

    /**
     * Print Date Wise General Expenses Report
     */
    public function printDateWiseGeneralExpenses(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $category_filter = $request->get('category_filter', 'all');
        $category_id = $request->get('category_id');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get expenses grouped by category (same logic as web view)
        $expenses_query = DB::table('transactions as t')
            ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'expense')
            ->where('t.status', 'final')
            ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);

        // Apply category filter
        if ($category_filter == 'single' && !empty($category_id) && $category_id != 'all') {
            $expenses_query->where('t.expense_category_id', $category_id);
        }

        // Apply location filter
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $expenses_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $expenses_query->whereIn('t.location_id', $permitted_locations);
        }

        $expenses = $expenses_query
            ->leftJoin('account_transactions as at', function($join) {
                $join->on('at.transaction_id', '=', 't.id')
                     ->where('at.type', '=', 'credit');
            })
            ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
            ->select(
                'ec.id as category_id',
                'ec.name as category_name',
                't.id as transaction_id',
                't.ref_no',
                't.additional_notes',
                't.final_total',
                'a.name as account_name',
                DB::raw("COALESCE(ec.name, 'Uncategorized') as expense_category")
            )
            ->orderBy('ec.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->get();

        // Group expenses by category
        $expenses_by_category = [];
        $total_expenses = 0;
        $serial = 1;

        foreach ($expenses as $expense) {
            $category_name = $expense->expense_category ?? 'Uncategorized';
            
            if (!isset($expenses_by_category[$category_name])) {
                $expenses_by_category[$category_name] = [
                    'category_name' => $category_name,
                    'items' => [],
                    'sub_total' => 0
                ];
            }

            // Get expense item name (account name, additional_notes, ref_no, or category name)
            $item_name = !empty($expense->account_name) ? $expense->account_name : 
                        (!empty($expense->additional_notes) ? $expense->additional_notes : 
                        (!empty($expense->ref_no) ? $expense->ref_no : 
                        ($expense->category_name ?? 'Expense')));

            $expenses_by_category[$category_name]['items'][] = [
                'serial' => $serial++,
                'name' => $item_name,
                'amount' => (float) $expense->final_total
            ];

            $expenses_by_category[$category_name]['sub_total'] += (float) $expense->final_total;
            $total_expenses += (float) $expense->final_total;
        }

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_date_wise_general_expenses', compact(
            'expenses_by_category',
            'total_expenses',
            'start_date',
            'end_date',
            'category_filter',
            'category_id',
            'location_id',
            'util',
            'business'
        ));
    }

    /**
     * Accounts Receivable Report
     */
    public function accountsReceivable(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $customer_filter = $request->get('customer_filter', 'all');
        $customer_id = $request->get('customer_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get customers for dropdown
        $customers = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select('id', DB::raw("COALESCE(supplier_business_name, name) as display_name"))
            ->orderBy('display_name', 'asc')
            ->pluck('display_name', 'id');

        // Get customers with receivable balances as of end_date
        $receivable_data = [];
        
        $customer_query = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.mobile',
                'contacts.address_line_1',
                'contacts.address_line_2',
                'contacts.city',
                'contacts.state',
                'contacts.contact_id',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                DB::raw("COALESCE(contacts.name, '') as proprietor_name"),
                DB::raw("CONCAT(
                    COALESCE(contacts.address_line_1, ''),
                    CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.address_line_2, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.city, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.state, '')
                ) as full_address")
            );

        // Apply customer filter
        if ($customer_filter == 'single' && !empty($customer_id) && $customer_id != 'all') {
            $customer_query->where('contacts.id', $customer_id);
        }

        $customers_list = $customer_query->orderBy('display_name', 'asc')->get();

        foreach ($customers_list as $customer) {
            // Get all transactions for this customer up to the end_date
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $customer->id)
                ->whereIn('t.type', ['sell', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $end_date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'sell')->where('t.status', 'final');
                    })->orWhere('t.type', 'opening_balance');
                });

            // Apply location filter
            if ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $transactions->whereIn('t.location_id', $permitted_locations);
            }

            $transactions = $transactions->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                if ($transaction->type == 'sell' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $end_date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate receivable balance (positive means customer owes us)
            $receivable_balance = $total_invoice - $total_paid;

            // Only include customers with receivable balance > 0.01
            if ($receivable_balance > 0.01) {
                $receivable_data[] = (object)[
                    'id' => $customer->id,
                    'accounts_name' => $customer->display_name,
                    'proprietor' => $customer->proprietor_name ?? '',
                    'address' => $customer->full_address ?? '',
                    'mobile' => $customer->mobile ?? '',
                    'balance' => $receivable_balance,
                ];
            }
        }

        // Calculate grand total
        $grand_total = collect($receivable_data)->sum('balance');

        $util = new Util();

        return view('businessmanagement::accounts_register.accounts_receivable', compact(
            'receivable_data',
            'grand_total',
            'start_date',
            'end_date',
            'customer_filter',
            'customer_id',
            'customers',
            'util'
        ));
    }

    /**
     * Print Accounts Receivable Report
     */
    public function printAccountsReceivable(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $customer_filter = $request->get('customer_filter', 'all');
        $customer_id = $request->get('customer_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get customers with receivable balances (same logic as web view)
        $receivable_data = [];
        
        $customer_query = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.mobile',
                'contacts.address_line_1',
                'contacts.address_line_2',
                'contacts.city',
                'contacts.state',
                'contacts.contact_id',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                DB::raw("COALESCE(contacts.name, '') as proprietor_name"),
                DB::raw("CONCAT(
                    COALESCE(contacts.address_line_1, ''),
                    CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.address_line_2, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.city, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.state, '')
                ) as full_address")
            );

        // Apply customer filter
        if ($customer_filter == 'single' && !empty($customer_id) && $customer_id != 'all') {
            $customer_query->where('contacts.id', $customer_id);
        }

        $customers_list = $customer_query->orderBy('display_name', 'asc')->get();

        foreach ($customers_list as $customer) {
            // Get all transactions for this customer up to the end_date
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $customer->id)
                ->whereIn('t.type', ['sell', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $end_date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'sell')->where('t.status', 'final');
                    })->orWhere('t.type', 'opening_balance');
                });

            // Apply location filter
            if ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $transactions->whereIn('t.location_id', $permitted_locations);
            }

            $transactions = $transactions->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                if ($transaction->type == 'sell' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $end_date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate receivable balance (positive means customer owes us)
            $receivable_balance = $total_invoice - $total_paid;

            // Only include customers with receivable balance > 0.01
            if ($receivable_balance > 0.01) {
                $receivable_data[] = (object)[
                    'id' => $customer->id,
                    'accounts_name' => $customer->display_name,
                    'proprietor' => $customer->proprietor_name ?? '',
                    'address' => $customer->full_address ?? '',
                    'mobile' => $customer->mobile ?? '',
                    'balance' => $receivable_balance,
                ];
            }
        }

        // Calculate grand total
        $grand_total = collect($receivable_data)->sum('balance');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_accounts_receivable', compact(
            'receivable_data',
            'grand_total',
            'start_date',
            'end_date',
            'customer_filter',
            'customer_id',
            'util',
            'business'
        ));
    }

    /**
     * Accounts Payable Report
     */
    public function accountsPayable(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $supplier_filter = $request->get('supplier_filter', 'all');
        $supplier_id = $request->get('supplier_id');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id);

        // Get suppliers for dropdown
        $suppliers = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->select('id', DB::raw("COALESCE(supplier_business_name, name) as display_name"))
            ->orderBy('display_name', 'asc')
            ->pluck('display_name', 'id');

        // Get suppliers with payable balances as of end_date
        $payable_data = [];
        
        $supplier_query = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.mobile',
                'contacts.address_line_1',
                'contacts.address_line_2',
                'contacts.city',
                'contacts.state',
                'contacts.contact_id',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                DB::raw("COALESCE(contacts.name, '') as proprietor_name"),
                DB::raw("CONCAT(
                    COALESCE(contacts.address_line_1, ''),
                    CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.address_line_2, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.city, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.state, '')
                ) as full_address")
            );

        // Apply supplier filter
        if ($supplier_filter == 'single' && !empty($supplier_id) && $supplier_id != 'all') {
            $supplier_query->where('contacts.id', $supplier_id);
        }

        $suppliers_list = $supplier_query->orderBy('display_name', 'asc')->get();

        foreach ($suppliers_list as $supplier) {
            // Get all transactions for this supplier up to the end_date
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $supplier->id)
                ->whereIn('t.type', ['purchase', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $end_date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'purchase')->where('t.status', 'received');
                    })->orWhere('t.type', 'opening_balance');
                });

            // Apply location filter
            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    continue;
                }
                $transactions->where('t.location_id', $location_id);
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $transactions->whereIn('t.location_id', $permitted_locations);
            }

            $transactions = $transactions->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                if ($transaction->type == 'purchase' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $end_date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate payable balance (positive means we owe supplier)
            $payable_balance = $total_invoice - $total_paid;

            // Only include suppliers with payable balance > 0.01
            if ($payable_balance > 0.01) {
                $payable_data[] = (object)[
                    'id' => $supplier->id,
                    'accounts_name' => $supplier->display_name,
                    'proprietor' => $supplier->proprietor_name ?? '',
                    'address' => $supplier->full_address ?? '',
                    'mobile' => $supplier->mobile ?? '',
                    'balance' => $payable_balance,
                ];
            }
        }

        // Calculate grand total
        $grand_total = collect($payable_data)->sum('balance');

        $util = new Util();

        return view('businessmanagement::accounts_register.accounts_payable', compact(
            'payable_data',
            'grand_total',
            'start_date',
            'end_date',
            'supplier_filter',
            'supplier_id',
            'location_id',
            'locations',
            'suppliers',
            'util'
        ));
    }

    /**
     * Print Accounts Payable Report
     */
    public function printAccountsPayable(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $supplier_filter = $request->get('supplier_filter', 'all');
        $supplier_id = $request->get('supplier_id');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get suppliers with payable balances (same logic as web view)
        $payable_data = [];
        
        $supplier_query = Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.mobile',
                'contacts.address_line_1',
                'contacts.address_line_2',
                'contacts.city',
                'contacts.state',
                'contacts.contact_id',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name"),
                DB::raw("COALESCE(contacts.name, '') as proprietor_name"),
                DB::raw("CONCAT(
                    COALESCE(contacts.address_line_1, ''),
                    CASE WHEN contacts.address_line_1 IS NOT NULL AND contacts.address_line_2 IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.address_line_2, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL) AND contacts.city IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.city, ''),
                    CASE WHEN (contacts.address_line_1 IS NOT NULL OR contacts.address_line_2 IS NOT NULL OR contacts.city IS NOT NULL) AND contacts.state IS NOT NULL THEN ', ' ELSE '' END,
                    COALESCE(contacts.state, '')
                ) as full_address")
            );

        // Apply supplier filter
        if ($supplier_filter == 'single' && !empty($supplier_id) && $supplier_id != 'all') {
            $supplier_query->where('contacts.id', $supplier_id);
        }

        $suppliers_list = $supplier_query->orderBy('display_name', 'asc')->get();

        foreach ($suppliers_list as $supplier) {
            // Get all transactions for this supplier up to the end_date
            $transactions = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $supplier->id)
                ->whereIn('t.type', ['purchase', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $end_date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'purchase')->where('t.status', 'received');
                    })->orWhere('t.type', 'opening_balance');
                });

            // Apply location filter
            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    continue;
                }
                $transactions->where('t.location_id', $location_id);
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $transactions->whereIn('t.location_id', $permitted_locations);
            }

            $transactions = $transactions->select('t.id', 't.type', 't.final_total')
                ->get();

            $total_invoice = 0;
            $total_paid = 0;

            foreach ($transactions as $transaction) {
                if ($transaction->type == 'purchase' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $end_date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate payable balance (positive means we owe supplier)
            $payable_balance = $total_invoice - $total_paid;

            // Only include suppliers with payable balance > 0.01
            if ($payable_balance > 0.01) {
                $payable_data[] = (object)[
                    'id' => $supplier->id,
                    'accounts_name' => $supplier->display_name,
                    'proprietor' => $supplier->proprietor_name ?? '',
                    'address' => $supplier->full_address ?? '',
                    'mobile' => $supplier->mobile ?? '',
                    'balance' => $payable_balance,
                ];
            }
        }

        // Calculate grand total
        $grand_total = collect($payable_data)->sum('balance');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_accounts_payable', compact(
            'payable_data',
            'grand_total',
            'start_date',
            'end_date',
            'supplier_filter',
            'supplier_id',
            'location_id',
            'util',
            'business'
        ));
    }

    /**
     * Customer Due With Ageing Report
     */
    public function customerDueWithAgeing(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $customer_filter = $request->get('customer_filter', 'all');
        $customer_id = $request->get('customer_id');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id);

        // Get customers for dropdown
        $customers = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select('id', DB::raw("COALESCE(supplier_business_name, name) as display_name"))
            ->orderBy('display_name', 'asc')
            ->pluck('display_name', 'id');

        // Get customers with due balances and ageing information
        $customer_due_data = [];
        
        $customer_query = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.customer_group_id',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name")
            );

        // Apply customer filter
        if ($customer_filter == 'single' && !empty($customer_id) && $customer_id != 'all') {
            $customer_query->where('contacts.id', $customer_id);
        }

        $customers_list = $customer_query->orderBy('display_name', 'asc')->get();

        foreach ($customers_list as $customer) {
            // Get all sales transactions for this customer up to end_date
            $sales_query = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $customer->id)
                ->whereIn('t.type', ['sell', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $end_date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'sell')->where('t.status', 'final');
                    })->orWhere('t.type', 'opening_balance');
                });

            // Apply location filter
            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    continue;
                }
                $sales_query->where('t.location_id', $location_id);
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $sales_query->whereIn('t.location_id', $permitted_locations);
            }

            $all_transactions = $sales_query->select('t.id', 't.type', 't.final_total', 't.transaction_date')
                ->orderBy('t.transaction_date', 'desc')
                ->get();

            // Calculate total invoice and total paid
            $total_invoice = 0;
            $total_paid = 0;

            foreach ($all_transactions as $transaction) {
                if ($transaction->type == 'sell' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $end_date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate current due balance
            $current_due = $total_invoice - $total_paid;

            // Only include customers with due balance > 0.01
            if ($current_due > 0.01) {
                // Get last sales transaction
                $last_sales = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $last_sales->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                    $last_sales->whereIn('t.location_id', $permitted_locations);
                }

                $last_sales = $last_sales->select('t.transaction_date', 't.final_total')
                    ->orderBy('t.transaction_date', 'desc')
                    ->first();

                $last_sales_date = $last_sales->transaction_date ?? null;
                $last_sales_amount = $last_sales->final_total ?? 0;
                $days_since_last_sales = $last_sales_date ? \Carbon\Carbon::parse($last_sales_date)->diffInDays(\Carbon\Carbon::parse($end_date)) : null;

                // Get last receipt/payment
                $last_receipt = DB::table('transaction_payments as tp')
                    ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['sell', 'opening_balance'])
                    ->whereDate('tp.paid_on', '<=', $end_date)
                    ->where('tp.is_return', 0);

                if (!empty($location_id)) {
                    $last_receipt->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                    $last_receipt->whereIn('t.location_id', $permitted_locations);
                }

                $last_receipt = $last_receipt->select('tp.paid_on', 'tp.amount')
                    ->orderBy('tp.paid_on', 'desc')
                    ->first();

                $last_receipt_date = $last_receipt->paid_on ?? null;
                $last_receipt_amount = $last_receipt->amount ?? 0;
                $days_since_last_receipt = $last_receipt_date ? \Carbon\Carbon::parse($last_receipt_date)->diffInDays(\Carbon\Carbon::parse($end_date)) : null;

                // Get customer group name for categorization
                $customer_group_name = '';
                if (!empty($customer->customer_group_id)) {
                    $customer_group = DB::table('customer_groups')
                        ->where('id', $customer->customer_group_id)
                        ->value('name');
                    $customer_group_name = $customer_group ?? '';
                }

                $customer_due_data[] = (object)[
                    'id' => $customer->id,
                    'customer_name' => $customer->display_name,
                    'customer_group_name' => $customer_group_name,
                    'last_sales_date' => $last_sales_date,
                    'last_sales_amount' => $last_sales_amount,
                    'days_since_last_sales' => $days_since_last_sales,
                    'last_receipt_date' => $last_receipt_date,
                    'last_receipt_amount' => $last_receipt_amount,
                    'days_since_last_receipt' => $days_since_last_receipt,
                    'current_due' => $current_due,
                ];
            }
        }

        // Group by customer group for subtotals
        $grouped_data = [];
        foreach ($customer_due_data as $item) {
            $group_key = !empty($item->customer_group_name) ? $item->customer_group_name : 'Other';
            if (!isset($grouped_data[$group_key])) {
                $grouped_data[$group_key] = [];
            }
            $grouped_data[$group_key][] = $item;
        }

        // Calculate grand total
        $grand_total = collect($customer_due_data)->sum('current_due');

        $util = new Util();

        return view('businessmanagement::accounts_register.customer_due_with_ageing', compact(
            'customer_due_data',
            'grouped_data',
            'grand_total',
            'end_date',
            'customer_filter',
            'customer_id',
            'location_id',
            'locations',
            'customers',
            'util'
        ));
    }

    /**
     * Print Customer Due With Ageing Report
     */
    public function printCustomerDueWithAgeing(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $customer_filter = $request->get('customer_filter', 'all');
        $customer_id = $request->get('customer_id');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get customers with due balances and ageing information (same logic as web view)
        $customer_due_data = [];
        
        $customer_query = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select(
                'contacts.id',
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.customer_group_id',
                DB::raw("COALESCE(contacts.supplier_business_name, contacts.name) as display_name")
            );

        // Apply customer filter
        if ($customer_filter == 'single' && !empty($customer_id) && $customer_id != 'all') {
            $customer_query->where('contacts.id', $customer_id);
        }

        $customers_list = $customer_query->orderBy('display_name', 'asc')->get();

        foreach ($customers_list as $customer) {
            // Get all sales transactions for this customer up to end_date
            $sales_query = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.contact_id', $customer->id)
                ->whereIn('t.type', ['sell', 'opening_balance'])
                ->whereDate('t.transaction_date', '<=', $end_date)
                ->where(function($q) {
                    $q->where(function($q2) {
                        $q2->where('t.type', 'sell')->where('t.status', 'final');
                    })->orWhere('t.type', 'opening_balance');
                });

            // Apply location filter
            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    continue;
                }
                $sales_query->where('t.location_id', $location_id);
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $sales_query->whereIn('t.location_id', $permitted_locations);
            }

            $all_transactions = $sales_query->select('t.id', 't.type', 't.final_total', 't.transaction_date')
                ->orderBy('t.transaction_date', 'desc')
                ->get();

            // Calculate total invoice and total paid
            $total_invoice = 0;
            $total_paid = 0;

            foreach ($all_transactions as $transaction) {
                if ($transaction->type == 'sell' || $transaction->type == 'opening_balance') {
                    $total_invoice += $transaction->final_total ?? 0;
                }

                $payments = DB::table('transaction_payments')
                    ->where('transaction_id', $transaction->id)
                    ->whereDate('paid_on', '<=', $end_date)
                    ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                $total_paid += $payments ?? 0;
            }

            // Calculate current due balance
            $current_due = $total_invoice - $total_paid;

            // Only include customers with due balance > 0.01
            if ($current_due > 0.01) {
                // Get last sales transaction
                $last_sales = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $end_date);

                if (!empty($location_id)) {
                    $last_sales->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                    $last_sales->whereIn('t.location_id', $permitted_locations);
                }

                $last_sales = $last_sales->select('t.transaction_date', 't.final_total')
                    ->orderBy('t.transaction_date', 'desc')
                    ->first();

                $last_sales_date = $last_sales->transaction_date ?? null;
                $last_sales_amount = $last_sales->final_total ?? 0;
                $days_since_last_sales = $last_sales_date ? \Carbon\Carbon::parse($last_sales_date)->diffInDays(\Carbon\Carbon::parse($end_date)) : null;

                // Get last receipt/payment
                $last_receipt = DB::table('transaction_payments as tp')
                    ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer->id)
                    ->whereIn('t.type', ['sell', 'opening_balance'])
                    ->whereDate('tp.paid_on', '<=', $end_date)
                    ->where('tp.is_return', 0);

                if (!empty($location_id)) {
                    $last_receipt->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                    $last_receipt->whereIn('t.location_id', $permitted_locations);
                }

                $last_receipt = $last_receipt->select('tp.paid_on', 'tp.amount')
                    ->orderBy('tp.paid_on', 'desc')
                    ->first();

                $last_receipt_date = $last_receipt->paid_on ?? null;
                $last_receipt_amount = $last_receipt->amount ?? 0;
                $days_since_last_receipt = $last_receipt_date ? \Carbon\Carbon::parse($last_receipt_date)->diffInDays(\Carbon\Carbon::parse($end_date)) : null;

                // Get customer group name for categorization
                $customer_group_name = '';
                if (!empty($customer->customer_group_id)) {
                    $customer_group = DB::table('customer_groups')
                        ->where('id', $customer->customer_group_id)
                        ->value('name');
                    $customer_group_name = $customer_group ?? '';
                }

                $customer_due_data[] = (object)[
                    'id' => $customer->id,
                    'customer_name' => $customer->display_name,
                    'customer_group_name' => $customer_group_name,
                    'last_sales_date' => $last_sales_date,
                    'last_sales_amount' => $last_sales_amount,
                    'days_since_last_sales' => $days_since_last_sales,
                    'last_receipt_date' => $last_receipt_date,
                    'last_receipt_amount' => $last_receipt_amount,
                    'days_since_last_receipt' => $days_since_last_receipt,
                    'current_due' => $current_due,
                ];
            }
        }

        // Group by customer group for subtotals
        $grouped_data = [];
        foreach ($customer_due_data as $item) {
            $group_key = !empty($item->customer_group_name) ? $item->customer_group_name : 'Other';
            if (!isset($grouped_data[$group_key])) {
                $grouped_data[$group_key] = [];
            }
            $grouped_data[$group_key][] = $item;
        }

        // Calculate grand total
        $grand_total = collect($customer_due_data)->sum('current_due');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_customer_due_with_ageing', compact(
            'customer_due_data',
            'grouped_data',
            'grand_total',
            'end_date',
            'customer_filter',
            'customer_id',
            'location_id',
            'util',
            'business'
        ));
    }

    /**
     * Cash Flows Report
     */
    public function cashFlows(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->startOfMonth()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id);

        // Get cash account IDs (accounts with 'cash' or 'petty' in name)
        $cash_accounts = DB::table('accounts')
            ->where('business_id', $business_id)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%cash%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%petty%']);
            })
            ->where('is_closed', 0)
            ->pluck('id')
            ->toArray();

        // Helper function to calculate cash flows for a specific date
        $calculateCashFlows = function($as_of_date) use ($business_id, $location_id, $permitted_locations, $cash_accounts) {
            $revenue_data = [];
            $expenses_data = [];
            
            // Revenue - Cash Receipts from sales
            $sales_receipts_query = DB::table('transaction_payments as tp')
                ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
                ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->where('tp.is_return', 0)
                ->whereDate('tp.paid_on', '<=', $as_of_date);

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    return ['revenue' => [], 'expenses' => []];
                }
                $sales_receipts_query->where('t.location_id', $location_id);
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $sales_receipts_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($cash_accounts)) {
                $sales_receipts_query->whereIn('tp.account_id', $cash_accounts);
            }

            $sales_receipts = $sales_receipts_query
                ->select(
                    'a.name as account_name',
                    DB::raw("COALESCE(a.name, 'Cash') as account"),
                    DB::raw('SUM(tp.amount) as total_amount')
                )
                ->groupBy('a.id', 'a.name')
                ->get();

            foreach ($sales_receipts as $receipt) {
                $account_name = $receipt->account ?? 'Cash';
                if (!isset($revenue_data[$account_name])) {
                    $revenue_data[$account_name] = 0;
                }
                $revenue_data[$account_name] += (float) $receipt->total_amount;
            }

            // Revenue - Direct receipts from account transactions
            $direct_receipts_query = DB::table('account_transactions as at')
                ->join('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'credit')
                ->whereDate('at.operation_date', '<=', $as_of_date)
                ->whereNull('at.deleted_at');

            if (!empty($location_id)) {
                $direct_receipts_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.location_id');
                });
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $direct_receipts_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.location_id');
                });
            }

            if (!empty($cash_accounts)) {
                $direct_receipts_query->whereIn('at.account_id', $cash_accounts);
            }

            $direct_receipts = $direct_receipts_query
                ->select(
                    'a.name as account_name',
                    DB::raw("COALESCE(a.name, 'Cash') as account"),
                    DB::raw('SUM(at.amount) as total_amount')
                )
                ->groupBy('a.id', 'a.name')
                ->get();

            foreach ($direct_receipts as $receipt) {
                $account_name = $receipt->account ?? 'Cash';
                if (!isset($revenue_data[$account_name])) {
                    $revenue_data[$account_name] = 0;
                }
                $revenue_data[$account_name] += (float) $receipt->total_amount;
            }

            // Expenses - Cash Payments from purchases
            $purchase_payments_query = DB::table('transaction_payments as tp')
                ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
                ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase')
                ->where('t.status', 'received')
                ->where('tp.is_return', 0)
                ->whereDate('tp.paid_on', '<=', $as_of_date);

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    return ['revenue' => $revenue_data, 'expenses' => []];
                }
                $purchase_payments_query->where('t.location_id', $location_id);
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $purchase_payments_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($cash_accounts)) {
                $purchase_payments_query->whereIn('tp.account_id', $cash_accounts);
            }

            $purchase_payments = $purchase_payments_query
                ->select(
                    'a.name as account_name',
                    DB::raw("COALESCE(a.name, 'Cash') as account"),
                    DB::raw('SUM(tp.amount) as total_amount')
                )
                ->groupBy('a.id', 'a.name')
                ->get();

            foreach ($purchase_payments as $payment) {
                $account_name = $payment->account ?? 'Cash';
                if (!isset($expenses_data[$account_name])) {
                    $expenses_data[$account_name] = 0;
                }
                $expenses_data[$account_name] += (float) $payment->total_amount;
            }

            // Expenses - Cash Payments from expenses
            $expense_payments_query = DB::table('transaction_payments as tp')
                ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
                ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'expense')
                ->where('t.status', 'final')
                ->whereDate('tp.paid_on', '<=', $as_of_date);

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    return ['revenue' => $revenue_data, 'expenses' => $expenses_data];
                }
                $expense_payments_query->where('t.location_id', $location_id);
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $expense_payments_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($cash_accounts)) {
                $expense_payments_query->whereIn('tp.account_id', $cash_accounts);
            }

            $expense_payments = $expense_payments_query
                ->select(
                    'a.name as account_name',
                    DB::raw("COALESCE(a.name, 'Cash') as account"),
                    DB::raw('SUM(tp.amount) as total_amount')
                )
                ->groupBy('a.id', 'a.name')
                ->get();

            foreach ($expense_payments as $payment) {
                $account_name = $payment->account ?? 'Cash';
                if (!isset($expenses_data[$account_name])) {
                    $expenses_data[$account_name] = 0;
                }
                $expenses_data[$account_name] += (float) $payment->total_amount;
            }

            // Expenses - Direct payments from account transactions
            $direct_payments_query = DB::table('account_transactions as at')
                ->join('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->where('a.business_id', $business_id)
                ->where('at.type', 'debit')
                ->whereDate('at.operation_date', '<=', $as_of_date)
                ->whereNull('at.deleted_at');

            if (!empty($location_id)) {
                $direct_payments_query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.location_id');
                });
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $direct_payments_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.location_id');
                });
            }

            if (!empty($cash_accounts)) {
                $direct_payments_query->whereIn('at.account_id', $cash_accounts);
            }

            $direct_payments = $direct_payments_query
                ->select(
                    'a.name as account_name',
                    DB::raw("COALESCE(a.name, 'Cash') as account"),
                    DB::raw('SUM(at.amount) as total_amount')
                )
                ->groupBy('a.id', 'a.name')
                ->get();

            foreach ($direct_payments as $payment) {
                $account_name = $payment->account ?? 'Cash';
                if (!isset($expenses_data[$account_name])) {
                    $expenses_data[$account_name] = 0;
                }
                $expenses_data[$account_name] += (float) $payment->total_amount;
            }

            return ['revenue' => $revenue_data, 'expenses' => $expenses_data];
        };

        // Calculate for period 1 (start_date)
        $period1 = $calculateCashFlows($start_date);
        $revenue_data_period1 = $period1['revenue'];
        $expenses_data_period1 = $period1['expenses'];

        // Calculate for period 2 (end_date)
        $period2 = $calculateCashFlows($end_date);
        $revenue_data_period2 = $period2['revenue'];
        $expenses_data_period2 = $period2['expenses'];

        // Get all unique account names from both periods
        $all_revenue_accounts = array_unique(array_merge(array_keys($revenue_data_period1), array_keys($revenue_data_period2)));
        $all_expense_accounts = array_unique(array_merge(array_keys($expenses_data_period1), array_keys($expenses_data_period2)));

        // Calculate totals
        $total_revenue_period1 = array_sum($revenue_data_period1);
        $total_revenue_period2 = array_sum($revenue_data_period2);
        $total_expenses_period1 = array_sum($expenses_data_period1);
        $total_expenses_period2 = array_sum($expenses_data_period2);

        $util = new Util();

        return view('businessmanagement::accounts_register.cash_flows', compact(
            'revenue_data_period1',
            'revenue_data_period2',
            'expenses_data_period1',
            'expenses_data_period2',
            'all_revenue_accounts',
            'all_expense_accounts',
            'total_revenue_period1',
            'total_revenue_period2',
            'total_expenses_period1',
            'total_expenses_period2',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'util'
        ));
    }

    /**
     * Print Cash Flows Report
     */
    public function printCashFlows(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->startOfMonth()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $location_id = $request->get('location_id');

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get cash account IDs (accounts with 'cash' or 'petty' in name)
        $cash_accounts = DB::table('accounts')
            ->where('business_id', $business_id)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%cash%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%petty%']);
            })
            ->where('is_closed', 0)
            ->pluck('id')
            ->toArray();

        // Revenue Section - Cash Receipts (same logic as web view)
        $revenue_data = [];
        
        // Get cash receipts from sales
        $sales_receipts_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->where('tp.is_return', 0)
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_receipts_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $sales_receipts_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($cash_accounts)) {
            $sales_receipts_query->whereIn('tp.account_id', $cash_accounts);
        }

        $sales_receipts = $sales_receipts_query
            ->select(
                'a.name as account_name',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(tp.amount) as total_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($sales_receipts as $receipt) {
            $account_name = $receipt->account ?? 'Cash';
            if (!isset($revenue_data[$account_name])) {
                $revenue_data[$account_name] = 0;
            }
            $revenue_data[$account_name] += (float) $receipt->total_amount;
        }

        // Get cash receipts from account transactions
        $direct_receipts_query = DB::table('account_transactions as at')
            ->join('accounts as a', 'at.account_id', '=', 'a.id')
            ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
            ->where('a.business_id', $business_id)
            ->where('at.type', 'credit')
            ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date])
            ->whereNull('at.deleted_at');

        if (!empty($location_id)) {
            $direct_receipts_query->where(function($q) use ($location_id) {
                $q->where('t.location_id', $location_id)
                  ->orWhereNull('t.location_id');
            });
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $direct_receipts_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('t.location_id', $permitted_locations)
                  ->orWhereNull('t.location_id');
            });
        }

        if (!empty($cash_accounts)) {
            $direct_receipts_query->whereIn('at.account_id', $cash_accounts);
        }

        $direct_receipts = $direct_receipts_query
            ->select(
                'a.name as account_name',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(at.amount) as total_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($direct_receipts as $receipt) {
            $account_name = $receipt->account ?? 'Cash';
            if (!isset($revenue_data[$account_name])) {
                $revenue_data[$account_name] = 0;
            }
            $revenue_data[$account_name] += (float) $receipt->total_amount;
        }

        // Expenses Section - Cash Payments (same logic as web view)
        $expenses_data = [];
        
        // Get cash payments from purchases
        $purchase_payments_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->where('tp.is_return', 0)
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $purchase_payments_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $purchase_payments_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($cash_accounts)) {
            $purchase_payments_query->whereIn('tp.account_id', $cash_accounts);
        }

        $purchase_payments = $purchase_payments_query
            ->select(
                'a.name as account_name',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(tp.amount) as total_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($purchase_payments as $payment) {
            $account_name = $payment->account ?? 'Cash';
            if (!isset($expenses_data[$account_name])) {
                $expenses_data[$account_name] = 0;
            }
            $expenses_data[$account_name] += (float) $payment->total_amount;
        }

        // Get cash payments from expenses
        $expense_payments_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'expense')
            ->where('t.status', 'final')
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $expense_payments_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $expense_payments_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($cash_accounts)) {
            $expense_payments_query->whereIn('tp.account_id', $cash_accounts);
        }

        $expense_payments = $expense_payments_query
            ->select(
                'a.name as account_name',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(tp.amount) as total_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($expense_payments as $payment) {
            $account_name = $payment->account ?? 'Cash';
            if (!isset($expenses_data[$account_name])) {
                $expenses_data[$account_name] = 0;
            }
            $expenses_data[$account_name] += (float) $payment->total_amount;
        }

        // Get cash payments from account transactions
        $direct_payments_query = DB::table('account_transactions as at')
            ->join('accounts as a', 'at.account_id', '=', 'a.id')
            ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
            ->where('a.business_id', $business_id)
            ->where('at.type', 'debit')
            ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date])
            ->whereNull('at.deleted_at');

        if (!empty($location_id)) {
            $direct_payments_query->where(function($q) use ($location_id) {
                $q->where('t.location_id', $location_id)
                  ->orWhereNull('t.location_id');
            });
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $direct_payments_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('t.location_id', $permitted_locations)
                  ->orWhereNull('t.location_id');
            });
        }

        if (!empty($cash_accounts)) {
            $direct_payments_query->whereIn('at.account_id', $cash_accounts);
        }

        $direct_payments = $direct_payments_query
            ->select(
                'a.name as account_name',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(at.amount) as total_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($direct_payments as $payment) {
            $account_name = $payment->account ?? 'Cash';
            if (!isset($expenses_data[$account_name])) {
                $expenses_data[$account_name] = 0;
            }
            $expenses_data[$account_name] += (float) $payment->total_amount;
        }

        // Calculate totals
        $total_revenue = array_sum($revenue_data);
        $total_expenses = array_sum($expenses_data);
        $net_cash_flow = $total_revenue - $total_expenses;

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_cash_flows', compact(
            'revenue_data',
            'expenses_data',
            'total_revenue',
            'total_expenses',
            'net_cash_flow',
            'start_date',
            'end_date',
            'location_id',
            'util',
            'business'
        ));
    }

    /**
     * Date Wise Receipt Payment Statement Report
     */
    public function dateWiseReceiptPaymentStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $receipt_payment_filter = $request->get('receipt_payment_filter', 'all');
        $receipt_account_id = $request->get('receipt_account_id');
        $payment_account_id = $request->get('payment_account_id');
        $location_id = $request->get('location_id');

        // Determine receipt and payment filters from combined filter
        $receipt_filter = 'all';
        $payment_filter = 'all';
        
        if ($receipt_payment_filter == 'receipt_all' || $receipt_payment_filter == 'receipt_single') {
            $receipt_filter = $receipt_payment_filter == 'receipt_single' ? 'single' : 'all';
        }
        
        if ($receipt_payment_filter == 'payment_all' || $receipt_payment_filter == 'payment_single') {
            $payment_filter = $receipt_payment_filter == 'payment_single' ? 'single' : 'all';
        }
        
        if ($receipt_payment_filter == 'all') {
            $receipt_filter = 'all';
            $payment_filter = 'all';
        }

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id);

        // Get cash account IDs
        $cash_accounts = DB::table('accounts')
            ->where('business_id', $business_id)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%cash%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%petty%']);
            })
            ->where('is_closed', 0)
            ->pluck('id')
            ->toArray();

        // Get bank account IDs
        $bank_accounts = DB::table('accounts')
            ->where('business_id', $business_id)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('id')
            ->toArray();

        // Get all accounts for dropdowns
        $all_accounts = DB::table('accounts')
            ->where('business_id', $business_id)
            ->where('is_closed', 0)
            ->select('id', 'name')
            ->orderBy('name', 'asc')
            ->get();

        $receipt_accounts = $all_accounts->pluck('name', 'id')->toArray();
        $payment_accounts = $all_accounts->pluck('name', 'id')->toArray();

        // Calculate Opening Balance (before start_date)
        $opening_cash = 0;
        $opening_bank = 0;

        if (!empty($cash_accounts)) {
            $opening_cash_query = DB::table('account_transactions as at')
                ->whereIn('at.account_id', $cash_accounts)
                ->whereDate('at.operation_date', '<', $start_date)
                ->whereNull('at.deleted_at');

            if (!empty($location_id)) {
                $opening_cash_query->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->where(function($q) use ($location_id) {
                        $q->where('t.location_id', $location_id)
                          ->orWhereNull('t.id');
                    });
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $opening_cash_query->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->where(function($q) use ($permitted_locations) {
                        $q->whereIn('t.location_id', $permitted_locations)
                          ->orWhereNull('t.id');
                    });
            }

            $opening_cash = $opening_cash_query
                ->select(DB::raw("SUM(IF(at.type='credit', at.amount, -1 * at.amount)) as balance"))
                ->value('balance') ?? 0;
        }

        if (!empty($bank_accounts)) {
            $opening_bank_query = DB::table('account_transactions as at')
                ->whereIn('at.account_id', $bank_accounts)
                ->whereDate('at.operation_date', '<', $start_date)
                ->whereNull('at.deleted_at');

            if (!empty($location_id)) {
                $opening_bank_query->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->where(function($q) use ($location_id) {
                        $q->where('t.location_id', $location_id)
                          ->orWhereNull('t.id');
                    });
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $opening_bank_query->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->where(function($q) use ($permitted_locations) {
                        $q->whereIn('t.location_id', $permitted_locations)
                          ->orWhereNull('t.id');
                    });
            }

            $opening_bank = $opening_bank_query
                ->select(DB::raw("SUM(IF(at.type='credit', at.amount, -1 * at.amount)) as balance"))
                ->value('balance') ?? 0;
        }

        $opening_total = $opening_cash + $opening_bank;

        // Get Receipts (money coming in) - grouped by account
        $receipt_data = [];

        // Receipts from sales
        $sales_receipts_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->where('tp.is_return', 0)
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_receipts_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $sales_receipts_query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply receipt account filter
        // When receipt_payment_filter is 'receipt_all', show all receipts (no filter)
        // When receipt_payment_filter is 'receipt_single', filter by selected account
        if ($receipt_payment_filter == 'receipt_single' && !empty($receipt_account_id) && $receipt_account_id != 'all') {
            $sales_receipts_query->where('tp.account_id', $receipt_account_id);
        }

        $sales_receipts = $sales_receipts_query
            ->select(
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as bank_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($sales_receipts as $receipt) {
            $account_name = $receipt->account ?? 'Cash';
            if (!isset($receipt_data[$account_name])) {
                $receipt_data[$account_name] = ['cash' => 0, 'bank' => 0, 'total' => 0];
            }
            $receipt_data[$account_name]['cash'] += (float) $receipt->cash_amount;
            $receipt_data[$account_name]['bank'] += (float) $receipt->bank_amount;
            $receipt_data[$account_name]['total'] += (float) $receipt->cash_amount + (float) $receipt->bank_amount;
        }

        // Receipts from account transactions (direct receipts)
        $direct_receipts_query = DB::table('account_transactions as at')
            ->join('accounts as a', 'at.account_id', '=', 'a.id')
            ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
            ->where('a.business_id', $business_id)
            ->where('at.type', 'credit')
            ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date])
            ->whereNull('at.deleted_at');

        if (!empty($location_id)) {
            $direct_receipts_query->where(function($q) use ($location_id) {
                $q->where('t.location_id', $location_id)
                  ->orWhereNull('t.id');
            });
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $direct_receipts_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('t.location_id', $permitted_locations)
                  ->orWhereNull('t.id');
            });
        }

        // Apply receipt account filter
        if ($receipt_payment_filter == 'receipt_single' && !empty($receipt_account_id) && $receipt_account_id != 'all') {
            $direct_receipts_query->where('at.account_id', $receipt_account_id);
        }

        $direct_receipts = $direct_receipts_query
            ->select(
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN at.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN at.amount ELSE 0 END) as bank_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($direct_receipts as $receipt) {
            $account_name = $receipt->account ?? 'Cash';
            if (!isset($receipt_data[$account_name])) {
                $receipt_data[$account_name] = ['cash' => 0, 'bank' => 0, 'total' => 0];
            }
            $receipt_data[$account_name]['cash'] += (float) $receipt->cash_amount;
            $receipt_data[$account_name]['bank'] += (float) $receipt->bank_amount;
            $receipt_data[$account_name]['total'] += (float) $receipt->cash_amount + (float) $receipt->bank_amount;
        }

        // Get Payments (money going out) - grouped by account/category
        $payment_data = [];

        // Payments from purchases
        $purchase_payments_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->where('tp.is_return', 0)
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $purchase_payments_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $purchase_payments_query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply payment account filter
        // When receipt_payment_filter is 'payment_all', show all payments (no filter)
        // When receipt_payment_filter is 'payment_single', filter by selected account
        if ($receipt_payment_filter == 'payment_single' && !empty($payment_account_id) && $payment_account_id != 'all') {
            $purchase_payments_query->where('tp.account_id', $payment_account_id);
        }

        $purchase_payments = $purchase_payments_query
            ->select(
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as bank_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($purchase_payments as $payment) {
            $account_name = $payment->account ?? 'Cash';
            if (!isset($payment_data[$account_name])) {
                $payment_data[$account_name] = ['cash' => 0, 'bank' => 0, 'total' => 0, 'items' => []];
            }
            $payment_data[$account_name]['cash'] += (float) $payment->cash_amount;
            $payment_data[$account_name]['bank'] += (float) $payment->bank_amount;
            $payment_data[$account_name]['total'] += (float) $payment->cash_amount + (float) $payment->bank_amount;
        }

        // Payments from expenses
        $expense_payments_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'expense')
            ->where('t.status', 'final')
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $expense_payments_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $expense_payments_query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply payment account filter
        if ($receipt_payment_filter == 'payment_single' && !empty($payment_account_id) && $payment_account_id != 'all') {
            $expense_payments_query->where('tp.account_id', $payment_account_id);
        }

        $expense_payments = $expense_payments_query
            ->select(
                'ec.name as category_name',
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(ec.name, 'Cash') as category"),
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as bank_amount'),
                't.final_total',
                't.ref_no',
                't.additional_notes'
            )
            ->groupBy('ec.id', 'ec.name', 'a.id', 'a.name', 't.id', 't.final_total', 't.ref_no', 't.additional_notes')
            ->get();

        foreach ($expense_payments as $payment) {
            $category_name = $payment->category ?? 'Cash';
            if (!isset($payment_data[$category_name])) {
                $payment_data[$category_name] = ['cash' => 0, 'bank' => 0, 'total' => 0, 'items' => []];
            }
            $payment_data[$category_name]['cash'] += (float) $payment->cash_amount;
            $payment_data[$category_name]['bank'] += (float) $payment->bank_amount;
            $payment_data[$category_name]['total'] += (float) $payment->cash_amount + (float) $payment->bank_amount;

            // Add expense item details
            $item_name = !empty($payment->ref_no) ? $payment->ref_no : 
                        (!empty($payment->additional_notes) ? $payment->additional_notes : 
                        ($payment->account ?? 'Expense'));
            $payment_data[$category_name]['items'][] = [
                'name' => $item_name,
                'cash' => (float) $payment->cash_amount,
                'bank' => (float) $payment->bank_amount,
                'total' => (float) $payment->cash_amount + (float) $payment->bank_amount
            ];
        }

        // Payments from account transactions (direct payments)
        $direct_payments_query = DB::table('account_transactions as at')
            ->join('accounts as a', 'at.account_id', '=', 'a.id')
            ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
            ->where('a.business_id', $business_id)
            ->where('at.type', 'debit')
            ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date])
            ->whereNull('at.deleted_at');

        if (!empty($location_id)) {
            $direct_payments_query->where(function($q) use ($location_id) {
                $q->where('t.location_id', $location_id)
                  ->orWhereNull('t.id');
            });
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $direct_payments_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('t.location_id', $permitted_locations)
                  ->orWhereNull('t.id');
            });
        }

        // Apply payment account filter
        if ($receipt_payment_filter == 'payment_single' && !empty($payment_account_id) && $payment_account_id != 'all') {
            $direct_payments_query->where('at.account_id', $payment_account_id);
        }

        $direct_payments = $direct_payments_query
            ->select(
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN at.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN at.amount ELSE 0 END) as bank_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($direct_payments as $payment) {
            $account_name = $payment->account ?? 'Cash';
            if (!isset($payment_data[$account_name])) {
                $payment_data[$account_name] = ['cash' => 0, 'bank' => 0, 'total' => 0, 'items' => []];
            }
            $payment_data[$account_name]['cash'] += (float) $payment->cash_amount;
            $payment_data[$account_name]['bank'] += (float) $payment->bank_amount;
            $payment_data[$account_name]['total'] += (float) $payment->cash_amount + (float) $payment->bank_amount;
        }

        // Calculate totals
        $total_receipt_cash = array_sum(array_column($receipt_data, 'cash'));
        $total_receipt_bank = array_sum(array_column($receipt_data, 'bank'));
        $total_receipt_total = array_sum(array_column($receipt_data, 'total'));

        $total_payment_cash = array_sum(array_column($payment_data, 'cash'));
        $total_payment_bank = array_sum(array_column($payment_data, 'bank'));
        $total_payment_total = array_sum(array_column($payment_data, 'total'));

        // Calculate Closing Balance
        $closing_cash = $opening_cash + $total_receipt_cash - $total_payment_cash;
        $closing_bank = $opening_bank + $total_receipt_bank - $total_payment_bank;
        $closing_total = $closing_cash + $closing_bank;

        $util = new Util();

        return view('businessmanagement::accounts_register.date_wise_receipt_payment_statement', compact(
            'opening_cash',
            'opening_bank',
            'opening_total',
            'receipt_data',
            'payment_data',
            'total_receipt_cash',
            'total_receipt_bank',
            'total_receipt_total',
            'total_payment_cash',
            'total_payment_bank',
            'total_payment_total',
            'closing_cash',
            'closing_bank',
            'closing_total',
            'start_date',
            'end_date',
            'receipt_payment_filter',
            'receipt_filter',
            'receipt_account_id',
            'payment_filter',
            'payment_account_id',
            'location_id',
            'locations',
            'receipt_accounts',
            'payment_accounts',
            'util'
        ));
    }

    /**
     * Print Date Wise Receipt Payment Statement Report
     */
    public function printDateWiseReceiptPaymentStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get filters
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        $receipt_payment_filter = $request->get('receipt_payment_filter', 'all');
        $receipt_account_id = $request->get('receipt_account_id');
        $payment_account_id = $request->get('payment_account_id');
        $location_id = $request->get('location_id');

        // Determine receipt and payment filters from combined filter
        $receipt_filter = 'all';
        $payment_filter = 'all';
        
        if ($receipt_payment_filter == 'receipt_all' || $receipt_payment_filter == 'receipt_single') {
            $receipt_filter = $receipt_payment_filter == 'receipt_single' ? 'single' : 'all';
        }
        
        if ($receipt_payment_filter == 'payment_all' || $receipt_payment_filter == 'payment_single') {
            $payment_filter = $receipt_payment_filter == 'payment_single' ? 'single' : 'all';
        }
        
        if ($receipt_payment_filter == 'all') {
            $receipt_filter = 'all';
            $payment_filter = 'all';
        }

        // Get permitted locations
        $permitted_locations = auth()->user()->permitted_locations();

        // Get cash account IDs
        $cash_accounts = DB::table('accounts')
            ->where('business_id', $business_id)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%cash%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%petty%']);
            })
            ->where('is_closed', 0)
            ->pluck('id')
            ->toArray();

        // Get bank account IDs
        $bank_accounts = DB::table('accounts')
            ->where('business_id', $business_id)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('id')
            ->toArray();

        // Calculate Opening Balance (before start_date) - same logic as web view
        $opening_cash = 0;
        $opening_bank = 0;

        if (!empty($cash_accounts)) {
            $opening_cash_query = DB::table('account_transactions as at')
                ->whereIn('at.account_id', $cash_accounts)
                ->whereDate('at.operation_date', '<', $start_date)
                ->whereNull('at.deleted_at');

            if (!empty($location_id)) {
                $opening_cash_query->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->where(function($q) use ($location_id) {
                        $q->where('t.location_id', $location_id)
                          ->orWhereNull('t.id');
                    });
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $opening_cash_query->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->where(function($q) use ($permitted_locations) {
                        $q->whereIn('t.location_id', $permitted_locations)
                          ->orWhereNull('t.id');
                    });
            }

            $opening_cash = $opening_cash_query
                ->select(DB::raw("SUM(IF(at.type='credit', at.amount, -1 * at.amount)) as balance"))
                ->value('balance') ?? 0;
        }

        if (!empty($bank_accounts)) {
            $opening_bank_query = DB::table('account_transactions as at')
                ->whereIn('at.account_id', $bank_accounts)
                ->whereDate('at.operation_date', '<', $start_date)
                ->whereNull('at.deleted_at');

            if (!empty($location_id)) {
                $opening_bank_query->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->where(function($q) use ($location_id) {
                        $q->where('t.location_id', $location_id)
                          ->orWhereNull('t.id');
                    });
            } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
                $opening_bank_query->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                    ->where(function($q) use ($permitted_locations) {
                        $q->whereIn('t.location_id', $permitted_locations)
                          ->orWhereNull('t.id');
                    });
            }

            $opening_bank = $opening_bank_query
                ->select(DB::raw("SUM(IF(at.type='credit', at.amount, -1 * at.amount)) as balance"))
                ->value('balance') ?? 0;
        }

        $opening_total = $opening_cash + $opening_bank;

        // Get Receipts (same logic as web view)
        $receipt_data = [];

        $sales_receipts_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->where('tp.is_return', 0)
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_receipts_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $sales_receipts_query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply receipt account filter
        if ($receipt_payment_filter == 'receipt_single' && !empty($receipt_account_id) && $receipt_account_id != 'all') {
            $sales_receipts_query->where('tp.account_id', $receipt_account_id);
        }

        $sales_receipts = $sales_receipts_query
            ->select(
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as bank_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($sales_receipts as $receipt) {
            $account_name = $receipt->account ?? 'Cash';
            if (!isset($receipt_data[$account_name])) {
                $receipt_data[$account_name] = ['cash' => 0, 'bank' => 0, 'total' => 0];
            }
            $receipt_data[$account_name]['cash'] += (float) $receipt->cash_amount;
            $receipt_data[$account_name]['bank'] += (float) $receipt->bank_amount;
            $receipt_data[$account_name]['total'] += (float) $receipt->cash_amount + (float) $receipt->bank_amount;
        }

        $direct_receipts_query = DB::table('account_transactions as at')
            ->join('accounts as a', 'at.account_id', '=', 'a.id')
            ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
            ->where('a.business_id', $business_id)
            ->where('at.type', 'credit')
            ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date])
            ->whereNull('at.deleted_at');

        if (!empty($location_id)) {
            $direct_receipts_query->where(function($q) use ($location_id) {
                $q->where('t.location_id', $location_id)
                  ->orWhereNull('t.id');
            });
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $direct_receipts_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('t.location_id', $permitted_locations)
                  ->orWhereNull('t.id');
            });
        }

        // Apply receipt account filter
        if ($receipt_payment_filter == 'receipt_single' && !empty($receipt_account_id) && $receipt_account_id != 'all') {
            $direct_receipts_query->where('at.account_id', $receipt_account_id);
        }

        $direct_receipts = $direct_receipts_query
            ->select(
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN at.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN at.amount ELSE 0 END) as bank_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($direct_receipts as $receipt) {
            $account_name = $receipt->account ?? 'Cash';
            if (!isset($receipt_data[$account_name])) {
                $receipt_data[$account_name] = ['cash' => 0, 'bank' => 0, 'total' => 0];
            }
            $receipt_data[$account_name]['cash'] += (float) $receipt->cash_amount;
            $receipt_data[$account_name]['bank'] += (float) $receipt->bank_amount;
            $receipt_data[$account_name]['total'] += (float) $receipt->cash_amount + (float) $receipt->bank_amount;
        }

        // Get Payments (same logic as web view)
        $payment_data = [];

        $purchase_payments_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->where('tp.is_return', 0)
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $purchase_payments_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $purchase_payments_query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply payment account filter
        if ($receipt_payment_filter == 'payment_single' && !empty($payment_account_id) && $payment_account_id != 'all') {
            $purchase_payments_query->where('tp.account_id', $payment_account_id);
        }

        $purchase_payments = $purchase_payments_query
            ->select(
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as bank_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($purchase_payments as $payment) {
            $account_name = $payment->account ?? 'Cash';
            if (!isset($payment_data[$account_name])) {
                $payment_data[$account_name] = ['cash' => 0, 'bank' => 0, 'total' => 0, 'items' => []];
            }
            $payment_data[$account_name]['cash'] += (float) $payment->cash_amount;
            $payment_data[$account_name]['bank'] += (float) $payment->bank_amount;
            $payment_data[$account_name]['total'] += (float) $payment->cash_amount + (float) $payment->bank_amount;
        }

        $expense_payments_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'expense')
            ->where('t.status', 'final')
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $expense_payments_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $expense_payments_query->whereIn('t.location_id', $permitted_locations);
        }

        // Apply payment account filter
        if ($receipt_payment_filter == 'payment_single' && !empty($payment_account_id) && $payment_account_id != 'all') {
            $expense_payments_query->where('tp.account_id', $payment_account_id);
        }

        $expense_payments = $expense_payments_query
            ->select(
                'ec.name as category_name',
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(ec.name, 'Cash') as category"),
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN tp.amount ELSE 0 END) as bank_amount'),
                't.final_total',
                't.ref_no',
                't.additional_notes'
            )
            ->groupBy('ec.id', 'ec.name', 'a.id', 'a.name', 't.id', 't.final_total', 't.ref_no', 't.additional_notes')
            ->get();

        foreach ($expense_payments as $payment) {
            $category_name = $payment->category ?? 'Cash';
            if (!isset($payment_data[$category_name])) {
                $payment_data[$category_name] = ['cash' => 0, 'bank' => 0, 'total' => 0, 'items' => []];
            }
            $payment_data[$category_name]['cash'] += (float) $payment->cash_amount;
            $payment_data[$category_name]['bank'] += (float) $payment->bank_amount;
            $payment_data[$category_name]['total'] += (float) $payment->cash_amount + (float) $payment->bank_amount;

            $item_name = !empty($payment->ref_no) ? $payment->ref_no : 
                        (!empty($payment->additional_notes) ? $payment->additional_notes : 
                        ($payment->account ?? 'Expense'));
            $payment_data[$category_name]['items'][] = [
                'name' => $item_name,
                'cash' => (float) $payment->cash_amount,
                'bank' => (float) $payment->bank_amount,
                'total' => (float) $payment->cash_amount + (float) $payment->bank_amount
            ];
        }

        $direct_payments_query = DB::table('account_transactions as at')
            ->join('accounts as a', 'at.account_id', '=', 'a.id')
            ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
            ->where('a.business_id', $business_id)
            ->where('at.type', 'debit')
            ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date])
            ->whereNull('at.deleted_at');

        if (!empty($location_id)) {
            $direct_payments_query->where(function($q) use ($location_id) {
                $q->where('t.location_id', $location_id)
                  ->orWhereNull('t.id');
            });
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $direct_payments_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('t.location_id', $permitted_locations)
                  ->orWhereNull('t.id');
            });
        }

        // Apply payment account filter
        if ($receipt_payment_filter == 'payment_single' && !empty($payment_account_id) && $payment_account_id != 'all') {
            $direct_payments_query->where('at.account_id', $payment_account_id);
        }

        $direct_payments = $direct_payments_query
            ->select(
                'a.name as account_name',
                'a.id as account_id',
                DB::raw("COALESCE(a.name, 'Cash') as account"),
                DB::raw('SUM(CASE WHEN ' . (!empty($cash_accounts) ? 'a.id IN (' . implode(',', $cash_accounts) . ')' : '1=0') . ' THEN at.amount ELSE 0 END) as cash_amount'),
                DB::raw('SUM(CASE WHEN ' . (!empty($bank_accounts) ? 'a.id IN (' . implode(',', $bank_accounts) . ')' : '1=0') . ' THEN at.amount ELSE 0 END) as bank_amount')
            )
            ->groupBy('a.id', 'a.name')
            ->get();

        foreach ($direct_payments as $payment) {
            $account_name = $payment->account ?? 'Cash';
            if (!isset($payment_data[$account_name])) {
                $payment_data[$account_name] = ['cash' => 0, 'bank' => 0, 'total' => 0, 'items' => []];
            }
            $payment_data[$account_name]['cash'] += (float) $payment->cash_amount;
            $payment_data[$account_name]['bank'] += (float) $payment->bank_amount;
            $payment_data[$account_name]['total'] += (float) $payment->cash_amount + (float) $payment->bank_amount;
        }

        // Calculate totals
        $total_receipt_cash = array_sum(array_column($receipt_data, 'cash'));
        $total_receipt_bank = array_sum(array_column($receipt_data, 'bank'));
        $total_receipt_total = array_sum(array_column($receipt_data, 'total'));

        $total_payment_cash = array_sum(array_column($payment_data, 'cash'));
        $total_payment_bank = array_sum(array_column($payment_data, 'bank'));
        $total_payment_total = array_sum(array_column($payment_data, 'total'));

        // Calculate Closing Balance
        $closing_cash = $opening_cash + $total_receipt_cash - $total_payment_cash;
        $closing_bank = $opening_bank + $total_receipt_bank - $total_payment_bank;
        $closing_total = $closing_cash + $closing_bank;

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_date_wise_receipt_payment_statement', compact(
            'opening_cash',
            'opening_bank',
            'opening_total',
            'receipt_data',
            'payment_data',
            'total_receipt_cash',
            'total_receipt_bank',
            'total_receipt_total',
            'total_payment_cash',
            'total_payment_bank',
            'total_payment_total',
            'closing_cash',
            'closing_bank',
            'closing_total',
            'start_date',
            'end_date',
            'receipt_payment_filter',
            'receipt_filter',
            'receipt_account_id',
            'payment_filter',
            'payment_account_id',
            'location_id',
            'util',
            'business'
        ));
    }

    /**
     * Date Wise Receive Report
     */
    public function dateWiseReceive(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $util = new \App\Utils\Util();
        
        $start_date = $request->get('start_date', date('Y-m-d'));
        $end_date = $request->get('end_date', date('Y-m-d'));
        $location_id = $request->get('location_id');

        $permitted_locations = auth()->user()->permitted_locations();
        $locations = BusinessLocation::forDropdown($business_id, false, false);

        // Get cash receipts from sales (customer payments)
        $sales_receipts_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('users as u', 't.created_by', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->where('tp.is_return', 0)
            ->where('tp.method', 'cash') // Only cash receipts
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_receipts_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $sales_receipts_query->whereIn('t.location_id', $permitted_locations);
        }

        $sales_receipts = $sales_receipts_query
            ->select(
                'tp.paid_on as date',
                't.invoice_no as receive_id',
                DB::raw("COALESCE(c.supplier_business_name, c.name, 'WALKING CUSTOMER') as received_from"),
                DB::raw("'CASH' as narration"),
                DB::raw("COALESCE(tp.payment_ref_no, '') as ref_no"),
                DB::raw("TRIM(CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, ''))) as prepared_by"),
                DB::raw("COALESCE(tp.cheque_number, '') as cheque_no"),
                DB::raw("NULL as cheque_date"),
                'tp.amount'
            )
            ->orderBy('tp.paid_on', 'asc')
            ->orderBy('tp.id', 'asc')
            ->get();

        // Get cash receipts from account transactions (direct receipts)
        $account_receipts_query = DB::table('account_transactions as at')
            ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
            ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
            ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
            ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
            ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
            ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
            ->where('a.business_id', $business_id)
            ->where('at.type', 'credit') // Credit = money received
            ->whereNull('at.deleted_at')
            ->where(function($q) {
                $q->where('tp.method', 'cash')
                  ->orWhereNull('tp.id');
            })
            ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $account_receipts_query->where(function($q) use ($location_id) {
                $q->where('t.location_id', $location_id)
                  ->orWhereNull('t.id');
            });
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $account_receipts_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('t.location_id', $permitted_locations)
                  ->orWhereNull('t.id');
            });
        }

        $account_receipts = $account_receipts_query
            ->select(
                'at.operation_date as date',
                DB::raw("CASE 
                    WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                    WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                    WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                    ELSE CONCAT('CRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                END as receive_id"),
                DB::raw("COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name, 'WALKING CUSTOMER') as received_from"),
                DB::raw("COALESCE(at.note, 'CASH') as narration"),
                DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                DB::raw("TRIM(CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, ''))) as prepared_by"),
                DB::raw("COALESCE(tp.cheque_number, '') as cheque_no"),
                DB::raw("NULL as cheque_date"),
                'at.amount'
            )
            ->orderBy('at.operation_date', 'asc')
            ->orderBy('at.id', 'asc')
            ->get();

        // Combine and sort all receipts
        $all_receipts = $sales_receipts->merge($account_receipts)
            ->sortBy(function($receipt) {
                return $receipt->date . ' ' . $receipt->receive_id;
            })
            ->values();

        $total_amount = $all_receipts->sum('amount');

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.date_wise_receive', compact(
            'all_receipts',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'total_amount',
            'util',
            'business'
        ));
    }

    /**
     * Print Date Wise Receive Report
     */
    public function printDateWiseReceive(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $util = new \App\Utils\Util();
        
        $start_date = $request->get('start_date', date('Y-m-d'));
        $end_date = $request->get('end_date', date('Y-m-d'));
        $location_id = $request->get('location_id');

        $permitted_locations = auth()->user()->permitted_locations();

        // Get cash receipts from sales (customer payments)
        $sales_receipts_query = DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('users as u', 't.created_by', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->where('tp.is_return', 0)
            ->where('tp.method', 'cash') // Only cash receipts
            ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $sales_receipts_query->where('t.location_id', $location_id);
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $sales_receipts_query->whereIn('t.location_id', $permitted_locations);
        }

        $sales_receipts = $sales_receipts_query
            ->select(
                'tp.paid_on as date',
                't.invoice_no as receive_id',
                DB::raw("COALESCE(c.supplier_business_name, c.name, 'WALKING CUSTOMER') as received_from"),
                DB::raw("'CASH' as narration"),
                DB::raw("COALESCE(tp.payment_ref_no, '') as ref_no"),
                DB::raw("TRIM(CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, ''))) as prepared_by"),
                DB::raw("COALESCE(tp.cheque_number, '') as cheque_no"),
                DB::raw("NULL as cheque_date"),
                'tp.amount'
            )
            ->orderBy('tp.paid_on', 'asc')
            ->orderBy('tp.id', 'asc')
            ->get();

        // Get cash receipts from account transactions (direct receipts)
        $account_receipts_query = DB::table('account_transactions as at')
            ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
            ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
            ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
            ->leftJoin('contacts as c_t', 't.contact_id', '=', 'c_t.id')
            ->leftJoin('contacts as c_p', 'tp.payment_for', '=', 'c_p.id')
            ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
            ->where('a.business_id', $business_id)
            ->where('at.type', 'credit') // Credit = money received
            ->whereNull('at.deleted_at')
            ->where(function($q) {
                $q->where('tp.method', 'cash')
                  ->orWhereNull('tp.id');
            })
            ->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);

        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $account_receipts_query->where(function($q) use ($location_id) {
                $q->where('t.location_id', $location_id)
                  ->orWhereNull('t.id');
            });
        } elseif ($permitted_locations != 'all' && is_array($permitted_locations)) {
            $account_receipts_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('t.location_id', $permitted_locations)
                  ->orWhereNull('t.id');
            });
        }

        $account_receipts = $account_receipts_query
            ->select(
                'at.operation_date as date',
                DB::raw("CASE 
                    WHEN at.reff_no IS NOT NULL AND at.reff_no != '' THEN at.reff_no
                    WHEN t.invoice_no IS NOT NULL AND t.invoice_no != '' THEN t.invoice_no
                    WHEN t.ref_no IS NOT NULL AND t.ref_no != '' THEN t.ref_no
                    ELSE CONCAT('CRV', DATE_FORMAT(at.operation_date, '%Y'), LPAD(at.id, 7, '0'))
                END as receive_id"),
                DB::raw("COALESCE(c_p.supplier_business_name, c_p.name, c_t.supplier_business_name, c_t.name, 'WALKING CUSTOMER') as received_from"),
                DB::raw("COALESCE(at.note, 'CASH') as narration"),
                DB::raw("COALESCE(NULLIF(at.reff_no, ''), NULLIF(tp.payment_ref_no, ''), NULLIF(t.ref_no, ''), '') as ref_no"),
                DB::raw("TRIM(CONCAT(COALESCE(u.surname, ''), ' ', COALESCE(u.first_name, ''), ' ', COALESCE(u.last_name, ''))) as prepared_by"),
                DB::raw("COALESCE(tp.cheque_number, '') as cheque_no"),
                DB::raw("NULL as cheque_date"),
                'at.amount'
            )
            ->orderBy('at.operation_date', 'asc')
            ->orderBy('at.id', 'asc')
            ->get();

        // Combine and sort all receipts
        $all_receipts = $sales_receipts->merge($account_receipts)
            ->sortBy(function($receipt) {
                return $receipt->date . ' ' . $receipt->receive_id;
            })
            ->values();

        $total_amount = $all_receipts->sum('amount');

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_date_wise_receive', compact(
            'all_receipts',
            'start_date',
            'end_date',
            'location_id',
            'total_amount',
            'util',
            'business'
        ));
    }

    /**
     * Date Wise Sales with Profit Report
     */
    public function dateWiseSalesWithProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $util = new \App\Utils\Util();
        
        $start_date = $request->get('start_date', date('Y-m-d'));
        $end_date = $request->get('end_date', date('Y-m-d'));
        $location_id = $request->get('location_id');

        $permitted_locations = auth()->user()->permitted_locations();
        $locations = BusinessLocation::forDropdown($business_id, false, false);

        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = date('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = date('Y-m-d');
                }
            }
        }

        $transactionUtil = new \App\Utils\TransactionUtil();
        $date_wise_data = [];
        $grand_total_sales = 0;
        $grand_total_cost = 0;
        $grand_total_expenses = 0;
        $grand_total_income = 0;
        $grand_total_profit = 0;

        // Loop through each date in the range
        $current_date = \Carbon\Carbon::parse($start_date);
        $end_date_obj = \Carbon\Carbon::parse($end_date);

        while ($current_date->lte($end_date_obj)) {
            $date_str = $current_date->format('Y-m-d');
            
            // Get profit/loss data for this date
            $pl_data = $transactionUtil->getProfitLossDetails(
                $business_id,
                $location_id,
                $date_str,
                $date_str,
                null,
                $permitted_locations
            );

            // Calculate sales amount
            $sales_amount = (float) ($pl_data['total_sell'] ?? 0);
            
            // Calculate cost amount (COGS)
            $opening_stock = (float) ($pl_data['opening_stock'] ?? 0);
            $total_purchase = (float) ($pl_data['total_purchase'] ?? 0);
            $closing_stock = (float) ($pl_data['closing_stock'] ?? 0);
            $cost_amount = $opening_stock + $total_purchase - $closing_stock;
            
            // Get expenses for this date
            $expenses_query = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'expense')
                ->where('t.status', 'final')
                ->whereDate('t.transaction_date', $date_str);

            if ($permitted_locations != 'all') {
                $expenses_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $expenses_query->where('t.location_id', $location_id);
            }

            $expenses = (float) ($expenses_query->sum('t.final_total') ?? 0);
            
            // Income (other income - can be from account transactions or other sources)
            // For now, set to 0 as it's not clearly defined in the image
            $income = 0;
            
            // Calculate profit
            $profit = $sales_amount - $cost_amount - $expenses + $income;

            $date_wise_data[] = [
                'date' => $date_str,
                'sales_amount' => $sales_amount,
                'cost_amount' => $cost_amount,
                'expenses' => $expenses,
                'income' => $income,
                'profit' => $profit
            ];

            $grand_total_sales += $sales_amount;
            $grand_total_cost += $cost_amount;
            $grand_total_expenses += $expenses;
            $grand_total_income += $income;
            $grand_total_profit += $profit;

            $current_date->addDay();
        }

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.date_wise_sales_with_profit', compact(
            'date_wise_data',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'grand_total_sales',
            'grand_total_cost',
            'grand_total_expenses',
            'grand_total_income',
            'grand_total_profit',
            'util',
            'business'
        ));
    }

    /**
     * Date Wise Net Profit Calculation Report
     */
    public function dateWiseNetProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register.date_wise_net_profit') && 
            !auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $util = new \App\Utils\Util();
        
        $start_date = $request->get('start_date', date('Y-m-d'));
        $end_date = $request->get('end_date', date('Y-m-d'));
        $location_id = $request->get('location_id');

        $permitted_locations = auth()->user()->permitted_locations();
        $locations = BusinessLocation::forDropdown($business_id, false, false);

        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = date('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = date('Y-m-d');
                }
            }
        }

        $transactionUtil = new \App\Utils\TransactionUtil();
        $date_wise_data = [];
        
        // Grand totals
        $grand_total_sales = 0;
        $grand_total_discount = 0;
        $grand_total_cogs = 0;
        $grand_total_gross_profit = 0;
        $grand_total_expenses = 0;
        $grand_total_net_profit = 0;

        // Loop through each date in the range
        $current_date = \Carbon\Carbon::parse($start_date);
        $end_date_obj = \Carbon\Carbon::parse($end_date);

        while ($current_date->lte($end_date_obj)) {
            $date_str = $current_date->format('Y-m-d');
            
            // Get profit/loss data for this date
            $pl_data = $transactionUtil->getProfitLossDetails(
                $business_id,
                $location_id,
                $date_str,
                $date_str,
                null,
                $permitted_locations
            );

            // Calculate sales amount
            $total_sales = (float) ($pl_data['total_sell'] ?? 0);
            $sales_discount = (float) ($pl_data['total_sell_discount'] ?? 0);
            
            // Calculate cost amount (COGS)
            $opening_stock = (float) ($pl_data['opening_stock'] ?? 0);
            $total_purchase = (float) ($pl_data['total_purchase'] ?? 0);
            $closing_stock = (float) ($pl_data['closing_stock'] ?? 0);
            $cogs = $opening_stock + $total_purchase - $closing_stock;
            
            // Calculate gross profit
            $gross_profit = $total_sales - abs($sales_discount) - $cogs;
            
            // Get expenses for this date
            $expenses_query = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'expense')
                ->where('t.status', 'final')
                ->whereDate('t.transaction_date', $date_str);

            if ($permitted_locations != 'all') {
                $expenses_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $expenses_query->where('t.location_id', $location_id);
            }

            $expenses = (float) ($expenses_query->sum('t.final_total') ?? 0);
            
            // Calculate net profit
            $net_profit = $gross_profit - $expenses;

            $date_wise_data[] = [
                'date' => $date_str,
                'total_sales' => $total_sales,
                'sales_discount' => abs($sales_discount),
                'cogs' => $cogs,
                'gross_profit' => $gross_profit,
                'expenses' => $expenses,
                'net_profit' => $net_profit
            ];

            $grand_total_sales += $total_sales;
            $grand_total_discount += abs($sales_discount);
            $grand_total_cogs += $cogs;
            $grand_total_gross_profit += $gross_profit;
            $grand_total_expenses += $expenses;
            $grand_total_net_profit += $net_profit;

            $current_date->addDay();
        }

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.date_wise_net_profit', compact(
            'date_wise_data',
            'start_date',
            'end_date',
            'location_id',
            'locations',
            'grand_total_sales',
            'grand_total_discount',
            'grand_total_cogs',
            'grand_total_gross_profit',
            'grand_total_expenses',
            'grand_total_net_profit',
            'util',
            'business'
        ));
    }

    /**
     * Print Date Wise Net Profit Report
     */
    public function printDateWiseNetProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register.date_wise_net_profit') && 
            !auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $util = new \App\Utils\Util();
        
        $start_date = $request->get('start_date', date('Y-m-d'));
        $end_date = $request->get('end_date', date('Y-m-d'));
        $location_id = $request->get('location_id');

        $permitted_locations = auth()->user()->permitted_locations();

        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = date('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = date('Y-m-d');
                }
            }
        }

        $transactionUtil = new \App\Utils\TransactionUtil();
        $date_wise_data = [];
        
        // Grand totals
        $grand_total_sales = 0;
        $grand_total_discount = 0;
        $grand_total_cogs = 0;
        $grand_total_gross_profit = 0;
        $grand_total_expenses = 0;
        $grand_total_net_profit = 0;

        // Loop through each date in the range
        $current_date = \Carbon\Carbon::parse($start_date);
        $end_date_obj = \Carbon\Carbon::parse($end_date);

        while ($current_date->lte($end_date_obj)) {
            $date_str = $current_date->format('Y-m-d');
            
            // Get profit/loss data for this date
            $pl_data = $transactionUtil->getProfitLossDetails(
                $business_id,
                $location_id,
                $date_str,
                $date_str,
                null,
                $permitted_locations
            );

            // Calculate sales amount
            $total_sales = (float) ($pl_data['total_sell'] ?? 0);
            $sales_discount = (float) ($pl_data['total_sell_discount'] ?? 0);
            
            // Calculate cost amount (COGS)
            $opening_stock = (float) ($pl_data['opening_stock'] ?? 0);
            $total_purchase = (float) ($pl_data['total_purchase'] ?? 0);
            $closing_stock = (float) ($pl_data['closing_stock'] ?? 0);
            $cogs = $opening_stock + $total_purchase - $closing_stock;
            
            // Calculate gross profit
            $gross_profit = $total_sales - abs($sales_discount) - $cogs;
            
            // Get expenses for this date
            $expenses_query = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'expense')
                ->where('t.status', 'final')
                ->whereDate('t.transaction_date', $date_str);

            if ($permitted_locations != 'all') {
                $expenses_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $expenses_query->where('t.location_id', $location_id);
            }

            $expenses = (float) ($expenses_query->sum('t.final_total') ?? 0);
            
            // Calculate net profit
            $net_profit = $gross_profit - $expenses;

            $date_wise_data[] = [
                'date' => $date_str,
                'total_sales' => $total_sales,
                'sales_discount' => abs($sales_discount),
                'cogs' => $cogs,
                'gross_profit' => $gross_profit,
                'expenses' => $expenses,
                'net_profit' => $net_profit
            ];

            $grand_total_sales += $total_sales;
            $grand_total_discount += abs($sales_discount);
            $grand_total_cogs += $cogs;
            $grand_total_gross_profit += $gross_profit;
            $grand_total_expenses += $expenses;
            $grand_total_net_profit += $net_profit;

            $current_date->addDay();
        }

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_date_wise_net_profit', compact(
            'date_wise_data',
            'start_date',
            'end_date',
            'location_id',
            'grand_total_sales',
            'grand_total_discount',
            'grand_total_cogs',
            'grand_total_gross_profit',
            'grand_total_expenses',
            'grand_total_net_profit',
            'util',
            'business'
        ));
    }

    /**
     * Print Date Wise Sales with Profit Report
     */
    public function printDateWiseSalesWithProfit(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $util = new \App\Utils\Util();
        
        $start_date = $request->get('start_date', date('Y-m-d'));
        $end_date = $request->get('end_date', date('Y-m-d'));
        $location_id = $request->get('location_id');

        $permitted_locations = auth()->user()->permitted_locations();

        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = date('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = date('Y-m-d');
                }
            }
        }

        $transactionUtil = new \App\Utils\TransactionUtil();
        $date_wise_data = [];
        $grand_total_sales = 0;
        $grand_total_cost = 0;
        $grand_total_expenses = 0;
        $grand_total_income = 0;
        $grand_total_profit = 0;

        // Loop through each date in the range
        $current_date = \Carbon\Carbon::parse($start_date);
        $end_date_obj = \Carbon\Carbon::parse($end_date);

        while ($current_date->lte($end_date_obj)) {
            $date_str = $current_date->format('Y-m-d');
            
            // Get profit/loss data for this date
            $pl_data = $transactionUtil->getProfitLossDetails(
                $business_id,
                $location_id,
                $date_str,
                $date_str,
                null,
                $permitted_locations
            );

            // Calculate sales amount
            $sales_amount = (float) ($pl_data['total_sell'] ?? 0);
            
            // Calculate cost amount (COGS)
            $opening_stock = (float) ($pl_data['opening_stock'] ?? 0);
            $total_purchase = (float) ($pl_data['total_purchase'] ?? 0);
            $closing_stock = (float) ($pl_data['closing_stock'] ?? 0);
            $cost_amount = $opening_stock + $total_purchase - $closing_stock;
            
            // Get expenses for this date
            $expenses_query = DB::table('transactions as t')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'expense')
                ->where('t.status', 'final')
                ->whereDate('t.transaction_date', $date_str);

            if ($permitted_locations != 'all') {
                $expenses_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $expenses_query->where('t.location_id', $location_id);
            }

            $expenses = (float) ($expenses_query->sum('t.final_total') ?? 0);
            
            // Income (other income - can be from account transactions or other sources)
            $income = 0;
            
            // Calculate profit
            $profit = $sales_amount - $cost_amount - $expenses + $income;

            $date_wise_data[] = [
                'date' => $date_str,
                'sales_amount' => $sales_amount,
                'cost_amount' => $cost_amount,
                'expenses' => $expenses,
                'income' => $income,
                'profit' => $profit
            ];

            $grand_total_sales += $sales_amount;
            $grand_total_cost += $cost_amount;
            $grand_total_expenses += $expenses;
            $grand_total_income += $income;
            $grand_total_profit += $profit;

            $current_date->addDay();
        }

        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.print_date_wise_sales_with_profit', compact(
            'date_wise_data',
            'start_date',
            'end_date',
            'location_id',
            'grand_total_sales',
            'grand_total_cost',
            'grand_total_expenses',
            'grand_total_income',
            'grand_total_profit',
            'util',
            'business'
        ));
    }

    /**
     * Profit and Loss Tally Style Report
     */
    public function profitLossTallyStyle()
    {
        if (!auth()->user()->can('businessmanagement.accounts_register.profit_loss_tally_style') && 
            !auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        return view('businessmanagement::accounts_register.profit_loss_tally_style', compact('locations'));
    }

    /**
     * Get Profit and Loss Data (AJAX)
     */
    public function getProfitLossData(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register.profit_loss_tally_style') && 
            !auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $transactionUtil = new TransactionUtil();
        $businessUtil = new BusinessUtil();
        
        $fy = $businessUtil->getCurrentFinancialYear($business_id);
        
        $location_id = !empty($request->input('location_id')) ? $request->input('location_id') : null;
        $start_date = !empty($request->input('start_date')) ? $request->input('start_date') : $fy['start'];
        $end_date = !empty($request->input('end_date')) ? $request->input('end_date') : $fy['end'];
        
        $user_id = $request->input('user_id') ?? null;
        
        $permitted_locations = auth()->user()->permitted_locations();
        
        try {
            // Get P&L data from TransactionUtil
            $data = $transactionUtil->getProfitLossDetails($business_id, $location_id, $start_date, $end_date, $user_id, $permitted_locations);
            
            // Ensure we have data (even if empty)
            if (!is_array($data)) {
                $data = [];
            }
            
            // Format as Tally-style P&L
            $pl_data = $this->formatTallyStylePL($data, $start_date, $end_date, $business_id, $location_id, $permitted_locations);
            
            return response()->json($pl_data);
        } catch (\Exception $e) {
            \Log::error('ProfitLoss getData Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'error' => 'Error generating Profit & Loss report: ' . $e->getMessage(),
                'left_side' => [],
                'right_side' => [],
                'subtotal_left' => 0,
                'subtotal_right' => 0,
                'gross_profit_c_o' => 0,
                'net_profit' => 0,
                'final_total_left' => 0,
                'final_total_right' => 0
            ], 500);
        }
    }

    /**
     * Format Profit Loss as Tally Style
     */
    protected function formatTallyStylePL($data, $start_date, $end_date, $business_id, $location_id, $permitted_locations)
    {
        // Ensure $data is an array and has the expected keys
        if (!is_array($data)) {
            $data = [];
        }
        
        $transactionUtil = new TransactionUtil();
        
        // LEFT SIDE (DEBIT/EXPENSES) - Tally Format
        $left_side = [];
        
        // 1. Opening Stock
        $opening_stock = (float) ($data['opening_stock'] ?? 0);
        $left_side[] = [
            'label' => 'Opening Stock',
            'value' => $opening_stock,
            'items' => [['label' => 'Opening Stock', 'value' => $opening_stock]]
        ];
        
        // 2. Purchase Accounts
        $purchase_items = [];
        $purchase_total = 0;
        
        $purchase_amount = (float) ($data['total_purchase'] ?? 0);
        if ($purchase_amount > 0.01 || $purchase_amount < -0.01) {
            $purchase_items[] = ['label' => 'Purchase - GST/IGST', 'value' => $purchase_amount];
            $purchase_total += $purchase_amount;
        }
        
        $purchase_shipping = (float) ($data['total_purchase_shipping_charge'] ?? 0);
        if (abs($purchase_shipping) > 0.01) {
            $purchase_items[] = ['label' => 'Delivery Charges on Purchase', 'value' => $purchase_shipping];
            $purchase_total += $purchase_shipping;
        }
        
        $purchase_additional = (float) ($data['total_purchase_additional_expense'] ?? 0);
        if (abs($purchase_additional) > 0.01) {
            $purchase_items[] = ['label' => 'Insurance Charges on Purchase', 'value' => $purchase_additional];
            $purchase_total += $purchase_additional;
        }
        
        $purchase_discount = (float) ($data['total_purchase_discount'] ?? 0);
        if (abs($purchase_discount) > 0.01) {
            $purchase_items[] = ['label' => 'Discount on Purchase', 'value' => -1 * abs($purchase_discount)];
            $purchase_total -= abs($purchase_discount);
        }
        
        $left_side[] = [
            'label' => 'Purchase Accounts',
            'value' => $purchase_total,
            'items' => !empty($purchase_items) ? $purchase_items : [['label' => 'No Purchases', 'value' => 0]]
        ];
        
        // 3. Direct Expenses
        $direct_expenses = [];
        $direct_total = 0;
        
        // Get direct expenses from account transactions
        try {
            $direct_expense_accounts = $this->getDirectExpenseAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            
            foreach ($direct_expense_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    $direct_expenses[] = ['label' => $account_name, 'value' => abs($amount)];
                    $direct_total += abs($amount);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting direct expenses: ' . $e->getMessage());
        }
        
        if (!empty($data['total_purchase_shipping_charge'])) {
            $direct_expenses[] = ['label' => 'Freight & Cartage', 'value' => $data['total_purchase_shipping_charge']];
            $direct_total += $data['total_purchase_shipping_charge'];
        }
        
        if (!empty($data['total_purchase_additional_expense'])) {
            $direct_expenses[] = ['label' => 'Handling & Other Charges', 'value' => $data['total_purchase_additional_expense']];
            $direct_total += $data['total_purchase_additional_expense'];
        }
        
        if (!empty($direct_expenses)) {
            $left_side[] = [
                'label' => 'Direct Expenses',
                'value' => $direct_total,
                'items' => $direct_expenses
            ];
        }
        
        // Calculate Gross Profit
        $opening_stock_val = (float) ($data['opening_stock'] ?? 0);
        $purchases = (float) $purchase_total;
        $direct_exp = (float) $direct_total;
        $closing_stock = (float) ($data['closing_stock'] ?? 0);
        
        $sales_gross = (float) ($data['total_sell'] ?? 0);
        $sales_discount_amount = (float) ($data['total_sell_discount'] ?? 0);
        $sales_net = (float) ($sales_gross - $sales_discount_amount);
        
        // Direct Incomes
        $direct_income_total = 0;
        try {
            $direct_income_accounts = $this->getDirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            foreach ($direct_income_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    $direct_income_total += abs($amount);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error calculating direct incomes: ' . $e->getMessage());
        }
        
        $trading_credit_side = $sales_net + $direct_income_total + $closing_stock;
        $trading_debit_side = $opening_stock_val + $purchases + $direct_exp;
        $gross_profit_c_o = (float) ($trading_credit_side - $trading_debit_side);
        
        // 4. Gross Profit c/o
        if (abs($gross_profit_c_o) > 0.01) {
            if ($gross_profit_c_o > 0) {
                $left_side[] = [
                    'label' => 'Gross Profit c/o',
                    'value' => $gross_profit_c_o,
                    'items' => []
                ];
            } else {
                $left_side[] = [
                    'label' => 'Gross Loss c/o',
                    'value' => abs($gross_profit_c_o),
                    'items' => []
                ];
            }
        } else {
            $left_side[] = [
                'label' => 'Gross Profit c/o',
                'value' => 0.00,
                'items' => []
            ];
        }
        
        $subtotal_left = (float) ($trading_debit_side + ($gross_profit_c_o > 0 ? $gross_profit_c_o : 0));
        
        // 5. Indirect Expenses
        $indirect_expenses = [];
        $indirect_total = 0;
        
        try {
            $indirect_expense_accounts = $this->getIndirectExpenseAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            
            $expense_categories = [
                'Administrative Expenses' => [],
                'Salaries & Staff Expenses' => [],
                'Selling & Distribution Expenses' => [],
                'Other Expenses' => []
            ];
            
            foreach ($indirect_expense_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    $display_name = trim($account_name);
                    if (empty($display_name)) {
                        $display_name = 'Unnamed Expense';
                    }
                    
                    $category = $this->categorizeExpense($display_name);
                    if (!isset($expense_categories[$category])) {
                        $category = 'Other Expenses';
                    }
                    $expense_categories[$category][] = ['label' => $display_name, 'value' => abs($amount)];
                }
            }
            
            foreach ($expense_categories as $category => $items) {
                if (!empty($items)) {
                    $category_total = (float) array_sum(array_column($items, 'value'));
                    $indirect_expenses[] = [
                        'label' => $category, 
                        'value' => $category_total, 
                        'is_group' => true, 
                        'sub_items' => $items
                    ];
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting indirect expenses: ' . $e->getMessage());
        }
        
        if (!empty($data['total_expense'])) {
            $indirect_expenses[] = ['label' => 'General Expenses', 'value' => $data['total_expense']];
        }
        
        if (!empty($data['total_sell_return'])) {
            $indirect_expenses[] = ['label' => 'Sales Returns', 'value' => $data['total_sell_return']];
        }
        
        foreach ($indirect_expenses as $item) {
            if (!empty($item['is_group']) && !empty($item['sub_items'])) {
                $group_total = 0;
                foreach ($item['sub_items'] as $sub_item) {
                    $group_total += (float) ($sub_item['value'] ?? 0);
                }
                $indirect_total += $group_total;
            } else {
                $indirect_total += (float) ($item['value'] ?? 0);
            }
        }
        
        $left_side[] = [
            'label' => 'Indirect Expenses',
            'value' => $indirect_total,
            'items' => !empty($indirect_expenses) ? $indirect_expenses : []
        ];
        
        // RIGHT SIDE (CREDIT/INCOME) - Tally Format
        $right_side = [];
        
        // 1. Sales Accounts
        $sales_items = [];
        $sales_total_val = 0;
        
        $sales_gross_val = (float) ($data['total_sell'] ?? 0);
        $sales_discount_val = (float) ($data['total_sell_discount'] ?? 0);
        $sales_net_val = (float) ($sales_gross_val - $sales_discount_val);
        
        if ($sales_gross_val > 0.01 || $sales_gross_val < -0.01) {
            $sales_items[] = ['label' => 'Sale - GST/IGST', 'value' => $sales_gross_val];
            $sales_total_val = $sales_gross_val;
            
            if (abs($sales_discount_val) > 0.01) {
                $sales_items[] = ['label' => 'Less: Price De-escalation (IGST-18%)', 'value' => -1 * abs($sales_discount_val)];
                $sales_total_val -= abs($sales_discount_val);
            }
        }
        
        $right_side[] = [
            'label' => 'Sales Accounts',
            'value' => $sales_total_val,
            'items' => !empty($sales_items) ? $sales_items : [['label' => 'No Sales', 'value' => 0]]
        ];
        
        // 2. Direct Incomes
        $direct_incomes = [];
        $direct_income_total_val = 0;
        
        try {
            $direct_income_accounts = $this->getDirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            
            foreach ($direct_income_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    $direct_incomes[] = ['label' => $account_name, 'value' => abs($amount)];
                    $direct_income_total_val += abs($amount);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting direct incomes: ' . $e->getMessage());
        }
        
        if (!empty($data['total_sell_shipping_charge'])) {
            $direct_incomes[] = ['label' => 'Service & Installation Charges', 'value' => $data['total_sell_shipping_charge']];
            $direct_income_total_val += $data['total_sell_shipping_charge'];
        }
        
        if (!empty($direct_incomes)) {
            $right_side[] = [
                'label' => 'Direct Incomes',
                'value' => $direct_income_total_val,
                'items' => $direct_incomes
            ];
        } else {
            $right_side[] = [
                'label' => 'Direct Incomes',
                'value' => 0,
                'items' => [['label' => 'No Direct Incomes', 'value' => 0]]
            ];
        }
        
        // 3. Closing Stock
        $closing_stock_val = (float) ($data['closing_stock'] ?? 0);
        $right_side[] = [
            'label' => 'Closing Stock',
            'value' => $closing_stock_val,
            'items' => [['label' => 'Closing Stock', 'value' => $closing_stock_val]]
        ];
        
        $subtotal_right = (float) ($sales_total_val + $direct_income_total_val + $closing_stock_val);
        
        // 4. Indirect Incomes
        $indirect_incomes = [];
        $indirect_income_total = 0;
        
        try {
            $indirect_income_accounts = $this->getIndirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            
            foreach ($indirect_income_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    $indirect_incomes[] = ['label' => $account_name, 'value' => abs($amount)];
                    $indirect_income_total += abs($amount);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting indirect incomes: ' . $e->getMessage());
        }
        
        if (!empty($data['total_purchase_return'])) {
            $indirect_incomes[] = ['label' => 'Purchase Returns', 'value' => abs($data['total_purchase_return'])];
            $indirect_income_total += abs($data['total_purchase_return']);
        }
        
        if (!empty($indirect_incomes)) {
            $right_side[] = [
                'label' => 'Indirect Incomes',
                'value' => $indirect_income_total,
                'items' => $indirect_incomes
            ];
        }
        
        // 5. Gross Profit b/f
        if (abs($gross_profit_c_o) > 0.01) {
            if ($gross_profit_c_o > 0) {
                $right_side[] = [
                    'label' => 'Gross Profit b/f',
                    'value' => $gross_profit_c_o,
                    'items' => []
                ];
            } else {
                $right_side[] = [
                    'label' => 'Gross Loss b/f',
                    'value' => abs($gross_profit_c_o),
                    'items' => []
                ];
            }
        } else {
            $right_side[] = [
                'label' => 'Gross Profit b/f',
                'value' => 0.00,
                'items' => []
            ];
        }
        
        // Net Profit Calculation
        $pl_credit_side = (float) ($gross_profit_c_o + $indirect_income_total);
        $pl_debit_side = (float) $indirect_total;
        $net_profit = (float) ($pl_credit_side - $pl_debit_side);
        
        // Add Net Profit/Loss to both sides
        if ($net_profit > 0.01) {
            $left_side[] = [
                'label' => 'Nett Profit',
                'value' => $net_profit,
                'items' => []
            ];
            $right_side[] = [
                'label' => 'Nett Profit',
                'value' => $net_profit,
                'items' => []
            ];
        } else if ($net_profit < -0.01) {
            $net_loss = abs($net_profit);
            $left_side[] = [
                'label' => 'Nett Loss',
                'value' => $net_loss,
                'items' => []
            ];
            $right_side[] = [
                'label' => 'Nett Loss',
                'value' => $net_loss,
                'items' => []
            ];
        }
        
        // Final totals
        $final_total_left = 0;
        foreach ($left_side as $section) {
            $final_total_left += (float) ($section['value'] ?? 0);
        }
        
        $final_total_right = 0;
        foreach ($right_side as $section) {
            $final_total_right += (float) ($section['value'] ?? 0);
        }
        
        // Ensure both sides balance
        $balance_diff = abs($final_total_left - $final_total_right);
        if ($balance_diff > 0.01) {
            $final_total = max($final_total_left, $final_total_right);
            $final_total_left = $final_total;
            $final_total_right = $final_total;
        }
        
        return [
            'left_side' => $left_side,
            'right_side' => $right_side,
            'subtotal_left' => $subtotal_left,
            'subtotal_right' => $subtotal_right,
            'gross_profit_c_o' => $gross_profit_c_o,
            'indirect_expenses_total' => $indirect_total,
            'indirect_incomes_total' => $indirect_income_total,
            'net_profit' => $net_profit,
            'final_total_left' => $final_total_left,
            'final_total_right' => $final_total_right,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'raw_data' => $data
        ];
    }

    /**
     * Get direct expense accounts
     */
    protected function getDirectExpenseAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations)
    {
        try {
            $direct_keywords = ['carriage', 'packaging', 'handling', 'assemble', 'assembly', 'freight', 'transport', 'delivery', 'loading', 'unloading'];
            
            $query = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->where('A.business_id', $business_id)
                ->where('account_transactions.type', 'debit')
                ->whereBetween(DB::raw('date(account_transactions.operation_date)'), [$start_date, $end_date])
                ->whereNull('account_transactions.deleted_at')
                ->whereNull('A.deleted_at');
            
            $expenses = $query->select(
                'A.name',
                DB::raw("COALESCE(SUM(account_transactions.amount), 0) as total")
            )
            ->groupBy('A.id', 'A.name')
            ->get();
            
            $direct_expenses = [];
            foreach ($expenses as $expense) {
                $account_name = trim($expense->name ?? '');
                if (empty($account_name)) {
                    $account_name = 'Unnamed Account';
                }
                
                if ($this->isAssetAccount($account_name)) {
                    continue;
                }
                
                foreach ($direct_keywords as $keyword) {
                    if (stripos($account_name, $keyword) !== false) {
                        $amount = (float) ($expense->total ?? 0);
                        if (abs($amount) > 0.01) {
                            if (isset($direct_expenses[$account_name])) {
                                $direct_expenses[$account_name] += $amount;
                            } else {
                                $direct_expenses[$account_name] = $amount;
                            }
                        }
                        break;
                    }
                }
            }
            
            return $direct_expenses;
        } catch (\Exception $e) {
            \Log::error('Error getting direct expense accounts: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get indirect expense accounts
     */
    protected function getIndirectExpenseAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations)
    {
        try {
            $direct_keywords = ['carriage', 'packaging', 'handling', 'assemble', 'assembly', 'freight', 'transport', 'delivery'];
            
            $query = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->where('A.business_id', $business_id)
                ->where('account_transactions.type', 'debit')
                ->whereBetween(DB::raw('date(account_transactions.operation_date)'), [$start_date, $end_date])
                ->whereNull('account_transactions.deleted_at')
                ->whereNull('A.deleted_at');
            
            $account_expenses = $query->select(
                'A.name',
                DB::raw("COALESCE(SUM(account_transactions.amount), 0) as total")
            )
            ->groupBy('A.id', 'A.name')
            ->get();
            
            $indirect_expenses = [];
            foreach ($account_expenses as $expense) {
                $account_name = trim($expense->name ?? '');
                if (empty($account_name)) {
                    $account_name = 'Unnamed Account';
                }
                
                $is_direct = false;
                foreach ($direct_keywords as $keyword) {
                    if (stripos($account_name, $keyword) !== false) {
                        $is_direct = true;
                        break;
                    }
                }
                
                if (!$is_direct && !$this->isAssetAccount($account_name)) {
                    $amount = (float) ($expense->total ?? 0);
                    if (isset($indirect_expenses[$account_name])) {
                        $indirect_expenses[$account_name] += $amount;
                    } else {
                        $indirect_expenses[$account_name] = $amount;
                    }
                }
            }
            
            // Add expense transactions
            try {
                $expense_query = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.type', 'expense')
                    ->where('t.status', 'final')
                    ->whereNull('t.deleted_at')
                    ->whereBetween('t.transaction_date', [$start_date, $end_date]);
                
                if ($permitted_locations != 'all') {
                    $expense_query->whereIn('t.location_id', $permitted_locations);
                }
                
                if (!empty($location_id)) {
                    $expense_query->where('t.location_id', $location_id);
                }
                
                $expense_transactions = $expense_query->select(
                    DB::raw("COALESCE(NULLIF(transaction_for, ''), 'Other Expenses') as expense_name"),
                    DB::raw("COALESCE(SUM(total_before_tax), 0) as total")
                )
                ->groupBy('transaction_for')
                ->get();
                
                foreach ($expense_transactions as $exp) {
                    $name = trim($exp->expense_name ?? '');
                    if (empty($name) || $name == '') {
                        $name = 'General Expenses';
                    }
                    
                    $amount = (float) ($exp->total ?? 0);
                    if (abs($amount) > 0.01) {
                        if (isset($indirect_expenses[$name])) {
                            $indirect_expenses[$name] += $amount;
                        } else {
                            $indirect_expenses[$name] = $amount;
                        }
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Error getting expense transactions: ' . $e->getMessage());
            }
            
            return $indirect_expenses;
        } catch (\Exception $e) {
            \Log::error('Error getting indirect expense accounts: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get direct income accounts
     */
    protected function getDirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations)
    {
        try {
            $direct_keywords = ['service', 'installation', 'AMC', 'maintenance', 'repair', 'commission'];
            
            $query = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->where('A.business_id', $business_id)
                ->where('account_transactions.type', 'credit')
                ->whereBetween(DB::raw('date(account_transactions.operation_date)'), [$start_date, $end_date])
                ->whereNull('account_transactions.deleted_at')
                ->whereNull('A.deleted_at');
            
            $incomes = $query->select(
                'A.name',
                DB::raw("COALESCE(SUM(account_transactions.amount), 0) as total")
            )
            ->groupBy('A.id', 'A.name')
            ->get();
            
            $direct_incomes = [];
            foreach ($incomes as $income) {
                $account_name = trim($income->name ?? '');
                if (empty($account_name)) {
                    $account_name = 'Unnamed Account';
                }
                
                if ($this->isAssetAccount($account_name)) {
                    continue;
                }
                
                foreach ($direct_keywords as $keyword) {
                    if (stripos($account_name, $keyword) !== false) {
                        $amount = (float) ($income->total ?? 0);
                        if (abs($amount) > 0.01) {
                            if (isset($direct_incomes[$account_name])) {
                                $direct_incomes[$account_name] += $amount;
                            } else {
                                $direct_incomes[$account_name] = $amount;
                            }
                        }
                        break;
                    }
                }
            }
            
            return $direct_incomes;
        } catch (\Exception $e) {
            \Log::error('Error getting direct income accounts: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get indirect income accounts
     */
    protected function getIndirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations)
    {
        try {
            $direct_keywords = ['service', 'installation', 'AMC', 'maintenance'];
            
            $query = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->where('A.business_id', $business_id)
                ->where('account_transactions.type', 'credit')
                ->whereBetween(DB::raw('date(account_transactions.operation_date)'), [$start_date, $end_date])
                ->whereNull('account_transactions.deleted_at')
                ->whereNull('A.deleted_at');
            
            $incomes = $query->select(
                'A.name',
                DB::raw("COALESCE(SUM(account_transactions.amount), 0) as total")
            )
            ->groupBy('A.id', 'A.name')
            ->get();
            
            $indirect_incomes = [];
            foreach ($incomes as $income) {
                $account_name = trim($income->name ?? '');
                if (empty($account_name)) {
                    $account_name = 'Unnamed Account';
                }
                
                if ($this->isAssetAccount($account_name)) {
                    continue;
                }
                
                $is_direct = false;
                foreach ($direct_keywords as $keyword) {
                    if (stripos($account_name, $keyword) !== false) {
                        $is_direct = true;
                        break;
                    }
                }
                
                if (!$is_direct) {
                    $amount = (float) ($income->total ?? 0);
                    if (abs($amount) > 0.01) {
                        if (isset($indirect_incomes[$account_name])) {
                            $indirect_incomes[$account_name] += $amount;
                        } else {
                            $indirect_incomes[$account_name] = $amount;
                        }
                    }
                }
            }
            
            return $indirect_incomes;
        } catch (\Exception $e) {
            \Log::error('Error getting indirect income accounts: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Check if account is an asset account
     */
    protected function isAssetAccount($account_name)
    {
        if (empty($account_name)) {
            return false;
        }
        
        $account_lower = strtolower($account_name);
        
        $asset_keywords = [
            'cash', 'bank', 'ebl', 'dbbl', 'brac', 'ucb', 'sbl', 'hsbc', 'standard chartered',
            'petty cash', 'cash in hand', 'cash at bank', 'bank balance',
            'accounts receivable', 'debtors', 'sundry debtors',
            'inventory', 'stock', 'goods',
            'fixed assets', 'plant', 'equipment', 'machinery', 'building', 'vehicle',
            'prepaid', 'advance', 'deposit', 'investment'
        ];
        
        foreach ($asset_keywords as $keyword) {
            if (stripos($account_lower, $keyword) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Categorize expense account
     */
    protected function categorizeExpense($account_name)
    {
        $name_lower = strtolower($account_name);
        
        if (stripos($name_lower, 'admin') !== false || 
            stripos($name_lower, 'office') !== false ||
            stripos($name_lower, 'utility') !== false ||
            stripos($name_lower, 'rent') !== false ||
            stripos($name_lower, 'electricity') !== false) {
            return 'Administrative Expenses';
        }
        
        if (stripos($name_lower, 'salary') !== false || 
            stripos($name_lower, 'wage') !== false ||
            stripos($name_lower, 'staff') !== false ||
            stripos($name_lower, 'employee') !== false ||
            stripos($name_lower, 'payroll') !== false) {
            return 'Salaries & Staff Expenses';
        }
        
        if (stripos($name_lower, 'selling') !== false || 
            stripos($name_lower, 'sales') !== false ||
            stripos($name_lower, 'marketing') !== false ||
            stripos($name_lower, 'advertisement') !== false ||
            stripos($name_lower, 'promotion') !== false) {
            return 'Selling & Distribution Expenses';
        }
        
        return 'Other Expenses';
    }

    /**
     * Customer Account Statement Report
     */
    public function customerAccountStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->startOfMonth()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        // Handle date format conversion if needed
        if (!empty($request->start_date)) {
            try {
                $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon\Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = \Carbon\Carbon::now()->startOfMonth()->format('Y-m-d');
                }
            }
        }
        
        if (!empty($request->end_date)) {
            try {
                $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon\Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                }
            }
        }
        
        $customer_id = $request->get('customer_id');
        $location_id = $request->get('location_id');

        // Get locations for dropdown
        $locations = BusinessLocation::forDropdown($business_id, true);

        // Get customers for dropdown
        $customers = Contact::where('business_id', $business_id)
            ->where('type', 'customer')
            ->select('id', 'name', 'supplier_business_name', 'contact_id', 'mobile', 'address_line_1', 'address_line_2', 'city', 'state', 'country')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function($contact) {
                return [$contact->id => ($contact->supplier_business_name ?? $contact->name)];
            })
            ->prepend(__('lang_v1.select_customer'), '');

        $customer = null;
        $statement_data = [];
        $previous_balance = 0;
        $running_balance = 0;
        $total_debit = 0;
        $total_credit = 0;

        if (!empty($customer_id)) {
            $customer = Contact::where('business_id', $business_id)
                ->where('type', 'customer')
                ->where('id', $customer_id)
                ->first();

            if ($customer) {
                // Apply location filter
                $permitted_locations = auth()->user()->permitted_locations();
                if ($permitted_locations != 'all') {
                    $opening_transactions_query = DB::table('transactions as t')
                        ->whereIn('t.location_id', $permitted_locations);
                } else {
                    $opening_transactions_query = DB::table('transactions as t');
                }

                if (!empty($location_id)) {
                    // Validate that user has access to the requested location
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        abort(403, 'Unauthorized location access.');
                    }
                }

                // Calculate previous balance (balance before start_date)
                $opening_transactions = $opening_transactions_query
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer_id)
                    ->whereIn('t.type', ['sell', 'opening_balance'])
                    ->whereDate('t.transaction_date', '<', $start_date)
                    ->where(function($q) {
                        $q->where(function($q2) {
                            $q2->where('t.type', 'sell')->where('t.status', 'final');
                        })->orWhere('t.type', 'opening_balance');
                    });

                if (!empty($location_id)) {
                    $opening_transactions->where('t.location_id', $location_id);
                }

                $opening_transactions = $opening_transactions
                    ->select('t.id', 't.type', 't.final_total')
                    ->get();

                $opening_invoice = 0;
                $opening_paid = 0;

                foreach ($opening_transactions as $transaction) {
                    if (in_array($transaction->type, ['sell', 'opening_balance'])) {
                        $opening_invoice += $transaction->final_total ?? 0;
                    }

                    $payments = DB::table('transaction_payments')
                        ->where('transaction_id', $transaction->id)
                        ->whereDate('paid_on', '<', $start_date)
                        ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                    $opening_paid += $payments ?? 0;
                }

                $previous_balance = $opening_invoice - $opening_paid;
                $running_balance = $previous_balance;

                // Add previous balance row
                if ($previous_balance != 0) {
                    $statement_data[] = (object)[
                        'date' => $start_date,
                        'transaction_type' => 'Previous Balance',
                        'reference' => '',
                        'description' => 'Previous Balance (Forwarded)',
                        'quantity' => 0,
                        'rate' => 0,
                        'debit' => 0,
                        'credit' => 0,
                        'balance' => $running_balance,
                    ];
                }

                // Get all sales transactions in date range
                $sales_query = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer_id)
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where(function($q) {
                        $q->where(function($q2) {
                            $q2->where('t.type', 'sell')->where('t.status', 'final');
                        })->orWhere('t.type', 'opening_balance');
                    });

                if (!empty($location_id)) {
                    $sales_query->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all') {
                    $sales_query->whereIn('t.location_id', $permitted_locations);
                }

                $sales_transactions = $sales_query
                    ->select(
                        't.id',
                        't.transaction_date',
                        't.type',
                        't.invoice_no',
                        't.ref_no',
                        't.final_total'
                    )
                    ->orderBy('t.transaction_date', 'asc')
                    ->orderBy('t.id', 'asc')
                    ->get();

                // Process sales transactions
                foreach ($sales_transactions as $transaction) {
                    if ($transaction->type == 'sell' && $transaction->final_total > 0) {
                        // Get product details from sell lines
                        $sell_lines = DB::table('transaction_sell_lines as tsl')
                            ->leftJoin('products as p', 'tsl.product_id', '=', 'p.id')
                            ->leftJoin('variations as v', 'tsl.variation_id', '=', 'v.id')
                            ->where('tsl.transaction_id', $transaction->id)
                            ->select(
                                'tsl.quantity',
                                'tsl.unit_price',
                                'p.name as product_name',
                                'v.name as variation_name'
                            )
                            ->get();

                        $description = '';
                        $total_qty = 0;
                        $avg_rate = 0;

                        if ($sell_lines->count() > 0) {
                            $product_names = [];
                            foreach ($sell_lines as $line) {
                                $product_name = $line->product_name ?? '';
                                if (!empty($line->variation_name) && $line->variation_name != 'DUMMY') {
                                    $product_name .= ' (' . $line->variation_name . ')';
                                }
                                $product_names[] = $product_name;
                                $total_qty += $line->quantity ?? 0;
                            }
                            $description = implode(', ', array_unique($product_names));
                            if (count($product_names) > 1) {
                                $description = 'Multiple Products';
                            }
                            $avg_rate = $total_qty > 0 ? ($transaction->final_total / $total_qty) : 0;
                        } else {
                            $description = 'Domestic Sales';
                            $total_qty = 0;
                            $avg_rate = $transaction->final_total;
                        }

                        $running_balance += $transaction->final_total;
                        $total_debit += $transaction->final_total;

                        $statement_data[] = (object)[
                            'date' => $transaction->transaction_date,
                            'transaction_type' => 'Domestic Sales',
                            'reference' => $transaction->invoice_no ?? $transaction->ref_no ?? '',
                            'description' => $description,
                            'quantity' => $total_qty,
                            'rate' => $avg_rate,
                            'debit' => $transaction->final_total,
                            'credit' => 0,
                            'balance' => $running_balance,
                        ];
                    }
                }

                // Get all payments in date range
                $payments_query = DB::table('transaction_payments as tp')
                    ->leftJoin('transactions as t', 'tp.transaction_id', '=', 't.id')
                    ->where('tp.business_id', $business_id)
                    ->where('tp.payment_for', $customer_id)
                    ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date])
                    ->whereNull('tp.parent_id');

                if (!empty($location_id)) {
                    $payments_query->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all') {
                    $payments_query->whereIn('t.location_id', $permitted_locations);
                }

                $payments = $payments_query
                    ->select(
                        'tp.id',
                        'tp.paid_on as transaction_date',
                        'tp.method',
                        'tp.amount',
                        'tp.is_return',
                        'tp.payment_ref_no',
                        't.invoice_no',
                        't.ref_no'
                    )
                    ->orderBy('tp.paid_on', 'asc')
                    ->orderBy('tp.id', 'asc')
                    ->get();

                // Process payments
                foreach ($payments as $payment) {
                    $payment_amount = $payment->is_return == 1 ? -$payment->amount : $payment->amount;
                    $running_balance -= $payment_amount;
                    $total_credit += abs($payment_amount);

                    $payment_method = ucfirst(str_replace('_', ' ', $payment->method ?? 'cash'));
                    if ($payment->is_return == 1) {
                        $payment_method = 'Payment Return';
                    }

                    $statement_data[] = (object)[
                        'date' => $payment->transaction_date,
                        'transaction_type' => 'Customer Payment',
                        'reference' => $payment->payment_ref_no ?? $payment->invoice_no ?? $payment->ref_no ?? '',
                        'description' => strtoupper($payment_method),
                        'quantity' => 0,
                        'rate' => 0,
                        'debit' => 0,
                        'credit' => abs($payment_amount),
                        'balance' => $running_balance,
                    ];
                }

                // Get credit memos (sell returns and other credit adjustments)
                $credit_memos_query = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer_id)
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where('t.type', 'sell_return');

                if (!empty($location_id)) {
                    $credit_memos_query->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all') {
                    $credit_memos_query->whereIn('t.location_id', $permitted_locations);
                }

                $credit_memos = $credit_memos_query
                    ->select(
                        't.id',
                        't.transaction_date',
                        't.invoice_no',
                        't.ref_no',
                        't.final_total',
                        't.additional_notes'
                    )
                    ->orderBy('t.transaction_date', 'asc')
                    ->orderBy('t.id', 'asc')
                    ->get();

                // Process credit memos
                foreach ($credit_memos as $memo) {
                    $credit_amount = abs($memo->final_total);
                    $running_balance -= $credit_amount;
                    $total_credit += $credit_amount;

                    $description = $memo->additional_notes ?? 'Customer Credit Memo';
                    if (empty($description)) {
                        $description = 'Customer Credit Memo';
                    }

                    $statement_data[] = (object)[
                        'date' => $memo->transaction_date,
                        'transaction_type' => 'Customer Credit Memo',
                        'reference' => $memo->invoice_no ?? $memo->ref_no ?? '',
                        'description' => $description,
                        'quantity' => 0,
                        'rate' => 0,
                        'debit' => 0,
                        'credit' => $credit_amount,
                        'balance' => $running_balance,
                    ];
                }

                // Sort all entries by date
                usort($statement_data, function($a, $b) {
                    $dateA = strtotime($a->date);
                    $dateB = strtotime($b->date);
                    if ($dateA == $dateB) {
                        // Previous Balance always first
                        if ($a->transaction_type == 'Previous Balance') return -1;
                        if ($b->transaction_type == 'Previous Balance') return 1;
                        return 0;
                    }
                    return ($dateA < $dateB) ? -1 : 1;
                });
            }
        }

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::accounts_register.customer_account_statement', compact(
            'statement_data',
            'start_date',
            'end_date',
            'customer_id',
            'location_id',
            'locations',
            'customers',
            'customer',
            'previous_balance',
            'running_balance',
            'total_debit',
            'total_credit',
            'util',
            'business'
        ));
    }

    /**
     * Print Customer Account Statement Report
     */
    public function printCustomerAccountStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.accounts_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        
        // Get date range filter
        $start_date = !empty($request->start_date) ? $request->start_date : \Carbon\Carbon::now()->startOfMonth()->format('Y-m-d');
        $end_date = !empty($request->end_date) ? $request->end_date : \Carbon\Carbon::now()->format('Y-m-d');
        
        $customer_id = $request->get('customer_id');
        $location_id = $request->get('location_id');

        $customer = null;
        $statement_data = [];
        $previous_balance = 0;
        $running_balance = 0;
        $total_debit = 0;
        $total_credit = 0;

        if (!empty($customer_id)) {
            $customer = Contact::where('business_id', $business_id)
                ->where('type', 'customer')
                ->where('id', $customer_id)
                ->first();

            if ($customer) {
                // Apply location filter
                $permitted_locations = auth()->user()->permitted_locations();
                if ($permitted_locations != 'all') {
                    $opening_transactions_query = DB::table('transactions as t')
                        ->whereIn('t.location_id', $permitted_locations);
                } else {
                    $opening_transactions_query = DB::table('transactions as t');
                }

                if (!empty($location_id)) {
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        abort(403, 'Unauthorized location access.');
                    }
                }

                // Calculate previous balance
                $opening_transactions = $opening_transactions_query
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer_id)
                    ->whereIn('t.type', ['sell', 'opening_balance'])
                    ->whereDate('t.transaction_date', '<', $start_date)
                    ->where(function($q) {
                        $q->where(function($q2) {
                            $q2->where('t.type', 'sell')->where('t.status', 'final');
                        })->orWhere('t.type', 'opening_balance');
                    });

                if (!empty($location_id)) {
                    $opening_transactions->where('t.location_id', $location_id);
                }

                $opening_transactions = $opening_transactions
                    ->select('t.id', 't.type', 't.final_total')
                    ->get();

                $opening_invoice = 0;
                $opening_paid = 0;

                foreach ($opening_transactions as $transaction) {
                    if (in_array($transaction->type, ['sell', 'opening_balance'])) {
                        $opening_invoice += $transaction->final_total ?? 0;
                    }

                    $payments = DB::table('transaction_payments')
                        ->where('transaction_id', $transaction->id)
                        ->whereDate('paid_on', '<', $start_date)
                        ->sum(DB::raw('IF(is_return = 1, -1 * amount, amount)'));

                    $opening_paid += $payments ?? 0;
                }

                $previous_balance = $opening_invoice - $opening_paid;
                $running_balance = $previous_balance;

                // Add previous balance row
                if ($previous_balance != 0) {
                    $statement_data[] = (object)[
                        'date' => $start_date,
                        'transaction_type' => 'Previous Balance',
                        'reference' => '',
                        'description' => 'Previous Balance (Forwarded)',
                        'quantity' => 0,
                        'rate' => 0,
                        'debit' => 0,
                        'credit' => 0,
                        'balance' => $running_balance,
                    ];
                }

                // Get all sales transactions
                $sales_query = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer_id)
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where(function($q) {
                        $q->where(function($q2) {
                            $q2->where('t.type', 'sell')->where('t.status', 'final');
                        })->orWhere('t.type', 'opening_balance');
                    });

                if (!empty($location_id)) {
                    $sales_query->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all') {
                    $sales_query->whereIn('t.location_id', $permitted_locations);
                }

                $sales_transactions = $sales_query
                    ->select(
                        't.id',
                        't.transaction_date',
                        't.type',
                        't.invoice_no',
                        't.ref_no',
                        't.final_total'
                    )
                    ->orderBy('t.transaction_date', 'asc')
                    ->orderBy('t.id', 'asc')
                    ->get();

                // Process sales transactions
                foreach ($sales_transactions as $transaction) {
                    if ($transaction->type == 'sell' && $transaction->final_total > 0) {
                        $sell_lines = DB::table('transaction_sell_lines as tsl')
                            ->leftJoin('products as p', 'tsl.product_id', '=', 'p.id')
                            ->leftJoin('variations as v', 'tsl.variation_id', '=', 'v.id')
                            ->where('tsl.transaction_id', $transaction->id)
                            ->select(
                                'tsl.quantity',
                                'tsl.unit_price',
                                'p.name as product_name',
                                'v.name as variation_name'
                            )
                            ->get();

                        $description = '';
                        $total_qty = 0;
                        $avg_rate = 0;

                        if ($sell_lines->count() > 0) {
                            $product_names = [];
                            foreach ($sell_lines as $line) {
                                $product_name = $line->product_name ?? '';
                                if (!empty($line->variation_name) && $line->variation_name != 'DUMMY') {
                                    $product_name .= ' (' . $line->variation_name . ')';
                                }
                                $product_names[] = $product_name;
                                $total_qty += $line->quantity ?? 0;
                            }
                            $description = implode(', ', array_unique($product_names));
                            if (count($product_names) > 1) {
                                $description = 'Multiple Products';
                            }
                            $avg_rate = $total_qty > 0 ? ($transaction->final_total / $total_qty) : 0;
                        } else {
                            $description = 'Domestic Sales';
                            $total_qty = 0;
                            $avg_rate = $transaction->final_total;
                        }

                        $running_balance += $transaction->final_total;
                        $total_debit += $transaction->final_total;

                        $statement_data[] = (object)[
                            'date' => $transaction->transaction_date,
                            'transaction_type' => 'Domestic Sales',
                            'reference' => $transaction->invoice_no ?? $transaction->ref_no ?? '',
                            'description' => $description,
                            'quantity' => $total_qty,
                            'rate' => $avg_rate,
                            'debit' => $transaction->final_total,
                            'credit' => 0,
                            'balance' => $running_balance,
                        ];
                    }
                }

                // Get all payments
                $payments_query = DB::table('transaction_payments as tp')
                    ->leftJoin('transactions as t', 'tp.transaction_id', '=', 't.id')
                    ->where('tp.business_id', $business_id)
                    ->where('tp.payment_for', $customer_id)
                    ->whereBetween(DB::raw('DATE(tp.paid_on)'), [$start_date, $end_date])
                    ->whereNull('tp.parent_id');

                if (!empty($location_id)) {
                    $payments_query->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all') {
                    $payments_query->whereIn('t.location_id', $permitted_locations);
                }

                $payments = $payments_query
                    ->select(
                        'tp.id',
                        'tp.paid_on as transaction_date',
                        'tp.method',
                        'tp.amount',
                        'tp.is_return',
                        'tp.payment_ref_no',
                        't.invoice_no',
                        't.ref_no'
                    )
                    ->orderBy('tp.paid_on', 'asc')
                    ->orderBy('tp.id', 'asc')
                    ->get();

                // Process payments
                foreach ($payments as $payment) {
                    $payment_amount = $payment->is_return == 1 ? -$payment->amount : $payment->amount;
                    $running_balance -= $payment_amount;
                    $total_credit += abs($payment_amount);

                    $payment_method = ucfirst(str_replace('_', ' ', $payment->method ?? 'cash'));
                    if ($payment->is_return == 1) {
                        $payment_method = 'Payment Return';
                    }

                    $statement_data[] = (object)[
                        'date' => $payment->transaction_date,
                        'transaction_type' => 'Customer Payment',
                        'reference' => $payment->payment_ref_no ?? $payment->invoice_no ?? $payment->ref_no ?? '',
                        'description' => strtoupper($payment_method),
                        'quantity' => 0,
                        'rate' => 0,
                        'debit' => 0,
                        'credit' => abs($payment_amount),
                        'balance' => $running_balance,
                    ];
                }

                // Get credit memos
                $credit_memos_query = DB::table('transactions as t')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $customer_id)
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where('t.type', 'sell_return');

                if (!empty($location_id)) {
                    $credit_memos_query->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all') {
                    $credit_memos_query->whereIn('t.location_id', $permitted_locations);
                }

                $credit_memos = $credit_memos_query
                    ->select(
                        't.id',
                        't.transaction_date',
                        't.invoice_no',
                        't.ref_no',
                        't.final_total',
                        't.additional_notes'
                    )
                    ->orderBy('t.transaction_date', 'asc')
                    ->orderBy('t.id', 'asc')
                    ->get();

                // Process credit memos
                foreach ($credit_memos as $memo) {
                    $credit_amount = abs($memo->final_total);
                    $running_balance -= $credit_amount;
                    $total_credit += $credit_amount;

                    $description = $memo->additional_notes ?? 'Customer Credit Memo';
                    if (empty($description)) {
                        $description = 'Customer Credit Memo';
                    }

                    $statement_data[] = (object)[
                        'date' => $memo->transaction_date,
                        'transaction_type' => 'Customer Credit Memo',
                        'reference' => $memo->invoice_no ?? $memo->ref_no ?? '',
                        'description' => $description,
                        'quantity' => 0,
                        'rate' => 0,
                        'debit' => 0,
                        'credit' => $credit_amount,
                        'balance' => $running_balance,
                    ];
                }

                // Sort all entries by date
                usort($statement_data, function($a, $b) {
                    $dateA = strtotime($a->date);
                    $dateB = strtotime($b->date);
                    if ($dateA == $dateB) {
                        if ($a->transaction_type == 'Previous Balance') return -1;
                        if ($b->transaction_type == 'Previous Balance') return 1;
                        return 0;
                    }
                    return ($dateA < $dateB) ? -1 : 1;
                });
            }
        }

        $util = new Util();
        $business = \App\Business::find($business_id);
        $logo = $business->logo ?? null;

        return view('businessmanagement::accounts_register.print_customer_account_statement', compact(
            'statement_data',
            'start_date',
            'end_date',
            'customer',
            'previous_balance',
            'running_balance',
            'total_debit',
            'total_credit',
            'util',
            'business',
            'logo'
        ));
    }
}

