<?php

namespace Modules\AccountingReports\Services;

use Modules\AccountingReports\Entities\ChartOfAccount;
use Modules\AccountingReports\Entities\JournalEntryHeader;
use Modules\AccountingReports\Entities\JournalEntryLine;
use Modules\AccountingReports\Entities\PeriodLock;
use Modules\AccountingReports\Entities\AuditLog;
use App\Transaction;
use App\TransactionPayment;
use App\TransactionSellLine;
use App\PurchaseLine;
use DB;

class PostingRulesService
{
    protected $businessId;
    protected $util;

    public function __construct()
    {
        $this->util = new \App\Utils\Util();
    }

    /**
     * Post a sale transaction to journal entries
     */
    public function postSale(Transaction $transaction)
    {
        if ($transaction->type !== 'sell') {
            throw new \Exception('Transaction is not a sale');
        }

        // Check period lock
        if (PeriodLock::isDateLocked($transaction->business_id, $transaction->transaction_date)) {
            throw new \Exception('Period is locked. Cannot post transactions for this date.');
        }

        DB::beginTransaction();
        try {
            $journal = $this->createJournalEntry([
                'business_id' => $transaction->business_id,
                'voucher_date' => $transaction->transaction_date,
                'source_module' => 'sale',
                'source_transaction_id' => $transaction->id,
                'location_id' => $transaction->location_id,
                'narration' => 'Sale Invoice: ' . $transaction->invoice_no,
                'reference' => $transaction->invoice_no,
                'created_by' => $transaction->created_by,
            ]);

            $totalRevenue = 0;
            $totalTax = 0;
            $totalDiscount = 0;
            $totalCogs = 0;

            // Post revenue and taxes from sale lines
            foreach ($transaction->sell_lines as $line) {
                $lineAmount = $line->quantity * $line->unit_price_inc_tax;
                $lineTax = $line->line_discount_type == 'fixed' 
                    ? ($lineAmount - $line->line_discount_amount) * ($line->tax->amount ?? 0) / 100
                    : ($lineAmount * (1 - $line->line_discount_amount / 100)) * ($line->tax->amount ?? 0) / 100;
                $lineRevenue = $lineAmount - $lineTax;

                $totalRevenue += $lineRevenue;
                $totalTax += $lineTax;

                // COGS will be posted separately via FIFO
            }

            // Get control accounts
            $arControl = $this->getControlAccount($transaction->business_id, 'receivables');
            $salesAccount = $this->getAccountByGroup($transaction->business_id, 'income');
            $taxAccount = $this->getControlAccount($transaction->business_id, 'tax');
            $discountAccount = $this->getControlAccount($transaction->business_id, 'discount');
            $cogsAccount = $this->getAccountByGroup($transaction->business_id, 'cogs');
            $inventoryAccount = $this->getControlAccount($transaction->business_id, 'inventory');

            // Debit: Accounts Receivable (or Cash/Bank if paid immediately)
            if ($transaction->payment_status == 'paid') {
                $paymentAccount = $this->getDefaultPaymentAccount($transaction->business_id, $transaction->location_id);
                $this->createJournalLine($journal->id, $paymentAccount->id, $transaction->final_total, 0, $transaction->contact_id);
            } else {
                $this->createJournalLine($journal->id, $arControl->id, $transaction->final_total, 0, $transaction->contact_id);
            }

            // Credit: Sales Revenue
            $this->createJournalLine($journal->id, $salesAccount->id, 0, $totalRevenue);

            // Credit: Output Tax
            if ($totalTax > 0) {
                $this->createJournalLine($journal->id, $taxAccount->id, 0, $totalTax);
            }

            // Debit: Discount (if any)
            if ($transaction->discount_amount > 0) {
                $this->createJournalLine($journal->id, $discountAccount->id, $transaction->discount_amount, 0);
            }

            // Post COGS via FIFO service
            $fifoService = new FifoCostingService();
            $cogsLines = $fifoService->calculateAndPostCogs($transaction);

            foreach ($cogsLines as $cogsLine) {
                // Debit: COGS
                $this->createJournalLine(
                    $journal->id, 
                    $cogsAccount->id, 
                    $cogsLine['amount'], 
                    0,
                    null,
                    $cogsLine['product_id'],
                    $cogsLine['variation_id'],
                    null,
                    'COGS for ' . $cogsLine['product_name'],
                    $cogsLine['fifo_layer_id']
                );

                // Credit: Inventory
                $this->createJournalLine(
                    $journal->id, 
                    $inventoryAccount->id, 
                    0, 
                    $cogsLine['amount'],
                    null,
                    $cogsLine['product_id'],
                    $cogsLine['variation_id'],
                    null,
                    'Inventory reduction for ' . $cogsLine['product_name'],
                    $cogsLine['fifo_layer_id']
                );
            }

            $journal->validateBalance();
            if (!$journal->is_balanced) {
                throw new \Exception('Journal entry does not balance');
            }

            $journal->post();
            $this->logAudit('create', 'journal_entry', $journal->id, null, $journal->toArray());

            DB::commit();
            return $journal;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Post a purchase transaction
     */
    public function postPurchase(Transaction $transaction)
    {
        if ($transaction->type !== 'purchase') {
            throw new \Exception('Transaction is not a purchase');
        }

        if (PeriodLock::isDateLocked($transaction->business_id, $transaction->transaction_date)) {
            throw new \Exception('Period is locked');
        }

        DB::beginTransaction();
        try {
            $journal = $this->createJournalEntry([
                'business_id' => $transaction->business_id,
                'voucher_date' => $transaction->transaction_date,
                'source_module' => 'purchase',
                'source_transaction_id' => $transaction->id,
                'location_id' => $transaction->location_id,
                'narration' => 'Purchase Bill: ' . $transaction->ref_no,
                'reference' => $transaction->ref_no,
                'created_by' => $transaction->created_by,
            ]);

            $inventoryAccount = $this->getControlAccount($transaction->business_id, 'inventory');
            $apControl = $this->getControlAccount($transaction->business_id, 'payables');
            $taxAccount = $this->getControlAccount($transaction->business_id, 'tax');

            $totalAmount = 0;
            $totalTax = 0;

            // Post inventory and input tax
            foreach ($transaction->purchase_lines as $line) {
                $lineAmount = $line->quantity * $line->purchase_price_inc_tax;
                $lineTax = $lineAmount * ($line->tax->amount ?? 0) / 100;
                $lineCost = $lineAmount - $lineTax;

                $totalAmount += $lineCost;
                $totalTax += $lineTax;

                // Debit: Inventory
                $this->createJournalLine(
                    $journal->id,
                    $inventoryAccount->id,
                    $lineCost,
                    0,
                    $transaction->contact_id,
                    $line->product_id,
                    $line->variation_id
                );

                // Create FIFO layer
                $fifoService = new FifoCostingService();
                $fifoService->createFifoLayer($transaction->business_id, $line, $transaction->location_id, $transaction->transaction_date);
            }

            // Credit: Input Tax
            if ($totalTax > 0) {
                $this->createJournalLine($journal->id, $taxAccount->id, 0, $totalTax);
            }

            // Credit: Accounts Payable
            if ($transaction->payment_status == 'due') {
                $this->createJournalLine($journal->id, $apControl->id, 0, $transaction->final_total, $transaction->contact_id);
            } else {
                $paymentAccount = $this->getDefaultPaymentAccount($transaction->business_id, $transaction->location_id);
                $this->createJournalLine($journal->id, $paymentAccount->id, 0, $transaction->final_total);
            }

            $journal->validateBalance();
            if (!$journal->is_balanced) {
                throw new \Exception('Journal entry does not balance');
            }

            $journal->post();
            $this->logAudit('create', 'journal_entry', $journal->id, null, $journal->toArray());

            DB::commit();
            return $journal;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Post customer receipt
     */
    public function postReceipt(TransactionPayment $payment)
    {
        if ($payment->transaction->type !== 'sell') {
            throw new \Exception('Payment is not for a sale');
        }

        DB::beginTransaction();
        try {
            $journal = $this->createJournalEntry([
                'business_id' => $payment->transaction->business_id,
                'voucher_date' => $payment->paid_on,
                'source_module' => 'receipt',
                'source_transaction_id' => $payment->transaction_id,
                'location_id' => $payment->transaction->location_id,
                'narration' => 'Customer Receipt: ' . $payment->transaction->invoice_no,
                'reference' => $payment->payment_ref_no,
                'created_by' => $payment->created_by,
            ]);

            $arControl = $this->getControlAccount($payment->transaction->business_id, 'receivables');
            $paymentAccount = $this->getAccountById($payment->account_id) 
                ?? $this->getDefaultPaymentAccount($payment->transaction->business_id, $payment->transaction->location_id);

            // Debit: Cash/Bank
            $this->createJournalLine($journal->id, $paymentAccount->id, $payment->amount, 0);

            // Credit: Accounts Receivable
            $this->createJournalLine($journal->id, $arControl->id, 0, $payment->amount, $payment->transaction->contact_id);

            $journal->validateBalance();
            $journal->post();

            DB::commit();
            return $journal;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Post expense
     */
    public function postExpense(Transaction $transaction)
    {
        if ($transaction->type !== 'expense') {
            throw new \Exception('Transaction is not an expense');
        }

        DB::beginTransaction();
        try {
            $journal = $this->createJournalEntry([
                'business_id' => $transaction->business_id,
                'voucher_date' => $transaction->transaction_date,
                'source_module' => 'expense',
                'source_transaction_id' => $transaction->id,
                'location_id' => $transaction->location_id,
                'narration' => 'Expense: ' . ($transaction->additional_notes ?? 'N/A'),
                'created_by' => $transaction->created_by,
            ]);

            $expenseAccount = $this->getAccountByGroup($transaction->business_id, 'expenses');
            $paymentAccount = $this->getDefaultPaymentAccount($transaction->business_id, $transaction->location_id);

            // Debit: Expense
            $this->createJournalLine($journal->id, $expenseAccount->id, $transaction->final_total, 0);

            // Credit: Cash/Bank
            $this->createJournalLine($journal->id, $paymentAccount->id, 0, $transaction->final_total);

            $journal->validateBalance();
            $journal->post();

            DB::commit();
            return $journal;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Helper methods
     */
    protected function createJournalEntry(array $data)
    {
        $voucherNo = $this->generateVoucherNumber($data['business_id'], $data['source_module']);
        
        return JournalEntryHeader::create([
            'business_id' => $data['business_id'],
            'voucher_no' => $voucherNo,
            'voucher_date' => $data['voucher_date'],
            'transaction_date' => now(),
            'source_module' => $data['source_module'],
            'source_transaction_id' => $data['source_transaction_id'] ?? null,
            'location_id' => $data['location_id'] ?? null,
            'narration' => $data['narration'] ?? null,
            'reference' => $data['reference'] ?? null,
            'created_by' => $data['created_by'],
            'status' => 'draft',
        ]);
    }

    protected function createJournalLine($journalId, $accountId, $debit, $credit, $contactId = null, $productId = null, $variationId = null, $taxId = null, $description = null, $fifoLayerId = null)
    {
        return JournalEntryLine::create([
            'journal_entry_id' => $journalId,
            'business_id' => JournalEntryHeader::find($journalId)->business_id,
            'account_id' => $accountId,
            'debit' => $debit,
            'credit' => $credit,
            'amount' => max($debit, $credit),
            'contact_id' => $contactId,
            'product_id' => $productId,
            'variation_id' => $variationId,
            'tax_id' => $taxId,
            'description' => $description,
            'fifo_layer_id' => $fifoLayerId,
            'line_number' => JournalEntryLine::where('journal_entry_id', $journalId)->count() + 1,
        ]);
    }

    protected function getControlAccount($businessId, $controlType)
    {
        return ChartOfAccount::where('business_id', $businessId)
            ->where('is_control_account', true)
            ->where('control_type', $controlType)
            ->firstOrFail();
    }

    protected function getAccountByGroup($businessId, $group)
    {
        return ChartOfAccount::where('business_id', $businessId)
            ->where('account_group', $group)
            ->where('is_active', true)
            ->first();
    }

    protected function getAccountById($accountId)
    {
        if (!$accountId) return null;
        
        // Try to find in Chart of Accounts first
        $coaAccount = ChartOfAccount::where('account_id', $accountId)->first();
        if ($coaAccount) {
            return $coaAccount;
        }
        
        // Fallback to mapping
        $account = \App\Account::find($accountId);
        if ($account) {
            // Try to find or create mapping
            return ChartOfAccount::firstOrCreate(
                ['business_id' => $account->business_id, 'account_id' => $accountId],
                [
                    'name' => $account->name,
                    'account_group' => 'assets_current',
                    'account_type' => 'asset',
                    'created_by' => auth()->id(),
                ]
            );
        }
        
        return null;
    }

    protected function getDefaultPaymentAccount($businessId, $locationId)
    {
        // Get default cash account from location settings
        $location = \App\BusinessLocation::find($locationId);
        if ($location && $location->default_payment_accounts) {
            $accounts = json_decode($location->default_payment_accounts, true);
            if (isset($accounts['cash']['account'])) {
                return $this->getAccountById($accounts['cash']['account']);
            }
        }
        
        // Fallback to default cash account
        return $this->getControlAccount($businessId, 'cash');
    }

    protected function generateVoucherNumber($businessId, $sourceModule)
    {
        $prefix = strtoupper(substr($sourceModule, 0, 3));
        $year = date('Y');
        
        $lastVoucher = JournalEntryHeader::where('business_id', $businessId)
            ->where('voucher_no', 'like', $prefix . '-' . $year . '%')
            ->orderBy('id', 'desc')
            ->first();
        
        if ($lastVoucher) {
            $lastNum = (int) substr($lastVoucher->voucher_no, -4);
            $nextNum = $lastNum + 1;
        } else {
            $nextNum = 1;
        }
        
        return $prefix . '-' . $year . '-' . str_pad($nextNum, 4, '0', STR_PAD_LEFT);
    }

    protected function logAudit($actionType, $entityType, $entityId, $oldValues, $newValues)
    {
        AuditLog::create([
            'business_id' => auth()->user()->business_id,
            'action_type' => $actionType,
            'entity_type' => $entityType,
            'entity_id' => $entityId,
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'user_id' => auth()->id(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);
    }
}


