<?php

namespace Modules\AccountingReports\Services;

use Modules\AccountingReports\Entities\FifoLayer;
use Modules\AccountingReports\Entities\JournalEntryLine;
use App\Transaction;
use App\TransactionSellLine;
use App\PurchaseLine;
use DB;

class FifoCostingService
{
    /**
     * Create FIFO layer from purchase
     */
    public function createFifoLayer($businessId, PurchaseLine $purchaseLine, $locationId, $date)
    {
        return FifoLayer::create([
            'business_id' => $businessId,
            'product_id' => $purchaseLine->product_id,
            'variation_id' => $purchaseLine->variation_id,
            'location_id' => $locationId,
            'transaction_id' => $purchaseLine->transaction_id,
            'transaction_line_id' => $purchaseLine->id,
            'layer_date' => $date,
            'quantity' => $purchaseLine->quantity,
            'unit_cost' => $purchaseLine->purchase_price,
            'total_cost' => $purchaseLine->quantity * $purchaseLine->purchase_price,
            'allocated_quantity' => 0,
            'status' => 'available',
        ]);
    }

    /**
     * Calculate COGS for sale using FIFO and return lines to be posted
     */
    public function calculateAndPostCogs(Transaction $saleTransaction)
    {
        $cogsLines = [];
        
        foreach ($saleTransaction->sell_lines as $sellLine) {
            $remainingQty = $sellLine->quantity;
            $productId = $sellLine->product_id;
            $variationId = $sellLine->variation_id;
            $locationId = $saleTransaction->location_id;
            
            // Get available FIFO layers (oldest first)
            $layers = FifoLayer::available()
                ->byProduct($productId, $variationId, $locationId)
                ->where('business_id', $saleTransaction->business_id)
                ->orderBy('layer_date', 'asc')
                ->orderBy('id', 'asc')
                ->get();
            
            foreach ($layers as $layer) {
                if ($remainingQty <= 0) break;
                
                $availableQty = $layer->available_quantity;
                if ($availableQty <= 0) continue;
                
                $allocateQty = min($remainingQty, $availableQty);
                $cogsAmount = $allocateQty * $layer->unit_cost;
                
                // Update layer
                $layer->allocated_quantity += $allocateQty;
                if ($layer->allocated_quantity >= $layer->quantity - 0.0001) {
                    $layer->status = 'fully_allocated';
                }
                $layer->save();
                
                $cogsLines[] = [
                    'product_id' => $productId,
                    'variation_id' => $variationId,
                    'product_name' => $layer->product->name ?? 'N/A',
                    'quantity' => $allocateQty,
                    'unit_cost' => $layer->unit_cost,
                    'amount' => $cogsAmount,
                    'fifo_layer_id' => $layer->id,
                ];
                
                $remainingQty -= $allocateQty;
            }
            
            // Handle negative stock scenario
            if ($remainingQty > 0.0001) {
                // Use average cost or last cost as fallback
                $avgCost = $this->getAverageCost($productId, $variationId, $locationId, $saleTransaction->business_id);
                $cogsAmount = $remainingQty * $avgCost;
                
                $cogsLines[] = [
                    'product_id' => $productId,
                    'variation_id' => $variationId,
                    'product_name' => $sellLine->product->name ?? 'N/A',
                    'quantity' => $remainingQty,
                    'unit_cost' => $avgCost,
                    'amount' => $cogsAmount,
                    'fifo_layer_id' => null,
                ];
            }
        }
        
        return $cogsLines;
    }

    /**
     * Get average cost for a product
     */
    protected function getAverageCost($productId, $variationId, $locationId, $businessId)
    {
        $layers = FifoLayer::where('business_id', $businessId)
            ->where('product_id', $productId)
            ->where('variation_id', $variationId)
            ->where('location_id', $locationId)
            ->where('status', '!=', 'closed')
            ->get();
        
        if ($layers->isEmpty()) {
            // Fallback to product's last cost or purchase price
            $product = \App\Product::find($productId);
            return $product->last_purchased_price ?? 0;
        }
        
        $totalCost = $layers->sum('total_cost');
        $totalQty = $layers->sum('quantity');
        
        return $totalQty > 0 ? $totalCost / $totalQty : 0;
    }

    /**
     * Get inventory valuation at a specific date
     */
    public function getInventoryValuation($businessId, $asOfDate = null, $locationId = null)
    {
        $query = FifoLayer::where('business_id', $businessId)
            ->where('status', '!=', 'closed');
        
        if ($asOfDate) {
            $query->where('layer_date', '<=', $asOfDate);
        }
        
        if ($locationId) {
            $query->where('location_id', $locationId);
        }
        
        $layers = $query->get();
        
        $valuation = [];
        foreach ($layers as $layer) {
            $availableQty = $layer->available_quantity;
            $value = $availableQty * $layer->unit_cost;
            
            $key = $layer->product_id . '_' . $layer->variation_id . '_' . $layer->location_id;
            if (!isset($valuation[$key])) {
                $valuation[$key] = [
                    'product_id' => $layer->product_id,
                    'variation_id' => $layer->variation_id,
                    'location_id' => $layer->location_id,
                    'quantity' => 0,
                    'value' => 0,
                ];
            }
            
            $valuation[$key]['quantity'] += $availableQty;
            $valuation[$key]['value'] += $value;
        }
        
        return array_values($valuation);
    }

    /**
     * Rebuild FIFO layers from historical data
     */
    public function rebuildFifoLayers($businessId, $fromDate = null, $toDate = null)
    {
        DB::beginTransaction();
        try {
            // Clear existing layers for the period
            if ($fromDate && $toDate) {
                FifoLayer::where('business_id', $businessId)
                    ->whereBetween('layer_date', [$fromDate, $toDate])
                    ->delete();
            }
            
            // Rebuild from purchases
            $purchases = \App\Transaction::where('business_id', $businessId)
                ->where('type', 'purchase')
                ->where('status', 'final');
            
            if ($fromDate) {
                $purchases->where('transaction_date', '>=', $fromDate);
            }
            if ($toDate) {
                $purchases->where('transaction_date', '<=', $toDate);
            }
            
            foreach ($purchases->get() as $purchase) {
                foreach ($purchase->purchase_lines as $line) {
                    $this->createFifoLayer(
                        $businessId,
                        $line,
                        $purchase->location_id,
                        $purchase->transaction_date
                    );
                }
            }
            
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
}


