<?php

namespace Modules\AccountingReports\Services;

use App\Transaction;
use App\TransactionPayment;
use App\AccountTransaction;
use App\Account;
use App\Contact;
use App\BusinessLocation;
use App\User;
use App\Utils\TransactionUtil;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

/**
 * DaybookQuery Service
 * 
 * Aggregates all financial transactions from various sources into a unified
 * chronological Day Book listing with running balance calculation.
 */
class DaybookQuery
{
    protected $transactionUtil;
    
    public function __construct(TransactionUtil $transactionUtil)
    {
        $this->transactionUtil = $transactionUtil;
    }

    /**
     * Get Day Book entries for a given date range with filters
     * 
     * @param int $business_id
     * @param string $start_date
     * @param string $end_date
     * @param array $filters
     * @return array
     */
    public function getDaybookEntries($business_id, $start_date, $end_date, array $filters = [])
    {
        $location_id = $filters['location_id'] ?? null;
        $user_id = $filters['user_id'] ?? null;
        $account_id = $filters['account_id'] ?? null;
        $module_filter = $filters['module_filter'] ?? null;
        $scope = $filters['scope'] ?? 'cash_bank'; // cash, bank, cash_bank, all
        $party_id = $filters['party_id'] ?? null;
        $payment_method = $filters['payment_method'] ?? null;
        $currency = $filters['currency'] ?? null;
        $use_created_date = $filters['use_created_date'] ?? false;
        $permitted_locations = $filters['permitted_locations'] ?? 'all';

        // Get account IDs based on scope
        $account_ids = $this->getAccountIdsByScope($business_id, $scope, $account_id);
        
        // Calculate opening balance
        $opening_balance = $this->calculateOpeningBalance(
            $business_id, 
            $start_date, 
            $location_id, 
            $permitted_locations,
            $account_ids
        );

        // Build union query for all transaction sources
        $entries = $this->buildDaybookQuery(
            $business_id,
            $start_date,
            $end_date,
            $location_id,
            $user_id,
            $account_ids,
            $party_id,
            $payment_method,
            $currency,
            $use_created_date,
            $module_filter,
            $permitted_locations
        );

        // Process and format entries
        $formatted_entries = [];
        $running_balance = $opening_balance;
        
        foreach ($entries as $entry) {
            // Convert to array if object
            $entry_array = is_object($entry) ? (array) $entry : $entry;
            if (!is_array($entry_array)) {
                continue;
            }
            
            $debit = (float) ($entry_array['debit'] ?? 0);
            $credit = (float) ($entry_array['credit'] ?? 0);
            
            $running_balance += ($debit - $credit);
            
            $formatted_entries[] = [
                'tx_datetime' => $entry_array['datetime'] ?? null,
                'voucher_no' => $entry_array['voucher_no'] ?? '-',
                'voucher_type' => $entry_array['module'] ?? 'Unknown',
                'module' => $entry_array['module'] ?? 'unknown',
                'party_name' => $entry_array['party'] ?? '-',
                'party_id' => $entry_array['party_id'] ?? null,
                'party_type' => null,
                'location_name' => $entry_array['location'] ?? '-',
                'location_id' => $entry_array['location_id'] ?? null,
                'payment_account_name' => $entry_array['account'] ?? '-',
                'payment_account_id' => $entry_array['account_id'] ?? null,
                'debit' => $debit,
                'credit' => $credit,
                'running_balance' => $running_balance,
                'narration' => $entry_array['narration'] ?? '-',
                'user_name' => $entry_array['user'] ?? '-',
                'user_id' => $entry_array['user_id'] ?? null,
                'currency_code' => null,
                'fx_rate' => 1.0,
                'source_table' => $entry_array['source_table'] ?? 'transactions',
                'source_pk' => $entry_array['source_pk'] ?? null,
                'transaction_id' => $entry_array['transaction_id'] ?? null,
                'transaction_payment_id' => $entry_array['transaction_payment_id'] ?? null,
                'account_transaction_id' => $entry_array['account_transaction_id'] ?? null,
                'created_at' => null,
                'updated_at' => null,
                'created_by_name' => null,
                'updated_by_name' => null,
            ];
        }

        // Calculate summary statistics
        $total_debit = array_sum(array_column($formatted_entries, 'debit'));
        $total_credit = array_sum(array_column($formatted_entries, 'credit'));
        $closing_balance = $opening_balance + ($total_debit - $total_credit);
        
        // Count by module
        $count_by_module = [];
        foreach ($formatted_entries as $entry) {
            $module = $entry['module'];
            $count_by_module[$module] = ($count_by_module[$module] ?? 0) + 1;
        }

        return [
            'entries' => $formatted_entries,
            'summary' => [
                'opening_balance' => $opening_balance,
                'total_debit' => $total_debit,
                'total_credit' => $total_credit,
                'closing_balance' => $closing_balance,
                'count_by_module' => $count_by_module,
                'total_entries' => count($formatted_entries),
            ],
            'filters_applied' => $filters,
            'is_scoped' => $scope !== 'all',
        ];
    }

    /**
     * Get account IDs based on scope
     */
    protected function getAccountIdsByScope($business_id, $scope, $specific_account_id = null)
    {
        if ($specific_account_id) {
            return [$specific_account_id];
        }

        if ($scope == 'all') {
            return null; // Don't filter by accounts
        }

        $account_ids = [];

        if (in_array($scope, ['cash', 'cash_bank'])) {
            $cash = Account::where('business_id', $business_id)
                ->where(function($q) {
                    $q->whereRaw('LOWER(name) LIKE ?', ['%cash%'])
                      ->orWhereRaw('LOWER(name) LIKE ?', ['%petty%']);
                })
                ->where('is_closed', 0)
                ->pluck('id')
                ->toArray();
            $account_ids = array_merge($account_ids, $cash);
        }

        if (in_array($scope, ['bank', 'cash_bank'])) {
            $bank = Account::where('business_id', $business_id)
                ->where(function($q) {
                    $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                      ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                      ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
                })
                ->where('is_closed', 0)
                ->pluck('id')
                ->toArray();
            $account_ids = array_merge($account_ids, $bank);
        }

        $result = !empty($account_ids) ? array_unique($account_ids) : null;
        
        return (is_array($result) && count($result) > 0) ? $result : null;
    }

    /**
     * Calculate opening balance from previous day's closing balance
     */
    protected function calculateOpeningBalance($business_id, $start_date, $location_id, $permitted_locations, $account_ids)
    {
        if (!$start_date) {
            return 0;
        }

        // Calculate opening balance from account_transactions
        // account_transactions doesn't have business_id, need to join with accounts table
        $opening_balance = 0;
        
        if ($account_ids && count($account_ids) > 0) {
            $query = AccountTransaction::join('accounts', 'account_transactions.account_id', '=', 'accounts.id')
                ->where('accounts.business_id', $business_id)
                ->whereDate('account_transactions.operation_date', '<', $start_date)
                ->whereIn('account_transactions.account_id', $account_ids)
                ->whereNull('account_transactions.deleted_at');

            // Filter by location if specified (via transaction relationship)
            if ($location_id) {
                $query->whereHas('transaction', function($q) use ($location_id) {
                    $q->where('location_id', $location_id);
                });
            }

            // Filter by permitted locations
            if ($permitted_locations != 'all') {
                $query->whereHas('transaction', function($q) use ($permitted_locations) {
                    $q->whereIn('location_id', $permitted_locations);
                });
            }

            $debit_total = (float) $query->clone()
                ->where('account_transactions.type', 'debit')
                ->sum('account_transactions.amount');
            
            $credit_total = (float) $query->clone()
                ->where('account_transactions.type', 'credit')
                ->sum('account_transactions.amount');
            
            $opening_balance = $debit_total - $credit_total;
        }

        return $opening_balance;
    }

    /**
     * Build union query for all transaction sources
     * Match existing /daybook logic using direct SQL queries
     */
    protected function buildDaybookQuery(
        $business_id,
        $start_date,
        $end_date,
        $location_id,
        $user_id,
        $account_ids,
        $party_id,
        $payment_method,
        $currency,
        $use_created_date,
        $module_filter,
        $permitted_locations
    ) {
        $queries = [];

        // 1. Sales (finalized) - Only show sales with actual cash payments (exclude credit/due sales)
        if (!$module_filter || $module_filter == 'sell') {
            $sales_query = DB::table('transactions as t')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->join('transaction_payments as tp', function($join) {
                    // Only include sales that have actual payments (exclude credit/due sales)
                    $join->on('t.id', '=', 'tp.transaction_id')
                         ->where('tp.is_return', 0)
                         ->whereNotNull('tp.account_id'); // Must have cash/bank account payment
                })
                ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
                ->leftJoin('users as u', 't.created_by', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final');
            
            // Apply date filter - use paid_on from transaction_payments (actual payment date)
            if ($start_date && $end_date) {
                $sales_query->whereBetween(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date))'), [$start_date, $end_date]);
            } elseif ($end_date && !$start_date) {
                $sales_query->where(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date))'), '<=', $end_date);
            }
            
            $sales_query->select(
                    DB::raw("COALESCE(tp.paid_on, t.transaction_date) as datetime"),
                    DB::raw("COALESCE(t.invoice_no, t.ref_no, CONCAT('SALE-', t.id)) as voucher_no"),
                    DB::raw("'sell' as module"),
                    DB::raw("COALESCE(c.name, 'Walk-in Customer') as party"),
                    DB::raw("COALESCE(c.id) as party_id"),
                    DB::raw("bl.name as location"),
                    DB::raw("bl.id as location_id"),
                    DB::raw("a.name as account"),
                    DB::raw("a.id as account_id"),
                    DB::raw("tp.amount as debit"),
                    DB::raw("0 as credit"),
                    DB::raw("CONCAT('Sale - ', COALESCE(t.invoice_no, t.ref_no)) as narration"),
                    DB::raw("CONCAT(u.first_name, ' ', u.last_name) as user"),
                    DB::raw("u.id as user_id"),
                    DB::raw("t.id as transaction_id"),
                    DB::raw("tp.id as transaction_payment_id"),
                    DB::raw("'transaction_payments' as source_table"),
                    DB::raw("tp.id as source_pk")
                );

            if ($location_id) {
                $sales_query->where('t.location_id', $location_id);
            }
            if ($user_id) {
                $sales_query->where('t.created_by', $user_id);
            }
            if ($party_id) {
                $sales_query->where('t.contact_id', $party_id);
            }
            if ($payment_method) {
                $sales_query->where('tp.method', $payment_method);
            }
            if ($permitted_locations != 'all') {
                $sales_query->whereIn('t.location_id', $permitted_locations);
            }
            // Account filter - only show payments for the specified account(s)
            if (!empty($account_ids) && is_array($account_ids) && count($account_ids) > 0) {
                $sales_query->whereIn('tp.account_id', $account_ids);
            }

            $sales_results = $sales_query->get();
            foreach ($sales_results as $row) {
                $queries[] = $row;
            }
        }

        // 2. Purchases (received) - Use payment date (paid_on) if available
        if (!$module_filter || $module_filter == 'purchase') {
            $purchase_query = DB::table('transactions as t')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
                ->leftJoin('users as u', 't.created_by', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase')
                ->where('t.status', 'received')
                ->where('tp.is_return', 0);
            
            // Apply date filter - use paid_on from transaction_payments
            if ($start_date && $end_date) {
                $purchase_query->whereBetween(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date, t.created_at))'), [$start_date, $end_date]);
            } elseif ($end_date && !$start_date) {
                $purchase_query->where(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date, t.created_at))'), '<=', $end_date);
            }
            
            $purchase_query->select(
                    DB::raw("COALESCE(tp.paid_on, t.transaction_date, t.created_at) as datetime"),
                    DB::raw("COALESCE(t.ref_no, CONCAT('PUR-', t.id)) as voucher_no"),
                    DB::raw("'purchase' as module"),
                    DB::raw("COALESCE(c.name, 'Supplier') as party"),
                    DB::raw("COALESCE(c.id) as party_id"),
                    DB::raw("bl.name as location"),
                    DB::raw("bl.id as location_id"),
                    DB::raw("a.name as account"),
                    DB::raw("a.id as account_id"),
                    DB::raw("0 as debit"),
                    DB::raw("COALESCE(tp.amount, t.final_total) as credit"),
                    DB::raw("CONCAT('Purchase - ', COALESCE(t.ref_no, t.id)) as narration"),
                    DB::raw("CONCAT(u.first_name, ' ', u.last_name) as user"),
                    DB::raw("u.id as user_id"),
                    DB::raw("t.id as transaction_id"),
                    DB::raw("tp.id as transaction_payment_id"),
                    DB::raw("'transaction_payments' as source_table"),
                    DB::raw("tp.id as source_pk")
                );

            if ($location_id) {
                $purchase_query->where('t.location_id', $location_id);
            }
            if ($user_id) {
                $purchase_query->where('t.created_by', $user_id);
            }
            if ($party_id) {
                $purchase_query->where('t.contact_id', $party_id);
            }
            if ($payment_method) {
                $purchase_query->where('tp.method', $payment_method);
            }
            if ($permitted_locations != 'all') {
                $purchase_query->whereIn('t.location_id', $permitted_locations);
            }
            if (!empty($account_ids) && is_array($account_ids) && count($account_ids) > 0) {
                $purchase_query->where(function($q) use ($account_ids) {
                    $q->whereIn('tp.account_id', $account_ids)
                      ->orWhereNull('tp.id');
                });
            }

            $purchase_results = $purchase_query->get();
            foreach ($purchase_results as $row) {
                $queries[] = $row;
            }
        }

        // 3. Expenses - Use payment date (paid_on) if available
        if (!$module_filter || $module_filter == 'expense') {
            $expense_query = DB::table('transactions as t')
                ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
                ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftJoin('transaction_payments as tp', 't.id', '=', 'tp.transaction_id')
                ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
                ->leftJoin('users as u', 't.created_by', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'expense');
            
            // Apply date filter - use paid_on from transaction_payments
            if ($start_date && $end_date) {
                $expense_query->whereBetween(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date, t.created_at))'), [$start_date, $end_date]);
            } elseif ($end_date && !$start_date) {
                $expense_query->where(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date, t.created_at))'), '<=', $end_date);
            }
            
            $expense_query->select(
                    DB::raw("COALESCE(tp.paid_on, t.transaction_date, t.created_at) as datetime"),
                    DB::raw("COALESCE(t.ref_no, CONCAT('EXP-', t.id)) as voucher_no"),
                    DB::raw("'expense' as module"),
                    DB::raw("COALESCE(ec.name, 'Expense') as party"),
                    DB::raw("NULL as party_id"),
                    DB::raw("bl.name as location"),
                    DB::raw("bl.id as location_id"),
                    DB::raw("a.name as account"),
                    DB::raw("a.id as account_id"),
                    DB::raw("0 as debit"),
                    DB::raw("t.final_total as credit"),
                    DB::raw("CONCAT('Expense - ', COALESCE(ec.name, ''), ' - ', COALESCE(t.additional_notes, '')) as narration"),
                    DB::raw("CONCAT(u.first_name, ' ', u.last_name) as user"),
                    DB::raw("u.id as user_id"),
                    DB::raw("t.id as transaction_id"),
                    DB::raw("tp.id as transaction_payment_id"),
                    DB::raw("'transaction_payments' as source_table"),
                    DB::raw("tp.id as source_pk")
                );

            if ($location_id) {
                $expense_query->where('t.location_id', $location_id);
            }
            if ($user_id) {
                $expense_query->where('t.created_by', $user_id);
            }
            if ($payment_method) {
                $expense_query->where('tp.method', $payment_method);
            }
            if ($permitted_locations != 'all') {
                $expense_query->whereIn('t.location_id', $permitted_locations);
            }
            if (!empty($account_ids) && is_array($account_ids) && count($account_ids) > 0) {
                $expense_query->where(function($q) use ($account_ids) {
                    $q->whereIn('tp.account_id', $account_ids)
                      ->orWhereNull('tp.id');
                });
            }

            $expense_results = $expense_query->get();
            foreach ($expense_results as $row) {
                $queries[] = $row;
            }
        }

        // 4. Account Transactions (Payments, Transfers, etc.)
        // Exclude sales, purchases, expenses, returns from account_transactions as they are handled separately
        if (!$module_filter || in_array($module_filter, ['payment', 'transfer', 'opening_balance', 'receive'])) {
            $account_trans_query = DB::table('account_transactions as at')
                ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
                ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
                ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
                ->leftJoin('contacts as c', function($join) {
                    $join->on('c.id', '=', 't.contact_id')
                         ->orOn('c.id', '=', DB::raw('tp.payment_for'));
                })
                ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
                ->where('a.business_id', $business_id)
                ->whereNull('at.deleted_at')
                // Exclude account_transactions linked to sales, purchases, expenses, and returns
                ->where(function($q) {
                    $q->whereNull('t.type')
                      ->orWhereNotIn('t.type', ['sell', 'sell_return', 'purchase', 'purchase_return', 'expense']);
                });
            
            // Apply account filters
            $has_account_filter = false;
            if (!empty($account_ids) && is_array($account_ids) && count($account_ids) > 0) {
                $account_trans_query->whereIn('at.account_id', $account_ids);
                $has_account_filter = true;
            }
            
            // Apply date filter
            if ($start_date && $end_date) {
                if ($has_account_filter) {
                    // For selected account(s), include opening balances regardless of date
                    $account_trans_query->where(function($q) use ($start_date, $end_date) {
                        $q->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);
                        $q->orWhere('at.sub_type', 'opening_balance');
                    });
                } else {
                    $account_trans_query->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);
                }
            } elseif ($end_date && !$start_date) {
                if ($has_account_filter) {
                    $account_trans_query->where(function($q) use ($end_date) {
                        $q->where(DB::raw('DATE(at.operation_date)'), '<=', $end_date);
                        $q->orWhere('at.sub_type', 'opening_balance');
                    });
                } else {
                    $account_trans_query->where(DB::raw('DATE(at.operation_date)'), '<=', $end_date);
                }
            }
            
            $account_trans_query->select(
                    DB::raw("at.operation_date as datetime"),
                    DB::raw("COALESCE(at.reff_no, CONCAT('ACC-', at.id)) as voucher_no"),
                    DB::raw("CASE 
                        WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND COALESCE(c.type, '') IN ('customer', 'both') THEN 'receive'
                        WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND c.type = 'supplier' THEN 'payment'
                        WHEN at.sub_type = 'opening_balance' THEN 'opening_balance'
                        ELSE COALESCE(at.sub_type, 'payment')
                    END as module"),
                    DB::raw("COALESCE(c.name, '-') as party"),
                    DB::raw("COALESCE(c.id) as party_id"),
                    DB::raw("COALESCE(bl.name, '-') as location"),
                    DB::raw("COALESCE(bl.id) as location_id"),
                    DB::raw("a.name as account"),
                    DB::raw("a.id as account_id"),
                    DB::raw("CASE 
                        WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND COALESCE(c.type, '') IN ('customer', 'both') THEN at.amount
                        WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND c.type = 'supplier' THEN 0
                        WHEN at.sub_type = 'opening_balance' THEN at.amount
                        WHEN COALESCE(at.type, '') = 'debit' THEN at.amount
                        ELSE 0
                    END as debit"),
                    DB::raw("CASE 
                        WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND COALESCE(c.type, '') IN ('customer', 'both') THEN 0
                        WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND c.type = 'supplier' THEN at.amount
                        WHEN at.sub_type = 'opening_balance' THEN 0
                        WHEN COALESCE(at.type, '') = 'credit' THEN at.amount
                        ELSE 0
                    END as credit"),
                    DB::raw("COALESCE(at.note, CASE 
                        WHEN at.sub_type = 'opening_balance' THEN CONCAT('Opening Balance - ', a.name)
                        ELSE CONCAT(COALESCE(at.sub_type, 'payment'), ' - ', a.name)
                    END) as narration"),
                    DB::raw("COALESCE(CONCAT(u.first_name, ' ', u.last_name), '-') as user"),
                    DB::raw("u.id as user_id"),
                    DB::raw("at.transaction_id"),
                    DB::raw("at.transaction_payment_id"),
                    DB::raw("at.id as account_transaction_id"),
                    DB::raw("'account_transactions' as source_table"),
                    DB::raw("at.id as source_pk")
                );
            
            if ($user_id) {
                $account_trans_query->where('at.created_by', $user_id);
            }
            if ($location_id) {
                $account_trans_query->where(function($q) use ($location_id) {
                    $q->where('bl.id', $location_id)
                      ->orWhereNull('bl.id');
                });
            }
            if ($party_id) {
                $account_trans_query->where(function($q) use ($party_id) {
                    $q->where('c.id', $party_id)
                      ->orWhereNull('c.id');
                });
            }
            if ($permitted_locations != 'all') {
                $account_trans_query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('bl.id', $permitted_locations)
                      ->orWhereNull('bl.id');
                });
            }

            $account_trans_results = $account_trans_query->get();
            foreach ($account_trans_results as $row) {
                $queries[] = $row;
            }
        }

        // 5. Journal Entries (if AccountingReports module tables exist)
        // Only include if all required tables exist
        if (!$module_filter || $module_filter == 'manual') {
            if (Schema::hasTable('ar_journal_entry_headers') && 
                Schema::hasTable('ar_journal_entry_lines') &&
                Schema::hasTable('ar_chart_of_accounts')) {
                
                try {
                    $journal_entries = DB::table('ar_journal_entry_headers as jeh')
                        ->join('ar_journal_entry_lines as jel', 'jeh.id', '=', 'jel.journal_entry_id')
                        ->leftJoin('ar_chart_of_accounts as coa', 'jel.account_id', '=', 'coa.id')
                        ->leftJoin('business_locations as bl', 'jeh.location_id', '=', 'bl.id')
                        ->leftJoin('users as u', 'jeh.created_by', '=', 'u.id')
                        ->where('jeh.business_id', $business_id)
                        ->where('jeh.status', 'posted')
                        ->whereBetween('jeh.voucher_date', [$start_date, $end_date])
                        ->whereNull('jeh.deleted_at')
                        ->select([
                            DB::raw("COALESCE(jeh.transaction_date, jeh.voucher_date) as datetime"),
                            'jeh.voucher_no',
                            DB::raw("'manual' as module"),
                            DB::raw("'-' as party_name"),
                            DB::raw("NULL as party_id"),
                            'jeh.location_id',
                            'bl.name as location_name',
                            'jel.account_id as account_id',
                            'coa.name as account',
                            'jel.debit',
                            'jel.credit',
                            'jeh.narration',
                            'jeh.created_by as user_id',
                            DB::raw("CONCAT(u.first_name, ' ', u.last_name) as user_name"),
                            DB::raw("'ar_journal_entry_lines' as source_table"),
                            'jel.id as source_pk',
                            DB::raw("NULL as transaction_id"),
                        ]);

                    if ($location_id) {
                        $journal_entries->where('jeh.location_id', $location_id);
                    }

                    if ($user_id) {
                        $journal_entries->where('jeh.created_by', $user_id);
                    }

                    if ($account_ids) {
                        $journal_entries->whereIn('jel.account_id', $account_ids);
                    }

                    if ($permitted_locations != 'all') {
                        $journal_entries->whereIn('jeh.location_id', $permitted_locations);
                    }

                    $journal_results = $journal_entries->get();
                    
                    foreach ($journal_results as $row) {
                        $queries[] = $row;
                    }
                } catch (\Exception $e) {
                    // Silently skip journal entries if there's an error (table might not be fully set up)
                    \Log::warning('DaybookQuery: Could not fetch journal entries: ' . $e->getMessage());
                }
            }
        }

        // Sort all entries by datetime
        usort($queries, function($a, $b) {
            $datetime_a = $a->datetime ?? $a->tx_datetime ?? '1970-01-01 00:00:00';
            $datetime_b = $b->datetime ?? $b->tx_datetime ?? '1970-01-01 00:00:00';
            return strcmp($datetime_a, $datetime_b);
        });

        return $queries;
    }
}

