<?php

namespace Modules\AccountingReports\Services;

use App\Account;
use App\Utils\TransactionUtil;
use Illuminate\Support\Facades\DB;

/**
 * BankbookQuery Service
 * 
 * Aggregates all bank transactions into a unified
 * chronological Bank Book listing with running balance calculation.
 * Bank Book shows only bank account transactions (deposits and withdrawals).
 */
class BankbookQuery
{
    protected $transactionUtil;
    
    public function __construct(TransactionUtil $transactionUtil)
    {
        $this->transactionUtil = $transactionUtil;
    }

    /**
     * Get Bank Book entries for a given date range with filters
     * 
     * @param int $business_id
     * @param string $start_date
     * @param string $end_date
     * @param array $filters
     * @return array
     */
    public function getBankbookEntries($business_id, $start_date, $end_date, array $filters = [])
    {
        $location_id = $filters['location_id'] ?? null;
        $user_id = $filters['user_id'] ?? null;
        $account_id = $filters['account_id'] ?? null;
        $permitted_locations = $filters['permitted_locations'] ?? 'all';

        // Get bank account IDs only
        $bank_account_ids = $this->getBankAccountIds($business_id, $account_id);
        
        if (empty($bank_account_ids)) {
            return [
                'entries' => [],
                'summary' => [
                    'opening_balance' => 0,
                    'total_debit' => 0,
                    'total_credit' => 0,
                    'closing_balance' => 0,
                    'total_deposits' => 0,
                    'total_withdrawals' => 0,
                    'total_entries' => 0,
                ],
                'filters_applied' => $filters,
            ];
        }
        
        // Calculate opening balance
        $opening_balance = $this->calculateOpeningBalance(
            $business_id, 
            $start_date, 
            $location_id, 
            $permitted_locations,
            $bank_account_ids
        );

        // Build union query for all transaction sources
        $entries = $this->buildBankbookQuery(
            $business_id,
            $start_date,
            $end_date,
            $location_id,
            $user_id,
            $bank_account_ids,
            $permitted_locations
        );

        // Process and format entries for DataTables (same format as DaybookController)
        $formatted_entries = [];
        $running_balance = $opening_balance;
        
        foreach ($entries as $entry) {
            // Convert to array if object
            $entry_array = is_object($entry) ? (array) $entry : $entry;
            if (!is_array($entry_array)) {
                continue;
            }
            
            $debit = (float) ($entry_array['debit'] ?? 0);
            $credit = (float) ($entry_array['credit'] ?? 0);
            
            // Determine type and amount (same as daybook format)
            if ($debit > 0) {
                $type = 'debit';
                $amount = $debit;
                $running_balance += $amount;
            } else {
                $type = 'credit';
                $amount = $credit;
                $running_balance -= $amount;
            }
            
            $formatted_entries[] = [
                'datetime' => $entry_array['datetime'] ?? null,
                'voucher_no' => $entry_array['voucher_no'] ?? '-',
                'module' => $entry_array['module'] ?? 'unknown',
                'party' => $entry_array['party'] ?? '-',
                'location' => $entry_array['location'] ?? '-',
                'account' => $entry_array['account'] ?? '-',
                'type' => $type,
                'amount' => $amount,
                'debit' => $debit,
                'credit' => $credit,
                'narration' => $entry_array['narration'] ?? '-',
                'user' => $entry_array['user'] ?? '-',
                'transaction_id' => $entry_array['transaction_id'] ?? null,
                'transaction_payment_id' => $entry_array['transaction_payment_id'] ?? null,
                'account_transaction_id' => $entry_array['account_transaction_id'] ?? null,
                'cheque_no' => $entry_array['cheque_no'] ?? null,
                'cheque_status' => $entry_array['cheque_status'] ?? null,
                'is_reconciled' => $entry_array['is_reconciled'] ?? false,
                'running_balance' => $running_balance,
            ];
        }

        // Calculate summary statistics
        $total_debit = array_sum(array_column($formatted_entries, 'debit'));
        $total_credit = array_sum(array_column($formatted_entries, 'credit'));
        $closing_balance = $opening_balance + ($total_debit - $total_credit);

        // Calculate opening balance for display (always in Debit side)
        $opening_balance_debit = abs($opening_balance);
        $opening_balance_credit = 0;

        // Calculate closing balance for display (always in Debit side)
        $closing_balance_debit = abs($closing_balance);
        $closing_balance_credit = 0;

        return [
            'entries' => $formatted_entries,
            'summary' => [
                'opening_balance' => $opening_balance,
                'opening_balance_debit' => $opening_balance_debit,
                'opening_balance_credit' => $opening_balance_credit,
                'total_debit' => $total_debit,
                'total_credit' => $total_credit,
                'closing_balance' => $closing_balance,
                'closing_balance_debit' => $closing_balance_debit,
                'closing_balance_credit' => $closing_balance_credit,
                'total_deposits' => $total_debit, // Deposits = Debit
                'total_withdrawals' => $total_credit, // Withdrawals = Credit
                'total_entries' => count($formatted_entries),
            ],
            'filters_applied' => $filters,
        ];
    }

    /**
     * Get bank account IDs only
     */
    protected function getBankAccountIds($business_id, $specific_account_id = null)
    {
        if ($specific_account_id) {
            // Verify it's a bank account
            $account = Account::find($specific_account_id);
            if ($account && $account->business_id == $business_id && !$account->is_closed) {
                $name_lower = strtolower($account->name);
                if (strpos($name_lower, 'bank') !== false || 
                    strpos($name_lower, 'cheque') !== false || 
                    strpos($name_lower, 'transfer') !== false) {
                    return [$specific_account_id];
                }
            }
            return [];
        }

        $bank_accounts = Account::where('business_id', $business_id)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('id')
            ->toArray();

        return !empty($bank_accounts) ? array_unique($bank_accounts) : [];
    }

    /**
     * Calculate opening balance from previous day's closing balance
     */
    protected function calculateOpeningBalance($business_id, $start_date, $location_id, $permitted_locations, $account_ids)
    {
        if (!$start_date || empty($account_ids)) {
            return 0;
        }

        $opening_balance = 0;
        
        $query = DB::table('account_transactions as at')
            ->join('accounts as a', 'at.account_id', '=', 'a.id')
            ->where('a.business_id', $business_id)
            ->whereDate('at.operation_date', '<', $start_date)
            ->whereIn('at.account_id', $account_ids)
            ->whereNull('at.deleted_at');

        // Filter by location if specified
        if ($location_id || $permitted_locations != 'all') {
            $query->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id');
            
            if ($location_id) {
                $query->where(function($q) use ($location_id) {
                    $q->where('t.location_id', $location_id)
                      ->orWhereNull('t.location_id');
                });
            }

            if ($permitted_locations != 'all') {
                $query->where(function($q) use ($permitted_locations) {
                    $q->whereIn('t.location_id', $permitted_locations)
                      ->orWhereNull('t.location_id');
                });
            }
        }

        $debit_total = (float) $query->clone()
            ->where('at.type', 'debit')
            ->sum('at.amount');
        
        $credit_total = (float) $query->clone()
            ->where('at.type', 'credit')
            ->sum('at.amount');
        
        $opening_balance = $debit_total - $credit_total;

        return $opening_balance;
    }

    /**
     * Build union query for all transaction sources (bank accounts only)
     */
    protected function buildBankbookQuery(
        $business_id,
        $start_date,
        $end_date,
        $location_id,
        $user_id,
        $bank_account_ids,
        $permitted_locations
    ) {
        $queries = [];

        // 1. Sales (finalized) - Only bank payments
        $sales_query = DB::table('transactions as t')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
            ->join('transaction_payments as tp', function($join) use ($bank_account_ids) {
                $join->on('t.id', '=', 'tp.transaction_id')
                     ->where('tp.is_return', 0)
                     ->whereIn('tp.account_id', $bank_account_ids); // Only bank accounts
            })
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->leftJoin('users as u', 't.created_by', '=', 'u.id')
            ->leftJoin('ar_cheque_book_entries as cbe', function($join) {
                $join->on('tp.id', '=', 'cbe.transaction_payment_id')
                     ->whereNull('cbe.deleted_at');
            })
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final');
        
        if ($start_date && $end_date) {
            $sales_query->whereBetween(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date))'), [$start_date, $end_date]);
        } elseif ($end_date && !$start_date) {
            $sales_query->where(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date))'), '<=', $end_date);
        }
        
        $sales_query->select(
                DB::raw("COALESCE(tp.paid_on, t.transaction_date) as datetime"),
                DB::raw("COALESCE(t.invoice_no, t.ref_no, CONCAT('SALE-', t.id)) as voucher_no"),
                DB::raw("'sell' as module"),
                DB::raw("COALESCE(c.name, 'Walk-in Customer') as party"),
                DB::raw("COALESCE(c.id) as party_id"),
                DB::raw("bl.name as location"),
                DB::raw("bl.id as location_id"),
                DB::raw("a.name as account"),
                DB::raw("a.id as account_id"),
                DB::raw("tp.amount as debit"), // Sales always debit (money deposited)
                DB::raw("0 as credit"),
                DB::raw("CONCAT('Sale - ', COALESCE(t.invoice_no, t.ref_no)) as narration"),
                DB::raw("CONCAT(u.first_name, ' ', u.last_name) as user"),
                DB::raw("u.id as user_id"),
                DB::raw("t.id as transaction_id"),
                DB::raw("tp.id as transaction_payment_id"),
                DB::raw("COALESCE(cbe.cheque_no, '') as cheque_no"),
                DB::raw("COALESCE(cbe.status, '') as cheque_status"),
                DB::raw("COALESCE(cbe.status, '') = 'cleared' as is_reconciled"),
                DB::raw("'transaction_payments' as source_table"),
                DB::raw("tp.id as source_pk")
            );

        if ($location_id) {
            $sales_query->where('t.location_id', $location_id);
        }
        if ($user_id) {
            $sales_query->where('t.created_by', $user_id);
        }
        if ($permitted_locations != 'all') {
            $sales_query->whereIn('t.location_id', $permitted_locations);
        }

        $sales_results = $sales_query->get();
        foreach ($sales_results as $row) {
            $queries[] = $row;
        }

        // 2. Purchases (received) - Only bank payments
        $purchase_query = DB::table('transactions as t')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
            ->join('transaction_payments as tp', function($join) use ($bank_account_ids) {
                $join->on('t.id', '=', 'tp.transaction_id')
                     ->where('tp.is_return', 0)
                     ->whereIn('tp.account_id', $bank_account_ids); // Only bank accounts
            })
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->leftJoin('users as u', 't.created_by', '=', 'u.id')
            ->leftJoin('ar_cheque_book_entries as cbe', function($join) {
                $join->on('tp.id', '=', 'cbe.transaction_payment_id')
                     ->whereNull('cbe.deleted_at');
            })
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received');
        
        if ($start_date && $end_date) {
            $purchase_query->whereBetween(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date, t.created_at))'), [$start_date, $end_date]);
        } elseif ($end_date && !$start_date) {
            $purchase_query->where(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date, t.created_at))'), '<=', $end_date);
        }
        
        $purchase_query->select(
                DB::raw("COALESCE(tp.paid_on, t.transaction_date, t.created_at) as datetime"),
                DB::raw("COALESCE(t.ref_no, CONCAT('PUR-', t.id)) as voucher_no"),
                DB::raw("'purchase' as module"),
                DB::raw("COALESCE(c.name, 'Supplier') as party"),
                DB::raw("COALESCE(c.id) as party_id"),
                DB::raw("bl.name as location"),
                DB::raw("bl.id as location_id"),
                DB::raw("a.name as account"),
                DB::raw("a.id as account_id"),
                DB::raw("0 as debit"),
                DB::raw("COALESCE(tp.amount, t.final_total) as credit"),
                DB::raw("CONCAT('Purchase - ', COALESCE(t.ref_no, t.id)) as narration"),
                DB::raw("CONCAT(u.first_name, ' ', u.last_name) as user"),
                DB::raw("u.id as user_id"),
                DB::raw("t.id as transaction_id"),
                DB::raw("tp.id as transaction_payment_id"),
                DB::raw("COALESCE(cbe.cheque_no, '') as cheque_no"),
                DB::raw("COALESCE(cbe.status, '') as cheque_status"),
                DB::raw("COALESCE(cbe.status, '') = 'cleared' as is_reconciled"),
                DB::raw("'transaction_payments' as source_table"),
                DB::raw("tp.id as source_pk")
            );

        if ($location_id) {
            $purchase_query->where('t.location_id', $location_id);
        }
        if ($user_id) {
            $purchase_query->where('t.created_by', $user_id);
        }
        if ($permitted_locations != 'all') {
            $purchase_query->whereIn('t.location_id', $permitted_locations);
        }

        $purchase_results = $purchase_query->get();
        foreach ($purchase_results as $row) {
            $queries[] = $row;
        }

        // 3. Expenses - Only bank payments
        $expense_query = DB::table('transactions as t')
            ->leftJoin('expense_categories as ec', 't.expense_category_id', '=', 'ec.id')
            ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
            ->join('transaction_payments as tp', function($join) use ($bank_account_ids) {
                $join->on('t.id', '=', 'tp.transaction_id')
                     ->whereIn('tp.account_id', $bank_account_ids); // Only bank accounts
            })
            ->leftJoin('accounts as a', 'tp.account_id', '=', 'a.id')
            ->leftJoin('users as u', 't.created_by', '=', 'u.id')
            ->leftJoin('ar_cheque_book_entries as cbe', function($join) {
                $join->on('tp.id', '=', 'cbe.transaction_payment_id')
                     ->whereNull('cbe.deleted_at');
            })
            ->where('t.business_id', $business_id)
            ->where('t.type', 'expense');
        
        if ($start_date && $end_date) {
            $expense_query->whereBetween(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date, t.created_at))'), [$start_date, $end_date]);
        } elseif ($end_date && !$start_date) {
            $expense_query->where(DB::raw('DATE(COALESCE(tp.paid_on, t.transaction_date, t.created_at))'), '<=', $end_date);
        }
        
        $expense_query->select(
                DB::raw("COALESCE(tp.paid_on, t.transaction_date, t.created_at) as datetime"),
                DB::raw("COALESCE(t.ref_no, CONCAT('EXP-', t.id)) as voucher_no"),
                DB::raw("'expense' as module"),
                DB::raw("COALESCE(ec.name, 'Expense') as party"),
                DB::raw("NULL as party_id"),
                DB::raw("bl.name as location"),
                DB::raw("bl.id as location_id"),
                DB::raw("a.name as account"),
                DB::raw("a.id as account_id"),
                DB::raw("0 as debit"),
                DB::raw("COALESCE(tp.amount, t.final_total) as credit"),
                DB::raw("CONCAT('Expense - ', COALESCE(ec.name, ''), ' - ', COALESCE(t.additional_notes, '')) as narration"),
                DB::raw("CONCAT(u.first_name, ' ', u.last_name) as user"),
                DB::raw("u.id as user_id"),
                DB::raw("t.id as transaction_id"),
                DB::raw("tp.id as transaction_payment_id"),
                DB::raw("COALESCE(cbe.cheque_no, '') as cheque_no"),
                DB::raw("COALESCE(cbe.status, '') as cheque_status"),
                DB::raw("COALESCE(cbe.status, '') = 'cleared' as is_reconciled"),
                DB::raw("'transaction_payments' as source_table"),
                DB::raw("tp.id as source_pk")
            );

        if ($location_id) {
            $expense_query->where('t.location_id', $location_id);
        }
        if ($user_id) {
            $expense_query->where('t.created_by', $user_id);
        }
        if ($permitted_locations != 'all') {
            $expense_query->whereIn('t.location_id', $permitted_locations);
        }

        $expense_results = $expense_query->get();
        foreach ($expense_results as $row) {
            $queries[] = $row;
        }

        // 4. Account Transactions (Direct payments/receipts) - Only bank accounts
        $account_trans_query = DB::table('account_transactions as at')
            ->leftJoin('accounts as a', 'at.account_id', '=', 'a.id')
            ->leftJoin('transactions as t', 'at.transaction_id', '=', 't.id')
            ->leftJoin('transaction_payments as tp', 'at.transaction_payment_id', '=', 'tp.id')
            ->leftJoin('contacts as c', function($join) {
                $join->on('c.id', '=', 't.contact_id')
                     ->orOn('c.id', '=', DB::raw('tp.payment_for'));
            })
            ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
            ->leftJoin('users as u', 'at.created_by', '=', 'u.id')
            ->leftJoin('ar_cheque_book_entries as cbe', function($join) {
                $join->on('at.transaction_payment_id', '=', 'cbe.transaction_payment_id')
                     ->whereNull('cbe.deleted_at');
            })
            ->where('a.business_id', $business_id)
            ->whereIn('at.account_id', $bank_account_ids) // Only bank accounts
            ->whereNull('at.deleted_at')
            ->where(function($q) {
                $q->whereNull('t.type')
                  ->orWhereNotIn('t.type', ['sell', 'sell_return', 'purchase', 'purchase_return', 'expense']);
            });
        
        if ($start_date && $end_date) {
            $account_trans_query->whereBetween(DB::raw('DATE(at.operation_date)'), [$start_date, $end_date]);
        } elseif ($end_date && !$start_date) {
            $account_trans_query->where(DB::raw('DATE(at.operation_date)'), '<=', $end_date);
        }
        
        $account_trans_query->select(
                DB::raw("at.operation_date as datetime"),
                DB::raw("COALESCE(at.reff_no, CONCAT('ACC-', at.id)) as voucher_no"),
                DB::raw("CASE 
                    WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND COALESCE(c.type, '') IN ('customer', 'both') THEN 'receive'
                    WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND c.type = 'supplier' THEN 'payment'
                    WHEN at.sub_type = 'opening_balance' THEN 'opening_balance'
                    ELSE COALESCE(at.sub_type, 'payment')
                END as module"),
                DB::raw("COALESCE(c.name, '-') as party"),
                DB::raw("COALESCE(c.id) as party_id"),
                DB::raw("COALESCE(bl.name, '-') as location"),
                DB::raw("COALESCE(bl.id) as location_id"),
                DB::raw("a.name as account"),
                DB::raw("a.id as account_id"),
                DB::raw("CASE 
                    WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND COALESCE(c.type, '') IN ('customer', 'both') THEN at.amount
                    WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND c.type = 'supplier' THEN 0
                    WHEN at.sub_type = 'opening_balance' THEN at.amount
                    WHEN COALESCE(at.type, '') = 'debit' THEN at.amount
                    ELSE 0
                END as debit"),
                DB::raw("CASE 
                    WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND COALESCE(c.type, '') IN ('customer', 'both') THEN 0
                    WHEN COALESCE(at.sub_type, 'payment') = 'payment' AND c.type = 'supplier' THEN at.amount
                    WHEN at.sub_type = 'opening_balance' THEN 0
                    WHEN COALESCE(at.type, '') = 'credit' THEN at.amount
                    ELSE 0
                END as credit"),
                DB::raw("COALESCE(at.note, CASE 
                    WHEN at.sub_type = 'opening_balance' THEN CONCAT('Opening Balance - ', a.name)
                    ELSE CONCAT(COALESCE(at.sub_type, 'payment'), ' - ', a.name)
                END) as narration"),
                DB::raw("COALESCE(CONCAT(u.first_name, ' ', u.last_name), '-') as user"),
                DB::raw("u.id as user_id"),
                DB::raw("at.transaction_id"),
                DB::raw("at.transaction_payment_id"),
                DB::raw("COALESCE(cbe.cheque_no, '') as cheque_no"),
                DB::raw("COALESCE(cbe.status, '') as cheque_status"),
                DB::raw("COALESCE(cbe.status, '') = 'cleared' as is_reconciled"),
                DB::raw("'account_transactions' as source_table"),
                DB::raw("at.id as source_pk")
            );
        
        if ($user_id) {
            $account_trans_query->where('at.created_by', $user_id);
        }
        if ($location_id) {
            $account_trans_query->where(function($q) use ($location_id) {
                $q->where('bl.id', $location_id)
                  ->orWhereNull('bl.id');
            });
        }
        if ($permitted_locations != 'all') {
            $account_trans_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('bl.id', $permitted_locations)
                  ->orWhereNull('bl.id');
            });
        }

        $account_trans_results = $account_trans_query->get();
        foreach ($account_trans_results as $row) {
            $queries[] = $row;
        }

        // 5. Cheque Book Entries (standalone entries not linked to transactions)
        $cheque_query = DB::table('ar_cheque_book_entries as cbe')
            ->leftJoin('accounts as a', 'cbe.account_id', '=', 'a.id')
            ->leftJoin('business_locations as bl', 'cbe.location_id', '=', 'bl.id')
            ->leftJoin('users as u', 'cbe.created_by', '=', 'u.id')
            ->where('cbe.business_id', $business_id)
            ->whereNull('cbe.deleted_at')
            ->whereNull('cbe.transaction_payment_id'); // Only standalone entries
        
        if ($start_date && $end_date) {
            $cheque_query->whereBetween(DB::raw('DATE(cbe.cheque_date)'), [$start_date, $end_date]);
        } elseif ($end_date && !$start_date) {
            $cheque_query->where(DB::raw('DATE(cbe.cheque_date)'), '<=', $end_date);
        }
        
        $cheque_query->select(
                DB::raw("cbe.cheque_date as datetime"),
                DB::raw("CONCAT('CHEQUE-', cbe.cheque_no) as voucher_no"),
                DB::raw("CASE WHEN cbe.type = 'issued' THEN 'cheque_issued' ELSE 'cheque_received' END as module"),
                DB::raw("COALESCE(cbe.payee_name, '-') as party"),
                DB::raw("NULL as party_id"),
                DB::raw("COALESCE(bl.name, '-') as location"),
                DB::raw("COALESCE(bl.id) as location_id"),
                DB::raw("a.name as account"),
                DB::raw("a.id as account_id"),
                DB::raw("CASE WHEN cbe.type = 'received' THEN cbe.amount ELSE 0 END as debit"),
                DB::raw("CASE WHEN cbe.type = 'issued' THEN cbe.amount ELSE 0 END as credit"),
                DB::raw("COALESCE(cbe.narration, CONCAT(cbe.type, ' Cheque - ', cbe.cheque_no)) as narration"),
                DB::raw("COALESCE(CONCAT(u.first_name, ' ', u.last_name), '-') as user"),
                DB::raw("u.id as user_id"),
                DB::raw("NULL as transaction_id"),
                DB::raw("NULL as transaction_payment_id"),
                DB::raw("cbe.cheque_no as cheque_no"),
                DB::raw("cbe.status as cheque_status"),
                DB::raw("CASE WHEN cbe.status = 'cleared' THEN 1 ELSE 0 END as is_reconciled"),
                DB::raw("'ar_cheque_book_entries' as source_table"),
                DB::raw("cbe.id as source_pk")
            );
        
        if ($location_id) {
            $cheque_query->where('cbe.location_id', $location_id);
        }
        if ($user_id) {
            $cheque_query->where('cbe.created_by', $user_id);
        }
        if ($permitted_locations != 'all') {
            $cheque_query->where(function($q) use ($permitted_locations) {
                $q->whereIn('bl.id', $permitted_locations)
                  ->orWhereNull('bl.id');
            });
        }

        $cheque_results = $cheque_query->get();
        foreach ($cheque_results as $row) {
            $queries[] = $row;
        }

        // Sort all entries by datetime
        usort($queries, function($a, $b) {
            $datetime_a = is_array($a) ? ($a['datetime'] ?? '1970-01-01 00:00:00') : ($a->datetime ?? '1970-01-01 00:00:00');
            $datetime_b = is_array($b) ? ($b['datetime'] ?? '1970-01-01 00:00:00') : ($b->datetime ?? '1970-01-01 00:00:00');
            return strcmp($datetime_a, $datetime_b);
        });

        return $queries;
    }
}

