@extends('layouts.app')

@section('title', __('accounting-reports::lang.direct_incomes'))

@section('content')

<!-- Content Header (Page header) -->
<section class="content-header">
    <h1>@lang('accounting-reports::lang.direct_incomes')</h1>
</section>

<!-- Main content -->
<section class="content">
    @component('components.widget', ['class' => 'box-primary'])
        <div class="row">
            <div class="col-md-12">
                <div class="btn-group">
                    <button type="button" class="btn btn-success btn-modal" data-href="{{ action([\Modules\AccountingReports\Http\Controllers\DirectIncomeController::class, 'create']) }}" data-container=".direct_income_modal">
                        <i class="fa fa-plus"></i> @lang('accounting-reports::lang.add_direct_income')
                    </button>
                </div>
            </div>
        </div>
        <br>
        <div class="row">
            <div class="col-md-3">
                <div class="form-group">
                    {!! Form::label('filter_location_id', __('purchase.business_location') . ':') !!}
                    {!! Form::select('filter_location_id', $locations, null, ['class' => 'form-control select2', 'placeholder' => __('lang_v1.all')]) !!}
                </div>
            </div>
            <div class="col-md-3">
                <div class="form-group">
                    {!! Form::label('filter_account_id', __('account.account') . ':') !!}
                    {!! Form::select('filter_account_id', $accounts, null, ['class' => 'form-control select2', 'placeholder' => __('lang_v1.all')]) !!}
                </div>
            </div>
            <div class="col-md-3">
                <div class="form-group">
                    {!! Form::label('start_date', __('report.start_date') . ':') !!}
                    {!! Form::text('start_date', null, ['class' => 'form-control date-picker', 'readonly']) !!}
                </div>
            </div>
            <div class="col-md-3">
                <div class="form-group">
                    {!! Form::label('end_date', __('report.end_date') . ':') !!}
                    {!! Form::text('end_date', null, ['class' => 'form-control date-picker', 'readonly']) !!}
                </div>
            </div>
        </div>
        
        <div class="table-responsive">
            <table class="table table-bordered table-striped table-hover" id="direct_incomes_table" style="width:100%;">
                <thead>
                    <tr>
                        <th>@lang('accounting-reports::lang.date')</th>
                        <th>@lang('accounting-reports::lang.name')</th>
                        <th>@lang('accounting-reports::lang.amount')</th>
                        <th>@lang('purchase.business_location')</th>
                        <th>@lang('account.account')</th>
                        <th>@lang('accounting-reports::lang.reference_no')</th>
                        <th>@lang('accounting-reports::lang.document')</th>
                        <th>@lang('messages.action')</th>
                    </tr>
                </thead>
                <tfoot>
                    <tr>
                        <th colspan="2" style="text-align:right">@lang('sale.total'):</th>
                        <th></th>
                        <th colspan="5"></th>
                    </tr>
                </tfoot>
            </table>
        </div>
    @endcomponent
    
    <div class="modal fade direct_income_modal" tabindex="-1" role="dialog"></div>
</section>

@endsection

@section('javascript')
<script type="text/javascript">
    $(document).ready(function() {
        var direct_incomes_table = $('#direct_incomes_table').DataTable({
            processing: true,
            serverSide: true,
            ajax: {
                url: '{{ action([\Modules\AccountingReports\Http\Controllers\DirectIncomeController::class, "getData"]) }}',
                data: function(d) {
                    d.location_id = $('#filter_location_id').val();
                    d.account_id = $('#filter_account_id').val();
                    d.start_date = $('#start_date').val();
                    d.end_date = $('#end_date').val();
                }
            },
            columns: [
                { data: 'income_date', name: 'income_date' },
                { data: 'name', name: 'name' },
                { data: 'amount', name: 'amount' },
                { data: 'location', name: 'location_id' },
                { data: 'account', name: 'account_id' },
                { data: 'reference_no', name: 'reference_no' },
                { data: 'document', name: 'document' },
                { data: 'action', name: 'action', orderable: false, searchable: false }
            ],
            footerCallback: function (row, data, start, end, display) {
                var api = this.api();
                var intVal = function (i) {
                    return typeof i === 'string' ? i.replace(/[\$,]/g, '') * 1 : typeof i === 'number' ? i : 0;
                };
                
                var total = api.column(2).data().reduce(function (a, b) {
                    return intVal(a) + intVal($(b).text());
                }, 0);
                
                $(api.column(2).footer()).html(__currency_trans_from_en(total, true));
            }
        });

        $('#filter_location_id, #filter_account_id, #start_date, #end_date').on('change', function() {
            direct_incomes_table.ajax.reload();
        });

        $(document).on('click', '.edit_income', function(e) {
            e.preventDefault();
            $('div.direct_income_modal').load($(this).data('href'), function() {
                $(this).modal('show');
                $('form#direct_income_form').submit(function(e) {
                    e.preventDefault();
                    $.ajax({
                        method: 'POST',
                        url: $(this).attr('action'),
                        data: new FormData(this),
                        processData: false,
                        contentType: false,
                        dataType: 'json',
                        success: function(response) {
                            if (response.success) {
                                $('div.direct_income_modal').modal('hide');
                                toastr.success(response.msg);
                                direct_incomes_table.ajax.reload();
                            } else {
                                toastr.error(response.msg);
                            }
                        }
                    });
                });
            });
        });

        $(document).on('click', '.view_income', function(e) {
            e.preventDefault();
            $('div.direct_income_modal').load($(this).data('href'), function() {
                $(this).modal('show');
            });
        });

        $(document).on('click', '.delete_income', function() {
            var url = $(this).data('href');
            swal({
                title: LANG.sure,
                text: LANG.confirm_delete_income,
                icon: "warning",
                buttons: true,
                dangerMode: true,
            }).then((confirmed) => {
                if (confirmed) {
                    $.ajax({
                        method: 'DELETE',
                        url: url,
                        dataType: 'json',
                        success: function(response) {
                            if (response.success) {
                                toastr.success(response.msg);
                                direct_incomes_table.ajax.reload();
                            } else {
                                toastr.error(response.msg);
                            }
                        }
                    });
                }
            });
        });

        $(document).on('submit', 'form#add_direct_income_form', function(e) {
            e.preventDefault();
            $(this).validate();
            if ($(this).valid()) {
                $.ajax({
                    method: 'POST',
                    url: $(this).attr('action'),
                    data: new FormData(this),
                    processData: false,
                    contentType: false,
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            $('div.direct_income_modal').modal('hide');
                            toastr.success(response.msg);
                            direct_incomes_table.ajax.reload();
                        } else {
                            toastr.error(response.msg);
                        }
                    }
                });
            }
        });
        
        // Auto-populate payment amount from income amount
        $(document).on('change blur', 'input#amount', function() {
            var amount = __read_number($(this)) || 0;
            if (amount > 0) {
                var current_payment = __read_number($('input#payment_amount')) || 0;
                if (current_payment == 0 || current_payment < amount) {
                    var formatted_amount = parseFloat(amount).toFixed(2);
                    $('input#payment_amount').val(formatted_amount);
                    $('input#payment_amount').trigger('change');
                }
            }
            calculatePaymentDue();
        });
        
        // Payment due calculation function
        function calculatePaymentDue() {
            var amount = __read_number($('input#amount')) || 0;
            var payment_amount = __read_number($('input#payment_amount')) || 0;
            var payment_due = amount - payment_amount;
            $('#payment_due').text(__currency_trans_from_en(payment_due, true, false));
        }
        
        // Calculate payment due when amount or payment amount changes
        $(document).on('change blur', 'input#amount, input#payment_amount', function() {
            calculatePaymentDue();
        });
        
        // Initialize date pickers when modal opens
        $(document).on('click', '.btn-modal', function(e) {
            e.preventDefault();
            $('div.direct_income_modal').load($(this).data('href'), function() {
                $(this).modal('show');
                $('.date-picker').datepicker(datepicker_params);
                $('.received_on').datetimepicker({
                    format: moment_date_format + ' ' + moment_time_format,
                    ignoreReadonly: true,
                });
                $('.select2').select2({width: '100%'});
                calculatePaymentDue();
                
                $('input#amount').on('change blur', function() {
                    var amount = __read_number($(this)) || 0;
                    if (amount > 0) {
                        var current_payment = __read_number($('input#payment_amount')) || 0;
                        if (current_payment == 0 || current_payment < amount) {
                            $('input#payment_amount').val(__currency_trans_from_en(amount, false, false));
                            $('input#payment_amount').trigger('change');
                        }
                    }
                    calculatePaymentDue();
                });
            });
        });
        
        // Initialize date pickers for edit modal
        $(document).on('click', '.edit_income', function(e) {
            e.preventDefault();
            $('div.direct_income_modal').load($(this).data('href'), function() {
                $(this).modal('show');
                $('.date-picker').datepicker(datepicker_params);
                $('.received_on').datetimepicker({
                    format: moment_date_format + ' ' + moment_time_format,
                    ignoreReadonly: true,
                });
                $('.select2').select2({width: '100%'});
                calculatePaymentDue();
                
                $('input#amount').on('change blur', function() {
                    var amount = __read_number($(this)) || 0;
                    if (amount > 0) {
                        var current_payment = __read_number($('input#payment_amount')) || 0;
                        if (current_payment == 0 || current_payment < amount) {
                            $('input#payment_amount').val(__currency_trans_from_en(amount, false, false));
                            $('input#payment_amount').trigger('change');
                        }
                    }
                    calculatePaymentDue();
                });
                
                $('form#direct_income_form').submit(function(e) {
                    e.preventDefault();
                    $.ajax({
                        method: 'POST',
                        url: $(this).attr('action'),
                        data: new FormData(this),
                        processData: false,
                        contentType: false,
                        dataType: 'json',
                        success: function(response) {
                            if (response.success) {
                                $('div.direct_income_modal').modal('hide');
                                toastr.success(response.msg);
                                direct_incomes_table.ajax.reload();
                            } else {
                                toastr.error(response.msg);
                            }
                        }
                    });
                });
            });
        });
    });
</script>
@endsection


