@extends('layouts.app')

@section('title', __('accounting-reports::lang.day_book'))

@section('content')

<!-- Content Header (Page header) -->
<section class="content-header">
    <h1>@lang('accounting-reports::lang.day_book')
        <small class="text-muted">@lang('accounting-reports::lang.tally_style')</small>
    </h1>
</section>

<!-- Main content -->
<section class="content">
    <div class="print_section">
        <h2 class="text-center">{{ session()->get('business.name') }} - @lang('accounting-reports::lang.day_book')</h2>
        <p class="text-center text-muted" id="db_period"></p>
    </div>

    <div class="row no-print">
        <div class="col-md-3 col-xs-12">
            <div class="input-group">
                <span class="input-group-addon bg-light-blue"><i class="fa fa-map-marker"></i></span>
                <select class="form-control select2" id="db_location_filter" style="width:100%">
                    <option value="">@lang('messages.all')</option>
                    @foreach ($locations as $key => $value)
                        <option value="{{ $key }}">{{ $value }}</option>
                    @endforeach
                </select>
            </div>
        </div>
    
        <div class="col-md-3 col-xs-12">
            <div class="form-group">
                <div class="input-group">
                    <span class="input-group-addon"><i class="fa fa-calendar"></i></span>
                    <input type="text" class="form-control" id="db_date_range" readonly placeholder="@lang('report.date_range')">
                </div>
            </div>
        </div>
        
        <div class="col-md-3 col-xs-12">
            <button type="button" class="btn btn-primary btn-block" id="generate_db_btn">
                <i class="fa fa-refresh"></i> @lang('accounting-reports::lang.generate')
            </button>
        </div>

        <div class="col-md-3 col-xs-12">
            <button class="btn btn-primary btn-block" aria-label="Print" onclick="window.print();">
                <i class="fa fa-print"></i> @lang('messages.print')
            </button>
        </div>
    </div>

    <div class="row">
        <div id="db_data_div" class="col-md-12">
            <div class="alert alert-info text-center">
                <i class="fa fa-info-circle"></i> @lang('accounting-reports::lang.select_date_range')
            </div>
        </div>
    </div>

</section>
<!-- /.content -->

@endsection

@section('javascript')
<script type="text/javascript">
    $(document).ready(function() {
        // Initialize date range picker (matching Profit & Loss)
        $('#db_date_range').daterangepicker(
            dateRangeSettings,
            function (start, end) {
                $('#db_date_range').val(start.format(moment_date_format) + ' ~ ' + end.format(moment_date_format));
            }
        );

        // Initialize select2
        $('#db_location_filter').select2();

        // Load Day Book on button click or location change
        $('#generate_db_btn, #db_location_filter').on('click change', function() {
            loadDayBook();
        });

        function loadDayBook() {
            var start_date = '';
            var end_date = '';
            
            if ($('#db_date_range').val() != '' && $('#db_date_range').data('daterangepicker')) {
                start_date = $('#db_date_range').data('daterangepicker').startDate.format('YYYY-MM-DD');
                end_date = $('#db_date_range').data('daterangepicker').endDate.format('YYYY-MM-DD');
            }
            
            var location_id = $('#db_location_filter').val() || '';
            
            if (!start_date || !end_date) {
                $('#db_data_div').html('<div class="alert alert-warning"><i class="fa fa-exclamation-triangle"></i> @lang("accounting-reports::lang.please_select_date_range")</div>');
                return;
            }

            $('#db_data_div').html('<div class="text-center"><i class="fa fa-spinner fa-spin fa-3x"></i><br>Loading...</div>');

            $.ajax({
                url: '{{ route("accounting-reports.day-book.get-data") }}',
                method: 'GET',
                data: {
                    start_date: start_date,
                    end_date: end_date,
                    location_id: location_id
                },
                success: function(response) {
                    if (response && response.success) {
                        renderTallyDayBook(response);
                    } else {
                        var errorMsg = response.error || 'Error loading data';
                        $('#db_data_div').html('<div class="alert alert-danger"><i class="fa fa-exclamation-triangle"></i> ' + errorMsg + '</div>');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error:', xhr.responseText);
                    var errorMsg = 'Error: Internal Server Error';
                    try {
                        if (xhr.responseJSON && xhr.responseJSON.error) {
                            errorMsg = xhr.responseJSON.error;
                        } else if (xhr.responseText) {
                            var response = JSON.parse(xhr.responseText);
                            if (response.error) {
                                errorMsg = response.error;
                            }
                        }
                    } catch(e) {
                        errorMsg = 'Error: ' + error + ' (Status: ' + xhr.status + ')';
                    }
                    $('#db_data_div').html('<div class="alert alert-danger"><i class="fa fa-exclamation-triangle"></i> ' + errorMsg + '</div>');
                }
            });
        }

        function renderTallyDayBook(data) {
            var html = '';
            var businessName = '{{ session()->get("business.name") }}';
            var startDate = data.start_date || '';
            var endDate = data.end_date || '';
            
            // Update period display
            $('#db_period').text('Period: ' + startDate + ' to ' + endDate);
            
            // Summary card
            html += '<div class="row">';
            html += '<div class="col-md-12">';
            html += '<div class="box box-primary">';
            html += '<div class="box-header with-border">';
            html += '<h3 class="box-title">@lang("accounting-reports::lang.summary")</h3>';
            html += '</div>';
            html += '<div class="box-body">';
            html += '<div class="row">';
            html += '<div class="col-md-3">';
            html += '<div class="info-box">';
            html += '<span class="info-box-icon bg-blue"><i class="fa fa-file-text"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">@lang("accounting-reports::lang.total_vouchers")</span>';
            html += '<span class="info-box-number">' + (data.entries && data.entries.length > 0 ? data.entries.length : 0) + '</span>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '<div class="col-md-3">';
            html += '<div class="info-box">';
            html += '<span class="info-box-icon bg-green"><i class="fa fa-list"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">@lang("accounting-reports::lang.total_entries")</span>';
            html += '<span class="info-box-number">' + (data.entries && data.entries.length > 0 ? data.entries.length : 0) + '</span>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '<div class="col-md-3">';
            html += '<div class="info-box">';
            html += '<span class="info-box-icon bg-yellow"><i class="fa fa-arrow-left"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">@lang("accounting-reports::lang.total_debit")</span>';
            html += '<span class="info-box-number">' + __currency_trans_from_en(data.total_debit || 0, true) + '</span>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '<div class="col-md-3">';
            html += '<div class="info-box">';
            html += '<span class="info-box-icon bg-red"><i class="fa fa-arrow-right"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">@lang("accounting-reports::lang.total_credit")</span>';
            html += '<span class="info-box-number">' + __currency_trans_from_en(data.total_credit || 0, true) + '</span>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            
            // Balance Status
            if (!data.is_balanced) {
                html += '<div class="row">';
                html += '<div class="col-md-12">';
                html += '<div class="alert alert-warning">';
                html += '<strong><i class="fa fa-exclamation-triangle"></i> @lang("accounting-reports::lang.not_balanced"):</strong> ';
                html += '@lang("accounting-reports::lang.debit"): ' + __currency_trans_from_en(data.total_debit || 0, true) + ' | ';
                html += '@lang("accounting-reports::lang.credit"): ' + __currency_trans_from_en(data.total_credit || 0, true);
                html += '</div>';
                html += '</div>';
                html += '</div>';
            }
            
            // Tally-style Day Book (Chronological Listing)
            html += '<div class="row">';
            html += '<div class="col-md-12">';
            html += '<div class="box box-primary">';
            html += '<div class="box-header with-border">';
            html += '<h3 class="box-title">@lang("accounting-reports::lang.day_book") - @lang("accounting-reports::lang.chronological_vouchers")</h3>';
            html += '</div>';
            html += '<div class="box-body">';
            html += '<div class="table-responsive">';
            html += '<table class="table table-bordered table-striped tally-daybook-table">';
            html += '<thead>';
            html += '<tr>';
            html += '<th width="10%">@lang("accounting-reports::lang.date")</th>';
            html += '<th width="12%">@lang("accounting-reports::lang.voucher_no")</th>';
            html += '<th width="12%">@lang("accounting-reports::lang.voucher_type")</th>';
            html += '<th width="25%">@lang("accounting-reports::lang.particulars")</th>';
            html += '<th width="12%">@lang("accounting-reports::lang.narration")</th>';
            html += '<th width="12%">@lang("accounting-reports::lang.debit")</th>';
            html += '<th width="12%">@lang("accounting-reports::lang.credit")</th>';
            html += '<th width="5%">@lang("accounting-reports::lang.location")</th>';
            html += '</tr>';
            html += '</thead>';
            html += '<tbody>';
            
            if (data.entries && data.entries.length > 0) {
                // Group entries by voucher_no for display (optional grouping)
                var entries_by_voucher = {};
                data.entries.forEach(function(entry) {
                    var voucher_key = entry.voucher_no || 'UNKNOWN';
                    if (!entries_by_voucher[voucher_key]) {
                        entries_by_voucher[voucher_key] = [];
                    }
                    entries_by_voucher[voucher_key].push(entry);
                });
                
                var voucher_keys = Object.keys(entries_by_voucher);
                voucher_keys.forEach(function(voucher_key, voucherIndex) {
                    var voucher_entries = entries_by_voucher[voucher_key];
                    var first_entry = voucher_entries[0];
                    
                    // Calculate voucher totals
                    var voucher_debit = 0;
                    var voucher_credit = 0;
                    voucher_entries.forEach(function(entry) {
                        voucher_debit += parseFloat(entry.debit || 0);
                        voucher_credit += parseFloat(entry.credit || 0);
                    });
                    
                    // Voucher header row
                    html += '<tr class="voucher-header-row">';
                    html += '<td>' + (first_entry.tx_datetime ? first_entry.tx_datetime.split(' ')[0] : '-') + '</td>';
                    html += '<td><strong>' + (first_entry.voucher_no || '-') + '</strong></td>';
                    html += '<td>' + (first_entry.voucher_type || first_entry.module || '-') + '</td>';
                    html += '<td colspan="2"><strong>@lang("accounting-reports::lang.voucher"): ' + (first_entry.voucher_no || '-') + '</strong>';
                    if (first_entry.narration) {
                        html += '<br><small>' + first_entry.narration + '</small>';
                    }
                    html += '</td>';
                    html += '<td class="text-right"><strong>' + __currency_trans_from_en(voucher_debit, true) + '</strong></td>';
                    html += '<td class="text-right"><strong>' + __currency_trans_from_en(voucher_credit, true) + '</strong></td>';
                    html += '<td>' + (first_entry.location_name || '-') + '</td>';
                    html += '</tr>';
                    
                    // Entry rows (Particulars)
                    voucher_entries.forEach(function(entry) {
                        html += '<tr class="voucher-line-row">';
                        html += '<td>' + (entry.tx_datetime ? entry.tx_datetime : '-') + '</td>';
                        html += '<td></td>'; // Empty voucher no column (already shown in header)
                        html += '<td></td>'; // Empty voucher type column
                        html += '<td class="particulars-cell">';
                        if (entry.payment_account_name && entry.payment_account_name !== '-') {
                            html += entry.payment_account_name;
                        }
                        if (entry.party_name && entry.party_name !== '-') {
                            html += '<br><small class="text-muted">@lang("accounting-reports::lang.party"): ' + entry.party_name + '</small>';
                        }
                        html += '</td>';
                        html += '<td>' + (entry.narration || '-') + '</td>';
                        html += '<td class="text-right">' + (parseFloat(entry.debit || 0) > 0 ? __currency_trans_from_en(entry.debit, true) : '-') + '</td>';
                        html += '<td class="text-right">' + (parseFloat(entry.credit || 0) > 0 ? __currency_trans_from_en(entry.credit, true) : '-') + '</td>';
                        html += '<td>' + (entry.location_name || '-') + '</td>';
                        html += '</tr>';
                    });
                    
                    // Add spacing row between vouchers (except for last voucher)
                    if (voucherIndex < voucher_keys.length - 1) {
                        html += '<tr class="voucher-spacer-row">';
                        html += '<td colspan="8">&nbsp;</td>';
                        html += '</tr>';
                    }
                });
            } else {
                html += '<tr>';
                html += '<td colspan="8" class="text-center">';
                html += '<div class="alert alert-info">@lang("accounting-reports::lang.no_vouchers_found")</div>';
                html += '</td>';
                html += '</tr>';
            }
            
            // Grand Total Row
            html += '<tr class="grand-total-row">';
            html += '<td colspan="5" class="text-right"><strong>@lang("accounting-reports::lang.grand_total"):</strong></td>';
            html += '<td class="text-right"><strong>' + __currency_trans_from_en(data.total_debit || 0, true) + '</strong></td>';
            html += '<td class="text-right"><strong>' + __currency_trans_from_en(data.total_credit || 0, true) + '</strong></td>';
            html += '<td></td>';
            html += '</tr>';
            
            html += '</tbody>';
            html += '</table>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            
            $('#db_data_div').html(html);
            
            // Apply currency conversion
            __currency_convert_recursively($('#db_data_div'));
        }

        // Load on date range change
        $('#db_date_range').on('apply.daterangepicker', function(ev, picker) {
            loadDayBook();
        });
    });
</script>
<style>
    .tally-daybook-table {
        font-size: 13px;
        margin-bottom: 0;
    }
    .tally-daybook-table th {
        background-color: #f8f9fa;
        font-weight: bold;
        border: 1px solid #dee2e6;
        text-align: center;
        padding: 10px 8px;
    }
    .tally-daybook-table td {
        border: 1px solid #dee2e6;
        padding: 8px;
        vertical-align: top;
    }
    .voucher-header-row {
        background-color: #e9ecef !important;
        font-weight: bold;
    }
    .voucher-header-row td {
        padding: 12px 8px;
        font-weight: bold;
    }
    .voucher-line-row td {
        padding-left: 30px;
    }
    .particulars-cell {
        padding-left: 40px !important;
    }
    .voucher-spacer-row {
        height: 5px;
    }
    .voucher-spacer-row td {
        padding: 0;
        border: none;
        background-color: transparent;
    }
    .grand-total-row {
        background-color: #f8f9fa !important;
        font-weight: bold;
        border-top: 2px solid #333;
    }
    .grand-total-row td {
        padding: 12px 8px;
        font-weight: bold;
    }
    @media print {
        .no-print { display: none !important; }
        .print_section { display: block !important; }
        .voucher-spacer-row { display: none !important; }
    }
</style>
@endsection
