@extends('layouts.app')

@section('title', __('accounting-reports::lang.cash_book'))

@section('content')
<section class="content-header">
    <h1>@lang('accounting-reports::lang.cash_book')</h1>
</section>

<section class="content">
    <div class="box box-solid">
        <div class="box-header with-border">
            <h3 class="box-title">@lang('accounting-reports::lang.cash_book') @lang('accounting-reports::lang.tally_style')</h3>
            <div class="box-tools">
                <button type="button" class="btn btn-sm btn-success" id="export_excel">
                    <i class="fa fa-file-excel-o"></i> @lang('daybook::lang.excel')
                </button>
                <button type="button" class="btn btn-sm btn-info" onclick="window.print()">
                    <i class="fa fa-print"></i> @lang('messages.print')
                </button>
            </div>
        </div>

        <div class="box-body">
            <div class="row">
                <!-- Date Range Picker -->
                <div class="col-md-4">
                    <div class="form-group">
                        {!! Form::label('date_range', __('report.date_range') . ':') !!}
                        <div class="input-group">
                            {!! Form::text('date_range', null, [
                                'class' => 'form-control', 
                                'id' => 'date_range', 
                                'readonly', 
                                'placeholder' => __('lang_v1.select_a_date_range')
                            ]) !!}
                            <span class="input-group-addon">
                                <i class="fa fa-calendar"></i>
                            </span>
                        </div>
                    </div>
                </div>
                <!-- Quick Date Filters -->
                <div class="col-md-8">
                    <div class="form-group">
                        {!! Form::label('quick_dates', __('daybook::lang.quick_filters') . ':') !!}
                        <div class="btn-group" style="display: flex; flex-wrap: wrap; gap: 5px;">
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="today">@lang('daybook::lang.today')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="yesterday">@lang('daybook::lang.yesterday')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="this_month">@lang('daybook::lang.this_month')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="last_month">@lang('daybook::lang.last_month')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="last_7_days">@lang('daybook::lang.last_7_days')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="last_30_days">@lang('daybook::lang.last_30_days')</button>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Hidden date fields for backend compatibility -->
            {!! Form::hidden('start_date', '', ['id' => 'start_date']) !!}
            {!! Form::hidden('end_date', '', ['id' => 'end_date']) !!}
            
            <div class="row">
                <div class="col-md-3">
                    <div class="form-group">
                        {!! Form::label('location_id', __('business.business_location') . ':') !!}
                        {!! Form::select('location_id', $locations, null, 
                            ['class' => 'form-control select2', 'id' => 'location_id', 'placeholder' => __('messages.all')]) 
                        !!}
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="form-group">
                        {!! Form::label('user_id', __('daybook::lang.user') . ':') !!}
                        {!! Form::select('user_id', $users, null, 
                            ['class' => 'form-control select2', 'id' => 'user_id', 'placeholder' => __('messages.all')]) 
                        !!}
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="form-group">
                        <br>
                        <button type="button" class="btn btn-primary" id="filter_btn">
                            <i class="fa fa-filter"></i> @lang('daybook::lang.filter')
                        </button>
                        <button type="button" class="btn btn-default" id="reset_btn">
                            <i class="fa fa-refresh"></i> @lang('daybook::lang.reset')
                        </button>
                    </div>
                </div>
            </div>

            <style>
                /* Excel-like minimal table styling */
                #cashbook_table {
                    border-collapse: collapse;
                    font-size: 12px;
                }
                #cashbook_table thead th {
                    background-color: #f2f2f2;
                    border: 1px solid #d0d0d0;
                    padding: 4px 6px;
                    font-weight: 600;
                    text-align: center;
                    font-size: 11px;
                }
                #cashbook_table tbody td {
                    padding: 3px 6px;
                    border: 1px solid #e0e0e0;
                    vertical-align: middle;
                }
                #cashbook_table tbody tr {
                    height: 22px;
                }
                #cashbook_table tbody tr:hover {
                    background-color: #f9f9f9;
                }
                #cashbook_table tfoot th {
                    padding: 4px 6px;
                    border: 1px solid #d0d0d0;
                    font-size: 11px;
                }
                #cashbook_table tfoot tr:last-child {
                    border-top: 2px solid #333;
                }
                #cashbook_table a {
                    color: #0066cc;
                    text-decoration: none;
                }
                #cashbook_table a:hover {
                    text-decoration: underline;
                }
                #cashbook_table tfoot tr {
                    background-color: #fafafa;
                }
            </style>
            
            <div class="table-responsive" style="margin-top: 15px;">
                <table class="table" id="cashbook_table" style="width: 100%; border-collapse: collapse;">
                    <thead>
                        <tr>
                            <th>@lang('daybook::lang.datetime')</th>
                            <th>@lang('daybook::lang.voucher_no')</th>
                            <th>@lang('daybook::lang.module')</th>
                            <th>@lang('daybook::lang.party')</th>
                            <th>@lang('daybook::lang.location')</th>
                            <th>@lang('daybook::lang.account')</th>
                            <th class="text-right">@lang('daybook::lang.debit')</th>
                            <th class="text-right">@lang('daybook::lang.credit')</th>
                            <th>@lang('daybook::lang.narration')</th>
                            <th>@lang('daybook::lang.user')</th>
                        </tr>
                    </thead>
                    <tbody></tbody>
                    <tfoot>
                        <tr>
                            <th colspan="6" class="text-right">@lang('daybook::lang.total'):</th>
                            <th class="text-right" id="total_debit">0.00</th>
                            <th class="text-right" id="total_credit">0.00</th>
                            <th colspan="2"></th>
                        </tr>
                        <tr>
                            <th colspan="6" class="text-right" style="font-weight: 600;">Opening Balance:</th>
                            <th class="text-right" id="opening_balance_debit" style="font-weight: 600;">0.00</th>
                            <th class="text-right" id="opening_balance_credit" style="font-weight: 600;">-</th>
                            <th colspan="2"></th>
                        </tr>
                        <tr>
                            <th colspan="6" class="text-right" style="font-weight: 600;">Current Total:</th>
                            <th class="text-right" id="current_total_debit" style="font-weight: 600;">0.00</th>
                            <th class="text-right" id="current_total_credit" style="font-weight: 600;">0.00</th>
                            <th colspan="2"></th>
                        </tr>
                        <tr style="border-top: 2px solid #333;">
                            <th colspan="6" class="text-right" style="font-weight: 700;">Closing Balance:</th>
                            <th class="text-right" id="closing_balance_debit" style="font-weight: 700; font-size: 12px;">0.00</th>
                            <th class="text-right" id="closing_balance_credit" style="font-weight: 700; font-size: 12px;">-</th>
                            <th colspan="2"></th>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
    </div>
</section>
@endsection

@section('javascript')
<script type="text/javascript">
$(document).ready(function() {
    // Declare cashbook_table variable first
    var cashbook_table;
    
    // Initialize date range picker
    if (typeof dateRangeSettings !== 'undefined' && $('#date_range').length && typeof moment !== 'undefined') {
        // Set default to today
        var today = moment();
        var todayFormatted = today.format(moment_date_format);
        $('#date_range').val(todayFormatted + ' ~ ' + todayFormatted);
        // Sync hidden fields for backend compatibility
        $('#start_date').val(todayFormatted);
        $('#end_date').val(todayFormatted);
        
        $('#date_range').daterangepicker(
            {
                ...dateRangeSettings,
                startDate: today,
                endDate: today,
                autoUpdateInput: true
            },
            function (start, end) {
                var startFormatted = start.format(moment_date_format);
                var endFormatted = end.format(moment_date_format);
                $('#date_range').val(startFormatted + ' ~ ' + endFormatted);
                // Update hidden fields for backend
                $('#start_date').val(startFormatted);
                $('#end_date').val(endFormatted);
                // Auto-reload when date range changes
                if (typeof cashbook_table !== 'undefined' && cashbook_table) {
                    cashbook_table.ajax.reload();
                }
            }
        );
        
        $('#date_range').on('cancel.daterangepicker', function(ev, picker) {
            $('#date_range').val(todayFormatted + ' ~ ' + todayFormatted);
            $('#start_date').val(todayFormatted);
            $('#end_date').val(todayFormatted);
            if (typeof cashbook_table !== 'undefined' && cashbook_table) {
                cashbook_table.ajax.reload();
            }
        });
    }
    
    // Quick date filter buttons
    $('.quick-date').click(function() {
        var range = $(this).data('range');
        var start, end;
        
        if (typeof moment === 'undefined') {
            alert('Moment.js is required for quick date filters');
            return;
        }
        
        switch(range) {
            case 'today':
                start = end = moment();
                break;
            case 'yesterday':
                start = end = moment().subtract(1, 'days');
                break;
            case 'this_month':
                start = moment().startOf('month');
                end = moment().endOf('month');
                break;
            case 'last_month':
                start = moment().subtract(1, 'month').startOf('month');
                end = moment().subtract(1, 'month').endOf('month');
                break;
            case 'last_7_days':
                start = moment().subtract(6, 'days');
                end = moment();
                break;
            case 'last_30_days':
                start = moment().subtract(29, 'days');
                end = moment();
                break;
            default:
                start = end = moment();
        }
        
        // Update date range picker
        if ($('#date_range').length && $('#date_range').data('daterangepicker')) {
            $('#date_range').data('daterangepicker').setStartDate(start);
            $('#date_range').data('daterangepicker').setEndDate(end);
            $('#date_range').val(start.format(moment_date_format) + ' ~ ' + end.format(moment_date_format));
        }
        
        // Update hidden date fields
        $('#start_date').val(start.format(moment_date_format));
        $('#end_date').val(end.format(moment_date_format));
        
        // Auto-reload data
        cashbook_table.ajax.reload();
    });
    
    // Initialize DataTable
    cashbook_table = $('#cashbook_table').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '{{ route("accounting-reports.cash-book.get-data") }}',
            type: 'GET',
            data: function(d) {
                d.start_date = $('#start_date').val();
                d.end_date = $('#end_date').val();
                d.location_id = $('#location_id').val();
                d.user_id = $('#user_id').val();
            },
            error: function(xhr, error, thrown) {
                console.error('Cash Book AJAX Error:', error);
                console.error('Response:', xhr.responseText);
                alert('Error loading cash book data. Please check console for details.');
            }
        },
        columns: [
            { data: 'datetime', name: 'datetime' },
            { data: 'voucher_no', name: 'voucher_no' },
            { data: 'module', name: 'module' },
            { data: 'party', name: 'party' },
            { data: 'location', name: 'location' },
            { data: 'account', name: 'account' },
            { data: 'debit', name: 'debit', className: 'text-right' },
            { data: 'credit', name: 'credit', className: 'text-right' },
            { data: 'narration', name: 'narration' },
            { data: 'user', name: 'user' }
        ],
        order: [[0, 'asc']],
        paging: false,
        pageLength: -1,
        lengthMenu: [[-1], ["All"]],
        footerCallback: function (row, data, start, end, display) {
            var api = this.api();
            
            // Get summary data from server response
            var json = api.ajax.json();
            var opening_balance = json.opening_balance || 0;
            var current_period_debit = json.current_period_debit || 0;
            var current_period_credit = json.current_period_credit || 0;
            var closing_balance = json.closing_balance || 0;
            
            // Calculate totals from visible rows
            var total_debit = 0;
            var total_credit = 0;

            // Sum debit column (index 6)
            api.column(6, {page: 'current'}).data().each(function (value, index) {
                if (value && value !== '-') {
                    var cleanVal = String(value).replace(/<[^>]*>/g, '').trim();
                    if (cleanVal !== '-' && cleanVal !== '') {
                        var numStr = cleanVal.replace(/[^\d.-]/g, '');
                        var num = parseFloat(numStr) || 0;
                        total_debit += Math.abs(num);
                    }
                }
            });

            // Sum credit column (index 7)
            api.column(7, {page: 'current'}).data().each(function (value, index) {
                if (value && value !== '-') {
                    var cleanVal = String(value).replace(/<[^>]*>/g, '').trim();
                    if (cleanVal !== '-' && cleanVal !== '') {
                        var numStr = cleanVal.replace(/[^\d.-]/g, '');
                        var num = parseFloat(numStr) || 0;
                        total_credit += Math.abs(num);
                    }
                }
            });

            // Update footer with formatted totals
            $('#total_debit').html(__currency_trans_from_en(total_debit, true));
            $('#total_credit').html(__currency_trans_from_en(total_credit, true));
            
            // Update Opening Balance - ALWAYS shows in Debit side
            $('#opening_balance_debit').html(__currency_trans_from_en(Math.abs(opening_balance), true));
            $('#opening_balance_credit').html('-');
            
            // Update Current Total
            var current_debit = current_period_debit || 0;
            var current_credit = current_period_credit || 0;
            
            if (Math.abs(opening_balance) < 0.01) {
                current_debit = total_debit;
                current_credit = total_credit;
            }
            
            $('#current_total_debit').html(__currency_trans_from_en(current_debit, true));
            $('#current_total_credit').html(__currency_trans_from_en(current_credit, true));
            
            // Update Closing Balance - ALWAYS shows in Debit side
            var closing_balance_formatted = closing_balance || 0;
            $('#closing_balance_debit').html(__currency_trans_from_en(Math.abs(closing_balance_formatted), true));
            $('#closing_balance_credit').html('-');
        },
        dom: 'Bfrtip',
        buttons: [
            {
                extend: 'excel',
                text: '<i class="fa fa-file-excel-o"></i> Excel',
                className: 'btn-success'
            },
            {
                extend: 'print',
                text: '<i class="fa fa-print"></i> Print',
                className: 'btn-info'
            }
        ]
    });

    // Filter button
    $('#filter_btn').click(function() {
        var startVal = $('#start_date').val().trim();
        var endVal = $('#end_date').val().trim();
        
        if (!startVal && !endVal) {
            // Set to current date if both empty
            var today = moment();
            var todayFormatted = today.format(moment_date_format);
            $('#start_date').val(todayFormatted);
            $('#end_date').val(todayFormatted);
            if ($('#date_range').length && $('#date_range').data('daterangepicker')) {
                $('#date_range').data('daterangepicker').setStartDate(today);
                $('#date_range').data('daterangepicker').setEndDate(today);
                $('#date_range').val(todayFormatted + ' ~ ' + todayFormatted);
            }
        }
        cashbook_table.ajax.reload();
    });

    // Reset button
    $('#reset_btn').click(function() {
        var today = moment();
        var todayFormatted = today.format(moment_date_format);
        $('#start_date').val(todayFormatted);
        $('#end_date').val(todayFormatted);
        if ($('#date_range').length && $('#date_range').data('daterangepicker')) {
            $('#date_range').data('daterangepicker').setStartDate(today);
            $('#date_range').data('daterangepicker').setEndDate(today);
            $('#date_range').val(todayFormatted + ' ~ ' + todayFormatted);
        }
        $('#location_id, #user_id').val('').trigger('change');
        cashbook_table.ajax.reload();
    });

    // Auto-reload on filter change
    $('#location_id, #user_id').change(function() {
        cashbook_table.ajax.reload();
    });

    // Export buttons
    $('#export_excel').click(function() {
        cashbook_table.button('.buttons-excel').trigger();
    });
});
</script>
@endsection
