# Menu Setup Guide - Accounting Reports Module

## Issue: Menu Not Showing After Installation

If the Accounting Reports menu is not showing in the admin sidebar after installation, follow these steps:

## Solution 1: Re-run Installation (Recommended)

1. Go to **Settings → Manage Modules**
2. Find "Accounting Reports" module
3. Click "Update" or re-run the install process
4. This will:
   - Set the module version in the system table
   - Create all required permissions
   - Refresh the menu

## Solution 2: Manual Fix (If Re-install Doesn't Work)

### Step 1: Set Module Version

Run this SQL query or use tinker:

```sql
INSERT INTO system (key, value, created_at, updated_at) 
VALUES ('accountingreports_version', '1.0.0', NOW(), NOW())
ON DUPLICATE KEY UPDATE value = '1.0.0';
```

Or via PHP Tinker:
```php
php artisan tinker
\App\System::firstOrCreate(
    ['key' => 'accountingreports_version'],
    ['value' => '1.0.0']
);
```

### Step 2: Create Permissions

The permissions should be created automatically during installation, but if needed, you can create them manually:

Go to **Settings → Roles & Permissions** and the permissions should appear when editing a role. If not, they will be created automatically when you assign them.

Or create via tinker:
```php
php artisan tinker
$permissions = [
    'accounting.view_all',
    'accounting.view_trial_balance',
    'accounting.view_balance_sheet',
    'accounting.view_pl',
    'accounting.view_cashflow',
    'accounting.view_ar',
    'accounting.view_ap',
    'accounting.view_daybook',
    'accounting.view_cashbook',
    'accounting.view_bankbook',
    'accounting.export_reports',
    'accounting.print_multi_account',
];

foreach ($permissions as $perm) {
    \Spatie\Permission\Models\Permission::firstOrCreate(
        ['name' => $perm, 'guard_name' => 'web']
    );
}
```

### Step 3: Assign Permissions to Roles

1. Go to **Settings → Roles & Permissions**
2. Edit your role (e.g., Admin)
3. Check the Accounting Reports permissions you want to grant
4. Save

### Step 4: Clear Cache

```bash
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

### Step 5: Refresh Browser

Hard refresh your browser (Ctrl+F5 or Cmd+Shift+R) to clear browser cache.

## Verification

To verify everything is set up correctly:

1. Check System table:
   ```sql
   SELECT * FROM system WHERE key = 'accountingreports_version';
   ```
   Should return a row with value '1.0.0'

2. Check Permissions table:
   ```sql
   SELECT * FROM permissions WHERE name LIKE 'accounting.%';
   ```
   Should return 12 permission rows

3. Check if DataController exists:
   The file should exist at:
   `Modules/AccountingReports/Http/Controllers/DataController.php`

4. Check if modifyAdminMenu method exists:
   The DataController should have a `modifyAdminMenu()` method

## Troubleshooting

### Menu Still Not Showing?

1. **Check if module is properly loaded:**
   ```php
   php artisan tinker
   Module::has('AccountingReports'); // Should return true
   ```

2. **Check module installation status:**
   ```php
   \App\System::getProperty('accountingreports_version'); // Should return '1.0.0'
   ```

3. **Check user permissions:**
   Make sure your user role has at least one accounting permission assigned:
   - `accounting.view_all` OR
   - `accounting.view_trial_balance` OR
   - Any other accounting permission

4. **Check browser console for JavaScript errors:**
   Open browser DevTools (F12) and check Console tab for errors

5. **Check Laravel logs:**
   ```bash
   tail -f storage/logs/laravel.log
   ```
   Look for any menu-related errors

### Common Issues

**Issue:** "Call to undefined method Menu::instance()"
- **Solution:** Make sure you're logged in as a user with permissions

**Issue:** Menu shows but empty
- **Solution:** Assign at least one accounting permission to your role

**Issue:** Menu shows but all items are disabled
- **Solution:** Your role doesn't have the required permissions for those menu items

## Expected Result

After completing these steps, you should see:
- **"Accounting Reports"** menu item in the sidebar
- A dropdown with sub-menu items:
  - Reports (Dashboard)
  - Trial Balance
  - Balance Sheet
  - Profit & Loss
  - Cash Flow
  - Receivables
  - Payables
  - Day Book
  - Cash Book
  - Bank Book
  - Ratio Analysis
  - Statistics
  - Batch Print
  - Integrity Tools

The menu items you see will depend on the permissions assigned to your role.

## Still Having Issues?

If the menu still doesn't appear after following all steps:

1. Verify the module structure is correct
2. Check that all files were uploaded correctly
3. Verify the service provider is registered in `module.json`
4. Check for PHP errors in Laravel logs
5. Ensure you're using the correct UltimatePOS version (Laravel 9.x/10.x)

