<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Transaction;
use App\Contact;
use App\BusinessLocation;
use App\Utils\Util;
use App\Utils\TransactionUtil;
use DB;
use Yajra\DataTables\Facades\DataTables;

class ReceivablesController extends Controller
{
    protected $commonUtil;
    protected $transactionUtil;

    public function __construct(Util $commonUtil, TransactionUtil $transactionUtil)
    {
        $this->commonUtil = $commonUtil;
        $this->transactionUtil = $transactionUtil;
    }

    public function index()
    {
        if (!auth()->user()->can('accounting.view_ar')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::receivables.index', compact('locations'));
    }

    /**
     * Get receivables data in Tally-style format
     * Groups by customer (Sundry Debtors) with invoice details
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_ar')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $location_id = $request->input('location_id');
        
        // Handle date input - convert from user format to Y-m-d if needed (matching Trial Balance pattern)
        $as_on_date = null;
        if ($request->has('as_on_date') && !empty($request->input('as_on_date'))) {
            try {
                $as_on_date = $this->transactionUtil->uf_date($request->input('as_on_date'));
                if (!$as_on_date) {
                    // Fallback: try parsing as Y-m-d format
                    $as_on_date = \Carbon\Carbon::createFromFormat('Y-m-d', $request->input('as_on_date'))->format('Y-m-d');
                }
            } catch (\Exception $e) {
                // If date parsing fails, use current date
                $as_on_date = \Carbon\Carbon::now()->format('Y-m-d');
            }
        } else {
            $as_on_date = \Carbon\Carbon::now()->format('Y-m-d');
        }
        
        $permitted_locations = auth()->user()->permitted_locations();

        try {
            // Get all customers with outstanding receivables
            $customers = Contact::where('contacts.business_id', $business_id)
                ->onlyCustomers()
                ->select('contacts.*');

            // Get receivables grouped by customer (Tally style)
            $receivables_data = $this->getReceivablesData($business_id, $as_on_date, $location_id, $permitted_locations);

            return response()->json([
                'success' => true,
                'customers' => $receivables_data['customers'],
                'summary' => $receivables_data['summary'],
                'as_on_date' => $as_on_date,
                'total_receivables' => $receivables_data['total_receivables'],
                'total_current' => $receivables_data['total_current'],
                'total_b0_7' => $receivables_data['total_b0_7'],
                'total_b8_30' => $receivables_data['total_b8_30'],
                'total_b31_60' => $receivables_data['total_b31_60'],
                'total_b61_90' => $receivables_data['total_b61_90'],
                'total_b90_plus' => $receivables_data['total_b90_plus'],
                // Month-wise totals
                'total_current_month' => $receivables_data['total_current_month'],
                'total_month_1' => $receivables_data['total_month_1'],
                'total_month_2' => $receivables_data['total_month_2'],
                'total_month_3' => $receivables_data['total_month_3'],
                'total_month_4_6' => $receivables_data['total_month_4_6'],
                'total_month_7_12' => $receivables_data['total_month_7_12'],
                'total_month_12_plus' => $receivables_data['total_month_12_plus'],
            ]);

        } catch (\Exception $e) {
            \Log::error('Receivables getData Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'success' => false,
                'error' => 'Error generating Receivables report: ' . $e->getMessage(),
                'customers' => [],
                'summary' => []
            ], 500);
        }
    }

    /**
     * Get receivables data in Tally format (grouped by customer)
     */
    protected function getReceivablesData($business_id, $as_on_date, $location_id, $permitted_locations)
    {
        // Initialize totals for aging buckets (days)
        $total_receivables = 0;
        $total_current = 0;
        $total_b0_7 = 0;
        $total_b8_30 = 0;
        $total_b31_60 = 0;
        $total_b61_90 = 0;
        $total_b90_plus = 0;

        // Initialize totals for month-wise aging buckets
        $total_current_month = 0;
        $total_month_1 = 0;
        $total_month_2 = 0;
        $total_month_3 = 0;
        $total_month_4_6 = 0;
        $total_month_7_12 = 0;
        $total_month_12_plus = 0;

        $customers_data = [];
        
        // Get all sell transactions with outstanding balance (due or partial)
        $transactions_query = Transaction::where('transactions.business_id', $business_id)
            ->where('transactions.type', 'sell')
            ->where('transactions.status', 'final')
            ->whereIn('transactions.payment_status', ['due', 'partial'])
            ->whereDate('transactions.transaction_date', '<=', $as_on_date)
            ->join('contacts', 'transactions.contact_id', '=', 'contacts.id')
            ->select(
                'transactions.*',
                'contacts.name as contact_name',
                'contacts.supplier_business_name',
                'contacts.id as contact_id',
                'contacts.contact_id as contact_code'
            );

        if ($location_id) {
            $transactions_query->where('transactions.location_id', $location_id);
        }

        if ($permitted_locations != 'all') {
            $transactions_query->whereIn('transactions.location_id', $permitted_locations);
        }

        $transactions = $transactions_query->get();

        // Group by customer
        $customers = [];
        foreach ($transactions as $transaction) {
            $contact_id = $transaction->contact_id;
            $contact_name = $transaction->contact_name ?: $transaction->supplier_business_name;

            // Calculate outstanding balance for this transaction
            $total_paid = DB::table('transaction_payments')
                ->where('transaction_id', $transaction->id)
                ->select(DB::raw('SUM(IF(is_return = 1, -1*amount, amount)) as total'))
                ->value('total') ?? 0;

            $outstanding = $transaction->final_total - $total_paid;

            // Skip if fully paid or negative
            if ($outstanding <= 0.01) {
                continue;
            }

            // Calculate due date
            $due_date = $this->calculateDueDate($transaction);

            // Calculate aging
            $aging = $this->calculateAging($due_date, $as_on_date);

            // Initialize customer if not exists
            if (!isset($customers[$contact_id])) {
                $customers[$contact_id] = [
                    'contact_id' => $contact_id,
                    'contact_name' => $contact_name,
                    'contact_code' => $transaction->contact_code,
                    'total_outstanding' => 0,
                    'invoices' => [],
                    'aging' => [
                        'current' => 0,
                        'b0_7' => 0,
                        'b8_30' => 0,
                        'b31_60' => 0,
                        'b61_90' => 0,
                        'b90_plus' => 0,
                    ]
                ];
            }

            // Add invoice to customer
            $customers[$contact_id]['invoices'][] = [
                'invoice_no' => $transaction->invoice_no,
                'transaction_date' => $transaction->transaction_date,
                'due_date' => $due_date,
                'invoice_amount' => $transaction->final_total,
                'paid_amount' => $total_paid,
                'outstanding' => $outstanding,
                'days_past_due' => $aging['days_past_due'],
                'aging_bucket' => $aging['bucket'],
                'months_past_due' => $aging['months_past_due'],
                'month_bucket' => $aging['month_bucket'],
                'payment_status' => $transaction->payment_status,
            ];

            // Update customer totals
            $customers[$contact_id]['total_outstanding'] += $outstanding;
            $customers[$contact_id]['aging'][$aging['bucket']] += $outstanding;

            // Update global totals (days)
            $total_receivables += $outstanding;
            $total_current += $aging['bucket'] == 'current' ? $outstanding : 0;
            $total_b0_7 += $aging['bucket'] == 'b0_7' ? $outstanding : 0;
            $total_b8_30 += $aging['bucket'] == 'b8_30' ? $outstanding : 0;
            $total_b31_60 += $aging['bucket'] == 'b31_60' ? $outstanding : 0;
            $total_b61_90 += $aging['bucket'] == 'b61_90' ? $outstanding : 0;
            $total_b90_plus += $aging['bucket'] == 'b90_plus' ? $outstanding : 0;

            // Update global totals (months)
            $total_current_month += $aging['month_bucket'] == 'current_month' ? $outstanding : 0;
            $total_month_1 += $aging['month_bucket'] == 'month_1' ? $outstanding : 0;
            $total_month_2 += $aging['month_bucket'] == 'month_2' ? $outstanding : 0;
            $total_month_3 += $aging['month_bucket'] == 'month_3' ? $outstanding : 0;
            $total_month_4_6 += $aging['month_bucket'] == 'month_4_6' ? $outstanding : 0;
            $total_month_7_12 += $aging['month_bucket'] == 'month_7_12' ? $outstanding : 0;
            $total_month_12_plus += $aging['month_bucket'] == 'month_12_plus' ? $outstanding : 0;
        }

        // Convert to array format for JSON
        $customers_array = [];
        foreach ($customers as $customer) {
            $customers_array[] = $customer;
        }

        return [
            'customers' => $customers_array,
            'summary' => [
                'total_customers' => count($customers_array),
                'total_invoices' => array_sum(array_map(function($c) { return count($c['invoices']); }, $customers_array)),
            ],
            'total_receivables' => $total_receivables,
            'total_current' => $total_current,
            'total_b0_7' => $total_b0_7,
            'total_b8_30' => $total_b8_30,
            'total_b31_60' => $total_b31_60,
            'total_b61_90' => $total_b61_90,
            'total_b90_plus' => $total_b90_plus,
            // Month-wise totals
            'total_current_month' => $total_current_month,
            'total_month_1' => $total_month_1,
            'total_month_2' => $total_month_2,
            'total_month_3' => $total_month_3,
            'total_month_4_6' => $total_month_4_6,
            'total_month_7_12' => $total_month_7_12,
            'total_month_12_plus' => $total_month_12_plus,
        ];
    }

    /**
     * Calculate due date from transaction
     */
    protected function calculateDueDate($transaction)
    {
        $transaction_date = \Carbon\Carbon::parse($transaction->transaction_date);
        
        if (!empty($transaction->pay_term_type) && !empty($transaction->pay_term_number)) {
            if ($transaction->pay_term_type == 'days') {
                $due_date = $transaction_date->copy()->addDays($transaction->pay_term_number);
            } else {
                $due_date = $transaction_date->copy()->addMonths($transaction->pay_term_number);
            }
        } else {
            // Default to transaction date if no payment terms
            $due_date = $transaction_date->copy();
        }

        return $due_date->format('Y-m-d');
    }

    /**
     * Calculate aging buckets (Tally style)
     */
    protected function calculateAging($due_date, $as_on_date)
    {
        $due = \Carbon\Carbon::parse($due_date);
        $as_on = \Carbon\Carbon::parse($as_on_date);
        
        // Calculate days past due (positive if past due, negative if not yet due)
        $days_past_due = $as_on->diffInDays($due, false);
        
        // If negative (not yet due), set to 0 and mark as current
        if ($days_past_due < 0) {
            $days_past_due = 0;
            $bucket = 'current';
        } elseif ($days_past_due == 0) {
            $bucket = 'current';
        } elseif ($days_past_due <= 7) {
            $bucket = 'b0_7';
        } elseif ($days_past_due <= 30) {
            $bucket = 'b8_30';
        } elseif ($days_past_due <= 60) {
            $bucket = 'b31_60';
        } elseif ($days_past_due <= 90) {
            $bucket = 'b61_90';
        } else {
            $bucket = 'b90_plus';
        }

        // Calculate month-wise aging
        $months_past_due = $as_on->diffInMonths($due, false);
        if ($months_past_due < 0) {
            $months_past_due = 0;
            $month_bucket = 'current_month';
        } elseif ($months_past_due == 0) {
            $month_bucket = 'current_month';
        } elseif ($months_past_due == 1) {
            $month_bucket = 'month_1';
        } elseif ($months_past_due == 2) {
            $month_bucket = 'month_2';
        } elseif ($months_past_due == 3) {
            $month_bucket = 'month_3';
        } elseif ($months_past_due <= 6) {
            $month_bucket = 'month_4_6';
        } elseif ($months_past_due <= 12) {
            $month_bucket = 'month_7_12';
        } else {
            $month_bucket = 'month_12_plus';
        }

        return [
            'days_past_due' => $days_past_due,
            'bucket' => $bucket,
            'months_past_due' => $months_past_due,
            'month_bucket' => $month_bucket
        ];
    }

    /**
     * Aging report - shows aging buckets summary
     */
    public function aging(Request $request)
    {
        if (!auth()->user()->can('accounting.view_ar')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $location_id = $request->input('location_id');
        
        // Handle date input - convert from user format to Y-m-d if needed (matching Trial Balance pattern)
        $as_on_date = null;
        if ($request->has('as_on_date') && !empty($request->input('as_on_date'))) {
            try {
                $as_on_date = $this->transactionUtil->uf_date($request->input('as_on_date'));
                if (!$as_on_date) {
                    // Fallback: try parsing as Y-m-d format
                    $as_on_date = \Carbon\Carbon::createFromFormat('Y-m-d', $request->input('as_on_date'))->format('Y-m-d');
                }
            } catch (\Exception $e) {
                // If date parsing fails, use current date
                $as_on_date = \Carbon\Carbon::now()->format('Y-m-d');
            }
        } else {
            $as_on_date = \Carbon\Carbon::now()->format('Y-m-d');
        }
        
        $permitted_locations = auth()->user()->permitted_locations();

        try {
            $receivables_data = $this->getReceivablesData($business_id, $as_on_date, $location_id, $permitted_locations);

            return response()->json([
                'success' => true,
                'aging_summary' => [
                    'current' => $receivables_data['total_current'],
                    'b0_7' => $receivables_data['total_b0_7'],
                    'b8_30' => $receivables_data['total_b8_30'],
                    'b31_60' => $receivables_data['total_b31_60'],
                    'b61_90' => $receivables_data['total_b61_90'],
                    'b90_plus' => $receivables_data['total_b90_plus'],
                    'total' => $receivables_data['total_receivables'],
                ],
                'as_on_date' => $as_on_date,
            ]);

        } catch (\Exception $e) {
            \Log::error('Receivables Aging Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'error' => 'Error generating Aging report: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Customer statement - detailed transaction list for a specific customer
     */
    public function statement($contact_id)
    {
        if (!auth()->user()->can('accounting.view_ar')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $contact = Contact::where('business_id', $business_id)
            ->where('id', $contact_id)
            ->onlyCustomers()
            ->firstOrFail();

        return view('accounting-reports::receivables.statement', compact('contact'));
    }

    /**
     * Export receivables report
     */
    public function export(Request $request)
    {
        if (!auth()->user()->can('accounting.view_ar')) {
            abort(403, 'Unauthorized action.');
        }

        // TODO: Implement export functionality
        return redirect()->back();
    }
}


