<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\BusinessLocation;
use App\Utils\TransactionUtil;
use App\Utils\BusinessUtil;
use App\Transaction;
use App\PurchaseLine;
use App\AccountTransaction;
use Modules\AccountingReports\Entities\DirectExpense;
use Modules\AccountingReports\Entities\DirectIncome;
use DB;

class ProfitLossController extends Controller
{
    protected $transactionUtil;
    protected $businessUtil;

    public function __construct(TransactionUtil $transactionUtil, BusinessUtil $businessUtil)
    {
        $this->transactionUtil = $transactionUtil;
        $this->businessUtil = $businessUtil;
    }

    public function index()
    {
        if (!auth()->user()->can('accounting.view_pl')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::profit-loss.index', compact('locations'));
    }

    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_pl')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        
        $fy = $this->businessUtil->getCurrentFinancialYear($business_id);
        
        $location_id = !empty($request->input('location_id')) ? $request->input('location_id') : null;
        $start_date = !empty($request->input('start_date')) ? $request->input('start_date') : $fy['start'];
        $end_date = !empty($request->input('end_date')) ? $request->input('end_date') : $fy['end'];
        
        $user_id = $request->input('user_id') ?? null;
        
        $permitted_locations = auth()->user()->permitted_locations();
        
        try {
            // Clear any cache that might be holding old data after deletions
            \Cache::forget('profit_loss_' . $business_id . '_' . $start_date . '_' . $end_date . '_' . $location_id);
            
            // Get existing P&L data - THIS IS THE REAL DATA FROM ULTIMATEPOS
            // getProfitLossDetails already handles deleted records through TransactionUtil methods
            $data = $this->transactionUtil->getProfitLossDetails($business_id, $location_id, $start_date, $end_date, $user_id, $permitted_locations);
            
            // Ensure we have data (even if empty)
            if (!is_array($data)) {
                $data = [];
            }
            
            // Format as Tally-style P&L
            $pl_data = $this->formatTallyStylePL($data, $start_date, $end_date, $business_id, $location_id, $permitted_locations);
            
            // Add debug info if needed (can be removed later)
            // $pl_data['debug'] = [
            //     'sales' => $data['total_sell'] ?? 0,
            //     'purchase' => $data['total_purchase'] ?? 0,
            //     'opening_stock' => $data['opening_stock'] ?? 0,
            //     'closing_stock' => $data['closing_stock'] ?? 0
            // ];
            
            return response()->json($pl_data);
        } catch (\Exception $e) {
            \Log::error('ProfitLoss getData Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'error' => 'Error generating Profit & Loss report: ' . $e->getMessage(),
                'left_side' => [],
                'right_side' => [],
                'subtotal_left' => 0,
                'subtotal_right' => 0,
                'gross_profit_c_o' => 0,
                'net_profit' => 0,
                'final_total_left' => 0,
                'final_total_right' => 0
            ], 500);
        }
    }

    protected function formatTallyStylePL($data, $start_date, $end_date, $business_id, $location_id, $permitted_locations)
    {
        // Ensure $data is an array and has the expected keys
        if (!is_array($data)) {
            $data = [];
        }
        
        // Get actual account transactions for expenses and incomes (optional - fallback to basic data if empty)
        
        // Get stock details dynamically (with error handling)
        try {
            $opening_stock_details = $this->getStockDetails($business_id, $start_date, $location_id, true, $permitted_locations);
        } catch (\Exception $e) {
            \Log::error('Error getting opening stock details: ' . $e->getMessage());
            $opening_stock_details = [];
        }
        
        try {
            $closing_stock_details = $this->getStockDetails($business_id, $end_date, $location_id, false, $permitted_locations);
        } catch (\Exception $e) {
            \Log::error('Error getting closing stock details: ' . $e->getMessage());
            $closing_stock_details = [];
        }
        
        // LEFT SIDE (DEBIT/EXPENSES) - Tally Format
        $left_side = [];
        
        // 1. Opening Stock (with detailed items)
        $opening_stock_items = [];
        $opening_stock_total = $data['opening_stock'] ?? 0;
        
        if (!empty($opening_stock_details) && is_array($opening_stock_details)) {
            foreach ($opening_stock_details as $stock_item) {
                if (isset($stock_item['name']) && isset($stock_item['stock_value'])) {
                    $opening_stock_items[] = [
                        'label' => $stock_item['name'],
                        'value' => $stock_item['stock_value']
                    ];
                }
            }
        }
        
        // If no detailed items, just show total
        if (empty($opening_stock_items) && $opening_stock_total > 0) {
            $opening_stock_items[] = ['label' => 'Opening Stock', 'value' => $opening_stock_total];
        }
        
        // Always show opening stock even if 0
        $left_side[] = [
            'label' => 'Opening Stock',
            'value' => $opening_stock_total,
            'items' => !empty($opening_stock_items) ? $opening_stock_items : [['label' => 'Opening Stock', 'value' => $opening_stock_total]]
        ];
        
        // 2. Purchase Accounts (from actual transactions - always show this section)
        $purchase_items = [];
        $purchase_total = 0;
        
        // Purchase transactions data is already in $data from getProfitLossDetails
        // This is the REAL data from UltimatePOS transactions
        
        $purchase_amount = (float) ($data['total_purchase'] ?? 0);
        if ($purchase_amount > 0.01 || $purchase_amount < -0.01) {
            $purchase_items[] = ['label' => 'Purchase - GST/IGST', 'value' => $purchase_amount];
            $purchase_total += $purchase_amount;
        }
        
        $purchase_shipping = (float) ($data['total_purchase_shipping_charge'] ?? 0);
        if (abs($purchase_shipping) > 0.01) {
            $purchase_items[] = ['label' => 'Delivery Charges on Purchase', 'value' => $purchase_shipping];
            $purchase_total += $purchase_shipping;
        }
        
        $purchase_additional = (float) ($data['total_purchase_additional_expense'] ?? 0);
        if (abs($purchase_additional) > 0.01) {
            $purchase_items[] = ['label' => 'Insurance Charges on Purchase', 'value' => $purchase_additional];
            $purchase_total += $purchase_additional;
        }
        
        $purchase_discount = (float) ($data['total_purchase_discount'] ?? 0);
        if (abs($purchase_discount) > 0.01) {
            $purchase_items[] = ['label' => 'Discount on Purchase', 'value' => -1 * abs($purchase_discount)]; // Negative
            $purchase_total -= abs($purchase_discount);
        }
        
        // Additional purchase related items from transactions
        $transfer_shipping = (float) ($data['total_transfer_shipping_charges'] ?? 0);
        if (abs($transfer_shipping) > 0.01) {
            $purchase_items[] = ['label' => 'Freight & Cartage - Outside', 'value' => $transfer_shipping];
            $purchase_total += $transfer_shipping;
        }
        
        // Always show purchase section (even if empty)
        $left_side[] = [
            'label' => 'Purchase Accounts',
            'value' => $purchase_total,
            'items' => !empty($purchase_items) ? $purchase_items : [['label' => 'No Purchases', 'value' => 0]]
        ];
        
        // 3. Direct Expenses (from direct expenses table + account transactions)
        $direct_expenses = [];
        $direct_total = 0;
        
        // Get direct expenses from ar_direct_expenses table
        try {
            $direct_expense_query = DirectExpense::where('business_id', $business_id)
                ->whereBetween('expense_date', [$start_date, $end_date])
                ->whereNull('deleted_at');
            
            if ($location_id) {
                $direct_expense_query->where('location_id', $location_id);
            }
            
            if ($permitted_locations != 'all') {
                $direct_expense_query->whereIn('location_id', $permitted_locations);
            }
            
            $direct_expense_records = $direct_expense_query->get();
            
            foreach ($direct_expense_records as $expense) {
                $amount = (float) $expense->amount;
                if (abs($amount) > 0.01) {
                    $expense_name = $expense->name;
                    if (isset($direct_expenses[$expense_name])) {
                        $direct_expenses[$expense_name] += $amount;
                    } else {
                        $direct_expenses[$expense_name] = $amount;
                    }
                    $direct_total += $amount;
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting direct expenses from table: ' . $e->getMessage());
        }
        
        // Also get direct expenses from account transactions
        // Direct expenses typically include: carriage, packaging, handling, assembly, etc.
        try {
            $direct_expense_accounts = $this->getDirectExpenseAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            
            foreach ($direct_expense_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    // Merge with existing direct expenses from table
                    if (isset($direct_expenses[$account_name])) {
                        $direct_expenses[$account_name] += abs($amount);
                    } else {
                        $direct_expenses[$account_name] = abs($amount);
                    }
                    $direct_total += abs($amount);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting direct expenses from accounts: ' . $e->getMessage());
        }
        
        // Convert to array format for display - all names are dynamic from database
        $direct_expenses_display = [];
        foreach ($direct_expenses as $name => $amount) {
            // All expense names come dynamically from database (from ar_direct_expenses table or account names)
            $direct_expenses_display[] = ['label' => $name, 'value' => $amount];
        }
        $direct_expenses = $direct_expenses_display;
        
        // Add purchase shipping/expenses - use dynamic account name if available
        if (!empty($data['total_purchase_shipping_charge'])) {
            // Try to get account name dynamically, fallback to generic label if needed
            $shipping_label = 'Freight & Cartage'; // Keep as fallback
            $direct_expenses[] = ['label' => $shipping_label, 'value' => $data['total_purchase_shipping_charge']];
            $direct_total += $data['total_purchase_shipping_charge'];
        }
        
        if (!empty($data['total_purchase_additional_expense'])) {
            // Try to get account name dynamically, fallback to generic label if needed
            $handling_label = 'Handling & Other Charges'; // Keep as fallback
            $direct_expenses[] = ['label' => $handling_label, 'value' => $data['total_purchase_additional_expense']];
            $direct_total += $data['total_purchase_additional_expense'];
        }
        
        // Add stock adjustment as other trading expenses - this is system generated, keep as is
        if (!empty($data['total_adjustment'])) {
            $direct_expenses[] = ['label' => 'Other Trading Expenses', 'value' => $data['total_adjustment']];
            $direct_total += $data['total_adjustment'];
        }
        
        // Add module direct expenses - these come dynamically from modules
        if (!empty($data['left_side_module_data'])) {
            foreach ($data['left_side_module_data'] as $module_data) {
                if (!empty($module_data['add_to_net_profit']) && $module_data['value'] > 0) {
                    // Check if it's direct expense
                    $is_direct = false;
                    $direct_keywords = ['packaging', 'carriage', 'handling', 'assemble', 'freight', 'transport', 'delivery', 'loading'];
                    foreach ($direct_keywords as $keyword) {
                        if (stripos($module_data['label'], $keyword) !== false) {
                            $is_direct = true;
                            break;
                        }
                    }
                    
                    if ($is_direct) {
                        // Module data labels are already dynamic from modules
                        $direct_expenses[] = ['label' => $module_data['label'], 'value' => $module_data['value']];
                        $direct_total += $module_data['value'];
                    }
                }
            }
        }
        
        // Keep "Direct Expenses" section grouping with all dynamic expense names
        if (!empty($direct_expenses)) {
            $left_side[] = [
                'label' => 'Direct Expenses',
                'value' => $direct_total,
                'items' => $direct_expenses
            ];
        }
        
        // Calculate Direct Incomes early (needed for Gross Profit calculation)
        $direct_income_total_precalc = 0;
        try {
            $direct_income_query = DirectIncome::where('business_id', $business_id)
                ->whereBetween('income_date', [$start_date, $end_date])
                ->whereNull('deleted_at');
            
            if ($location_id) {
                $direct_income_query->where('location_id', $location_id);
            }
            
            if ($permitted_locations != 'all') {
                $direct_income_query->whereIn('location_id', $permitted_locations);
            }
            
            $direct_income_total_precalc = (float) $direct_income_query->sum('amount');
            
            // Also get from account transactions
            $direct_income_accounts = $this->getDirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            foreach ($direct_income_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    $direct_income_total_precalc += abs($amount);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error calculating direct incomes total: ' . $e->getMessage());
        }
        
        // Calculate Gross Profit (Tally Trading Account format)
        // TRADING ACCOUNT:
        // Debit Side: Opening Stock + Purchases + Direct Expenses
        // Credit Side: Sales + Direct Incomes + Closing Stock
        // Gross Profit = Credit Side - Debit Side
        
        $opening_stock = (float) ($data['opening_stock'] ?? 0);
        $purchases = (float) $purchase_total;
        $direct_exp = (float) $direct_total;
        $closing_stock = (float) ($data['closing_stock'] ?? 0);
        
        // Sales side total (for Trading Account calculation)
        // IMPORTANT: total_sell from UltimatePOS is total_before_tax (GROSS sales before discount)
        // We need to subtract discount to get NET sales for Trading Account
        $sales_gross = (float) ($data['total_sell'] ?? 0);
        $sales_discount_amount = (float) ($data['total_sell_discount'] ?? 0);
        $sales_net = (float) ($sales_gross - $sales_discount_amount);
        
        // Trading Account Credit Side (before Gross Profit)
        // Credit = Sales (NET = gross - discount) + Direct Incomes + Closing Stock
        $trading_credit_side = $sales_net + $direct_income_total_precalc + $closing_stock;
        
        // Trading Account Debit Side (before Gross Profit)
        // Debit = Opening Stock + Purchases + Direct Expenses
        $trading_debit_side = $opening_stock + $purchases + $direct_exp;
        
        // Gross Profit calculation (Credit - Debit)
        // If positive = Gross Profit, if negative = Gross Loss
        $gross_profit_c_o = (float) ($trading_credit_side - $trading_debit_side);
        
        // Log calculation for debugging
        \Log::info('Gross Profit Calculation:', [
            'sales_gross' => $sales_gross,
            'sales_discount' => $sales_discount_amount,
            'sales_net' => $sales_net,
            'direct_income_total' => $direct_income_total_precalc,
            'closing_stock' => $closing_stock,
            'opening_stock' => $opening_stock,
            'purchases' => $purchases,
            'direct_exp' => $direct_exp,
            'trading_credit_side' => $trading_credit_side,
            'trading_debit_side' => $trading_debit_side,
            'gross_profit_c_o' => $gross_profit_c_o
        ]);
        
        // 4. Gross Profit c/o (carried over)
        // If positive, it goes on left side as "Gross Profit c/o"
        // If negative, it's Gross Loss and goes on right side
        // ALWAYS show Gross Profit/Loss (required for Tally format)
        if (abs($gross_profit_c_o) > 0.01) {
            // Non-zero gross profit/loss
            if ($gross_profit_c_o > 0) {
                $left_side[] = [
                    'label' => 'Gross Profit c/o',
                    'value' => $gross_profit_c_o,
                    'items' => []
                ];
            } else {
                // Gross Loss
                $left_side[] = [
                    'label' => 'Gross Loss c/o',
                    'value' => abs($gross_profit_c_o),
                    'items' => []
                ];
            }
        } else {
            // Zero or very small - still show it (required for proper P&L structure)
            $left_side[] = [
                'label' => 'Gross Profit c/o',
                'value' => 0.00,
                'items' => []
            ];
        }
        
        // Subtotal (before indirect expenses) - Trading Account subtotal on left
        $subtotal_left = (float) ($trading_debit_side + ($gross_profit_c_o > 0 ? $gross_profit_c_o : 0));
        
        // 5. Indirect Expenses (from actual account transactions)
        $indirect_expenses = [];
        $indirect_total = 0;
        
        // Get indirect expenses from account transactions (expenses that are not direct)
        try {
            $indirect_expense_accounts = $this->getIndirectExpenseAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            
            // Group expenses by category
            $expense_categories = [
                'Administrative Expenses' => [],
                'Salaries & Staff Expenses' => [],
                'Selling & Distribution Expenses' => [],
                'Other Expenses' => []
            ];
            
            foreach ($indirect_expense_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    // Ensure account name is not empty
                    $display_name = trim($account_name);
                    if (empty($display_name)) {
                        $display_name = 'Unnamed Expense';
                    }
                    
                    $category = $this->categorizeExpense($display_name);
                    if (!isset($expense_categories[$category])) {
                        $category = 'Other Expenses';
                    }
                    $expense_categories[$category][] = ['label' => $display_name, 'value' => abs($amount)];
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting indirect expenses: ' . $e->getMessage());
            $expense_categories = [
                'Administrative Expenses' => [],
                'Salaries & Staff Expenses' => [],
                'Selling & Distribution Expenses' => [],
                'Other Expenses' => []
            ];
        }
        
        // Add expense transactions if available from main data
        if (!empty($data['total_expense'])) {
            // Split expenses into categories based on available data
            $expense_categories['Other Expenses'][] = ['label' => 'General Expenses', 'value' => $data['total_expense']];
        }
        
        // Add categorized expenses (as groups)
        foreach ($expense_categories as $category => $items) {
            if (!empty($items)) {
                $category_total = (float) array_sum(array_column($items, 'value'));
                $indirect_expenses[] = [
                    'label' => $category, 
                    'value' => $category_total, 
                    'is_group' => true, 
                    'sub_items' => $items
                ];
            }
        }
        
        // Sales Discount should NOT be in expenses - it's already deducted from sales
        // (Sales discount reduces sales revenue, not an expense)
        
        if (!empty($data['total_sell_return'])) {
            $indirect_expenses[] = ['label' => 'Sales Returns', 'value' => $data['total_sell_return']];
        }
        
        if (!empty($data['total_reward_amount'])) {
            $indirect_expenses[] = ['label' => 'Reward Amount', 'value' => $data['total_reward_amount']];
        }
        
        if (!empty($data['total_sell_round_off'])) {
            $indirect_expenses[] = ['label' => 'Rounded Off (+/-)', 'value' => abs($data['total_sell_round_off'])];
        }
        
        // Left side module indirect expenses
        if (!empty($data['left_side_module_data'])) {
            foreach ($data['left_side_module_data'] as $module_data) {
                if (!empty($module_data['add_to_net_profit']) && $module_data['value'] > 0) {
                    // Skip if already added to direct expenses
                    $is_direct = false;
                    $direct_keywords = ['packaging', 'carriage', 'handling', 'assemble', 'freight', 'transport', 'delivery'];
                    foreach ($direct_keywords as $keyword) {
                        if (stripos($module_data['label'], $keyword) !== false) {
                            $is_direct = true;
                            break;
                        }
                    }
                    
                    if (!$is_direct) {
                        $indirect_expenses[] = ['label' => $module_data['label'], 'value' => $module_data['value']];
                    }
                }
            }
        }
        
        // Calculate total from all indirect expenses
        // For grouped items, sum the sub_items
        foreach ($indirect_expenses as $item) {
            if (!empty($item['is_group']) && !empty($item['sub_items'])) {
                // Grouped item - sum the sub_items
                $group_total = 0;
                foreach ($item['sub_items'] as $sub_item) {
                    $group_total += (float) ($sub_item['value'] ?? 0);
                }
                $item['value'] = $group_total; // Update item value to match sum
                $indirect_total += $group_total;
            } else {
                $indirect_total += (float) ($item['value'] ?? 0);
            }
        }
        
        // Always show indirect expenses section (even if empty, for proper structure)
        // Ensure value matches the sum of all items
        $calculated_indirect_total = 0;
        foreach ($indirect_expenses as $item) {
            if (!empty($item['is_group']) && !empty($item['sub_items'])) {
                $calculated_indirect_total += (float) ($item['value'] ?? 0);
            } else {
                $calculated_indirect_total += (float) ($item['value'] ?? 0);
            }
        }
        
        // Use the calculated total (more accurate)
        $indirect_total = $calculated_indirect_total;
        
        $left_side[] = [
            'label' => 'Indirect Expenses',
            'value' => $indirect_total,
            'items' => !empty($indirect_expenses) ? $indirect_expenses : []
        ];
        
        // RIGHT SIDE (CREDIT/INCOME) - Tally Format
        $right_side = [];
        
        // 1. Sales Accounts (from actual sales transactions - ALWAYS SHOW THIS)
        $sales_items = [];
        $sales_total_val = 0;
        
        // Get REAL sales data from UltimatePOS
        // NOTE: total_sell from UltimatePOS is total_before_tax (GROSS sales before discount)
        // We calculate NET sales = gross - discount
        $sales_gross = (float) ($data['total_sell'] ?? 0);
        $sales_discount = (float) ($data['total_sell_discount'] ?? 0);
        $sales_net = (float) ($sales_gross - $sales_discount);
        
        if ($sales_gross > 0.01 || $sales_gross < -0.01) {
            $sales_items[] = ['label' => 'Sale - GST/IGST', 'value' => $sales_gross];
            $sales_total_val = $sales_gross;
            
            // Show discount separately
            if (abs($sales_discount) > 0.01) {
                $sales_items[] = ['label' => 'Less: Price De-escalation (IGST-18%)', 'value' => -1 * abs($sales_discount)]; // Negative for display
                $sales_total_val -= abs($sales_discount); // Subtract discount to get NET sales
            }
        }
        
        // Always show sales section (even if empty)
        $right_side[] = [
            'label' => 'Sales Accounts',
            'value' => $sales_total_val, // This should be NET sales
            'items' => !empty($sales_items) ? $sales_items : [['label' => 'No Sales', 'value' => 0]]
        ];
        
        // 2. Direct Incomes (from direct incomes table + account transactions)
        $direct_incomes = [];
        $direct_income_total = 0;
        
        // Get direct incomes from ar_direct_incomes table
        try {
            $direct_income_query = DirectIncome::where('business_id', $business_id)
                ->whereBetween('income_date', [$start_date, $end_date])
                ->whereNull('deleted_at');
            
            if ($location_id) {
                $direct_income_query->where('location_id', $location_id);
            }
            
            if ($permitted_locations != 'all') {
                $direct_income_query->whereIn('location_id', $permitted_locations);
            }
            
            $direct_income_records = $direct_income_query->get();
            
            foreach ($direct_income_records as $income) {
                $amount = (float) $income->amount;
                if (abs($amount) > 0.01) {
                    $income_name = $income->name;
                    if (isset($direct_incomes[$income_name])) {
                        $direct_incomes[$income_name] += $amount;
                    } else {
                        $direct_incomes[$income_name] = $amount;
                    }
                    $direct_income_total += $amount;
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting direct incomes from table: ' . $e->getMessage());
        }
        
        // Also get direct income accounts (service charges, installation, AMC, etc.)
        try {
            $direct_income_accounts = $this->getDirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            
            foreach ($direct_income_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    // Merge with existing direct incomes from table
                    if (isset($direct_incomes[$account_name])) {
                        $direct_incomes[$account_name] += abs($amount);
                    } else {
                        $direct_incomes[$account_name] = abs($amount);
                    }
                    $direct_income_total += abs($amount);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting direct incomes from accounts: ' . $e->getMessage());
        }
        
        // Convert to array format for display - all names are dynamic from database
        $direct_incomes_display = [];
        foreach ($direct_incomes as $name => $amount) {
            // All income names come dynamically from database (from ar_direct_incomes table or account names)
            $direct_incomes_display[] = ['label' => $name, 'value' => $amount];
        }
        $direct_incomes = $direct_incomes_display;
        
        // Add shipping charges from sales (this comes from actual transaction data)
        if (!empty($data['total_sell_shipping_charge'])) {
            $shipping_label = 'Service & Installation Charges'; // Keep as fallback
            // Check if we already have an entry with this name from direct incomes table
            $label_exists = false;
            foreach ($direct_incomes as $item) {
                if (stripos($item['label'], 'service') !== false || stripos($item['label'], 'installation') !== false) {
                    $label_exists = true;
                    break;
                }
            }
            if (!$label_exists) {
                $direct_incomes[] = ['label' => $shipping_label, 'value' => $data['total_sell_shipping_charge']];
                $direct_income_total += $data['total_sell_shipping_charge'];
            }
        }
        
        // Add additional expenses from sales (service income)
        if (!empty($data['total_sell_additional_expense'])) {
            $other_service_label = 'Other Service Income'; // Keep as fallback
            // Check if we already have an entry with this name
            $label_exists = false;
            foreach ($direct_incomes as $item) {
                if (stripos($item['label'], 'other service') !== false) {
                    $label_exists = true;
                    break;
                }
            }
            if (!$label_exists) {
                $direct_incomes[] = ['label' => $other_service_label, 'value' => $data['total_sell_additional_expense']];
                $direct_income_total += $data['total_sell_additional_expense'];
            }
        }
        
        // Stock recovered
        if (!empty($data['total_recovered'])) {
            $stock_recovered_label = 'Stock Recovered'; // Keep as fallback
            // Check if we already have an entry with this name
            $label_exists = false;
            foreach ($direct_incomes as $item) {
                if (stripos($item['label'], 'stock recovered') !== false || stripos($item['label'], 'recovered') !== false) {
                    $label_exists = true;
                    break;
                }
            }
            if (!$label_exists) {
                $direct_incomes[] = ['label' => $stock_recovered_label, 'value' => $data['total_recovered']];
                $direct_income_total += $data['total_recovered'];
            }
        }
        
        // Right side module direct income
        if (!empty($data['right_side_module_data'])) {
            foreach ($data['right_side_module_data'] as $module_data) {
                if (!empty($module_data['add_to_net_profit']) && $module_data['value'] > 0) {
                    $direct_incomes[] = ['label' => $module_data['label'], 'value' => $module_data['value']];
                    $direct_income_total += $module_data['value'];
                }
            }
        }
        
        // Always show Direct Incomes section (even if empty, to match Direct Expenses behavior)
        if (!empty($direct_incomes)) {
            $right_side[] = [
                'label' => 'Direct Incomes',
                'value' => $direct_income_total,
                'items' => $direct_incomes
            ];
        } else {
            // Show empty section if no direct incomes found
            $right_side[] = [
                'label' => 'Direct Incomes',
                'value' => 0,
                'items' => [['label' => 'No Direct Incomes', 'value' => 0]]
            ];
        }
        
        // 3. Closing Stock (with detailed items)
        $closing_stock_items = [];
        $closing_stock_total = $data['closing_stock'] ?? 0;
        
        if (!empty($closing_stock_details) && is_array($closing_stock_details)) {
            foreach ($closing_stock_details as $stock_item) {
                if (isset($stock_item['name']) && isset($stock_item['stock_value'])) {
                    $closing_stock_items[] = [
                        'label' => $stock_item['name'],
                        'value' => $stock_item['stock_value']
                    ];
                }
            }
        }
        
        // If no detailed items, just show total
        if (empty($closing_stock_items) && $closing_stock_total > 0) {
            $closing_stock_items[] = ['label' => 'Closing Stock', 'value' => $closing_stock_total];
        }
        
        // Always show closing stock even if 0
        $right_side[] = [
            'label' => 'Closing Stock',
            'value' => $closing_stock_total,
            'items' => !empty($closing_stock_items) ? $closing_stock_items : [['label' => 'Closing Stock', 'value' => $closing_stock_total]]
        ];
        
        // Subtotal (before Indirect Incomes) - Trading Account credit side
        // Use NET sales (after discount) + Direct Incomes + Closing Stock for subtotal
        $subtotal_right = (float) ($sales_total_val + $direct_income_total + $closing_stock_total);
        
        // 4. Indirect Incomes (from account transactions) - Profit & Loss Account
        $indirect_incomes = [];
        $indirect_income_total = 0;
        
        // Get indirect income accounts
        try {
            $indirect_income_accounts = $this->getIndirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations);
            
            foreach ($indirect_income_accounts as $account_name => $amount) {
                if (abs($amount) > 0.01) {
                    $indirect_incomes[] = ['label' => $account_name, 'value' => abs($amount)];
                    $indirect_income_total += abs($amount);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting indirect incomes: ' . $e->getMessage());
        }
        
        // Purchase returns (from actual transaction data) - Indirect Income
        if (!empty($data['total_purchase_return'])) {
            $purchase_return_amount = (float) $data['total_purchase_return'];
            if (abs($purchase_return_amount) > 0.01) {
                $indirect_incomes[] = ['label' => 'Purchase Returns', 'value' => abs($purchase_return_amount)];
                $indirect_income_total += abs($purchase_return_amount);
            }
        }
        
        // Purchase discount (treated as income) - but this is usually negative, so we might not include it
        // if (!empty($data['total_purchase_discount'])) {
        //     $indirect_incomes[] = ['label' => 'Purchase Discount Received', 'value' => $data['total_purchase_discount']];
        //     $indirect_income_total += $data['total_purchase_discount'];
        // }
        
        // Right side module indirect income
        if (!empty($data['right_side_module_data'])) {
            foreach ($data['right_side_module_data'] as $module_data) {
                if (!empty($module_data['add_to_net_profit']) && $module_data['value'] > 0) {
                    // Skip if already added to direct income
                    $is_direct = false;
                    $direct_keywords = ['AMC', 'Service', 'Installation', 'Maintenance'];
                    foreach ($direct_keywords as $keyword) {
                        if (stripos($module_data['label'], $keyword) !== false) {
                            $is_direct = true;
                            break;
                        }
                    }
                    
                    if (!$is_direct) {
                        $indirect_incomes[] = ['label' => $module_data['label'], 'value' => $module_data['value']];
                        $indirect_income_total += $module_data['value'];
                    }
                }
            }
        }
        
        if (!empty($indirect_incomes)) {
            $right_side[] = [
                'label' => 'Indirect Incomes',
                'value' => $indirect_income_total,
                'items' => $indirect_incomes
            ];
        }
        
        // 5. Gross Profit b/f (brought forward from Trading Account to Profit & Loss)
        // Position: After Indirect Incomes, before Net Profit (Tally style)
        // CRITICAL: This must be the SAME value as Gross Profit c/o (the actual gross profit)
        // ALWAYS show (required for proper Tally format)
        if (abs($gross_profit_c_o) > 0.01) {
            // Non-zero gross profit/loss
            if ($gross_profit_c_o > 0) {
                $right_side[] = [
                    'label' => 'Gross Profit b/f',
                    'value' => $gross_profit_c_o,  // This is the actual gross profit
                    'items' => []
                ];
            } else {
                // Gross Loss b/f
                $right_side[] = [
                    'label' => 'Gross Loss b/f',
                    'value' => abs($gross_profit_c_o),
                    'items' => []
                ];
            }
        } else {
            // Zero or very small - still show it for proper structure
            $right_side[] = [
                'label' => 'Gross Profit b/f',
                'value' => 0.00,
                'items' => []
            ];
        }
        
        // Net Profit Calculation (Profit & Loss Account)
        // P&L Credit Side: Gross Profit b/f + Indirect Incomes
        // P&L Debit Side: Indirect Expenses
        // Net Profit = Credit - Debit
        
        $pl_credit_side = (float) ($gross_profit_c_o + $indirect_income_total);
        $pl_debit_side = (float) $indirect_total;
        $net_profit = (float) ($pl_credit_side - $pl_debit_side);
        
        // Add Net Profit/Loss to appropriate side (Tally format)
        if ($net_profit > 0.01) {
            // Net Profit - shown on both sides for balancing
            $left_side[] = [
                'label' => 'Nett Profit',
                'value' => $net_profit,
                'items' => []
            ];
            $right_side[] = [
                'label' => 'Nett Profit',
                'value' => $net_profit,
                'items' => []
            ];
        } else if ($net_profit < -0.01) {
            // Net Loss - shown on both sides for balancing
            $net_loss = abs($net_profit);
            $left_side[] = [
                'label' => 'Nett Loss',
                'value' => $net_loss,
                'items' => []
            ];
            $right_side[] = [
                'label' => 'Nett Loss',
                'value' => $net_loss,
                'items' => []
            ];
        }
        
        // Final totals calculation (both sides must balance in Tally format)
        // Tally P&L Structure:
        // LEFT SIDE (Debit): Opening Stock + Purchases + Direct Expenses + Gross Profit c/o + Indirect Expenses + Net Profit (if profit)
        // RIGHT SIDE (Credit): Sales (NET) + Closing Stock + Gross Profit b/f + Indirect Incomes + Net Profit (if profit)
        
        // Calculate left side total by summing all left side items
        $final_total_left = 0;
        foreach ($left_side as $section) {
            $final_total_left += (float) ($section['value'] ?? 0);
        }
        
        // Calculate right side total by summing all right side items
        $final_total_right = 0;
        foreach ($right_side as $section) {
            $final_total_right += (float) ($section['value'] ?? 0);
        }
        
        // Verify both sides balance (Tally requirement - both sides MUST be equal)
        $balance_diff = abs($final_total_left - $final_total_right);
        
        // Recalculate to verify - sum all actual values from sections
        $verify_left = 0;
        foreach ($left_side as $section) {
            $verify_left += (float) ($section['value'] ?? 0);
        }
        
        $verify_right = 0;
        foreach ($right_side as $section) {
            $verify_right += (float) ($section['value'] ?? 0);
        }
        
        // If there's still a difference after verification, it means we have a calculation error
        // Log it for debugging, but ensure both sides balance
        if ($balance_diff > 0.01 || abs($verify_left - $verify_right) > 0.01) {
            $actual_diff = abs($verify_left - $verify_right);
            \Log::warning('P&L Balance Mismatch Detected:', [
                'calculated_left' => $final_total_left,
                'calculated_right' => $final_total_right,
                'calculated_diff' => $balance_diff,
                'verified_left' => $verify_left,
                'verified_right' => $verify_right,
                'verified_diff' => $actual_diff,
                'business_id' => $business_id,
                'start_date' => $start_date,
                'end_date' => $end_date
            ]);
            
            // Use verified totals (they're more accurate as they sum actual section values)
            $final_total_left = $verify_left;
            $final_total_right = $verify_right;
            
            // Ensure both sides balance - use the larger total
            // In Tally, both sides must be equal (this is a fundamental accounting principle)
            $final_total = max($final_total_left, $final_total_right);
            $final_total_left = $final_total;
            $final_total_right = $final_total;
        } else {
            // Both sides already balance - use the calculated total
            $final_total = $final_total_left; // or $final_total_right, they're equal
        }
        
        return [
            'left_side' => $left_side,
            'right_side' => $right_side,
            'subtotal_left' => $subtotal_left,
            'subtotal_right' => $subtotal_right,
            'gross_profit_c_o' => $gross_profit_c_o,
            'indirect_expenses_total' => $indirect_total,
            'indirect_incomes_total' => $indirect_income_total,
            'net_profit' => $net_profit,
            'final_total_left' => $final_total_left,
            'final_total_right' => $final_total_right,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'raw_data' => $data
        ];
    }

    /**
     * Get stock details by product/category for opening/closing stock
     */
    protected function getStockDetails($business_id, $date, $location_id, $is_opening, $permitted_locations)
    {
        try {
            $day_before = $is_opening ? \Carbon\Carbon::createFromFormat('Y-m-d', $date)->subDay()->format('Y-m-d') : $date;
            
            $query = PurchaseLine::join(
                'transactions as purchase',
                'purchase_lines.transaction_id',
                '=',
                'purchase.id'
            )
            ->leftjoin('variations as v', 'v.id', '=', 'purchase_lines.variation_id')
            ->leftjoin('products as p', 'p.id', '=', 'purchase_lines.product_id')
            ->leftjoin('categories as c', 'c.id', '=', 'p.category_id')
            ->where('purchase.business_id', $business_id)
            ->where('purchase.type', '!=', 'purchase_order')
            ->whereNull('purchase.deleted_at')
            ->whereNull('purchase_lines.deleted_at');
            
            if ($is_opening) {
                $next_day = \Carbon\Carbon::createFromFormat('Y-m-d', $date)->addDay()->format('Y-m-d');
                $query->where(function ($q) use ($day_before, $next_day) {
                    $q->whereRaw("date(transaction_date) <= '$day_before'")
                      ->orWhereRaw("date(transaction_date) = '$next_day' AND purchase.type='opening_stock'");
                });
            } else {
                $query->whereRaw("date(transaction_date) <= '$date'");
            }
            
            if ($location_id) {
                $query->where('purchase.location_id', $location_id);
            }
            
            if ($permitted_locations != 'all') {
                $query->whereIn('purchase.location_id', $permitted_locations);
            }
            
            $stock_details = $query->select(
                DB::raw("COALESCE(c.name, 'No Category') as category_name"),
                DB::raw("COALESCE(p.name, 'Unnamed Product') as product_name"),
                DB::raw("COALESCE(SUM((purchase_lines.quantity - COALESCE(purchase_lines.quantity_returned, 0) - COALESCE(purchase_lines.quantity_adjusted, 0) -
                            (SELECT COALESCE(SUM(COALESCE(tspl.quantity, 0) - COALESCE(tspl.qty_returned, 0)), 0) FROM 
                            transaction_sell_lines_purchase_lines AS tspl
                            JOIN transaction_sell_lines as tsl ON tspl.sell_line_id=tsl.id 
                            JOIN transactions as sale ON tsl.transaction_id=sale.id 
                            WHERE tspl.purchase_line_id = purchase_lines.id AND 
                            date(sale.transaction_date) <= '$date' AND sale.deleted_at IS NULL) ) * 
                            (COALESCE(purchase_lines.purchase_price, 0) + COALESCE(purchase_lines.item_tax, 0))
                        ), 0) as stock_value")
            )
            ->groupBy('c.id', 'c.name', 'p.id', 'p.name')
            ->havingRaw('ABS(stock_value) > 0.01')
            ->get();
            
            $details = [];
            foreach ($stock_details as $stock) {
                $name = $stock->product_name;
                if ($stock->category_name && $stock->category_name != 'No Category') {
                    $name .= ' (' . $stock->category_name . ')';
                }
                $details[] = [
                    'name' => $name,
                    'stock_value' => (float) ($stock->stock_value ?? 0)
                ];
            }
            
            return $details;
        } catch (\Exception $e) {
            \Log::error('Error getting stock details: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get direct expense accounts (carriage, packaging, handling, etc.)
     */
    protected function getDirectExpenseAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations)
    {
        try {
            $direct_keywords = ['carriage', 'packaging', 'handling', 'assemble', 'assembly', 'freight', 'transport', 'delivery', 'loading', 'unloading'];
            
            $query = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->where('A.business_id', $business_id)
                ->where('account_transactions.type', 'debit')
                ->whereBetween(DB::raw('date(account_transactions.operation_date)'), [$start_date, $end_date])
                ->whereNull('account_transactions.deleted_at')
                ->whereNull('A.deleted_at');
            
            if ($location_id) {
                $location = BusinessLocation::find($location_id);
                if (!empty($location->default_payment_accounts)) {
                    $account_ids = [];
                    $default_accounts = json_decode($location->default_payment_accounts, true);
                    if (is_array($default_accounts)) {
                        foreach ($default_accounts as $acc) {
                            if (!empty($acc['is_enabled']) && !empty($acc['account'])) {
                                $account_ids[] = $acc['account'];
                            }
                        }
                        if (!empty($account_ids)) {
                            $query->whereIn('A.id', $account_ids);
                        }
                    }
                }
            }
            
            $expenses = $query->select(
                'A.name',
                DB::raw("COALESCE(SUM(account_transactions.amount), 0) as total")
            )
            ->groupBy('A.id', 'A.name')
            ->get();
            
            $direct_expenses = [];
            foreach ($expenses as $expense) {
                // Ensure we have a valid account name
                $account_name = trim($expense->name ?? '');
                if (empty($account_name)) {
                    $account_name = 'Unnamed Account';
                }
                
                // Exclude asset accounts (cash, bank, etc.) from P&L
                if ($this->isAssetAccount($account_name)) {
                    continue;
                }
                
                foreach ($direct_keywords as $keyword) {
                    if (stripos($account_name, $keyword) !== false) {
                        $amount = (float) ($expense->total ?? 0);
                        if (abs($amount) > 0.01) {
                            // Ensure unique keys
                            if (isset($direct_expenses[$account_name])) {
                                $direct_expenses[$account_name] += $amount;
                            } else {
                                $direct_expenses[$account_name] = $amount;
                            }
                        }
                        break;
                    }
                }
            }
            
            return $direct_expenses;
        } catch (\Exception $e) {
            \Log::error('Error getting direct expense accounts: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Check if account is an asset account (Cash, Bank, etc.) - should not be in P&L
     */
    protected function isAssetAccount($account_name)
    {
        if (empty($account_name)) {
            return false;
        }
        
        $account_lower = strtolower($account_name);
        
        // Asset account keywords (Balance Sheet accounts, not P&L)
        $asset_keywords = [
            'cash', 'bank', 'ebl', 'dbbl', 'brac', 'ucb', 'sbl', 'hsbc', 'standard chartered',
            'petty cash', 'cash in hand', 'cash at bank', 'bank balance',
            'accounts receivable', 'debtors', 'sundry debtors',
            'inventory', 'stock', 'goods',
            'fixed assets', 'plant', 'equipment', 'machinery', 'building', 'vehicle',
            'prepaid', 'advance', 'deposit', 'investment'
        ];
        
        foreach ($asset_keywords as $keyword) {
            if (stripos($account_lower, $keyword) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Get indirect expense accounts
     */
    protected function getIndirectExpenseAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations)
    {
        try {
            $direct_keywords = ['carriage', 'packaging', 'handling', 'assemble', 'assembly', 'freight', 'transport', 'delivery'];
            
            $query = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->where('A.business_id', $business_id)
                ->where('account_transactions.type', 'debit')
                ->whereBetween(DB::raw('date(account_transactions.operation_date)'), [$start_date, $end_date])
                ->whereNull('account_transactions.deleted_at')
                ->whereNull('A.deleted_at');
            
            // Also get from expense transactions (exclude deleted)
            $expense_query = Transaction::where('business_id', $business_id)
                ->where('type', 'expense')
                ->whereNull('deleted_at')
                ->whereBetween('transaction_date', [$start_date, $end_date]);
            
            if ($location_id) {
                $location = BusinessLocation::find($location_id);
                if (!empty($location->default_payment_accounts)) {
                    $account_ids = [];
                    $default_accounts = json_decode($location->default_payment_accounts, true);
                    if (is_array($default_accounts)) {
                        foreach ($default_accounts as $acc) {
                            if (!empty($acc['is_enabled']) && !empty($acc['account'])) {
                                $account_ids[] = $acc['account'];
                            }
                        }
                        if (!empty($account_ids)) {
                            $query->whereIn('A.id', $account_ids);
                            $expense_query->where('location_id', $location_id);
                        }
                    }
                }
            }
            
            $account_expenses = $query->select(
                'A.name',
                DB::raw("COALESCE(SUM(account_transactions.amount), 0) as total")
            )
            ->groupBy('A.id', 'A.name')
            ->get();
            
            $indirect_expenses = [];
            foreach ($account_expenses as $expense) {
                // Ensure we have a valid name
                $account_name = trim($expense->name ?? '');
                if (empty($account_name)) {
                    $account_name = 'Unnamed Account';
                }
                
                $is_direct = false;
                foreach ($direct_keywords as $keyword) {
                    if (stripos($account_name, $keyword) !== false) {
                        $is_direct = true;
                        break;
                    }
                }
                
                // Exclude asset accounts (cash, bank, etc.) from P&L
                if (!$is_direct && !$this->isAssetAccount($account_name)) {
                    $amount = (float) ($expense->total ?? 0);
                    // Use account name as key, but make sure it's unique
                    $key = $account_name;
                    if (isset($indirect_expenses[$key])) {
                        $indirect_expenses[$key] += $amount;
                    } else {
                        $indirect_expenses[$key] = $amount;
                    }
                }
            }
            
            // Add expense transactions with proper names
            try {
                $expense_transactions = $expense_query->select(
                    DB::raw("COALESCE(NULLIF(transaction_for, ''), 'Other Expenses') as expense_name"),
                    DB::raw("COALESCE(SUM(total_before_tax), 0) as total")
                )
                ->groupBy('transaction_for')
                ->get();
                
                foreach ($expense_transactions as $exp) {
                    $name = trim($exp->expense_name ?? '');
                    if (empty($name) || $name == '') {
                        $name = 'General Expenses';
                    }
                    
                    $amount = (float) ($exp->total ?? 0);
                    if (abs($amount) > 0.01) {
                        // Add to existing or create new
                        if (isset($indirect_expenses[$name])) {
                            $indirect_expenses[$name] += $amount;
                        } else {
                            $indirect_expenses[$name] = $amount;
                        }
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Error getting expense transactions: ' . $e->getMessage());
            }
            
            return $indirect_expenses;
        } catch (\Exception $e) {
            \Log::error('Error getting indirect expense accounts: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get direct income accounts (service charges, installation, AMC, etc.)
     */
    protected function getDirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations)
    {
        try {
            $direct_keywords = ['service', 'installation', 'AMC', 'maintenance', 'repair', 'commission'];
            
            $query = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->where('A.business_id', $business_id)
                ->where('account_transactions.type', 'credit')
                ->whereBetween(DB::raw('date(account_transactions.operation_date)'), [$start_date, $end_date])
                ->whereNull('account_transactions.deleted_at')
                ->whereNull('A.deleted_at');
            
            if ($location_id) {
                $location = BusinessLocation::find($location_id);
                if (!empty($location->default_payment_accounts)) {
                    $account_ids = [];
                    $default_accounts = json_decode($location->default_payment_accounts, true);
                    if (is_array($default_accounts)) {
                        foreach ($default_accounts as $acc) {
                            if (!empty($acc['is_enabled']) && !empty($acc['account'])) {
                                $account_ids[] = $acc['account'];
                            }
                        }
                        if (!empty($account_ids)) {
                            $query->whereIn('A.id', $account_ids);
                        }
                    }
                }
            }
            
            $incomes = $query->select(
                'A.name',
                DB::raw("COALESCE(SUM(account_transactions.amount), 0) as total")
            )
            ->groupBy('A.id', 'A.name')
            ->get();
            
            $direct_incomes = [];
            foreach ($incomes as $income) {
                // Ensure we have a valid account name
                $account_name = trim($income->name ?? '');
                if (empty($account_name)) {
                    $account_name = 'Unnamed Account';
                }
                
                // Exclude asset accounts (cash, bank, etc.) from P&L
                if ($this->isAssetAccount($account_name)) {
                    continue;
                }
                
                foreach ($direct_keywords as $keyword) {
                    if (stripos($account_name, $keyword) !== false) {
                        $amount = (float) ($income->total ?? 0);
                        if (abs($amount) > 0.01) {
                            // Ensure unique keys
                            if (isset($direct_incomes[$account_name])) {
                                $direct_incomes[$account_name] += $amount;
                            } else {
                                $direct_incomes[$account_name] = $amount;
                            }
                        }
                        break;
                    }
                }
            }
            
            return $direct_incomes;
        } catch (\Exception $e) {
            \Log::error('Error getting direct income accounts: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get indirect income accounts
     */
    protected function getIndirectIncomeAccounts($business_id, $start_date, $end_date, $location_id, $permitted_locations)
    {
        try {
            $direct_keywords = ['service', 'installation', 'AMC', 'maintenance'];
            
            $query = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->where('A.business_id', $business_id)
                ->where('account_transactions.type', 'credit')
                ->whereBetween(DB::raw('date(account_transactions.operation_date)'), [$start_date, $end_date])
                ->whereNull('account_transactions.deleted_at')
                ->whereNull('A.deleted_at');
            
            if ($location_id) {
                $location = BusinessLocation::find($location_id);
                if (!empty($location->default_payment_accounts)) {
                    $account_ids = [];
                    $default_accounts = json_decode($location->default_payment_accounts, true);
                    if (is_array($default_accounts)) {
                        foreach ($default_accounts as $acc) {
                            if (!empty($acc['is_enabled']) && !empty($acc['account'])) {
                                $account_ids[] = $acc['account'];
                            }
                        }
                        if (!empty($account_ids)) {
                            $query->whereIn('A.id', $account_ids);
                        }
                    }
                }
            }
            
            $incomes = $query->select(
                'A.name',
                DB::raw("COALESCE(SUM(account_transactions.amount), 0) as total")
            )
            ->groupBy('A.id', 'A.name')
            ->get();
            
            $indirect_incomes = [];
            foreach ($incomes as $income) {
                // Ensure we have a valid account name
                $account_name = trim($income->name ?? '');
                if (empty($account_name)) {
                    $account_name = 'Unnamed Account';
                }
                
                // Exclude asset accounts (cash, bank, etc.) from P&L
                if ($this->isAssetAccount($account_name)) {
                    continue;
                }
                
                $is_direct = false;
                foreach ($direct_keywords as $keyword) {
                    if (stripos($account_name, $keyword) !== false) {
                        $is_direct = true;
                        break;
                    }
                }
                
                if (!$is_direct) {
                    $amount = (float) ($income->total ?? 0);
                    if (abs($amount) > 0.01) {
                        // Ensure unique keys
                        if (isset($indirect_incomes[$account_name])) {
                            $indirect_incomes[$account_name] += $amount;
                        } else {
                            $indirect_incomes[$account_name] = $amount;
                        }
                    }
                }
            }
            
            return $indirect_incomes;
        } catch (\Exception $e) {
            \Log::error('Error getting indirect income accounts: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Categorize expense account
     */
    protected function categorizeExpense($account_name)
    {
        $name_lower = strtolower($account_name);
        
        // Administrative Expenses
        if (stripos($name_lower, 'admin') !== false || 
            stripos($name_lower, 'office') !== false ||
            stripos($name_lower, 'utility') !== false ||
            stripos($name_lower, 'rent') !== false ||
            stripos($name_lower, 'electricity') !== false) {
            return 'Administrative Expenses';
        }
        
        // Salaries & Staff Expenses
        if (stripos($name_lower, 'salary') !== false || 
            stripos($name_lower, 'wage') !== false ||
            stripos($name_lower, 'staff') !== false ||
            stripos($name_lower, 'employee') !== false ||
            stripos($name_lower, 'payroll') !== false) {
            return 'Salaries & Staff Expenses';
        }
        
        // Selling & Distribution Expenses
        if (stripos($name_lower, 'selling') !== false || 
            stripos($name_lower, 'sales') !== false ||
            stripos($name_lower, 'marketing') !== false ||
            stripos($name_lower, 'advertisement') !== false ||
            stripos($name_lower, 'promotion') !== false) {
            return 'Selling & Distribution Expenses';
        }
        
        return 'Other Expenses';
    }

    /**
     * Get account transactions by category (legacy method - not used but kept for compatibility)
     */
    protected function getAccountTransactionsByCategory($business_id, $start_date, $end_date, $location_id, $category, $permitted_locations)
    {
        // This method can be used for future enhancements
        return [];
    }

    public function export(Request $request)
    {
        // TODO: Implement export
        return redirect()->back();
    }
}


