<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Entities\DirectExpenseName;
use App\Utils\Util;
use DB;
use Yajra\DataTables\Facades\DataTables;

class DirectExpenseNameController extends Controller
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of direct expense names
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::direct-expense-names.index');
    }

    /**
     * Get direct expense names data for DataTables
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;

        $query = DirectExpenseName::with(['creator'])
            ->where('business_id', $business_id);

        // Filter by active status if provided
        if ($request->has('is_active') && $request->is_active !== '') {
            $query->where('is_active', $request->is_active);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-info btn-xs view_name" data-href="' . 
                        action([self::class, 'show'], [$row->id]) . '"><i class="fa fa-eye"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-primary btn-xs edit_name" data-href="' . 
                        action([self::class, 'edit'], [$row->id]) . '"><i class="fa fa-edit"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_name" data-href="' . 
                        action([self::class, 'destroy'], [$row->id]) . '"><i class="fa fa-trash"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->editColumn('is_active', function ($row) {
                if ($row->is_active) {
                    return '<span class="label label-success">' . __('accounting-reports::lang.active') . '</span>';
                } else {
                    return '<span class="label label-danger">' . __('accounting-reports::lang.inactive') . '</span>';
                }
            })
            ->editColumn('description', function ($row) {
                return !empty($row->description) ? $row->description : '-';
            })
            ->editColumn('created_at', function ($row) {
                return $this->commonUtil->format_date($row->created_at, true);
            })
            ->rawColumns(['action', 'is_active'])
            ->toJson();
    }

    /**
     * Show the form for creating a new direct expense name
     */
    public function create()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::direct-expense-names.create');
    }

    /**
     * Store a newly created direct expense name
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $user_id = auth()->user()->id;

            $request->validate([
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_direct_expense_names,name,NULL,id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'description' => 'nullable|string',
            ]);

            $data = $request->only(['name', 'description']);
            $data['business_id'] = $business_id;
            $data['created_by'] = $user_id;
            $data['is_active'] = $request->has('is_active') ? 1 : 1; // Default to active

            $expense_name = DirectExpenseName::create($data);

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.direct_expense_name_added_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Display the specified direct expense name
     */
    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $expense_name = DirectExpenseName::where('business_id', $business_id)
            ->with(['creator'])
            ->findOrFail($id);

        return view('accounting-reports::direct-expense-names.show', compact('expense_name'));
    }

    /**
     * Show the form for editing the specified direct expense name
     */
    public function edit($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $expense_name = DirectExpenseName::where('business_id', $business_id)->findOrFail($id);

        return view('accounting-reports::direct-expense-names.edit', compact('expense_name'));
    }

    /**
     * Update the specified direct expense name
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $expense_name = DirectExpenseName::where('business_id', $business_id)->findOrFail($id);

            $request->validate([
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_direct_expense_names,name,' . $id . ',id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'description' => 'nullable|string',
            ]);

            $data = $request->only(['name', 'description']);
            $data['is_active'] = $request->has('is_active') ? 1 : 0;

            $expense_name->update($data);

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.direct_expense_name_updated_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Remove the specified direct expense name
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $expense_name = DirectExpenseName::where('business_id', $business_id)->findOrFail($id);

            $expense_name->delete();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.direct_expense_name_deleted_success')
            ];

            return response()->json($output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }
}
