<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Entities\DirectExpense;
use App\BusinessLocation;
use App\Account;
use App\AccountTransaction;
use App\Utils\Util;
use App\Utils\TransactionUtil;
use App\Utils\ModuleUtil;
use DB;
use Yajra\DataTables\Facades\DataTables;

class DirectExpenseController extends Controller
{
    protected $commonUtil;
    protected $transactionUtil;
    protected $moduleUtil;

    public function __construct(Util $commonUtil, TransactionUtil $transactionUtil, ModuleUtil $moduleUtil)
    {
        $this->commonUtil = $commonUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of direct expenses
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($business_id, true);
        $accounts = Account::forDropdown($business_id, false);

        return view('accounting-reports::direct-expenses.index', compact('locations', 'accounts'));
    }

    /**
     * Get direct expenses data for DataTables
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;

        $query = DirectExpense::with(['location', 'account', 'creator'])
            ->where('business_id', $business_id);

        // Apply filters
        if ($request->has('location_id') && !empty($request->location_id)) {
            $query->where('location_id', $request->location_id);
        }

        if ($request->has('account_id') && !empty($request->account_id)) {
            $query->where('account_id', $request->account_id);
        }

        if ($request->has('start_date') && !empty($request->start_date)) {
            $query->whereDate('expense_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && !empty($request->end_date)) {
            $query->whereDate('expense_date', '<=', $request->end_date);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-info btn-xs view_expense" data-href="' . 
                        action([self::class, 'show'], [$row->id]) . '"><i class="fa fa-eye"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-primary btn-xs edit_expense" data-href="' . 
                        action([self::class, 'edit'], [$row->id]) . '"><i class="fa fa-edit"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_expense" data-href="' . 
                        action([self::class, 'destroy'], [$row->id]) . '"><i class="fa fa-trash"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->editColumn('expense_date', function ($row) {
                return $this->commonUtil->format_date($row->expense_date, true);
            })
            ->editColumn('name', function ($row) {
                return '<strong>' . e($row->name) . '</strong>' . 
                       (!empty($row->description) ? '<br><small class="text-muted">' . e($row->description) . '</small>' : '');
            })
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->amount . '</span>';
            })
            ->editColumn('location', function ($row) {
                return $row->location ? $row->location->name : __('lang_v1.all');
            })
            ->editColumn('account', function ($row) {
                return $row->account ? $row->account->name : '-';
            })
            ->editColumn('document', function ($row) {
                if ($row->document) {
                    return '<a href="' . asset('uploads/documents/' . $row->document) . '" target="_blank"><i class="fa fa-file"></i></a>';
                }
                return '-';
            })
            ->rawColumns(['action', 'amount', 'document', 'name'])
            ->toJson();
    }

    /**
     * Show the form for creating a new direct expense
     */
    public function create()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($business_id, true);
        $accounts = Account::forDropdown($business_id, false);
        $payment_types = $this->transactionUtil->payment_types(null, false, $business_id);
        
        // Payment accounts - same way as ExpenseController (use $accounts variable)
        $payment_accounts = [];
        if ($this->moduleUtil->isModuleEnabled('account')) {
            $payment_accounts = Account::forDropdown($business_id, true, false, true);
        }

        return view('accounting-reports::direct-expenses.create', compact('locations', 'accounts', 'payment_types', 'payment_accounts'));
    }

    /**
     * Store a newly created direct expense
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $user_id = auth()->user()->id;

            $request->validate([
                'name' => 'required|string|max:255',
                'amount' => 'required|numeric|min:0',
                'expense_date' => 'required|date',
                'location_id' => 'nullable|exists:business_locations,id',
                'account_id' => 'nullable|exists:accounts,id',
                'document' => 'nullable|file|max:' . (config('constants.document_size_limit', 10000) / 1000),
                'payment_method' => 'nullable|string',
                'payment_account_id' => 'nullable|exists:accounts,id',
                'paid_on' => 'nullable|date',
                'payment_note' => 'nullable|string',
            ]);

            $data = $request->only([
                'name', 'description', 'amount', 'expense_date', 
                'location_id', 'account_id', 'reference_no',
                'payment_method', 'payment_account_id', 'payment_note'
            ]);

            $data['business_id'] = $business_id;
            $data['created_by'] = $user_id;
            
            // Format amount
            $data['amount'] = $this->commonUtil->num_uf($data['amount']);
            
            // Format date (false = date only, no time)
            $data['expense_date'] = $this->commonUtil->uf_date($data['expense_date'], false);
            
            // Handle payment data
            $payment_amount = $request->input('payment_amount', 0);
            if ($payment_amount > 0) {
                $data['payment_amount'] = $this->commonUtil->num_uf($payment_amount);
                if ($request->has('paid_on') && !empty($request->paid_on)) {
                    $data['paid_on'] = $this->commonUtil->uf_date($request->paid_on, true);
                }
                
                // Calculate payment status
                $total_amount = $data['amount'];
                if ($data['payment_amount'] >= $total_amount) {
                    $data['payment_status'] = 'paid';
                } elseif ($data['payment_amount'] > 0) {
                    $data['payment_status'] = 'partial';
                } else {
                    $data['payment_status'] = 'due';
                }
            } else {
                $data['payment_status'] = 'due';
            }

            // Upload document if provided
            if ($request->hasFile('document')) {
                $document_name = $this->commonUtil->uploadFile($request, 'document', 'documents', 'document');
                if (!empty($document_name)) {
                    $data['document'] = $document_name;
                }
            }

            DB::beginTransaction();
            
            $direct_expense = DirectExpense::create($data);

            // Create account transaction if payment account is selected and payment amount > 0
            if (!empty($data['payment_account_id']) && !empty($data['payment_amount']) && $data['payment_amount'] > 0) {
                if ($this->moduleUtil->isModuleEnabled('account', $business_id)) {
                    $account_transaction_data = [
                        'amount' => $data['payment_amount'],
                        'account_id' => $data['payment_account_id'],
                        'type' => 'debit', // Expense payment = debit from account
                        'operation_date' => $data['paid_on'] ?? $data['expense_date'],
                        'created_by' => $user_id,
                        'note' => 'Direct Expense: ' . $data['name'] . ($data['payment_note'] ? ' - ' . $data['payment_note'] : ''),
                    ];
                    
                    AccountTransaction::createAccountTransaction($account_transaction_data);
                }
            }

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.direct_expense_added_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Display the specified direct expense
     */
    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $direct_expense = DirectExpense::where('business_id', $business_id)
            ->with(['location', 'account', 'creator'])
            ->findOrFail($id);

        return view('accounting-reports::direct-expenses.show', compact('direct_expense'));
    }

    /**
     * Show the form for editing the specified direct expense
     */
    public function edit($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $direct_expense = DirectExpense::where('business_id', $business_id)->findOrFail($id);
        
        $locations = BusinessLocation::forDropdown($business_id, true);
        $accounts = Account::forDropdown($business_id, false);
        $payment_types = $this->transactionUtil->payment_types(null, false, $business_id);
        
        // Payment accounts - same way as ExpenseController
        $payment_accounts = [];
        if ($this->moduleUtil->isModuleEnabled('account')) {
            $payment_accounts = Account::forDropdown($business_id, true, false, true);
        }

        return view('accounting-reports::direct-expenses.edit', compact('direct_expense', 'locations', 'accounts', 'payment_types', 'payment_accounts'));
    }

    /**
     * Update the specified direct expense
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $direct_expense = DirectExpense::where('business_id', $business_id)->findOrFail($id);

            $request->validate([
                'name' => 'required|string|max:255',
                'amount' => 'required|numeric|min:0',
                'expense_date' => 'required|date',
                'location_id' => 'nullable|exists:business_locations,id',
                'account_id' => 'nullable|exists:accounts,id',
                'document' => 'nullable|file|max:' . (config('constants.document_size_limit', 10000) / 1000),
                'payment_method' => 'nullable|string',
                'payment_account_id' => 'nullable|exists:accounts,id',
                'paid_on' => 'nullable|date',
                'payment_note' => 'nullable|string',
            ]);

            $data = $request->only([
                'name', 'description', 'amount', 'expense_date', 
                'location_id', 'account_id', 'reference_no',
                'payment_method', 'payment_account_id', 'payment_note'
            ]);

            // Format amount
            $data['amount'] = $this->commonUtil->num_uf($data['amount']);
            
            // Format date (false = date only, no time)
            $data['expense_date'] = $this->commonUtil->uf_date($data['expense_date'], false);
            
            // Handle payment data
            $payment_amount = $request->input('payment_amount', 0);
            if ($payment_amount > 0) {
                $data['payment_amount'] = $this->commonUtil->num_uf($payment_amount);
                if ($request->has('paid_on') && !empty($request->paid_on)) {
                    $data['paid_on'] = $this->commonUtil->uf_date($request->paid_on, true);
                }
                
                // Calculate payment status
                $total_amount = $data['amount'];
                if ($data['payment_amount'] >= $total_amount) {
                    $data['payment_status'] = 'paid';
                } elseif ($data['payment_amount'] > 0) {
                    $data['payment_status'] = 'partial';
                } else {
                    $data['payment_status'] = 'due';
                }
            } else {
                $data['payment_status'] = 'due';
            }

            // Upload document if provided
            if ($request->hasFile('document')) {
                // Delete old document if exists
                if ($direct_expense->document && file_exists(public_path('uploads/documents/' . $direct_expense->document))) {
                    @unlink(public_path('uploads/documents/' . $direct_expense->document));
                }

                $document_name = $this->commonUtil->uploadFile($request, 'document', 'documents', 'document');
                if (!empty($document_name)) {
                    $data['document'] = $document_name;
                }
            }

            DB::beginTransaction();
            
            // Delete old account transaction if exists
            AccountTransaction::where('note', 'LIKE', 'Direct Expense: ' . $direct_expense->name . '%')
                ->whereNull('transaction_id')
                ->whereNull('transaction_payment_id')
                ->delete();

            $direct_expense->update($data);

            // Create new account transaction if payment account is selected and payment amount > 0
            if (!empty($data['payment_account_id']) && !empty($data['payment_amount']) && $data['payment_amount'] > 0) {
                if ($this->moduleUtil->isModuleEnabled('account', $business_id)) {
                    $account_transaction_data = [
                        'amount' => $data['payment_amount'],
                        'account_id' => $data['payment_account_id'],
                        'type' => 'debit', // Expense payment = debit from account
                        'operation_date' => $data['paid_on'] ?? $data['expense_date'],
                        'created_by' => auth()->user()->id,
                        'note' => 'Direct Expense: ' . $data['name'] . ($data['payment_note'] ? ' - ' . $data['payment_note'] : ''),
                    ];
                    
                    AccountTransaction::createAccountTransaction($account_transaction_data);
                }
            }

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.direct_expense_updated_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Remove the specified direct expense
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $direct_expense = DirectExpense::where('business_id', $business_id)->findOrFail($id);

            DB::beginTransaction();

            // Delete associated account transaction
            AccountTransaction::where('note', 'LIKE', 'Direct Expense: ' . $direct_expense->name . '%')
                ->whereNull('transaction_id')
                ->whereNull('transaction_payment_id')
                ->delete();

            // Delete document if exists
            if ($direct_expense->document && file_exists(public_path('uploads/documents/' . $direct_expense->document))) {
                @unlink(public_path('uploads/documents/' . $direct_expense->document));
            }

            $direct_expense->delete();

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.direct_expense_deleted_success')
            ];

            return response()->json($output);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }
}

