<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Account;
use App\BusinessLocation;
use Modules\AccountingReports\Entities\ChequeBookEntry;
use Yajra\DataTables\Facades\DataTables;
use DB;
use Carbon\Carbon;

class ChequeBookEntryController extends Controller
{
    public function index()
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        
        // Get bank accounts only
        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id')
            ->prepend(__('messages.all'), '');

        return view('accounting-reports::cheque-book.index', compact('accounts'));
    }

    public function create()
    {
        if (!auth()->user()->can('accounting.add_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        
        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id');

        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::cheque-book.create', compact('accounts', 'locations'));
    }

    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.add_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $businessId = auth()->user()->business_id;

            // Validate cheque number uniqueness per account
            $existing = ChequeBookEntry::where('business_id', $businessId)
                ->where('account_id', $request->account_id)
                ->where('cheque_no', $request->cheque_no)
                ->whereNull('deleted_at')
                ->first();

            if ($existing) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', __('accounting-reports::lang.cheque_no_already_exists'));
            }

            $cheque = new ChequeBookEntry();
            $cheque->business_id = $businessId;
            $cheque->account_id = $request->account_id;
            $cheque->cheque_no = $request->cheque_no;
            $cheque->cheque_date = $this->convert_date($request->cheque_date);
            $cheque->amount = $request->amount;
            $cheque->type = $request->type;
            $cheque->status = $request->status ?? 'pending';
            $cheque->payee_name = $request->payee_name;
            $cheque->narration = $request->narration;
            $cheque->reference_no = $request->reference_no;
            $cheque->location_id = $request->location_id;
            $cheque->created_by = auth()->id();
            
            if ($request->has('cleared_date') && !empty($request->cleared_date)) {
                $cheque->cleared_date = $this->convert_date($request->cleared_date);
            }
            
            if ($request->has('bank_statement_ref')) {
                $cheque->bank_statement_ref = $request->bank_statement_ref;
            }

            $cheque->save();

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg' => __('accounting-reports::lang.cheque_entry_added_success')
                ]);
            }

            return redirect()->route('accounting-reports.cheque-book.index')
                ->with('success', __('accounting-reports::lang.cheque_entry_added_success'));

        } catch (\Exception $e) {
            \Log::error('Cheque Entry Store Error: ' . $e->getMessage());
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
                ], 422);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', __('messages.something_went_wrong'));
        }
    }

    public function edit($id)
    {
        if (!auth()->user()->can('accounting.edit_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        $cheque = ChequeBookEntry::findOrFail($id);
        $businessId = auth()->user()->business_id;

        if ($cheque->business_id != $businessId) {
            abort(403, 'Unauthorized action.');
        }

        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id');

        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::cheque-book.edit', compact('cheque', 'accounts', 'locations'));
    }

    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.edit_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $cheque = ChequeBookEntry::findOrFail($id);
            $businessId = auth()->user()->business_id;

            if ($cheque->business_id != $businessId) {
                abort(403, 'Unauthorized action.');
            }

            // Validate cheque number uniqueness (excluding current entry)
            if ($cheque->cheque_no != $request->cheque_no || $cheque->account_id != $request->account_id) {
                $existing = ChequeBookEntry::where('business_id', $businessId)
                    ->where('account_id', $request->account_id)
                    ->where('cheque_no', $request->cheque_no)
                    ->where('id', '!=', $id)
                    ->whereNull('deleted_at')
                    ->first();

                if ($existing) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', __('accounting-reports::lang.cheque_no_already_exists'));
                }
            }

            $cheque->account_id = $request->account_id;
            $cheque->cheque_no = $request->cheque_no;
            $cheque->cheque_date = $this->convert_date($request->cheque_date);
            $cheque->amount = $request->amount;
            $cheque->type = $request->type;
            $cheque->status = $request->status;
            $cheque->payee_name = $request->payee_name;
            $cheque->narration = $request->narration;
            $cheque->reference_no = $request->reference_no;
            $cheque->location_id = $request->location_id;
            
            if ($request->has('cleared_date') && !empty($request->cleared_date)) {
                $cheque->cleared_date = $this->convert_date($request->cleared_date);
            } else {
                $cheque->cleared_date = null;
            }
            
            if ($request->has('bounced_date') && !empty($request->bounced_date)) {
                $cheque->bounced_date = $this->convert_date($request->bounced_date);
            }
            
            if ($request->has('bounce_reason')) {
                $cheque->bounce_reason = $request->bounce_reason;
            }
            
            if ($request->has('bank_statement_ref')) {
                $cheque->bank_statement_ref = $request->bank_statement_ref;
            }
            
            if ($request->has('bank_statement_date') && !empty($request->bank_statement_date)) {
                $cheque->bank_statement_date = $this->convert_date($request->bank_statement_date);
            }

            $cheque->save();

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg' => __('accounting-reports::lang.cheque_entry_updated_success')
                ]);
            }

            return redirect()->route('accounting-reports.cheque-book.index')
                ->with('success', __('accounting-reports::lang.cheque_entry_updated_success'));

        } catch (\Exception $e) {
            \Log::error('Cheque Entry Update Error: ' . $e->getMessage());
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
                ], 422);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', __('messages.something_went_wrong'));
        }
    }

    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.delete_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $cheque = ChequeBookEntry::findOrFail($id);
            $businessId = auth()->user()->business_id;

            if ($cheque->business_id != $businessId) {
                abort(403, 'Unauthorized action.');
            }

            $cheque->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg' => __('accounting-reports::lang.cheque_entry_deleted_success')
                ]);
            }

            return redirect()->route('accounting-reports.cheque-book.index')
                ->with('success', __('accounting-reports::lang.cheque_entry_deleted_success'));

        } catch (\Exception $e) {
            \Log::error('Cheque Entry Delete Error: ' . $e->getMessage());
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->with('error', __('messages.something_went_wrong'));
        }
    }

    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;

        $query = ChequeBookEntry::where('business_id', $businessId)
            ->with(['account', 'location', 'createdBy']);

        if ($request->has('account_id') && !empty($request->account_id)) {
            $query->where('account_id', $request->account_id);
        }

        if ($request->has('type') && !empty($request->type)) {
            $query->where('type', $request->type);
        }

        if ($request->has('status') && !empty($request->status)) {
            $query->where('status', $request->status);
        }

        if ($request->has('start_date') && !empty($request->start_date)) {
            $query->whereDate('cheque_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && !empty($request->end_date)) {
            $query->whereDate('cheque_date', '<=', $request->end_date);
        }

        return DataTables::of($query)
            ->addColumn('account_name', function ($row) {
                return $row->account->name ?? '-';
            })
            ->addColumn('location_name', function ($row) {
                return $row->location->name ?? '-';
            })
            ->addColumn('created_by_name', function ($row) {
                return $row->createdBy ? $row->createdBy->user_full_name : '-';
            })
            ->editColumn('cheque_date', function ($row) {
                return $row->cheque_date ? \Carbon\Carbon::parse($row->cheque_date)->format('d/m/Y') : '-';
            })
            ->editColumn('amount', function ($row) {
                return number_format($row->amount, 2);
            })
            ->editColumn('type', function ($row) {
                return ucfirst($row->type);
            })
            ->editColumn('status', function ($row) {
                $colors = [
                    'pending' => 'warning',
                    'cleared' => 'success',
                    'bounced' => 'danger',
                    'cancelled' => 'default',
                    'deposited' => 'info',
                ];
                $color = $colors[$row->status] ?? 'default';
                return '<span class="label label-' . $color . '">' . ucfirst($row->status) . '</span>';
            })
            ->addColumn('action', function ($row) {
                $html = '';
                
                if (auth()->user()->can('accounting.edit_cheque_entry')) {
                    $html .= '<a href="' . route('accounting-reports.cheque-book.edit', [$row->id]) . '" class="btn btn-xs btn-primary">
                        <i class="fa fa-edit"></i> ' . __('messages.edit') . '
                    </a> ';
                }
                
                if (auth()->user()->can('accounting.delete_cheque_entry')) {
                    $html .= '<button type="button" class="btn btn-xs btn-danger delete_cheque_entry" data-href="' . route('accounting-reports.cheque-book.destroy', [$row->id]) . '">
                        <i class="fa fa-trash"></i> ' . __('messages.delete') . '
                    </button>';
                }
                
                return $html;
            })
            ->rawColumns(['status', 'action'])
            ->make(true);
    }

    private function convert_date($date)
    {
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            return $date;
        }
        
        try {
            $dt = Carbon::createFromFormat('d/m/Y', $date);
            return $dt->format('Y-m-d');
        } catch (\Exception $e) {
            try {
                $dt = Carbon::createFromFormat('Y-m-d', $date);
                return $dt->format('Y-m-d');
            } catch (\Exception $e2) {
                return $date;
            }
        }
    }
}

