<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Account;
use App\BusinessLocation;
use Modules\AccountingReports\Entities\ChequeBook;
use Modules\AccountingReports\Entities\ChequeBookEntry;
use Yajra\DataTables\Facades\DataTables;
use DB;
use Carbon\Carbon;

class ChequeBookController extends Controller
{
    public function index()
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        
        // Get bank accounts only
        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id')
            ->prepend(__('messages.all'), '');

        return view('accounting-reports::cheque-books.index', compact('accounts'));
    }

    public function create()
    {
        if (!auth()->user()->can('accounting.add_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        
        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id');

        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::cheque-books.create', compact('accounts', 'locations'));
    }

    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.add_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $businessId = auth()->user()->business_id;

            // Validate cheque number range
            $start_no = (int) $request->start_cheque_no;
            $end_no = (int) $request->end_cheque_no;
            
            if ($start_no >= $end_no) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', __('accounting-reports::lang.start_cheque_no_must_be_less_than_end'));
            }

            $total_cheques = $end_no - $start_no + 1;
            
            // Check if cheque book name already exists
            $existing = ChequeBook::where('business_id', $businessId)
                ->where('account_id', $request->account_id)
                ->where('book_name', $request->book_name)
                ->whereNull('deleted_at')
                ->first();

            if ($existing) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', __('accounting-reports::lang.cheque_book_name_already_exists'));
            }

            $chequeBook = new ChequeBook();
            $chequeBook->business_id = $businessId;
            $chequeBook->account_id = $request->account_id;
            $chequeBook->book_name = $request->book_name;
            $chequeBook->book_number = $request->book_number;
            $chequeBook->start_cheque_no = str_pad($start_no, strlen($request->start_cheque_no), '0', STR_PAD_LEFT);
            $chequeBook->end_cheque_no = str_pad($end_no, strlen($request->end_cheque_no), '0', STR_PAD_LEFT);
            $chequeBook->total_cheques = $total_cheques;
            $chequeBook->status = 'active';
            $chequeBook->issue_date = $request->issue_date ? $this->convert_date($request->issue_date) : null;
            $chequeBook->expiry_date = $request->expiry_date ? $this->convert_date($request->expiry_date) : null;
            $chequeBook->location_id = $request->location_id;
            $chequeBook->notes = $request->notes;
            $chequeBook->created_by = auth()->id();
            $chequeBook->save();

            // Optionally generate all cheque entries
            if ($request->has('generate_entries') && $request->generate_entries == '1') {
                $this->generateChequeEntries($chequeBook);
            }

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg' => __('accounting-reports::lang.cheque_book_added_success'),
                    'redirect' => route('accounting-reports.cheque-books.show', $chequeBook->id)
                ]);
            }

            return redirect()->route('accounting-reports.cheque-books.show', $chequeBook->id)
                ->with('success', __('accounting-reports::lang.cheque_book_added_success'));

        } catch (\Exception $e) {
            \Log::error('Cheque Book Store Error: ' . $e->getMessage());
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
                ], 422);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', __('messages.something_went_wrong'));
        }
    }

    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $chequeBook = ChequeBook::with(['account', 'location', 'chequeEntries'])
            ->findOrFail($id);
        $businessId = auth()->user()->business_id;

        if ($chequeBook->business_id != $businessId) {
            abort(403, 'Unauthorized action.');
        }

        // Get all cheque numbers in sequence
        $chequeSequence = $chequeBook->getChequeNumberSequence();
        
        // Get used cheque numbers
        $usedCheques = $chequeBook->chequeEntries()
            ->pluck('cheque_no')
            ->toArray();

        return view('accounting-reports::cheque-books.show', compact('chequeBook', 'chequeSequence', 'usedCheques'));
    }

    public function edit($id)
    {
        if (!auth()->user()->can('accounting.edit_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        $chequeBook = ChequeBook::findOrFail($id);
        $businessId = auth()->user()->business_id;

        if ($chequeBook->business_id != $businessId) {
            abort(403, 'Unauthorized action.');
        }

        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id');

        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::cheque-books.edit', compact('chequeBook', 'accounts', 'locations'));
    }

    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.edit_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $chequeBook = ChequeBook::findOrFail($id);
            $businessId = auth()->user()->business_id;

            if ($chequeBook->business_id != $businessId) {
                abort(403, 'Unauthorized action.');
            }

            // Don't allow changing cheque range if entries exist
            if ($chequeBook->chequeEntries()->count() > 0) {
                if ($chequeBook->start_cheque_no != $request->start_cheque_no || 
                    $chequeBook->end_cheque_no != $request->end_cheque_no) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', __('accounting-reports::lang.cannot_change_cheque_range_if_entries_exist'));
                }
            } else {
                // Validate new range
                $start_no = (int) $request->start_cheque_no;
                $end_no = (int) $request->end_cheque_no;
                
                if ($start_no >= $end_no) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', __('accounting-reports::lang.start_cheque_no_must_be_less_than_end'));
                }

                $total_cheques = $end_no - $start_no + 1;
                $chequeBook->start_cheque_no = str_pad($start_no, strlen($request->start_cheque_no), '0', STR_PAD_LEFT);
                $chequeBook->end_cheque_no = str_pad($end_no, strlen($request->end_cheque_no), '0', STR_PAD_LEFT);
                $chequeBook->total_cheques = $total_cheques;
            }

            $chequeBook->book_name = $request->book_name;
            $chequeBook->book_number = $request->book_number;
            $chequeBook->account_id = $request->account_id;
            $chequeBook->status = $request->status;
            $chequeBook->issue_date = $request->issue_date ? $this->convert_date($request->issue_date) : null;
            $chequeBook->expiry_date = $request->expiry_date ? $this->convert_date($request->expiry_date) : null;
            $chequeBook->location_id = $request->location_id;
            $chequeBook->notes = $request->notes;
            $chequeBook->save();

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg' => __('accounting-reports::lang.cheque_book_updated_success')
                ]);
            }

            return redirect()->route('accounting-reports.cheque-books.show', $chequeBook->id)
                ->with('success', __('accounting-reports::lang.cheque_book_updated_success'));

        } catch (\Exception $e) {
            \Log::error('Cheque Book Update Error: ' . $e->getMessage());
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
                ], 422);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', __('messages.something_went_wrong'));
        }
    }

    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.delete_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $chequeBook = ChequeBook::findOrFail($id);
            $businessId = auth()->user()->business_id;

            if ($chequeBook->business_id != $businessId) {
                abort(403, 'Unauthorized action.');
            }

            // Don't allow deletion if cheque entries exist
            if ($chequeBook->chequeEntries()->count() > 0) {
                if (request()->ajax()) {
                    return response()->json([
                        'success' => false,
                        'msg' => __('accounting-reports::lang.cannot_delete_cheque_book_with_entries')
                    ], 422);
                }
                return redirect()->back()
                    ->with('error', __('accounting-reports::lang.cannot_delete_cheque_book_with_entries'));
            }

            $chequeBook->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg' => __('accounting-reports::lang.cheque_book_deleted_success')
                ]);
            }

            return redirect()->route('accounting-reports.cheque-books.index')
                ->with('success', __('accounting-reports::lang.cheque_book_deleted_success'));

        } catch (\Exception $e) {
            \Log::error('Cheque Book Delete Error: ' . $e->getMessage());
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->with('error', __('messages.something_went_wrong'));
        }
    }

    /**
     * Generate all cheque entries from cheque book
     */
    public function generateEntries($id)
    {
        if (!auth()->user()->can('accounting.add_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $chequeBook = ChequeBook::findOrFail($id);
            $businessId = auth()->user()->business_id;

            if ($chequeBook->business_id != $businessId) {
                abort(403, 'Unauthorized action.');
            }

            $generated = $this->generateChequeEntries($chequeBook);

            return response()->json([
                'success' => true,
                'msg' => __('accounting-reports::lang.cheque_entries_generated_success', ['count' => $generated])
            ]);

        } catch (\Exception $e) {
            \Log::error('Generate Cheque Entries Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate cheque entries for a cheque book
     */
    protected function generateChequeEntries($chequeBook)
    {
        $sequence = $chequeBook->getChequeNumberSequence();
        $existing = $chequeBook->chequeEntries()
            ->pluck('cheque_no')
            ->toArray();
        
        $generated = 0;
        
        foreach ($sequence as $cheque_no) {
            if (!in_array($cheque_no, $existing)) {
                $entry = new ChequeBookEntry();
                $entry->business_id = $chequeBook->business_id;
                $entry->account_id = $chequeBook->account_id;
                $entry->cheque_book_id = $chequeBook->id;
                $entry->cheque_no = $cheque_no;
                $entry->cheque_date = $chequeBook->issue_date ?? Carbon::now();
                $entry->amount = 0; // Will be updated when used
                $entry->type = 'issued';
                $entry->status = 'pending';
                $entry->location_id = $chequeBook->location_id;
                $entry->created_by = auth()->id();
                $entry->save();
                $generated++;
            }
        }
        
        return $generated;
    }

    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;

        $query = ChequeBook::where('business_id', $businessId)
            ->with(['account', 'location', 'createdBy']);

        if ($request->has('account_id') && !empty($request->account_id)) {
            $query->where('account_id', $request->account_id);
        }

        if ($request->has('status') && !empty($request->status)) {
            $query->where('status', $request->status);
        }

        return DataTables::of($query)
            ->addColumn('account_name', function ($row) {
                return $row->account->name ?? '-';
            })
            ->addColumn('location_name', function ($row) {
                return $row->location->name ?? '-';
            })
            ->addColumn('created_by_name', function ($row) {
                return $row->createdBy ? $row->createdBy->user_full_name : '-';
            })
            ->addColumn('cheque_range', function ($row) {
                return $row->start_cheque_no . ' - ' . $row->end_cheque_no;
            })
            ->addColumn('used_cheques', function ($row) {
                return $row->used_cheques;
            })
            ->addColumn('remaining_cheques', function ($row) {
                return $row->remaining_cheques;
            })
            ->editColumn('issue_date', function ($row) {
                return $row->issue_date ? \Carbon\Carbon::parse($row->issue_date)->format('d/m/Y') : '-';
            })
            ->editColumn('status', function ($row) {
                $colors = [
                    'active' => 'success',
                    'completed' => 'info',
                    'cancelled' => 'default',
                ];
                $color = $colors[$row->status] ?? 'default';
                return '<span class="label label-' . $color . '">' . ucfirst($row->status) . '</span>';
            })
            ->addColumn('action', function ($row) {
                $html = '';
                
                $html .= '<a href="' . route('accounting-reports.cheque-books.show', [$row->id]) . '" class="btn btn-xs btn-info">
                    <i class="fa fa-eye"></i> ' . __('messages.view') . '
                </a> ';
                
                if (auth()->user()->can('accounting.edit_cheque_entry')) {
                    $html .= '<a href="' . route('accounting-reports.cheque-books.edit', [$row->id]) . '" class="btn btn-xs btn-primary">
                        <i class="fa fa-edit"></i> ' . __('messages.edit') . '
                    </a> ';
                }
                
                if (auth()->user()->can('accounting.add_cheque_entry')) {
                    $html .= '<button type="button" class="btn btn-xs btn-success generate-entries" data-href="' . route('accounting-reports.cheque-books.generate-entries', [$row->id]) . '" data-book-name="' . htmlspecialchars($row->book_name, ENT_QUOTES) . '">
                        <i class="fa fa-plus"></i> ' . __('accounting-reports::lang.generate_entries') . '
                    </button> ';
                }
                
                if (auth()->user()->can('accounting.delete_cheque_entry')) {
                    $html .= '<button type="button" class="btn btn-xs btn-danger delete_cheque_book" data-href="' . route('accounting-reports.cheque-books.destroy', [$row->id]) . '">
                        <i class="fa fa-trash"></i> ' . __('messages.delete') . '
                    </button>';
                }
                
                return $html;
            })
            ->rawColumns(['status', 'action'])
            ->make(true);
    }

    private function convert_date($date)
    {
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            return $date;
        }
        
        try {
            $dt = Carbon::createFromFormat('d/m/Y', $date);
            return $dt->format('Y-m-d');
        } catch (\Exception $e) {
            try {
                $dt = Carbon::createFromFormat('Y-m-d', $date);
                return $dt->format('Y-m-d');
            } catch (\Exception $e2) {
                return $date;
            }
        }
    }
}

