<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\BusinessLocation;
use App\Account;
use App\User;
use App\Utils\TransactionUtil;
use App\Utils\Util;
use Modules\AccountingReports\Services\BankbookQuery;
use Modules\AccountingReports\Entities\BankReconciliation;
use Yajra\DataTables\Facades\DataTables;
use DB;
use Carbon\Carbon;

class BankBookController extends Controller
{
    protected $transactionUtil;
    protected $commonUtil;
    protected $bankbookQuery;

    public function __construct(TransactionUtil $transactionUtil, Util $commonUtil, BankbookQuery $bankbookQuery)
    {
        $this->transactionUtil = $transactionUtil;
        $this->commonUtil = $commonUtil;
        $this->bankbookQuery = $bankbookQuery;
    }

    public function index()
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($businessId, true);
        
        // Get bank accounts only
        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id')
            ->prepend(__('messages.all'), '');
        
        $users = User::forDropdown($businessId, false, true, false);

        return view('accounting-reports::bank-book.index', compact(
            'locations',
            'accounts',
            'users'
        ));
    }

    /**
     * Get Bank Book data in Tally-style format
     * Shows only bank account transactions with deposits (debit) and withdrawals (credit)
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;

            // Handle date range input
            $start_date = null;
            $end_date = null;
            
            if ($request->has('start_date') && !empty($request->input('start_date'))) {
                $date_input = trim($request->input('start_date'));
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_input)) {
                    $start_date = $date_input;
                } else {
                    try {
                        $start_date = $this->transactionUtil->uf_date($date_input);
                        if (!$start_date) {
                            $start_date = Carbon::now()->format('Y-m-d');
                        }
                    } catch (\Exception $e) {
                        \Log::warning('Bank Book start_date parsing error: ' . $e->getMessage());
                        $start_date = Carbon::now()->format('Y-m-d');
                    }
                }
            } else {
                $start_date = Carbon::now()->format('Y-m-d');
            }

            if ($request->has('end_date') && !empty($request->input('end_date'))) {
                $date_input = trim($request->input('end_date'));
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_input)) {
                    $end_date = $date_input;
                } else {
                    try {
                        $end_date = $this->transactionUtil->uf_date($date_input);
                        if (!$end_date) {
                            $end_date = Carbon::now()->format('Y-m-d');
                        }
                    } catch (\Exception $e) {
                        \Log::warning('Bank Book end_date parsing error: ' . $e->getMessage());
                        $end_date = Carbon::now()->format('Y-m-d');
                    }
                }
            } else {
                $end_date = Carbon::now()->format('Y-m-d');
            }

            // Get filters
            $filters = [
                'location_id' => $request->input('location_id'),
                'user_id' => $request->input('user_id'),
                'account_id' => $request->input('account_id'),
                'permitted_locations' => auth()->user()->permitted_locations(),
            ];

            // Get bankbook data
            $bankbook_data = $this->bankbookQuery->getBankbookEntries(
                $business_id,
                $start_date,
                $end_date,
                $filters
            );

            $entries = $bankbook_data['entries'];
            $opening_balance = $bankbook_data['summary']['opening_balance'];
            
            // Calculate current period totals
            $current_period_debit = 0;
            $current_period_credit = 0;
            
            foreach ($entries as $entry) {
                $type = $entry['type'] ?? '';
                $amount = floatval($entry['amount'] ?? 0);
                
                if ($type == 'debit') {
                    $current_period_debit += $amount;
                } else {
                    $current_period_credit += $amount;
                }
            }
            
            // Current Total = Current Period + Opening Balance
            if (abs($opening_balance) > 0.01 && $start_date) {
                $current_period_debit += abs($opening_balance);
            }
            
            // Get closing balance
            $closing_balance = 0;
            if (!empty($entries)) {
                $last_entry = end($entries);
                $closing_balance = $last_entry['running_balance'] ?? $opening_balance;
            } else {
                $closing_balance = $opening_balance;
            }

            return DataTables::of($entries)
                ->with([
                    'opening_balance' => $opening_balance,
                    'current_period_debit' => $current_period_debit,
                    'current_period_credit' => $current_period_credit,
                    'closing_balance' => $closing_balance
                ])
                ->editColumn('datetime', function ($row) {
                    $datetime = $row['datetime'] ?? null;
                    return $datetime ? $this->commonUtil->format_date($datetime, true) : '-';
                })
                ->editColumn('voucher_no', function ($row) {
                    $voucher_no = $row['voucher_no'] ?? '';
                    $transaction_id = $row['transaction_id'] ?? null;
                    $transaction_payment_id = $row['transaction_payment_id'] ?? null;
                    $cheque_no = $row['cheque_no'] ?? null;
                    $module = $row['module'] ?? '';
                    
                    if ($transaction_id || $transaction_payment_id) {
                        $data_attr = '';
                        if ($transaction_id) {
                            $data_attr = 'data-transaction-id="' . $transaction_id . '" data-module="' . htmlspecialchars($module, ENT_QUOTES) . '"';
                        } elseif ($transaction_payment_id) {
                            $data_attr = 'data-payment-id="' . $transaction_payment_id . '" data-module="' . htmlspecialchars($module, ENT_QUOTES) . '"';
                        }
                        $display = $voucher_no;
                        if ($cheque_no) {
                            $display .= ' <small style="color: #666;">(' . $cheque_no . ')</small>';
                        }
                        return '<a href="#" class="voucher-details-link" ' . $data_attr . ' style="cursor: pointer; color: #0066cc;">' . $display . '</a>';
                    }
                    return $voucher_no;
                })
                ->editColumn('module', function ($row) {
                    $module = $row['module'] ?? '';
                    return ucfirst(str_replace('_', ' ', $module));
                })
                ->editColumn('party', function ($row) {
                    return $row['party'] ?? '-';
                })
                ->editColumn('location', function ($row) {
                    return $row['location'] ?? '-';
                })
                ->editColumn('account', function ($row) {
                    return $row['account'] ?? '-';
                })
                ->editColumn('debit', function ($row) {
                    $type = $row['type'] ?? '';
                    $amount = floatval($row['amount'] ?? 0);
                    return $type == 'debit' ? $this->commonUtil->num_f($amount) : '-';
                })
                ->editColumn('credit', function ($row) {
                    $type = $row['type'] ?? '';
                    $amount = floatval($row['amount'] ?? 0);
                    return $type == 'credit' ? $this->commonUtil->num_f($amount) : '-';
                })
                ->editColumn('cheque_no', function ($row) {
                    $cheque_no = $row['cheque_no'] ?? '';
                    $cheque_status = $row['cheque_status'] ?? '';
                    if ($cheque_no) {
                        $status_color = $cheque_status == 'cleared' ? 'green' : ($cheque_status == 'bounced' ? 'red' : 'orange');
                        return '<span style="color: ' . $status_color . ';">' . $cheque_no . '</span>';
                    }
                    return '-';
                })
                ->editColumn('is_reconciled', function ($row) {
                    $is_reconciled = $row['is_reconciled'] ?? false;
                    return $is_reconciled ? '<span class="label label-success">✓</span>' : '<span class="label label-default">-</span>';
                })
                ->editColumn('narration', function ($row) {
                    return $row['narration'] ?? '-';
                })
                ->editColumn('user', function ($row) {
                    return $row['user'] ?? '-';
                })
                ->rawColumns(['voucher_no', 'module', 'debit', 'credit', 'cheque_no', 'is_reconciled'])
                ->make(true);

        } catch (\Exception $e) {
            \Log::error('Bank Book Error: ' . $e->getMessage());
            \Log::error('Bank Book Trace: ' . $e->getTraceAsString());
            
            $draw = intval($request->get('draw', 1));
            return response()->json([
                'draw' => $draw,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading bank book data: ' . $e->getMessage()
            ], 200);
        }
    }

    /**
     * Bank Reconciliation page
     */
    public function reconciliation($accountId)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $account = Account::findOrFail($accountId);
        $businessId = auth()->user()->business_id;

        if ($account->business_id != $businessId) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::bank-book.reconciliation', compact('account'));
    }

    /**
     * Get reconciliation data
     */
    public function getReconciliationData(Request $request, $accountId)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $account = Account::findOrFail($accountId);
            
            if ($account->business_id != $business_id) {
                abort(403, 'Unauthorized action.');
            }

            $end_date = $request->input('end_date', Carbon::now()->format('Y-m-d'));
            
            // Get unreconciled transactions
            $start_date = $request->input('start_date', Carbon::now()->subMonths(3)->format('Y-m-d'));
            
            $filters = [
                'location_id' => $request->input('location_id'),
                'user_id' => $request->input('user_id'),
                'account_id' => $accountId,
                'permitted_locations' => auth()->user()->permitted_locations(),
            ];

            $bankbook_data = $this->bankbookQuery->getBankbookEntries(
                $business_id,
                $start_date,
                $end_date,
                $filters
            );

            $entries = $bankbook_data['entries'];
            
            // Filter reconciled/unreconciled
            $unreconciled = array_filter($entries, function($entry) {
                return !($entry['is_reconciled'] ?? false);
            });

            return response()->json([
                'success' => true,
                'unreconciled' => array_values($unreconciled),
                'all_entries' => $entries,
            ]);

        } catch (\Exception $e) {
            \Log::error('Bank Reconciliation Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => 'Error loading reconciliation data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark transaction as reconciled
     */
    public function markReconciled(Request $request)
    {
        if (!auth()->user()->can('accounting.reconcile_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $transaction_payment_id = $request->input('transaction_payment_id');
            $cheque_entry_id = $request->input('cheque_entry_id');
            $cleared_date = $request->input('cleared_date', Carbon::now()->format('Y-m-d'));
            $bank_statement_ref = $request->input('bank_statement_ref');

            if ($cheque_entry_id) {
                // Update cheque entry
                $cheque = \Modules\AccountingReports\Entities\ChequeBookEntry::findOrFail($cheque_entry_id);
                $cheque->status = 'cleared';
                $cheque->cleared_date = $cleared_date;
                $cheque->bank_statement_ref = $bank_statement_ref;
                $cheque->save();
            } else {
                // Create reconciliation entry
                // This would link to journal entry lines if using full accounting module
                // For now, we'll just track in cheque book or a separate reconciliation table
            }

            return response()->json([
                'success' => true,
                'msg' => __('accounting-reports::lang.reconciliation_marked_success')
            ]);

        } catch (\Exception $e) {
            \Log::error('Mark Reconciled Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }

    public function export(Request $request)
    {
        // TODO: Implement export
        return redirect()->back();
    }
}


