<?php

namespace Modules\AccountingReports\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class ReceivablePayable extends Model
{
    use SoftDeletes;

    protected $table = 'ar_receivables_payables';

    protected $guarded = ['id'];

    protected $casts = [
        'document_date' => 'date',
        'due_date' => 'date',
        'invoice_amount' => 'decimal:4',
        'allocated_amount' => 'decimal:4',
        'balance' => 'decimal:4',
        'discount_amount' => 'decimal:4',
        'tax_amount' => 'decimal:4',
        'exchange_rate' => 'decimal:6',
    ];

    /**
     * Relationships
     */
    public function business()
    {
        return $this->belongsTo(\App\Business::class, 'business_id');
    }

    public function contact()
    {
        return $this->belongsTo(\App\Contact::class, 'contact_id');
    }

    public function transaction()
    {
        return $this->belongsTo(\App\Transaction::class, 'transaction_id');
    }

    public function journalEntry()
    {
        return $this->belongsTo(JournalEntryHeader::class, 'journal_entry_id');
    }

    /**
     * Update balance and aging
     */
    public function updateBalance()
    {
        $this->balance = $this->invoice_amount - $this->allocated_amount;
        
        if ($this->balance <= 0.01) {
            $this->status = 'paid';
        } elseif ($this->allocated_amount > 0) {
            $this->status = 'partially_paid';
        } else {
            $this->status = 'open';
        }

        // Calculate days past due
        if ($this->due_date && $this->balance > 0) {
            $this->days_past_due = max(0, now()->diffInDays($this->due_date, false));
            
            // Set aging bucket
            if ($this->days_past_due <= 0) {
                $this->aging_bucket = 'current';
            } elseif ($this->days_past_due <= 7) {
                $this->aging_bucket = 'b0_7';
            } elseif ($this->days_past_due <= 30) {
                $this->aging_bucket = 'b8_30';
            } elseif ($this->days_past_due <= 60) {
                $this->aging_bucket = 'b31_60';
            } elseif ($this->days_past_due <= 90) {
                $this->aging_bucket = 'b61_90';
            } else {
                $this->aging_bucket = 'b90_plus';
            }
        }
    }
}


