<?php

namespace Modules\AccountingReports\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class JournalEntryHeader extends Model
{
    use SoftDeletes;

    protected $table = 'ar_journal_entry_headers';

    protected $guarded = ['id'];

    protected $casts = [
        'voucher_date' => 'date',
        'transaction_date' => 'datetime',
        'total_debit' => 'decimal:4',
        'total_credit' => 'decimal:4',
        'balance_diff' => 'decimal:4',
        'exchange_rate' => 'decimal:6',
        'is_balanced' => 'boolean',
        'is_reversed' => 'boolean',
        'posted_at' => 'datetime',
    ];

    /**
     * Relationships
     */
    public function business()
    {
        return $this->belongsTo(\App\Business::class, 'business_id');
    }

    public function location()
    {
        return $this->belongsTo(\App\BusinessLocation::class, 'location_id');
    }

    public function sourceTransaction()
    {
        return $this->belongsTo(\App\Transaction::class, 'source_transaction_id');
    }

    public function journalEntryLines()
    {
        return $this->hasMany(JournalEntryLine::class, 'journal_entry_id');
    }

    public function reversedBy()
    {
        return $this->belongsTo(JournalEntryHeader::class, 'reversed_by_journal_id');
    }

    public function createdBy()
    {
        return $this->belongsTo(\App\User::class, 'created_by');
    }

    public function postedBy()
    {
        return $this->belongsTo(\App\User::class, 'posted_by');
    }

    /**
     * Scopes
     */
    public function scopePosted($query)
    {
        return $query->where('status', 'posted');
    }

    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    public function scopeByDateRange($query, $fromDate, $toDate)
    {
        return $query->whereBetween('voucher_date', [$fromDate, $toDate]);
    }

    public function scopeBySourceModule($query, $module)
    {
        return $query->where('source_module', $module);
    }

    /**
     * Validate that debit equals credit
     */
    public function validateBalance()
    {
        $totalDebit = $this->journalEntryLines()->sum('debit');
        $totalCredit = $this->journalEntryLines()->sum('credit');
        
        $this->total_debit = $totalDebit;
        $this->total_credit = $totalCredit;
        $this->balance_diff = $totalDebit - $totalCredit;
        $this->is_balanced = abs($this->balance_diff) < 0.01; // Allow rounding differences
        
        return $this->is_balanced;
    }

    /**
     * Post the journal entry
     */
    public function post()
    {
        if (!$this->validateBalance()) {
            throw new \Exception('Journal entry does not balance. Debit: ' . $this->total_debit . ', Credit: ' . $this->total_credit);
        }

        $this->status = 'posted';
        $this->posted_by = auth()->id();
        $this->posted_at = now();
        $this->save();
    }

    /**
     * Reverse this journal entry
     */
    public function reverse()
    {
        if ($this->is_reversed) {
            throw new \Exception('Journal entry is already reversed');
        }

        $reversal = $this->replicate();
        $reversal->voucher_no = $this->voucher_no . '-REV';
        $reversal->status = 'posted';
        $reversal->is_reversed = true;
        $reversal->reversed_by_journal_id = $this->id;
        $reversal->narration = 'Reversal of ' . $this->voucher_no;
        $reversal->posted_by = auth()->id();
        $reversal->posted_at = now();
        $reversal->save();

        // Reverse all lines
        foreach ($this->journalEntryLines as $line) {
            $reversedLine = $line->replicate();
            $reversedLine->journal_entry_id = $reversal->id;
            $reversedLine->debit = $line->credit;
            $reversedLine->credit = $line->debit;
            $reversedLine->save();
        }

        $this->is_reversed = true;
        $this->status = 'reversed';
        $this->save();

        $reversal->validateBalance();
        $reversal->save();

        return $reversal;
    }
}


