<?php

namespace Modules\AccountingReports\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class ChartOfAccount extends Model
{
    use SoftDeletes;

    protected $table = 'ar_chart_of_accounts';

    protected $guarded = ['id'];

    protected $casts = [
        'is_control_account' => 'boolean',
        'is_active' => 'boolean',
        'is_closed' => 'boolean',
        'allow_negative_balance' => 'boolean',
        'requires_contact' => 'boolean',
    ];

    /**
     * Relationships
     */
    public function business()
    {
        return $this->belongsTo(\App\Business::class, 'business_id');
    }

    public function parent()
    {
        return $this->belongsTo(ChartOfAccount::class, 'parent_id');
    }

    public function children()
    {
        return $this->hasMany(ChartOfAccount::class, 'parent_id');
    }

    public function account()
    {
        return $this->belongsTo(\App\Account::class, 'account_id');
    }

    public function journalEntryLines()
    {
        return $this->hasMany(JournalEntryLine::class, 'account_id');
    }

    public function createdBy()
    {
        return $this->belongsTo(\App\User::class, 'created_by');
    }

    public function updatedBy()
    {
        return $this->belongsTo(\App\User::class, 'updated_by');
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeNotClosed($query)
    {
        return $query->where('is_closed', false);
    }

    public function scopeByGroup($query, $group)
    {
        return $query->where('account_group', $group);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('account_type', $type);
    }

    public function scopeControlAccounts($query)
    {
        return $query->where('is_control_account', true);
    }

    /**
     * Get account balance (Dr - Cr) up to a specific date
     */
    public function getBalance($asOfDate = null, $locationId = null)
    {
        $query = JournalEntryLine::where('account_id', $this->id)
            ->whereHas('journalEntry', function($q) use ($asOfDate) {
                if ($asOfDate) {
                    $q->where('voucher_date', '<=', $asOfDate);
                }
                $q->where('status', 'posted');
            });

        if ($locationId) {
            $query->whereHas('journalEntry', function($q) use ($locationId) {
                $q->where('location_id', $locationId);
            });
        }

        $result = $query->selectRaw('SUM(debit) as total_debit, SUM(credit) as total_credit')
            ->first();

        return ($result->total_debit ?? 0) - ($result->total_credit ?? 0);
    }
}


