<?php
/**
 * Business Management Module - Stock Register Controller
 * 
 * Developer: Posify
 * Website: www.posifyme.com
 * WhatsApp: +8801675444143
 */

namespace Modules\BusinessManagement\Http\Controllers;

use App\Http\Controllers\Controller;
use App\VariationLocationDetails;
use App\BusinessLocation;
use App\Product;
use App\Category;
use App\Brands;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use App\Utils\Util;
use Illuminate\Support\Facades\DB;

class StockRegisterController extends Controller
{
    /**
     * Display a listing of the stock register.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id);

        if ($request->ajax()) {
            $stock = VariationLocationDetails::join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->join('business_locations as bl', 'variation_location_details.location_id', '=', 'bl.id')
                ->where('p.business_id', $business_id)
                ->select(
                    'p.name as product_name',
                    'p.sku',
                    'v.sub_sku',
                    'bl.name as location_name',
                    'variation_location_details.qty_available',
                    'variation_location_details.stock_value'
                );

            // Apply location filter based on user permissions
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $stock->whereIn('variation_location_details.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $stock->where('variation_location_details.location_id', $request->location_id);
            }

            if (!empty($request->product_name)) {
                $stock->where('p.name', 'like', '%' . $request->product_name . '%');
            }

            $util = new Util();
            
            return DataTables::of($stock)
                ->editColumn('qty_available', function ($row) use ($util) {
                    return '<span class="display_currency" data-currency_symbol="false">' . $util->num_f($row->qty_available) . '</span>';
                })
                ->editColumn('stock_value', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->stock_value . '</span>';
                })
                ->rawColumns(['qty_available', 'stock_value'])
                ->make(true);
        }

        return view('businessmanagement::stock_register.index', compact('locations'));
    }

    /**
     * Current Stock (Combined) Report
     */
    public function currentStockCombined(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->where('enable_stock', 1)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->pluck('name', 'id')
            ->prepend(__('lang_v1.all'), 'all');

        // Get date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in currentStockCombined: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get stock data
        $query = VariationLocationDetails::join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->leftJoin('business_locations as bl', 'variation_location_details.location_id', '=', 'bl.id')
            ->where('p.business_id', $business_id)
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->where('variation_location_details.qty_available', '>', 0)
            ->select(
                'p.id as product_id',
                'p.name as product_name',
                'p.sku',
                'v.sub_sku',
                'v.id as variation_id',
                'v.default_purchase_price as unit_cost',
                'variation_location_details.qty_available as qty',
                DB::raw('(variation_location_details.qty_available * COALESCE(v.default_purchase_price, 0)) as total_cost'),
                'u.short_name as unit_name',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                'b.id as brand_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                'bl.id as location_id',
                'bl.name as location_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            )
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('variation_location_details.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('variation_location_details.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        // Apply date filter - filter by stock as of date
        // For current stock, we typically show stock as of a specific date
        // This would require calculating stock history, but for now we'll show current stock
        // If date filter is provided, we could filter by last purchase date or similar
        // For simplicity, we'll show current stock regardless of date filter
        // (Date filter might be used for "stock as of date" in future enhancement)

        $stock_data = $query->orderBy('c.name', 'asc')
            ->orderBy('b.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group data by category and brand
        $grouped_data = [];
        foreach ($stock_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            $brand_name = $item->brand_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            if (!isset($grouped_data[$category_name][$brand_name])) {
                $grouped_data[$category_name][$brand_name] = [];
            }
            
            $grouped_data[$category_name][$brand_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = $stock_data->sum('qty');
        $grand_total_cost = $stock_data->sum('total_cost');

        $util = new Util();

        return view('businessmanagement::stock_register.current_stock_combined', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'grand_total_qty',
            'grand_total_cost',
            'util'
        ));
    }

    /**
     * Print Current Stock (Combined) Report
     */
    public function printCurrentStockCombined(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Get date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in printCurrentStockCombined: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get stock data (same query as web view)
        $query = VariationLocationDetails::join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->leftJoin('business_locations as bl', 'variation_location_details.location_id', '=', 'bl.id')
            ->where('p.business_id', $business_id)
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->where('variation_location_details.qty_available', '>', 0)
            ->select(
                'p.id as product_id',
                'p.name as product_name',
                'p.sku',
                'v.sub_sku',
                'v.id as variation_id',
                'v.default_purchase_price as unit_cost',
                'variation_location_details.qty_available as qty',
                DB::raw('(variation_location_details.qty_available * COALESCE(v.default_purchase_price, 0)) as total_cost'),
                'u.short_name as unit_name',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                'b.id as brand_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                'bl.id as location_id',
                'bl.name as location_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            )
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('variation_location_details.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('variation_location_details.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        $stock_data = $query->orderBy('c.name', 'asc')
            ->orderBy('b.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group data by category and brand
        $grouped_data = [];
        foreach ($stock_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            $brand_name = $item->brand_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            if (!isset($grouped_data[$category_name][$brand_name])) {
                $grouped_data[$category_name][$brand_name] = [];
            }
            
            $grouped_data[$category_name][$brand_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = $stock_data->sum('qty');
        $grand_total_cost = $stock_data->sum('total_cost');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::stock_register.print_current_stock_combined', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'grand_total_qty',
            'grand_total_cost',
            'util',
            'business'
        ));
    }

    /**
     * Current Stock (c) Report
     */
    public function currentStockC(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->where('enable_stock', 1)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->pluck('name', 'id')
            ->prepend(__('lang_v1.all'), 'all');

        // Get date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in currentStockC: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get stock data
        $query = VariationLocationDetails::join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->leftJoin('business_locations as bl', 'variation_location_details.location_id', '=', 'bl.id')
            ->where('p.business_id', $business_id)
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->where('variation_location_details.qty_available', '>', 0)
            ->select(
                'p.id as product_id',
                'p.name as product_name',
                'p.sku',
                'v.sub_sku',
                'v.id as variation_id',
                'variation_location_details.qty_available as qty',
                'u.short_name as unit_name',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                'b.id as brand_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                'bl.id as location_id',
                'bl.name as location_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            )
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('variation_location_details.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('variation_location_details.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        $stock_data = $query->orderBy('c.name', 'asc')
            ->orderBy('b.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group data by category and brand
        $grouped_data = [];
        foreach ($stock_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            $brand_name = $item->brand_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            if (!isset($grouped_data[$category_name][$brand_name])) {
                $grouped_data[$category_name][$brand_name] = [];
            }
            
            $grouped_data[$category_name][$brand_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = $stock_data->sum('qty');

        $util = new Util();

        return view('businessmanagement::stock_register.current_stock_c', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'grand_total_qty',
            'util'
        ));
    }

    /**
     * Print Current Stock (c) Report
     */
    public function printCurrentStockC(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Get date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in printCurrentStockC: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get stock data (same query as web view)
        $query = VariationLocationDetails::join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->leftJoin('business_locations as bl', 'variation_location_details.location_id', '=', 'bl.id')
            ->where('p.business_id', $business_id)
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->where('variation_location_details.qty_available', '>', 0)
            ->select(
                'p.id as product_id',
                'p.name as product_name',
                'p.sku',
                'v.sub_sku',
                'v.id as variation_id',
                'variation_location_details.qty_available as qty',
                'u.short_name as unit_name',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                'b.id as brand_id',
                DB::raw("COALESCE(b.name, 'NO BRAND') as brand_name"),
                'bl.id as location_id',
                'bl.name as location_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            )
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('variation_location_details.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('variation_location_details.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        $stock_data = $query->orderBy('c.name', 'asc')
            ->orderBy('b.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group data by category and brand
        $grouped_data = [];
        foreach ($stock_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            $brand_name = $item->brand_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            if (!isset($grouped_data[$category_name][$brand_name])) {
                $grouped_data[$category_name][$brand_name] = [];
            }
            
            $grouped_data[$category_name][$brand_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = $stock_data->sum('qty');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::stock_register.print_current_stock_c', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'grand_total_qty',
            'util',
            'business'
        ));
    }

    /**
     * Current Stock With Serial Report
     */
    public function currentStockWithSerial(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->where('enable_stock', 1)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->pluck('name', 'id')
            ->prepend(__('lang_v1.all'), 'all');

        // Get date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in currentStockWithSerial: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get serial numbers with stock status = 'available'
        // Join with purchase_lines to get purchase invoice number (if exists)
        $query = DB::table('serial_numbers as sn')
            ->join('products as p', 'sn.product_id', '=', 'p.id')
            ->join('variations as v', 'sn.variation_id', '=', 'v.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('purchase_lines as pl', function($join) {
                $join->on('sn.serial_purchase_code', '=', 'pl.serial_purchase_code')
                     ->whereNotNull('sn.serial_purchase_code');
            })
            ->leftJoin('transactions as t', function($join) {
                $join->on('pl.transaction_id', '=', 't.id')
                     ->where('t.type', '=', 'purchase');
            })
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('sn.business_id', $business_id)
            ->where('sn.stock_status', 'available')
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->select(
                'sn.id as serial_id',
                'sn.serial_number',
                'sn.transaction_id',
                'p.id as product_id',
                'p.name as product_name',
                'v.id as variation_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("COALESCE(t.invoice_no, '') as purchase_invoice_no"),
                'sn.location_id'
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('sn.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('sn.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        // Apply date filter - filter by purchase date if transaction exists
        // Show all serial numbers, but filter by purchase date if purchase exists
        if (!empty($start_date) && !empty($end_date)) {
            $query->where(function($q) use ($start_date, $end_date) {
                $q->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                  ->orWhereNull('t.id')
                  ->orWhereNull('pl.id');
            });
        }

        $serial_data = $query->orderBy('c.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->orderBy('sn.serial_number', 'asc')
            ->get();

        // Group data by product description
        $grouped_data = [];
        foreach ($serial_data as $item) {
            $product_key = $item->product_description;
            
            if (!isset($grouped_data[$product_key])) {
                $grouped_data[$product_key] = [];
            }
            
            $grouped_data[$product_key][] = $item;
        }

        // Calculate totals
        $grand_total_qty = count($serial_data);

        $util = new Util();

        return view('businessmanagement::stock_register.current_stock_with_serial', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'grand_total_qty',
            'util'
        ));
    }

    /**
     * Print Current Stock With Serial Report
     */
    public function printCurrentStockWithSerial(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Get date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in printCurrentStockWithSerial: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get serial numbers with stock status = 'available' (same query as web view)
        // Join with purchase_lines to get purchase invoice number (if exists)
        $query = DB::table('serial_numbers as sn')
            ->join('products as p', 'sn.product_id', '=', 'p.id')
            ->join('variations as v', 'sn.variation_id', '=', 'v.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('purchase_lines as pl', function($join) {
                $join->on('sn.serial_purchase_code', '=', 'pl.serial_purchase_code')
                     ->whereNotNull('sn.serial_purchase_code');
            })
            ->leftJoin('transactions as t', function($join) {
                $join->on('pl.transaction_id', '=', 't.id')
                     ->where('t.type', '=', 'purchase');
            })
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('sn.business_id', $business_id)
            ->where('sn.stock_status', 'available')
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->select(
                'sn.id as serial_id',
                'sn.serial_number',
                'sn.transaction_id',
                'p.id as product_id',
                'p.name as product_name',
                'v.id as variation_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("COALESCE(t.invoice_no, '') as purchase_invoice_no"),
                'sn.location_id'
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('sn.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('sn.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        // Apply date filter - filter by purchase date if transaction exists
        // Show all serial numbers, but filter by purchase date if purchase exists
        if (!empty($start_date) && !empty($end_date)) {
            $query->where(function($q) use ($start_date, $end_date) {
                $q->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                  ->orWhereNull('t.id')
                  ->orWhereNull('pl.id');
            });
        }

        $serial_data = $query->orderBy('c.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->orderBy('sn.serial_number', 'asc')
            ->get();

        // Group data by product description
        $grouped_data = [];
        foreach ($serial_data as $item) {
            $product_key = $item->product_description;
            
            if (!isset($grouped_data[$product_key])) {
                $grouped_data[$product_key] = [];
            }
            
            $grouped_data[$product_key][] = $item;
        }

        // Calculate totals
        $grand_total_qty = count($serial_data);

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::stock_register.print_current_stock_with_serial', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'grand_total_qty',
            'util',
            'business'
        ));
    }

    /**
     * Combined Current Stock (All BR Summary) Report
     */
    public function combinedCurrentStockAllBrSummary(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);

        // Get date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in combinedCurrentStockAllBrSummary: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');

        // Get stock data - combine from all locations
        $query = VariationLocationDetails::join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('p.business_id', $business_id)
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->where('variation_location_details.qty_available', '>', 0)
            ->select(
                'p.id as product_id',
                'p.name as product_name',
                'v.id as variation_id',
                'v.default_purchase_price as unit_cost',
                DB::raw('SUM(variation_location_details.qty_available) as qty'),
                DB::raw('SUM(variation_location_details.qty_available * COALESCE(v.default_purchase_price, 0)) as total_cost'),
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            )
            ->groupBy('p.id', 'v.id', 'c.id', 'c.name', 'p.name', 'pv.name', 'v.name', 'p.type', 'v.default_purchase_price');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('variation_location_details.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('variation_location_details.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        $stock_data = $query->orderBy('c.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group data by category
        $grouped_data = [];
        foreach ($stock_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = $stock_data->sum('qty');
        $grand_total_cost = $stock_data->sum('total_cost');

        $util = new Util();

        return view('businessmanagement::stock_register.combined_current_stock_all_br_summary', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'locations',
            'categories',
            'grand_total_qty',
            'grand_total_cost',
            'util'
        ));
    }

    /**
     * Print Combined Current Stock (All BR Summary) Report
     */
    public function printCombinedCurrentStockAllBrSummary(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Get date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in printCombinedCurrentStockAllBrSummary: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');

        // Get stock data - combine from all locations (same query as web view)
        $query = VariationLocationDetails::join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('p.business_id', $business_id)
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->where('variation_location_details.qty_available', '>', 0)
            ->select(
                'p.id as product_id',
                'p.name as product_name',
                'v.id as variation_id',
                'v.default_purchase_price as unit_cost',
                DB::raw('SUM(variation_location_details.qty_available) as qty'),
                DB::raw('SUM(variation_location_details.qty_available * COALESCE(v.default_purchase_price, 0)) as total_cost'),
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            )
            ->groupBy('p.id', 'v.id', 'c.id', 'c.name', 'p.name', 'pv.name', 'v.name', 'p.type', 'v.default_purchase_price');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('variation_location_details.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('variation_location_details.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        $stock_data = $query->orderBy('c.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group data by category
        $grouped_data = [];
        foreach ($stock_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = $stock_data->sum('qty');
        $grand_total_cost = $stock_data->sum('total_cost');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::stock_register.print_combined_current_stock_all_br_summary', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'grand_total_qty',
            'grand_total_cost',
            'util',
            'business'
        ));
    }

    /**
     * Date Wise Product Status Report
     */
    public function dateWiseProductStatus(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->where('enable_stock', 1)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->pluck('name', 'id')
            ->prepend(__('lang_v1.all'), 'all');

        // Get date range filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in dateWiseProductStatus: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        // Default to today if no date provided
        if (empty($start_date) || empty($end_date)) {
            $start_date = date('Y-m-d');
            $end_date = date('Y-m-d');
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get all products with variations
        $products_query = Product::join('variations as v', 'products.id', '=', 'v.product_id')
            ->leftJoin('categories as c', 'products.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('products.business_id', $business_id)
            ->where('products.type', '!=', 'modifier')
            ->where('products.enable_stock', 1)
            ->select(
                'products.id as product_id',
                'v.id as variation_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as model_no")
            );

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $products_query->where('products.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $products_query->where('products.id', $product_id);
        }

        $products_data = $products_query->get();

        // Calculate stock movements for each product/variation
        $product_status_data = [];
        $permitted_locations = auth()->user()->permitted_locations();
        $location_ids = [];
        
        if ($permitted_locations != 'all') {
            $location_ids = $permitted_locations;
        } else {
            $location_ids = BusinessLocation::where('business_id', $business_id)->pluck('id')->toArray();
        }
        
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location_ids = [$location_id];
        }

        // Calculate opening stock date (day before start_date)
        $opening_date = \Carbon\Carbon::parse($start_date)->subDay()->format('Y-m-d');

        foreach ($products_data as $product) {
            $opening = 0;
            $purchase = 0;
            $p_return = 0;
            $sales = 0;
            $s_return = 0;
            $menu_in = 0;
            $menu_out = 0;
            $item_issue = 0;
            $others_in = 0;
            $others_out = 0;
            $transferred = 0;
            $received = 0;
            $closing = 0;

            foreach ($location_ids as $loc_id) {
                // Opening stock (as of opening_date)
                $opening_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->whereIn('t.type', ['purchase', 'opening_stock'])
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                $opening_sale_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                $opening_sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $opening_adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                $opening += $opening_qty - $opening_sale_qty + $opening_sell_return_qty + $opening_adjustment_qty;

                // Purchase (in date range)
                $purchase += DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                // Purchase Return (in date range)
                $p_return += DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase_return')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity_returned, 0)'));

                // Sales (in date range)
                $sales += DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                // Sales Return (in date range)
                $s_return += DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                // Transferred (sell_transfer - stock transferred out)
                $transferred += DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_transfer')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                // Received (purchase_transfer - stock received/transferred in)
                $received += DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase_transfer')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                // Stock Adjustments - positive (Others In)
                $others_in += DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where(DB::raw('COALESCE(sal.quantity, 0)'), '>', 0)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                // Stock Adjustments - negative (Others Out)
                $others_out += abs(DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where(DB::raw('COALESCE(sal.quantity, 0)'), '<', 0)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)')));
            }

            // Menu In, Menu Out, Item Issue - These might be specific transaction types or stock adjustments
            // For now, setting to 0 as they may not be standard transaction types
            // If they exist in your system, they would need to be mapped accordingly

            // Closing = Opening + Purchase - P.Return - Sales + S.Return + Menu.In - Menu.Out - Item Issue + Others In - Others Out - Transferred + Received
            $closing = $opening + $purchase - $p_return - $sales + $s_return + $menu_in - $menu_out - $item_issue + $others_in - $others_out - $transferred + $received;

            // Only include products with activity or stock
            if ($opening != 0 || $purchase != 0 || $p_return != 0 || $sales != 0 || $s_return != 0 || 
                $menu_in != 0 || $menu_out != 0 || $item_issue != 0 || $others_in != 0 || 
                $others_out != 0 || $transferred != 0 || $received != 0 || $closing != 0) {
                
                $product_status_data[] = (object)[
                    'product_id' => $product->product_id,
                    'variation_id' => $product->variation_id,
                    'model_no' => $product->model_no,
                    'category_name' => $product->category_name,
                    'opening' => $opening,
                    'purchase' => $purchase,
                    'p_return' => $p_return,
                    'sales' => $sales,
                    's_return' => $s_return,
                    'menu_in' => $menu_in,
                    'menu_out' => $menu_out,
                    'item_issue' => $item_issue,
                    'others_in' => $others_in,
                    'others_out' => $others_out,
                    'transferred' => $transferred,
                    'received' => $received,
                    'closing' => $closing,
                ];
            }
        }

        // Group data by category
        $grouped_data = [];
        foreach ($product_status_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Calculate totals
        $grand_total_opening = collect($product_status_data)->sum('opening');
        $grand_total_purchase = collect($product_status_data)->sum('purchase');
        $grand_total_p_return = collect($product_status_data)->sum('p_return');
        $grand_total_sales = collect($product_status_data)->sum('sales');
        $grand_total_s_return = collect($product_status_data)->sum('s_return');
        $grand_total_menu_in = collect($product_status_data)->sum('menu_in');
        $grand_total_menu_out = collect($product_status_data)->sum('menu_out');
        $grand_total_item_issue = collect($product_status_data)->sum('item_issue');
        $grand_total_others_in = collect($product_status_data)->sum('others_in');
        $grand_total_others_out = collect($product_status_data)->sum('others_out');
        $grand_total_transferred = collect($product_status_data)->sum('transferred');
        $grand_total_received = collect($product_status_data)->sum('received');
        $grand_total_closing = collect($product_status_data)->sum('closing');

        $util = new Util();
        $location_name = !empty($location_id) ? BusinessLocation::find($location_id)->name ?? 'All Locations' : 'All Locations';

        return view('businessmanagement::stock_register.date_wise_product_status', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'location_name',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'grand_total_opening',
            'grand_total_purchase',
            'grand_total_p_return',
            'grand_total_sales',
            'grand_total_s_return',
            'grand_total_menu_in',
            'grand_total_menu_out',
            'grand_total_item_issue',
            'grand_total_others_in',
            'grand_total_others_out',
            'grand_total_transferred',
            'grand_total_received',
            'grand_total_closing',
            'util'
        ));
    }

    /**
     * Print Date Wise Product Status Report
     */
    public function printDateWiseProductStatus(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Get date range filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in printDateWiseProductStatus: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        // Default to today if no date provided
        if (empty($start_date) || empty($end_date)) {
            $start_date = date('Y-m-d');
            $end_date = date('Y-m-d');
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get all products with variations (same query as web view)
        $products_query = Product::join('variations as v', 'products.id', '=', 'v.product_id')
            ->leftJoin('categories as c', 'products.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('products.business_id', $business_id)
            ->where('products.type', '!=', 'modifier')
            ->where('products.enable_stock', 1)
            ->select(
                'products.id as product_id',
                'v.id as variation_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as model_no")
            );

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $products_query->where('products.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $products_query->where('products.id', $product_id);
        }

        $products_data = $products_query->get();

        // Calculate stock movements for each product/variation (same logic as web view)
        $product_status_data = [];
        $permitted_locations = auth()->user()->permitted_locations();
        $location_ids = [];
        
        if ($permitted_locations != 'all') {
            $location_ids = $permitted_locations;
        } else {
            $location_ids = BusinessLocation::where('business_id', $business_id)->pluck('id')->toArray();
        }
        
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location_ids = [$location_id];
        }

        // Calculate opening stock date (day before start_date)
        $opening_date = \Carbon\Carbon::parse($start_date)->subDay()->format('Y-m-d');

        foreach ($products_data as $product) {
            $opening = 0;
            $purchase = 0;
            $p_return = 0;
            $sales = 0;
            $s_return = 0;
            $menu_in = 0;
            $menu_out = 0;
            $item_issue = 0;
            $others_in = 0;
            $others_out = 0;
            $transferred = 0;
            $received = 0;
            $closing = 0;

            foreach ($location_ids as $loc_id) {
                // Opening stock (as of opening_date) - same logic as web view
                $opening_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->whereIn('t.type', ['purchase', 'opening_stock'])
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                $opening_sale_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                $opening_sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $opening_adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                $opening += $opening_qty - $opening_sale_qty + $opening_sell_return_qty + $opening_adjustment_qty;

                // Purchase, P.Return, Sales, S.Return, Transferred, Received, Others In, Others Out - same as web view
                $purchase += DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                $p_return += DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase_return')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity_returned, 0)'));

                $sales += DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                $s_return += DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $transferred += DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_transfer')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $received += DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase_transfer')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                $others_in += DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where(DB::raw('COALESCE(sal.quantity, 0)'), '>', 0)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                $others_out += abs(DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->where(DB::raw('COALESCE(sal.quantity, 0)'), '<', 0)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)')));
            }

            // Closing calculation
            $closing = $opening + $purchase - $p_return - $sales + $s_return + $menu_in - $menu_out - $item_issue + $others_in - $others_out - $transferred + $received;

            // Only include products with activity or stock
            if ($opening != 0 || $purchase != 0 || $p_return != 0 || $sales != 0 || $s_return != 0 || 
                $menu_in != 0 || $menu_out != 0 || $item_issue != 0 || $others_in != 0 || 
                $others_out != 0 || $transferred != 0 || $received != 0 || $closing != 0) {
                
                $product_status_data[] = (object)[
                    'product_id' => $product->product_id,
                    'variation_id' => $product->variation_id,
                    'model_no' => $product->model_no,
                    'category_name' => $product->category_name,
                    'opening' => $opening,
                    'purchase' => $purchase,
                    'p_return' => $p_return,
                    'sales' => $sales,
                    's_return' => $s_return,
                    'menu_in' => $menu_in,
                    'menu_out' => $menu_out,
                    'item_issue' => $item_issue,
                    'others_in' => $others_in,
                    'others_out' => $others_out,
                    'transferred' => $transferred,
                    'received' => $received,
                    'closing' => $closing,
                ];
            }
        }

        // Group data by category
        $grouped_data = [];
        foreach ($product_status_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Calculate totals
        $grand_total_opening = collect($product_status_data)->sum('opening');
        $grand_total_purchase = collect($product_status_data)->sum('purchase');
        $grand_total_p_return = collect($product_status_data)->sum('p_return');
        $grand_total_sales = collect($product_status_data)->sum('sales');
        $grand_total_s_return = collect($product_status_data)->sum('s_return');
        $grand_total_menu_in = collect($product_status_data)->sum('menu_in');
        $grand_total_menu_out = collect($product_status_data)->sum('menu_out');
        $grand_total_item_issue = collect($product_status_data)->sum('item_issue');
        $grand_total_others_in = collect($product_status_data)->sum('others_in');
        $grand_total_others_out = collect($product_status_data)->sum('others_out');
        $grand_total_transferred = collect($product_status_data)->sum('transferred');
        $grand_total_received = collect($product_status_data)->sum('received');
        $grand_total_closing = collect($product_status_data)->sum('closing');

        $util = new Util();
        $business = \App\Business::find($business_id);
        $location_name = !empty($location_id) ? BusinessLocation::find($location_id)->name ?? 'All Locations' : 'All Locations';

        return view('businessmanagement::stock_register.print_date_wise_product_status', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'location_name',
            'category_id',
            'product_id',
            'grand_total_opening',
            'grand_total_purchase',
            'grand_total_p_return',
            'grand_total_sales',
            'grand_total_s_return',
            'grand_total_menu_in',
            'grand_total_menu_out',
            'grand_total_item_issue',
            'grand_total_others_in',
            'grand_total_others_out',
            'grand_total_transferred',
            'grand_total_received',
            'grand_total_closing',
            'util',
            'business'
        ));
    }

    /**
     * Date Wise Stock Report
     */
    public function dateWiseStock(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->where('enable_stock', 1)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->pluck('name', 'id')
            ->prepend(__('lang_v1.all'), 'all');

        // Get date filter - this is the date to calculate stock as of
        $stock_date = null;
        if (!empty($request->date)) {
            try {
                $stock_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $stock_date = \Carbon\Carbon::parse($request->date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    \Log::error('Date parsing error in dateWiseStock: ' . $e2->getMessage());
                }
            }
        } elseif (!empty($request->stock_date)) {
            $stock_date = $request->stock_date;
        }

        // Default to today if no date provided
        if (empty($stock_date)) {
            $stock_date = date('Y-m-d');
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get all products with variations
        $products_query = Product::join('variations as v', 'products.id', '=', 'v.product_id')
            ->leftJoin('categories as c', 'products.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('products.business_id', $business_id)
            ->where('products.type', '!=', 'modifier')
            ->where('products.enable_stock', 1)
            ->select(
                'products.id as product_id',
                'v.id as variation_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as product_description")
            );

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $products_query->where('products.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $products_query->where('products.id', $product_id);
        }

        $products_data = $products_query->get();

        // Calculate stock as of date for each product/variation
        $stock_data = [];
        $permitted_locations = auth()->user()->permitted_locations();
        $location_ids = [];
        
        if ($permitted_locations != 'all') {
            $location_ids = $permitted_locations;
        } else {
            $location_ids = BusinessLocation::where('business_id', $business_id)->pluck('id')->toArray();
        }
        
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location_ids = [$location_id];
        }

        foreach ($products_data as $product) {
            $total_qty = 0;

            foreach ($location_ids as $loc_id) {
                // Calculate purchases up to date
                $purchase_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->whereIn('t.type', ['purchase', 'opening_stock'])
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '<=', $stock_date)
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                // Calculate sales up to date
                $sale_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $stock_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                // Calculate sell returns up to date
                $sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $stock_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                // Calculate stock adjustments up to date
                $adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereDate('t.transaction_date', '<=', $stock_date)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                // Stock as of date = purchases - sales + returns + adjustments
                $qty = $purchase_qty - $sale_qty + $sell_return_qty + $adjustment_qty;
                $total_qty += $qty;
            }

            if ($total_qty > 0) {
                $stock_data[] = (object)[
                    'product_id' => $product->product_id,
                    'variation_id' => $product->variation_id,
                    'product_description' => $product->product_description,
                    'qty' => $total_qty,
                    'category_name' => $product->category_name,
                ];
            }
        }

        // Group data by category
        $grouped_data = [];
        foreach ($stock_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = collect($stock_data)->sum('qty');

        $util = new Util();

        return view('businessmanagement::stock_register.date_wise_stock', compact(
            'grouped_data',
            'stock_date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'grand_total_qty',
            'util'
        ));
    }

    /**
     * Print Date Wise Stock Report
     */
    public function printDateWiseStock(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Get date filter - this is the date to calculate stock as of
        $stock_date = null;
        if (!empty($request->date)) {
            try {
                $stock_date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $stock_date = \Carbon\Carbon::parse($request->date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    \Log::error('Date parsing error in printDateWiseStock: ' . $e2->getMessage());
                }
            }
        } elseif (!empty($request->stock_date)) {
            $stock_date = $request->stock_date;
        }

        // Default to today if no date provided
        if (empty($stock_date)) {
            $stock_date = date('Y-m-d');
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get all products with variations (same query as web view)
        $products_query = Product::join('variations as v', 'products.id', '=', 'v.product_id')
            ->leftJoin('categories as c', 'products.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('products.business_id', $business_id)
            ->where('products.type', '!=', 'modifier')
            ->where('products.enable_stock', 1)
            ->select(
                'products.id as product_id',
                'v.id as variation_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as product_description")
            );

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $products_query->where('products.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $products_query->where('products.id', $product_id);
        }

        $products_data = $products_query->get();

        // Calculate stock as of date for each product/variation (same logic as web view)
        $stock_data = [];
        $permitted_locations = auth()->user()->permitted_locations();
        $location_ids = [];
        
        if ($permitted_locations != 'all') {
            $location_ids = $permitted_locations;
        } else {
            $location_ids = BusinessLocation::where('business_id', $business_id)->pluck('id')->toArray();
        }
        
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location_ids = [$location_id];
        }

        foreach ($products_data as $product) {
            $total_qty = 0;

            foreach ($location_ids as $loc_id) {
                // Calculate purchases up to date
                $purchase_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->whereIn('t.type', ['purchase', 'opening_stock'])
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '<=', $stock_date)
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                // Calculate sales up to date
                $sale_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $stock_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                // Calculate sell returns up to date
                $sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $stock_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                // Calculate stock adjustments up to date
                $adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereDate('t.transaction_date', '<=', $stock_date)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                // Stock as of date = purchases - sales + returns + adjustments
                $qty = $purchase_qty - $sale_qty + $sell_return_qty + $adjustment_qty;
                $total_qty += $qty;
            }

            if ($total_qty > 0) {
                $stock_data[] = (object)[
                    'product_id' => $product->product_id,
                    'variation_id' => $product->variation_id,
                    'product_description' => $product->product_description,
                    'qty' => $total_qty,
                    'category_name' => $product->category_name,
                ];
            }
        }

        // Group data by category
        $grouped_data = [];
        foreach ($stock_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = collect($stock_data)->sum('qty');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::stock_register.print_date_wise_stock', compact(
            'grouped_data',
            'stock_date',
            'location_id',
            'category_id',
            'product_id',
            'grand_total_qty',
            'util',
            'business'
        ));
    }

    /**
     * Product Price List Report
     */
    public function productPriceList(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->pluck('name', 'id')
            ->prepend(__('lang_v1.all'), 'all');

        // Get date filter (for future use if needed)
        $date = null;
        if (!empty($request->date)) {
            try {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $date = \Carbon\Carbon::parse($request->date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    \Log::error('Date parsing error in productPriceList: ' . $e2->getMessage());
                }
            }
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get selling price groups
        $price_groups = \App\SellingPriceGroup::where('business_id', $business_id)
            ->where('is_active', 1)
            ->orderBy('name', 'asc')
            ->get();

        // Get all products with variations
        $products_query = Product::join('variations as v', 'products.id', '=', 'v.product_id')
            ->leftJoin('categories as c', 'products.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('units as u', 'products.unit_id', '=', 'u.id')
            ->where('products.business_id', $business_id)
            ->where('products.type', '!=', 'modifier')
            ->select(
                'products.id as product_id',
                'v.id as variation_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as product_group"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as product_model"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as product_description"),
                'u.short_name as unit',
                'v.default_purchase_price as purchase_price',
                'v.default_sell_price as mrp',
                'v.sell_price_inc_tax as lower_price'
            );

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $products_query->where('products.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $products_query->where('products.id', $product_id);
        }

        $products_data = $products_query->orderBy('c.name', 'asc')
            ->orderBy('products.name', 'asc')
            ->get();

        // Get group prices for each variation
        $price_list_data = [];
        foreach ($products_data as $product) {
            // Get group prices for this variation
            $group_prices = \App\VariationGroupPrice::where('variation_id', $product->variation_id)
                ->join('selling_price_groups as spg', 'variation_group_prices.price_group_id', '=', 'spg.id')
                ->where('spg.business_id', $business_id)
                ->where('spg.is_active', 1)
                ->select('spg.name as group_name', 'variation_group_prices.price_inc_tax', 'variation_group_prices.price_type')
                ->get()
                ->keyBy('group_name');

            // Get prices for specific groups
            $dealer_a = 0;
            $dealer_b = 0;
            $dealer_c = 0;
            $retailer_a = 0;

            foreach ($group_prices as $group_name => $group_price) {
                $price = $group_price->price_inc_tax;
                if ($group_price->price_type == 'percentage') {
                    $price = (new Util())->calc_percentage($product->mrp, $group_price->price_inc_tax);
                }

                if (stripos($group_name, 'Dealer-A') !== false || stripos($group_name, 'Dealer A') !== false) {
                    $dealer_a = $price;
                } elseif (stripos($group_name, 'Dealer-B') !== false || stripos($group_name, 'Dealer B') !== false) {
                    $dealer_b = $price;
                } elseif (stripos($group_name, 'Dealer-C') !== false || stripos($group_name, 'Dealer C') !== false) {
                    $dealer_c = $price;
                } elseif (stripos($group_name, 'Retailer-A') !== false || stripos($group_name, 'Retailer A') !== false) {
                    $retailer_a = $price;
                }
            }

            $price_list_data[] = (object)[
                'product_id' => $product->product_id,
                'variation_id' => $product->variation_id,
                'product_group' => $product->product_group,
                'product_model' => $product->product_model,
                'product_description' => $product->product_description,
                'unit' => $product->unit ?? 'PCS',
                'purchase_price' => $product->purchase_price ?? 0,
                'mrp' => $product->mrp ?? 0,
                'lower_price' => $product->lower_price ?? $product->mrp ?? 0,
                'dealer_a' => $dealer_a,
                'dealer_b' => $dealer_b,
                'dealer_c' => $dealer_c,
                'retailer_a' => $retailer_a,
            ];
        }

        // Group data by product group (category)
        $grouped_data = [];
        foreach ($price_list_data as $item) {
            $product_group = $item->product_group ?? 'NO BRAND';
            
            if (!isset($grouped_data[$product_group])) {
                $grouped_data[$product_group] = [];
            }
            
            $grouped_data[$product_group][] = $item;
        }

        $util = new Util();

        return view('businessmanagement::stock_register.product_price_list', compact(
            'grouped_data',
            'date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'util'
        ));
    }

    /**
     * Print Product Price List Report
     */
    public function printProductPriceList(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Get date filter (for future use if needed)
        $date = null;
        if (!empty($request->date)) {
            try {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $date = \Carbon\Carbon::parse($request->date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    \Log::error('Date parsing error in printProductPriceList: ' . $e2->getMessage());
                }
            }
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get all products with variations (same query as web view)
        $products_query = Product::join('variations as v', 'products.id', '=', 'v.product_id')
            ->leftJoin('categories as c', 'products.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('units as u', 'products.unit_id', '=', 'u.id')
            ->where('products.business_id', $business_id)
            ->where('products.type', '!=', 'modifier')
            ->select(
                'products.id as product_id',
                'v.id as variation_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as product_group"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as product_model"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as product_description"),
                'u.short_name as unit',
                'v.default_purchase_price as purchase_price',
                'v.default_sell_price as mrp',
                'v.sell_price_inc_tax as lower_price'
            );

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $products_query->where('products.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $products_query->where('products.id', $product_id);
        }

        $products_data = $products_query->orderBy('c.name', 'asc')
            ->orderBy('products.name', 'asc')
            ->get();

        // Get group prices for each variation (same logic as web view)
        $price_list_data = [];
        foreach ($products_data as $product) {
            // Get group prices for this variation
            $group_prices = \App\VariationGroupPrice::where('variation_id', $product->variation_id)
                ->join('selling_price_groups as spg', 'variation_group_prices.price_group_id', '=', 'spg.id')
                ->where('spg.business_id', $business_id)
                ->where('spg.is_active', 1)
                ->select('spg.name as group_name', 'variation_group_prices.price_inc_tax', 'variation_group_prices.price_type')
                ->get()
                ->keyBy('group_name');

            // Get prices for specific groups
            $dealer_a = 0;
            $dealer_b = 0;
            $dealer_c = 0;
            $retailer_a = 0;

            foreach ($group_prices as $group_name => $group_price) {
                $price = $group_price->price_inc_tax;
                if ($group_price->price_type == 'percentage') {
                    $price = (new Util())->calc_percentage($product->mrp, $group_price->price_inc_tax);
                }

                if (stripos($group_name, 'Dealer-A') !== false || stripos($group_name, 'Dealer A') !== false) {
                    $dealer_a = $price;
                } elseif (stripos($group_name, 'Dealer-B') !== false || stripos($group_name, 'Dealer B') !== false) {
                    $dealer_b = $price;
                } elseif (stripos($group_name, 'Dealer-C') !== false || stripos($group_name, 'Dealer C') !== false) {
                    $dealer_c = $price;
                } elseif (stripos($group_name, 'Retailer-A') !== false || stripos($group_name, 'Retailer A') !== false) {
                    $retailer_a = $price;
                }
            }

            $price_list_data[] = (object)[
                'product_id' => $product->product_id,
                'variation_id' => $product->variation_id,
                'product_group' => $product->product_group,
                'product_model' => $product->product_model,
                'product_description' => $product->product_description,
                'unit' => $product->unit ?? 'PCS',
                'purchase_price' => $product->purchase_price ?? 0,
                'mrp' => $product->mrp ?? 0,
                'lower_price' => $product->lower_price ?? $product->mrp ?? 0,
                'dealer_a' => $dealer_a,
                'dealer_b' => $dealer_b,
                'dealer_c' => $dealer_c,
                'retailer_a' => $retailer_a,
            ];
        }

        // Group data by product group (category)
        $grouped_data = [];
        foreach ($price_list_data as $item) {
            $product_group = $item->product_group ?? 'NO BRAND';
            
            if (!isset($grouped_data[$product_group])) {
                $grouped_data[$product_group] = [];
            }
            
            $grouped_data[$product_group][] = $item;
        }

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::stock_register.print_product_price_list', compact(
            'grouped_data',
            'date',
            'location_id',
            'category_id',
            'product_id',
            'util',
            'business'
        ));
    }

    /**
     * Supplier Wise Current Stock Report
     */
    public function supplierWiseCurrentStock(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->where('enable_stock', 1)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->pluck('name', 'id')
            ->prepend(__('lang_v1.all'), 'all');

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get current stock with supplier information
        $query = VariationLocationDetails::join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('p.business_id', $business_id)
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->where('variation_location_details.qty_available', '>', 0)
            ->select(
                'p.id as product_id',
                'v.id as variation_id',
                'variation_location_details.location_id',
                'variation_location_details.qty_available as qty',
                'v.default_purchase_price as unit_cost',
                DB::raw('variation_location_details.qty_available * COALESCE(v.default_purchase_price, 0) as total_cost'),
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('variation_location_details.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('variation_location_details.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        $stock_data = $query->get();

        // Get supplier for each product/variation from last purchase
        $supplier_stock_data = [];
        foreach ($stock_data as $item) {
            // Get the last purchase transaction for this product/variation to find supplier
            $last_purchase = DB::table('purchase_lines as pl')
                ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase')
                ->where('t.status', 'received')
                ->where('pl.product_id', $item->product_id)
                ->where('pl.variation_id', $item->variation_id)
                ->where('t.location_id', $item->location_id)
                ->select('c.name as supplier_name', 'c.supplier_business_name', 'c.id as supplier_id')
                ->orderBy('t.transaction_date', 'desc')
                ->orderBy('t.id', 'desc')
                ->first();

            $supplier_name = 'NO SUPPLIER';
            $supplier_id = null;
            if ($last_purchase) {
                $supplier_name = $last_purchase->supplier_business_name ?? $last_purchase->supplier_name ?? 'NO SUPPLIER';
                $supplier_id = $last_purchase->supplier_id;
            }

            $supplier_stock_data[] = (object)[
                'product_id' => $item->product_id,
                'variation_id' => $item->variation_id,
                'product_description' => $item->product_description,
                'qty' => $item->qty,
                'unit_cost' => $item->unit_cost ?? 0,
                'total_cost' => $item->total_cost ?? 0,
                'category_name' => $item->category_name,
                'supplier_name' => $supplier_name,
                'supplier_id' => $supplier_id,
            ];
        }

        // Group data by supplier, then by category
        $grouped_data = [];
        foreach ($supplier_stock_data as $item) {
            $supplier_name = $item->supplier_name ?? 'NO SUPPLIER';
            
            if (!isset($grouped_data[$supplier_name])) {
                $grouped_data[$supplier_name] = [];
            }
            
            $category_name = $item->category_name ?? 'NO BRAND';
            if (!isset($grouped_data[$supplier_name][$category_name])) {
                $grouped_data[$supplier_name][$category_name] = [];
            }
            
            $grouped_data[$supplier_name][$category_name][] = $item;
        }

        $util = new Util();

        return view('businessmanagement::stock_register.supplier_wise_current_stock', compact(
            'grouped_data',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'util'
        ));
    }

    /**
     * Print Supplier Wise Current Stock Report
     */
    public function printSupplierWiseCurrentStock(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get current stock with supplier information (same query as web view)
        $query = VariationLocationDetails::join('variations as v', 'variation_location_details.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('categories as c', 'p.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->where('p.business_id', $business_id)
            ->where('p.type', '!=', 'modifier')
            ->where('p.enable_stock', 1)
            ->where('variation_location_details.qty_available', '>', 0)
            ->select(
                'p.id as product_id',
                'v.id as variation_id',
                'variation_location_details.location_id',
                'variation_location_details.qty_available as qty',
                'v.default_purchase_price as unit_cost',
                DB::raw('variation_location_details.qty_available * COALESCE(v.default_purchase_price, 0) as total_cost'),
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('variation_location_details.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('variation_location_details.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        $stock_data = $query->get();

        // Get supplier for each product/variation from last purchase (same logic as web view)
        $supplier_stock_data = [];
        foreach ($stock_data as $item) {
            // Get the last purchase transaction for this product/variation to find supplier
            $last_purchase = DB::table('purchase_lines as pl')
                ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase')
                ->where('t.status', 'received')
                ->where('pl.product_id', $item->product_id)
                ->where('pl.variation_id', $item->variation_id)
                ->where('t.location_id', $item->location_id)
                ->select('c.name as supplier_name', 'c.supplier_business_name', 'c.id as supplier_id')
                ->orderBy('t.transaction_date', 'desc')
                ->orderBy('t.id', 'desc')
                ->first();

            $supplier_name = 'NO SUPPLIER';
            $supplier_id = null;
            if ($last_purchase) {
                $supplier_name = $last_purchase->supplier_business_name ?? $last_purchase->supplier_name ?? 'NO SUPPLIER';
                $supplier_id = $last_purchase->supplier_id;
            }

            $supplier_stock_data[] = (object)[
                'product_id' => $item->product_id,
                'variation_id' => $item->variation_id,
                'product_description' => $item->product_description,
                'qty' => $item->qty,
                'unit_cost' => $item->unit_cost ?? 0,
                'total_cost' => $item->total_cost ?? 0,
                'category_name' => $item->category_name,
                'supplier_name' => $supplier_name,
                'supplier_id' => $supplier_id,
            ];
        }

        // Group data by supplier, then by category
        $grouped_data = [];
        foreach ($supplier_stock_data as $item) {
            $supplier_name = $item->supplier_name ?? 'NO SUPPLIER';
            
            if (!isset($grouped_data[$supplier_name])) {
                $grouped_data[$supplier_name] = [];
            }
            
            $category_name = $item->category_name ?? 'NO BRAND';
            if (!isset($grouped_data[$supplier_name][$category_name])) {
                $grouped_data[$supplier_name][$category_name] = [];
            }
            
            $grouped_data[$supplier_name][$category_name][] = $item;
        }

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::stock_register.print_supplier_wise_current_stock', compact(
            'grouped_data',
            'location_id',
            'category_id',
            'product_id',
            'util',
            'business'
        ));
    }

    /**
     * Date Wise Product In Out Statement Report
     */
    public function dateWiseProductInOutStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->where('enable_stock', 1)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->pluck('name', 'id')
            ->prepend(__('lang_v1.all'), 'all');

        // Get date range filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in dateWiseProductInOutStatement: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        // Default to today if no date provided
        if (empty($start_date) || empty($end_date)) {
            $start_date = date('Y-m-d');
            $end_date = date('Y-m-d');
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get products with variations - include all products with stock or activity
        $stock_query = Product::join('variations as v', 'products.id', '=', 'v.product_id')
            ->leftJoin('categories as c', 'products.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('units as u', 'products.unit_id', '=', 'u.id')
            ->leftJoin('variation_location_details as vld', function($join) use ($business_id, $start_date, $end_date) {
                $join->on('v.id', '=', 'vld.variation_id');
            })
            ->where('products.business_id', $business_id)
            ->where('products.type', '!=', 'modifier')
            ->where('products.enable_stock', 1)
            ->where(function($query) use ($start_date, $end_date, $business_id) {
                // Include products with stock or with activity in date range
                $query->where('vld.qty_available', '>', 0)
                    ->orWhereExists(function($subquery) use ($start_date, $end_date, $business_id) {
                        $subquery->select(DB::raw(1))
                            ->from('transactions as t')
                            ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
                            ->whereColumn('pl.variation_id', 'v.id')
                            ->where('t.business_id', $business_id)
                            ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
                    })
                    ->orWhereExists(function($subquery) use ($start_date, $end_date, $business_id) {
                        $subquery->select(DB::raw(1))
                            ->from('transactions as t')
                            ->join('transaction_sell_lines as tsl', 't.id', '=', 'tsl.transaction_id')
                            ->whereColumn('tsl.variation_id', 'v.id')
                            ->where('t.business_id', $business_id)
                            ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
                    });
            })
            ->select(
                'products.id as product_id',
                'v.id as variation_id',
                'vld.location_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as product_description"),
                'u.short_name as unit'
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $stock_query->whereIn('vld.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $stock_query->where('vld.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $stock_query->where('products.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $stock_query->where('products.id', $product_id);
        }

        // Get location IDs for filtering
        $location_ids = [];
        if ($permitted_locations != 'all') {
            $location_ids = $permitted_locations;
        } else {
            $location_ids = BusinessLocation::where('business_id', $business_id)->pluck('id')->toArray();
        }
        if (!empty($location_id)) {
            $location_ids = [$location_id];
        }

        $products_data = $stock_query->groupBy('products.id', 'v.id', 'c.id', 'c.name', 'products.name', 'pv.name', 'v.name', 'products.type', 'u.short_name')
            ->get();

        // Calculate opening, movements, and closing stock for each product/variation
        $in_out_data = [];

        // Calculate opening stock date (day before start_date)
        $opening_date = \Carbon\Carbon::parse($start_date)->subDay()->format('Y-m-d');

        foreach ($products_data as $product) {
            $opening = 0;
            $qty2 = 0; // Purchase
            $qty3 = 0; // Sales
            $qty4 = 0; // Returns/Adjustments
            $closing = 0;

            foreach ($location_ids as $loc_id) {
                // Opening stock (as of opening_date)
                $opening_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->whereIn('t.type', ['purchase', 'opening_stock'])
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                $opening_sale_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                $opening_sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $opening_adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                $opening += $opening_qty - $opening_sale_qty + $opening_sell_return_qty + $opening_adjustment_qty;

                // Quantity 2: Purchase (in date range)
                $qty2 += DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                // Quantity 3: Sales (in date range)
                $qty3 += DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                // Quantity 4: Returns/Adjustments (in date range)
                $purchase_return_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase_return')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity_returned, 0)'));

                $sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                $qty4 += $purchase_return_qty - $sell_return_qty + $adjustment_qty;

                // Closing stock (as of end_date)
                $closing_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->whereIn('t.type', ['purchase', 'opening_stock'])
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '<=', $end_date)
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                $closing_sale_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $end_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                $closing_sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $end_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $closing_adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereDate('t.transaction_date', '<=', $end_date)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                $closing += $closing_qty - $closing_sale_qty + $closing_sell_return_qty + $closing_adjustment_qty;
            }

            // Only include products with activity or stock
            if ($opening != 0 || $qty2 != 0 || $qty3 != 0 || $qty4 != 0 || $closing != 0) {
                $in_out_data[] = (object)[
                    'product_id' => $product->product_id,
                    'variation_id' => $product->variation_id,
                    'product_description' => $product->product_description,
                    'unit' => $product->unit ?? 'PCS',
                    'category_name' => $product->category_name,
                    'qty1' => $opening,
                    'qty2' => $qty2,
                    'qty3' => $qty3,
                    'qty4' => $qty4,
                    'qty5' => $closing,
                ];
            }
        }

        // Group data by category
        $grouped_data = [];
        foreach ($in_out_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Sort each category by product description
        foreach ($grouped_data as $category => $items) {
            usort($grouped_data[$category], function($a, $b) {
                return strcmp($a->product_description, $b->product_description);
            });
        }

        // Calculate totals
        $grand_total_qty1 = collect($in_out_data)->sum('qty1');
        $grand_total_qty2 = collect($in_out_data)->sum('qty2');
        $grand_total_qty3 = collect($in_out_data)->sum('qty3');
        $grand_total_qty4 = collect($in_out_data)->sum('qty4');
        $grand_total_qty5 = collect($in_out_data)->sum('qty5');

        $util = new Util();

        return view('businessmanagement::stock_register.date_wise_product_in_out_statement', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'grand_total_qty1',
            'grand_total_qty2',
            'grand_total_qty3',
            'grand_total_qty4',
            'grand_total_qty5',
            'util'
        ));
    }

    /**
     * Print Date Wise Product In Out Statement Report
     */
    public function printDateWiseProductInOutStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Get date range filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = preg_split('/\s*[~-]\s*/', $request->date_range);
            if (count($date_range) == 2) {
                try {
                    $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[0]))->format('Y-m-d');
                    $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', trim($date_range[1]))->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon\Carbon::parse(trim($date_range[0]))->format('Y-m-d');
                        $end_date = \Carbon\Carbon::parse(trim($date_range[1]))->format('Y-m-d');
                    } catch (\Exception $e2) {
                        \Log::error('Date parsing error in printDateWiseProductInOutStatement: ' . $e2->getMessage());
                    }
                }
            }
        } elseif (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
        }

        // Default to today if no date provided
        if (empty($start_date) || empty($end_date)) {
            $start_date = date('Y-m-d');
            $end_date = date('Y-m-d');
        }

        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get products with variations - include all products with stock or activity (same as web view)
        $stock_query = Product::join('variations as v', 'products.id', '=', 'v.product_id')
            ->leftJoin('categories as c', 'products.category_id', '=', 'c.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('units as u', 'products.unit_id', '=', 'u.id')
            ->leftJoin('variation_location_details as vld', function($join) use ($business_id, $start_date, $end_date) {
                $join->on('v.id', '=', 'vld.variation_id');
            })
            ->where('products.business_id', $business_id)
            ->where('products.type', '!=', 'modifier')
            ->where('products.enable_stock', 1)
            ->where(function($query) use ($start_date, $end_date, $business_id) {
                // Include products with stock or with activity in date range
                $query->where('vld.qty_available', '>', 0)
                    ->orWhereExists(function($subquery) use ($start_date, $end_date, $business_id) {
                        $subquery->select(DB::raw(1))
                            ->from('transactions as t')
                            ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
                            ->whereColumn('pl.variation_id', 'v.id')
                            ->where('t.business_id', $business_id)
                            ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
                    })
                    ->orWhereExists(function($subquery) use ($start_date, $end_date, $business_id) {
                        $subquery->select(DB::raw(1))
                            ->from('transactions as t')
                            ->join('transaction_sell_lines as tsl', 't.id', '=', 'tsl.transaction_id')
                            ->whereColumn('tsl.variation_id', 'v.id')
                            ->where('t.business_id', $business_id)
                            ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date]);
                    });
            })
            ->select(
                'products.id as product_id',
                'v.id as variation_id',
                'c.id as category_id',
                DB::raw("COALESCE(c.name, 'NO BRAND') as category_name"),
                DB::raw("CASE 
                    WHEN products.type = 'variable' THEN CONCAT(COALESCE(products.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(products.name, 'N/A')
                END as product_description"),
                'u.short_name as unit'
            );

        // Apply location filter - we'll filter in the loop
        $permitted_locations = auth()->user()->permitted_locations();
        $location_ids = [];
        if ($permitted_locations != 'all') {
            $location_ids = $permitted_locations;
        } else {
            $location_ids = BusinessLocation::where('business_id', $business_id)->pluck('id')->toArray();
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $location_ids = [$location_id];
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $stock_query->where('products.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $stock_query->where('products.id', $product_id);
        }

        $products_data = $stock_query->groupBy('products.id', 'v.id', 'c.id', 'c.name', 'products.name', 'pv.name', 'v.name', 'products.type', 'u.short_name')
            ->get();

        // Calculate opening, movements, and closing stock (same logic as web view)
        $in_out_data = [];

        // Calculate opening stock date (day before start_date)
        $opening_date = \Carbon\Carbon::parse($start_date)->subDay()->format('Y-m-d');

        foreach ($products_data as $product) {
            $opening = 0;
            $qty2 = 0;
            $qty3 = 0;
            $qty4 = 0;
            $closing = 0;

            foreach ($location_ids as $loc_id) {
                // Opening, Qty2, Qty3, Qty4, Closing - same logic as web view
                $opening_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->whereIn('t.type', ['purchase', 'opening_stock'])
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                $opening_sale_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                $opening_sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $opening_adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereDate('t.transaction_date', '<=', $opening_date)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                $opening += $opening_qty - $opening_sale_qty + $opening_sell_return_qty + $opening_adjustment_qty;

                $qty2 += DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                $qty3 += DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                $purchase_return_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->where('t.type', 'purchase_return')
                    ->where('t.status', 'received')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(pl.quantity_returned, 0)'));

                $sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereBetween(DB::raw('DATE(t.transaction_date)'), [$start_date, $end_date])
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                $qty4 += $purchase_return_qty - $sell_return_qty + $adjustment_qty;

                $closing_qty = DB::table('purchase_lines as pl')
                    ->join('transactions as t', 'pl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('pl.variation_id', $product->variation_id)
                    ->whereIn('t.type', ['purchase', 'opening_stock'])
                    ->where('t.status', 'received')
                    ->whereDate('t.transaction_date', '<=', $end_date)
                    ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

                $closing_sale_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $end_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0) - COALESCE(tsl.quantity_returned, 0)'));

                $closing_sell_return_qty = DB::table('transaction_sell_lines as tsl')
                    ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('tsl.variation_id', $product->variation_id)
                    ->where('t.type', 'sell_return')
                    ->where('t.status', 'final')
                    ->whereDate('t.transaction_date', '<=', $end_date)
                    ->sum(DB::raw('COALESCE(tsl.quantity, 0)'));

                $closing_adjustment_qty = DB::table('stock_adjustment_lines as sal')
                    ->join('transactions as t', 'sal.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.location_id', $loc_id)
                    ->where('sal.variation_id', $product->variation_id)
                    ->where('t.type', 'stock_adjustment')
                    ->whereDate('t.transaction_date', '<=', $end_date)
                    ->sum(DB::raw('COALESCE(sal.quantity, 0)'));

                $closing += $closing_qty - $closing_sale_qty + $closing_sell_return_qty + $closing_adjustment_qty;
            }

            // Only include products with activity or stock
            if ($opening != 0 || $qty2 != 0 || $qty3 != 0 || $qty4 != 0 || $closing != 0) {
                $in_out_data[] = (object)[
                    'product_id' => $product->product_id,
                    'variation_id' => $product->variation_id,
                    'product_description' => $product->product_description,
                    'unit' => $product->unit ?? 'PCS',
                    'category_name' => $product->category_name,
                    'qty1' => $opening,
                    'qty2' => $qty2,
                    'qty3' => $qty3,
                    'qty4' => $qty4,
                    'qty5' => $closing,
                ];
            }
        }

        // Group data by category
        $grouped_data = [];
        foreach ($in_out_data as $item) {
            $category_name = $item->category_name ?? 'NO BRAND';
            
            if (!isset($grouped_data[$category_name])) {
                $grouped_data[$category_name] = [];
            }
            
            $grouped_data[$category_name][] = $item;
        }

        // Sort each category by product description
        foreach ($grouped_data as $category => $items) {
            usort($grouped_data[$category], function($a, $b) {
                return strcmp($a->product_description, $b->product_description);
            });
        }

        // Calculate totals
        $grand_total_qty1 = collect($in_out_data)->sum('qty1');
        $grand_total_qty2 = collect($in_out_data)->sum('qty2');
        $grand_total_qty3 = collect($in_out_data)->sum('qty3');
        $grand_total_qty4 = collect($in_out_data)->sum('qty4');
        $grand_total_qty5 = collect($in_out_data)->sum('qty5');

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::stock_register.print_date_wise_product_in_out_statement', compact(
            'grouped_data',
            'start_date',
            'end_date',
            'location_id',
            'category_id',
            'product_id',
            'grand_total_qty1',
            'grand_total_qty2',
            'grand_total_qty3',
            'grand_total_qty4',
            'grand_total_qty5',
            'util',
            'business'
        ));
    }

    /**
     * Date Wise Stock Serial (M) Report
     */
    public function dateWiseStockSerialM(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $categories = Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->where('enable_stock', 1)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->pluck('name', 'id')
            ->prepend(__('lang_v1.all'), 'all');

        // Get date filter
        $date = !empty($request->date) ? $request->date : \Carbon\Carbon::now()->format('Y-m-d');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get purchase transactions with serial numbers
        $query = DB::table('transactions as t')
            ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
            ->join('variations as v', 'pl.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('serial_numbers as sn', function($join) {
                $join->on('sn.transaction_id', '=', 't.id')
                     ->on('sn.product_id', '=', 'p.id')
                     ->on('sn.variation_id', '=', 'v.id');
            })
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', $date)
            ->select(
                't.id as transaction_id',
                DB::raw("COALESCE(NULLIF(t.invoice_no, ''), NULLIF(t.ref_no, ''), '') as invoice_no"),
                't.transaction_date',
                DB::raw("COALESCE(c.supplier_business_name, c.name, 'NO SUPPLIER') as supplier_name"),
                'pl.quantity as qty',
                't.status',
                'sn.serial_number',
                'pl.purchase_price_inc_tax as unit_cost',
                'p.id as product_id',
                'v.id as variation_id',
                'cat.id as category_id',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE((SELECT pv.name FROM product_variations pv WHERE pv.id = v.product_variation_id), ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        $data = $query->orderBy('c.supplier_business_name', 'asc')
            ->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.id', 'asc')
            ->get();

        // Group data by supplier
        $grouped_data = [];
        foreach ($data as $item) {
            $supplier_name = $item->supplier_name ?? 'NO SUPPLIER';
            
            if (!isset($grouped_data[$supplier_name])) {
                $grouped_data[$supplier_name] = [];
            }
            
            $grouped_data[$supplier_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = collect($data)->sum('qty');
        $grand_total_cost = collect($data)->sum(function($item) {
            return $item->qty * $item->unit_cost;
        });

        $util = new Util();

        return view('businessmanagement::stock_register.date_wise_stock_serial_m', compact(
            'grouped_data',
            'date',
            'location_id',
            'category_id',
            'product_id',
            'locations',
            'categories',
            'products',
            'grand_total_qty',
            'grand_total_cost',
            'util'
        ));
    }

    /**
     * Print Date Wise Stock Serial (M) Report
     */
    public function printDateWiseStockSerialM(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.stock_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Get date filter
        $date = !empty($request->date) ? $request->date : \Carbon\Carbon::now()->format('Y-m-d');
        $location_id = $request->get('location_id');
        $category_id = $request->get('category_id');
        $product_id = $request->get('product_id');

        // Get purchase transactions with serial numbers (same query as web view)
        $query = DB::table('transactions as t')
            ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
            ->join('variations as v', 'pl.variation_id', '=', 'v.id')
            ->join('products as p', 'v.product_id', '=', 'p.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('serial_numbers as sn', function($join) {
                $join->on('sn.transaction_id', '=', 't.id')
                     ->on('sn.product_id', '=', 'p.id')
                     ->on('sn.variation_id', '=', 'v.id');
            })
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', $date)
            ->select(
                't.id as transaction_id',
                DB::raw("COALESCE(NULLIF(t.invoice_no, ''), NULLIF(t.ref_no, ''), '') as invoice_no"),
                't.transaction_date',
                DB::raw("COALESCE(c.supplier_business_name, c.name, 'NO SUPPLIER') as supplier_name"),
                'pl.quantity as qty',
                't.status',
                'sn.serial_number',
                'pl.purchase_price_inc_tax as unit_cost',
                'p.id as product_id',
                'v.id as variation_id',
                'cat.id as category_id',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE((SELECT pv.name FROM product_variations pv WHERE pv.id = v.product_variation_id), ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
            $query->where('t.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id) && $category_id != 'all') {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('p.id', $product_id);
        }

        $data = $query->orderBy('c.supplier_business_name', 'asc')
            ->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.id', 'asc')
            ->get();

        // Group data by supplier
        $grouped_data = [];
        foreach ($data as $item) {
            $supplier_name = $item->supplier_name ?? 'NO SUPPLIER';
            
            if (!isset($grouped_data[$supplier_name])) {
                $grouped_data[$supplier_name] = [];
            }
            
            $grouped_data[$supplier_name][] = $item;
        }

        // Calculate totals
        $grand_total_qty = collect($data)->sum('qty');
        $grand_total_cost = collect($data)->sum(function($item) {
            return $item->qty * $item->unit_cost;
        });

        $util = new Util();
        $business = \App\Business::find($business_id);

        return view('businessmanagement::stock_register.print_date_wise_stock_serial_m', compact(
            'grouped_data',
            'date',
            'location_id',
            'category_id',
            'product_id',
            'grand_total_qty',
            'grand_total_cost',
            'util',
            'business'
        ));
    }
}

