<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Warehouse extends Model
{
    use SoftDeletes;

    /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * Get the business that owns the warehouse.
     */
    public function business()
    {
        return $this->belongsTo(\App\Business::class);
    }

    /**
     * Get the user who created the warehouse.
     */
    public function creator()
    {
        return $this->belongsTo(\App\User::class, 'created_by');
    }

    /**
     * Get products in this warehouse.
     */
    public function products()
    {
        return $this->belongsToMany(\App\Product::class, 'product_warehouse')
            ->withPivot(['variation_id', 'qty_available', 'qty_reserved'])
            ->withTimestamps();
    }

    /**
     * Get stock movements for this warehouse.
     */
    public function stockMovements()
    {
        return $this->hasMany(\App\StockMovement::class);
    }

    /**
     * Scope a query to only include active warehouses.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', 1);
    }

    /**
     * Scope a query to only include warehouses for a business.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  int  $business_id
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForBusiness($query, $business_id)
    {
        return $query->where('business_id', $business_id);
    }
}

