<?php

namespace App\Utils;

use App\Product;
use App\Variation;
use App\Warehouse;
use App\StockMovement;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class InventoryService
{
    /**
     * Add stock to a warehouse
     *
     * @param int $product_id
     * @param int $variation_id
     * @param int $warehouse_id
     * @param float $quantity
     * @param string $type
     * @param mixed $reference
     * @param string|null $notes
     * @return array
     */
    public function addStock($product_id, $variation_id, $warehouse_id, $quantity, $type = 'purchase', $reference = null, $notes = null)
    {
        return $this->adjustStock($product_id, $variation_id, $warehouse_id, abs($quantity), $type, $reference, $notes);
    }

    /**
     * Remove stock from a warehouse
     *
     * @param int $product_id
     * @param int $variation_id
     * @param int $warehouse_id
     * @param float $quantity
     * @param string $type
     * @param mixed $reference
     * @param string|null $notes
     * @return array
     */
    public function removeStock($product_id, $variation_id, $warehouse_id, $quantity, $type = 'sale', $reference = null, $notes = null)
    {
        return $this->adjustStock($product_id, $variation_id, $warehouse_id, -abs($quantity), $type, $reference, $notes);
    }

    /**
     * Adjust stock in a warehouse (add or remove)
     * Uses database locks to prevent race conditions
     *
     * @param int $product_id
     * @param int $variation_id
     * @param int $warehouse_id
     * @param float $quantity Positive for increase, negative for decrease
     * @param string $type
     * @param mixed $reference
     * @param string|null $notes
     * @return array ['success' => bool, 'qty_before' => float, 'qty_after' => float, 'movement' => StockMovement]
     */
    public function adjustStock($product_id, $variation_id, $warehouse_id, $quantity, $type = 'adjustment', $reference = null, $notes = null)
    {
        DB::beginTransaction();
        try {
            // Lock the product_warehouse row for update to prevent race conditions
            $productWarehouse = DB::table('product_warehouse')
                ->where('product_id', $product_id)
                ->where('variation_id', $variation_id)
                ->where('warehouse_id', $warehouse_id)
                ->lockForUpdate()
                ->first();

            $qty_before = $productWarehouse ? (float)$productWarehouse->qty_available : 0;
            $qty_after = $qty_before + $quantity;

            // Prevent negative stock (unless it's an adjustment that allows it)
            if ($qty_after < 0 && $type !== 'adjustment') {
                DB::rollBack();
                return [
                    'success' => false,
                    'msg' => __('lang_v1.insufficient_stock'),
                    'qty_before' => $qty_before,
                    'qty_after' => $qty_before,
                ];
            }

            // Update or create product_warehouse record
            if ($productWarehouse) {
                DB::table('product_warehouse')
                    ->where('id', $productWarehouse->id)
                    ->update([
                        'qty_available' => $qty_after,
                        'updated_at' => now(),
                    ]);
            } else {
                DB::table('product_warehouse')->insert([
                    'product_id' => $product_id,
                    'variation_id' => $variation_id,
                    'warehouse_id' => $warehouse_id,
                    'qty_available' => $qty_after,
                    'qty_reserved' => 0,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            // Get business_id from warehouse
            $warehouse = Warehouse::find($warehouse_id);
            $business_id = $warehouse->business_id;

            // Create stock movement record (immutable ledger)
            $movement = StockMovement::create([
                'business_id' => $business_id,
                'product_id' => $product_id,
                'variation_id' => $variation_id,
                'warehouse_id' => $warehouse_id,
                'type' => $type,
                'quantity' => $quantity,
                'qty_before' => $qty_before,
                'qty_after' => $qty_after,
                'reference_type' => $reference ? get_class($reference) : null,
                'reference_id' => $reference ? $reference->id : null,
                'reference_no' => $this->getReferenceNo($reference),
                'notes' => $notes,
                'created_by' => auth()->id() ?? 1,
                'movement_date' => now(),
            ]);

            DB::commit();

            return [
                'success' => true,
                'qty_before' => $qty_before,
                'qty_after' => $qty_after,
                'movement' => $movement,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('InventoryService::adjustStock Error: ' . $e->getMessage());
            
            return [
                'success' => false,
                'msg' => __('messages.something_went_wrong'),
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Reserve stock for POS cart/order
     *
     * @param int $product_id
     * @param int $variation_id
     * @param int $warehouse_id
     * @param float $quantity
     * @return array
     */
    public function reserveStock($product_id, $variation_id, $warehouse_id, $quantity)
    {
        DB::beginTransaction();
        try {
            $productWarehouse = DB::table('product_warehouse')
                ->where('product_id', $product_id)
                ->where('variation_id', $variation_id)
                ->where('warehouse_id', $warehouse_id)
                ->lockForUpdate()
                ->first();

            if (!$productWarehouse) {
                DB::rollBack();
                return [
                    'success' => false,
                    'msg' => __('lang_v1.product_not_found_in_warehouse'),
                ];
            }

            $available = (float)$productWarehouse->qty_available - (float)$productWarehouse->qty_reserved;
            
            if ($available < $quantity) {
                DB::rollBack();
                return [
                    'success' => false,
                    'msg' => __('lang_v1.insufficient_stock'),
                    'available' => $available,
                ];
            }

            DB::table('product_warehouse')
                ->where('id', $productWarehouse->id)
                ->increment('qty_reserved', $quantity);

            DB::commit();

            return [
                'success' => true,
                'reserved' => $quantity,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('InventoryService::reserveStock Error: ' . $e->getMessage());
            
            return [
                'success' => false,
                'msg' => __('messages.something_went_wrong'),
            ];
        }
    }

    /**
     * Release reserved stock
     *
     * @param int $product_id
     * @param int $variation_id
     * @param int $warehouse_id
     * @param float $quantity
     * @return array
     */
    public function releaseReservedStock($product_id, $variation_id, $warehouse_id, $quantity)
    {
        DB::beginTransaction();
        try {
            $productWarehouse = DB::table('product_warehouse')
                ->where('product_id', $product_id)
                ->where('variation_id', $variation_id)
                ->where('warehouse_id', $warehouse_id)
                ->lockForUpdate()
                ->first();

            if (!$productWarehouse) {
                DB::rollBack();
                return ['success' => false];
            }

            $current_reserved = (float)$productWarehouse->qty_reserved;
            $release_qty = min($quantity, $current_reserved);

            DB::table('product_warehouse')
                ->where('id', $productWarehouse->id)
                ->decrement('qty_reserved', $release_qty);

            DB::commit();

            return [
                'success' => true,
                'released' => $release_qty,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('InventoryService::releaseReservedStock Error: ' . $e->getMessage());
            
            return [
                'success' => false,
                'msg' => __('messages.something_went_wrong'),
            ];
        }
    }

    /**
     * Transfer stock between warehouses
     *
     * @param int $product_id
     * @param int $variation_id
     * @param int $from_warehouse_id
     * @param int $to_warehouse_id
     * @param float $quantity
     * @param string|null $notes
     * @return array
     */
    public function transferStock($product_id, $variation_id, $from_warehouse_id, $to_warehouse_id, $quantity, $notes = null)
    {
        DB::beginTransaction();
        try {
            // Remove from source warehouse
            $remove_result = $this->removeStock(
                $product_id,
                $variation_id,
                $from_warehouse_id,
                $quantity,
                'transfer_out',
                null,
                $notes
            );

            if (!$remove_result['success']) {
                DB::rollBack();
                return $remove_result;
            }

            // Add to destination warehouse
            $add_result = $this->addStock(
                $product_id,
                $variation_id,
                $to_warehouse_id,
                $quantity,
                'transfer_in',
                null,
                $notes
            );

            if (!$add_result['success']) {
                DB::rollBack();
                return $add_result;
            }

            DB::commit();

            return [
                'success' => true,
                'from_warehouse' => $remove_result,
                'to_warehouse' => $add_result,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('InventoryService::transferStock Error: ' . $e->getMessage());
            
            return [
                'success' => false,
                'msg' => __('messages.something_went_wrong'),
            ];
        }
    }

    /**
     * Get available stock (qty_available - qty_reserved)
     *
     * @param int $product_id
     * @param int $variation_id
     * @param int $warehouse_id
     * @return float
     */
    public function getAvailableStock($product_id, $variation_id, $warehouse_id)
    {
        $productWarehouse = DB::table('product_warehouse')
            ->where('product_id', $product_id)
            ->where('variation_id', $variation_id)
            ->where('warehouse_id', $warehouse_id)
            ->first();

        if (!$productWarehouse) {
            return 0;
        }

        return max(0, (float)$productWarehouse->qty_available - (float)$productWarehouse->qty_reserved);
    }

    /**
     * Get total stock across all warehouses for a product variation
     *
     * @param int $product_id
     * @param int $variation_id
     * @param int|null $business_id
     * @return float
     */
    public function getTotalStock($product_id, $variation_id, $business_id = null)
    {
        $query = DB::table('product_warehouse')
            ->where('product_id', $product_id)
            ->where('variation_id', $variation_id);

        if ($business_id) {
            $query->join('warehouses', 'product_warehouse.warehouse_id', '=', 'warehouses.id')
                ->where('warehouses.business_id', $business_id);
        }

        return (float)$query->sum('qty_available');
    }

    /**
     * Get reference number from reference object
     *
     * @param mixed $reference
     * @return string|null
     */
    private function getReferenceNo($reference)
    {
        if (!$reference) {
            return null;
        }

        // Try common reference number fields
        if (isset($reference->invoice_no)) {
            return $reference->invoice_no;
        }
        if (isset($reference->ref_no)) {
            return $reference->ref_no;
        }
        if (isset($reference->document_no)) {
            return $reference->document_no;
        }
        if (isset($reference->id)) {
            return '#' . $reference->id;
        }

        return null;
    }
}

