<?php

namespace App\Services;

use App\Account;
use App\Brands;
use App\Business;
use App\BusinessLocation;
use App\Category;
use App\Contact;
use App\CustomerGroup;
use App\InvoiceLayout;
use App\InvoiceScheme;
use App\SellingPriceGroup;
use App\TaxRate;
use App\TypesOfService;
use App\User;
use App\Utils\BusinessUtil;
use App\Utils\CashRegisterUtil;
use App\Utils\ContactUtil;
use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Illuminate\Support\Facades\Cache;

class PosService
{
    protected $contactUtil;
    protected $productUtil;
    protected $businessUtil;
    protected $transactionUtil;
    protected $cashRegisterUtil;
    protected $moduleUtil;

    public function __construct(
        ContactUtil $contactUtil,
        ProductUtil $productUtil,
        BusinessUtil $businessUtil,
        TransactionUtil $transactionUtil,
        CashRegisterUtil $cashRegisterUtil,
        ModuleUtil $moduleUtil
    ) {
        $this->contactUtil = $contactUtil;
        $this->productUtil = $productUtil;
        $this->businessUtil = $businessUtil;
        $this->transactionUtil = $transactionUtil;
        $this->cashRegisterUtil = $cashRegisterUtil;
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Prepare all data needed for POS create view
     *
     * @param int $business_id
     * @param int $user_id
     * @param string|null $sub_type
     * @return array
     */
    public function prepareCreateViewData($business_id, $user_id, $sub_type = null)
    {
        // Get cash register details
        $register_details = $this->cashRegisterUtil->getCurrentCashRegister($user_id);

        // Get and validate default location
        $default_location = $this->getDefaultLocation($register_details, $business_id);

        // Get business details with caching
        $business_details = $this->getBusinessDetails($business_id);

        // Get POS settings with caching
        $pos_settings = $this->getPosSettings($business_id, $business_details);

        // Get business locations
        $business_locations_data = BusinessLocation::forDropdown($business_id, false, true);
        $bl_attributes = $business_locations_data['attributes'];
        $business_locations = $business_locations_data['locations'];

        // Get walk-in customer
        $walk_in_customer = $this->contactUtil->getWalkInCustomer($business_id);

        // Prepare dropdowns
        $dropdowns = $this->prepareDropdowns($business_id, $default_location, $business_details);

        // Get featured products
        $featured_products = !empty($default_location) ? $default_location->getFeaturedProducts() : [];

        // Get module data
        $pos_module_data = $this->moduleUtil->getModuleData('get_pos_screen_view', [
            'sub_type' => $sub_type,
            'job_sheet_id' => request()->get('job_sheet_id')
        ]);

        // Get warehouses with location permission validation
        $warehouses_data = $this->getWarehouses($business_id);
        $warehouses = $warehouses_data['warehouses'];
        $default_warehouse = $warehouses_data['default_warehouse'];

        // Get user permissions
        $edit_discount = auth()->user()->can('edit_product_discount_from_pos_screen');
        $edit_price = auth()->user()->can('edit_product_price_from_pos_screen');

        // Get users for contact assignment
        $users = config('constants.enable_contact_assign') 
            ? User::forDropdown($business_id, false, false, false, true) 
            : [];

        // Get common settings
        $common_settings = session()->get('business.common_settings');

        // Get currency details
        $currency_details = $this->transactionUtil->purchaseCurrencyDetails($business_id);

        // Get shortcuts
        $shortcuts = json_decode($business_details->keyboard_shortcuts, true);

        // Get payment lines
        $payment_lines = [$this->getDummyPaymentLine()];
        $change_return = $this->getDummyPaymentLine();

        // Get default datetime
        $default_datetime = $this->businessUtil->format_date('now', true);

        return [
            'currency_details' => $currency_details,
            'edit_discount' => $edit_discount,
            'common_settings' => $common_settings,
            'edit_price' => $edit_price,
            'business_locations' => $business_locations,
            'bl_attributes' => $bl_attributes,
            'business_details' => $business_details,
            'taxes' => $dropdowns['taxes'],
            'payment_types' => $dropdowns['payment_types'],
            'walk_in_customer' => $walk_in_customer,
            'payment_lines' => $payment_lines,
            'default_location' => $default_location,
            'shortcuts' => $shortcuts,
            'commission_agent' => $dropdowns['commission_agent'],
            'categories' => $dropdowns['categories'],
            'brands' => $dropdowns['brands'],
            'pos_settings' => $pos_settings,
            'change_return' => $change_return,
            'types' => $dropdowns['types'],
            'customer_groups' => $dropdowns['customer_groups'],
            'accounts' => $dropdowns['accounts'],
            'price_groups' => $dropdowns['price_groups'],
            'types_of_service' => $dropdowns['types_of_service'],
            'default_price_group_id' => $dropdowns['default_price_group_id'],
            'shipping_statuses' => $dropdowns['shipping_statuses'],
            'default_datetime' => $default_datetime,
            'featured_products' => $featured_products,
            'sub_type' => $sub_type,
            'pos_module_data' => $pos_module_data,
            'invoice_schemes' => $dropdowns['invoice_schemes'],
            'default_invoice_schemes' => $dropdowns['default_invoice_schemes'],
            'invoice_layouts' => $dropdowns['invoice_layouts'],
            'warehouses' => $warehouses,
            'default_warehouse' => $default_warehouse,
            'users' => $users,
        ];
    }

    /**
     * Get default location with validation
     *
     * @param object|null $register_details
     * @param int $business_id
     * @return \App\BusinessLocation|null
     */
    protected function getDefaultLocation($register_details, $business_id)
    {
        $business_locations = BusinessLocation::forDropdown($business_id, false, true);
        $locations = $business_locations['locations'];

        // Validate register location if exists
        if (!empty($register_details->location_id)) {
            $this->validateLocationAccess($register_details->location_id);
            return BusinessLocation::findOrFail($register_details->location_id);
        }

        // Get first permitted location
        if (!empty($locations)) {
            $first_location_id = array_key_first($locations);
            $this->validateLocationAccess($first_location_id);
            return BusinessLocation::findOrFail($first_location_id);
        }

        return null;
    }

    /**
     * Validate location access
     *
     * @param mixed $location_id
     * @return void
     * @throws \Illuminate\Auth\Access\AuthorizationException
     */
    protected function validateLocationAccess($location_id)
    {
        if (empty($location_id)) {
            return;
        }

        $permitted_locations = auth()->user()->permitted_locations();

        if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
            abort(403, 'Unauthorized location access.');
        }
    }

    /**
     * Get business details with caching
     *
     * @param int $business_id
     * @return \App\Business
     */
    protected function getBusinessDetails($business_id)
    {
        return Cache::remember(
            "business_details_{$business_id}",
            now()->addMinutes(30),
            fn() => $this->businessUtil->getDetails($business_id)
        );
    }

    /**
     * Get POS settings with caching
     *
     * @param int $business_id
     * @param \App\Business $business_details
     * @return array
     */
    protected function getPosSettings($business_id, $business_details)
    {
        return Cache::remember(
            "pos_settings_{$business_id}",
            now()->addMinutes(15),
            function() use ($business_details) {
                return empty($business_details->pos_settings)
                    ? $this->businessUtil->defaultPosSettings()
                    : json_decode($business_details->pos_settings, true);
            }
        );
    }

    /**
     * Prepare all dropdowns
     *
     * @param int $business_id
     * @param \App\BusinessLocation|null $default_location
     * @param \App\Business $business_details
     * @return array
     */
    protected function prepareDropdowns($business_id, $default_location, $business_details)
    {
        // Taxes
        $taxes = TaxRate::forBusinessDropdown($business_id, true, true);

        // Payment types
        $payment_types = $this->productUtil->payment_types(null, true, $business_id);

        // Commission agent
        $commission_agent = [];
        $commsn_agnt_setting = $business_details->sales_cmsn_agnt;
        if ($commsn_agnt_setting == 'user') {
            $commission_agent = User::forDropdown($business_id, false);
        } elseif ($commsn_agnt_setting == 'cmsn_agnt') {
            $commission_agent = User::saleCommissionAgentsDropdown($business_id, false);
        }

        // Categories
        $categories = (request()->session()->get('business.enable_category') == 1)
            ? Category::catAndSubCategories($business_id)
            : false;

        // Brands
        $brands = (request()->session()->get('business.enable_brand') == 1)
            ? Brands::forDropdown($business_id)->prepend(__('lang_v1.all_brands'), 'all')
            : false;

        // Contact types
        $types = Contact::getContactTypes();

        // Customer groups
        $customer_groups = CustomerGroup::forDropdown($business_id);

        // Accounts
        $accounts = [];
        if ($this->moduleUtil->isModuleEnabled('account')) {
            $accounts = Account::forDropdown($business_id, true, false, true);
        }

        // Price groups
        $price_groups = SellingPriceGroup::forDropdown($business_id);
        $default_price_group_id = !empty($default_location->selling_price_group_id) 
            && array_key_exists($default_location->selling_price_group_id, $price_groups)
            ? $default_location->selling_price_group_id
            : null;

        // Types of service
        $types_of_service = [];
        if ($this->moduleUtil->isModuleEnabled('types_of_service')) {
            $types_of_service = TypesOfService::forDropdown($business_id);
        }

        // Shipping statuses
        $shipping_statuses = $this->transactionUtil->shipping_statuses();

        // Invoice layouts
        $invoice_layouts = InvoiceLayout::forDropdown($business_id);

        // Invoice schemes
        $invoice_schemes = InvoiceScheme::forDropdown($business_id);
        $default_invoice_schemes = InvoiceScheme::getDefault($business_id);

        return [
            'taxes' => $taxes,
            'payment_types' => $payment_types,
            'commission_agent' => $commission_agent,
            'categories' => $categories,
            'brands' => $brands,
            'types' => $types,
            'customer_groups' => $customer_groups,
            'accounts' => $accounts,
            'price_groups' => $price_groups,
            'default_price_group_id' => $default_price_group_id,
            'types_of_service' => $types_of_service,
            'shipping_statuses' => $shipping_statuses,
            'invoice_layouts' => $invoice_layouts,
            'invoice_schemes' => $invoice_schemes,
            'default_invoice_schemes' => $default_invoice_schemes,
        ];
    }

    /**
     * Get warehouses for business
     * Note: Warehouses are business-level entities, not location-specific
     *
     * @param int $business_id
     * @return array
     */
    protected function getWarehouses($business_id)
    {
        // Warehouses are business-scoped, not location-scoped
        // So we don't filter by location_id (which doesn't exist in warehouses table)
        $warehouses_query = \App\Warehouse::where('business_id', $business_id)
            ->where('is_active', 1);

        $warehouses = $warehouses_query->pluck('name', 'id')->toArray();
        $default_warehouse = $this->productUtil->getDefaultWarehouse($business_id);

        return [
            'warehouses' => $warehouses,
            'default_warehouse' => $default_warehouse,
        ];
    }

    /**
     * Get dummy payment line
     *
     * @return array
     */
    protected function getDummyPaymentLine()
    {
        return [
            'method' => 'cash',
            'amount' => 0,
            'note' => '',
            'card_transaction_number' => '',
            'card_number' => '',
            'card_type' => '',
            'card_holder_name' => '',
            'card_month' => '',
            'card_year' => '',
            'card_security' => '',
            'cheque_number' => '',
            'bank_account_number' => '',
            'is_return' => 0,
            'transaction_no' => '',
        ];
    }
}

