<?php

namespace App\Http\Controllers;

use App\Business;
use App\Warehouse;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class WarehouseController extends Controller
{
    protected $moduleUtil;

    /**
     * Constructor
     *
     * @param  ModuleUtil  $moduleUtil
     * @return void
     */
    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if (!auth()->user()->can('warehouse.view')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        if (request()->ajax()) {
            $warehouses = Warehouse::where('business_id', $business_id)
                ->select(['id', 'name', 'code', 'city', 'state', 'is_active', 'created_at']);

            return Datatables::of($warehouses)
                ->addColumn('action', function ($row) {
                    $html = '<div class="btn-group">';
                    $html .= '<button type="button" data-href="' . action([WarehouseController::class, 'edit'], [$row->id]) . '" class="btn btn-xs btn-primary btn-modal" data-container=".warehouse_modal"><i class="glyphicon glyphicon-edit"></i> ' . __('messages.edit') . '</button>';
                    
                    if (auth()->user()->can('warehouse.delete')) {
                        $html .= '<button type="button" data-href="' . action([WarehouseController::class, 'destroy'], [$row->id]) . '" class="btn btn-xs btn-danger delete_warehouse_button"><i class="glyphicon glyphicon-trash"></i> ' . __('messages.delete') . '</button>';
                    }
                    
                    $html .= '</div>';
                    return $html;
                })
                ->editColumn('is_active', function ($row) {
                    if ($row->is_active) {
                        return '<span class="label label-success">' . __('lang_v1.active') . '</span>';
                    } else {
                        return '<span class="label label-danger">' . __('lang_v1.inactive') . '</span>';
                    }
                })
                ->removeColumn('id')
                ->rawColumns(['action', 'is_active'])
                ->make(true);
        }

        return view('warehouse.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if (!auth()->user()->can('warehouse.create')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        return view('warehouse.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('warehouse.create')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = $request->session()->get('user.business_id');

            $input = $request->only(['name', 'code', 'address', 'city', 'state', 'country', 'zip_code', 'phone', 'email', 'notes']);
            $input['business_id'] = $business_id;
            $input['created_by'] = auth()->user()->id;
            $input['is_active'] = $request->has('is_active') ? 1 : 0;

            // Generate code if not provided
            if (empty($input['code'])) {
                $input['code'] = $this->generateWarehouseCode($business_id);
            }

            $warehouse = Warehouse::create($input);

            // If this is the first warehouse, set it as default
            $business = Business::find($business_id);
            if (!$business->default_warehouse_id) {
                $business->default_warehouse_id = $warehouse->id;
                $business->save();
            }

            $output = [
                'success' => true,
                'msg' => __('lang_v1.warehouse_added_successfully')
            ];
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];
        }

        return response()->json($output);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        if (!auth()->user()->can('warehouse.update')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $warehouse = Warehouse::where('business_id', $business_id)->findOrFail($id);

        return view('warehouse.edit')->with(compact('warehouse'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('warehouse.update')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = $request->session()->get('user.business_id');
            $warehouse = Warehouse::where('business_id', $business_id)->findOrFail($id);

            $input = $request->only(['name', 'code', 'address', 'city', 'state', 'country', 'zip_code', 'phone', 'email', 'notes']);
            $input['is_active'] = $request->has('is_active') ? 1 : 0;

            $warehouse->update($input);

            $output = [
                'success' => true,
                'msg' => __('lang_v1.warehouse_updated_successfully')
            ];
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];
        }

        return response()->json($output);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('warehouse.delete')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = request()->session()->get('user.business_id');
            $warehouse = Warehouse::where('business_id', $business_id)->findOrFail($id);

            // Check if warehouse has stock
            $has_stock = DB::table('product_warehouse')
                ->where('warehouse_id', $id)
                ->where('qty_available', '>', 0)
                ->exists();

            if ($has_stock) {
                $output = [
                    'success' => false,
                    'msg' => __('lang_v1.warehouse_has_stock_cannot_delete')
                ];
                return response()->json($output);
            }

            // Check if it's the default warehouse
            $business = Business::find($business_id);
            if ($business->default_warehouse_id == $id) {
                $output = [
                    'success' => false,
                    'msg' => __('lang_v1.cannot_delete_default_warehouse')
                ];
                return response()->json($output);
            }

            $warehouse->delete();

            $output = [
                'success' => true,
                'msg' => __('lang_v1.warehouse_deleted_successfully')
            ];
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];
        }

        return response()->json($output);
    }

    /**
     * Set default warehouse for business
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function setDefault(Request $request)
    {
        if (!auth()->user()->can('warehouse.update')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = $request->session()->get('user.business_id');
            $warehouse_id = $request->input('warehouse_id');

            $warehouse = Warehouse::where('business_id', $business_id)->findOrFail($warehouse_id);

            $business = Business::find($business_id);
            $business->default_warehouse_id = $warehouse_id;
            $business->save();

            $output = [
                'success' => true,
                'msg' => __('lang_v1.default_warehouse_set_successfully')
            ];
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];
        }

        return response()->json($output);
    }

    /**
     * Get warehouses dropdown for business
     *
     * @return \Illuminate\Http\Response
     */
    public function getDropdown()
    {
        $business_id = request()->session()->get('user.business_id');
        $warehouses = Warehouse::where('business_id', $business_id)
            ->where('is_active', 1)
            ->select('id', 'name', 'code')
            ->get();

        $html = '<option value="">' . __('lang_v1.please_select') . '</option>';
        foreach ($warehouses as $warehouse) {
            $html .= '<option value="' . $warehouse->id . '">' . $warehouse->name . ($warehouse->code ? ' (' . $warehouse->code . ')' : '') . '</option>';
        }

        return $html;
    }

    /**
     * Generate unique warehouse code
     *
     * @param  int  $business_id
     * @return string
     */
    private function generateWarehouseCode($business_id)
    {
        $count = Warehouse::where('business_id', $business_id)->count();
        $code = 'WH' . str_pad($count + 1, 3, '0', STR_PAD_LEFT);
        
        // Ensure uniqueness
        while (Warehouse::where('business_id', $business_id)->where('code', $code)->exists()) {
            $count++;
            $code = 'WH' . str_pad($count + 1, 3, '0', STR_PAD_LEFT);
        }

        return $code;
    }
}

