<?php

namespace App\Http\Controllers;

use App\Contact;
use App\Transaction;

use App\BusinessLocation;
use App\Utils\ModuleUtil;
use App\Utils\ContactUtil;
use App\TransactionPayment;
use App\Utils\BusinessUtil;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\NotificationTemplate;
use App\Utils\TransactionUtil;
use App\Utils\CashRegisterUtil;
use App\Utils\NotificationUtil;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Events\TransactionPaymentAdded;
use Yajra\DataTables\Facades\DataTables;
use App\Events\TransactionPaymentUpdated;
use App\Exceptions\AdvanceBalanceNotAvailable;
use App\Http\Controllers\NotificationController;
use Modules\Accounting\Entities\AccountingAccount;
use Modules\Accounting\Entities\AccountingAccountsTransaction;

class TransactionPaymentController extends Controller
{

    protected $transactionUtil;
    protected $moduleUtil;
    protected $contactUtil;
    protected $businessUtil;
    protected $cashRegisterUtil;
    protected $notificationUtil;

    /**
     * Constructor
     *
     * @param  TransactionUtil  $transactionUtil
     * @return void
     */
    public function __construct(
        TransactionUtil $transactionUtil,
        ModuleUtil $moduleUtil,
        BusinessUtil $businessUtil,
        CashRegisterUtil $cashRegisterUtil,
        ContactUtil $contactUtil,
        NotificationUtil $notificationUtil
    ) {

        $this->notificationUtil = $notificationUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
        $this->contactUtil = $contactUtil;
        $this->cashRegisterUtil = $cashRegisterUtil;
        $this->businessUtil = $businessUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {

        $business_id = request()->session()->get('user.business_id');
        $query = Contact::where('contacts.business_id', $business_id);

        $register_details = $this->cashRegisterUtil->getCurrentCashRegister(auth()->user()->id);

        $default_location = ! empty($register_details->location_id) ? BusinessLocation::findOrFail($register_details->location_id) : null;

        $business_locations = BusinessLocation::forDropdown($business_id, false, true);
        $bl_attributes = $business_locations['attributes'];
        $business_locations = $business_locations['locations'];

        //set first location as default locaton
        if (empty($default_location)) {
            foreach ($business_locations as $id => $name) {
                $default_location = BusinessLocation::findOrFail($id);
                break;
            }
        }

        $all_contact = $query->select(
            DB::raw("IF(contacts.contact_id IS NULL OR contacts.contact_id='', contacts.name, CONCAT(contacts.name, ' (', contacts.contact_id, ')', IF(contacts.supplier_business_name IS NOT NULL AND contacts.supplier_business_name != '', CONCAT(' - ', contacts.supplier_business_name), ''))) AS contact_name"),
            'id'
        )->pluck('contact_name', 'id')->toArray();



        $payment_types = $this->transactionUtil->payment_types(null, true, $business_id);


        if (request()->ajax()) {

            $start_date = request()->start_date;
            $end_date = request()->end_date;
            $payment_method = request()->payment_method;
            $contact_id = request()->contact_id;
            $location_id = request()->location_id;
            $due_payment_type = request()->due_payment_type;

            $payment_list = Contact::join('transactions AS t', 'contacts.id', '=', 't.contact_id')
                // ->whereDate('t.transaction_date', '<', $start_date)
                ->join('transaction_payments', 't.id', '=', 'transaction_payments.transaction_id')
                ->join('business_locations AS bl', 't.location_id', '=', 'bl.id')
                ->whereBetween('transaction_payments.paid_on', [$start_date, $end_date])
                ->select(
                    'transaction_payments.id',
                    'contacts.id as customer_id',
                    'contacts.contact_id',
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.type as contact_type',

                    'bl.name as business_location',

                    't.invoice_no',
                    't.ref_no',
                    't.type as transaction_type',
                    't.id as transaction_id',
                    'transaction_payments.paid_on',
                    DB::raw('COALESCE(SUM(transaction_payments.amount), 0) as total_pay'),
                );


            if (!empty($contact_id)) {
                $payment_list->where('contacts.id', $contact_id);
            }
            if (!empty($payment_method)) {
                $payment_list->where('transaction_payments.method', $payment_method);
            }
            if (!empty($location_id)) {
                $payment_list->where('t.location_id', $location_id);
            }
            if (!empty($due_payment_type)) {
                $payment_list->where('t.type', $due_payment_type);
            }

            $payment_list->groupBy('transaction_payments.id');

            $datatable = Datatables::of($payment_list)

                ->editColumn('action', function ($row) {
                    return '<a class="print-invoice" href="#" data-href="' . route('print.voucher', ['id' => $row->id]) . '">' . __('Print Voucher') . '</a>';
                })


                ->addColumn('conatct_name', '@if(!empty($supplier_business_name)) {{$supplier_business_name}}, <br> @endif {{$name}}')
                ->filterColumn('conatct_name', function ($query, $keyword) {
                    $query->where(function ($q) use ($keyword) {
                        $q->where('contacts.name', 'like', "%{$keyword}%")
                            ->orWhere('contacts.supplier_business_name', 'like', "%{$keyword}%");
                    });
                })
                ->editColumn('business_location', function ($row) {
                    return $row->business_location;
                })

                ->editColumn('paid_on', '{{@format_datetime($paid_on)}}')
                ->editColumn('contact_id', function ($row) {
                    return $row->contact_id;
                })
                ->editColumn('business_location', function ($row) {
                    return $row->business_location;
                })
                ->editColumn('due', function ($row) use ($location_id, $due_payment_type) {
                    $type = $due_payment_type;
                    if (empty($due_payment_type)) {
                        if ($row->contact_type == "customer") {
                            $type = 'sell';
                        } else {
                            $type = 'purchase';
                        }
                    }
                    $contact_id = $row->customer_id;

                    $contact_due = $this->getContactDueForLucationAndType($contact_id, $type, $location_id);

                    $total_old_due = $row->total_pay + $contact_due;
                    $total_old_html = '<span class="due" data-orig-value="' . $total_old_due . '">' . $this->transactionUtil->num_f($total_old_due, true) . '</span>';
                    return $total_old_html;
                })
                ->addColumn('today_pay', function ($row) {
                    $total_pay = $row->total_pay;
                    $total_pay_html = '<span class="today_pay" data-orig-value="' . $total_pay . '">' . $this->transactionUtil->num_f($total_pay, true) . '</span>';
                    return $total_pay_html;
                })
                ->addColumn('balance', function ($row) use ($location_id, $due_payment_type) {
                    $type = $due_payment_type;
                    if (empty($due_payment_type)) {
                        if ($row->contact_type == "customer") {
                            $type = 'sell';
                        } else {
                            $type = 'purchase';
                        }
                    }
                    $contact_id = $row->customer_id;

                    $balance = $this->getContactDueForLucationAndType($contact_id, $type, $location_id);
                    // $balance = 0;
                    $balance_html = '<span class="balance" data-orig-value="' . $balance . '">' . $this->transactionUtil->num_f($balance, true) . '</span>';
                    return $balance_html;
                });

            $rawColumns = [
                'business_location',
                'due',
                'action',
                'today_pay',
                'balance',
                'paid_on',
                'conatct_name'
            ];

            return $datatable->rawColumns($rawColumns)->make(true);
        }


        return view('transaction_payment.index')->with(
            compact(
                'all_contact',
                'payment_types',
                'default_location',
                'business_locations',
                'bl_attributes',
                'business_locations',
            )
        );
    }




    public function PrintVoucher($id)
    {

        try {

            $business_id = request()->session()->get('business.id');

            $invoice_layout = $this->businessUtil->invoiceLayout($business_id, 1);

            $receipt_details = TransactionPayment::where('id', $id)->first();

            $word_format = isset($invoice_layout->common_settings['num_to_word_format']) ? $invoice_layout->common_settings['num_to_word_format'] : 'international';
            $total_in_words = $this->moduleUtil->numToWord($receipt_details->amount, null, $word_format);


            $payment_for = Contact::where('id', $receipt_details->payment_for)->first();
            $business_details = $this->businessUtil->getDetails($business_id);

            $receipt_details['due_payment_type'] = $receipt_details->payment_type = '' ? '' : '';

            $receipt_details['contact_info'] = $payment_for;
            $receipt_details['total_in_words'] = $total_in_words;
            $receipt_details['user'] = Auth::user();
            $receipt_details['business_details'] = $business_details;
            $receipt_details['voucher_title'] = 'payment_voucher';


            $output = [
                'success' => 1,
                'receipt' => [],
                'print_title' => 'payment_voucher'
            ];

            $output['receipt']['html_content'] = view('sale_pos.receipts.many_receipt', compact('receipt_details'))->render();
        } catch (\Exception $e) {

            DB::rollBack();
            Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg' => 'File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage(),
            ];
        }


        return $output;
    }


    public function PaymentRecived()
    {

        if (! (auth()->user()->can('sell.payments') || auth()->user()->can('purchase.payments'))) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');


        $register_details = $this->cashRegisterUtil->getCurrentCashRegister(auth()->user()->id);

        $default_location = ! empty($register_details->location_id) ? BusinessLocation::findOrFail($register_details->location_id) : null;

        $business_locations = BusinessLocation::forDropdown($business_id, false, true);
        $bl_attributes = $business_locations['attributes'];
        $business_locations = $business_locations['locations'];

        //set first location as default locaton
        if (empty($default_location)) {
            foreach ($business_locations as $id => $name) {
                $default_location = BusinessLocation::findOrFail($id);
                break;
            }
        }



        $walk_in_customer = $this->contactUtil->getWalkInCustomer($business_id);
        $query = Contact::where('contacts.business_id', $business_id);

        $all_contact = $query->select(
            DB::raw("IF(contacts.contact_id IS NULL OR contacts.contact_id='', contacts.name, CONCAT(contacts.name, ' (', contacts.contact_id, ')', IF(contacts.supplier_business_name IS NOT NULL AND contacts.supplier_business_name != '', CONCAT(' - ', contacts.supplier_business_name), ''))) AS contact_name"),
            'id'
        )
            ->pluck('contact_name', 'id')
            ->toArray();

        $payment_types = $this->transactionUtil->payment_types(null, false, $business_id);

        //Accounts
        $accounts = $this->moduleUtil->accountsDropdown($business_id, true);


        return view('transaction_payment.create')->with(compact(
            'payment_types',
            'accounts',
            'walk_in_customer',
            'default_location',
            'business_locations',
            'bl_attributes',
            'business_locations',
            'all_contact'
        ));
    }
    /**
     * Shows contact's payment due modal
     *
     * @param  int  $contact_id
     * @return \Illuminate\Http\Response
     */
    public function getPayContactDueForm($contact_id)
    {
        if (! (auth()->user()->can('sell.payments') || auth()->user()->can('purchase.payments'))) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');

            $due_payment_type = request()->input('type');

            $location_id = request()->input('location_id');


            $query = Contact::where('contacts.id', $contact_id)
                ->leftjoin('transactions AS t', 'contacts.id', '=', 't.contact_id')
                ->where('t.location_id', $location_id);


            if ($due_payment_type == 'purchase') {
                $query->select(
                    DB::raw("SUM(IF(t.type = 'purchase', final_total, 0)) as total_purchase"),
                    DB::raw("SUM(IF(t.type = 'purchase', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_paid"),
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.id as contact_id'
                );
            } elseif ($due_payment_type == 'purchase_return') {
                $query->select(
                    DB::raw("SUM(IF(t.type = 'purchase_return', final_total, 0)) as total_purchase_return"),
                    DB::raw("SUM(IF(t.type = 'purchase_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_return_paid"),
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.id as contact_id'
                );
            } elseif ($due_payment_type == 'sell') {
                $query->select(
                    DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', final_total, 0)) as total_invoice"),
                    DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', (SELECT SUM(IF(is_return = 1,-1*amount,amount)) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_paid"),
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.id as contact_id'
                );
            } elseif ($due_payment_type == 'sell_return') {
                $query->select(
                    DB::raw("SUM(IF(t.type = 'sell_return', final_total, 0)) as total_sell_return"),
                    DB::raw("SUM(IF(t.type = 'sell_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_return_paid"),
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.id as contact_id'
                );
            }

            //Query for opening balance details
            $query->addSelect(
                DB::raw("SUM(IF(t.type = 'opening_balance', final_total, 0)) as opening_balance"),
                DB::raw("SUM(IF(t.type = 'opening_balance', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as opening_balance_paid")
            );
            $contact_details = $query->first();

            $payment_line = new TransactionPayment();
            if ($due_payment_type == 'purchase') {
                $contact_details->total_purchase = empty($contact_details->total_purchase) ? 0 : $contact_details->total_purchase;
                $payment_line->amount = $contact_details->total_purchase - $contact_details->total_paid;
            } elseif ($due_payment_type == 'purchase_return') {
                $payment_line->amount = $contact_details->total_purchase_return - $contact_details->total_return_paid;
            } elseif ($due_payment_type == 'sell') {
                $contact_details->total_invoice = empty($contact_details->total_invoice) ? 0 : $contact_details->total_invoice;
                $payment_line->amount = $contact_details->total_invoice - $contact_details->total_paid;
            } elseif ($due_payment_type == 'sell_return') {
                $payment_line->amount = $contact_details->total_sell_return - $contact_details->total_return_paid;
            }

            //If opening balance due exists add to payment amount
            $contact_details->opening_balance = ! empty($contact_details->opening_balance) ? $contact_details->opening_balance : 0;
            $contact_details->opening_balance_paid = ! empty($contact_details->opening_balance_paid) ? $contact_details->opening_balance_paid : 0;
            $ob_due = $contact_details->opening_balance - $contact_details->opening_balance_paid;
            if ($ob_due > 0) {
                $payment_line->amount += $ob_due;
            }

            $amount_formated = $this->transactionUtil->num_f($payment_line->amount);

            $contact_details->total_paid = empty($contact_details->total_paid) ? 0 : $contact_details->total_paid;

            $payment_line->method = 'cash';
            $payment_line->paid_on = \Carbon::now()->toDateTimeString();

            $payment_types = $this->transactionUtil->payment_types(null, false, $business_id);

            //Accounts
            $accounts = $this->moduleUtil->accountsDropdown($business_id, true);

            return view('transaction_payment.pay_supplier_due_form')->with(
                compact(
                    'contact_details',
                    'payment_types',
                    'payment_line',
                    'due_payment_type',
                    'ob_due',
                    'amount_formated',
                    'location_id',
                    'accounts'
                )
            );
        }
    }
    public function postPayContactDueForm(Request $request)
    {

        if (! (auth()->user()->can('sell.payments') || auth()->user()->can('purchase.payments'))) {
            abort(403, 'Unauthorized action.');
        }

        // Check if there is a open register, if no then redirect to Create Register screen.
        $location_id = $request->location_id;

        try {

            DB::beginTransaction();

            $business_id = request()->session()->get('business.id');



            if ($request->amount <= 0) {

                $output = [
                    'success' => false,
                    'msg' => 'Pay Amount must be geterdan 0',
                ];

                return $output;
            }


            $receipt_details = $this->transactionUtil->payContactDue($request);

            $pos_settings = ! empty(session()->get('business.pos_settings')) ? json_decode(session()->get('business.pos_settings'), true) : [];
            $enable_cash_denomination_for_payment_methods = ! empty($pos_settings['enable_cash_denomination_for_payment_methods']) ? $pos_settings['enable_cash_denomination_for_payment_methods'] : [];
            //add cash denomination
            if (in_array($receipt_details->method, $enable_cash_denomination_for_payment_methods) && ! empty($request->input('denominations')) && ! empty($pos_settings['enable_cash_denomination_on']) && $pos_settings['enable_cash_denomination_on'] == 'all_screens') {
                $denominations = [];

                foreach ($request->input('denominations') as $key => $value) {
                    if (! empty($value)) {
                        $denominations[] = [
                            'business_id' => $business_id,
                            'amount' => $key,
                            'total_count' => $value,
                        ];
                    }
                }

                if (! empty($denominations)) {
                    $receipt_details->denominations()->createMany($denominations);
                }
            }


            $invoice_layout = $this->businessUtil->invoiceLayout($business_id, 1);
            $word_format = isset($invoice_layout->common_settings['num_to_word_format']) ? $invoice_layout->common_settings['num_to_word_format'] : 'international';
            $total_in_words = $this->moduleUtil->numToWord($request->amount, null, $word_format);

            DB::commit();

            $payment_for = Contact::where('id', $receipt_details->payment_for)->first();
            $business_details = $this->businessUtil->getDetails($business_id);

            $receipt_details['due_payment_type'] = $request->due_payment_type;
            $receipt_details['contact_info'] = $payment_for;
            $receipt_details['total_in_words'] = $total_in_words;
            $receipt_details['user'] = Auth::user();
            $receipt_details['business_details'] = $business_details;
            $receipt_details['voucher_title'] = 'payment_voucher';

            $output = [
                'success' => true,
                'print_title' => 'Payment Voucher',
                'html_content' => view('sale_pos.receipts.many_receipt', compact('receipt_details'))->render(),
                'msg' => __('purchase.payment_added_success'),
            ];
        } catch (\Exception $e) {

            DB::rollBack();
            Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage(),
            ];
        }

        return $output;
    }




    public function getContactDueForLucationAndType($contact_id, $due_payment_type, $location_id)
    {

        $query = Contact::where('contacts.id', $contact_id)
            ->leftjoin('transactions AS t', 'contacts.id', '=', 't.contact_id')
            ->where('t.location_id', $location_id);


        if ($due_payment_type == 'purchase') {
            $query->select(
                DB::raw("SUM(IF(t.type = 'purchase', final_total, 0)) as total_purchase"),
                DB::raw("SUM(IF(t.type = 'purchase', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_paid"),
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.id as contact_id'
            );
        } elseif ($due_payment_type == 'purchase_return') {
            $query->select(
                DB::raw("SUM(IF(t.type = 'purchase_return', final_total, 0)) as total_purchase_return"),
                DB::raw("SUM(IF(t.type = 'purchase_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_return_paid"),
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.id as contact_id'
            );
        } elseif ($due_payment_type == 'sell') {
            $query->select(
                DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', final_total, 0)) as total_invoice"),
                DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', (SELECT SUM(IF(is_return = 1,-1*amount,amount)) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_paid"),
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.id as contact_id'
            );
        } elseif ($due_payment_type == 'sell_return') {
            $query->select(
                DB::raw("SUM(IF(t.type = 'sell_return', final_total, 0)) as total_sell_return"),
                DB::raw("SUM(IF(t.type = 'sell_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_return_paid"),
                'contacts.name',
                'contacts.supplier_business_name',
                'contacts.id as contact_id'
            );
        }

        //Query for opening balance details
        $query->addSelect(
            DB::raw("SUM(IF(t.type = 'opening_balance', final_total, 0)) as opening_balance"),
            DB::raw("SUM(IF(t.type = 'opening_balance', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as opening_balance_paid")
        );
        $contact_details = $query->first();

        $payment_line_amount = 0;

        if ($due_payment_type == 'purchase') {
            $contact_details->total_purchase = empty($contact_details->total_purchase) ? 0 : $contact_details->total_purchase;
            $payment_line_amount = $contact_details->total_purchase - $contact_details->total_paid;
        } elseif ($due_payment_type == 'purchase_return') {
            $payment_line_amount = $contact_details->total_purchase_return - $contact_details->total_return_paid;
        } elseif ($due_payment_type == 'sell') {
            $contact_details->total_invoice = empty($contact_details->total_invoice) ? 0 : $contact_details->total_invoice;
            $payment_line_amount = $contact_details->total_invoice - $contact_details->total_paid;
        } elseif ($due_payment_type == 'sell_return') {
            $payment_line_amount = $contact_details->total_sell_return - $contact_details->total_return_paid;
        }

        //If opening balance due exists add to payment amount
        $contact_details->opening_balance = ! empty($contact_details->opening_balance) ? $contact_details->opening_balance : 0;
        $contact_details->opening_balance_paid = ! empty($contact_details->opening_balance_paid) ? $contact_details->opening_balance_paid : 0;
        $ob_due = $contact_details->opening_balance - $contact_details->opening_balance_paid;
        if ($ob_due > 0) {
            $payment_line_amount += $ob_due;
        }

        $amount_un_formated = $payment_line_amount;
        $amount_formated = $this->transactionUtil->num_f($payment_line_amount);

        $contact_details->total_paid = empty($contact_details->total_paid) ? 0 : $contact_details->total_paid;

        $output = [];
        $output['amount_un_formated'] = $amount_un_formated;
        $output['amount_formated'] = $amount_formated;

        return $amount_un_formated;
    }



    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        try {
            $business_id = $request->session()->get('user.business_id');
            $transaction_id = $request->input('transaction_id');
            $transaction = Transaction::where('business_id', $business_id)->with(['contact'])->findOrFail($transaction_id);





            if (!empty($transaction) && !empty($transaction->return_group_id)) {


                $all_returns = Transaction::where('return_group_id', $transaction->return_group_id)
                    ->where('business_id', $business_id)
                    ->where('type', 'sell_return')
                    ->get();
                $inputs = $request->only([
                    'amount',
                    'method',
                    'note',
                    'card_number',
                    'card_holder_name',
                    'card_transaction_number',
                    'card_type',
                    'card_month',
                    'card_year',
                    'card_security',
                    'cheque_number',
                    'bank_account_number',
                ]);
                $inputs['paid_on'] = $this->transactionUtil->uf_date($request->input('paid_on'), true);

                if ($inputs['method'] == 'custom_pay_1') {
                    $inputs['transaction_no'] = $request->input('transaction_no_1');
                } elseif ($inputs['method'] == 'custom_pay_2') {
                    $inputs['transaction_no'] = $request->input('transaction_no_2');
                } elseif ($inputs['method'] == 'custom_pay_3') {
                    $inputs['transaction_no'] = $request->input('transaction_no_3');
                }

                if (! empty($request->input('account_id')) && $inputs['method'] != 'advance') {
                    $inputs['account_id'] = $request->input('account_id');
                }
                $prefix_type = 'sell_payment';
                $inputs['business_id'] = $request->session()->get('business.id');
                $inputs['document'] = $this->transactionUtil->uploadFile($request, 'document', 'documents');



                DB::beginTransaction();

                $total_payment_amount = $this->transactionUtil->num_uf($inputs['amount']);
                dd("inside sell return", $request->all());

                foreach ($all_returns as $return) {

                    if ($return->payment_status != 'paid') {

                        $inputs['transaction_id'] = $return->id;

                        $paid_amount = $this->transactionUtil->getTotalPaid($return->id);

                        $final_total = $return->final_total;
                        $paid_amount = $paid_amount;
                        $this_transaction_amount = $final_total - $paid_amount;

                        if ($total_payment_amount > 0) {
                            $inputs['amount'] = $this->transactionUtil->num_uf(min($total_payment_amount, $this_transaction_amount));
                        }

                        $total_payment_amount -= $inputs['amount'];

                        $inputs['created_by'] = auth()->user()->id;
                        $inputs['payment_for'] = $return->contact_id;


                        $ref_count = $this->transactionUtil->setAndGetReferenceCount($prefix_type);
                        //Generate reference number
                        $inputs['payment_ref_no'] = $this->transactionUtil->generateReferenceNumber($prefix_type, $ref_count);

                        if (! empty($inputs['amount'])) {

                            $tp = TransactionPayment::create($inputs);
                            //accounting account transaction by ashiq
                            try {
                                $accountingAccount = AccountingAccount::where('account_id', $transaction->contact_id)->where('business_id', $inputs['business_id'])->first();
                                if ($accountingAccount) {
                                    $forCredit = [
                                        'amount'                => $inputs['amount'],
                                        'accounting_account_id' => $accountingAccount->id,
                                        'created_by'            => $inputs['created_by'],
                                        'operation_date'        => $inputs["paid_on"],
                                        'note'                  => $request['note'],
                                        'type'                  => 'debit',
                                        'sub_type'              => "payment",
                                        'transaction_id'        => $request["transaction_id"],
                                        // 'transaction_payment_id' => $parent_payment->id
                                    ];
                                    AccountingAccountsTransaction::createTransaction($forCredit);
                                }
                            } catch (\Exception $e) {
                                Log::emergency('Accounting transaction error: ' . $e->getMessage());
                            }

                            //accounting account transaction by ashiq end
                            if (! empty($request->input('denominations'))) {
                                $this->transactionUtil->addCashDenominations($tp, $request->input('denominations'));
                            }

                            $inputs['transaction_type'] = $return->type;

                            try {
                                event(new TransactionPaymentAdded($tp, $inputs));
                            } catch (\Exception $e) {
                                Log::emergency('Event dispatch error: ' . $e->getMessage());
                            }
                        }


                        $transaction_before = $return->replicate();

                        //update payment status
                        $payment_status = $this->transactionUtil->updatePaymentStatus($return->id, $return->final_total);
                        $return->payment_status = $payment_status;

                        $this->transactionUtil->activityLog($return, 'payment_edited', $transaction_before);
                    }
                }
            } else {

                $transaction_before = $transaction->replicate();


                if (!(auth()->user()->can('purchase.payments') || auth()->user()->can('hms.add_booking_payment') || auth()->user()->can('sell.payments') || auth()->user()->can('all_expense.access') || auth()->user()->can('view_own_expense'))) {
                    abort(403, 'Unauthorized action.');
                }

                if ($transaction->payment_status != 'paid') {
                    $inputs = $request->only([
                        'amount',
                        'method',
                        'note',
                        'card_number',
                        'card_holder_name',
                        'card_transaction_number',
                        'card_type',
                        'card_month',
                        'card_year',
                        'card_security',
                        'cheque_number',
                        'bank_account_number',
                    ]);
                    $inputs['paid_on'] = $this->transactionUtil->uf_date($request->input('paid_on'), true);
                    $inputs['transaction_id'] = $transaction->id;
                    $inputs['amount'] = $this->transactionUtil->num_uf($inputs['amount']);
                    $inputs['created_by'] = auth()->user()->id;
                    $inputs['payment_for'] = $transaction->contact_id;

                    if ($inputs['method'] == 'custom_pay_1') {
                        $inputs['transaction_no'] = $request->input('transaction_no_1');
                    } elseif ($inputs['method'] == 'custom_pay_2') {
                        $inputs['transaction_no'] = $request->input('transaction_no_2');
                    } elseif ($inputs['method'] == 'custom_pay_3') {
                        $inputs['transaction_no'] = $request->input('transaction_no_3');
                    }

                    if (!empty($request->input('account_id')) && $inputs['method'] != 'advance') {
                        $inputs['account_id'] = $request->input('account_id');
                    }

                    $prefix_type = 'purchase_payment';
                    if (in_array($transaction->type, ['sell', 'sell_return'])) {
                        $prefix_type = 'sell_payment';
                    } elseif (in_array($transaction->type, ['expense', 'expense_refund'])) {
                        $prefix_type = 'expense_payment';
                    }

                    DB::beginTransaction();

                    $ref_count = $this->transactionUtil->setAndGetReferenceCount($prefix_type);
                    //Generate reference number
                    $inputs['payment_ref_no'] = $this->transactionUtil->generateReferenceNumber($prefix_type, $ref_count);

                    $inputs['business_id'] = $request->session()->get('business.id');
                    $inputs['document'] = $this->transactionUtil->uploadFile($request, 'document', 'documents');

                    //Pay from advance balance
                    $payment_amount = $inputs['amount'];
                    $contact_balance = !empty($transaction->contact) ? $transaction->contact->balance : 0;
                    if ($inputs['method'] == 'advance' && $inputs['amount'] > $contact_balance) {
                        throw new AdvanceBalanceNotAvailable(__('lang_v1.required_advance_balance_not_available'));
                    }

                    if (!empty($inputs['amount'])) {
                        $tp = TransactionPayment::create($inputs);

                        if (!empty($request->input('denominations'))) {
                            $this->transactionUtil->addCashDenominations($tp, $request->input('denominations'));
                        }

                        $inputs['transaction_type'] = $transaction->type;
                        try {
                            event(new TransactionPaymentAdded($tp, $inputs));
                        } catch (\Exception $e) {
                            Log::emergency('Event dispatch error: ' . $e->getMessage());
                        }
                    }

                    //update payment status
                    $payment_status = $this->transactionUtil->updatePaymentStatus($transaction_id, $transaction->final_total);
                    $transaction->payment_status = $payment_status;

                    $this->transactionUtil->activityLog($transaction, 'payment_edited', $transaction_before);

                    DB::commit();
                }
            }


            $output = [
                'success' => true,
                'msg' => __('purchase.payment_added_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            $msg = __('messages.something_went_wrong');

            if (get_class($e) == \App\Exceptions\AdvanceBalanceNotAvailable::class) {
                $msg = $e->getMessage();
            } else {
                \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());
            }

            $output = [
                'success' => false,
                'msg' => $msg,
            ];
        }

        return redirect()->back()->with(['status' => $output]);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        if (!(auth()->user()->can('sell.payments') || auth()->user()->can('purchase.payments') || auth()->user()->can('hms.add_booking_payment'))) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {




            if (Str::startsWith($id, 'g')) {

                $group_id = (int) Str::after($id, 'g');

                $transaction = Transaction::where('return_group_id', $group_id)
                    ->where('type', 'sell_return')
                    ->first();

                $all_return_ids = Transaction::where('return_group_id', $group_id)
                    ->where('type', 'sell_return')
                    ->pluck('id');
                $payments_query = TransactionPayment::whereIn('transaction_id', $all_return_ids);


                $accounts_enabled = false;
                if ($this->moduleUtil->isModuleEnabled('account')) {
                    $accounts_enabled = true;
                    $payments_query->with(['payment_account']);
                }
                $payments = $payments_query->get();

                $location_id = ! empty($transaction->location_id) ? $transaction->location_id : null;
                $payment_types = $this->transactionUtil->payment_types($location_id, true);

                return view('transaction_payment.show_payments')->with(compact('transaction', 'payments', 'payment_types', 'accounts_enabled'));
            } else {


                $transaction = Transaction::where('id', $id)
                    ->with(['contact', 'business', 'transaction_for'])
                    ->first();
                $payments_query = TransactionPayment::where('transaction_id', $id);

                $accounts_enabled = false;
                if ($this->moduleUtil->isModuleEnabled('account')) {
                    $accounts_enabled = true;
                    $payments_query->with(['payment_account']);
                }

                $payments = $payments_query->get();
                $location_id = !empty($transaction->location_id) ? $transaction->location_id : null;
                $payment_types = $this->transactionUtil->payment_types($location_id, true);

                return view('transaction_payment.show_payments')
                    ->with(compact('transaction', 'payments', 'payment_types', 'accounts_enabled'));
            }
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        if (!auth()->user()->can('edit_purchase_payment') && !auth()->user()->can('edit_sell_payment') && !auth()->user()->can('hms.edit_booking_payment')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');

            $payment_line = TransactionPayment::with(['denominations'])->where('method', '!=', 'advance')->findOrFail($id);

            $transaction = Transaction::where('id', $payment_line->transaction_id)
                ->where('business_id', $business_id)
                ->with(['contact', 'location'])
                ->first();

            $payment_types = $this->transactionUtil->payment_types($transaction->location);

            //Accounts
            $accounts = $this->moduleUtil->accountsDropdown($business_id, true, false, true);

            return view('transaction_payment.edit_payment_row')
                ->with(compact('transaction', 'payment_types', 'payment_line', 'accounts'));
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('edit_purchase_payment') && !auth()->user()->can('edit_sell_payment') && !auth()->user()->can('all_expense.access') && !auth()->user()->can('view_own_expense') && !auth()->user()->can('hms.edit_booking_payment')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = request()->session()->get('user.business_id');

            $inputs = $request->only([
                'amount',
                'method',
                'note',
                'card_number',
                'card_holder_name',
                'card_transaction_number',
                'card_type',
                'card_month',
                'card_year',
                'card_security',
                'cheque_number',
                'bank_account_number',
            ]);
            $inputs['paid_on'] = $this->transactionUtil->uf_date($request->input('paid_on'), true);
            $inputs['amount'] = $this->transactionUtil->num_uf($inputs['amount']);

            if ($inputs['method'] == 'custom_pay_1') {
                $inputs['transaction_no'] = $request->input('transaction_no_1');
            } elseif ($inputs['method'] == 'custom_pay_2') {
                $inputs['transaction_no'] = $request->input('transaction_no_2');
            } elseif ($inputs['method'] == 'custom_pay_3') {
                $inputs['transaction_no'] = $request->input('transaction_no_3');
            }

            if (!empty($request->input('account_id'))) {
                $inputs['account_id'] = $request->input('account_id');
            }

            $payment = TransactionPayment::where('method', '!=', 'advance')->findOrFail($id);

            if (!empty($request->input('denominations'))) {
                $this->transactionUtil->updateCashDenominations($payment, $request->input('denominations'));
            }

            //Update parent payment if exists
            if (!empty($payment->parent_id)) {
                $parent_payment = TransactionPayment::find($payment->parent_id);
                $parent_payment->amount = $parent_payment->amount - ($payment->amount - $inputs['amount']);

                $parent_payment->save();
            }

            $business_id = $request->session()->get('user.business_id');

            $transaction = Transaction::where('business_id', $business_id)
                ->find($payment->transaction_id);

            $transaction_before = $transaction->replicate();
            $document_name = $this->transactionUtil->uploadFile($request, 'document', 'documents');
            if (!empty($document_name)) {
                $inputs['document'] = $document_name;
            }

            DB::beginTransaction();

            $payment->update($inputs);
            // dd($payment);
            // 🧾 Accounting transaction update

            // 1. আগের accounting entry খুঁজে বের করুন
            try {
                $accountingEntry = AccountingAccountsTransaction::where('transaction_id', $payment->transaction_id)
                    ->where('transaction_payment_id', $payment->id)
                    ->first();

                    if ($accountingEntry) {
                        $accountingEntry->update([
                            'amount'                => $payment->amount ?? $inputs['amount'],
                            'note'                  => $payment->note ?? null,
                        ]);
                    };
            } catch (\Exception $e) {
                Log::emergency('Accounting transaction update error: ' . $e->getMessage());
            }



            //update payment status
            $payment_status = $this->transactionUtil->updatePaymentStatus($payment->transaction_id);
            $transaction->payment_status = $payment_status;

            $this->transactionUtil->activityLog($transaction, 'payment_edited', $transaction_before);

            DB::commit();

            //event
            try {
                event(new TransactionPaymentUpdated($payment, $transaction->type));
            } catch (\Exception $e) {
                Log::emergency('Event dispatch error: ' . $e->getMessage());
            }

            $output = [
                'success' => true,
                'msg' => __('purchase.payment_updated_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->back()->with(['status' => $output]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('delete_purchase_payment') && !auth()->user()->can('delete_sell_payment') && !auth()->user()->can('all_expense.access') && !auth()->user()->can('view_own_expense') && !auth()->user()->can('hms.delete_booking_payment')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            try {
                $payment = TransactionPayment::findOrFail($id);

                DB::beginTransaction();

                if (!empty($payment->transaction_id)) {
                    TransactionPayment::deletePayment($payment);
                } else { //advance payment
                    $adjusted_payments = TransactionPayment::where(
                        'parent_id',
                        $payment->id
                    )
                        ->get();

                    $total_adjusted_amount = $adjusted_payments->sum('amount');

                    //Get customer advance share from payment and deduct from advance balance
                    $total_customer_advance = $payment->amount - $total_adjusted_amount;
                    if ($total_customer_advance > 0) {
                        $this->transactionUtil->updateContactBalance($payment->payment_for, $total_customer_advance, 'deduct');
                    }

                    //Delete all child payments
                    foreach ($adjusted_payments as $adjusted_payment) {
                        //Make parent payment null as it will get deleted
                        $adjusted_payment->parent_id = null;
                        TransactionPayment::deletePayment($adjusted_payment);
                    }

                    //Delete advance payment
                    TransactionPayment::deletePayment($payment);


                }
                try {
                    AccountingAccountsTransaction::where('transaction_id', $payment->transaction_id)
                    ->where('transaction_payment_id', $payment->id)
                    ->delete();
                } catch (\Exception $e) {
                    Log::emergency('Accounting transaction delete error: ' . $e->getMessage());
                }

                DB::commit();

                $output = [
                    'success' => true,
                    'msg' => __('purchase.payment_deleted_success'),
                ];
            } catch (\Exception $e) {
                DB::rollBack();

                \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

                $output = [
                    'success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    /**
     * Adds new payment to the given transaction.
     *
     * @param  int  $transaction_id
     * @return \Illuminate\Http\Response
     */
    public function addPayment($transaction_id)
    {
        if (!auth()->user()->can('purchase.payments') && !auth()->user()->can('sell.payments') && !auth()->user()->can('all_expense.access') && !auth()->user()->can('view_own_expense') && !auth()->user()->can('hms.add_booking_payment')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');



            if (Str::startsWith($transaction_id, 'g')) {

                $group_id = (int) Str::after($transaction_id, 'g');


                $transaction = Transaction::where('return_group_id', $group_id)
                    ->where('type', 'sell_return')->first();

                $show_advance = false;
                $payment_types = $this->transactionUtil->payment_types($transaction->location, $show_advance);

                $all_returns = Transaction::where('return_group_id', $group_id)->where('business_id', $business_id)->where('type', 'sell_return')->get();

                $amount = 0;
                foreach ($all_returns as $return) {

                    if ($return->payment_status != 'paid') {
                        $paid_amount = $this->transactionUtil->getTotalPaid($return->id);
                        $single_amount = $return->final_total - $paid_amount;
                        if ($single_amount < 0) {
                            $single_amount = 0;
                        }
                        $amount += $single_amount;
                    }
                }


                $amount_formated = $this->transactionUtil->num_f($amount);

                $payment_line = new TransactionPayment();
                $payment_line->amount = $amount;
                $payment_line->method = 'cash';
                $payment_line->paid_on = \Carbon::now()->toDateTimeString();



                if ($amount > 0) {
                    //Accounts
                    $accounts = $this->moduleUtil->accountsDropdown($business_id, true, false, true);

                    $view = view('transaction_payment.payment_row')->with(compact('transaction', 'payment_types', 'payment_line', 'amount_formated', 'accounts'))->render();

                    $output = [
                        'status' => 'due',
                        'view' => $view,
                    ];
                } else {
                    $output = [
                        'status' => 'paid',
                        'view' => '',
                        'msg' => __('purchase.amount_already_paid'),
                    ];
                }
            } else {

                $transaction = Transaction::where('business_id', $business_id)
                    ->with(['contact', 'location'])
                    ->findOrFail($transaction_id);
                if ($transaction->payment_status != 'paid') {
                    $show_advance = in_array($transaction->type, ['sell', 'purchase']) ? true : false;
                    $payment_types = $this->transactionUtil->payment_types($transaction->location, $show_advance);

                    $paid_amount = $this->transactionUtil->getTotalPaid($transaction_id);
                    $amount = $transaction->final_total - $paid_amount;
                    if ($amount < 0) {
                        $amount = 0;
                    }

                    $amount_formated = $this->transactionUtil->num_f($amount);

                    $payment_line = new TransactionPayment();
                    $payment_line->amount = $amount;
                    $payment_line->method = 'cash';
                    $payment_line->paid_on = \Carbon::now()->toDateTimeString();

                    //Accounts
                    $accounts = $this->moduleUtil->accountsDropdown($business_id, true, false, true);

                    $view = view('transaction_payment.payment_row')->with(
                        compact(
                            'transaction',
                            'payment_types',
                            'payment_line',
                            'amount_formated',
                            'accounts'
                        )
                    )->render();

                    $output = [
                        'status' => 'due',
                        'view' => $view,
                    ];
                } else {
                    $output = [
                        'status' => 'paid',
                        'view' => '',
                        'msg' => __('purchase.amount_already_paid'),
                    ];
                }
            }

            return json_encode($output);
        }
    }

    /**
     * Shows contact's payment due modal
     *
     * @param  int  $contact_id
     * @return \Illuminate\Http\Response
     */
    public function getPayContactDue($contact_id)
    {
        if (!(auth()->user()->can('sell.payments') || auth()->user()->can('purchase.payments'))) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');

            $due_payment_type = request()->input('type');
            $query = Contact::where('contacts.id', $contact_id)
                ->leftjoin('transactions AS t', 'contacts.id', '=', 't.contact_id');
            if ($due_payment_type == 'purchase') {
                $query->select(
                    DB::raw("SUM(IF(t.type = 'purchase', final_total, 0)) as total_purchase"),
                    DB::raw("SUM(IF(t.type = 'purchase', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_paid"),
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.id as contact_id'
                );
            } elseif ($due_payment_type == 'purchase_return') {
                $query->select(
                    DB::raw("SUM(IF(t.type = 'purchase_return', final_total, 0)) as total_purchase_return"),
                    DB::raw("SUM(IF(t.type = 'purchase_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_return_paid"),
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.id as contact_id'
                );
            } elseif ($due_payment_type == 'sell') {
                $query->select(
                    DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', final_total, 0)) as total_invoice"),
                    DB::raw("SUM(IF(t.type = 'sell' AND t.status = 'final', (SELECT SUM(IF(is_return = 1,-1*amount,amount)) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_paid"),
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.id as contact_id'
                );
            } elseif ($due_payment_type == 'sell_return') {
                $query->select(
                    DB::raw("SUM(IF(t.type = 'sell_return', final_total, 0)) as total_sell_return"),
                    DB::raw("SUM(IF(t.type = 'sell_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as total_return_paid"),
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'contacts.id as contact_id'
                );
            }

            //Query for opening balance details
            $query->addSelect(
                DB::raw("SUM(IF(t.type = 'opening_balance', final_total, 0)) as opening_balance"),
                DB::raw("SUM(IF(t.type = 'opening_balance', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=t.id), 0)) as opening_balance_paid")
            );
            $contact_details = $query->first();

            $payment_line = new TransactionPayment();
            if ($due_payment_type == 'purchase') {
                $contact_details->total_purchase = empty($contact_details->total_purchase) ? 0 : $contact_details->total_purchase;
                $payment_line->amount = $contact_details->total_purchase -
                    $contact_details->total_paid;
            } elseif ($due_payment_type == 'purchase_return') {
                $payment_line->amount = $contact_details->total_purchase_return -
                    $contact_details->total_return_paid;
            } elseif ($due_payment_type == 'sell') {
                $contact_details->total_invoice = empty($contact_details->total_invoice) ? 0 : $contact_details->total_invoice;

                $payment_line->amount = $contact_details->total_invoice -
                    $contact_details->total_paid;
            } elseif ($due_payment_type == 'sell_return') {
                $payment_line->amount = $contact_details->total_sell_return -
                    $contact_details->total_return_paid;
            }

            //If opening balance due exists add to payment amount
            $contact_details->opening_balance = !empty($contact_details->opening_balance) ? $contact_details->opening_balance : 0;
            $contact_details->opening_balance_paid = !empty($contact_details->opening_balance_paid) ? $contact_details->opening_balance_paid : 0;
            $ob_due = $contact_details->opening_balance - $contact_details->opening_balance_paid;
            if ($ob_due > 0) {
                $payment_line->amount += $ob_due;
            }

            $amount_formated = $this->transactionUtil->num_f($payment_line->amount);

            $contact_details->total_paid = empty($contact_details->total_paid) ? 0 : $contact_details->total_paid;

            $payment_line->method = 'cash';
            $payment_line->paid_on = \Carbon::now()->toDateTimeString();

            $payment_types = $this->transactionUtil->payment_types(null, false, $business_id);

            //Accounts
            $accounts = $this->moduleUtil->accountsDropdown($business_id, true);

            return view('transaction_payment.pay_supplier_due_modal')->with(
                compact(
                    'contact_details',
                    'payment_types',
                    'payment_line',
                    'due_payment_type',
                    'ob_due',
                    'amount_formated',
                    'accounts'
                )
            );
        }
    }

    /**
     * Adds Payments for Contact due
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function postPayContactDue(Request $request)
    {
        if (!(auth()->user()->can('sell.payments') || auth()->user()->can('purchase.payments'))) {
            abort(403, 'Unauthorized action.');
        }
        //upload document

        try {
            DB::beginTransaction();

            $business_id = request()->session()->get('business.id');
            $tp = $this->transactionUtil->payContact($request);

            $pos_settings = !empty(session()->get('business.pos_settings')) ? json_decode(session()->get('business.pos_settings'), true) : [];
            $enable_cash_denomination_for_payment_methods = !empty($pos_settings['enable_cash_denomination_for_payment_methods']) ? $pos_settings['enable_cash_denomination_for_payment_methods'] : [];
            //add cash denomination
            if (in_array($tp->method, $enable_cash_denomination_for_payment_methods) && !empty($request->input('denominations')) && !empty($pos_settings['enable_cash_denomination_on']) && $pos_settings['enable_cash_denomination_on'] == 'all_screens') {
                $denominations = [];

                foreach ($request->input('denominations') as $key => $value) {
                    if (!empty($value)) {
                        $denominations[] = [
                            'business_id' => $business_id,
                            'amount' => $key,
                            'total_count' => $value,
                        ];
                    }
                }

                if (!empty($denominations)) {
                    $tp->denominations()->createMany($denominations);
                }
            }

            DB::commit();
            $transactions = clone $tp;
            //transactionsArray is no longer necessary as a data array as been created
            // Convert $transactions to an array
            $transactionsArray = $transactions->toArray();
            if (array_key_exists('payment_for', $transactionsArray)) {
                // Move the value to 'contact' key
                $contact = Contact::find($transactionsArray['payment_for']);
                // dd($contact->supplier_business_name);
                $contact = Contact::find($transactionsArray['payment_for']);

                if ($contact && !empty($contact->name)) {
                    $transactionsArray['contact'] = $contact->name;
                    $transactionsArray['mobile'] = $contact->mobile;
                } else {
                    // If 'contact' is null or empty, use 'supplier_business_name' and 'mobile'
                    $transactionsArray['contact'] = !empty($contact->supplier_business_name) ? $contact->supplier_business_name : null;
                    $transactionsArray['mobile'] = !empty($contact->mobile) ? $contact->mobile : null;
                }


                $transactionsArray['transaction_id'] = $transactionsArray['id'];
                // unset($transactionsArray['payment_for']);
                unset($transactionsArray['method']);
                unset($transactionsArray['card_type']);
                unset($transactionsArray['payment_type']);
                unset($transactionsArray['updated_at']);
                unset($transactionsArray['created_at']);
                unset($transactionsArray['payment_type']);
                unset($transactionsArray['payment_ref_no']);
                unset($transactionsArray['paid_on']);
            }

            $transaction_pay = TransactionPayment::where('payment_for', $transactionsArray['payment_for'])
                ->whereNotNull('transaction_id')
                ->latest('created_at')
                ->first();

            if (!$transaction_pay) {
                // If $transaction_pay is not found, create a new instance
                $transaction_pay = new TransactionPayment();
            }

            // Add the 'contact' key to the $transaction_pay object
            $transaction_pay->contact = (object)['name' => $transactionsArray['contact']];
            //   dd($transactionsArray);
            $notification_type = 'payment_received';
            $notification_template = NotificationTemplate::where('business_id', $business_id)
                ->where('template_for', $notification_type)
                ->select('email_body', 'sms_body', 'whatsapp_text', 'subject', 'cc', 'bcc', 'template_for', 'auto_send_sms')
                ->first();
            // dd($notification_template);
            $email = $notification_template->email_body;
            $sms = $notification_template->sms_body;
            $whatsapp = $notification_template->whatsapp_text;
            $subject = $notification_template->subject;
            $cc = $notification_template->cc;
            $bcc = $notification_template->bcc;
            $template = $notification_template->template_for;
            $auto_send_sms = $notification_template->auto_send_sms;

            $data = [
                'template_for' => 'payment_received',
                'transaction_id' => $transaction_pay->transaction_id,
                'notification_type' => ['sms'],
                'mobile_number' => $transactionsArray['mobile'],
                'to_email' => '',
                'email_body' => $email,
                'sms_body' => $sms,
                'cc' => $cc,
                'bcc' => $bcc,
                'subject' => $subject,
                'whatsapp_text' => $whatsapp,
            ];


            $notificationUtil = new NotificationUtil();
            $transactionUtil = new TransactionUtil();

            if (!empty($auto_send_sms)) {
                // dd('Before send method call');
                $notificationController = new NotificationController($notificationUtil, $transactionUtil);
                $request = new Request($data);
                $notificationController->send($request);
                // dd('After send method call');
            }
            // dd($transaction_pay);

            //Auto send notification whatsapp auto send can be added here
            //  $this->notificationUtil->autoSendNotification($business_id, 'payment_received', $transaction_pay, $transaction_pay->contact);

            $output = [
                'success' => true,
                'msg' => __('purchase.payment_added_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage(),
            ];
        }

        return redirect()->back()->with(['status' => $output]);
    }

    /**
     * view details of single..,
     * payment.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function viewPayment($payment_id)
    {
        if (!(auth()->user()->can('sell.payments') ||
            auth()->user()->can('purchase.payments') ||
            auth()->user()->can('edit_sell_payment') ||
            auth()->user()->can('delete_sell_payment') ||
            auth()->user()->can('edit_purchase_payment') ||
            auth()->user()->can('delete_purchase_payment') ||
            auth()->user()->can('hms.add_booking_payment')
        )) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('business.id');
            $single_payment_line = TransactionPayment::findOrFail($payment_id);

            $transaction = null;
            if (!empty($single_payment_line->transaction_id)) {
                $transaction = Transaction::where('id', $single_payment_line->transaction_id)
                    ->with(['contact', 'location', 'transaction_for'])
                    ->first();
            } else {
                $child_payment = TransactionPayment::where('business_id', $business_id)
                    ->where('parent_id', $payment_id)
                    ->with(['transaction', 'transaction.contact', 'transaction.location', 'transaction.transaction_for'])
                    ->first();
                $transaction = !empty($child_payment) ? $child_payment->transaction : null;
            }

            $payment_types = $this->transactionUtil->payment_types(null, false, $business_id);

            return view('transaction_payment.single_payment_view')
                ->with(compact('single_payment_line', 'transaction', 'payment_types'));
        }
    }

    /**
     * Retrieves all the child payments of a parent payments
     * payment.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function showChildPayments($payment_id)
    {
        if (!(auth()->user()->can('sell.payments') ||
            auth()->user()->can('purchase.payments') ||
            auth()->user()->can('edit_sell_payment') ||
            auth()->user()->can('delete_sell_payment') ||
            auth()->user()->can('edit_purchase_payment') ||
            auth()->user()->can('delete_purchase_payment')
        )) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('business.id');

            $child_payments = TransactionPayment::where('business_id', $business_id)
                ->where('parent_id', $payment_id)
                ->with(['transaction', 'transaction.contact'])
                ->get();

            $payment_types = $this->transactionUtil->payment_types(null, false, $business_id);

            return view('transaction_payment.show_child_payments')
                ->with(compact('child_payments', 'payment_types'));
        }
    }

    /**
     * Retrieves list of all opening balance payments.
     *
     * @param  int  $contact_id
     * @return \Illuminate\Http\Response
     */
    public function getOpeningBalancePayments($contact_id)
    {
        if (!(auth()->user()->can('sell.payments') ||
            auth()->user()->can('purchase.payments') ||
            auth()->user()->can('edit_sell_payment') ||
            auth()->user()->can('delete_sell_payment') ||
            auth()->user()->can('edit_purchase_payment') ||
            auth()->user()->can('delete_purchase_payment')
        )) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('business.id');
        if (request()->ajax()) {
            $query = TransactionPayment::leftjoin('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'opening_balance')
                ->where('t.contact_id', $contact_id)
                ->where('transaction_payments.business_id', $business_id)
                ->select(
                    'transaction_payments.amount',
                    'method',
                    'paid_on',
                    'transaction_payments.payment_ref_no',
                    'transaction_payments.document',
                    'transaction_payments.id',
                    'cheque_number',
                    'card_transaction_number',
                    'bank_account_number'
                )
                ->groupBy('transaction_payments.id');

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            return Datatables::of($query)
                ->editColumn('paid_on', '{{@format_datetime($paid_on)}}')
                ->editColumn('method', function ($row) {
                    $method = __('lang_v1.' . $row->method);
                    if ($row->method == 'cheque') {
                        $method .= '<br>(' . __('lang_v1.cheque_no') . ': ' . $row->cheque_number . ')';
                    } elseif ($row->method == 'card') {
                        $method .= '<br>(' . __('lang_v1.card_transaction_no') . ': ' . $row->card_transaction_number . ')';
                    } elseif ($row->method == 'bank_transfer') {
                        $method .= '<br>(' . __('lang_v1.bank_account_no') . ': ' . $row->bank_account_number . ')';
                    } elseif ($row->method == 'custom_pay_1') {
                        $method = __('lang_v1.custom_payment_1') . '<br>(' . __('lang_v1.transaction_no') . ': ' . $row->transaction_no . ')';
                    } elseif ($row->method == 'custom_pay_2') {
                        $method = __('lang_v1.custom_payment_2') . '<br>(' . __('lang_v1.transaction_no') . ': ' . $row->transaction_no . ')';
                    } elseif ($row->method == 'custom_pay_3') {
                        $method = __('lang_v1.custom_payment_3') . '<br>(' . __('lang_v1.transaction_no') . ': ' . $row->transaction_no . ')';
                    }

                    return $method;
                })
                ->editColumn('amount', function ($row) {
                    return '<span class="display_currency paid-amount" data-orig-value="' . $row->amount . '" data-currency_symbol = true>' . $row->amount . '</span>';
                })
                ->addColumn('action', '<button type="button" class="btn btn-primary btn-xs view_payment" data-href="{{ action([\App\Http\Controllers\TransactionPaymentController::class, \'viewPayment\'], [$id]) }}"><i class="fas fa-eye"></i> @lang("messages.view")
                    </button> <button type="button" class="btn btn-info btn-xs edit_payment" 
                    data-href="{{action([\App\Http\Controllers\TransactionPaymentController::class, \'edit\'], [$id]) }}"><i class="glyphicon glyphicon-edit"></i> @lang("messages.edit")</button>
                    &nbsp; <button type="button" class="btn btn-danger btn-xs delete_payment" 
                    data-href="{{ action([\App\Http\Controllers\TransactionPaymentController::class, \'destroy\'], [$id]) }}"
                    ><i class="fa fa-trash" aria-hidden="true"></i> @lang("messages.delete")</button> @if(!empty($document))<a href="{{asset("/uploads/documents/" . $document)}}" class="btn btn-success btn-xs" download=""><i class="fa fa-download"></i> @lang("purchase.download_document")</a>@endif')
                ->rawColumns(['amount', 'method', 'action'])
                ->make(true);
        }
    }
}
