<?php

namespace App\Http\Controllers;

use App\User;
use App\Contact;
use App\Product;
use App\Variation;
use App\ReturnGroup;
use App\Transaction;
use App\SerialNumber;
use App\CustomerGroup;
use App\BusinessLocation;
use App\Utils\ModuleUtil;
use App\Utils\ContactUtil;
use App\Utils\ProductUtil;
use App\Utils\BusinessUtil;
use App\TransactionSellLine;
use Illuminate\Http\Request;
use App\Utils\TransactionUtil;
use App\Utils\CashRegisterUtil;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Spatie\Activitylog\Models\Activity;
use Yajra\DataTables\Facades\DataTables;
use App\Events\TransactionPaymentDeleted;
use App\TransactionSellLinesPurchaseLines;
use Modules\Accounting\Entities\AccountingAccount;
use Modules\Accounting\Entities\AccountingAccTransMapping;
use Modules\Accounting\Entities\AccountingAccountsTransaction;

class SellReturnController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $productUtil;

    protected $transactionUtil;

    protected $contactUtil;

    protected $businessUtil;

    protected $moduleUtil;

    protected $cashRegisterUtil;
    protected $dummyPaymentLine;

    /**
     * Constructor
     *
     * @param  ProductUtils  $product
     * @return void
     */
    public function __construct(ProductUtil $productUtil, TransactionUtil $transactionUtil, CashRegisterUtil $cashRegisterUtil, ContactUtil $contactUtil, BusinessUtil $businessUtil, ModuleUtil $moduleUtil)
    {
        $this->productUtil = $productUtil;
        $this->transactionUtil = $transactionUtil;
        $this->contactUtil = $contactUtil;
        $this->businessUtil = $businessUtil;
        $this->moduleUtil = $moduleUtil;
        $this->cashRegisterUtil = $cashRegisterUtil;
        
        
        $this->dummyPaymentLine = [
            'method' => '', 'amount' => 0, 
            'note' => '', 'card_transaction_number' => '', 
            'card_number' => '', 'card_type' => '', 
            'card_holder_name' => '', 'card_month' => '', 
            'card_year' => '', 'card_security' => '', 
            'cheque_number' => '', 'bank_account_number' => '',
            'is_return' => 0, 'transaction_no' => '', 
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {


        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        if (request()->ajax()) {


            $sells = ReturnGroup::where('return_groups.business_id', $business_id)
                                    ->join('transactions','transactions.return_group_id','=','return_groups.id')
                                    ->where('transactions.type','sell_return')
                                    ->join('contacts', 'transactions.contact_id', '=', 'contacts.id')
                                    ->join('business_locations AS bl', 'transactions.location_id', '=', 'bl.id')
                                    ->groupBy('return_groups.id')
                                    ->select(
                                        'return_groups.id',
                                        DB::raw("MIN(transactions.transaction_date) as transaction_date"),
                                        DB::raw("GROUP_CONCAT(DISTINCT CASE WHEN contacts.name != '' THEN contacts.name ELSE contacts.supplier_business_name END SEPARATOR ', ') as customer_names"),
                                        DB::raw("GROUP_CONCAT(DISTINCT bl.name SEPARATOR ', ') as business_locations"),

                                        DB::raw("GROUP_CONCAT(DISTINCT transactions.invoice_no SEPARATOR ', ') as invoice_nos"),

                                        DB::raw('SUM(transactions.final_total) as final_total'),
                                        DB::raw("SUM(IF(transactions.type = 'sell_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=transactions.id), 0)) as total_paid")
                                    );

 
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sells->whereIn('transactions.location_id', $permitted_locations);
            }

            if (! auth()->user()->can('access_sell_return') && auth()->user()->can('access_own_sell_return')) {
                $sells->where('transactions.created_by', request()->session()->get('user.id'));
            }

            //Add condition for created_by,used in sales representative sales report
            if (request()->has('created_by')) {
                $created_by = request()->get('created_by');
                if (! empty($created_by)) {
                    $sells->where('transactions.created_by', $created_by);
                }
            }

            //Add condition for location,used in sales representative expense report
            if (request()->has('location_id')) {
                $location_id = request()->get('location_id');
                if (! empty($location_id)) {
                    $sells->where('transactions.location_id', $location_id);
                }
            }

    
            if (! empty(request()->customer_id)) {
                $customer_id = request()->customer_id;
                $sells->where('contacts.id', $customer_id);
            }
            if (! empty(request()->start_date) && ! empty(request()->end_date)) {
                $start = request()->start_date;
                $end = request()->end_date;
                $sells->whereDate('transactions.transaction_date', '>=', $start)
                        ->whereDate('transactions.transaction_date', '<=', $end);
            }

                        
            return Datatables::of($sells)
                ->addColumn(
                    'action',
                    function($row) {
                        $id = $row->id;

                        $payment_status = 'paid';

                        $due_amount = $row->final_total - $row->total_paid;
                        if($due_amount > 0 && $due_amount == $row->final_total){
                            $payment_status = 'due';
                        }
                        if($due_amount > 0 && $due_amount < $row->final_total){
                            $payment_status = 'partial';
                        }
                
                        $html = '<div class="btn-group">
                                    <button type="button" class="btn btn-info dropdown-toggle btn-xs" 
                                        data-toggle="dropdown" aria-expanded="false">'.
                                        __('messages.actions').
                                        '<span class="caret"></span><span class="sr-only">Toggle Dropdown</span>
                                    </button>
                                    <ul class="dropdown-menu dropdown-menu-right" role="menu">';
                
                        $view_url = action('App\Http\Controllers\SellReturnController@show', [$id]);
                        $delete_url = action('App\Http\Controllers\SellReturnController@destroy', [$id]);
                        $print_url = action('App\Http\Controllers\SellReturnController@printInvoice', [$id]);
                        // $edit_url = action('App\Http\Controllers\SellReturnController@edit', [$id]);

                        $add_payment_url = action('App\Http\Controllers\TransactionPaymentController@addPayment', ['g'.$id]);
                        $view_payment_url = action('App\Http\Controllers\TransactionPaymentController@show', ['g'.$id]);
                
                        $html .= '<li><a href="#" class="btn-modal" data-container=".view_modal" data-href="' . $view_url . '"><i class="fas fa-eye" aria-hidden="true"></i> ' . __("messages.view") . '</a></li>';
                        $html .= '<li><a href="' . $delete_url . '" class="delete_sell_return"><i class="fa fa-trash" aria-hidden="true"></i> ' . __("messages.delete") . '</a></li>';
                        $html .= '<li><a href="#" class="print-invoice" data-href="' . $print_url . '"><i class="fa fa-print" aria-hidden="true"></i> ' . __("messages.print") . '</a></li>';             
                        // $html .= '<li><a href="'.$edit_url.'" class=""><i class="fas fa-edit" aria-hidden="true"></i> ' . __("messages.edit") . '</a></li>';
                
                        if ($payment_status != "paid") {
                            $html .= '<li><a href="'.$add_payment_url.'" class="add_payment_modal"><i class="fas fa-money-bill-alt"></i> ' . __("purchase.add_payment") . '</a></li>';
                        }
                
                        $html .= '<li><a href="'.$view_payment_url.'" class="view_payment_modal"><i class="fas fa-money-bill-alt"></i> ' . __("purchase.view_payments") . '</a></li>';
                
                        $html .= '</ul></div>';
                
                        return $html;
                    }
                )
                ->removeColumn('id')
                ->editColumn(
                    'final_total',
                    '<span class="display_currency final_total" data-currency_symbol="true" data-orig-value="{{$final_total}}">{{$final_total}}</span>'
                )
    
                ->editColumn(
                    'name',
                    function($row) {
                        return $row->customer_names;
                    }
                )
                ->editColumn(
                    'invoice_no',
                    function($row) {
                        return $row->invoice_nos;
                    }
                   
                )   
                ->editColumn(
                    'business_location',
                    function($row) {
                        return $row->business_locations;
                    }
                   
                )
                ->editColumn(
                    'payment_status',
                    function($row) {
                        $payment_status = 'paid';
                        $due_amount = $row->final_total - $row->total_paid;
                        
                        if ($due_amount > 0 && $due_amount == $row->final_total) {
                            $payment_status = 'due';
                        }
                        if ($due_amount > 0 && $due_amount < $row->final_total) {
                            $payment_status = 'partial';
                        }
                        
                        return (string) view('sell.partials.payment_status', ['payment_status' => $payment_status, 'id' => 'g'.$row->id]);
                    }
                )
                ->addColumn('payment_due', function ($row) {
                    $due = $row->final_total - $row->total_paid;
                    return '<span class="display_currency payment_due" data-currency_symbol="true" data-orig-value="'.$due.'">'.$due.'</sapn>';
                })
                ->setRowAttr([
                    'data-href' => function ($row) {
                        if (auth()->user()->can('sell.view')) {
                            return  action([\App\Http\Controllers\SellReturnController::class, 'show'], [$row->id]);
                        } else {
                            return '';
                        }
                    }, ])
                ->rawColumns(['final_total', 'action', 'parent_sale', 'payment_status', 'payment_due', 'name'])
                ->make(true);
        }


        $business_locations = BusinessLocation::forDropdown($business_id, false);
        $customers = Contact::customersDropdown($business_id, false);

        $sales_representative = User::forDropdown($business_id, false, false, true);


        return view('sell_return.index')->with(compact('business_locations', 'customers', 'sales_representative'));
    }





    public function addsellReturnList(){

        if (!auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        if (request()->ajax()) {

            $sells = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
                                ->join(
                                    'business_locations AS bl',
                                    'transactions.location_id',
                                    '=',
                                    'bl.id'
                                )
                                ->join(
                                    'transactions as T1',
                                    'transactions.return_parent_id',
                                    '=',
                                    'T1.id'
                                )
                                ->leftJoin(
                                    'transaction_payments AS TP',
                                    'transactions.id',
                                    '=',
                                    'TP.transaction_id'
                                )
                                ->where('transactions.business_id', $business_id)
                                ->where('transactions.type', 'sell_return')
                                ->where('transactions.status', 'final')
                                ->whereNull('transactions.return_group_id')
                                ->select(
                                    'transactions.id',
                                    'transactions.transaction_date',
                                    'transactions.invoice_no',
                                    'contacts.name',
                                    'contacts.supplier_business_name',
                                    'transactions.final_total',
                                    'transactions.payment_status',
                                    'bl.name as business_location',
                                    'T1.invoice_no as parent_sale',
                                    'T1.id as parent_sale_id',
                                    DB::raw('SUM(TP.amount) as amount_paid')
                                );

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sells->whereIn('transactions.location_id', $permitted_locations);
            }

            if (! auth()->user()->can('access_sell_return') && auth()->user()->can('access_own_sell_return')) {
                $sells->where('transactions.created_by', request()->session()->get('user.id'));
            }

            //Add condition for created_by,used in sales representative sales report
            if (request()->has('created_by')) {
                $created_by = request()->get('created_by');
                if (! empty($created_by)) {
                    $sells->where('transactions.created_by', $created_by);
                }
            }

            //Add condition for location,used in sales representative expense report
            if (request()->has('location_id')) {
                $location_id = request()->get('location_id');
                if (! empty($location_id)) {
                    $sells->where('transactions.location_id', $location_id);
                }
            }

            if (! empty(request()->customer_id)) {
                $customer_id = request()->customer_id;
                $sells->where('contacts.id', $customer_id);
            }
            if (! empty(request()->start_date) && ! empty(request()->end_date)) {
                $start = request()->start_date;
                $end = request()->end_date;
                $sells->whereDate('transactions.transaction_date', '>=', $start)
                        ->whereDate('transactions.transaction_date', '<=', $end);
            }

            $sells->groupBy('transactions.id');

            return Datatables::of($sells)
                ->addColumn(
                    'action',
                    '<div class="btn-group">
                    <button type="button" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline tw-dw-btn-info tw-w-max dropdown-toggle" 
                        data-toggle="dropdown" aria-expanded="false">'.
                        __('messages.actions').
                        '<span class="caret"></span><span class="sr-only">Toggle Dropdown
                        </span>
                    </button>
                    <ul class="dropdown-menu dropdown-menu-right" role="menu">
                        <li><a href="#" class="btn-modal" data-container=".view_modal" data-href="{{action(\'App\Http\Controllers\SellReturnController@showmain\', [$parent_sale_id])}}"><i class="fas fa-eye" aria-hidden="true"></i> @lang("messages.view")</a></li>
                        <li><a href="{{action(\'App\Http\Controllers\SellReturnController@add\', [$parent_sale_id])}}" ><i class="fa fa-edit" aria-hidden="true"></i> @lang("messages.edit")</a></li>
                        <li><a href="{{action(\'App\Http\Controllers\SellReturnController@destroyMain\', [$id])}}" class="delete_sell_return" ><i class="fa fa-trash" aria-hidden="true"></i> @lang("messages.delete")</a></li>
                        <li><a href="#" class="print-invoice" data-href="{{action(\'App\Http\Controllers\SellReturnController@printInvoiceMain\', [$id])}}"><i class="fa fa-print" aria-hidden="true"></i> @lang("messages.print")</a></li>

                    @if($payment_status != "paid")
                        <li><a href="{{action(\'App\Http\Controllers\TransactionPaymentController@addPayment\', [$id])}}" class="add_payment_modal"><i class="fas fa-money-bill-alt"></i> @lang("purchase.add_payment")</a></li>
                    @endif

                    <li><a href="{{action(\'App\Http\Controllers\TransactionPaymentController@show\', [$id])}}" class="view_payment_modal"><i class="fas fa-money-bill-alt"></i> @lang("purchase.view_payments")</a></li>
                    </ul>
                    </div>'
                )
                ->removeColumn('id')
                ->editColumn(
                    'final_total',
                    '<span class="display_currency final_total" data-currency_symbol="true" data-orig-value="{{$final_total}}">{{$final_total}}</span>'
                )
                ->editColumn('parent_sale', function ($row) {
                    return '<button type="button" class="btn btn-link btn-modal" data-container=".view_modal" data-href="'.action([\App\Http\Controllers\SellController::class, 'show'], [$row->parent_sale_id]).'">'.$row->parent_sale.'</button>';
                })
                ->editColumn('name', '@if(!empty($supplier_business_name)) {{$supplier_business_name}}, <br> @endif {{$name}}')
                ->editColumn('transaction_date', '{{@format_datetime($transaction_date)}}')
                ->editColumn(
                    'payment_status',
                    '<a href="{{ action([\App\Http\Controllers\TransactionPaymentController::class, \'show\'], [$id])}}" class="view_payment_modal payment-status payment-status-label" data-orig-value="{{$payment_status}}" data-status-name="{{__(\'lang_v1.\' . $payment_status)}}"><span class="label @payment_status($payment_status)">{{__(\'lang_v1.\' . $payment_status)}}</span></a>'
                )
                ->addColumn('payment_due', function ($row) {
                    $due = $row->final_total - $row->amount_paid;

                    return '<span class="display_currency payment_due" data-currency_symbol="true" data-orig-value="'.$due.'">'.$due.'</sapn>';
                })
                ->setRowAttr([
                    'data-href' => function ($row) {
                        if (auth()->user()->can('sell.view')) {
                            return  action([\App\Http\Controllers\SellReturnController::class, 'showmain'], [$row->parent_sale_id]);
                        } else {
                            return '';
                        }
                    }, ])
                ->rawColumns(['final_total', 'action', 'parent_sale', 'payment_status', 'payment_due', 'name'])
                ->make(true);
        }
        $business_locations = BusinessLocation::forDropdown($business_id, false);
        $customers = Contact::customersDropdown($business_id, false);

        $sales_representative = User::forDropdown($business_id, false, false, true);

        return view('sell_return.index2')->with(compact('business_locations', 'customers', 'sales_representative'));

            
    }




    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(){

        if (!auth()->user()->can('sell.create')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        //Check if subscribed or not
        if (!$this->moduleUtil->isSubscribed($business_id)) {
            return $this->moduleUtil->expiredResponse(action([\App\Http\Controllers\SellReturnController::class, 'index']));
        }


        $sub_type = request()->get('sub_type');
        //Check if there is a open register, if no then redirect to Create Register screen.
        if ($this->cashRegisterUtil->countOpenedRegister() == 0) {
            return redirect()->action([\App\Http\Controllers\CashRegisterController::class, 'create'], ['sub_type' => $sub_type]);
        }

        $register_details = $this->cashRegisterUtil->getCurrentCashRegister(auth()->user()->id);
        $default_location = ! empty($register_details->location_id) ? BusinessLocation::findOrFail($register_details->location_id) : null;

        $business_locations = BusinessLocation::forDropdown($business_id, false, true);
        $bl_attributes = $business_locations['attributes'];
        $business_locations = $business_locations['locations'];


        //set first location as default locaton
        if (empty($default_location)) {
            foreach ($business_locations as $id => $name) {
                $default_location = BusinessLocation::findOrFail($id);
                break;
            }
        }

        $types = Contact::getContactTypes();
        $customer_groups = CustomerGroup::forDropdown($business_id);
        $walk_in_customer = $this->contactUtil->getWalkInCustomer($business_id);
        $business_details = $this->businessUtil->getDetails($business_id);


        $pos_settings = empty($business_details->pos_settings) ? $this->businessUtil->defaultPosSettings() : json_decode($business_details->pos_settings, true);
        $default_datetime = $this->businessUtil->format_date('now', true);

        $payment_line = $this->dummyPaymentLine;
        $payment_types = $this->transactionUtil->payment_types(null, false, $business_id);
        $change_return = $this->dummyPaymentLine;


        return view('sell_return.create')->with(compact(
            'business_locations',
            'bl_attributes',
            'default_location',
            'customer_groups',
            'walk_in_customer',
            'business_details',
            'default_datetime',
            'pos_settings',
            'types',
            'payment_line',
            'payment_types',
            'change_return',
        ));


    }


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function add($id)
    {
        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        //Check if subscribed or not
        if (! $this->moduleUtil->isSubscribed($business_id)) {
            return $this->moduleUtil->expiredResponse();
        }

        $sell = Transaction::where('business_id', $business_id)
                            ->with([
                                'sell_lines', 
                                'location', 
                                'return_parent', 
                                'contact', 
                                'tax', 
                                'sell_lines.sub_unit', 
                                'sell_lines.product', 
                                'sell_lines.product.unit',
                                'sell_lines.product.first_conversion_unit',
                                'sell_lines.product.second_conversion_unit'
                            ])
                            ->find($id);
                            
        foreach ($sell->sell_lines as $key => $value) {

            if(!empty($value->sub_unit_id)) {
                $formated_sell_line = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);
                $sell->sell_lines[$key] = $formated_sell_line;
            }   

            $sell->sell_lines[$key]->formatted_qty = $this->transactionUtil->num_f($value->quantity, false, null, true);

            if(!empty($sell->return_parent)){
                $return_sell_line = TransactionSellLine::where('return_sell_line_id', $value->id)
                                                        ->where('transaction_id', $sell->return_parent->id)
                                                        ->where('product_id', $value->product_id)
                                                        ->where('variation_id', $value->variation_id)
                                                        ->first();


                $total_return_quantity = $return_sell_line->quantity;

                $sell->sell_lines[$key]->quantity_returned = $total_return_quantity;


            }
            
        }

        
        $common_settings = session()->get('business.common_settings');

        return view('sell_return.add')->with(compact('sell','common_settings'));


    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function storeAdd(Request $request)
    {
        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }
        $product_total_purchase_array  = [];
        $product_total_sell_array = [];
        $return_total_amount = [];
        try {

            $input = $request->except('_token');

            if (! empty($input['products'])) {
                $business_id = $request->session()->get('user.business_id');


                //Check if subscribed or not
                if (! $this->moduleUtil->isSubscribed($business_id)) {
                    return $this->moduleUtil->expiredResponse(action([\App\Http\Controllers\SellReturnController::class, 'index']));
                }

                $user_id = $request->session()->get('user.id');

                DB::beginTransaction();

                $sell_return = $this->transactionUtil->addSellReturn($input, $business_id, $user_id);

                
                //accounting account transaction by ashiq
                foreach ($input['products'] as $product) {
                    $product_total_purchase_array[] = Variation::where('id', $product['variation_id'])->where('product_id', $product['product_id'])->first()->default_purchase_price;
                    $product_total_sell_array[]     = Variation::where('id', $product['variation_id'])->where('product_id', $product['product_id'])->first()->default_sell_price;
                    $return_total_amount[] = $product['sell_quantity'] * $product['sell_unit_price'];
                }
                $total_purchase = array_sum($product_total_purchase_array);
                $total_sell     = array_sum($product_total_sell_array);
                $total_return_amount  = array_sum($return_total_amount);
                // dd($sell_return, $sell_return['invoice_no'], $input['products'], $total_purchase, $total_sell, $total_return_amount);
                $inventory_accounting_account = AccountingAccount::where('business_id', $business_id)->where('account_type', 'inventory')->first();
                $revenue_of_product_service_accounting_account =  AccountingAccount::where("business_id", $business_id)->where("account_type", "revenue_of_products_and_services_sales")->first();
                $customer_accounting_account = AccountingAccount::where('business_id', $business_id)->where('account_id', $sell_return['contact_id'])->first();
                $cost_of_goods_sold_accounting_account = AccountingAccount::where('business_id',$business_id )->where("account_type", "cost_of_goods_sold")->first();

                $user_id = request()->session()->get('user.id');

                $journal_type = 'journal_entry';
                $journal_ref_no = $sell_return['invoice_no'];

                $acc_trans_mapping                 = new AccountingAccTransMapping();
                $acc_trans_mapping->business_id    = $sell_return['business_id'];
                $acc_trans_mapping->ref_no         = $journal_ref_no;
                $acc_trans_mapping->note           = null;
                $acc_trans_mapping->type           = $journal_type;
                $acc_trans_mapping->created_by     = $user_id;
                $acc_trans_mapping->operation_date = $sell_return['transaction_date'];
                $acc_trans_mapping->save();

                // Only create accounting transactions if all required accounts exist
                if ($inventory_accounting_account && $revenue_of_product_service_accounting_account && $customer_accounting_account && $cost_of_goods_sold_accounting_account) {
                    $inventoryInfo = [
                        "accounting_account_id" => $inventory_accounting_account->id,
                        "amount"                => $total_purchase,
                        "type"                  => "debit",
                        "created_by"            => $user_id,
                        "operation_date"        => $sell_return['transaction_date'],
                        'sub_type'              => "journal_entry",
                        "acc_trans_mapping_id"  => $acc_trans_mapping->id,
                    ];
                    AccountingAccountsTransaction::createTransaction($inventoryInfo);

                    $revenueAndServiceInfo = [
                        "accounting_account_id" => $revenue_of_product_service_accounting_account->id,
                        "amount"                => $sell_return['final_total'],
                        "type"                  => "debit",
                        "created_by"            => $user_id,
                        "operation_date"        => $sell_return['transaction_date'],
                        'sub_type'              => "journal_entry",
                        "acc_trans_mapping_id"  => $acc_trans_mapping->id,
                    ];
                    AccountingAccountsTransaction::createTransaction($revenueAndServiceInfo);

                    $customerInfo = [
                        'amount'                => $sell_return['final_total'],
                        'accounting_account_id' => $customer_accounting_account->id,
                        'created_by'            => $sell_return["created_by"],
                        'operation_date'        => $sell_return['transaction_date'],
                        'note'                  => null,
                        'type'                  => 'credit',
                        'sub_type'              => $sell_return["type"],
                        'transaction_id'        => $sell_return['id'],
                        "acc_trans_mapping_id"  => $acc_trans_mapping->id,
                        // 'transaction_payment_id' => $parent_payment->id
                    ];
                    AccountingAccountsTransaction::createTransaction($customerInfo);

                    $costOfGoodsSoldInfo = [
                        "accounting_account_id" => $cost_of_goods_sold_accounting_account->id,
                        "amount"                => $total_purchase,
                        "type"                  => "credit",
                        "created_by"            => $user_id,
                        "operation_date"        => $sell_return['transaction_date'],
                        'sub_type'              => "journal_entry",
                        "acc_trans_mapping_id"  => $acc_trans_mapping->id,
                    ];
                    AccountingAccountsTransaction::createTransaction($costOfGoodsSoldInfo);
                } else {
                    // Log warning if accounting accounts are missing
                    Log::warning('Sell return accounting transaction skipped - missing accounting accounts', [
                        'business_id' => $business_id,
                        'sell_return_id' => $sell_return['id'] ?? null,
                        'inventory_account' => $inventory_accounting_account ? 'exists' : 'missing',
                        'revenue_account' => $revenue_of_product_service_accounting_account ? 'exists' : 'missing',
                        'customer_account' => $customer_accounting_account ? 'exists' : 'missing',
                        'cogs_account' => $cost_of_goods_sold_accounting_account ? 'exists' : 'missing',
                    ]);
                }


                //accounting account transaction by ashiq end

                $receipt = $this->receiptContent($business_id, $sell_return->location_id, $sell_return->id);

                $product_lines = $input['products'];
                foreach ($product_lines as $product_line) {

                    $product_id = $product_line['product_id'];
                    $variation_id = $product_line['variation_id'];
                    $transaction_id = $input['transaction_id'];
                    $check_serial_no_status = Product::where('id', $product_id)->where('enable_serial_number', '1')->exists();
                    if($check_serial_no_status){
                        $old_serial_check = SerialNumber::where('location_id', $sell_return->location_id)
                                                        ->where('product_id', $product_id)
                                                        ->where('variation_id', $variation_id)
                                                        ->where('transaction_id', $transaction_id)
                                                        ->where('is_return', 1)
                                                        ->exists();
                        if($old_serial_check){
                            SerialNumber::where('location_id', $sell_return->location_id)
                                        ->where('product_id', $product_id)
                                        ->where('transaction_id', $transaction_id)
                                        ->where('is_return', 1)
                                        ->update([
                                            'is_return' => null,
                                            'stock_status' => 'out',
                                        ]);
                        }
                        
                        if(isset($product_line['product_serial_number'])){
                            $serial_numbers = $product_line['product_serial_number'];
                            SerialNumber::where('location_id', $sell_return->location_id)
                                        ->where('product_id', $product_id)
                                        ->where('variation_id', $variation_id)
                                        ->whereIn('serial_number', $serial_numbers)
                                        ->update([
                                            'stock_status' => 'available',
                                            'is_return' => 1,
                                        ]);
                            
                        }
                    }

                }

                DB::commit();

                $output = [
                    'success' => 1,
                    'msg' => __('lang_v1.success'),
                    'receipt' => $receipt,
                ];

            }

        } catch (\Exception $e) {

            DB::rollBack();

            if (get_class($e) == \App\Exceptions\PurchaseSellMismatch::class) {
                $msg = $e->getMessage();
            } else {
                \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
                $msg = __('messages.something_went_wrong');
            }

            $output = [
                'success' => 0,
                'msg' => $msg,
            ];

        }

        return $output;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request){
      
        try {
            
            $input = $request->except('_token');

            if (!empty($input['products'])) {
                $business_id = $request->session()->get('user.business_id');

                //Check if subscribed or not
                if (!$this->moduleUtil->isSubscribed($business_id)) {
                    return $this->moduleUtil->expiredResponse(action([\App\Http\Controllers\SellReturnController::class, 'index']));
                }

                $user_id = $request->session()->get('user.id');

                DB::beginTransaction();

                $return_group = ReturnGroup::create([
                    'business_id' => $business_id
                ]);


                $input_except_products = $request->except('_token', 'products', 'payment');

                $grouped = [];

                foreach ($input['products'] as $gt_product) {
                    $tid = $gt_product['transaction_id'];
                    $grouped[$tid]['products'][] = $gt_product;
                }

                $transfer_location_id = $request->select_location_id;
                
                
                //Update reference count
                $ref_count = $this->transactionUtil->setAndGetReferenceCount('sell_return', $business_id);

                $input_except_products['invoice_no'] = $this->transactionUtil->generateReferenceNumber('sell_return', $ref_count, $business_id);

                $return_transaction_ids = [];
                if(!empty($grouped)){
                    foreach ($grouped as $key => $value) {
                        $input_except_products['products'] = $value['products'];
                        $input_except_products['transaction_id'] = $key;
                        $input_except_products['return_transfer_location'] = $transfer_location_id;
                        $input_except_products['return_group_id'] = $return_group->id;
                        if (!empty($value['products'][0]['location_id'])) {
                            $input_except_products['location_id'] = $value['products'][0]['location_id'];
                        }

                        $sell_return = $this->transactionUtil->addMultiSellReturn($input_except_products, $business_id, $user_id);
                        $this->transactionUtil->createOrUpdateSellReturnLines($sell_return, $value['products'], $transfer_location_id);
          
                        $return_transaction_ids[] = $sell_return->id;

                    }
                }

                dd($sell_return);


                

                if(!empty($sell_return->sell_lines)){
    
                    // SELL Transfer Logic
                    $grouped_location = [];
                    foreach ($input['products'] as $gl_product) {
                        if($transfer_location_id != $gl_product['location_id']){
                            $lid = $gl_product['location_id'];
                            $grouped_location[$lid]['products'][] = $gl_product;
                        }
                    }

                    $input_for_transfer = [];
                    $return_transfer_ids = [];
                    if(!empty($grouped_location)){
                        foreach ($grouped_location as $key => $l_value) {
                            if (!empty($l_value['products'][0]['location_id'])) {
                                $input_for_transfer['location_id'] = $l_value['products'][0]['location_id'];
                            }

                            if ($transfer_location_id != $input_for_transfer['location_id']) {

                                $input_for_transfer['transaction_date'] = $request->transaction_date;
                                $input_for_transfer['status'] = "completed";
            
                                $input_for_transfer['shipping_charges'] = 0;
                                $input_for_transfer['additional_notes'] = null;
                                $input_for_transfer['return_group_id'] = $return_group->id;

                                $input_for_transfer['products'] = $l_value['products'];
                                $input_for_transfer['transaction_id'] = $key;
    
                                $invoice_total = $this->productUtil->calculateInvoiceTotal($input_for_transfer['products'], null, null, true);
    
                                //Update reference count
                                $ref_count = $this->productUtil->setAndGetReferenceCount('stock_transfer');
                                //Generate reference number
                                if (empty($input_data['ref_no'])) {
                                    $input_for_transfer['ref_no'] = $this->productUtil->generateReferenceNumber('stock_transfer', $ref_count);
                                }
    
                                if (!empty($l_value['products'][0]['location_id'])) {
                                    $input_for_transfer['location_id'] = $l_value['products'][0]['location_id'];
                                }
    
                                $input_for_transfer['final_total'] = $invoice_total['final_total'];
                                $input_for_transfer['total_before_tax'] = $invoice_total['final_total'];
                                $input_for_transfer['type'] = 'sell_transfer';
                                $input_for_transfer['business_id'] = $business_id;
                                $input_for_transfer['created_by'] = $user_id;
                                $input_for_transfer['payment_status'] = "paid";
                                $input_for_transfer['status'] = "final";

                                //Create Sell Transfer transaction
                                $sell_transfer = Transaction::create($input_for_transfer);

                                $input_for_transfer['location_id'] = $request->select_location_id; //transfer_location_id
                                $input_for_transfer['type'] = 'purchase_transfer';
                                $input_for_transfer['transfer_parent_id'] = $sell_transfer->id;
                                $input_for_transfer['status'] = 'received';

                                $purchase_transfer = Transaction::create($input_for_transfer);


                                $sell_lines = [];
                                $purchase_lines = [];

                                if (!empty($input_for_transfer['products'])) {
                                    foreach ($input_for_transfer['products'] as $product) {

                                        $sell_line_arr = [
                                            'product_id' => $product['product_id'],
                                            'variation_id' => $product['variation_id'],
                                            'quantity' => $this->productUtil->num_uf($product['quantity']),
                                            'item_tax' => 0,
                                            'tax_id' => null, 
                                            'sell_line_id' => null, 
                                            'is_return_transfer' => 1, 
                                        ];

                                        if (!empty($product['product_unit_id'])) {
                                            $sell_line_arr['product_unit_id'] = $product['product_unit_id'];
                                        }


                                        $purchase_line_arr = $sell_line_arr;

                                        if (!empty($product['base_unit_multiplier'])) {
                                            $sell_line_arr['base_unit_multiplier'] = $product['base_unit_multiplier'];
                                        }

                                        $sell_line_arr['unit_price'] = $this->productUtil->num_uf($product['unit_price']);
                                        $sell_line_arr['unit_price_inc_tax'] = $sell_line_arr['unit_price'];

                                        $purchase_line_arr['purchase_price'] = $sell_line_arr['unit_price'];
                                        $purchase_line_arr['purchase_price_inc_tax'] = $sell_line_arr['unit_price'];
                                        $purchase_line_arr['transfer_return'] = 1;

                                        if (!empty($product['base_unit_multiplier'])) {
                                            $purchase_line_arr['quantity'] = $purchase_line_arr['quantity'] * $product['base_unit_multiplier'];
                                            $purchase_line_arr['purchase_price'] = $purchase_line_arr['purchase_price'] / $product['base_unit_multiplier'];
                                            $purchase_line_arr['purchase_price_inc_tax'] = $purchase_line_arr['purchase_price_inc_tax'] / $product['base_unit_multiplier'];
                                        }



                                        if (isset($purchase_line_arr['sub_unit_id']) && $purchase_line_arr['sub_unit_id'] == $purchase_line_arr['product_unit_id']) {
                                            unset($purchase_line_arr['sub_unit_id']);
                                        }

                                        unset($purchase_line_arr['product_unit_id']);

                                        $sell_lines[] = $sell_line_arr;
                                        $purchase_lines[] = $purchase_line_arr;
                                    }
                                }

                                //Sell Product from first location
                                if (!empty($sell_lines)) {
                                    $this->transactionUtil->createOrUpdateSellReturnLines($sell_transfer, $sell_lines, $input_for_transfer['location_id'], false, null, [], false, true);
                                }
                                //Purchase product in second location
                                if (!empty($purchase_lines)) {
                                    $purchase_transfer->purchase_lines()->createMany($purchase_lines);
                                }

                                //Adjust stock over selling if found
                                $this->productUtil->adjustStockOverSelling($purchase_transfer);

                                $this->transactionUtil->activityLog($sell_transfer, 'added');

                                $return_purchase_transfer_ids[] = $purchase_transfer->id;
                                $return_transfer_ids[] = $sell_transfer->id;

                            }


                        }
                    }

                }

                // Add change return
                if (!empty($input['payment']['change_return'])) {
                    unset($input['payment']['change_return']);
                }

                if (!empty($input['payment'])) {

                    $total_payment_amount = $this->transactionUtil->num_uf($input['payment'][0]['amount']);

                    if(!empty($return_transaction_ids)){
                        foreach($return_transaction_ids as $return_transaction){

                            $return = Transaction::find($return_transaction);

                            $paid_amount = $this->transactionUtil->getTotalPaid($return->id);
                            $this_transaction_amount = $return->final_total - $paid_amount;
    
                            if($total_payment_amount > 0){
                                $input['payment'][0]['amount'] = $this->transactionUtil->num_uf(min($total_payment_amount, $this_transaction_amount));
                            }
    
               
                            $total_payment_amount -= $input['payment'][0]['amount'];
    
                            $this->transactionUtil->createOrUpdatePaymentLines($return, $input['payment']);
                            //Update payment status
                            $payment_status = $this->transactionUtil->updatePaymentStatus($return->id, $return->final_total);
                            $return->payment_status = $payment_status;

                        }
                    }
                    
                }

                $receipt = $this->receiptContentMultiReturn($business_id, $transfer_location_id, $return_group->id, null, false, true, null);

                DB::commit();

                $output = [
                    'success' => 1,
                    'msg' => __('lang_v1.success'),
                    'receipt' => $receipt,
                ];

            }


        } catch (\Exception $e) {
            DB::rollBack();

            if (get_class($e) == \App\Exceptions\PurchaseSellMismatch::class) {
                $msg = $e->getMessage();
            } else {
                \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
                $msg = __('messages.something_went_wrong');
            }

            $output = [
                'success' => 0,
                'msg' => $msg,
            ];
        }

        return $output;
    }



    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        if(! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }


        $business_id = request()->session()->get('user.business_id');


        $single_sell = ReturnGroup::where('return_groups.business_id', $business_id)
                                ->where('return_groups.id', $id)
                                ->where('transactions.type', 'sell_return')
                                ->join('transactions', 'transactions.return_group_id', '=', 'return_groups.id')
                                ->leftJoin('transactions AS ps', 'ps.id', '=', 'transactions.return_parent_id')
                                ->leftJoin('contacts AS pc', 'ps.contact_id', '=', 'pc.id')
                                ->leftJoin('business_locations AS pbl', 'ps.location_id', '=', 'pbl.id')
                                ->join('contacts', 'transactions.contact_id', '=', 'contacts.id')
                                ->join('business_locations AS bl', 'transactions.location_id', '=', 'bl.id')
                                ->groupBy('return_groups.id')
                                ->select(
                                    'return_groups.id',
                                    DB::raw("MIN(transactions.transaction_date) as transaction_date"),
                                    DB::raw("GROUP_CONCAT(DISTINCT CASE WHEN contacts.name != '' THEN contacts.name ELSE contacts.supplier_business_name END SEPARATOR ', ') as customer_name"),
                                    DB::raw("GROUP_CONCAT(DISTINCT bl.name SEPARATOR ', ') as business_location"),
                                    DB::raw("GROUP_CONCAT(DISTINCT transactions.invoice_no SEPARATOR ', ') as invoice_no"),
                                    
                                    
                                    
                                    DB::raw("GROUP_CONCAT(DISTINCT ps.transaction_date SEPARATOR ', ') as parent_transaction_date"),

                                    DB::raw("GROUP_CONCAT(DISTINCT CASE WHEN pc.name != '' THEN pc.name ELSE pc.supplier_business_name END SEPARATOR ', ') as parent_customer_name"),
                                    DB::raw("GROUP_CONCAT(DISTINCT pbl.name SEPARATOR ', ') as parent_business_location"),
                                    DB::raw("GROUP_CONCAT(DISTINCT ps.invoice_no SEPARATOR ', ') as parent_invoice_no"),



                                    DB::raw('SUM(transactions.final_total) as final_total'),
                                    DB::raw("SUM(IF(transactions.type = 'sell_return', (SELECT SUM(amount) FROM transaction_payments WHERE transaction_payments.transaction_id=transactions.id), 0)) as total_paid")
                                )->first();
    


        $sell_returns = ReturnGroup::where('business_id', $business_id)
                                ->where('id', $id)
                                ->with(
                                    'transactions',

                                    'transactions.return_parent',
                                    'transactions.tax',
                                    'transactions.contact',
                                    'transactions.sell_lines',
                                    'transactions.sell_lines.product',
                                    'transactions.sell_lines.variations',
                                    'transactions.sell_lines.sub_unit',
                                    'transactions.sell_lines.product',
                                    'transactions.sell_lines.product.unit',
                                    'transactions.location',
                                )->first();

   

        $sell_taxes = [];
        $total_discount = 0;
        $activities = [];

        foreach ($sell_returns->transactions as $key => $sell) {

            if($sell->type == 'sell_return'){ 


       
                foreach ($sell->sell_lines as $key => $value) {
                    if (!empty($value->sub_unit_id)) {
                        $formated_sell_line = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);
                        $sell->sell_lines[$key] = $formated_sell_line;
                    }
                }





            // if (! empty($sell->return_parent->tax)) {
            //     if ($sell->return_parent->tax->is_tax_group) {
            //         $sell_taxes = $this->transactionUtil->sumGroupTaxDetails($this->transactionUtil->groupTaxDetails($sell->return_parent->tax, $sell->return_parent->tax_amount));
            //     } else {
            //         $sell_taxes[$sell->return_parent->tax->name] = $sell->return_parent->tax_amount;
            //     }
            // }


            // if ($sell->return_parent->discount_type == 'fixed') {
            //     $total_discount = $sell->return_parent->discount_amount;
            // } elseif ($sell->return_parent->discount_type == 'percentage') {
            //     $discount_percent = $sell->return_parent->discount_amount;
            //     if ($discount_percent == 100) {
            //         $total_discount = $sell->return_parent->total_before_tax;
            //     } else {
            //         $total_after_discount = $sell->return_parent->final_total - $sell->return_parent->tax_amount;
            //         $total_before_discount = $total_after_discount * 100 / (100 - $discount_percent);
            //         $total_discount = $total_before_discount - $total_after_discount;
            //     }
            // }

            }
        }

        return view('sell_return.show')->with(compact(
            'single_sell', 
            'sell_returns', 
            'sell_taxes', 
            'total_discount', 
            'activities'
        ));


    }


    public function showmain($id){

        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $query = Transaction::where('business_id', $business_id)
                                ->where('id', $id)
                                ->with(
                                    'contact',
                                    'return_parent',
                                    'tax',
                                    'sell_lines',
                                    'sell_lines.product',
                                    'sell_lines.variations',
                                    'sell_lines.sub_unit',
                                    'sell_lines.product',
                                    'sell_lines.product.unit',
                                    'location'
                                );

        if (! auth()->user()->can('access_sell_return') && auth()->user()->can('access_own_sell_return')) {
            $sells->where('created_by', request()->session()->get('user.id'));
        }
        $sell = $query->first();

        foreach ($sell->sell_lines as $key => $value) {
            if (! empty($value->sub_unit_id)) {
                $formated_sell_line = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);
                $sell->sell_lines[$key] = $formated_sell_line;
            }
        }

        $sell_taxes = [];
        if (! empty($sell->return_parent->tax)) {
            if ($sell->return_parent->tax->is_tax_group) {
                $sell_taxes = $this->transactionUtil->sumGroupTaxDetails($this->transactionUtil->groupTaxDetails($sell->return_parent->tax, $sell->return_parent->tax_amount));
            } else {
                $sell_taxes[$sell->return_parent->tax->name] = $sell->return_parent->tax_amount;
            }
        }

        $total_discount = 0;
        if ($sell->return_parent->discount_type == 'fixed') {
            $total_discount = $sell->return_parent->discount_amount;
        } elseif ($sell->return_parent->discount_type == 'percentage') {
            $discount_percent = $sell->return_parent->discount_amount;
            if ($discount_percent == 100) {
                $total_discount = $sell->return_parent->total_before_tax;
            } else {
                $total_after_discount = $sell->return_parent->final_total - $sell->return_parent->tax_amount;
                $total_before_discount = $total_after_discount * 100 / (100 - $discount_percent);
                $total_discount = $total_before_discount - $total_after_discount;
            }
        }

        $activities = Activity::forSubject($sell->return_parent)->with(['causer', 'subject'])->latest()->get();

        return view('sell_return.show-main')->with(compact('sell', 'sell_taxes', 'total_discount', 'activities'));

    }



    public function destroyMain($id){
        
        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            try {
                $business_id = request()->session()->get('user.business_id');
                //Begin transaction
                DB::beginTransaction();

                $query = Transaction::where('id', $id)
                    ->where('business_id', $business_id)
                    ->where('type', 'sell_return')
                    ->with(['sell_lines', 'payment_lines']);

                if (! auth()->user()->can('access_sell_return') && auth()->user()->can('access_own_sell_return')) {
                    $sells->where('created_by', request()->session()->get('user.id'));
                }
                $sell_return = $query->first();

                $sell_lines = TransactionSellLine::where('transaction_id',
                                            $sell_return->return_parent_id)
                                    ->get();

                if (! empty($sell_return)) {
                    $transaction_payments = $sell_return->payment_lines;

                    foreach ($sell_lines as $sell_line) {
                        if ($sell_line->quantity_returned > 0) {
                            $quantity = 0;
                            $quantity_before = $this->transactionUtil->num_f($sell_line->quantity_returned);

                            $sell_line->quantity_returned = 0;
                            $sell_line->save();

                            //update quantity sold in corresponding purchase lines
                            $this->transactionUtil->updateQuantitySoldFromSellLine($sell_line, 0, $quantity_before);

                            // Update quantity in variation location details
                            // Get warehouse_id from sell_return or use default
                            $business_id = request()->session()->get('user.business_id');
                            $warehouse_id = isset($sell_return->warehouse_id) ? $sell_return->warehouse_id : null;
                            if (empty($warehouse_id)) {
                                $warehouse_id = $this->productUtil->getWarehouseIdOrDefault(null, $business_id);
                            }
                            
                            if (!empty($warehouse_id)) {
                                $this->productUtil->updateProductQuantityWarehouse($warehouse_id, $sell_line->product_id, $sell_line->variation_id, 0, $quantity_before, 'sale_return', null, 'Sell return deleted');
                            } else {
                                $this->productUtil->updateProductQuantity($sell_return->location_id, $sell_line->product_id, $sell_line->variation_id, 0, $quantity_before);
                            }
                        }
                    }

                    $sell_return->delete();
                    foreach ($transaction_payments as $payment) {
                        event(new TransactionPaymentDeleted($payment));
                    }
                }

                DB::commit();
                $output = ['success' => 1,
                    'msg' => __('lang_v1.success'),
                ];
            } catch (\Exception $e) {
                DB::rollBack();

                if (get_class($e) == \App\Exceptions\PurchaseSellMismatch::class) {
                    $msg = $e->getMessage();
                } else {
                    \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
                    $msg = __('messages.something_went_wrong');
                }

                $output = ['success' => 0,
                    'msg' => $msg,
                ];
            }

            return $output;
        }

    }
    
    public function printInvoiceMain(Request $request, $transaction_id){
        
        if (request()->ajax()) {
            try {
                $output = ['success' => 0,
                    'msg' => trans('messages.something_went_wrong'),
                ];

                $business_id = $request->session()->get('user.business_id');

                $transaction = Transaction::where('business_id', $business_id)
                                ->where('id', $transaction_id)
                                ->first();

                if (empty($transaction)) {
                    return $output;
                }

                $receipt = $this->receiptContent($business_id, $transaction->location_id, $transaction_id, 'browser');

                if (! empty($receipt)) {
                    $output = ['success' => 1, 'receipt' => $receipt];
                }
            } catch (\Exception $e) {
                $output = ['success' => 0,
                    'msg' => trans('messages.something_went_wrong'),
                ];
            }

            return $output;
        }

    }
    

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if (! auth()->user()->can('access_sell_return') && ! auth()->user()->can('access_own_sell_return')) {
            abort(403, 'Unauthorized action.');
        }

         if (request()->ajax()) {
            try {

                $business_id = request()->session()->get('user.business_id');
                //Begin transaction
                DB::beginTransaction();



                $returns = Transaction::where('return_group_id', $id)
                                ->where('business_id', $business_id)
                                ->where('type', 'sell_return')
                                ->with(['sell_lines', 'payment_lines'])
                                ->get();


                if ($returns->isNotEmpty()) {
                    $return_sell_transfers = Transaction::where('return_group_id', $id)
                                    ->where('business_id', $business_id)
                                    ->where('type', 'sell_transfer')
                                    ->get();   

   
                    if ($return_sell_transfers->isNotEmpty()){

                        $return_purchase_transfer = Transaction::where('return_group_id', $id)
                                                                ->where('business_id', $business_id)
                                                                ->where('type', 'purchase_transfer')
                                                                ->get();

                        foreach($return_purchase_transfer as $purchase_transfer){
                            //Check if any transfer stock is deleted and delete purchase lines
                            $purchase_lines = $purchase_transfer->purchase_lines;
                            foreach ($purchase_lines as $purchase_line) {
                                if ($purchase_line->quantity_sold > 0) {
                                    $output = [
                                        'success' => 0,
                                        'msg' => __('lang_v1.stock_transfer_cannot_be_deleted'),
                                    ];
                                    return $output;
                                }
                            }
                        }

                        $deleted_sell_purchase_ids = [];

                        foreach ($return_sell_transfers as $sell_transfer_s) {
                            foreach ($sell_transfer_s->sell_lines as $sell_line) {
                                $purchase_sell_line = TransactionSellLinesPurchaseLines::where('sell_line_id', $sell_line->id)->first();
                                if (! empty($purchase_sell_line)) {   
                                    $deleted_sell_purchase_ids[] = $purchase_sell_line->id;
                                }
                            }
                        }


                        //Delete sale line purchase line
                        if(!empty($deleted_sell_purchase_ids)) {
                            TransactionSellLinesPurchaseLines::whereIn('id', $deleted_sell_purchase_ids)->delete();
                        }

                        $return_purchase_transfer->each->delete();
                        $return_sell_transfers->each->delete();


                    }else{

                        foreach($returns as $sell_return){

                            $sell_lines = TransactionSellLine::where('transaction_id', $sell_return->id)->get();

                            if (! empty($sell_lines)) {
                                $transaction_payments = $sell_return->payment_lines;

                                foreach ($sell_lines as $sell_line) {
                                    if ($sell_line->quantity > 0 && $sell_line->transfer_return != 1) {
                                        $quantity_before = $this->transactionUtil->num_f($sell_line->quantity);
                                        //update quantity sold in corresponding purchase lines
                                        $this->transactionUtil->updateQuantitySoldFromSellLine($sell_line, 0, $quantity_before);
                                        // Update quantity in variation location details
                                        $this->productUtil->updateProductQuantity($sell_return->location_id, $sell_line->product_id, $sell_line->variation_id, 0, $quantity_before);
                                    }
                                }

                                $sell_return->delete();
                                foreach ($transaction_payments as $payment) {
                                    event(new TransactionPaymentDeleted($payment));
                                }
                            }


                        }

                    }  

                }




                $sell_return =$returns->first();


                if (empty($sell_return)) {
                    $output = [
                        'success' => 0,
                        'msg' => __('lang_v1.sell_return_not_found'),
                    ];
                    return $output;
                }

                $sell_lines = TransactionSellLine::where('transaction_id', $sell_return->return_parent_id)->get();


                if (!empty($sell_return)) {
                    
                    $transaction_payments = $sell_return->payment_lines;

                    foreach ($sell_lines as $sell_line) {
                        if ($sell_line->quantity_returned > 0) {

                            $quantity = 0;
                            $quantity_before = $this->transactionUtil->num_f($sell_line->quantity_returned);

                            $sell_line->quantity_returned = 0;
                            $sell_line->save();

                            if(Product::where('id', $sell_line->product_id)->where('enable_serial_number', '1')->exists()){
                                SerialNumber::where('location_id', $sell_return->location_id)
                                            ->where('stock_status', "available")
                                            ->where('product_id', $sell_line->product_id)
                                            ->where('variation_id', $sell_line->variation_id)
                                            ->where('transaction_id', $sell_line->transaction_id)
                                            ->where('is_return', 1)->update([
                                                'is_return' => null,
                                                'stock_status' => "out"
                                            ]);
                            }

                            // update quantity sold in corresponding purchase lines
                            $this->transactionUtil->updateQuantitySoldFromSellLine($sell_line, 0, $quantity_before);

                            // Update quantity in variation location details
                            // Get warehouse_id from sell_return or use default
                            $business_id = request()->session()->get('user.business_id');
                            $warehouse_id = isset($sell_return->warehouse_id) ? $sell_return->warehouse_id : null;
                            if (empty($warehouse_id)) {
                                $warehouse_id = $this->productUtil->getWarehouseIdOrDefault(null, $business_id);
                            }
                            
                            if (!empty($warehouse_id)) {
                                $this->productUtil->updateProductQuantityWarehouse($warehouse_id, $sell_line->product_id, $sell_line->variation_id, 0, $quantity_before, 'sale_return', null, 'Sell return deleted');
                            } else {
                                $this->productUtil->updateProductQuantity($sell_return->location_id, $sell_line->product_id, $sell_line->variation_id, 0, $quantity_before);
                            }

                        }
                    }

                    foreach ($transaction_payments as $payment) {
                        event(new TransactionPaymentDeleted($payment));
                    }

                }


                $returns->each->delete();


                DB::commit();

                $output = [
                    'success' => 1,
                    'msg' => __('lang_v1.success'),
                ];

            } catch (\Exception $e) {
                DB::rollBack();

                if (get_class($e) == \App\Exceptions\PurchaseSellMismatch::class) {
                    $msg = $e->getMessage();
                } else {
                    \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
                    $msg = __('messages.something_went_wrong');
                }

                $output = [
                    'success' => 0,
                    'msg' => $msg,
                ];
            }

            return $output;
        }


    }

    /**
     * Returns the content for the receipt
     *
     * @param  int  $business_id
     * @param  int  $location_id
     * @param  int  $transaction_id
     * @param  string  $printer_type = null
     * @return array
     */
    private function receiptContent(
        $business_id,
        $location_id,
        $transaction_id,
        $printer_type = null
    ) {
        $output = [
            'is_enabled' => false,
            'print_type' => 'browser',
            'html_content' => null,
            'printer_config' => [],
            'data' => [],
        ];

        $business_details = $this->businessUtil->getDetails($business_id);
        $location_details = BusinessLocation::find($location_id);

        //Check if printing of invoice is enabled or not.
        if ($location_details->print_receipt_on_invoice == 1) {
            //If enabled, get print type.
            $output['is_enabled'] = true;

            $invoice_layout = $this->businessUtil->invoiceLayout($business_id, $location_details->invoice_layout_id);

            //Check if printer setting is provided.
            $receipt_printer_type = is_null($printer_type) ? $location_details->receipt_printer_type : $printer_type;

            $receipt_details = $this->transactionUtil->getReceiptDetails($transaction_id, $location_id, $invoice_layout, $business_details, $location_details, $receipt_printer_type);

            //If print type browser - return the content, printer - return printer config data, and invoice format config
            $output['print_title'] = $receipt_details->invoice_no;
            if ($receipt_printer_type == 'printer') {
                $output['print_type'] = 'printer';
                $output['printer_config'] = $this->businessUtil->printerConfig($business_id, $location_details->printer_id);
                $output['data'] = $receipt_details;
            } else {
                $output['html_content'] = view('sell_return.receipt', compact('receipt_details'))->render();
            }
        }

        return $output;
    }


    /**
     * Returns the content for the receipt
     *
     * @param  int  $business_id
     * @param  int  $location_id
     * @param  int  $transaction_id
     * @param  string  $printer_type = null
     * @return array
     */
    private function receiptContentMultiReturn( $business_id, $location_id, $return_group_id, $printer_type = null, $is_package_slip = false, $from_pos_screen = true, $invoice_layout_id = null, $is_delivery_note = false){
        
        $output = [
            'is_enabled' => false,
            'print_type' => 'browser',
            'html_content' => null,
            'printer_config' => [],
            'data' => [],
        ];

        $business_details = $this->businessUtil->getDetails($business_id);
        $location_details = BusinessLocation::find($location_id);

        //Check if printing of invoice is enabled or not.
        if ($location_details->print_receipt_on_invoice == 1) {
            //If enabled, get print type.
            $output['is_enabled'] = true;

            $invoice_layout = $this->businessUtil->invoiceLayout($business_id, $location_details->invoice_layout_id);

            //Check if printer setting is provided.
            $receipt_printer_type = is_null($printer_type) ? $location_details->receipt_printer_type : $printer_type;

            $receipt_details = $this->transactionUtil->getMultiReceiptDetails($return_group_id, $location_id, $invoice_layout, $business_details, $location_details, $receipt_printer_type);

            //If print type browser - return the content, printer - return printer config data, and invoice format config
            $output['print_title'] = $receipt_details->invoice_no;
            if ($receipt_printer_type == 'printer') {
                $output['print_type'] = 'printer';
                $output['printer_config'] = $this->businessUtil->printerConfig($business_id, $location_details->printer_id);
                $output['data'] = $receipt_details;
            } else {
                $output['html_content'] = view('sell_return.receipt', compact('receipt_details'))->render();
            }
        }

        return $output;
    }

    /**
     * Prints invoice for sell
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function printInvoice(Request $request, $return_group_id)
    {
        if (request()->ajax()) {


            try {
                $output = [
                    'success' => 0,
                    'msg' => trans('messages.something_went_wrong'),
                ];

                $business_id = $request->session()->get('user.business_id');

                $transaction = Transaction::where('business_id', $business_id)
                                ->where('return_group_id', $return_group_id)
                                ->first();

                if (empty($transaction)) {
                    return $output;
                }
                                
                $location_id = $transaction->return_transfer_location ? $transaction->return_transfer_location : $transaction->location_id; 
                $receipt = $this->receiptContentMultiReturn($business_id, $location_id, $return_group_id, null, false, true, null);

                if (! empty($receipt)) {
                    $output = [
                        'success' => 1, 
                        'receipt' => $receipt
                    ];
                }

            } catch (\Exception $e) {
                $output = [
                    'success' => 0,
                    'msg' => trans('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    /**
     * Function to validate sell for sell return
     */
    public function validateInvoiceToReturn($invoice_no)
    {
        if (! auth()->user()->can('sell.create') && ! auth()->user()->can('direct_sell.access') && ! auth()->user()->can('view_own_sell_only')) {
            return ['success' => 0,
                'msg' => trans('lang_v1.permission_denied'),
            ];
        }

        $business_id = request()->session()->get('user.business_id');
        $query = Transaction::where('business_id', $business_id)
                            ->where('invoice_no', $invoice_no);

        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('transactions.location_id', $permitted_locations);
        }

        if (! auth()->user()->can('direct_sell.access') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', auth()->user()->id);
        }

        $sell = $query->first();

        if (empty($sell)) {
            return ['success' => 0,
                'msg' => trans('lang_v1.sell_not_found'),
            ];
        }

        return ['success' => 1,
            'redirect_url' => action([\App\Http\Controllers\SellReturnController::class, 'add'], [$sell->id]),
        ];
    }


    

    public function validateInvoiceToExchange($invoice_no)
    {
        if (! auth()->user()->can('sell.create') && ! auth()->user()->can('direct_sell.access') && ! auth()->user()->can('view_own_sell_only')) {
            return ['success' => 0,
                'msg' => trans('lang_v1.permission_denied'),
            ];
        }

        $business_id = request()->session()->get('user.business_id');
        $query = Transaction::where('business_id', $business_id)
                            ->where('invoice_no', $invoice_no);

        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('transactions.location_id', $permitted_locations);
        }

        if (! auth()->user()->can('direct_sell.access') && auth()->user()->can('view_own_sell_only')) {
            $query->where('created_by', auth()->user()->id);
        }

        $sell = $query->first();

        if (empty($sell)) {
            return ['success' => 0,
                'msg' => trans('lang_v1.sell_not_found'),
            ];
        }

        return ['success' => 1,
            'redirect_url' => action([\App\Http\Controllers\SellExchangeController::class, 'createExchange'], [$sell->id]),
        ];
    }




}
