<?php

namespace App\Http\Controllers;

use App\User;
use Exception;
use App\Account;
use App\Contact;
use App\Product;
use App\TaxRate;
use App\Transaction;
use App\PurchaseLine;
use App\SellExchange;
use App\SerialNumber;
use App\InvoiceScheme;
use App\BusinessLocation;
use App\Utils\ModuleUtil;
use App\Utils\ContactUtil;
use App\Utils\ProductUtil;
use App\AccountTransaction;
use App\TransactionPayment;
use App\Utils\BusinessUtil;
use App\TransactionSellLine;
use Illuminate\Http\Request;
use App\Utils\TransactionUtil;
use App\CashRegisterTransaction;
use App\VariationLocationDetails;
use Illuminate\Support\Facades\DB;
use Spatie\Activitylog\Models\Activity;
use Yajra\DataTables\Facades\DataTables;
use App\TransactionSellLinesPurchaseLines;

class SellExchangeController extends Controller
{

    protected $contactUtil;

    protected $businessUtil;

    protected $transactionUtil;

    protected $productUtil;
    protected $moduleUtil;
    protected $dummyPaymentLine;
    protected $shipping_status_colors;

    public function __construct(ContactUtil $contactUtil, BusinessUtil $businessUtil, TransactionUtil $transactionUtil, ModuleUtil $moduleUtil, ProductUtil $productUtil)
    {
        $this->contactUtil     = $contactUtil;
        $this->businessUtil    = $businessUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil      = $moduleUtil;
        $this->productUtil     = $productUtil;

        $this->dummyPaymentLine = [
            'method' => '',
            'amount'        => 0,
            'note' => '',
            'card_transaction_number' => '',
            'card_number' => '',
            'card_type' => '',
            'card_holder_name' => '',
            'card_month' => '',
            'card_year' => '',
            'card_security' => '',
            'cheque_number' => '',
            'bank_account_number' => '',
            'is_return'                         => 0,
            'transaction_no' => ''
        ];

        $this->shipping_status_colors = [
            'ordered'   => 'bg-yellow',
            'packed'    => 'bg-info',
            'shipped'   => 'bg-navy',
            'delivered' => 'bg-green',
            'cancelled' => 'bg-red',
        ];
    }

    public function index()
    {
        if (! auth()->user()->can('access_sell_exchange') && ! auth()->user()->can('access_own_sell_exchange')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        if (request()->ajax()) {
            $sell_exchanges = SellExchange::where('sell_exchange.business_id', $business_id)
                ->join('transactions', 'transactions.id', '=', 'sell_exchange.transaction_id')
                ->join('contacts', 'transactions.contact_id', '=', 'contacts.id')
                ->join('business_locations AS bl', 'transactions.location_id', '=', 'bl.id')
                ->leftJoin('transaction_payments AS tp', 'transactions.id', '=', 'tp.transaction_id')
                ->leftJoin('sell_exchange_product AS sep', 'sell_exchange.id', '=', 'sep.sell_exchange_id')
                ->select(
                    'sell_exchange.id',
                    'transactions.transaction_date',
                    'transactions.invoice_no',
                    DB::raw("CASE WHEN contacts.name != '' THEN contacts.name ELSE contacts.supplier_business_name END as customer_name"),
                    'bl.name as business_location',
                    'transactions.final_total',
                    DB::raw('SUM(IF(tp.is_return = 0, tp.amount, 0)) as total_paid'),

                    'sell_exchange.reason',
                    'sell_exchange.amount_before_exchange',
                    'sell_exchange.amount_after_exchange',
                     'sell_exchange.created_at as sell_exchange_created_at',
                    'sell_exchange.updated_at as sell_exchange_updated_at',
                    'sep.exchange_product_id as exchange_product',
                    'sep.exchange_product_variation_id as exchange_product_variation',
                    'sep.new_product_id as new_product',
                    'sep.new_product_variation_id as new_product_variation',
                    'sep.exchange_product_quantity as exchange_product_quantity',
                    'sep.new_product_quantity as new_product_quantity',

                )
                ->groupBy(
                    'sell_exchange.id',
                    'transactions.transaction_date',
                    'transactions.invoice_no',
                    'contacts.name',
                    'contacts.supplier_business_name',
                    'bl.name',
                    'transactions.final_total',

                    'sell_exchange.reason',
                    'sell_exchange.amount_before_exchange',
                    'sell_exchange.amount_after_exchange',
                    'sep.exchange_product_id',
                    'sep.exchange_product_variation_id',
                      'sell_exchange.created_at',
                    'sell_exchange.updated_at',
                    'sep.new_product_id',
                    'sep.new_product_variation_id',
                    'sep.exchange_product_quantity',
                    'sep.new_product_quantity'
                );

            //   dd($sell_exchanges->get());

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $sell_exchanges->whereIn('transactions.location_id', $permitted_locations);
            }

            if (! auth()->user()->can('access_sell_exchange') && auth()->user()->can('access_own_sell_exchange')) {
                $sell_exchanges->where('transactions.created_by', request()->session()->get('user.id'));
            }

            // Filter by created_by (sales representative)
            if (request()->has('created_by')) {
                $created_by = request()->get('created_by');
                if (! empty($created_by)) {
                    $sell_exchanges->where('transactions.created_by', $created_by);
                }
            }

            // Filter by location
            if (request()->has('location_id')) {
                $location_id = request()->get('location_id');
                if (! empty($location_id)) {
                    $sell_exchanges->where('transactions.location_id', $location_id);
                }
            }

            // Filter by customer
            if (! empty(request()->customer_id)) {
                $customer_id = request()->customer_id;
                $sell_exchanges->where('contacts.id', $customer_id);
            }

            // Date filter
            if (! empty(request()->start_date) && ! empty(request()->end_date)) {
                $start = request()->start_date;
                $end   = request()->end_date;
                $sell_exchanges->whereDate('sell_exchange.created_at', '>=', $start)
                    ->whereDate('sell_exchange.created_at', '<=', $end);
            }

            return DataTables::of($sell_exchanges)

                ->addColumn('action', function ($row) {
                    return '<a href="' . action('SellExchangeController@show', [$row->id]) . '" class="btn btn-info btn-xs view_sell_exchange" data-orig-value="' . $row->id . '">
                <i class="fa fa-eye"></i> View
            </a>';
                })

                ->editColumn(
                    'amount_before_exchange',
                    '<span class="display_currency" data-currency_symbol="true" data-orig-value="{{ $amount_before_exchange }}">{{ $amount_before_exchange }}</span>'
                )
                ->editColumn(
                    'amount_after_exchange',
                    '<span class="display_currency" data-currency_symbol="true" data-orig-value="{{$amount_after_exchange}}">{{$amount_after_exchange}}</span>'
                )
                ->editColumn('transaction_date', '{{@format_date($sell_exchange_created_at)}}')


                ->addColumn('payment_due', function ($row) {
                    $due = $row->final_total - $row->total_paid;
                    return '<span class="display_currency payment_due" data-currency_symbol="true" data-orig-value="' . $due . '">' . $due . '</span>';
                })
                ->setRowAttr([
                    'data-href' => function ($row) {
                        if (auth()->user()->can('sell_exchange.view')) {
                            // return action('SellExchangeController@show', [$row->id]);
                        } else {
                            return '';
                        }
                    },
                ])

                ->rawColumns(['amount_before_exchange', 'amount_after_exchange', 'payment_due', 'action'])
                ->make(true);
        }

        $business_locations   = BusinessLocation::forDropdown($business_id, false);
        $customers            = Contact::customersDropdown($business_id, false);
        $sales_representative = User::forDropdown($business_id, false, false, true);

        return view('sell_exchange.index')->with(compact('business_locations', 'customers', 'sales_representative'));
    }

    public function createExchange($id)
    {

        $business_id = request()->session()->get('user.business_id');
        $taxes       = TaxRate::where('business_id', $business_id)
            ->pluck('name', 'id');
        $query = Transaction::where('business_id', $business_id)
            ->where('id', $id)
            ->with(['contact', 'delivery_person_user', 'sell_lines' => function ($q) {
                $q->whereNull('parent_sell_line_id');
            }, 'sell_lines.product', 'sell_lines.product.unit', 'sell_lines.product.second_unit', 'sell_lines.variations', 'sell_lines.variations.product_variation', 'payment_lines', 'sell_lines.modifiers', 'sell_lines.lot_details', 'tax', 'sell_lines.sub_unit', 'table', 'service_staff', 'sell_lines.service_staff', 'types_of_service', 'sell_lines.warranties', 'media']);

        if (! auth()->user()->can('sell.view') && ! auth()->user()->can('direct_sell.access') && auth()->user()->can('view_own_sell_only')) {
            $query->where('transactions.created_by', request()->session()->get('user.id'));
        }

        $sell = $query->firstOrFail();

        $activities = Activity::forSubject($sell)
            ->with(['causer', 'subject'])
            ->latest()
            ->get();

        $line_taxes = [];
        foreach ($sell->sell_lines as $key => $value) {
            if (! empty($value->sub_unit_id)) {
                $formated_sell_line     = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);
                $sell->sell_lines[$key] = $formated_sell_line;
            }

            if (! empty($taxes[$value->tax_id])) {
                if (isset($line_taxes[$taxes[$value->tax_id]])) {
                    $line_taxes[$taxes[$value->tax_id]] += ($value->item_tax * $value->quantity);
                } else {
                    $line_taxes[$taxes[$value->tax_id]] = ($value->item_tax * $value->quantity);
                }
            }
        }

        $payment_types = $this->transactionUtil->payment_types($sell->location_id, true);
        $order_taxes   = [];
        if (! empty($sell->tax)) {
            if ($sell->tax->is_tax_group) {
                $order_taxes = $this->transactionUtil->sumGroupTaxDetails($this->transactionUtil->groupTaxDetails($sell->tax, $sell->tax_amount));
            } else {
                $order_taxes[$sell->tax->name] = $sell->tax_amount;
            }
        }

        $business_details       = $this->businessUtil->getDetails($business_id);
        $pos_settings           = empty($business_details->pos_settings) ? $this->businessUtil->defaultPosSettings() : json_decode($business_details->pos_settings, true);
        $shipping_statuses      = $this->transactionUtil->shipping_statuses();
        $shipping_status_colors = $this->shipping_status_colors;
        $common_settings        = session()->get('business.common_settings');
        $is_warranty_enabled    = ! empty($common_settings['enable_product_warranty']) ? true : false;

        $statuses = Transaction::sell_statuses();
        if ($sell->type == 'sales_order') {
            $sales_order_statuses = Transaction::sales_order_statuses(true);
            $statuses             = array_merge($statuses, $sales_order_statuses);
        }

        $status_color_in_activity = Transaction::sales_order_statuses();
        $sales_orders             = $sell->salesOrders();
        $default_location         = null;
        $business_locations       = BusinessLocation::forDropdown($business_id, false, true);
        $bl_attributes = $business_locations['attributes'];
        $business_locations       = $business_locations['locations'];

        foreach ($business_locations as $id => $name) {
            $default_location = BusinessLocation::findOrFail($id);
            break;
        }
        if (! empty($default_location) && ! empty($default_location->sale_invoice_scheme_id)) {
            $default_invoice_schemes = InvoiceScheme::where('business_id', $business_id)
                ->findorfail($default_location->sale_invoice_scheme_id);
        }

        $taxesDropdown = TaxRate::forBusinessDropdown($business_id, true, true);
        //Accounts
        $accounts = [];
        if ($this->moduleUtil->isModuleEnabled('account')) {
            $accounts = Account::forDropdown($business_id, true, false);
        }

        $change_return = $this->dummyPaymentLine;

        return view('sell_exchange.create_exchange')
            ->with(compact(
                'taxes',
                'taxesDropdown',
                'sell',
                'business_details',
                'default_location',
                'business_locations',
                'bl_attributes',
                'common_settings',
                'payment_types',
                'order_taxes',
                'pos_settings',
                'shipping_statuses',
                'shipping_status_colors',
                'is_warranty_enabled',
                'activities',
                'statuses',
                'status_color_in_activity',
                'sales_orders',
                'line_taxes',
                'change_return',
                'accounts'
            ));
    }

    public function store(Request $request)
    {
        // dd($request->all());
        DB::beginTransaction();

        try {
            $exchangeProductIds          = [];
            $exchangeProductVaraitionIds = [];
            $newProductIds               = [];
            $newProductVaraitionIds      = [];
            $exchangeProductUnitPrice    = [];
            $exchangeProductTotalPrice   = [];
            $exchangeProductQuantity     = [];
            $newProductQuantity          = [];
            $newProductUnitPrice         = [];
            $newProductTotalPrice        = [];
            $amountBeforeExchange        = 0;
            $amountAfterExchange         = 0;
            if (! empty($request->line_id)) {
                $transaction        = Transaction::where('id', $request->transaction_id)->first();
                $cashRegisterTransaction = CashRegisterTransaction::where('transaction_id', $request->transaction_id)->first();
                $originalFinalTotal = $transaction->final_total;
                $originalFinalTotalBeforeTax = $transaction->total_before_tax;
                $originalFinalAmountDiscount = $transaction->discount_amount;
                $originalDiscountType = $transaction->discount_type;
                $exchangeLineTotal  = 0;
                foreach ($request->line_id as $key => $value) {
                    $line = TransactionSellLine::where('id', $value)->first();
                    $exchangeLineTotal = $line->unit_price_inc_tax * $line->quantity;

                    if (! empty($line)) {
                        $exchangeProductIds[]          = $line->product_id;
                        $exchangeProductVaraitionIds[] = $line->variation_id;
                        $exchangeProductQuantity[]     = $line->quantity;
                        $exchangeProductUnitPrice[]    = $line->unit_price_inc_tax;
                        $exchangeProductTotalPrice[]   = $line->unit_price_inc_tax * $line->quantity;

                        $enable_stock = Product::find($request->product_id[$value])->enable_stock;
                        if ($enable_stock == 1) {
                            VariationLocationDetails::where('variation_id', $request->product_variation_id[$value])
                                ->where('product_id', $request->product_id[$value])
                                ->where('location_id', $request->location_id)
                                ->increment('qty_available', $request->quantity[$value]);
                        }
                    }


                    $allocations = TransactionSellLinesPurchaseLines::where('sell_line_id', $value)->get();
                    foreach ($allocations as $allocation) {
                        // dd($allocations, $allocation, $allocation->purchase_line_id);
                        // Increment the available quantity in purchase_lines
                        PurchaseLine::where('id', $allocation->purchase_line_id)
                            ->decrement('quantity_sold', $allocation->quantity);
                    }

                    SerialNumber::where('transaction_id', $request->transaction_id)
                        ->where('product_id', $request->product_id[$value])
                        ->where('variation_id', $request->product_variation_id[$value])
                        ->where('location_id', $request->location_id)
                        ->update([
                            'stock_status'   => 'available',
                            'transaction_id' => null,
                        ]);

                    TransactionSellLinesPurchaseLines::where('sell_line_id', $value)->delete();
                    TransactionSellLine::where('id', $value)->delete();
                }
            }

            $discount = [
                'discount_type'   => $request->discount_type,
                'discount_amount' => $request->discount_amount,
            ];

            $invoice_total = $this->productUtil->calculateInvoiceTotal($request->products, $request->tax_rate_id, $discount);
            // $amountBeforeExchange = $originalFinalTotal - $exchangeLineTotal;
            $amountBeforeExchange = ($originalFinalTotalBeforeTax - $exchangeLineTotal) == 0
                ? 0
                : ($originalFinalTotalBeforeTax - $exchangeLineTotal);

            $amountAfterExchangeWithoutDiscount = $invoice_total['total_before_tax'] + $amountBeforeExchange;
            $amountAfterExchangeWithDiscount = $originalFinalAmountDiscount > 0
                ? $amountAfterExchangeWithoutDiscount - ($originalDiscountType == 'percentage'
                    ? ($amountAfterExchangeWithoutDiscount * ($originalFinalAmountDiscount / 100))
                    : $originalFinalAmountDiscount)
                : $amountAfterExchangeWithoutDiscount;

            // dd($amountBeforeExchange,  $exchangeLineTotal, $invoice_total, $amountAfterExchangeWithoutDiscount, $amountAfterExchangeWithDiscount);



            // dd($request->all(), $amountBeforeExchange, $amountAfterExchangeWithDiscount,$amountAfterExchangeWithoutDiscount, $ammountDiff, $originalFinalTotal, $exchangeLineTotal, $invoice_total);

            //create sell line

            $this->addSellLine($transaction, $request->products, $request->location_id, false, null, [], false);

            foreach ($request->products as $key => $product) {
                // dd($product);
                $this->productUtil->decreaseProductQuantity(
                    $product['product_id'],
                    $product['variation_id'],
                    $request->location_id,
                    $product['quantity'],
                );

                $newProductIds[]          = $product['product_id'];
                $newProductVaraitionIds[] = $product['variation_id'];
                $newProductQuantity[]     = $product['quantity'];
                $newProductUnitPrice[]    = $product['unit_price_inc_tax'];
                $newProductTotalPrice[]   = $product['unit_price_inc_tax'] * $product['quantity'];
            }

            $transaction->discount_type    = $originalDiscountType;
            $transaction->discount_amount  = $originalFinalAmountDiscount;
            $transaction->total_before_tax = $invoice_total['total_before_tax'];
            $transaction->tax_id           = $request->tax_rate_id;
            $transaction->tax_amount       = $invoice_total['tax'];
            $transaction->total_before_tax = $this->transactionUtil->num_uf($amountAfterExchangeWithoutDiscount);
            $transaction->final_total      = $this->transactionUtil->num_uf($amountAfterExchangeWithDiscount);
            $transaction->save();

            //create exhchange
            $sellExchange = SellExchange::create([
                'business_id'            => $transaction->business_id,
                'location_id'            => $transaction->location_id,
                'transaction_id'         => $transaction->id,
                'status'                 => 'created',
                'reason'                 => $request->reason ?? null,
            'amount_before_exchange' => $this->transactionUtil->num_uf($request->total_current_paid) ?? $amountBeforeExchange,

                'amount_after_exchange'  => $amountAfterExchangeWithDiscount,
            ]);

            // dd($exchangeProductUnitPrice, $exchangeProductTotalPrice, $newProductUnitPrice, $newProductTotalPrice);

            $sellExchange->sellExchangeProduct()->create([
                'exchange_product_id'           => json_encode($exchangeProductIds),
                'exchange_product_variation_id' => json_encode($exchangeProductVaraitionIds),
                'exchange_product_quantity'     => json_encode($exchangeProductQuantity),
                'exchange_product_unit_price'   => json_encode($exchangeProductUnitPrice),
                'exchange_product_total_price'  => json_encode($exchangeProductTotalPrice),
                'new_product_id'                => json_encode($newProductIds),
                'new_product_variation_id'      => json_encode($newProductVaraitionIds),
                'new_product_quantity'          => json_encode($newProductQuantity),
                'new_product_unit_price'        => json_encode($newProductUnitPrice),
                'new_product_total_price'       => json_encode($newProductTotalPrice),

            ]);
            //create exhchange end

            foreach ($request->payment_line_id as $key => $value) {
                $payment_line = TransactionPayment::where('id', $value)->latest()->first();

                if (! empty($payment_line) && $key == 0) {
                    $original_amount = $request->total_current_paid ?? 0;
                    $new_amount      = $this->transactionUtil->num_uf($request->final_total) ?? 0;
                    $difference      = $new_amount - $original_amount;
                    // dd($original_amount, $new_amount, $difference);

                    if ($difference != 0) {
                        foreach ($request->payment as $payment) {
                            $prefix_type    = 'sell_payment';
                            $ref_count      = $this->transactionUtil->setAndGetReferenceCount($prefix_type, $transaction->business_id);
                            $payment_ref_no = $this->transactionUtil->generateReferenceNumber($prefix_type, $ref_count, $transaction->business_id);
                            $payment_data   = [
                                'amount'                  => $payment['amount'],
                                'transaction_id'          => $transaction->id,
                                'method'                  => $payment['method'],
                                'business_id'             => $transaction->business_id,
                                'is_return'               => 0,
                                'card_transaction_number' => $payment['card_transaction_number'] ?? null,
                                'card_number'             => $payment['card_number'] ?? null,
                                'card_type'               => $payment['card_type'] ?? null,
                                'card_holder_name'        => $payment['card_holder_name'] ?? null,
                                'card_month'              => $payment['card_month'] ?? null,
                                'card_security'           => $payment['card_security'] ?? null,
                                'cheque_number'           => $payment['cheque_number'] ?? null,
                                'bank_account_number'     => $payment['bank_account_number'] ?? null,
                                'note'                    => $payment['note'] ?? null,
                                'paid_on'                 => now(),
                                'created_by'              => auth()->id(),
                                'payment_for'             => $transaction->contact_id,
                                'payment_ref_no'          => $payment_ref_no,
                                'account_id'              => ! empty($payment['account_id']) && $payment['method'] != 'advance' ? $payment['account_id'] : null,
                            ];

                            for ($i = 1; $i < 8; $i++) {
                                if ($payment['method'] == 'custom_pay_' . $i) {
                                    $payment_data['transaction_no'] = $payment["transaction_no_{$i}"];
                                }
                            }

                            $transaction_payment = TransactionPayment::create($payment_data);
                            CashRegisterTransaction::create([
                                'cash_register_id'  => $cashRegisterTransaction->cash_register_id,
                                'amount'            => $payment['amount'],
                                'pay_method'        => $payment['method'],
                                'type'              => 'credit',
                                'transaction_type'  => 'sell',
                                'transaction_id'    => $transaction->id,
                            ]);

                            if ($payment['account_id']) {
                                $transaction_data = [
                                    'amount'                  => $payment['amount'],
                                    'account_id'              => $payment['account_id'],
                                    'type'                    => 'credit',
                                    'sub_type'                => 'sell_exchange',
                                    'operation_date'          => \Carbon::now(),
                                    'created_by'              => auth()->id(),
                                    'transaction_id'          => $transaction->id,
                                    'transaction_payment_id'  => $transaction_payment->id,
                                    'note'                    => ! empty($payment['note']) ? $payment['note'] : null,
                                    'transfer_transaction_id' =>  null,
                                ];

                                AccountTransaction::create($transaction_data);
                            }
                        }
                    }
                    break;
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'msg'     => __('lang_v1.success'),
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            // \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());
            // dd($e->getMessage());
            throw $e;

            return response()->json([
                'success' => false,
                'msg'     => __('lang_v1.something_went_wrong', $e->getMessage()),
                'error'   => $e->getMessage(), // Optional: remove in production
            ], 500);
        }
    }

    public function show($id)
    {
        $sell_exchange   = SellExchange::with('sellExchangeProduct', 'transaction')->find($id);
        $exchangeDetails = [];

        $sellExchangeProduct = $sell_exchange->sellExchangeProduct()->get();
        if (! $sellExchangeProduct->isEmpty()) {
            foreach ($sellExchangeProduct as $item) {
                $exchangeProductIds   = json_decode($item->exchange_product_id, true);
                $exchangeVariationIds = json_decode($item->exchange_product_variation_id, true);
                $exchangeQuantities   = json_decode($item->exchange_product_quantity, true);
                $exchangeUnitPrice    = json_decode($item->exchange_product_unit_price, true);
                $exchangeTotalPrice   = json_decode($item->exchange_product_total_price, true);

                $newProductIds   = json_decode($item->new_product_id, true);
                $newVariationIds = json_decode($item->new_product_variation_id, true);
                $newQuantities   = json_decode($item->new_product_quantity, true);
                $newUnitPrice    = json_decode($item->new_product_unit_price, true);
                $newTotalPrice   = json_decode($item->new_product_total_price, true);

                // Get exchanged product details
                $exchangeProducts = Product::with(['variations' => function ($q) {
                    $q->select('id', 'name', 'product_id');
                }])->whereIn('id', $exchangeProductIds)->get();

                // Get new product details
                $newProducts = Product::with(['variations' => function ($q) {
                    $q->select('id', 'name', 'product_id');
                }])->whereIn('id', $newProductIds)->get();

                $exchangeDetails[] = [
                    'exchange_products'      => $exchangeProducts,
                    'exchange_variation_ids' => $exchangeVariationIds,
                    'exchange_quantities'    => $exchangeQuantities,
                    'new_products'           => $newProducts,
                    'new_variation_ids'      => $newVariationIds,
                    'new_quantities'         => $newQuantities,
                    'exchange_unit_price'    => $exchangeUnitPrice,
                    'exchange_total_price'   => $exchangeTotalPrice,
                    'new_unit_price'         => $newUnitPrice,
                    'new_total_price'        => $newTotalPrice,
                ];
            }
        }
        // dd($sell_exchange, $sellExchangeProduct, $exchangeDetails);

        return view('sell_exchange.show')->with(compact('sell_exchange', 'exchangeDetails'));
    }

    //function name sell line
    public function addSellLine($transaction, $products, $location_id, $return_deleted = false, $status_before = null, $extra_line_parameters = [], $uf_data = true)
    {
        $lines_formatted         = [];
        $modifiers_array         = [];
        $combo_lines             = [];
        $products_modified_combo = [];

        foreach ($products as $product) {
            $multiplier                   = 1;
            $common_settings              = session()->get('business.common_settings');
            $is_enable_product_multi_unit = ! empty($common_settings['enable_product_multi_unit']);

            if ($is_enable_product_multi_unit) {
                if (isset($product['is_enable_product_multi_unit']) && $product['is_enable_product_multi_unit'] == "1") {
                    if (isset($product['sub_unit_id']) && $product['sub_unit_id'] != $product['product_unit_id']) {
                        if ($product['sub_unit_id'] == $product['first_conversion_unit_id']) {
                            $multiplier = $this->transactionUtil->num_uf($product['first_conversion_unit_rate']);
                        }
                        if (! empty($product['second_conversion_unit_id'])) {
                            if ($product['sub_unit_id'] == $product['second_conversion_unit_id']) {
                                $multiplier = ($this->transactionUtil->num_uf($product['first_conversion_unit_rate']) * $this->transactionUtil->num_uf($product['second_conversion_unit_rate']));
                            }
                        }
                        $product['sell_unit_id'] = $product['sub_unit_id'];
                        $product['sub_unit_id']  = null;
                    }
                }
            } else {
                if (isset($product['sub_unit_id']) && $product['sub_unit_id'] == $product['product_unit_id']) {
                    unset($product['sub_unit_id']);
                }
                if (! empty($product['sub_unit_id']) && ! empty($product['base_unit_multiplier'])) {
                    $multiplier = $product['base_unit_multiplier'];
                }
            }

            // Only create new lines (skip edit logic)
            if (empty($product['transaction_sell_lines_id'])) {
                $products_modified_combo[] = $product;

                //calculate unit price and unit price before discount
                $uf_unit_price              = $uf_data ? $this->transactionUtil->num_uf($product['unit_price']) : $product['unit_price'];
                $unit_price_before_discount = $uf_unit_price / $multiplier;
                $unit_price                 = $unit_price_before_discount;

                if (! empty($product['line_discount_type']) && $product['line_discount_amount']) {
                    $discount_amount = $uf_data ? $this->transactionUtil->num_uf($product['line_discount_amount']) : $product['line_discount_amount'];
                    if ($product['line_discount_type'] == 'fixed') {
                        $unit_price = $unit_price_before_discount - ($discount_amount / $multiplier);
                    } elseif ($product['line_discount_type'] == 'percentage') {
                        $unit_price = ((100 - $discount_amount) * $unit_price_before_discount) / 100;
                    }
                }

                $uf_quantity           = $uf_data ? $this->transactionUtil->num_uf($product['quantity']) : $product['quantity'];
                $uf_item_tax           = $uf_data ? $this->transactionUtil->num_uf($product['item_tax']) : $product['item_tax'];
                $uf_unit_price_inc_tax = $uf_data ? $this->transactionUtil->num_uf($product['unit_price_inc_tax']) : $product['unit_price_inc_tax'];

                $line_discount_amount = 0;
                if (! empty($product['line_discount_amount'])) {
                    $line_discount_amount = $uf_data ? $this->transactionUtil->num_uf($product['line_discount_amount']) : $product['line_discount_amount'];
                    if ($product['line_discount_type'] == 'fixed') {
                        $line_discount_amount = $line_discount_amount / $multiplier;
                    }
                }

                $line = [
                    'product_id'                 => $product['product_id'],
                    'variation_id'               => $product['variation_id'],
                    'quantity'                   => $uf_quantity * $multiplier,
                    'unit_price_before_discount' => $unit_price_before_discount,
                    'unit_price'                 => $unit_price,
                    'line_discount_type'         => ! empty($product['line_discount_type']) ? $product['line_discount_type'] : null,
                    'line_discount_amount'       => $line_discount_amount,
                    'item_tax'                   => $uf_item_tax / $multiplier,
                    'tax_id'                     => $product['tax_id'],
                    'unit_price_inc_tax'         => $uf_unit_price_inc_tax / $multiplier,
                    'sell_line_note'             => ! empty($product['sell_line_note']) ? $product['sell_line_note'] : '',
                    'sub_unit_id'                => ! empty($product['sub_unit_id']) ? $product['sub_unit_id'] : null,
                    'sell_unit_id'               => ! empty($product['sell_unit_id']) ? $product['sell_unit_id'] : null,
                    'discount_id'                => ! empty($product['discount_id']) ? $product['discount_id'] : null,
                    'res_service_staff_id'       => ! empty($product['res_service_staff_id']) ? $product['res_service_staff_id'] : null,
                    'res_line_order_status'      => ! empty($product['res_service_staff_id']) ? 'received' : null,
                    'so_line_id'                 => ! empty($product['so_line_id']) ? $product['so_line_id'] : null,
                    'secondary_unit_quantity'    => ! empty($product['secondary_unit_quantity']) ? $this->transactionUtil->num_uf($product['secondary_unit_quantity']) : 0,
                ];

                foreach ($extra_line_parameters as $key => $value) {
                    $line[$key] = isset($product[$value]) ? $product[$value] : '';
                }

                if (! empty($product['lot_no_line_id'])) {
                    $line['lot_no_line_id'] = $product['lot_no_line_id'];
                }

                //Check if restaurant module is enabled then add more data related to that.
                if ($this->transactionUtil->isModuleEnabled('modifiers')) {
                    $sell_line_modifiers = [];

                    if (! empty($product['modifier'])) {
                        foreach ($product['modifier'] as $key => $value) {
                            if (! empty($product['modifier_price'][$key])) {
                                $this_price            = $uf_data ? $this->transactionUtil->num_uf($product['modifier_price'][$key]) : $product['modifier_price'][$key];
                                $modifier_quantity     = isset($product['modifier_quantity'][$key]) ? $product['modifier_quantity'][$key] : 1;
                                $sell_line_modifiers[] = [
                                    'product_id'                 => $product['modifier_set_id'][$key],
                                    'variation_id'               => $value,
                                    'quantity'                   => $modifier_quantity,
                                    'unit_price_before_discount' => $this_price,
                                    'unit_price'                 => $this_price,
                                    'unit_price_inc_tax'         => $this_price,
                                    'children_type'              => 'modifier',
                                ];
                            }
                        }
                    }
                    $modifiers_array[] = $sell_line_modifiers;
                }

                $lines_formatted[] = new TransactionSellLine($line);

                //Update purchase order line quantity received
                $this->updateSalesOrderLine($line['so_line_id'], $line['quantity'], 0);
            }
        }

        if (! is_object($transaction)) {
            $transaction = Transaction::findOrFail($transaction);
        }

        if (! empty($lines_formatted)) {
            $transaction->sell_lines()->saveMany($lines_formatted);

            //Add corresponding modifier sell lines if exists
            if ($this->transactionUtil->isModuleEnabled('modifiers')) {
                $modifiers_formatted = [];
                foreach ($lines_formatted as $key => $value) {
                    if (! empty($modifiers_array[$key])) {
                        foreach ($modifiers_array[$key] as $modifier) {
                            $modifier['parent_sell_line_id'] = $value->id;
                            $modifiers_formatted[]           = new TransactionSellLine($modifier);
                        }
                    }
                }

                if (! empty($modifiers_formatted)) {
                    $transaction->sell_lines()->saveMany($modifiers_formatted);
                }
            }

            //Combo product lines
            foreach ($lines_formatted as $key => $value) {
                if (! empty($products_modified_combo[$key]['product_type']) && $products_modified_combo[$key]['product_type'] == 'combo') {
                    $combo_lines = array_merge($combo_lines, $this->__makeLinesForComboProduct($products_modified_combo[$key]['combo'], $value));
                }
                // dd($lines_formatted, $value, $value->product_id);
                $product_id   = $value->product_id;
                $variation_id = $value->variation_id;
                $quantity     = $value->quantity;

                $purchase_lines = PurchaseLine::join('transactions', 'purchase_lines.transaction_id', '=', 'transactions.id')
                    ->where('purchase_lines.product_id', $product_id)
                    ->where('purchase_lines.variation_id', $variation_id)
                    ->where('transactions.location_id', $location_id)
                    ->where('transactions.status', 'received')
                    ->where('purchase_lines.quantity', '>', 0)
                    ->orderBy('purchase_lines.created_at', 'asc')
                    ->select('purchase_lines.*')
                    ->get();

                $remaining_quantity = $quantity;

                foreach ($purchase_lines as $purchase_line) {
                    if ($remaining_quantity <= 0) {
                        break;
                    }

                    $allocated_quantity = min($remaining_quantity, $purchase_line->quantity);

                    // Create mapping record
                    DB::table('transaction_sell_lines_purchase_lines')->insert([
                        'sell_line_id'     => $value->id,
                        'purchase_line_id' => $purchase_line->id,
                        'quantity'         => $allocated_quantity,
                        'created_at'       => now(),
                        'updated_at'       => now(),
                    ]);

                    // Update purchase line's available quantity
                    $purchase_line->quantity_sold += $allocated_quantity;
                    $purchase_line->save();

                    $remaining_quantity -= $allocated_quantity;
                }
            }

            if (! empty($combo_lines)) {
                $transaction->sell_lines()->saveMany($combo_lines);
            }
        }

        return true;
    }

    private function updateSalesOrderLine($so_line_id, $new_qty, $old_qty = 0)
    {
        $diff = $new_qty - $old_qty;
        if (! empty($so_line_id) && ! empty($diff)) {
            $so_line = TransactionSellLine::find($so_line_id);
            $so_line->so_quantity_invoiced += ($diff);
            $so_line->save();
        }
    }
}
