<?php

namespace App\Http\Controllers;

use App\Brands;
use App\Business;
use App\BusinessLocation;
use App\Category;
use App\Exports\ProductsExport;
use App\Media;
use App\Product;
use App\ProductVariation;
use App\PurchaseLine;
use App\SellingPriceGroup;
use App\TaxRate;
use App\Unit;
use App\Utils\ModuleUtil;
use App\Utils\ProductUtil;
use App\Variation;
use App\VariationGroupPrice;
use App\VariationLocationDetails;
use App\VariationTemplate;
use App\Warranty;
use Spatie\Permission\Models\Role;
use App\User;
use App\Discount;
use App\SerialNumber;
use App\ServiceWarranty;
use App\PremadeLine;


use Excel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Facades\DataTables;
use App\Events\ProductsCreatedOrModified;
use App\TransactionSellLine;
use App\Utils\BusinessUtil;
use App\Utils\ContactUtil;
use App\Utils\TransactionUtil;




  
class ProductController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $productUtil;
    protected $businessUtil;
    protected $contactUtil;
    protected $transactionUtil;
    protected $moduleUtil;

    private $barcode_types;

    /**
     * Product type constants
     */
    const PRODUCT_TYPE_SINGLE = 'single';
    const PRODUCT_TYPE_VARIABLE = 'variable';
    const PRODUCT_TYPE_COMBO = 'combo';
    const PRODUCT_TYPE_MODIFIER = 'modifier';

    /**
     * Location update type constants
     */
    const LOCATION_UPDATE_ADD = 'add';
    const LOCATION_UPDATE_REMOVE = 'remove';

    /**
     * Constructor
     *
     * @param  ProductUtils  $product
     * @return void
     */
    public function __construct(ProductUtil $productUtil, ModuleUtil $moduleUtil,BusinessUtil $businessUtil,ContactUtil $contactUtil,TransactionUtil $transactionUtil)
    {
        $this->productUtil = $productUtil;
        $this->moduleUtil = $moduleUtil;
        $this->businessUtil = $businessUtil;
        $this->contactUtil = $contactUtil;
        $this->transactionUtil = $transactionUtil;
        //barcode types
        $this->barcode_types = $this->productUtil->barcode_types();
    }

    /**
     * Validate that user has permission to access the given locations
     *
     * @param array $location_ids
     * @return void
     * @throws \Exception
     */
    protected function validateLocationPermissions(array $location_ids): void
    {
        if (empty($location_ids)) {
            return;
        }

        $permitted_locations = auth()->user()->permitted_locations();
        
        if ($permitted_locations != 'all') {
            $invalid_locations = array_diff($location_ids, $permitted_locations);
            if (!empty($invalid_locations)) {
                throw new \Exception(__('lang_v1.unauthorized_location_access'));
            }
        }
    }


    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if (! auth()->user()->can('product.view') && ! auth()->user()->can('product.create')) {
            abort(403, 'Unauthorized action.');
        }
        $business_id = request()->session()->get('user.business_id');
        $selling_price_group_count = SellingPriceGroup::countSellingPriceGroups($business_id);
        $is_woocommerce = $this->moduleUtil->isModuleInstalled('Woocommerce');

        $price_groups = SellingPriceGroup::where('business_id', $business_id)->get();

        if (request()->ajax()) {
            //Filter by location
            $location_id = request()->get('location_id', null);
            $permitted_locations = auth()->user()->permitted_locations();

            $query = Product::with(['media'])
                            ->leftJoin('brands', 'products.brand_id', '=', 'brands.id')
                            ->join('units', 'products.unit_id', '=', 'units.id')

                            ->leftjoin('units as first_conversion_unit', 'products.first_conversion_unit_id', '=', 'first_conversion_unit.id')
                            ->leftjoin('units as second_conversion_unit', 'products.second_conversion_unit_id', '=', 'second_conversion_unit.id')

                            ->leftJoin('categories as c1', 'products.category_id', '=', 'c1.id')
                            ->leftJoin('categories as c2', 'products.sub_category_id', '=', 'c2.id')
                            ->leftJoin('tax_rates', 'products.tax', '=', 'tax_rates.id')
                            ->join('variations as v', 'v.product_id', '=', 'products.id')

                            // ->leftjoin('variation_group_prices', 'v.id', '=', 'variation_group_prices.variation_id')
                            // ->leftjoin('selling_price_groups', 'variation_group_prices.price_group_id', '=', 'selling_price_groups.id')

                            ->leftJoin('variation_location_details as vld', function ($join) use ($permitted_locations) {
                                $join->on('vld.variation_id', '=', 'v.id');
                                if ($permitted_locations != 'all') {
                                    $join->whereIn('vld.location_id', $permitted_locations);
                                }
                            })
                            ->whereNull('v.deleted_at')
                            ->where('products.business_id', $business_id)
                            ->where('products.type', '!=', self::PRODUCT_TYPE_MODIFIER);


            if (! empty($location_id) && $location_id != 'none') {
                if ($permitted_locations == 'all' || in_array($location_id, $permitted_locations)) {
                    $query->whereHas('product_locations', function ($query) use ($location_id) {
                        $query->where('product_locations.location_id', '=', $location_id);
                    });
                }
            } elseif ($location_id == 'none') {
                $query->doesntHave('product_locations');
            } else {
                if ($permitted_locations != 'all') {
                    $query->whereHas('product_locations', function ($query) use ($permitted_locations) {
                        $query->whereIn('product_locations.location_id', $permitted_locations);
                    });
                } else {
                    $query->with('product_locations');
                }
            }

            $products = $query->select(

                    'products.id',
                    'v.id as variation_id',

                    // 'variation_group_prices.price_inc_tax as group_price',
                    // 'variation_group_prices.price_group_id',


                    'products.name as product',
                    'products.type',
                    'c1.name as category',
                    'c2.name as sub_category',
                    'units.actual_name as unit',
                    'brands.name as brand',
                    'tax_rates.name as tax',
                    'products.sku',
                    'products.image',
                    'products.enable_stock',
                    'products.is_inactive',
                    'products.not_for_selling',

                    'products.use_multi_unit',
                    'products.first_conversion_unit_id',
                    'products.second_conversion_unit_id',
                    'products.first_conversion_unit_rate',
                    'products.second_conversion_unit_rate',
                    'first_conversion_unit.short_name as first_conversion_short_name',
                    'first_conversion_unit.actual_name as first_conversion_name',
                    'second_conversion_unit.actual_name as second_conversion_name',
                    'second_conversion_unit.short_name as second_conversion_short_name',

                    'products.product_custom_field1', 
                    'products.product_custom_field2', 
                    'products.product_custom_field3', 
                    'products.product_custom_field4', 
                    'products.product_custom_field5', 
                    'products.product_custom_field6',
                    'products.product_custom_field7', 
                    'products.product_custom_field8', 
                    'products.product_custom_field9',
                    'products.product_custom_field10', 
                    'products.product_custom_field11', 
                    'products.product_custom_field12',
                    'products.product_custom_field13', 
                    'products.product_custom_field14', 
                    'products.product_custom_field15',
                    'products.product_custom_field16', 
                    'products.product_custom_field17', 
                    'products.product_custom_field18', 
                    'products.product_custom_field19', 
                    'products.product_custom_field20',
                    'products.alert_quantity',

                    
                    DB::raw('MAX(v.sell_price_inc_tax) as max_price'),
                    DB::raw('MIN(v.sell_price_inc_tax) as min_price'),
                    DB::raw('MAX(v.dpp_inc_tax) as max_purchase_price'),
                    DB::raw('MIN(v.dpp_inc_tax) as min_purchase_price'),
                    
                    
                    'v.first_unit_sku',
                    DB::raw('MAX(v.fu_sell_price_inc_tax) as fu_max_price'),
                    DB::raw('MIN(v.fu_sell_price_inc_tax) as fu_min_price'),
                    DB::raw('MAX(v.fu_dpp_inc_tax) as fu_max_purchase_price'),
                    DB::raw('MIN(v.fu_dpp_inc_tax) as fu_min_purchase_price'),
                    
                    'v.second_unit_sku',
                    DB::raw('MAX(v.su_sell_price_inc_tax) as su_max_price'),
                    DB::raw('MIN(v.su_sell_price_inc_tax) as su_min_price'),
                    DB::raw('MAX(v.su_dpp_inc_tax) as su_max_purchase_price'),
                    DB::raw('MIN(v.su_dpp_inc_tax) as su_min_purchase_price'),
                    
                    DB::raw('vld.qty_available as current_stock')
                );

            //if woocomerce enabled add field to query
            if ($is_woocommerce) {
                $products->addSelect('woocommerce_disable_sync');
            }

            $products->groupBy('products.id');

            $type = request()->get('type', null);
            if (! empty($type)) {
                $products->where('products.type', $type);
            }

            $category_id = request()->get('category_id', null);
            if (! empty($category_id)) {
                $products->where('products.category_id', $category_id);
            }

            $brand_id = request()->get('brand_id', null);
            if (! empty($brand_id)) {
                $products->where('products.brand_id', $brand_id);
            }

            $unit_id = request()->get('unit_id', null);
            if (! empty($unit_id)) {
                $products->where('products.unit_id', $unit_id);
            }

            $tax_id = request()->get('tax_id', null);
            if (! empty($tax_id)) {
                $products->where('products.tax', $tax_id);
            }

            $active_state = request()->get('active_state', null);
            if ($active_state == 'active') {
                $products->Active();
            }
            if ($active_state == 'inactive') {
                $products->Inactive();
            }
            $not_for_selling = request()->get('not_for_selling', null);
            if ($not_for_selling == 'true') {
                $products->ProductNotForSales();
            }

            $woocommerce_enabled = request()->get('woocommerce_enabled', 0);
            if ($woocommerce_enabled == 1) {
                $products->where('products.woocommerce_disable_sync', 0);
            }

            if (! empty(request()->get('repair_model_id'))) {
                $products->where('products.repair_model_id', request()->get('repair_model_id'));
            }
		    $custom_labels = json_decode(session('business.custom_labels'), true);
            $databases =  Datatables::of($products)
                ->addColumn(
                    'product_locations',
                    function ($row) {
                        return $row->product_locations->implode('name', ', ');
                    }
                )
                ->editColumn('category', '{{$category}} @if(!empty($sub_category))<br/> -- {{$sub_category}}@endif')
                ->addColumn(
                    'action',
                    function ($row) use ($selling_price_group_count) {
                        $html =
                        '<div class="btn-group"><button type="button" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline  tw-dw-btn-info tw-w-max dropdown-toggle" data-toggle="dropdown" aria-expanded="false">'.__('messages.actions').'<span class="caret"></span><span class="sr-only">Toggle Dropdown</span></button><ul class="dropdown-menu dropdown-menu-left" role="menu"><li><a href="'.action([\App\Http\Controllers\LabelsController::class, 'show']).'?product_id='.$row->id.'" data-toggle="tooltip" title="'.__('lang_v1.label_help').'"><i class="fa fa-barcode"></i> '.__('barcode.labels').'</a></li>';

                        if (auth()->user()->can('product.view')) {
                            $html .=
                            '<li><a href="'.action([\App\Http\Controllers\ProductController::class, 'view'], [$row->id]).'" class="view-product"><i class="fa fa-eye"></i> '.__('messages.view').'</a></li>';
                        }

                        if (auth()->user()->can('product.update')) {
                            $html .=
                            '<li><a href="'.action([\App\Http\Controllers\ProductController::class, 'edit'], [$row->id]).'"><i class="glyphicon glyphicon-edit"></i> '.__('messages.edit').'</a></li>';
                        }

                        if (auth()->user()->can('product.delete')) {
                            $html .=
                            '<li><a href="'.action([\App\Http\Controllers\ProductController::class, 'destroy'], [$row->id]).'" class="delete-product"><i class="fa fa-trash"></i> '.__('messages.delete').'</a></li>';
                        }

                        if ($row->is_inactive == 1) {
                            $html .=
                            '<li><a href="'.action([\App\Http\Controllers\ProductController::class, 'activate'], [$row->id]).'" class="activate-product"><i class="fas fa-check-circle"></i> '.__('lang_v1.reactivate').'</a></li>';
                        }

                        $html .= '<li class="divider"></li>';

                        if ($row->enable_stock == 1 && auth()->user()->can('product.opening_stock')) {
                            $html .=
                            '<li><a href="#" data-href="'.action([\App\Http\Controllers\OpeningStockController::class, 'add'], ['product_id' => $row->id]).'" class="add-opening-stock"><i class="fa fa-database"></i> '.__('lang_v1.add_edit_opening_stock').'</a></li>';
                        }

                        if (auth()->user()->can('product.view')) {
                            $html .=
                            '<li><a href="'.action([\App\Http\Controllers\ProductController::class, 'productStockHistory'], [$row->id]).'"><i class="fas fa-history"></i> '.__('lang_v1.product_stock_history').'</a></li>';
                        }

                        if (auth()->user()->can('product.create')) {
                            if ($selling_price_group_count > 0) {
                                $html .=
                                '<li><a href="'.action([\App\Http\Controllers\ProductController::class, 'addSellingPrices'], [$row->id]).'"><i class="fas fa-money-bill-alt"></i> '.__('lang_v1.add_selling_price_group_prices').'</a></li>';
                            }

                            $html .=
                                '<li><a href="'.action([\App\Http\Controllers\ProductController::class, 'create'], ['d' => $row->id]).'"><i class="fa fa-copy"></i> '.__('lang_v1.duplicate_product').'</a></li>';
                        }

                        if (! empty($row->media->first())) {
                            $html .=
                                '<li><a href="'.$row->media->first()->display_url.'" download="'.$row->media->first()->display_name.'"><i class="fas fa-download"></i> '.__('lang_v1.product_brochure').'</a></li>';
                        }

                        $html .= '</ul></div>';

                        return $html;
                    }
                )
                ->editColumn('product', function ($row) use ($is_woocommerce) {
                    $product = $row->is_inactive == 1 ? $row->product.' <span class="label bg-gray">'.__('lang_v1.inactive').'</span>' : $row->product;

                    $product = $row->not_for_selling == 1 ? $product.' <span class="label bg-gray">'.__('lang_v1.not_for_selling').
                        '</span>' : $product;

                    if ($is_woocommerce && ! $row->woocommerce_disable_sync) {
                        $product = $product.'<br><i class="fab fa-wordpress"></i>';
                    }

                    return $product;
                })
                ->editColumn('image', function ($row) {
                    return '<div style="display: flex;"><img src="'.$row->image_url.'" alt="Product image" class="product-thumbnail-small"></div>';
                })
                ->editColumn('type', '@lang("lang_v1." . $type)')
                ->addColumn('mass_delete', function ($row) {
                    return  '<input type="checkbox" class="row-select" value="'.$row->id.'">';
                })
                ->editColumn('current_stock', function ($row) {
                     if ($row->enable_stock) {


                        $stock = $this->productUtil->num_f($row->current_stock, false, null, true);
                        $first_unit_stock = '--';
                        $second_unit_stock = '--';

                        if($row->use_multi_unit == 1){
                            if($row->first_conversion_unit_rate >0){
                                $first_unit_stock = $row->current_stock/$row->first_conversion_unit_rate;
                            }      
                            if($row->second_conversion_unit_rate >0){
                                $second_unit_stock = $row->current_stock/($row->first_conversion_unit_rate*$row->second_conversion_unit_rate);
                            }
                        }


                        return $stock.' '.$row->unit.'<br>'.$first_unit_stock.' '.$row->first_conversion_short_name.'<br>'.$second_unit_stock.' '.$row->second_conversion_short_name;
                    } else {
                        return '--';
                    }
                })
                ->addColumn(
                    'purchase_price',
                    '<div style="white-space: nowrap;">@format_currency($min_purchase_price) @if($max_purchase_price != $min_purchase_price && $type == "variable") -  @format_currency($max_purchase_price)@endif </div>'
                )
                ->addColumn(
                    'selling_price',
                    '<div style="white-space: nowrap;">@format_currency($min_price) @if($max_price != $min_price && $type == "variable") -  @format_currency($max_price)@endif </div>'
                )


                ->addColumn('product_custom_field1', function($row) use($custom_labels){
                    $product_cf_details = !empty($custom_labels['product_cf_details']) ? $custom_labels['product_cf_details'] : [];
	                $cf_type = !empty($product_cf_details[1]['type']) ? $product_cf_details[1]['type'] : 'text';
                    $dropdown = !empty($product_cf_details[1]['dropdown_options']) ? explode(PHP_EOL, $product_cf_details[1]['dropdown_options']) : [];   
					$value = $row->product_custom_field1 ?? null;

                    if(in_array($cf_type, ['text', 'date'])){
                        return $value;
                    }else{
                        return $dropdown[$row->product_custom_field1] ?? '';

                    }
                })
                ->addColumn('product_custom_field2', function($row) use($custom_labels){
                    $product_cf_details = !empty($custom_labels['product_cf_details']) ? $custom_labels['product_cf_details'] : [];
	                $cf_type = !empty($product_cf_details[2]['type']) ? $product_cf_details[2]['type'] : 'text';
                    $dropdown = !empty($product_cf_details[2]['dropdown_options']) ? explode(PHP_EOL, $product_cf_details[2]['dropdown_options']) : [];   
					$value = $row->product_custom_field2 ?? null;

                    if(in_array($cf_type, ['text', 'date'])){
                        return $value;
                    }else{
                        return $dropdown[$row->product_custom_field2] ?? '';

                    }
                })
                ->editColumn('sku', function ($row) {
                  
                    $first_unit_sku =   '--';
                    $second_unit_sku =  '--';
                    $sku = $row->sku;

                    if(!empty($row->first_unit_sku)){
                        $first_unit_sku = $row->first_unit_sku;
                    }
                    if(!empty($row->second_unit_sku)){
                        $second_unit_sku = $row->second_unit_sku;
                    }

                    return $sku.'<br>'.$first_unit_sku.'<br>'.$second_unit_sku;

                });

                $raw_columns = [
                    'action', 'image', 'sku',
                    'mass_delete', 'product', 'selling_price', 'purchase_price',
                    'category', 'current_stock', 
                ];


                foreach($price_groups as $price_group){
                    $column_name = 'group_price'.$price_group->id;
                    $raw_columns[] = $column_name;
                    $databases->editColumn($column_name, function ($row)  use ($price_group) {

                        $due_html = '';
                        $group_price = (float) 0;
                        if(!empty($price_group->id)){
                            $variation_group_price = VariationGroupPrice::where('variation_id', $row->variation_id)->where('price_group_id', $price_group->id)->first();
                            if(!empty($variation_group_price)){
                                $group_price = (float) $variation_group_price->price_inc_tax;
                            }
                        }
                        $due_html .= '<span class="group_price" data-orig-value="'.$group_price.'">'.$this->productUtil->num_f($group_price, true).'</span>';

                        return $due_html;

                    });
                }

               return $databases->setRowAttr([
                    'data-href' => function ($row) {
                        if (auth()->user()->can('product.view')) {
                            return  action([\App\Http\Controllers\ProductController::class, 'view'], [$row->id]);
                        } else {
                            return '';
                        }
                    }, 
                ])
                ->rawColumns($raw_columns)
                ->make(true);
        }

        $rack_enabled = (request()->session()->get('business.enable_racks') || request()->session()->get('business.enable_row') || request()->session()->get('business.enable_position'));

        $categories = Category::forDropdown($business_id, 'product');

        $brands = Brands::forDropdown($business_id);

        $units = Unit::forDropdown($business_id);

        $tax_dropdown = TaxRate::forBusinessDropdown($business_id, false);
        $taxes = $tax_dropdown['tax_rates'];

        $business_locations = BusinessLocation::forDropdown($business_id);
        $business_locations->prepend(__('lang_v1.none'), 'none');

        if ($this->moduleUtil->isModuleInstalled('Manufacturing') && (auth()->user()->can('superadmin') || $this->moduleUtil->hasThePermissionInSubscription($business_id, 'manufacturing_module'))) {
            $show_manufacturing_data = true;
        } else {
            $show_manufacturing_data = false;
        }

        //list product screen filter from module
        $pos_module_data = $this->moduleUtil->getModuleData('get_filters_for_list_product_screen');

        $is_admin = $this->productUtil->is_admin(auth()->user());
  

        return view('product.index')
            ->with(compact(

                'price_groups',

                'rack_enabled',
                'categories',
                'brands',
                'units',
                'taxes',
                'business_locations',
                'show_manufacturing_data',
                'pos_module_data',
                'is_woocommerce',
                'is_admin'
            ));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if (! auth()->user()->can('product.create')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        // Check if subscribed or not, then check for products quota
        if (! $this->moduleUtil->isSubscribed($business_id)) {
            return $this->moduleUtil->expiredResponse();
        } elseif (! $this->moduleUtil->isQuotaAvailable('products', $business_id)) {
            return $this->moduleUtil->quotaExpiredResponse('products', $business_id, action([\App\Http\Controllers\ProductController::class, 'index']));
        }

        $categories = Category::forDropdown($business_id, 'product');
        $brands = Brands::forDropdown($business_id);
        $units = Unit::forDropdown($business_id, true);

        $tax_dropdown = TaxRate::forBusinessDropdown($business_id, true, true);
        $taxes = $tax_dropdown['tax_rates'];
        $tax_attributes = $tax_dropdown['attributes'];

        $barcode_types = $this->barcode_types;
        $barcode_default = $this->productUtil->barcode_default();

        $default_profit_percent = request()->session()->get('business.default_profit_percent');

        //Get all business locations
        $business_locations = BusinessLocation::forDropdown($business_id);

        //Duplicate product
        $duplicate_product = null;
        $rack_details = null;

        $sub_categories = [];
        if (! empty(request()->input('d'))) {
            $duplicate_product = Product::where('business_id', $business_id)->find(request()->input('d'));
            $duplicate_product->name .= ' (copy)';

            if (! empty($duplicate_product->category_id)) {
                $sub_categories = Category::where('business_id', $business_id)
                        ->where('parent_id', $duplicate_product->category_id)
                        ->pluck('name', 'id')
                        ->toArray();
            }

            //Rack details
            if (! empty($duplicate_product->id)) {
                $rack_details = $this->productUtil->getRackDetails($business_id, $duplicate_product->id);
            }
        }

        $selling_price_group_count = SellingPriceGroup::countSellingPriceGroups($business_id);

        $module_form_parts = $this->moduleUtil->getModuleData('product_form_part');
        $product_types = $this->product_types();

        $common_settings = session()->get('business.common_settings');
        $warranties = Warranty::forDropdown($business_id);

        //product screen view from module
        $pos_module_data = $this->moduleUtil->getModuleData('get_product_screen_top_view');
        $use_multi_unit = 0;


        $discounts = Discount::forDropdown($business_id, true, false);

        $price_groups = SellingPriceGroup::forDropdown($business_id);

        return view('product.create')->with(
            compact(
                'categories', 
                'discounts', 

                'price_groups', 

                'brands', 
                'units','use_multi_unit', 'taxes', 
                'barcode_types', 'default_profit_percent', 
                'tax_attributes', 'barcode_default', 
                'business_locations', 'duplicate_product', 
                'sub_categories', 'rack_details', 
                'selling_price_group_count', 'module_form_parts', 
                'product_types', 'common_settings', 
                'warranties', 'pos_module_data'
            )
        );


    }

    private function product_types()
    {
        //Product types also includes modifier.
        return [self::PRODUCT_TYPE_SINGLE => __('lang_v1.single'),
            self::PRODUCT_TYPE_VARIABLE => __('lang_v1.variable'),
            self::PRODUCT_TYPE_COMBO => __('lang_v1.combo'),
        ];
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (! auth()->user()->can('product.create')) {
            abort(403, 'Unauthorized action.');
        }
        try {
            $business_id = $request->session()->get('user.business_id');
            $form_fields = [
                'name', 'brand_id', 'unit_id', 'category_id', 'tax', 'type', 
                'barcode_type', 'sku', 'alert_quantity', 'tax_type', 
                'weight', 
                'discount_id', 


                'pre_made',
                'use_multi_unit', 

                'product_description', 
                'sub_unit_ids', 
                'preparation_time_in_minutes', 
                'product_custom_field1', 
                'product_custom_field2', 
                'product_custom_field3', 
                'product_custom_field4', 
                'product_custom_field5', 
                'product_custom_field6', 
                'product_custom_field7', 
                'product_custom_field8', 
                'product_custom_field9', 
                'product_custom_field10', 
                'product_custom_field11', 
                'product_custom_field12', 
                'product_custom_field13', 
                'product_custom_field14', 
                'product_custom_field15', 
                'product_custom_field16', 
                'product_custom_field17', 
                'product_custom_field18', 
                'product_custom_field19', 
                'product_custom_field20',
            ];

            $module_form_fields = $this->moduleUtil->getModuleFormField('product_form_fields');
            if (! empty($module_form_fields)) {
                $form_fields = array_merge($form_fields, $module_form_fields);
            }

            $product_details = $request->only($form_fields);
            $product_details['business_id'] = $business_id;
            $product_details['created_by'] = $request->session()->get('user.id');

            $product_details['enable_stock'] = (! empty($request->input('enable_stock')) && $request->input('enable_stock') == 1) ? 1 : 0;
            $product_details['not_for_selling'] = (! empty($request->input('not_for_selling')) && $request->input('not_for_selling') == 1) ? 1 : 0;

            if (! empty($request->input('sub_category_id'))) {
                $product_details['sub_category_id'] = $request->input('sub_category_id');
            }

            if (! empty($request->input('secondary_unit_id'))) {
                $product_details['secondary_unit_id'] = $request->input('secondary_unit_id');
            }

            if (empty($product_details['sku'])) {
                $product_details['sku'] = ' ';
            }

            if (! empty($product_details['alert_quantity'])) {
                $product_details['alert_quantity'] = $this->productUtil->num_uf($product_details['alert_quantity']);
            }

            $expiry_enabled = $request->session()->get('business.enable_product_expiry');
            if (! empty($request->input('expiry_period_type')) && ! empty($request->input('expiry_period')) && ! empty($expiry_enabled) && ($product_details['enable_stock'] == 1)) {
                $product_details['expiry_period_type'] = $request->input('expiry_period_type');
                $product_details['expiry_period'] = $this->productUtil->num_uf($request->input('expiry_period'));
            }

            if (! empty($request->input('enable_sr_no')) && $request->input('enable_sr_no') == 1) {
                $product_details['enable_sr_no'] = 1;
            }

            if (! empty($request->input('enable_serial_number')) && $request->input('enable_serial_number') == 1) {
                $product_details['enable_serial_number'] = 1;
            }
            $common_settings = session()->get('business.common_settings');


            if(!empty($common_settings['enable_product_multi_unit'])){
                if (!empty($request->input('use_multi_unit')) && $request->input('use_multi_unit') == 1) {
                    $product_details['first_conversion_unit_id'] = $request->input('first_conversion_unit_id');
                    $product_details['first_conversion_unit_rate'] = $request->input('first_conversion_unit_rate');
                    $product_details['second_conversion_unit_id'] = $request->input('second_conversion_unit_id');
                    $product_details['second_conversion_unit_rate'] = $request->input('second_conversion_unit_rate');
                }
            }

     

            //upload document
            $product_details['image'] = $this->productUtil->uploadFile($request, 'image', config('constants.product_img_path'), 'image');

            $product_details['warranty_id'] = ! empty($request->input('warranty_id')) ? $request->input('warranty_id') : null;

            DB::beginTransaction();

            $product = Product::create($product_details);

            event(new ProductsCreatedOrModified($product_details, 'added'));

            if (empty(trim($request->input('sku')))) {
                $sku = $this->productUtil->generateProductSku($product->id);
                $product->sku = $sku;
                $product->save();
            }

            //Add product locations
            $product_locations = $request->input('product_locations');
            if (! empty($product_locations)) {
                // Validate location permissions
                $this->validateLocationPermissions($product_locations);
                $product->product_locations()->sync($product_locations);
            }

            if ($product->type == self::PRODUCT_TYPE_SINGLE) {
      
                $fu_single_dpp = $request->input('fu_single_dpp', null);
                $fu_single_dpp_inc_tax = $request->input('fu_single_dpp_inc_tax', null);
                $fu_profit_percent = $request->input('fu_profit_percent', null);
                $fu_single_dsp = $request->input('fu_single_dsp', null);
                $fu_single_dsp_inc_tax = $request->input('fu_single_dsp_inc_tax', null);
                $fu_sub_sku = $request->input('fu_sub_sku', null);
                $su_single_dpp = $request->input('su_single_dpp', null);
                $su_single_dpp_inc_tax = $request->input('su_single_dpp_inc_tax', null);
                $su_profit_percent = $request->input('su_profit_percent', null);
                $su_single_dsp = $request->input('su_single_dsp', null);
                $su_single_dsp_inc_tax = $request->input('su_single_dsp_inc_tax', null);
                $su_sub_sku = $request->input('su_sub_sku', null);
                


                $this->productUtil->createSingleProductVariation(
                    $product->id, 
                    $product->sku,
                    $request->input('single_dpp'), 
                    $request->input('single_dpp_inc_tax'), 
                    $request->input('profit_percent'), 
                    $request->input('single_dsp'), 
                    $request->input('single_dsp_inc_tax'),
                    [],
                    $fu_single_dpp,
                    $fu_single_dpp_inc_tax,
                    $fu_profit_percent,
                    $fu_single_dsp,
                    $fu_single_dsp_inc_tax,
                    $fu_sub_sku,
                    $su_single_dpp,
                    $su_single_dpp_inc_tax,
                    $su_profit_percent,
                    $su_single_dsp,
                    $su_single_dsp_inc_tax,
                    $su_sub_sku
                );

            } elseif ($product->type == self::PRODUCT_TYPE_VARIABLE) {
                if (!empty($request->input('product_variation'))) {

                    $input_variations = $request->input('product_variation');
                    
                    $this->productUtil->createVariableProductVariations(
                        $product->id, 
                        $input_variations, 
                        $request->input('sku_type')
                    );


                }
            } elseif ($product->type == self::PRODUCT_TYPE_COMBO) {

                //Create combo_variations array by combining variation_id and quantity.
                $combo_variations = [];
                if (! empty($request->input('composition_variation_id'))) {
                    $composition_variation_id = $request->input('composition_variation_id');
                    $quantity = $request->input('quantity');
                    $unit = $request->input('unit');

                    foreach ($composition_variation_id as $key => $value) {
                        $combo_variations[] = [
                            'variation_id' => $value,
                            'quantity' => $this->productUtil->num_uf($quantity[$key]),
                            'unit_id' => $unit[$key],
                        ];
                    }
                }


                $fu_single_dpp = $request->input('fu_single_dpp', null);
                $fu_single_dpp_inc_tax = $request->input('fu_single_dpp_inc_tax', null);
                $fu_profit_percent = $request->input('fu_profit_percent', null);
                $fu_single_dsp = $request->input('fu_single_dsp', null);
                $fu_single_dsp_inc_tax = $request->input('fu_single_dsp_inc_tax', null);
                $fu_sub_sku = $request->input('fu_sub_sku', null);
                $su_single_dpp = $request->input('su_single_dpp', null);
                $su_single_dpp_inc_tax = $request->input('su_single_dpp_inc_tax', null);
                $su_profit_percent = $request->input('su_profit_percent', null);
                $su_single_dsp = $request->input('su_single_dsp', null);
                $su_single_dsp_inc_tax = $request->input('su_single_dsp_inc_tax', null);
                $su_sub_sku = $request->input('su_sub_sku', null);
                


                $this->productUtil->createSingleProductVariation(
                    $product->id, 
                    $product->sku, 
                    $request->input('item_level_purchase_price_total'), 
                    $request->input('purchase_price_inc_tax'), 
                    $request->input('profit_percent'), 
                    $request->input('selling_price'), 
                    $request->input('selling_price_inc_tax'), 
                    $combo_variations,
                    $fu_single_dpp,
                    $fu_single_dpp_inc_tax,
                    $fu_profit_percent,
                    $fu_single_dsp,
                    $fu_single_dsp_inc_tax,
                    $fu_sub_sku,
                    $su_single_dpp,
                    $su_single_dpp_inc_tax,
                    $su_profit_percent,
                    $su_single_dsp,
                    $su_single_dsp_inc_tax,
                    $su_sub_sku
                );
            }


            //Add product racks details.
            $product_racks = $request->get('product_racks', null);
            if (! empty($product_racks)) {
                $this->productUtil->addRackDetails($business_id, $product->id, $product_racks);
            }

            //Set Module fields
            if (! empty($request->input('has_module_data'))) {
                $this->moduleUtil->getModuleData('after_product_saved', ['product' => $product, 'request' => $request]);
            }

            if(!empty($common_settings['enable_product_premade'])){
                if (!empty($request->input('pre_made')) && $request->input('pre_made') == 1) {
                    $premade_products = $request->get('products', []);
                    $this->productUtil->createOrUpdatePremade($business_id, $product->id, $premade_products);
                }
            }

            Media::uploadMedia($product->business_id, $product, $request, 'product_brochure', true);

            $discount_id = $request->input('discount_id');
            
            if(!empty($discount_id)){
                $discount = Discount::find($discount_id);
                // Retrieve the variation ids associated with the product
                $variation_ids = $product->variations->pluck('id')->toArray();
                // Use the variation ids in the sync method
                $discount->variations()->sync($variation_ids);
            }


            DB::commit();
            $output = [
                'success' => 1,
                'msg' => __('product.product_added_success'),
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];

            return redirect('products')->with('status', $output);
        }

        if ($request->input('submit_type') == 'submit_n_add_opening_stock') {
            return redirect()->action([\App\Http\Controllers\OpeningStockController::class, 'add'],
                ['product_id' => $product->id]
            );
        } elseif ($request->input('submit_type') == 'submit_n_add_selling_prices') {
            return redirect()->action([\App\Http\Controllers\ProductController::class, 'addSellingPrices'],
                [$product->id]
            );
        } elseif ($request->input('submit_type') == 'save_n_add_another') {
            return redirect()->action([\App\Http\Controllers\ProductController::class, 'create']
            )->with('status', $output);
        }

        return redirect('products')->with('status', $output);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        if (! auth()->user()->can('product.view')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $details = $this->productUtil->getRackDetails($business_id, $id, true);

        return view('product.show')->with(compact('details'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        if (! auth()->user()->can('product.update')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $brands = Brands::forDropdown($business_id);

        $tax_dropdown = TaxRate::forBusinessDropdown($business_id, true, true);
        $taxes = $tax_dropdown['tax_rates'];
        $tax_attributes = $tax_dropdown['attributes'];

        $barcode_types = $this->barcode_types;

        $product = Product::where('business_id', $business_id)
                            ->with(['product_locations','premadeline'])
                            ->where('id', $id)
                            ->firstOrFail();


        $premade_details = PremadeLine::join('products AS p', 'premade_lines.premade_product_id', '=', 'p.id')
                            ->join('products', 'premade_lines.product_id', '=', 'products.id')
                            ->join('variations AS variations', 'premade_lines.variation_id', '=', 'variations.id')
                            ->leftjoin('units', 'units.id', '=', 'p.unit_id')
                            ->leftjoin('units as u', 'p.secondary_unit_id', '=', 'u.id')
                            
                            ->leftjoin('units as use_unit', 'premade_lines.sub_unit_id', '=', 'use_unit.id')
                            ->where('premade_lines.premade_product_id', $id)
                
                            ->select(

                                'products.id as product_id',
                                'products.name as product_name',
                                
                                'products.pre_made',

                                'variations.default_sell_price',
                                'variations.dpp_inc_tax as default_purchase_price',
                                'variations.sell_price_inc_tax',
                                'variations.profit_percent',
                                'variations.id as variation_id',
                                'variations.combo_variations',  //Used in combo products

                                'use_unit.short_name as unit',
                                'use_unit.allow_decimal as unit_allow_decimal',
                                'use_unit.id as unit_id',
                                
                                'premade_lines.id',
                                'premade_lines.business_id',
                                'premade_lines.product_id',
                                'premade_lines.variation_id',
                                'premade_lines.quantity as quantity_ordered',
                                'premade_lines.premade_price',
                                'premade_lines.premade_price_inc_tax',
                                'premade_lines.premade_product_id',
                                'premade_lines.sub_unit_id',
                                'premade_lines.sell_unit_id',
                         
                            )
                            ->get();


        //Sub-category
        $sub_categories = [];
        $sub_categories = Category::where('business_id', $business_id)
                        ->where('parent_id', $product->category_id)
                        ->pluck('name', 'id')
                        ->toArray();
        $sub_categories = ['' => 'None'] + $sub_categories;

        $default_profit_percent = request()->session()->get('business.default_profit_percent');

        //Get units.
        $units = Unit::forDropdown($business_id, true);
        $sub_units = $this->productUtil->getSubUnits($business_id, $product->unit_id, true);

        //Get all business locations
        $business_locations = BusinessLocation::forDropdown($business_id);
        //Rack details
        $rack_details = $this->productUtil->getRackDetails($business_id, $id);

        $selling_price_group_count = SellingPriceGroup::countSellingPriceGroups($business_id);

        $module_form_parts = $this->moduleUtil->getModuleData('product_form_part');
        $product_types = $this->product_types();
        $common_settings = session()->get('business.common_settings');
        $warranties = Warranty::forDropdown($business_id);

        //product screen view from module
        $pos_module_data = $this->moduleUtil->getModuleData('get_product_screen_top_view');

        $alert_quantity = ! is_null($product->alert_quantity) ? $this->productUtil->num_f($product->alert_quantity, false, null, true) : null;

        $discounts = Discount::forDropdown($business_id, true, false);
        return view('product.edit')->with(
            compact(
                'categories', 
                'discounts', 
                'premade_details', 
                'brands', 'units', 
                'sub_units', 'taxes', 'tax_attributes', 
                'barcode_types', 'product', 'sub_categories', 
                'default_profit_percent', 'business_locations', 
                'rack_details', 'selling_price_group_count', 'module_form_parts', 
                'product_types', 'common_settings', 'warranties', 
                'pos_module_data', 'alert_quantity'
            )
        );


    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        if (! auth()->user()->can('product.update')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = $request->session()->get('user.business_id');
            $common_settings = session()->get('business.common_settings');
            $product_details = $request->only([
                'name', 'brand_id', 'unit_id', 'category_id', 'tax', 'barcode_type', 'sku', 'alert_quantity', 
                'tax_type', 'weight', 'product_description', 'sub_unit_ids', 
                
                'discount_id', 
                'use_multi_unit', 
                'pre_made', 

                'preparation_time_in_minutes', 'product_custom_field1', 
                'product_custom_field2', 'product_custom_field3', 'product_custom_field4', 'product_custom_field5', 
                'product_custom_field6', 'product_custom_field7', 'product_custom_field8', 'product_custom_field9', 'product_custom_field10', 
                'product_custom_field11', 'product_custom_field12', 'product_custom_field13', 'product_custom_field14', 
                'product_custom_field15', 'product_custom_field16', 'product_custom_field17', 
                'product_custom_field18', 'product_custom_field19', 'product_custom_field20',
            ]);

            DB::beginTransaction();

            $product = Product::where('business_id', $business_id)->where('id', $id)->with(['product_variations'])->first();

            if(!empty($common_settings['enable_product_multi_unit'])){
                if($request->input('use_multi_unit') == 1) {

                    $product->first_conversion_unit_id = $request->input('first_conversion_unit_id');
                    $product->first_conversion_unit_rate = $request->input('first_conversion_unit_rate');
                    $product->second_conversion_unit_id = $request->input('second_conversion_unit_id');
                    $product->second_conversion_unit_rate = $request->input('second_conversion_unit_rate');
                    $product->use_multi_unit = $product_details['use_multi_unit'] ?? '';
                    
                }else {
    
                    $product->first_conversion_unit_id = null;
                    $product->first_conversion_unit_rate = null;
                    $product->second_conversion_unit_id = null;
                    $product->second_conversion_unit_rate = null;
    
                }
            }

            


            $module_form_fields = $this->moduleUtil->getModuleFormField('product_form_fields');
            if (! empty($module_form_fields)) {
                foreach ($module_form_fields as $column) {
                    $product->$column = $request->input($column);
                }
            }


            $product->discount_id = $product_details['discount_id'] ?? '';

            $product->name = $product_details['name'];
            $product->brand_id = $product_details['brand_id'];
            $product->unit_id = $product_details['unit_id'];
            $product->category_id = $product_details['category_id'];
            $product->tax = $product_details['tax'];
            $product->barcode_type = $product_details['barcode_type'];
            $product->sku = $product_details['sku'];
            $product->alert_quantity = ! empty($product_details['alert_quantity']) ? $this->productUtil->num_uf($product_details['alert_quantity']) : $product_details['alert_quantity'];
            $product->tax_type = $product_details['tax_type'];
            $product->weight = $product_details['weight'];
            $product->product_custom_field1 = $product_details['product_custom_field1'] ?? '';
            $product->product_custom_field2 = $product_details['product_custom_field2'] ?? '';
            $product->product_custom_field3 = $product_details['product_custom_field3'] ?? '';
            $product->product_custom_field4 = $product_details['product_custom_field4'] ?? '';
            $product->product_custom_field5 = $product_details['product_custom_field5'] ?? '';
            $product->product_custom_field6 = $product_details['product_custom_field6'] ?? '';
            $product->product_custom_field7 = $product_details['product_custom_field7'] ?? '';
            $product->product_custom_field8 = $product_details['product_custom_field8'] ?? '';
            $product->product_custom_field9 = $product_details['product_custom_field9'] ?? '';
            $product->product_custom_field10 = $product_details['product_custom_field10'] ?? '';
            $product->product_custom_field11 = $product_details['product_custom_field11'] ?? '';
            $product->product_custom_field12 = $product_details['product_custom_field12'] ?? '';
            $product->product_custom_field13 = $product_details['product_custom_field13'] ?? '';
            $product->product_custom_field14 = $product_details['product_custom_field14'] ?? '';
            $product->product_custom_field15 = $product_details['product_custom_field15'] ?? '';
            $product->product_custom_field16 = $product_details['product_custom_field16'] ?? '';
            $product->product_custom_field17 = $product_details['product_custom_field17'] ?? '';
            $product->product_custom_field18 = $product_details['product_custom_field18'] ?? '';
            $product->product_custom_field19 = $product_details['product_custom_field19'] ?? '';
            $product->product_custom_field20 = $product_details['product_custom_field20'] ?? '';

            $product->product_description = $product_details['product_description'];
            $product->sub_unit_ids = ! empty($product_details['sub_unit_ids']) ? $product_details['sub_unit_ids'] : null;
            $product->preparation_time_in_minutes = $product_details['preparation_time_in_minutes'];
            $product->warranty_id = ! empty($request->input('warranty_id')) ? $request->input('warranty_id') : null;
            $product->secondary_unit_id = ! empty($request->input('secondary_unit_id')) ? $request->input('secondary_unit_id') : null;

            if (! empty($request->input('enable_stock')) && $request->input('enable_stock') == 1) {
                $product->enable_stock = 1;
            } else {
                $product->enable_stock = 0;
            }

            $product->not_for_selling = (! empty($request->input('not_for_selling')) && $request->input('not_for_selling') == 1) ? 1 : 0;

            if (! empty($request->input('sub_category_id'))) {
                $product->sub_category_id = $request->input('sub_category_id');
            } else {
                $product->sub_category_id = null;
            }

            $expiry_enabled = $request->session()->get('business.enable_product_expiry');
            if (! empty($expiry_enabled)) {
                if (! empty($request->input('expiry_period_type')) && ! empty($request->input('expiry_period')) && ($product->enable_stock == 1)) {
                    $product->expiry_period_type = $request->input('expiry_period_type');
                    $product->expiry_period = $this->productUtil->num_uf($request->input('expiry_period'));
                } else {
                    $product->expiry_period_type = null;
                    $product->expiry_period = null;
                }
            }

            if (! empty($request->input('enable_sr_no')) && $request->input('enable_sr_no') == 1) {
                $product->enable_sr_no = 1;
            } else {
                $product->enable_sr_no = 0;
            }



            // enable_product_premade

            if(!empty($common_settings['enable_product_premade'])){
                if (!empty($request->input('pre_made')) && $request->input('pre_made') == 1) {
                    $product->pre_made = 1;
                } else {
                    $product->pre_made = 0;
                }
            }


            if (! empty($request->input('enable_serial_number')) && $request->input('enable_serial_number') == 1) {
                $product->enable_serial_number = 1;
            } else {
                $product->enable_serial_number = 0;
            }




            //upload document
            $file_name = $this->productUtil->uploadFile($request, 'image', config('constants.product_img_path'), 'image');
            if (! empty($file_name)) {

                //If previous image found then remove
                if (! empty($product->image_path) && file_exists($product->image_path)) {
                    unlink($product->image_path);
                }

                $product->image = $file_name;
                //If product image is updated update woocommerce media id
                if (! empty($product->woocommerce_media_id)) {
                    $product->woocommerce_media_id = null;
                }
            }

            $product->save();
            $product->touch();

            event(new ProductsCreatedOrModified($product, 'updated'));

            //Add product locations
            $product_locations = ! empty($request->input('product_locations')) ?
                                $request->input('product_locations') : [];

            $permitted_locations = auth()->user()->permitted_locations();
            
            // Validate new location permissions
            if (!empty($product_locations)) {
                $this->validateLocationPermissions($product_locations);
            }
            
            //If not assigned location exists don't remove it
            if ($permitted_locations != 'all') {
                $existing_product_locations = $product->product_locations()->pluck('id');

                foreach ($existing_product_locations as $pl) {
                    if (! in_array($pl, $permitted_locations)) {
                        $product_locations[] = $pl;
                    }
                }
            }

            $product->product_locations()->sync($product_locations);

            if ($product->type == self::PRODUCT_TYPE_SINGLE) {
                $single_data = $request->only([
                    'single_variation_id', 
                    'single_dpp', 
                    'single_dpp_inc_tax', 
                    'single_dsp_inc_tax', 
                    'profit_percent', 
                    'single_dsp'
                ]);

                $variation = Variation::find($single_data['single_variation_id']);

                $variation->sub_sku = $product->sku;
                $variation->default_purchase_price = $this->productUtil->num_uf($single_data['single_dpp']);
                $variation->dpp_inc_tax = $this->productUtil->num_uf($single_data['single_dpp_inc_tax']);
                $variation->profit_percent = $this->productUtil->num_uf($single_data['profit_percent']);
                $variation->default_sell_price = $this->productUtil->num_uf($single_data['single_dsp']);
                $variation->sell_price_inc_tax = $this->productUtil->num_uf($single_data['single_dsp_inc_tax']);


                $fu_single_dpp = $request->has('fu_single_dpp') ? $request->input('fu_single_dpp') : 0;
                $fu_single_dpp_inc_tax = $request->has('fu_single_dpp_inc_tax') ? $request->input('fu_single_dpp_inc_tax') : 0;
                $fu_profit_percent = $request->has('fu_profit_percent') ? $request->input('fu_profit_percent') : 0;
                $fu_single_dsp = $request->has('fu_single_dsp') ? $request->input('fu_single_dsp') : 0;
                $fu_single_dsp_inc_tax = $request->has('fu_single_dsp_inc_tax') ? $request->input('fu_single_dsp_inc_tax') : 0;
                $fu_sub_sku = $request->has('fu_sub_sku') ? $request->input('fu_sub_sku') : 0;
                $su_single_dpp = $request->has('su_single_dpp') ? $request->input('su_single_dpp') : 0;
                $su_single_dpp_inc_tax = $request->has('su_single_dpp_inc_tax') ? $request->input('su_single_dpp_inc_tax') : 0;
                $su_profit_percent = $request->has('su_profit_percent') ? $request->input('su_profit_percent') : 0;
                $su_single_dsp = $request->has('su_single_dsp') ? $request->input('su_single_dsp') : 0;
                $su_single_dsp_inc_tax = $request->has('su_single_dsp_inc_tax') ? $request->input('su_single_dsp_inc_tax') : 0;
                $su_sub_sku = $request->has('su_sub_sku') ? $request->input('su_sub_sku') : 0;

       

                $variation->first_unit_sku = $fu_sub_sku;
                $variation->second_unit_sku = $su_sub_sku;
                $variation->fu_default_purchase_price = $this->productUtil->num_uf($fu_single_dpp);
                $variation->fu_dpp_inc_tax = $this->productUtil->num_uf($fu_single_dpp_inc_tax);
                $variation->fu_profit_percent = $this->productUtil->num_uf($fu_profit_percent);
                $variation->fu_default_sell_price = $this->productUtil->num_uf($fu_single_dsp);
                $variation->fu_sell_price_inc_tax = $this->productUtil->num_uf($fu_single_dsp_inc_tax);
                $variation->su_default_purchase_price = $this->productUtil->num_uf($su_single_dpp);
                $variation->su_dpp_inc_tax = $this->productUtil->num_uf($su_single_dpp_inc_tax);
                $variation->su_default_sell_price = $this->productUtil->num_uf($su_single_dsp);
                $variation->su_profit_percent = $this->productUtil->num_uf($su_profit_percent);
                $variation->su_sell_price_inc_tax = $this->productUtil->num_uf($su_single_dsp_inc_tax);



                $variation->save();

                Media::uploadMedia($product->business_id, $variation, $request, 'variation_images');

            } elseif ($product->type == self::PRODUCT_TYPE_VARIABLE) {

                //Update existing variations
                $input_variations_edit = $request->get('product_variation_edit');
                if (! empty($input_variations_edit)) {
                    $this->productUtil->updateVariableProductVariations($product->id, $input_variations_edit,$request->input('sku_type'));
                }

                //Add new variations created.
                $input_variations = $request->input('product_variation');
                if (! empty($input_variations)) {
                    $this->productUtil->createVariableProductVariations($product->id, $input_variations, $request->input('sku_type'));
                }
            } elseif ($product->type == self::PRODUCT_TYPE_COMBO) {

                //Create combo_variations array by combining variation_id and quantity.
                $combo_variations = [];
                if (! empty($request->input('composition_variation_id'))) {
                    $composition_variation_id = $request->input('composition_variation_id');
                    $quantity = $request->input('quantity');
                    $unit = $request->input('unit');

                    foreach ($composition_variation_id as $key => $value) {
                        $combo_variations[] = [
                            'variation_id' => $value,
                            'quantity' => $quantity[$key],
                            'unit_id' => $unit[$key],
                        ];
                    }
                }

                $variation = Variation::find($request->input('combo_variation_id'));
                $variation->sub_sku = $product->sku;
                $variation->default_purchase_price = $this->productUtil->num_uf($request->input('item_level_purchase_price_total'));
                $variation->dpp_inc_tax = $this->productUtil->num_uf($request->input('purchase_price_inc_tax'));
                $variation->profit_percent = $this->productUtil->num_uf($request->input('profit_percent'));
                $variation->default_sell_price = $this->productUtil->num_uf($request->input('selling_price'));
                $variation->sell_price_inc_tax = $this->productUtil->num_uf($request->input('selling_price_inc_tax'));
                $variation->combo_variations = $combo_variations;


                $fu_single_dpp = $request->has('fu_single_dpp') ? $request->input('fu_single_dpp') : $variation->fu_default_purchase_price;
                $fu_single_dpp_inc_tax = $request->has('fu_single_dpp_inc_tax') ? $request->input('fu_single_dpp_inc_tax') : $variation->fu_dpp_inc_tax;
                $fu_profit_percent = $request->has('fu_profit_percent') ? $request->input('fu_profit_percent') : $variation->fu_profit_percent;
                $fu_single_dsp = $request->has('fu_single_dsp') ? $request->input('fu_single_dsp') : $variation->fu_default_sell_price;
                $fu_single_dsp_inc_tax = $request->has('fu_single_dsp_inc_tax') ? $request->input('fu_single_dsp_inc_tax') : $variation->fu_sell_price_inc_tax;
                $fu_sub_sku = $request->has('fu_sub_sku') ? $request->input('fu_sub_sku') : $variation->first_unit_sku;
                $su_single_dpp = $request->has('su_single_dpp') ? $request->input('su_single_dpp') : $variation->su_default_purchase_price;
                $su_single_dpp_inc_tax = $request->has('su_single_dpp_inc_tax') ? $request->input('su_single_dpp_inc_tax') : $variation->su_dpp_inc_tax;
                $su_profit_percent = $request->has('su_profit_percent') ? $request->input('su_profit_percent') : $variation->su_profit_percent;
                $su_single_dsp = $request->has('su_single_dsp') ? $request->input('su_single_dsp') : $variation->su_default_sell_price;
                $su_single_dsp_inc_tax = $request->has('su_single_dsp_inc_tax') ? $request->input('su_single_dsp_inc_tax') : $variation->su_sell_price_inc_tax;
                $su_sub_sku = $request->has('su_sub_sku') ? $request->input('su_sub_sku') : $variation->second_unit_sku;



                $variation->first_unit_sku = $fu_sub_sku;
                $variation->second_unit_sku = $su_sub_sku;
                $variation->fu_default_purchase_price = $this->productUtil->num_uf($fu_single_dpp);
                $variation->fu_dpp_inc_tax = $this->productUtil->num_uf($fu_single_dpp_inc_tax);
                $variation->fu_profit_percent = $this->productUtil->num_uf($fu_profit_percent);
                $variation->fu_default_sell_price = $this->productUtil->num_uf($fu_single_dsp);
                $variation->fu_sell_price_inc_tax = $this->productUtil->num_uf($fu_single_dsp_inc_tax);
                $variation->su_default_purchase_price = $this->productUtil->num_uf($su_single_dpp);
                $variation->su_dpp_inc_tax = $this->productUtil->num_uf($su_single_dpp_inc_tax);
                $variation->su_default_sell_price = $this->productUtil->num_uf($su_single_dsp);
                $variation->su_profit_percent = $this->productUtil->num_uf($su_profit_percent);
                $variation->su_sell_price_inc_tax = $this->productUtil->num_uf($su_single_dsp_inc_tax);



                $variation->save();
            }

            //Add product racks details.
            $product_racks = $request->get('product_racks', null);
            if (! empty($product_racks)) {
                $this->productUtil->addRackDetails($business_id, $product->id, $product_racks);
            }

            $product_racks_update = $request->get('product_racks_update', null);
            if (! empty($product_racks_update)) {
                $this->productUtil->updateRackDetails($business_id, $product->id, $product_racks_update);
            }

            //Set Module fields
            if (! empty($request->input('has_module_data'))) {
                $this->moduleUtil->getModuleData('after_product_saved', ['product' => $product, 'request' => $request]);
            }


            if(!empty($common_settings['enable_product_premade'])){
                $premade_products = $request->get('products', []);
                $this->productUtil->createOrUpdatePremade($business_id, $product->id, $premade_products);
            }


            Media::uploadMedia($product->business_id, $product, $request, 'product_brochure', true);
            $discount_id = $request->input('discount_id');
            
            if(!empty($discount_id)){
                $discount = Discount::find($discount_id);
                // Retrieve the variation ids associated with the product
                $variation_ids = $product->variations->pluck('id')->toArray();
                // Use the variation ids in the sync method
                $discount->variations()->sync($variation_ids);
            }



            DB::commit();
            $output = [
                'success' => 1,
                'msg' => __('product.product_updated_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => $e->getMessage(),
            ];
        }

        if ($request->input('submit_type') == 'update_n_edit_opening_stock') {
            return redirect()->action([\App\Http\Controllers\OpeningStockController::class, 'add'],
                ['product_id' => $product->id]
            );
        } elseif ($request->input('submit_type') == 'submit_n_add_selling_prices') {
            return redirect()->action([\App\Http\Controllers\ProductController::class, 'addSellingPrices'],
                [$product->id]
            );
        } elseif ($request->input('submit_type') == 'save_n_add_another') {
            return redirect()->action([\App\Http\Controllers\ProductController::class, 'create']
            )->with('status', $output);
        }

        return redirect('products')->with('status', $output);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if (! auth()->user()->can('product.delete')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            try {

                $business_id = request()->session()->get('user.business_id');

                $can_be_deleted = true;
                $error_msg = '';

                //Check if any purchase or transfer exists
                $count = PurchaseLine::join('transactions as T','purchase_lines.transaction_id','=','T.id')
                                    ->whereIn('T.type', ['purchase'])
                                    ->where('T.business_id', $business_id)
                                    ->where('purchase_lines.product_id', $id)
                                    ->count();
                if ($count > 0) {
                    $can_be_deleted = false;
                    $error_msg = __('lang_v1.purchase_already_exist');
                } else {
                    //Check if any opening stock sold
                    $count = PurchaseLine::join('transactions as T','purchase_lines.transaction_id','=','T.id')
                                    ->where('T.type', 'opening_stock')
                                    ->where('T.business_id', $business_id)
                                    ->where('purchase_lines.product_id', $id)
                                    ->where('purchase_lines.quantity_sold', '>', 0)
                                    ->count();
                    if ($count > 0) {
                        $can_be_deleted = false;
                        $error_msg = __('lang_v1.opening_stock_sold');
                    } else {
                        //Check if any stock is adjusted
                        $count = PurchaseLine::join('transactions as T','purchase_lines.transaction_id','=','T.id')
                                    ->where('T.business_id', $business_id)
                                    ->where('purchase_lines.product_id', $id)
                                    ->where('purchase_lines.quantity_adjusted', '>', 0)
                                    ->count();
                        if ($count > 0) {
                            $can_be_deleted = false;
                            $error_msg = __('lang_v1.stock_adjusted');
                        }
                    }
                }

                $product = Product::where('id', $id)
                                ->where('business_id', $business_id)
                                ->with('variations')
                                ->first();


                // check for enable stock = 0 product
                if($product->enable_stock == 0){
                    $t_count = TransactionSellLine::join('transactions as T','transaction_sell_lines.transaction_id','=','T.id')
                                                ->where('T.business_id', $business_id)
                                                ->where('transaction_sell_lines.product_id', $id)
                                                ->count();

                    if ($t_count > 0) {
                        $can_be_deleted = false;
                        $error_msg = "can't delete product exit in sell";
                    }
                }

                //Check if product is added as an ingredient of any recipe
                if ($this->moduleUtil->isModuleInstalled('Manufacturing')) {
                    $variation_ids = $product->variations->pluck('id');

                    $exists_as_ingredient = \Modules\Manufacturing\Entities\MfgRecipeIngredient::whereIn('variation_id', $variation_ids)
                        ->exists();
                    if ($exists_as_ingredient) {
                        $can_be_deleted = false;
                        $error_msg = __('manufacturing::lang.added_as_ingredient');
                    }
                }
            
                if ($can_be_deleted) {
                    if (! empty($product)) {
                        DB::beginTransaction();
                        //Delete variation location details
                        VariationLocationDetails::where('product_id', $id)->delete();

                        SerialNumber::where('product_id', $id)->delete();
                        ServiceWarranty::where('product_id', $id)->delete();

                        $common_settings = session()->get('business.common_settings');
                        if(!empty($common_settings['enable_product_premade'])){
                            PremadeLine::where('business_id', $business_id)->where('premade_product_id', $id)->delete();
                        }

                        $product->delete();
                        event(new ProductsCreatedOrModified($product, 'deleted'));
                        DB::commit();
                    }

                    $output = [
                        'success' => true,
                        'msg' => __('lang_v1.product_delete_success'),
                    ];
                } else {
                    $output = ['success' => false,
                        'msg' => $error_msg,
                    ];
                }
            } catch (\Exception $e) {
                DB::rollBack();
                Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = ['success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    /**
     * Get subcategories list for a category.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getSubCategories(Request $request)
    {
        if (! empty($request->input('cat_id'))) {
            $category_id = $request->input('cat_id');
            $business_id = $request->session()->get('user.business_id');
            $sub_categories = Category::where('business_id', $business_id)
                        ->where('parent_id', $category_id)
                        ->select(['name', 'id'])
                        ->get();
            $html = '<option value="">None</option>';
            if (! empty($sub_categories)) {
                foreach ($sub_categories as $sub_category) {
                    $html .= '<option value="'.$sub_category->id.'">'.$sub_category->name.'</option>';
                }
            }
            echo $html;
            exit;
        }
    }

    /**
     * Get product form parts.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getProductVariationFormPart(Request $request){

        $business_id = $request->session()->get('user.business_id');
        $business = Business::findorfail($business_id);
        $profit_percent = $business->default_profit_percent;
        $common_settings = session()->get('business.common_settings');

        $use_multi_unit = $request->input('use_multi_unit');
        $first_conversion_unit_id = $request->input('first_conversion_unit_id');
        $second_conversion_unit_id = $request->input('second_conversion_unit_id');

        if($first_conversion_unit_id){
            $first_conversion_unit_id = 1;
        }else{
            $first_conversion_unit_id = 0;
        }
        if($second_conversion_unit_id){
            $second_conversion_unit_id = 1;
        }else{
            $second_conversion_unit_id = 0;
        }


        $action = $request->input('action');
        if ($request->input('action') == 'add') {
            if ($request->input('type') == 'single') {

                return view('product.partials.single_product_form_part')->with([
                    'profit_percent' => $profit_percent, 
                    'common_settings' => $common_settings, 

                    'use_multi_unit' => $use_multi_unit,
                    'first_conversion_unit_id' => $first_conversion_unit_id,
                    'second_conversion_unit_id' => $second_conversion_unit_id,

                ]);
            
            } elseif ($request->input('type') == 'variable') {

                $variation_templates = VariationTemplate::where('business_id', $business_id)->pluck('name', 'id')->toArray();
                $variation_templates = ['' => __('messages.please_select')] + $variation_templates;

                return view('product.partials.variable_product_form_part')->with(
                    compact(
                        'variation_templates', 
                        'use_multi_unit', 
                        'first_conversion_unit_id', 
                        'second_conversion_unit_id', 
                        'profit_percent', 
                        'common_settings', 
                        'action'
                    ));

            } elseif ($request->input('type') == 'combo') {

                return view('product.partials.combo_product_form_part')->with(
                    compact(
                        'profit_percent', 
                        'use_multi_unit', 
                        'first_conversion_unit_id', 
                        'second_conversion_unit_id', 
                        'action'
                    ));

            }

        } elseif ($request->input('action') == 'edit' || $request->input('action') == 'duplicate') {
            $product_id = $request->input('product_id');

            $product = Product::where('business_id', $business_id)->findOrFail($product_id);

            $action = $request->input('action');
            if ($request->input('type') == 'single') {
                $product_deatails = ProductVariation::where('product_id', $product_id)->with(['variations', 'variations.media'])->first();
                return view('product.partials.edit_single_product_form_part')->with(
                    compact(
                        'product_deatails',
                        'product',

                        'use_multi_unit', 
                        'first_conversion_unit_id', 
                        'second_conversion_unit_id', 

                        'common_settings', 
                        'action'
                    ));
            } elseif ($request->input('type') == 'variable') {
                $product_variations = ProductVariation::where('product_id', $product_id)->with(['variations', 'variations.media'])->get();
                return view('product.partials.variable_product_form_part')->with(
                    compact(
                        'product_variations',
                        'product',
                        'use_multi_unit', 
                        'first_conversion_unit_id', 
                        'second_conversion_unit_id', 
                        'profit_percent', 
                        'common_settings', 
                        'action'
                    )
                );

            } elseif ($request->input('type') == 'combo') {
                $product_deatails = ProductVariation::where('product_id', $product_id)->with(['variations', 'variations.media'])->first();
                $combo_variations = $this->productUtil->__getComboProductDetails($product_deatails['variations'][0]->combo_variations, $business_id);
                $variation_id = $product_deatails['variations'][0]->id;
                $profit_percent = $product_deatails['variations'][0]->profit_percent;
                return view('product.partials.combo_product_form_part')->with(
                    compact(
                        'combo_variations',
                        'product', 
                        'use_multi_unit', 
                        'first_conversion_unit_id', 
                        'second_conversion_unit_id', 
                        'profit_percent',
                        'common_settings', 
                        'action', 
                        'variation_id'
                    )
                );


            }
        }


    }

    /**
     * Get product form parts.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getVariationValueRow(Request $request)
    {
        $business_id = $request->session()->get('user.business_id');
        $business = Business::findorfail($business_id);
        $profit_percent = $business->default_profit_percent;

        $variation_index = $request->input('variation_row_index');
        $value_index = $request->input('value_index') + 1;

        $row_type = $request->input('row_type', 'add');

        $use_multi_unit = $request->input('use_multi_unit');
        $first_conversion_unit_id = $request->input('first_conversion_unit_id');
        $second_conversion_unit_id = $request->input('second_conversion_unit_id');

        if($first_conversion_unit_id){
            $first_conversion_unit_id = 1;
        }else{
            $first_conversion_unit_id = 0;
        }
        if($second_conversion_unit_id){
            $second_conversion_unit_id = 1;
        }else{
            $second_conversion_unit_id = 0;
        }

        return view('product.partials.variation_value_row')->with(
            compact(
                'profit_percent', 

                'use_multi_unit', 
                'first_conversion_unit_id', 
                'second_conversion_unit_id', 

                'variation_index', 
                'value_index', 
                'row_type'
            )
        );

    }

    /**
     * Get product form parts.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getProductVariationRow(Request $request)
    {
        $business_id = $request->session()->get('user.business_id');
        $business = Business::findorfail($business_id);
        $profit_percent = $business->default_profit_percent;

        $variation_templates = VariationTemplate::where('business_id', $business_id)
                                                ->pluck('name', 'id')->toArray();
        $variation_templates = ['' => __('messages.please_select')] + $variation_templates;

        $row_index = $request->input('row_index', 0);
        $action = $request->input('action');


        $use_multi_unit = $request->input('use_multi_unit');
        $first_conversion_unit_id = $request->input('first_conversion_unit_id');
        $second_conversion_unit_id = $request->input('second_conversion_unit_id');

        if($first_conversion_unit_id){
            $first_conversion_unit_id = 1;
        }else{
            $first_conversion_unit_id = 0;
        }
        if($second_conversion_unit_id){
            $second_conversion_unit_id = 1;
        }else{
            $second_conversion_unit_id = 0;
        }


        return view('product.partials.product_variation_row')->with(
            compact(
                'variation_templates', 
                'row_index', 
                'use_multi_unit', 
                'first_conversion_unit_id', 
                'second_conversion_unit_id', 
                'action', 
                'profit_percent'
            )
        );


    }

    /**
     * Get product form parts.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getVariationTemplate(Request $request)
    {
        $business_id = $request->session()->get('user.business_id');
        $business = Business::findorfail($business_id);
        $profit_percent = $business->default_profit_percent;

        $template = VariationTemplate::where('id', $request->input('template_id'))
                                                ->with(['values'])
                                                ->first();
        $row_index = $request->input('row_index');

        $values = [];
        foreach ($template->values as $v) {
            $values[] = [
                'id' => $v->id,
                'text' => $v->name,
            ];
        }



        $use_multi_unit = $request->input('use_multi_unit');
        $first_conversion_unit_id = $request->input('first_conversion_unit_id');
        $second_conversion_unit_id = $request->input('second_conversion_unit_id');

        if($first_conversion_unit_id){
            $first_conversion_unit_id = 1;
        }else{
            $first_conversion_unit_id = 0;
        }
        if($second_conversion_unit_id){
            $second_conversion_unit_id = 1;
        }else{
            $second_conversion_unit_id = 0;
        }

        


        return [
            'html' => view('product.partials.product_variation_template')->with(
                compact(
                    'template', 

                    'use_multi_unit', 
                    'first_conversion_unit_id', 
                    'second_conversion_unit_id', 

                    'row_index', 
                    'profit_percent'
                )
            )->render(),

            'values' => $values,
        ];

    }

    /**
     * Return the view for combo product row
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getComboProductEntryRow(Request $request)
    {
        if (request()->ajax()) {
            $product_id = $request->input('product_id');
            $variation_id = $request->input('variation_id');
            $business_id = $request->session()->get('user.business_id');

            if (! empty($product_id)) {
                $product = Product::where('id', $product_id)
                        ->with(['unit'])
                        ->first();

                $query = Variation::where('product_id', $product_id)
                        ->with(['product_variation']);

                if ($variation_id !== '0') {
                    $query->where('id', $variation_id);
                }
                $variations = $query->get();

                $sub_units = $this->productUtil->getSubUnits($business_id, $product['unit']->id);


                $use_multi_unit = $request->input('use_multi_unit');
                $first_conversion_unit_id = $request->input('first_conversion_unit_id');
                $second_conversion_unit_id = $request->input('second_conversion_unit_id');
        
                if($first_conversion_unit_id){
                    $first_conversion_unit_id = 1;
                }else{
                    $first_conversion_unit_id = 0;
                }
                if($second_conversion_unit_id){
                    $second_conversion_unit_id = 1;
                }else{
                    $second_conversion_unit_id = 0;
                }
        

                return view('product.partials.combo_product_entry_row')->with(
                    compact(
                        'product', 

                        'use_multi_unit', 
                        'first_conversion_unit_id', 
                        'second_conversion_unit_id', 

                        'variations', 
                        'sub_units'
                    )
                );

            }
        }
    }

    /**
     * Retrieves products list.
     *
     * @param  string  $q
     * @param  bool  $check_qty
     * @return JSON
     */
    public function getProducts()
    {
        if (request()->ajax()) {
            $search_term = request()->input('term', '');
            $location_id = request()->input('location_id', null);
            $check_qty = request()->input('check_qty', false);
            $price_group_id = request()->input('price_group', null);
            $business_id = request()->session()->get('user.business_id');
            $not_for_selling = request()->get('not_for_selling', null);
            $price_group_id = request()->input('price_group', '');
            $product_types = request()->get('product_types', []);
            $limit = request()->input('limit', 50); // Default limit of 50 for better performance

            $search_fields = request()->get('search_fields', ['name', 'sku']);
            if (in_array('sku', $search_fields)) {
                $search_fields[] = 'sub_sku';
            }

            
            $result = $this->productUtil->filterProduct($business_id, $search_term, $location_id, $not_for_selling, $price_group_id, $product_types, $search_fields, $check_qty, 'like', $limit);

            return json_encode($result);
        }
    }



    public function getSellProducts(){
        if (request()->ajax()) {
            
            $search_term = request()->input('term', '');
            $location_id = request()->input('location_id', null);

            $customer_id = request()->get('customer_id', null);
            $search_fields = request()->get('search_fields', ['name', 'sku']);
            $business_id = request()->session()->get('user.business_id');
            $product_types = request()->get('product_types', []);

            $price_group_id = request()->input('price_group', '');
            $check_qty = request()->input('check_qty', false);


            $result = $this->productUtil->filterSellProduct($business_id, $search_term, $location_id, $customer_id, $price_group_id, $product_types, $search_fields, $check_qty);

            return json_encode($result);

        }
        
    }


        // getPremadeProducts
        /**
     * Retrieves products list.
     *
     * @param  string  $q
     * @param  bool  $check_qty
     * @return JSON
     */
    public function getPremadeProductRaw($variation_id) {

        if (request()->ajax()) {

            $location_id = null;
            $quantity = request()->input('quantity', 1);
            $purchase_line_id = request()->input('purchase_line_id', null);
            $row_count = request()->input('product_row', null);
            $check_qty = false;
            $business_id = request()->session()->get('user.business_id');

            $common_settings = session()->get('business.common_settings');

            $business_details = $this->businessUtil->getDetails($business_id);

            $pos_settings = empty($business_details->pos_settings) ? $this->businessUtil->defaultPosSettings() : json_decode($business_details->pos_settings, true);

            $product = $this->productUtil->getDetailsFromVariation($variation_id, $business_id, $location_id, $check_qty);

            if (!isset($product->quantity_ordered)) {
                $product->quantity_ordered = $quantity;
            }
            $product->secondary_unit_quantity = ! isset($product->secondary_unit_quantity) ? 0 : $product->secondary_unit_quantity;
    
            $product->formatted_qty_available = $this->productUtil->num_f($product->qty_available, false, null, true);
    
            $sub_units = $this->productUtil->getSubUnits($business_id, $product->unit_id, false, $product->product_id);

            $output['success'] = true;
            $so_line = null;


            $output['html_content'] = view('product.partials.premade_row')->with(
                compact(
                    'product', 
                    'common_settings', 
                    'business_details', 
                    'row_count', 
                    'pos_settings', 
                    'sub_units', 
                    'so_line', 
                    'purchase_line_id', 
                    'quantity', 
                )
            )->render();

            return $output;


        }  

    }



    public function getProductsByBarcode(){
        if (request()->ajax()) {

            $output = [];

            try {

                $location_id = request()->input('location_id', null);
                $check_qty = request()->input('check_qty', false);
                $business_id = request()->session()->get('user.business_id');
                $product_types = request()->get('product_types', []);
                $price_group_id = request()->input('price_group', null);


                $business_details = $this->businessUtil->getDetails($business_id);
                $pos_settings = empty($business_details->pos_settings) ? $this->businessUtil->defaultPosSettings() : json_decode($business_details->pos_settings, true);
                
                $check_qty = ! empty($pos_settings['allow_overselling']) ? false : true;



                $is_sales_order = request()->has('is_sales_order') && request()->input('is_sales_order') == 'true' ? true : false;
                $is_draft = request()->has('is_draft') && request()->input('is_draft') == 'true' ? true : false;

                $is_overselling_allowed = request()->get('is_overselling_allowed');
                $for_so = request()->get('for_so');
                $is_draft = request()->get('is_draft');
                $quantity = request()->get('quantity', 1);

                $sku = request()->get('sku');
                if(empty($sku)){
                    $sku = request()->input('sku');
                }

                $query = Variation::join('products AS p', 'variations.product_id', '=', 'p.id')
                                ->join('product_variations AS pv', 'variations.product_variation_id', '=', 'pv.id')

                                ->leftjoin('variation_location_details AS vld', 'variations.id', '=', 'vld.variation_id')


                                ->leftjoin('units', 'p.unit_id', '=', 'units.id')
                                ->leftjoin('units as u', 'p.secondary_unit_id', '=', 'u.id')
                                ->leftjoin('units as first_conversion_unit', 'p.first_conversion_unit_id', '=', 'first_conversion_unit.id')
                                ->leftjoin('units as second_conversion_unit', 'p.second_conversion_unit_id', '=', 'second_conversion_unit.id')
                                ->leftjoin('brands', function ($join) {
                                    $join->on('p.brand_id', '=', 'brands.id')
                                        ->whereNull('brands.deleted_at');
                                })
                                ->where('p.business_id', $business_id);

                                if (!empty($location_id) && $check_qty) {
                                    //Check for enable stock, if enabled check for location id.
                                    $query->where(function ($query) use ($location_id) {
                                        $query->where('p.enable_stock', '!=', 1)
                                              ->orWhere('vld.location_id', $location_id);
                                    });
                                }

              

                $query->where('p.business_id', $business_id)
                        ->where('p.type', '!=', 'modifier')
                        ->where(function ($query) use ($sku) {
                            $query->where('p.sku', '=', $sku)
                                ->orWhere('variations.sub_sku', '=', $sku)
                                ->orWhere('variations.first_unit_sku', '=', $sku)
                                ->orWhere('variations.second_unit_sku', '=', $sku);
                        });


                 //Add condition for check of quantity. (if stock is not enabled or qty_available > 0)
                // if ($check_qty) {
                //     $query->where(function ($query) {
                //         $query->where('p.enable_stock', '!=', 1)
                //             ->orWhere('vld.qty_available', '>', 0);
                //     });
                // }
                $common_settings = session()->get('business.common_settings');
                $product = $query->select(
                        DB::raw("IF(pv.is_dummy = 0, CONCAT(p.name, ' (', pv.name, ':',variations.name, ')'), p.name) AS product_name"),
                        'p.id as product_id',
                        'p.brand_id',
                        
                        'p.enable_serial_number',

                        'p.pre_made',
                        'p.category_id',
                        'p.tax as tax_id',
                        'p.enable_stock',
                        'p.enable_sr_no',
                        'p.type as product_type',
                        'p.name as product_actual_name',
                        'p.warranty_id',
                        'p.product_custom_field1',
                        'p.product_custom_field2',
                        'p.product_custom_field3',
                        'p.product_custom_field4',
                        'p.product_custom_field5',
                        'p.product_custom_field6',
                        'p.product_custom_field7',
                        'p.product_custom_field8',
                        'p.product_custom_field9',
                        'p.product_custom_field10',
                        'p.product_custom_field11',
                        'p.product_custom_field12',
                        'p.product_custom_field13',
                        'p.product_custom_field14',
                        'p.product_custom_field15',
                        'p.product_custom_field16',
                        'p.product_custom_field17',
                        'p.product_custom_field18',
                        'p.product_custom_field19',
                        'p.product_custom_field20',




                        'pv.name as product_variation_name',
                        'pv.is_dummy as is_dummy',
                        'variations.name as variation_name',
                        'variations.sub_sku',
                        'p.barcode_type',



                        'vld.qty_available',
                 
                        'variations.combo_variations',  //Used in combo products


                                            
                        'p.use_multi_unit',
                        'p.first_conversion_unit_id',
                        'p.second_conversion_unit_id',
                        'p.first_conversion_unit_rate',
                        'p.second_conversion_unit_rate',
                        'first_conversion_unit.short_name as first_conversion_short_name',
                        'first_conversion_unit.actual_name as first_conversion_name',
                        'second_conversion_unit.actual_name as second_conversion_name',
                        'second_conversion_unit.short_name as second_conversion_short_name',


                        'variations.first_unit_sku',
                        'variations.fu_default_purchase_price',
                        'variations.fu_dpp_inc_tax',
                        'variations.fu_default_sell_price',
                        'variations.fu_sell_price_inc_tax',
                        
                        'variations.second_unit_sku',
                        'variations.su_default_purchase_price',
                        'variations.su_dpp_inc_tax',
                        'variations.su_default_sell_price',
                        'variations.su_sell_price_inc_tax',

                        'variations.id as variation_id',
                        'variations.default_sell_price',
                        'variations.dpp_inc_tax as default_purchase_price',
                        'variations.sell_price_inc_tax',
                        'units.short_name as unit',
                        'units.id as unit_id',
                        'units.allow_decimal as unit_allow_decimal',
                        'u.short_name as second_unit',
                        'brands.name as brand',

                        DB::raw('(SELECT purchase_price_inc_tax FROM purchase_lines WHERE variation_id=variations.id ORDER BY id DESC LIMIT 1) as last_purchased_price'),

                        DB::raw("CASE WHEN variations.first_unit_sku = '$sku' THEN p.first_conversion_unit_id WHEN variations.second_unit_sku = '$sku' THEN p.second_conversion_unit_id ELSE null END as search_unit_id"),

                    )->groupBy('variations.id')
                    ->orderBy('vld.qty_available', 'desc')->first();

                if(empty($product)){
                    $output['success'] = false;
                    $output['msg'] = __('lang_v1.no_products_found');
                    return $output;
                }else{
                    
                    $enable_stock = $product->enable_stock;
                    $qty_available = $product->qty_available;
            
                    $variation_id = $product->variation_id;
                    if(!isset($product->quantity_ordered)){
                        $product->quantity_ordered = $quantity;
                    }
                    
                    $product->secondary_unit_quantity = ! isset($product->secondary_unit_quantity) ? 0 : $product->secondary_unit_quantity;
                    $product->formatted_qty_available = $this->productUtil->num_f($product->qty_available, false, null, true);
  
                    if ($enable_stock != 1 || $qty_available > 0 || $is_overselling_allowed || $for_so || $is_draft) {

                        $row_count = request()->get('product_row');
                        $row_count = $row_count + 1;
                
                        $is_direct_sell = false;
                        $so_line = null;
                        $last_sell_line = null;

          
        
                        $lot_numbers = [];
                        if (request()->session()->get('business.enable_lot_number') == 1 || request()->session()->get('business.enable_product_expiry') == 1) {
                            $lot_number_obj = $this->transactionUtil->getLotNumbersFromVariation($variation_id, $business_id, $location_id, true);
                            foreach ($lot_number_obj as $lot_number) {
                                $lot_number->qty_formated = $this->productUtil->num_f($lot_number->qty_available);
                                $lot_numbers[] = $lot_number;
                            }
                        }
                        $product->lot_numbers = $lot_numbers;
                        
                        $purchase_line_id = request()->get('purchase_line_id');
                        
                        $price_group = request()->input('price_group');
                        $output['variation_id'] = $variation_id;
                        $output['success'] = true;
                        $output['enable_sr_no'] = $product->enable_sr_no;
                        $output['search_unit_id'] = $product->search_unit_id;
                        
                        $waiters = [];
                        if ($this->productUtil->isModuleEnabled('service_staff') && ! empty($pos_settings['inline_service_staff'])) {
                            $waiters_enabled = true;
                            $waiters = $this->productUtil->serviceStaffDropdown($business_id, $location_id);
                        }
                   
                        if (!empty($price_group) && $price_group != "NaN") {
                            $variation_group_prices = $this->productUtil->getVariationGroupPrice($variation_id, $price_group, $product->tax_id);
                            if (!empty($variation_group_prices['price_inc_tax'])) {
                                $product->sell_price_inc_tax = $variation_group_prices['price_inc_tax'];
                                $product->default_sell_price = $variation_group_prices['price_exc_tax'];
                            }
                        }
                        
                        $warranties = $this->__getwarranties();
                        $tax_dropdown = TaxRate::forBusinessDropdown($business_id, true, true);
                        $enabled_modules = $this->transactionUtil->allModulesEnabled();
                        $sub_units = $this->productUtil->getSubUnits($business_id, $product->unit_id, false, $product->product_id);


                        //Get customer group and change the price accordingly
                        $customer_id = request()->get('customer_id', null);
                        $cg = $this->contactUtil->getCustomerGroup($business_id, $customer_id);
                        $percent = (empty($cg) || empty($cg->amount) || $cg->price_calculation_type != 'percentage') ? 0 : $cg->amount;
                        $product->default_sell_price = $product->default_sell_price + ($percent * $product->default_sell_price / 100);
                        $product->sell_price_inc_tax = $product->sell_price_inc_tax + ($percent * $product->sell_price_inc_tax / 100);


                        if (request()->get('type') == 'sell-return') {
                            $output['html_content'] = view('sell_return.partials.product_row')->with(compact(
                                'product', 
                                'row_count', 
                                'tax_dropdown', 
                                'enabled_modules', 
                                'sub_units'
                            ))->render();
                        } else {
                            $is_cg = ! empty($cg->id) ? true : false;
                            $discount = $this->productUtil->getProductDiscount($product, $business_id, $location_id, $is_cg, $price_group, $variation_id);
                
                         
                            $edit_discount = auth()->user()->can('edit_product_discount_from_pos_screen');
                            $edit_price = auth()->user()->can('edit_product_price_from_pos_screen');
                            
                
                            $output['html_content'] = view('sale_pos.product_row')->with(compact(
                                'product', 
                                'row_count', 
                                'tax_dropdown', 

                                'location_id',  
                                'common_settings',

                                'enabled_modules', 
                                'pos_settings', 
                                'sub_units', 
                                'discount', 
                                'waiters', 
                                'edit_discount', 
                                'edit_price', 
                                'purchase_line_id', 
                                'warranties', 
                                'quantity', 
                                'is_direct_sell', 
                                'so_line', 
                                'is_sales_order', 
                                
                                'last_sell_line'
                            ))->render();


                        }
              
                        return $output;

                    } else {
                        $output['success'] = false;
                        $output['msg'] = __('lang_v1.item_out_of_stock');
                        return $output;
                    }

                }

                
            } catch (\Exception $e) {
                Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
                $output['success'] = false;
                $output['msg'] = __('lang_v1.item_out_of_stock');
            }

            
            return $output;

        }

    }











    
    public function getProductsByBarcodeTime(){

        $output = [];
            
        $location_id = request()->input('location_id', null);
        $business_id = request()->session()->get('user.business_id');
        
        $sku = request()->input('sku', null);
        $row_count = request()->input('product_row');
        $price_group_id = request()->input('price_group', null);


        $is_overselling_allowed = request()->get('is_overselling_allowed');
        


        if(!empty($sku)){



            $query = Variation::join('products AS p', 'variations.product_id', '=', 'p.id')
                                ->join('product_variations AS pv', 'variations.product_variation_id', '=', 'pv.id')
                                ->leftJoin('variation_location_details AS vld', 'variations.id', '=', 'vld.variation_id')
                                ->leftJoin('units', 'p.unit_id', '=', 'units.id')
                                
                                // Optional joins for secondary units (remove if not needed)
                                ->leftJoin('units as u', 'p.secondary_unit_id', '=', 'u.id')
                                ->leftJoin('units as first_conversion_unit', 'p.first_conversion_unit_id', '=', 'first_conversion_unit.id')
                                ->leftJoin('units as second_conversion_unit', 'p.second_conversion_unit_id', '=', 'second_conversion_unit.id')
                                
                                ->leftJoin('brands', 'p.brand_id', '=', 'brands.id')
                                
                                ->where('p.business_id', $business_id)
                                ->where('p.type', '!=', 'modifier');
                                if (strlen($sku) >= 2) {
                                    $query->where(function ($query) use ($sku) {
                                        // This grouping ensures SKU conditions are correctly handled
                                        $query->where('p.sku', $sku)
                                            ->orWhere('variations.sub_sku', $sku)
                                            ->orWhere('variations.first_unit_sku', $sku)
                                            ->orWhere('variations.second_unit_sku', $sku);
                                    });
                                }else{
                                    $output['success'] = false;
                                    $output['msg'] = __('lang_v1.no_products_found');
                                    return $output;
                                }



                $query->select(
                    DB::raw("IF(pv.is_dummy = 0, CONCAT(p.name, ' (', pv.name, ':',variations.name, ')'), p.name) AS product_name"),
                    'vld.qty_available',

                    'p.id as product_id',
                    'p.brand_id',
                    'p.enable_serial_number',
                    'p.category_id',
                    'p.tax as tax_id',
                    'p.enable_stock',
                    'p.enable_sr_no',
                    'p.type as product_type',
                    'p.name as product_actual_name',
                    'p.warranty_id',


                    'p.pre_made',

        
                    'p.use_multi_unit',
                    'p.first_conversion_unit_id',
                    'p.second_conversion_unit_id',
                    'p.first_conversion_unit_rate',
                    'p.second_conversion_unit_rate',
                    'first_conversion_unit.short_name as first_conversion_short_name',
                    'first_conversion_unit.actual_name as first_conversion_name',
                    'second_conversion_unit.actual_name as second_conversion_name',
                    'second_conversion_unit.short_name as second_conversion_short_name',
                    'pv.name as product_variation_name',
                    'pv.is_dummy as is_dummy',
                    'variations.name as variation_name',
                    'variations.sub_sku',
                    'p.barcode_type',


                    'variations.first_unit_sku',
                    'variations.fu_default_purchase_price',
                    'variations.fu_dpp_inc_tax',
                    'variations.fu_default_sell_price',
                    'variations.fu_sell_price_inc_tax',
                    
                    'variations.second_unit_sku',
                    'variations.su_default_purchase_price',
                    'variations.su_dpp_inc_tax',
                    'variations.su_default_sell_price',
                    'variations.su_sell_price_inc_tax',


                    'variations.id as variation_id',
                    'variations.default_sell_price',
                    'variations.dpp_inc_tax as default_purchase_price',
                    'variations.sell_price_inc_tax',
                    'units.short_name as unit',
                    'units.id as unit_id',
                    'units.allow_decimal as unit_allow_decimal',
                    'u.short_name as second_unit',
                    'brands.name as brand',


                    DB::raw('(SELECT purchase_price_inc_tax FROM purchase_lines WHERE variation_id=variations.id ORDER BY id DESC LIMIT 1) as last_purchased_price'),


                    'variations.combo_variations',

                    DB::raw("CASE WHEN variations.first_unit_sku = '$sku' THEN p.first_conversion_unit_id WHEN variations.second_unit_sku = '$sku' THEN p.second_conversion_unit_id ELSE null END as search_unit_id"),
                );
            
            $product = $query->groupBy('variations.id')
                            ->orderBy('vld.qty_available', 'desc')
                            ->first();

            
    

            // search_unit_id
            if(empty($product)){
                $output['success'] = false;
                $output['msg'] = __('lang_v1.no_products_found');
                return $output;
            }else{


                if($is_overselling_allowed !== true && $is_overselling_allowed !== 'true' && $is_overselling_allowed !== 1) {
                    if($product->qty_available <= 0){
                        $output['success'] = false;
                        $output['msg'] = __('lang_v1.item_out_of_stock');
                        return $output;
                    }
                }

            

                $row_count = $row_count + 1;

                $output['success'] = true;


                $warranties = $this->__getwarranties();
                $tax_dropdown = TaxRate::forBusinessDropdown($business_id, true, true);
                // $enabled_modules = $this->transactionUtil->allModulesEnabled();
                $sub_units = $this->productUtil->getSubUnits($business_id, $product->unit_id, false, $product->product_id);


                $business_details = $this->businessUtil->getDetails($business_id);
                $pos_settings = empty($business_details->pos_settings) ? $this->businessUtil->defaultPosSettings() : json_decode($business_details->pos_settings, true);
                


                $now = \Carbon::now()->toDateTimeString();
                $discount = Discount::where('business_id', $business_id)
                                    ->where('location_id', $location_id)
                                    ->where('is_active', 1)
                                    ->where('starts_at', '<=', $now)
                                ->where('ends_at', '>=', $now)
                                ->where(function ($q) use ($product) {
                                    $q->where(function ($sub_q) use ($product) {
                                        if (! empty($product->brand_id)) {
                                            $sub_q->where('brand_id', $product->brand_id);
                                        }
                                        if (! empty($product->category_id)) {
                                            $sub_q->where('category_id', $product->category_id);
                                        }
                                    })
                                    ->orWhereHas('variations', function ($sub_q) use ($product) {
                                        $sub_q->where('variation_id', $product->variation_id);
                                    })
                                    ->orWhere(function ($sub_q) use ($product) {
                                        $sub_q->whereRaw('(brand_id = ? AND category_id IS NULL)', [$product->brand_id])
                                        ->orWhereRaw('(category_id = ? AND brand_id IS NULL)', [$product->category_id]);
                                    });
                                })
                                ->orderBy('priority', 'desc')
                                ->first();
        

                    if (! empty($discount)) {
                        $discount->formated_starts_at = $this->format_date($discount->starts_at->toDateTimeString(), true);
                        $discount->formated_ends_at = $this->format_date($discount->ends_at->toDateTimeString(), true);
                    }

                    $warranties = $this->__getwarranties();


                    $waiters = [];
                    //Get all service staff roles
                    $service_staff_roles = Role::where('business_id', $business_id)
                                                ->where('is_service_staff', 1)
                                                ->pluck('name')
                                                ->toArray();
            
                    //Get all users of service staff roles
                    if (! empty($service_staff_roles)) {
                        $waiters = User::where('business_id', $business_id)
                            ->role($service_staff_roles);
            
                        if (! empty($location_id)) {
                            $waiters->permission(['location.'.$location_id, 'access_all_locations']);
                        }
                        $waiters = $waiters->select('id', DB::raw('CONCAT(COALESCE(first_name, ""), " ", COALESCE(last_name, "")) as full_name'))->get()->pluck('full_name', 'id');
                    
                    }


            
                $variation_id = $product->variation_id;
                $output['variation_id'] = $variation_id;
                $output['search_unit_id'] = $product->search_unit_id;

                $edit_discount = auth()->user()->can('edit_product_discount_from_pos_screen');
                $edit_price = auth()->user()->can('edit_product_price_from_pos_screen');
                        
                $output['html_content'] = view('sale_pos.product_row_barcode')->with(compact(
                    'product', 
                    'row_count',
                    'location_id',
                    'edit_discount', 
                    'tax_dropdown', 
                    'pos_settings', 
                    'edit_price', 
                    'sub_units', 
                    'discount', 
                    'warranties', 
                    'waiters'
                    
                ))->render();


            }


            return $output;


        }

       

 

    }










    private function __getwarranties()
    {
        $business_id = session()->get('user.business_id');
        $common_settings = session()->get('business.common_settings');
        $is_warranty_enabled = ! empty($common_settings['enable_product_warranty']) ? true : false;
        $warranties = $is_warranty_enabled ? Warranty::forDropdown($business_id) : [];

        return $warranties;
    }
    
    /**
     * Parse the weighing barcode.
     *
     * @return array
     */
    private function __parseWeighingBarcode($scale_barcode)
    {
        $business_id = session()->get('user.business_id');

        $scale_setting = session()->get('business.weighing_scale_setting');

        $error_msg = trans('messages.something_went_wrong');

        //Check for prefix.
        if ((strlen($scale_setting['label_prefix']) == 0) || Str::startsWith($scale_barcode, $scale_setting['label_prefix'])) {
            $scale_barcode = substr($scale_barcode, strlen($scale_setting['label_prefix']));

            //Get product sku, trim left side 0
            $sku = ltrim(substr($scale_barcode, 0, $scale_setting['product_sku_length'] + 1), '0');

            //Get quantity integer
            $qty_int = substr($scale_barcode, $scale_setting['product_sku_length'] + 1, $scale_setting['qty_length'] + 1);

            //Get quantity decimal
            $qty_decimal = '0.'.substr($scale_barcode, $scale_setting['product_sku_length'] + $scale_setting['qty_length'] + 2, $scale_setting['qty_length_decimal'] + 1);

            $qty = (float) $qty_int + (float) $qty_decimal;

            //Find the variation id
            $result = $this->productUtil->filterProduct($business_id, $sku, null, false, null, [], ['sub_sku'], false, 'exact')->first();

            if (! empty($result)) {
                return ['variation_id' => $result->variation_id,
                    'qty' => $qty,
                    'success' => true,
                ];
            } else {
                $error_msg = trans('lang_v1.sku_not_match', ['sku' => $sku]);
            }
        } else {
            $error_msg = trans('lang_v1.prefix_did_not_match');
        }

        return [
            'success' => false,
            'msg' => $error_msg,
        ];
    }



    /**
     * Retrieves products list without variation list
     *
     * @param  string  $q
     * @param  bool  $check_qty
     * @return JSON
     */
    public function getProductsWithoutVariations()
    {
        if (request()->ajax()) {
            $term = request()->input('term', '');
            //$location_id = request()->input('location_id', '');

            //$check_qty = request()->input('check_qty', false);

            $business_id = request()->session()->get('user.business_id');

            $products = Product::join('variations', 'products.id', '=', 'variations.product_id')
                ->where('products.business_id', $business_id)
                ->where('products.type', '!=', 'modifier');

            //Include search
            if (! empty($term)) {
                $products->where(function ($query) use ($term) {
                    $query->where('products.name', 'like', '%'.$term.'%');
                    $query->orWhere('sku', 'like', '%'.$term.'%');
                    $query->orWhere('sub_sku', 'like', '%'.$term.'%');
                });
            }

            //Include check for quantity
            // if($check_qty){
            //     $products->where('VLD.qty_available', '>', 0);
            // }

            $products = $products->groupBy('products.id')
                ->select(
                    'products.id as product_id',
                    'products.name',
                    'products.type',
                    'products.enable_stock',
                    'products.sku',
                    'products.id as id',
                    DB::raw('CONCAT(products.name, " - ", products.sku) as text')
                )
                    ->orderBy('products.name')
                    ->get();

            return json_encode($products);
        }
    }

    /**
     * Checks if product sku already exists.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function checkProductSku(Request $request)
    {
        $business_id = $request->session()->get('user.business_id');
        $sku = $request->input('sku');
        $product_id = $request->input('product_id');

        //check in products table
        $query = Product::where('business_id', $business_id)
                        ->where('sku', $sku);
        if (! empty($product_id)) {
            $query->where('id', '!=', $product_id);
        }
        $count = $query->count();

        //check in variation table if $count = 0
        if ($count == 0) {
            $query2 = Variation::where('sub_sku', $sku)
                            ->join('products', 'variations.product_id', '=', 'products.id')
                            ->where('business_id', $business_id);

            if (! empty($product_id)) {
                $query2->where('product_id', '!=', $product_id);
            }

            if (! empty($request->input('variation_id'))) {
                $query2->where('variations.id', '!=', $request->input('variation_id'));
            }
            $count = $query2->count();
        }
        if ($count == 0) {
            echo 'true';
            exit;
        } else {
            echo 'false';
            exit;
        }
    }

    /**
     * Validates multiple variation skus
     */
    public function validateVaritionSkus(Request $request)
    {
        $business_id = $request->session()->get('user.business_id');
        $all_skus = $request->input('skus');

        $skus = [];
        foreach ($all_skus as $key => $value) {
            $skus[] = $value['sku'];
        }

        //check product table is sku present
        $product = Product::where('business_id', $business_id)
                        ->whereIn('sku', $skus)
                        ->first();

        if (! empty($product)) {
            return ['success' => 0, 'sku' => $product->sku];
        }

        foreach ($all_skus as $key => $value) {
            $query = Variation::where('sub_sku', $value['sku'])
                            ->join('products', 'variations.product_id', '=', 'products.id')
                            ->where('business_id', $business_id);

            if (! empty($value['variation_id'])) {
                $query->where('variations.id', '!=', $value['variation_id']);
            }
            $variation = $query->first();

            if (! empty($variation)) {
                return ['success' => 0, 'sku' => $variation->sub_sku];
            }
        }

        return ['success' => 1];
    }

    /**
     * Loads quick add product modal.
     *
     * @return \Illuminate\Http\Response
     */
    public function quickAdd()
    {
        if (! auth()->user()->can('product.create')) {
            abort(403, 'Unauthorized action.');
        }

        $product_name = ! empty(request()->input('product_name')) ? request()->input('product_name') : '';

        $product_for = ! empty(request()->input('product_for')) ? request()->input('product_for') : null;

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $brands = Brands::forDropdown($business_id);
        $units = Unit::forDropdown($business_id, true);

        $tax_dropdown = TaxRate::forBusinessDropdown($business_id, true, true);
        $taxes = $tax_dropdown['tax_rates'];
        $tax_attributes = $tax_dropdown['attributes'];

        $barcode_types = $this->barcode_types;

        $default_profit_percent = Business::where('id', $business_id)->value('default_profit_percent');

        $locations = BusinessLocation::forDropdown($business_id);

        $enable_expiry = request()->session()->get('business.enable_product_expiry');
        $enable_lot = request()->session()->get('business.enable_lot_number');

        $module_form_parts = $this->moduleUtil->getModuleData('product_form_part');

        //Get all business locations
        $business_locations = BusinessLocation::forDropdown($business_id);

        $common_settings = session()->get('business.common_settings');
        $warranties = Warranty::forDropdown($business_id);
        $use_multi_unit = 0;

        return view('product.partials.quick_add_product')->with(
            compact(
                'categories', 'brands', 'units', 
                'taxes', 'use_multi_unit', 
                'barcode_types', 
                'default_profit_percent', 
                'tax_attributes', 'product_name', 
                'locations', 'product_for', 'enable_expiry', 
                'enable_lot', 'module_form_parts', 
                'business_locations', 'common_settings', 
                'warranties'
            )
        );

    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function saveQuickProduct(Request $request)
    {
        if (! auth()->user()->can('product.create')) {
            abort(403, 'Unauthorized action.');
        }


        try {
            $business_id = $request->session()->get('user.business_id');
            $form_fields = [
                'name', 'brand_id','unit_id', 
                'category_id', 'tax', 'barcode_type', 
                'tax_type', 'sku',
                'alert_quantity', 'type', 
                'sub_unit_ids', 
                'sub_category_id', 
                'sub_category_id', 
                'weight', 
                'product_description', 
                'product_custom_field1', 
                'product_custom_field2', 
                'product_custom_field3', 
                'product_custom_field4', 
                'product_custom_field5', 
                'product_custom_field6', 
                'product_custom_field7', 
                'product_custom_field8', 
                'product_custom_field9', 
                'product_custom_field10', 
                'product_custom_field11', 
                'product_custom_field12', 
                'product_custom_field13', 
                'product_custom_field14', 
                'product_custom_field15', 
                'product_custom_field16', 'product_custom_field17', 'product_custom_field18', 'product_custom_field19', 
                'product_custom_field20'
            ];

            $module_form_fields = $this->moduleUtil->getModuleData('product_form_fields');
            if (! empty($module_form_fields)) {
                foreach ($module_form_fields as $key => $value) {
                    if (! empty($value) && is_array($value)) {
                        $form_fields = array_merge($form_fields, $value);
                    }
                }
            }
            $product_details = $request->only($form_fields);

            $product_details['type'] = empty($product_details['type']) ? 'single' : $product_details['type'];
            $product_details['business_id'] = $business_id;
            $product_details['created_by'] = $request->session()->get('user.id');
            if (! empty($request->input('enable_stock')) && $request->input('enable_stock') == 1) {
                $product_details['enable_stock'] = 1;
                //TODO: Save total qty
                //$product_details['total_qty_available'] = 0;
            }
            if (! empty($request->input('not_for_selling')) && $request->input('not_for_selling') == 1) {
                $product_details['not_for_selling'] = 1;
            }
            if (empty($product_details['sku'])) {
                $product_details['sku'] = ' ';
            }


            $common_settings = session()->get('business.common_settings');
     

            if (!empty($request->input('enable_serial_number')) && $request->input('enable_serial_number') == 1) {
                $product_details['enable_serial_number'] = $request->input('enable_serial_number');
            }

            if(!empty($common_settings['enable_product_multi_unit'])){
                if (!empty($request->input('use_multi_unit')) && $request->input('use_multi_unit') == 1) {
                    $product_details['use_multi_unit'] = $request->input('use_multi_unit');
                    $product_details['first_conversion_unit_id'] = $request->input('first_conversion_unit_id');
                    $product_details['first_conversion_unit_rate'] = $request->input('first_conversion_unit_rate');
                    $product_details['second_conversion_unit_id'] = $request->input('second_conversion_unit_id');
                    $product_details['second_conversion_unit_rate'] = $request->input('second_conversion_unit_rate');
                }
            }


            if (! empty($product_details['alert_quantity'])) {
                $product_details['alert_quantity'] = $this->productUtil->num_uf($product_details['alert_quantity']);
            }

            $expiry_enabled = $request->session()->get('business.enable_product_expiry');
            if (! empty($request->input('expiry_period_type')) && ! empty($request->input('expiry_period')) && ! empty($expiry_enabled)) {
                $product_details['expiry_period_type'] = $request->input('expiry_period_type');
                $product_details['expiry_period'] = $this->productUtil->num_uf($request->input('expiry_period'));
            }

            if (! empty($request->input('enable_sr_no')) && $request->input('enable_sr_no') == 1) {
                $product_details['enable_sr_no'] = 1;
            }

            $product_details['warranty_id'] = ! empty($request->input('warranty_id')) ? $request->input('warranty_id') : null;

            DB::beginTransaction();

            $product = Product::create($product_details);
            event(new ProductsCreatedOrModified($product_details, 'added'));

            if (empty(trim($request->input('sku')))) {
                $sku = $this->productUtil->generateProductSku($product->id);
                $product->sku = $sku;
                $product->save();
            }




            $fu_single_dpp = $request->input('fu_single_dpp', null);
            $fu_single_dpp_inc_tax = $request->input('fu_single_dpp_inc_tax', null);
            $fu_profit_percent = $request->input('fu_profit_percent', null);
            $fu_single_dsp = $request->input('fu_single_dsp', null);
            $fu_single_dsp_inc_tax = $request->input('fu_single_dsp_inc_tax', null);
            $fu_sub_sku = $request->input('fu_sub_sku', null);
            $su_single_dpp = $request->input('su_single_dpp', null);
            $su_single_dpp_inc_tax = $request->input('su_single_dpp_inc_tax', null);
            $su_profit_percent = $request->input('su_profit_percent', null);
            $su_single_dsp = $request->input('su_single_dsp', null);
            $su_single_dsp_inc_tax = $request->input('su_single_dsp_inc_tax', null);
            $su_sub_sku = $request->input('su_sub_sku', null);
            


            $this->productUtil->createSingleProductVariation(
                $product->id, 
                $product->sku,
                $request->input('single_dpp'), 
                $request->input('single_dpp_inc_tax'), 
                $request->input('profit_percent'), 
                $request->input('single_dsp'), 
                $request->input('single_dsp_inc_tax'),
                [],
                $fu_single_dpp,
                $fu_single_dpp_inc_tax,
                $fu_profit_percent,
                $fu_single_dsp,
                $fu_single_dsp_inc_tax,
                $fu_sub_sku,
                $su_single_dpp,
                $su_single_dpp_inc_tax,
                $su_profit_percent,
                $su_single_dsp,
                $su_single_dsp_inc_tax,
                $su_sub_sku
            );



            if ($product->enable_stock == 1 && ! empty($request->input('opening_stock'))) {
                $user_id = $request->session()->get('user.id');

                $transaction_date = $request->session()->get('financial_year.start');
                $transaction_date = \Carbon::createFromFormat('Y-m-d', $transaction_date)->toDateTimeString();

                $this->productUtil->addSingleProductOpeningStock($business_id, $product, $request->input('opening_stock'), $transaction_date, $user_id);
            }

            //Add product locations
            $product_locations = $request->input('product_locations');
            if (! empty($product_locations)) {
                // Validate location permissions
                $this->validateLocationPermissions($product_locations);
                $product->product_locations()->sync($product_locations);
            }

            DB::commit();

            $output = [
                'success' => 1,
                'msg' => __('product.product_added_success'),
                'product' => $product,
                'variation' => $product->variations->first(),
                'locations' => $product_locations,
            ];


        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return $output;
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function view($id)
    {
        if (! auth()->user()->can('product.view')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = request()->session()->get('user.business_id');

            $product = Product::where('business_id', $business_id)
                        ->with([
                            'brand', 
                            'unit', 
                            'category', 
                            'sub_category', 
                            'product_tax', 
                            'variations', 
                            'variations.product_variation', 
                            'variations.group_prices', 
                            'variations.media', 
                            'product_locations', 
                            'warranty', 
                            'media'
                        ])
                        ->findOrFail($id);

            // Check location access
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $product_location_ids = $product->product_locations->pluck('id')->toArray();
                $has_access = empty($product_location_ids) || !empty(array_intersect($product_location_ids, $permitted_locations));
                
                if (!$has_access) {
                    abort(403, 'Unauthorized: You do not have access to this product\'s locations.');
                }
            }

            $premade_details = PremadeLine::join('products AS p', 'premade_lines.premade_product_id', '=', 'p.id')
                        ->join('products', 'premade_lines.product_id', '=', 'products.id')
                        ->join('variations AS variations', 'premade_lines.variation_id', '=', 'variations.id')
                        ->leftjoin('units', 'units.id', '=', 'p.unit_id')
                        ->leftjoin('units as u', 'p.secondary_unit_id', '=', 'u.id')
                        
                        ->leftjoin('units as use_unit', 'premade_lines.sub_unit_id', '=', 'use_unit.id')
                        ->where('premade_lines.premade_product_id', $id)
            
                        ->select(
                            
                            'products.id as product_id',
                            'products.name as product_name',
                            
                            'variations.default_sell_price',
                            'variations.dpp_inc_tax as default_purchase_price',
                            'variations.sell_price_inc_tax',
                            'variations.profit_percent',
                            'variations.id as variation_id',
                            'variations.combo_variations',  //Used in combo products

                            'use_unit.short_name as unit',
                            'use_unit.allow_decimal as unit_allow_decimal',
                            'use_unit.id as unit_id',
                            
                            'premade_lines.id',
                            'premade_lines.business_id',
                            'premade_lines.product_id',
                            'premade_lines.variation_id',
                            'premade_lines.quantity as quantity_ordered',
                            'premade_lines.premade_price',
                            'premade_lines.premade_price_inc_tax',
                            'premade_lines.premade_product_id',
                            'premade_lines.sub_unit_id',
                            'premade_lines.sell_unit_id',
                     
                        )
                        ->get();


            $price_groups = SellingPriceGroup::where('business_id', $business_id)->active()->pluck('name', 'id');

            $allowed_group_prices = [];
            foreach ($price_groups as $key => $value) {
                if (auth()->user()->can('selling_price_group.'.$key)) {
                    $allowed_group_prices[$key] = $value;
                }
            }

            $group_price_details = [];

            foreach ($product->variations as $variation) {
                foreach ($variation->group_prices as $group_price) {
                    $group_price_details[$variation->id][$group_price->price_group_id] = ['price' => $group_price->price_inc_tax, 'price_type' => $group_price->price_type, 'calculated_price' => $group_price->calculated_price];
                }
            }

            $rack_details = $this->productUtil->getRackDetails($business_id, $id, true);

            $combo_variations = [];
            if ($product->type == self::PRODUCT_TYPE_COMBO) {
                $combo_variations = $this->productUtil->__getComboProductDetails($product['variations'][0]->combo_variations, $business_id);
            }

            return view('product.view-modal')->with(compact(
                'product',
                'premade_details',
                'rack_details',
                'allowed_group_prices',
                'group_price_details',
                'combo_variations'
            ));
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong')
                ], 500);
            }
            
            return redirect()->back()->with('status', [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ]);
        }
    }

    /**
     * Mass deletes products.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function massDestroy(Request $request)
    {
        if (! auth()->user()->can('product.delete')) {
            abort(403, 'Unauthorized action.');
        }
        try {
            $purchase_exist = false;

            if (!empty($request->input('selected_rows'))) {
                $business_id = $request->session()->get('user.business_id');

                $selected_rows = explode(',', $request->input('selected_rows'));

                $products = Product::where('business_id', $business_id)
                                    ->whereIn('id', $selected_rows)
                                    ->with(['purchase_lines', 'variations'])
                                    ->get();

                $deletable_products = [];

                $is_mfg_installed = $this->moduleUtil->isModuleInstalled('Manufacturing');

                DB::beginTransaction();



            

                foreach ($products as $product) {


              
                    $can_be_deleted = true;

                    //Check if product is added as an ingredient of any recipe
                    if ($is_mfg_installed) {
                        $variation_ids = $product->variations->pluck('id');
                        $exists_as_ingredient = \Modules\Manufacturing\Entities\MfgRecipeIngredient::whereIn('variation_id', $variation_ids)->exists();
                        $can_be_deleted = !$exists_as_ingredient;
                    }


                    // return $can_be_deleted;

                    // die();


                    //Delete if no purchase found
                    if (empty($product->purchase_lines->toArray()) && $can_be_deleted) {
                        //Delete variation location details
                        VariationLocationDetails::where('product_id', $product->id)->delete();
                        $product->delete();
                        event(new ProductsCreatedOrModified($product, 'Deleted'));
                    } else {
                        $purchase_exist = true;
                    }


                }

                DB::commit();
            }

            if (!$purchase_exist) {
                $output = [
                    'success' => 1,
                    'msg' => __('lang_v1.deleted_success'),
                ];
            } else {
                $output = ['success' => 0,
                    'msg' => __('lang_v1.products_could_not_be_deleted'),
                ];
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->back()->with(['status' => $output]);
    }


    
    /**
     * Shows form to add selling price group prices for a product.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function addSellingPrices($id)
    {
        if (! auth()->user()->can('product.create')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $product = Product::where('business_id', $business_id)
                    ->with(['variations', 'variations.group_prices', 'variations.product_variation'])
                            ->findOrFail($id);

        $price_groups = SellingPriceGroup::where('business_id', $business_id)
                                            ->active()
                                            ->get();
        $variation_prices = [];
        foreach ($product->variations as $variation) {
            foreach ($variation->group_prices as $group_price) {
                $variation_prices[$variation->id][$group_price->price_group_id] = ['price' => $group_price->price_inc_tax, 'price_type' => $group_price->price_type];
            }
        }

        return view('product.add-selling-prices')->with(compact('product', 'price_groups', 'variation_prices'));
    }

    /**
     * Saves selling price group prices for a product.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function saveSellingPrices(Request $request)
    {
        if (! auth()->user()->can('product.create')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = $request->session()->get('user.business_id');
            $product = Product::where('business_id', $business_id)
                            ->with(['variations'])
                            ->findOrFail($request->input('product_id'));
            DB::beginTransaction();
            foreach ($product->variations as $variation) {
                $variation_group_prices = [];
                foreach ($request->input('group_prices') as $key => $value) {
                    if (isset($value[$variation->id])) {
                        $variation_group_price =
                        VariationGroupPrice::where('variation_id', $variation->id)
                                            ->where('price_group_id', $key)
                                            ->first();
                        if (empty($variation_group_price)) {
                            $variation_group_price = new VariationGroupPrice([
                                'variation_id' => $variation->id,
                                'price_group_id' => $key,
                            ]);
                        }

                        $variation_group_price->price_inc_tax = $this->productUtil->num_uf($value[$variation->id]['price']);
                        $variation_group_price->price_type = $value[$variation->id]['price_type'];
                        $variation_group_prices[] = $variation_group_price;
                    }
                }

                if (! empty($variation_group_prices)) {
                    $variation->group_prices()->saveMany($variation_group_prices);
                }
            }
            //Update product updated_at timestamp
            $product->touch();

            DB::commit();
            $output = ['success' => 1,
                'msg' => __('lang_v1.updated_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        if ($request->input('submit_type') == 'submit_n_add_opening_stock') {
            return redirect()->action([\App\Http\Controllers\OpeningStockController::class, 'add'],
                ['product_id' => $product->id]
            );
        } elseif ($request->input('submit_type') == 'save_n_add_another') {
            return redirect()->action([\App\Http\Controllers\ProductController::class, 'create']
            )->with('status', $output);
        }

        return redirect('products')->with('status', $output);
    }

    public function viewGroupPrice($id)
    {
        if (! auth()->user()->can('product.view')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        $product = Product::where('business_id', $business_id)
                            ->where('id', $id)
                            ->with(['variations', 'variations.product_variation', 'variations.group_prices'])
                            ->first();

        $price_groups = SellingPriceGroup::where('business_id', $business_id)->active()->pluck('name', 'id');

        $allowed_group_prices = [];
        foreach ($price_groups as $key => $value) {
            if (auth()->user()->can('selling_price_group.'.$key)) {
                $allowed_group_prices[$key] = $value;
            }
        }

        $group_price_details = [];

        foreach ($product->variations as $variation) {
            foreach ($variation->group_prices as $group_price) {
                $group_price_details[$variation->id][$group_price->price_group_id] = $group_price->price_inc_tax;
            }
        }

        return view('product.view-product-group-prices')->with(compact('product', 'allowed_group_prices', 'group_price_details'));
    }

    /**
     * Mass deactivates products.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function massDeactivate(Request $request)
    {
        if (! auth()->user()->can('product.update')) {
            abort(403, 'Unauthorized action.');
        }
        try {
            if (! empty($request->input('selected_products'))) {
                $business_id = $request->session()->get('user.business_id');

                $selected_products = explode(',', $request->input('selected_products'));

                DB::beginTransaction();

                $products = Product::where('business_id', $business_id)
                                    ->whereIn('id', $selected_products)
                                    ->update(['is_inactive' => 1]);

                DB::commit();
            }

            $output = ['success' => 1,
                'msg' => __('lang_v1.products_deactivated_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return $output;
    }

    /**
     * Activates the specified resource from storage.
     *
     * @param  \App\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function activate($id)
    {
        if (! auth()->user()->can('product.update')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            try {
                $business_id = request()->session()->get('user.business_id');
                $product = Product::where('id', $id)
                                ->where('business_id', $business_id)
                                ->update(['is_inactive' => 0]);

                $output = ['success' => true,
                    'msg' => __('lang_v1.updated_success'),
                ];
            } catch (\Exception $e) {
                Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = ['success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    /**
     * Deletes a media file from storage and database.
     *
     * @param  int  $media_id
     * @return json
     */
    public function deleteMedia($media_id)
    {
        if (! auth()->user()->can('product.update')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            try {
                $business_id = request()->session()->get('user.business_id');

                Media::deleteMedia($business_id, $media_id);

                $output = ['success' => true,
                    'msg' => __('lang_v1.file_deleted_successfully'),
                ];
            } catch (\Exception $e) {
                Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = ['success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    public function getProductsApi($id = null)
    {
        try {
            $api_token = request()->header('API-TOKEN');
            $filter_string = request()->header('FILTERS');
            $order_by = request()->header('ORDER-BY');

            parse_str($filter_string, $filters);

            $api_settings = $this->moduleUtil->getApiSettings($api_token);

            $limit = ! empty(request()->input('limit')) ? request()->input('limit') : 10;

            $location_id = $api_settings->location_id;

            $query = Product::where('business_id', $api_settings->business_id)
                            ->active()
                            ->with(['brand', 'unit', 'category', 'sub_category',
                                'product_variations', 'product_variations.variations', 'product_variations.variations.media',
                                'product_variations.variations.variation_location_details' => function ($q) use ($location_id) {
                                    $q->where('location_id', $location_id);
                                }, ]);

            if (! empty($filters['categories'])) {
                $query->whereIn('category_id', $filters['categories']);
            }

            if (! empty($filters['brands'])) {
                $query->whereIn('brand_id', $filters['brands']);
            }

            if (! empty($filters['category'])) {
                $query->where('category_id', $filters['category']);
            }

            if (! empty($filters['sub_category'])) {
                $query->where('sub_category_id', $filters['sub_category']);
            }

            if ($order_by == 'name') {
                $query->orderBy('name', 'asc');
            } elseif ($order_by == 'date') {
                $query->orderBy('created_at', 'desc');
            }

            if (empty($id)) {
                $products = $query->paginate($limit);
            } else {
                $products = $query->find($id);
            }
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            return $this->respondWentWrong($e);
        }

        return $this->respond($products);
    }

    public function getVariationsApi()
    {
        try {
            $api_token = request()->header('API-TOKEN');
            $variations_string = request()->header('VARIATIONS');

            if (is_numeric($variations_string)) {
                $variation_ids = intval($variations_string);
            } else {
                parse_str($variations_string, $variation_ids);
            }

            $api_settings = $this->moduleUtil->getApiSettings($api_token);
            $location_id = $api_settings->location_id;
            $business_id = $api_settings->business_id;

            $query = Variation::with([
                'product_variation',
                'product' => function ($q) use ($business_id) {
                    $q->where('business_id', $business_id);
                },
                'product.unit',
                'variation_location_details' => function ($q) use ($location_id) {
                    $q->where('location_id', $location_id);
                },
            ]);

            $variations = is_array($variation_ids) ? $query->whereIn('id', $variation_ids)->get() : $query->where('id', $variation_ids)->first();
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            return $this->respondWentWrong($e);
        }

        return $this->respond($variations);
    }

    /**
     * Shows form to edit multiple products at once.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function bulkEdit(Request $request)
    {
        if (! auth()->user()->can('product.update')) {
            abort(403, 'Unauthorized action.');
        }

        $selected_products_string = $request->input('selected_products');
        if (! empty($selected_products_string)) {
            $selected_products = explode(',', $selected_products_string);
            $business_id = $request->session()->get('user.business_id');

            $products = Product::where('business_id', $business_id)
                                ->whereIn('id', $selected_products)
                                ->with(['variations', 'variations.product_variation', 'variations.group_prices', 'product_locations'])
                                ->get();

            $all_categories = Category::catAndSubCategories($business_id);

            $categories = [];
            $sub_categories = [];
            foreach ($all_categories as $category) {
                $categories[$category['id']] = $category['name'];

                if (! empty($category['sub_categories'])) {
                    foreach ($category['sub_categories'] as $sub_category) {
                        $sub_categories[$category['id']][$sub_category['id']] = $sub_category['name'];
                    }
                }
            }

            $brands = Brands::forDropdown($business_id);

            $tax_dropdown = TaxRate::forBusinessDropdown($business_id, true, true);
            $taxes = $tax_dropdown['tax_rates'];
            $tax_attributes = $tax_dropdown['attributes'];

            $price_groups = SellingPriceGroup::where('business_id', $business_id)->active()->pluck('name', 'id');
            $business_locations = BusinessLocation::forDropdown($business_id);

            return view('product.bulk-edit')->with(compact(
                'products',
                'categories',
                'brands',
                'taxes',
                'tax_attributes',
                'sub_categories',
                'price_groups',
                'business_locations'
            ));
        }
    }

    /**
     * Updates multiple products at once.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function bulkUpdate(Request $request)
    {
        if (! auth()->user()->can('product.update')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $products = $request->input('products');
            $business_id = $request->session()->get('user.business_id');

            DB::beginTransaction();
            foreach ($products as $id => $product_data) {
                $update_data = [
                    'category_id' => $product_data['category_id'],
                    'sub_category_id' => $product_data['sub_category_id'],
                    'brand_id' => $product_data['brand_id'],
                    'tax' => $product_data['tax'],
                ];

                //Update product
                $product = Product::where('business_id', $business_id)
                                ->findOrFail($id);

                $product->update($update_data);

                //Add product locations
                $product_locations = ! empty($product_data['product_locations']) ?
                                    $product_data['product_locations'] : [];
                
                // Validate location permissions
                if (!empty($product_locations)) {
                    $this->validateLocationPermissions($product_locations);
                }
                
                $product->product_locations()->sync($product_locations);

                $variations_data = [];

                //Format variations data
                foreach ($product_data['variations'] as $key => $value) {
                    $variation = Variation::where('product_id', $product->id)->findOrFail($key);
                    $variation->default_purchase_price = $this->productUtil->num_uf($value['default_purchase_price']);
                    $variation->dpp_inc_tax = $this->productUtil->num_uf($value['dpp_inc_tax']);
                    $variation->profit_percent = $this->productUtil->num_uf($value['profit_percent']);
                    $variation->default_sell_price = $this->productUtil->num_uf($value['default_sell_price']);
                    $variation->sell_price_inc_tax = $this->productUtil->num_uf($value['sell_price_inc_tax']);
                    $variations_data[] = $variation;

                    //Update price groups
                    if (! empty($value['group_prices'])) {
                        foreach ($value['group_prices'] as $k => $v) {
                            VariationGroupPrice::updateOrCreate(
                                ['price_group_id' => $k, 'variation_id' => $variation->id],
                                ['price_inc_tax' => $this->productUtil->num_uf($v)]
                            );
                        }
                    }
                }
                $product->variations()->saveMany($variations_data);
            }
            DB::commit();

            $output = ['success' => 1,
                'msg' => __('lang_v1.updated_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return redirect('products')->with('status', $output);
    }

    /**
     * Adds product row to edit in bulk edit product form
     *
     * @param  int  $product_id
     * @return \Illuminate\Http\Response
     */
    public function getProductToEdit($product_id)
    {
        if (! auth()->user()->can('product.update')) {
            abort(403, 'Unauthorized action.');
        }
        $business_id = request()->session()->get('user.business_id');

        $product = Product::where('business_id', $business_id)
                            ->with(['variations', 'variations.product_variation', 'variations.group_prices'])
                            ->findOrFail($product_id);
        $all_categories = Category::catAndSubCategories($business_id);

        $categories = [];
        $sub_categories = [];
        foreach ($all_categories as $category) {
            $categories[$category['id']] = $category['name'];

            if (! empty($category['sub_categories'])) {
                foreach ($category['sub_categories'] as $sub_category) {
                    $sub_categories[$category['id']][$sub_category['id']] = $sub_category['name'];
                }
            }
        }

        $brands = Brands::forDropdown($business_id);

        $tax_dropdown = TaxRate::forBusinessDropdown($business_id, true, true);
        $taxes = $tax_dropdown['tax_rates'];
        $tax_attributes = $tax_dropdown['attributes'];

        $price_groups = SellingPriceGroup::where('business_id', $business_id)->active()->pluck('name', 'id');
        $business_locations = BusinessLocation::forDropdown($business_id);

        return view('product.partials.bulk_edit_product_row')->with(compact(
            'product',
            'categories',
            'brands',
            'taxes',
            'tax_attributes',
            'sub_categories',
            'price_groups',
            'business_locations'
        ));
    }

    /**
     * Gets the sub units for the given unit.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $unit_id
     * @return \Illuminate\Http\Response
     */
    public function getSubUnits(Request $request)
    {
        if (! empty($request->input('unit_id'))) {
            $unit_id = $request->input('unit_id');
            $business_id = $request->session()->get('user.business_id');
            $sub_units = $this->productUtil->getSubUnits($business_id, $unit_id, true);

            //$html = '<option value="">' . __('lang_v1.all') . '</option>';
            $html = '';
            if (! empty($sub_units)) {
                foreach ($sub_units as $id => $sub_unit) {
                    $html .= '<option value="'.$id.'">'.$sub_unit['name'].'</option>';
                }
            }

            return $html;
        }
    }




    /**
     * Gets the sub units for the given unit.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $unit_id
     * @return \Illuminate\Http\Response
    */
    public function getOthersUnits(Request $request){

        if (!empty($request->input('unit_id'))) {

            $unit_id = $request->input('unit_id');
            $first_conversion_unit_id = $request->input('first_conversion_unit_id');
            $business_id = $request->session()->get('user.business_id');

            $query = Unit::where('business_id', $business_id)
                            ->whereNotIn('id', [$unit_id]);
                            if(!empty($first_conversion_unit_id)){
                                $query->whereNotIn('id', [$first_conversion_unit_id]);
                            }
            $units = $query->get();

            $html = '<option value="">'.__('messages.please_select').'</option>';
            if (!empty($units)) {
                foreach ($units as $key => $unit) {
                    $html .= '<option value="'.$unit->id.'">'.$unit->actual_name.'</option>';
                }
            }

            return $html;
        }
    }








    public function updateProductLocation(Request $request)
    {
        if (! auth()->user()->can('product.update')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $selected_products = $request->input('products');
            $update_type = $request->input('update_type');
            $location_ids = $request->input('product_location');

            $business_id = $request->session()->get('user.business_id');

            // Validate update_type
            if (!in_array($update_type, [self::LOCATION_UPDATE_ADD, self::LOCATION_UPDATE_REMOVE])) {
                return response()->json([
                    'success' => 0,
                    'msg' => __('lang_v1.invalid_update_type')
                ], 400);
            }

            // Validate location permissions
            if (!empty($location_ids)) {
                $this->validateLocationPermissions($location_ids);
            }

            $product_ids = explode(',', $selected_products);

            $products = Product::where('business_id', $business_id)
                                ->whereIn('id', $product_ids)
                                ->with(['product_locations'])
                                ->get();
            DB::beginTransaction();
            foreach ($products as $product) {
                $product_locations = $product->product_locations->pluck('id')->toArray();

                if ($update_type == self::LOCATION_UPDATE_ADD) {
                    $product_locations = array_unique(array_merge($location_ids, $product_locations));
                    $product->product_locations()->sync($product_locations);
                } elseif ($update_type == self::LOCATION_UPDATE_REMOVE) {
                    foreach ($product_locations as $key => $value) {
                        if (in_array($value, $location_ids)) {
                            unset($product_locations[$key]);
                        }
                    }
                    $product->product_locations()->sync($product_locations);
                }
            }
            DB::commit();
            $output = ['success' => 1,
                'msg' => __('lang_v1.updated_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return $output;
    }


    public function productStockHistory($id){

        if (!auth()->user()->can('product.view')) {
            abort(403, 'Unauthorized action.');
        }
        
        $business_id = request()->session()->get('user.business_id');


        $product_q = Product::where('business_id', $business_id)
                            ->with([
                                'variations', 
                                'variations.product_variation',
                                'first_conversion_unit',
                                'second_conversion_unit'
                            ]);
                            if(!empty($product_id)){
                                $product_q->where('id', $product_id);
                            }else{
                                $product_q->where('id',$id);
                            }

        $product = $product_q->first();

        if (request()->ajax()) {
   

            //for ajax call $id is variation id else it is product id
            $stock_details = $this->productUtil->getVariationStockDetails($business_id, $id, request()->input('location_id'));
            $stock_history = $this->productUtil->getVariationStockHistory($business_id, $id, request()->input('location_id'));

            //if mismach found update stock in variation location details
            if (isset($stock_history[0]) && (float) $stock_details['current_stock'] != (float) $stock_history[0]['stock']) {
                VariationLocationDetails::where('variation_id', $id)->where('location_id', request()->input('location_id'))->update(['qty_available' => $stock_history[0]['stock']]);
                $stock_details['current_stock'] = $stock_history[0]['stock'];
            }

            return view('product.stock_history_details')->with(compact('stock_details','stock_history'));
        }



        //Get all business locations
        $business_locations = BusinessLocation::forDropdown($business_id);

        return view('product.stock_history')->with(compact('product', 'business_locations'));
    }








    /**
     * Toggle WooComerce sync
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function toggleWooCommerceSync(Request $request)
    {
        try {
            $selected_products = $request->input('woocommerce_products_sync');
            $woocommerce_disable_sync = $request->input('woocommerce_disable_sync');

            $business_id = $request->session()->get('user.business_id');
            $product_ids = explode(',', $selected_products);

            DB::beginTransaction();
            if ($this->moduleUtil->isModuleInstalled('Woocommerce')) {
                Product::where('business_id', $business_id)
                        ->whereIn('id', $product_ids)
                        ->update(['woocommerce_disable_sync' => $woocommerce_disable_sync]);
            }
            DB::commit();
            $output = [
                'success' => 1,
                'msg' => __('lang_v1.success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = [
                'success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return $output;
    }

    /**
     * Function to download all products in xlsx format
     */
    public function downloadExcel()
    {
        $is_admin = $this->productUtil->is_admin(auth()->user());
        if (! $is_admin) {
            abort(403, 'Unauthorized action.');
        }

        $filename = 'products-export-'.\Carbon::now()->format('Y-m-d').'.xlsx';

        return Excel::download(new ProductsExport, $filename);
    }
}
