<?php

namespace App\Http\Controllers;

use App\System;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Validation\ValidationException;

class LicenseManagementController extends Controller
{
    /**
     * Display license management page
     */
    public function index()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        // Get all module licenses
        $licenses = $this->getAllModuleLicenses();

        return view('license_management.index', compact('licenses'));
    }

    /**
     * Generate license for a module
     */
    public function generateLicense(Request $request)
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'module_name' => 'required|string',
            'username' => 'required|string',
            'email' => 'required|email',
            'license_type' => 'required|in:single,multi,unlimited',
            'expiry_date' => 'nullable|date',
        ]);

        try {
            // Generate license code
            $license_code = $this->generateLicenseCode(
                $request->module_name,
                $request->username,
                $request->email,
                $request->license_type
            );

            // Store license in database
            // Status starts as 'inactive' until module is installed
            $license_data = [
                'module_name' => $request->module_name,
                'username' => $request->username,
                'email' => $request->email,
                'license_code' => $license_code,
                'license_type' => $request->license_type,
                'expiry_date' => $request->expiry_date,
                'status' => 'inactive', // Will change to 'active' when module is installed
                'created_at' => now(),
                'updated_at' => now(),
            ];

            // Store in licenses table if exists, otherwise use System properties
            if (Schema::hasTable('module_licenses')) {
                DB::table('module_licenses')->insert($license_data);
            } else {
                // Store in System properties as fallback
                System::addProperty('license_' . $request->module_name . '_' . $license_code, json_encode($license_data));
            }

            $output = [
                'success' => true,
                'msg' => 'License generated successfully',
                'license_code' => $license_code,
                'license_data' => $license_data
            ];

            return redirect()->back()->with('status', $output);
        } catch (\Exception $e) {
            \Log::error('License generation error: ' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'Failed to generate license: ' . $e->getMessage()
            ];

            return redirect()->back()->with('status', $output);
        }
    }

    /**
     * Validate license
     */
    public function validateLicense(Request $request)
    {
        $request->validate([
            'module_name' => 'required|string',
            'license_code' => 'required|string',
            'username' => 'required|string',
        ]);

        $is_valid = $this->checkLicenseValidity(
            $request->module_name,
            $request->license_code,
            $request->username
        );

        return response()->json([
            'valid' => $is_valid,
            'message' => $is_valid ? 'License is valid' : 'License is invalid'
        ]);
    }

    /**
     * Get all module licenses with installation status
     */
    private function getAllModuleLicenses()
    {
        $licenses = [];

        // Check if licenses table exists
        if (Schema::hasTable('module_licenses')) {
            $licenses = DB::table('module_licenses')
                ->orderBy('created_at', 'desc')
                ->get();

            // Update status based on module installation state
            foreach ($licenses as $license) {
                $is_installed = $this->isModuleInstalled($license->module_name, $license->license_code, $license->username);
                
                // If module is installed, status should be 'active'
                // If module is not installed, status should be 'inactive' (unless expired/revoked)
                if ($is_installed) {
                    if ($license->status !== 'active' && $license->status !== 'expired' && $license->status !== 'revoked') {
                        // Update to active if module is installed
                        DB::table('module_licenses')
                            ->where('id', $license->id)
                            ->update(['status' => 'active', 'updated_at' => now()]);
                        $license->status = 'active';
                    }
                } else {
                    // Module not installed - set to inactive (unless already expired/revoked)
                    if ($license->status === 'active') {
                        DB::table('module_licenses')
                            ->where('id', $license->id)
                            ->update(['status' => 'inactive', 'updated_at' => now()]);
                        $license->status = 'inactive';
                    }
                }
            }

            $licenses = $licenses->toArray();
        } else {
            // Get from System properties
            $system_properties = DB::table('system')
                ->where('key', 'like', 'license_%')
                ->get();

            foreach ($system_properties as $property) {
                $license_data = json_decode($property->value, true);
                if ($license_data) {
                    $licenses[] = (object) $license_data;
                }
            }
        }

        return $licenses;
    }

    /**
     * Check if module is installed by checking System properties
     *
     * @param string $module_name
     * @param string $license_code
     * @param string $username
     * @return bool
     */
    private function isModuleInstalled($module_name, $license_code, $username)
    {
        // Check if module version exists in System properties
        $module_version = System::getProperty(strtolower($module_name) . '_version');
        
        if (empty($module_version)) {
            return false;
        }

        // Also verify the license code matches
        $stored_license_code = System::getProperty(strtolower($module_name) . '_license_code');
        $stored_username = System::getProperty(strtolower($module_name) . '_license_username');

        if ($stored_license_code === $license_code && $stored_username === $username) {
            return true;
        }

        return false;
    }

    /**
     * Generate license code
     */
    private function generateLicenseCode($module_name, $username, $email, $license_type)
    {
        // Generate a unique license code
        $data = $module_name . $username . $email . $license_type . time();
        $hash = hash('sha256', $data);
        
        // Format: BM-XXXX-XXXX-XXXX-XXXX (Business Management)
        $license_code = 'BM-' . strtoupper(substr($hash, 0, 4) . '-' . substr($hash, 4, 4) . '-' . substr($hash, 8, 4) . '-' . substr($hash, 12, 4));
        
        return $license_code;
    }

    /**
     * Check license validity
     */
    private function checkLicenseValidity($module_name, $license_code, $username)
    {
        // Check in database
        if (Schema::hasTable('module_licenses')) {
            $license = DB::table('module_licenses')
                ->where('module_name', $module_name)
                ->where('license_code', $license_code)
                ->where('username', $username)
                ->where('status', 'active')
                ->first();

            if ($license) {
                // Check expiry
                if ($license->expiry_date && strtotime($license->expiry_date) < time()) {
                    return false;
                }
                return true;
            }
        }

        // Check in System properties
        $property_key = 'license_' . $module_name . '_' . $license_code;
        $license_data = System::getProperty($property_key);
        
        if ($license_data) {
            $data = json_decode($license_data, true);
            if ($data && $data['username'] == $username && $data['status'] == 'active') {
                // Check expiry
                if (isset($data['expiry_date']) && strtotime($data['expiry_date']) < time()) {
                    return false;
                }
                return true;
            }
        }

        return false;
    }
}

