<?php

namespace App\Http\Controllers;

use App\BusinessLocation;
use App\Charts\CommonChart;
use App\Contact;
use App\Currency;
use App\Media;
use App\Product;
use App\Transaction;
use App\User;
use App\Utils\BusinessUtil;
use App\Utils\ModuleUtil;
use App\Utils\RestaurantUtil;
use App\Utils\TransactionUtil;
use App\Utils\ProductUtil;
use App\Utils\Util;
use App\VariationLocationDetails;
use Datatables;
use DB;
use Illuminate\Http\Request;
use Illuminate\Notifications\DatabaseNotification;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class HomeController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $businessUtil;

    protected $transactionUtil;

    protected $moduleUtil;

    protected $commonUtil;

    protected $restUtil;
    protected $productUtil;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(
        BusinessUtil $businessUtil,
        TransactionUtil $transactionUtil,
        ModuleUtil $moduleUtil,
        Util $commonUtil,
        RestaurantUtil $restUtil,
        ProductUtil $productUtil,
    ) {
        $this->businessUtil = $businessUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
        $this->commonUtil = $commonUtil;
        $this->restUtil = $restUtil;
        $this->productUtil = $productUtil;
    }

    /**
     * Validate location access
     *
     * @param mixed $location_id
     * @return mixed|null
     */
    private function validateLocationAccess($location_id)
    {
        if (empty($location_id) || $location_id == 'null' || $location_id == '') {
            return null;
        }
        
        $permitted_locations = auth()->user()->permitted_locations();
        
        if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
            abort(403, 'Unauthorized location access.');
        }
        
        return $location_id;
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = auth()->user();
        if ($user->user_type == 'user_customer') {
            return redirect()->action([\Modules\Crm\Http\Controllers\DashboardController::class, 'index']);
        }

        $business_id = request()->session()->get('user.business_id');

        $is_admin = $this->businessUtil->is_admin(auth()->user());

        if (! auth()->user()->can('dashboard.data')) {
            return view('home.index');
        }

        $fy = $this->businessUtil->getCurrentFinancialYear($business_id);

        $currency = Currency::where('id', request()->session()->get('business.currency_id'))->first();
        //ensure start date starts from at least 30 days before to get sells last 30 days
        $least_30_days = \Carbon::parse($fy['start'])->subDays(30)->format('Y-m-d');

        //get all sells
        $sells_this_fy = $this->transactionUtil->getSellsCurrentFy($business_id, $least_30_days, $fy['end']);

        // Get all locations filtered by user permissions
        // forDropdown() with check_permission=true (default) already filters by permissions
        $all_locations = BusinessLocation::forDropdown($business_id, false, false, true, true)->toArray();
        
        // Additional filter: ensure only permitted locations are shown
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $all_locations = array_intersect_key($all_locations, array_flip($permitted_locations));
        }

        //Chart for sells last 30 days (OPTIMIZED: Pre-process data once instead of multiple collection operations)
        $labels = [];
        $all_sell_values = [];
        $dates = [];
        
        // Pre-process sells data into arrays for O(1) lookups instead of O(n) collection searches
        $sells_by_date = [];
        $sells_by_date_location = [];
        foreach ($sells_this_fy as $sell) {
            $date = $sell->date ?? null;
            if ($date) {
                if (!isset($sells_by_date[$date])) {
                    $sells_by_date[$date] = 0;
                }
                $sells_by_date[$date] += (float) ($sell->total_sells ?? 0);
                
                $loc_id = $sell->location_id ?? null;
                if ($loc_id) {
                    if (!isset($sells_by_date_location[$date])) {
                        $sells_by_date_location[$date] = [];
                    }
                    if (!isset($sells_by_date_location[$date][$loc_id])) {
                        $sells_by_date_location[$date][$loc_id] = 0;
                    }
                    $sells_by_date_location[$date][$loc_id] += (float) ($sell->total_sells ?? 0);
                }
            }
        }
        
        for ($i = 29; $i >= 0; $i--) {
            $date = \Carbon::now()->subDays($i)->format('Y-m-d');
            $dates[] = $date;
            $labels[] = date('j M Y', strtotime($date));
            $all_sell_values[] = (float) ($sells_by_date[$date] ?? 0);
        }

        //Group sells by location (OPTIMIZED: Use pre-processed arrays)
        $location_sells = [];
        foreach ($all_locations as $loc_id => $loc_name) {
            $values = [];
            foreach ($dates as $date) {
                $values[] = (float) ($sells_by_date_location[$date][$loc_id] ?? 0);
            }
            $location_sells[$loc_id]['loc_label'] = $loc_name;
            $location_sells[$loc_id]['values'] = $values;
        }

        $sells_chart_1 = new CommonChart;

        $sells_chart_1->labels($labels)
                        ->options($this->__chartOptions(__(
                            'home.total_sells',
                            ['currency' => $currency->code]
                            )));

        if (! empty($location_sells)) {
            foreach ($location_sells as $location_sell) {
                $sells_chart_1->dataset($location_sell['loc_label'], 'area', $location_sell['values'])->color('#4db2ff');
            }
        }

        if (count($all_locations) > 1) {
            $sells_chart_1->dataset(__('report.all_locations'), 'area', $all_sell_values)->color('#4db2ff');
        }

        // Financial Year Chart (OPTIMIZED: Pre-process data once)
        $labels = [];
        $values = [];
        $date = strtotime($fy['start']);
        $last = date('m-Y', strtotime($fy['end']));
        $fy_months = [];
        
        // Pre-process sells data by month for O(1) lookups
        $sells_by_month = [];
        $sells_by_month_location = [];
        foreach ($sells_this_fy as $sell) {
            $month_year = $sell->yearmonth ?? null;
            if ($month_year) {
                if (!isset($sells_by_month[$month_year])) {
                    $sells_by_month[$month_year] = 0;
                }
                $sells_by_month[$month_year] += (float) ($sell->total_sells ?? 0);
                
                $loc_id = $sell->location_id ?? null;
                if ($loc_id) {
                    if (!isset($sells_by_month_location[$month_year])) {
                        $sells_by_month_location[$month_year] = [];
                    }
                    if (!isset($sells_by_month_location[$month_year][$loc_id])) {
                        $sells_by_month_location[$month_year][$loc_id] = 0;
                    }
                    $sells_by_month_location[$month_year][$loc_id] += (float) ($sell->total_sells ?? 0);
                }
            }
        }
        
        do {
            $month_year = date('m-Y', $date);
            $fy_months[] = $month_year;
            $labels[] = \Carbon::createFromFormat('m-Y', $month_year)->format('M-Y');
            $date = strtotime('+1 month', $date);
            $values[] = (float) ($sells_by_month[$month_year] ?? 0);
        } while ($month_year != $last);

        $fy_sells_by_location_data = [];
        foreach ($all_locations as $loc_id => $loc_name) {
            $values_data = [];
            foreach ($fy_months as $month) {
                $values_data[] = (float) ($sells_by_month_location[$month][$loc_id] ?? 0);
            }
            $fy_sells_by_location_data[$loc_id]['loc_label'] = $loc_name;
            $fy_sells_by_location_data[$loc_id]['values'] = $values_data;
        }

        $sells_chart_2 = new CommonChart;
        $sells_chart_2->labels($labels)
                    ->options($this->__chartOptions(__(
                        'home.total_sells',
                        ['currency' => $currency->code]
                            )));
        if (! empty($fy_sells_by_location_data)) {
            foreach ($fy_sells_by_location_data as $location_sell) {
                $sells_chart_2->dataset($location_sell['loc_label'], 'area', $location_sell['values'])->color('#4db2ff');
            }
        }
        if (count($all_locations) > 1) {
            $sells_chart_2->dataset(__('report.all_locations'), 'area', $values)->color('#4db2ff');
        }

        //Get Dashboard widgets from module
        $module_widgets = $this->moduleUtil->getModuleData('dashboard_widget');

        $widgets = [];

        foreach ($module_widgets as $widget_array) {
            if (! empty($widget_array['position'])) {
                $widgets[$widget_array['position']][] = $widget_array['widget'];
            }
        }

        $common_settings = ! empty(session('business.common_settings')) ? session('business.common_settings') : [];
    
        $is_admin = 0;

        // Fetch totals for dashboard charts - respect date filter from request, default to today
        $start = request()->get('start', \Carbon::now()->startOfDay()->toDateString());
        $end = request()->get('end', \Carbon::now()->endOfDay()->toDateString());
        
        // Ensure dates are properly formatted
        try {
            $start = \Carbon::parse($start)->startOfDay()->toDateString();
            $end = \Carbon::parse($end)->endOfDay()->toDateString();
        } catch (\Exception $e) {
            // If date parsing fails, default to today
            $start = \Carbon::now()->startOfDay()->toDateString();
            $end = \Carbon::now()->endOfDay()->toDateString();
        }
        $purchase_details = $this->transactionUtil->getPurchaseTotals($business_id, $start, $end, null, null);
        $sell_details = $this->transactionUtil->getSellTotals($business_id, $start, $end, null, null);
        $transaction_types = ['purchase_return', 'sell_return', 'expense'];
        $transaction_totals = $this->transactionUtil->getTransactionTotals($business_id, $transaction_types, $start, $end);
        $total_sell = !empty($sell_details['total_sell_inc_tax']) ? $sell_details['total_sell_inc_tax'] : 0;
        $total_purchase = !empty($purchase_details['total_purchase_inc_tax']) ? $purchase_details['total_purchase_inc_tax'] : 0;
        $total_expense = !empty($transaction_totals['total_expense']) ? $transaction_totals['total_expense'] : 0;
        $invoice_due = !empty($sell_details['invoice_due']) ? $sell_details['invoice_due'] : 0;
        $purchase_due = !empty($purchase_details['purchase_due']) ? $purchase_details['purchase_due'] : 0;
        $net = $total_sell - $invoice_due - $total_expense;

        // Revenue Distribution variables - Calculate cash and credit sales
        // Cash sales = sum of cash payments from transaction_payments
        // Credit sales = sum of outstanding amounts (final_total - total_paid) for transactions with payment_status != 'paid'
        $cash_sales_query = \DB::table('transaction_payments as tp')
            ->join('transactions as t', 'tp.transaction_id', '=', 't.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->where('tp.is_return', 0)
            ->where('tp.method', 'cash')
            ->whereDate('t.transaction_date', '>=', $start)
            ->whereDate('t.transaction_date', '<=', $end)
            ->selectRaw('COALESCE(SUM(tp.amount), 0) as cash_sales')
            ->first();
        
        // Credit sales = outstanding amount (final_total - total_paid) for transactions with payment_status != 'paid'
        // Calculate total paid per transaction including returns
        $credit_sales_query = \DB::table('transactions as t')
            ->leftJoin(\DB::raw('(SELECT transaction_id, 
                                  SUM(CASE WHEN is_return = 0 THEN amount ELSE -amount END) as total_paid 
                                  FROM transaction_payments 
                                  WHERE is_return = 0 OR is_return = 1
                                  GROUP BY transaction_id) as tp'), 't.id', '=', 'tp.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->whereIn('t.payment_status', ['due', 'partial'])
            ->whereDate('t.transaction_date', '>=', $start)
            ->whereDate('t.transaction_date', '<=', $end)
            ->selectRaw('COALESCE(SUM(GREATEST(t.final_total - COALESCE(tp.total_paid, 0), 0)), 0) as credit_sales')
            ->first();
        
        $cash_sales = $cash_sales_query->cash_sales ?? 0;
        $credit_sales = $credit_sales_query->credit_sales ?? 0;
        $transaction_types = ['sell_return'];
        $transaction_totals = $this->transactionUtil->getTransactionTotals($business_id, $transaction_types, $start, $end);
        $total_sell_return = !empty($transaction_totals['total_sell_return_inc_tax']) ? $transaction_totals['total_sell_return_inc_tax'] : 0;

        // Prepare monthly sales and purchases for the current year (OPTIMIZED: Use cache and reduce queries)
        $current_year = date('Y');
        $cache_key = 'dashboard_monthly_data_' . $business_id . '_' . $current_year;
        $monthly_data = Cache::remember($cache_key, 300, function () use ($business_id, $current_year) {
            $monthly_sales = [];
            $monthly_purchases = [];
            $monthly_expenses = [];
            
            // Get all transactions for the year in fewer queries
            $year_start = \Carbon::create($current_year, 1, 1)->startOfYear()->toDateString();
            $year_end = \Carbon::create($current_year, 12, 31)->endOfYear()->toDateString();
            
            // Fetch all sell transactions for the year with revenue
            $all_sells = \DB::table('transactions')
                ->where('business_id', $business_id)
                ->where('type', 'sell')
                ->where('status', 'final')
                ->whereBetween('transaction_date', [$year_start, $year_end])
                ->selectRaw('
                    MONTH(transaction_date) as month,
                    SUM(final_total) as total_sell
                ')
                ->groupBy('month')
                ->pluck('total_sell', 'month')
                ->toArray();
            
            // Fetch cost of goods sold (COGS) from sell transactions - actual cost of items sold
            $all_cogs = \DB::table('transactions as t')
                ->join('transaction_sell_lines as tsl', 't.id', '=', 'tsl.transaction_id')
                ->leftJoin('variations as v', 'tsl.variation_id', '=', 'v.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->whereNull('tsl.parent_sell_line_id')
                ->whereBetween('t.transaction_date', [$year_start, $year_end])
                ->selectRaw('
                    MONTH(t.transaction_date) as month,
                    SUM(COALESCE(v.default_purchase_price, tsl.unit_price_inc_tax * 0.7) * (tsl.quantity - COALESCE(tsl.quantity_returned, 0))) as total_cogs
                ')
                ->groupBy('month')
                ->pluck('total_cogs', 'month')
                ->toArray();
            
            // Fetch all purchase transactions for the year (for other charts)
            $all_purchases = \DB::table('transactions')
                ->where('business_id', $business_id)
                ->where('type', 'purchase')
                ->where('status', 'received')
                ->whereBetween('transaction_date', [$year_start, $year_end])
                ->selectRaw('
                    MONTH(transaction_date) as month,
                    SUM(final_total) as total_purchase
                ')
                ->groupBy('month')
                ->pluck('total_purchase', 'month')
                ->toArray();
            
            // Fetch all expenses for the year
            $all_expenses = \DB::table('transactions')
                ->where('business_id', $business_id)
                ->where('type', 'expense')
                ->whereBetween('transaction_date', [$year_start, $year_end])
                ->selectRaw('
                    MONTH(transaction_date) as month,
                    SUM(final_total) as total_expense
                ')
                ->groupBy('month')
                ->pluck('total_expense', 'month')
                ->toArray();
            
            // Initialize all months with 0
            for ($month = 1; $month <= 12; $month++) {
                $monthly_sales[$month] = $all_sells[$month] ?? 0;
                $monthly_purchases[$month] = $all_purchases[$month] ?? 0;
                $monthly_expenses[$month] = $all_expenses[$month] ?? 0;
                $monthly_cogs[$month] = $all_cogs[$month] ?? 0;
            }
            
            return [
                'sales' => $monthly_sales,
                'purchases' => $monthly_purchases,
                'expenses' => $monthly_expenses,
                'cogs' => $monthly_cogs
            ];
        });
        
        $monthly_sales = $monthly_data['sales'];
        $monthly_purchases = $monthly_data['purchases'];
        $monthly_expenses = $monthly_data['expenses'];
        $monthly_cogs = $monthly_data['cogs'] ?? [];

        // Additional KPIs (OPTIMIZED: Cache and combine queries)
        // Include date in cache key to respect date filters
        $kpi_cache_key = 'dashboard_kpis_' . $business_id . '_' . $start . '_' . $end;
        $kpis = Cache::remember($kpi_cache_key, 180, function () use ($business_id, $start, $end) {
            return [
                'total_customers' => Contact::where('business_id', $business_id)
                    ->whereIn('type', ['customer', 'both'])
                    ->count(),
                'total_products' => Product::where('business_id', $business_id)
                    ->where('type', '!=', 'modifier')
                    ->count(),
                'total_transactions_today' => Transaction::where('business_id', $business_id)
                    ->where('type', 'sell')
                    ->where('status', 'final')
                    ->whereDate('transaction_date', '>=', $start)
                    ->whereDate('transaction_date', '<=', $end)
                    ->count()
            ];
        });
        
        $total_customers = $kpis['total_customers'];
        $total_products = $kpis['total_products'];
        $total_transactions_today = $kpis['total_transactions_today'];
        
        // Calculate profit margin
        $profit_margin = 0;
        if ($total_sell > 0) {
            $profit_margin = (($total_sell - $total_purchase - $total_expense) / $total_sell) * 100;
        }
        
        // Calculate average order value
        $average_order_value = 0;
        if ($total_transactions_today > 0) {
            $average_order_value = $total_sell / $total_transactions_today;
        }
        
        // Calculate growth rate (compare with previous period - day before selected start date)
        $selected_date = \Carbon::parse($start);
        $previous_period_start = $selected_date->copy()->subDay()->startOfDay()->toDateString();
        $previous_period_end = $selected_date->copy()->subDay()->endOfDay()->toDateString();
        $yesterday_sell_details = $this->transactionUtil->getSellTotals($business_id, $previous_period_start, $previous_period_end, null, null);
        $yesterday_sales = !empty($yesterday_sell_details['total_sell_inc_tax']) ? $yesterday_sell_details['total_sell_inc_tax'] : 0;
        $growth_rate = 0;
        if ($yesterday_sales > 0) {
            $growth_rate = (($total_sell - $yesterday_sales) / $yesterday_sales) * 100;
        }

        return view('home.index', compact(
            'sells_chart_1', 'sells_chart_2', 'widgets', 'all_locations', 'common_settings', 'is_admin',
            'total_sell', 'total_purchase', 'total_expense',
            'monthly_sales', 'monthly_purchases',
            'monthly_expenses',
            'invoice_due', 'purchase_due',
            'net',
            'cash_sales', 'credit_sales', 'total_sell_return',
            'total_customers', 'total_products', 'total_transactions_today',
            'profit_margin', 'average_order_value', 'growth_rate',
            'start', 'end'
        ));
    }

    /**
     * Retrieves purchase and sell details for a given time period.
     *
     * @return \Illuminate\Http\Response
     */
    public function getTotals()
    {
        if (request()->ajax()) {
            $start = request()->start;
            $end = request()->end;
            $location_id = $this->validateLocationAccess(request()->location_id);
            $business_id = request()->session()->get('user.business_id');

            // get user id parameter
            $created_by = request()->user_id;

            $purchase_details = $this->transactionUtil->getPurchaseTotals($business_id, $start, $end, $location_id, $created_by);

            $sell_details = $this->transactionUtil->getSellTotals($business_id, $start, $end, $location_id, $created_by);

            $total_ledger_discount = $this->transactionUtil->getTotalLedgerDiscount($business_id, $start, $end);

            $purchase_details['purchase_due'] = $purchase_details['purchase_due'] - $total_ledger_discount['total_purchase_discount'];

            $transaction_types = [
                'purchase_return', 'sell_return', 'expense',
            ];

            $transaction_totals = $this->transactionUtil->getTransactionTotals(
                $business_id,
                $transaction_types,
                $start,
                $end,
                $location_id,
                $created_by
            );

            $total_purchase_inc_tax = ! empty($purchase_details['total_purchase_inc_tax']) ? $purchase_details['total_purchase_inc_tax'] : 0;
            $total_purchase_return_inc_tax = $transaction_totals['total_purchase_return_inc_tax'];

            $output = $purchase_details;
            $output['total_purchase'] = $total_purchase_inc_tax;
            $output['total_purchase_return'] = $total_purchase_return_inc_tax;
            $output['total_purchase_return_paid'] = $this->transactionUtil->getTotalPurchaseReturnPaid($business_id, $start, $end, $location_id);

            $total_sell_inc_tax = ! empty($sell_details['total_sell_inc_tax']) ? $sell_details['total_sell_inc_tax'] : 0;
            $total_sell_return_inc_tax = ! empty($transaction_totals['total_sell_return_inc_tax']) ? $transaction_totals['total_sell_return_inc_tax'] : 0;
            $output['total_sell_return_paid'] = $this->transactionUtil->getTotalSellReturnPaid($business_id, $start, $end, $location_id);

            $output['total_sell'] = $total_sell_inc_tax;
            $output['total_sell_return'] = $total_sell_return_inc_tax;

            $output['invoice_due'] = $sell_details['invoice_due'] - $total_ledger_discount['total_sell_discount'];
            $output['total_expense'] = $transaction_totals['total_expense'];

            //NET = TOTAL SALES - INVOICE DUE - EXPENSE
            $output['net'] = $output['total_sell'] - $output['invoice_due'] - $output['total_expense'];

            return $output;
        }
    }

    /**
     * Retrieves sell products whose available quntity is less than alert quntity.
     *
     * @return \Illuminate\Http\Response
     */
    public function getProductStockAlert()
    {
        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $permitted_locations = auth()->user()->permitted_locations();
            $products = $this->productUtil->getProductAlert($business_id, $permitted_locations);

            return Datatables::of($products)
                ->editColumn('product', function ($row) {
                    if ($row->type == 'single') {
                        return $row->product.' ('.$row->sku.')';
                    } else {
                        return $row->product.' - '.$row->product_variation.' - '.$row->variation.' ('.$row->sub_sku.')';
                    }
                })
                ->editColumn('stock', function ($row) {
                    $stock = $row->stock ? $row->stock : 0;

                    return '<span data-is_quantity="true" class="display_currency" data-currency_symbol=false>'.(float) $stock.'</span> '.$row->unit;
                })
                ->removeColumn('sku')
                ->removeColumn('sub_sku')
                ->removeColumn('unit')
                ->removeColumn('type')
                ->removeColumn('product_variation')
                ->removeColumn('variation')
                ->rawColumns([2])
                ->make(false);
        }
    }

    /**
     * Retrieves payment dues for the purchases.
     *
     * @return \Illuminate\Http\Response
     */
    public function getPurchasePaymentDues()
    {
        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $today = \Carbon::now()->format('Y-m-d H:i:s');

            $query = Transaction::join(
                'contacts as c',
                'transactions.contact_id',
                '=',
                'c.id'
            )
                    ->leftJoin(
                        'transaction_payments as tp',
                        'transactions.id',
                        '=',
                        'tp.transaction_id'
                    )
                    ->where('transactions.business_id', $business_id)
                    ->where('transactions.type', 'purchase')
                    ->where('transactions.payment_status', '!=', 'paid')
                    ->whereRaw("DATEDIFF( DATE_ADD( transaction_date, INTERVAL IF(transactions.pay_term_type = 'days', transactions.pay_term_number, 30 * transactions.pay_term_number) DAY), '$today') <= 7");

            //Check for permitted locations of a user
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('transactions.location_id', $permitted_locations);
            }

            // Validate specific location if provided
            $location_id = $this->validateLocationAccess(request()->input('location_id'));
            if (!empty($location_id)) {
                $query->where('transactions.location_id', $location_id);
            }

            $dues = $query->select(
                'transactions.id as id',
                'c.name as supplier',
                'c.supplier_business_name',
                'ref_no',
                'final_total',
                DB::raw('SUM(tp.amount) as total_paid')
            )
                        ->groupBy('transactions.id');

            return Datatables::of($dues)
                ->addColumn('due', function ($row) {
                    $total_paid = ! empty($row->total_paid) ? $row->total_paid : 0;
                    $due = $row->final_total - $total_paid;

                    return '<span class="display_currency" data-currency_symbol="true">'.
                    $due.'</span>';
                })
                ->addColumn('action', '@can("purchase.create") <a href="{{action([\App\Http\Controllers\TransactionPaymentController::class, \'addPayment\'], [$id])}}" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline tw-dw-btn-accent add_payment_modal"><i class="fas fa-money-bill-alt"></i> @lang("purchase.add_payment")</a> @endcan')
                ->removeColumn('supplier_business_name')
                ->editColumn('supplier', '@if(!empty($supplier_business_name)) {{$supplier_business_name}}, <br> @endif {{$supplier}}')
                ->editColumn('ref_no', function ($row) {
                    if (auth()->user()->can('purchase.view')) {
                        return  '<a href="#" data-href="'.action([\App\Http\Controllers\PurchaseController::class, 'show'], [$row->id]).'"
                                    class="btn-modal" data-container=".view_modal">'.$row->ref_no.'</a>';
                    }

                    return $row->ref_no;
                })
                ->removeColumn('id')
                ->removeColumn('final_total')
                ->removeColumn('total_paid')
                ->rawColumns([0, 1, 2, 3])
                ->make(false);
        }
    }

    /**
     * Retrieves payment dues for the purchases.
     *
     * @return \Illuminate\Http\Response
     */
    public function getSalesPaymentDues()
    {
        if (request()->ajax()) {
            try {
                $business_id = request()->session()->get('user.business_id');
                if (!$business_id) {
                    return response()->json(['error' => 'Business ID not found'], 400);
                }

                $today = \Carbon::now()->format('Y-m-d H:i:s');

                $query = Transaction::join(
                    'contacts as c',
                    'transactions.contact_id',
                    '=',
                    'c.id'
                )
                        ->leftJoin(
                            'transaction_payments as tp',
                            'transactions.id',
                            '=',
                            'tp.transaction_id'
                        )
                        ->where('transactions.business_id', $business_id)
                        ->where('transactions.type', 'sell')
                        ->where('transactions.payment_status', '!=', 'paid')
                        ->whereNotNull('transactions.pay_term_number')
                        ->whereNotNull('transactions.pay_term_type')
                        ->whereRaw("DATEDIFF( DATE_ADD( transaction_date, INTERVAL IF(transactions.pay_term_type = 'days', transactions.pay_term_number, 30 * transactions.pay_term_number) DAY), '$today') <= 7");

                //Check for permitted locations of a user
                $permitted_locations = auth()->user()->permitted_locations();
                if ($permitted_locations != 'all') {
                    $query->whereIn('transactions.location_id', $permitted_locations);
                }

                // Validate specific location if provided
                $location_id = $this->validateLocationAccess(request()->input('location_id'));
                if (!empty($location_id)) {
                    $query->where('transactions.location_id', $location_id);
                }

                $dues = $query->select(
                    'transactions.id as id',
                    'c.name as customer',
                    'c.supplier_business_name',
                    'transactions.invoice_no',
                    'final_total',
                    DB::raw('SUM(tp.amount) as total_paid')
                )
                            ->groupBy('transactions.id');

                return Datatables::of($dues)
                    ->addColumn('due', function ($row) {
                        $total_paid = ! empty($row->total_paid) ? $row->total_paid : 0;
                        $due = $row->final_total - $total_paid;

                        return '<span class="display_currency" data-currency_symbol="true">'.
                        $due.'</span>';
                    })
                    ->editColumn('invoice_no', function ($row) {
                        if (auth()->user()->can('sell.view')) {
                            return  '<a href="#" data-href="'.action([\App\Http\Controllers\SellController::class, 'show'], [$row->id]).'"
                                        class="btn-modal" data-container=".view_modal">'.$row->invoice_no.'</a>';
                        }

                        return $row->invoice_no;
                    })
                    ->addColumn('action', '@if(auth()->user()->can("sell.create") || auth()->user()->can("direct_sell.access")) <a href="{{action([\App\Http\Controllers\TransactionPaymentController::class, \'addPayment\'], [$id])}}" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline tw-dw-btn-accent add_payment_modal"><i class="fas fa-money-bill-alt"></i> @lang("purchase.add_payment")</a> @endif')
                    ->editColumn('customer', '@if(!empty($supplier_business_name)) {{$supplier_business_name}}, <br> @endif {{$customer}}')
                    ->removeColumn('supplier_business_name')
                    ->removeColumn('id')
                    ->removeColumn('final_total')
                    ->removeColumn('total_paid')
                    ->rawColumns([0, 1, 2, 3])
                    ->make(false);
            } catch (\Exception $e) {
                Log::error('Error in getSalesPaymentDues: ' . $e->getMessage());
                return response()->json([
                    'draw' => request()->input('draw', 0),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'An error occurred while loading data. Please try again.'
                ], 500);
            }
        }

        return response()->json(['error' => 'Invalid request'], 400);
    }

    /**
     * Get Top 10 Customers by Sales Amount
     */
    public function getTopCustomers()
    {
        if (request()->ajax() || request()->wantsJson()) {
            try {
                $business_id = request()->session()->get('user.business_id');
                if (!$business_id) {
                    return response()->json(['data' => [], 'error' => 'Business ID not found'], 400);
                }
                
                $location_id = $this->validateLocationAccess(request()->get('location_id'));
                // Use a wider default range - last 365 days if no dates provided
                $start = request()->get('start', \Carbon::now()->subYear()->toDateString());
                $end = request()->get('end', \Carbon::now()->toDateString());

                // Validate location permissions
                $permitted_locations = auth()->user()->permitted_locations();
                if ($permitted_locations != 'all') {
                    $query = \DB::table('transactions as t')
                        ->join('contacts as c', 't.contact_id', '=', 'c.id')
                        ->where('t.business_id', $business_id)
                        ->where('t.type', 'sell')
                        ->where('t.status', 'final')
                        ->whereDate('t.transaction_date', '>=', $start)
                        ->whereDate('t.transaction_date', '<=', $end)
                        ->whereNotNull('t.contact_id')
                        ->whereIn('t.location_id', $permitted_locations);
                } else {
                    $query = \DB::table('transactions as t')
                        ->join('contacts as c', 't.contact_id', '=', 'c.id')
                        ->where('t.business_id', $business_id)
                        ->where('t.type', 'sell')
                        ->where('t.status', 'final')
                        ->whereDate('t.transaction_date', '>=', $start)
                        ->whereDate('t.transaction_date', '<=', $end)
                        ->whereNotNull('t.contact_id');
                }

                // Apply specific location filter if provided and validate permission
                if (!empty($location_id) && $location_id != 'null' && $location_id != '') {
                    // Validate location permission
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        return response()->json(['data' => [], 'error' => 'Unauthorized location access.'], 403);
                    }
                    $query->where('t.location_id', $location_id);
                }

                $customers = $query->select(
                    'c.id',
                    'c.name as customer_name',
                    'c.supplier_business_name',
                    \DB::raw('SUM(t.final_total) as total_amount'),
                    \DB::raw('COUNT(t.id) as transaction_count')
                )
                ->groupBy('c.id', 'c.name', 'c.supplier_business_name')
                ->orderBy('total_amount', 'desc')
                ->limit(10)
                ->get();

                $data = [];
                $rank = 1;
                foreach ($customers as $customer) {
                    $display_name = !empty($customer->supplier_business_name) 
                        ? $customer->supplier_business_name . ' - ' . $customer->customer_name
                        : ($customer->customer_name ?? 'Unknown Customer');

                    $data[] = [
                        'rank' => $rank++,
                        'customer_name' => $display_name,
                        'amount' => floatval($customer->total_amount ?? 0)
                    ];
                }

                return response()->json(['data' => $data]);
            } catch (\Exception $e) {
                \Log::error('Error in getTopCustomers: ' . $e->getMessage());
                \Log::error('Stack trace: ' . $e->getTraceAsString());
                return response()->json(['data' => [], 'error' => $e->getMessage()], 500);
            }
        }
        
        return response()->json(['data' => [], 'error' => 'Invalid request'], 400);
    }

    /**
     * Get Top 10 Products by Quantity Sold
     */
    public function getTopProducts()
    {
        if (request()->ajax() || request()->wantsJson()) {
            try {
                $business_id = request()->session()->get('user.business_id');
                if (!$business_id) {
                    return response()->json(['data' => [], 'error' => 'Business ID not found'], 400);
                }
                
                $location_id = $this->validateLocationAccess(request()->get('location_id'));
                // Use a wider default range - last 365 days if no dates provided
                $start = request()->get('start', \Carbon::now()->subYear()->toDateString());
                $end = request()->get('end', \Carbon::now()->toDateString());

                // Validate location permissions
                $permitted_locations = auth()->user()->permitted_locations();
                if ($permitted_locations != 'all') {
                    $query = \DB::table('transaction_sell_lines as tsl')
                        ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                        ->join('variations as v', 'tsl.variation_id', '=', 'v.id')
                        ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                        ->join('products as p', 'pv.product_id', '=', 'p.id')
                        ->where('t.business_id', $business_id)
                        ->where('t.type', 'sell')
                        ->where('t.status', 'final')
                        ->whereDate('t.transaction_date', '>=', $start)
                        ->whereDate('t.transaction_date', '<=', $end)
                        ->whereNull('tsl.parent_sell_line_id')
                        ->whereIn('t.location_id', $permitted_locations);
                } else {
                    $query = \DB::table('transaction_sell_lines as tsl')
                        ->join('transactions as t', 'tsl.transaction_id', '=', 't.id')
                        ->join('variations as v', 'tsl.variation_id', '=', 'v.id')
                        ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                        ->join('products as p', 'pv.product_id', '=', 'p.id')
                        ->where('t.business_id', $business_id)
                        ->where('t.type', 'sell')
                        ->where('t.status', 'final')
                        ->whereDate('t.transaction_date', '>=', $start)
                        ->whereDate('t.transaction_date', '<=', $end)
                        ->whereNull('tsl.parent_sell_line_id');
                }

                // Apply specific location filter if provided and validate permission
                if (!empty($location_id) && $location_id != 'null' && $location_id != '') {
                    // Validate location permission
                    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                        return response()->json(['data' => [], 'error' => 'Unauthorized location access.'], 403);
                    }
                    $query->where('t.location_id', $location_id);
                }

                $products = $query->select(
                    'p.id',
                    'p.name as product_name',
                    'pv.name as variation_name',
                    'v.name as sub_sku',
                    \DB::raw('SUM(tsl.quantity - COALESCE(tsl.quantity_returned, 0)) as total_quantity')
                )
                ->groupBy('p.id', 'p.name', 'pv.name', 'v.name')
                ->orderBy('total_quantity', 'desc')
                ->limit(10)
                ->get();

                $data = [];
                $rank = 1;
                foreach ($products as $product) {
                    $display_name = $product->product_name ?? 'Unknown Product';
                    if (!empty($product->variation_name) && $product->variation_name != 'DUMMY') {
                        $display_name .= ' - ' . $product->variation_name;
                    }
                    if (!empty($product->sub_sku) && $product->sub_sku != 'DUMMY') {
                        $display_name .= ' (' . $product->sub_sku . ')';
                    }

                    $data[] = [
                        'rank' => $rank++,
                        'product_name' => $display_name,
                        'quantity' => floatval($product->total_quantity ?? 0)
                    ];
                }

                return response()->json(['data' => $data]);
            } catch (\Exception $e) {
                \Log::error('Error in getTopProducts: ' . $e->getMessage());
                \Log::error('Stack trace: ' . $e->getTraceAsString());
                return response()->json(['data' => [], 'error' => $e->getMessage()], 500);
            }
        }
        
        return response()->json(['data' => [], 'error' => 'Invalid request'], 400);
    }

    public function loadMoreNotifications()
    {
        $notifications = auth()->user()->notifications()->orderBy('created_at', 'DESC')->paginate(10);

        if (request()->input('page') == 1) {
            auth()->user()->unreadNotifications->markAsRead();
        }
        $notifications_data = $this->commonUtil->parseNotifications($notifications);

        return view('layouts.partials.notification_list', compact('notifications_data'));
    }

    /**
     * Function to count total number of unread notifications
     *
     * @return json
     */
    public function getTotalUnreadNotifications()
    {
        $unread_notifications = auth()->user()->unreadNotifications;
        $total_unread = $unread_notifications->count();

        $notification_html = '';
        $modal_notifications = [];
        foreach ($unread_notifications as $unread_notification) {
            if (isset($data['show_popup'])) {
                $modal_notifications[] = $unread_notification;
                $unread_notification->markAsRead();
            }
        }
        if (! empty($modal_notifications)) {
            $notification_html = view('home.notification_modal')->with(['notifications' => $modal_notifications])->render();
        }

        return [
            'total_unread' => $total_unread,
            'notification_html' => $notification_html,
        ];
    }

    private function __chartOptions($title)
    {
        return [
            'yAxis' => [
                'title' => [
                    'text' => $title,
                ],
            ],
            'legend' => [
                'align' => 'right',
                'verticalAlign' => 'top',
                'floating' => true,
                'layout' => 'vertical',
                'padding' => 20,
            ],
        ];
    }

    public function getCalendar()
    {
        $business_id = request()->session()->get('user.business_id');
        $is_admin = $this->restUtil->is_admin(auth()->user(), $business_id);
        $is_superadmin = auth()->user()->can('superadmin');
        if (request()->ajax()) {
            $data = [
                'start_date' => request()->start,
                'end_date' => request()->end,
                'user_id' => ($is_admin || $is_superadmin) && ! empty(request()->user_id) ? request()->user_id : auth()->user()->id,
                'location_id' => $this->validateLocationAccess(request()->location_id),
                'business_id' => $business_id,
                'events' => request()->events ?? [],
                'color' => '#007FFF',
            ];
            $events = [];

            if (in_array('bookings', $data['events'])) {
                $events = $this->restUtil->getBookingsForCalendar($data);
            }

            $module_events = $this->moduleUtil->getModuleData('calendarEvents', $data);

            foreach ($module_events as $module_event) {
                $events = array_merge($events, $module_event);
            }

            return $events;
        }

        $all_locations = BusinessLocation::forDropdown($business_id)->toArray();
        $users = [];
        if ($is_admin) {
            $users = User::forDropdown($business_id, false);
        }

        $event_types = [
            'bookings' => [
                'label' => __('restaurant.bookings'),
                'color' => '#007FFF',
            ],
        ];
        $module_event_types = $this->moduleUtil->getModuleData('eventTypes');
        foreach ($module_event_types as $module_event_type) {
            $event_types = array_merge($event_types, $module_event_type);
        }

        return view('home.calendar')->with(compact('all_locations', 'users', 'event_types'));
    }

    public function showNotification($id)
    {
        $notification = DatabaseNotification::find($id);

        $data = $notification->data;

        $notification->markAsRead();

        return view('home.notification_modal')->with([
            'notifications' => [$notification],
        ]);
    }

    public function attachMediasToGivenModel(Request $request)
    {
        if ($request->ajax()) {
            try {
                $business_id = request()->session()->get('user.business_id');

                $model_id = $request->input('model_id');
                $model = $request->input('model_type');
                $model_media_type = $request->input('model_media_type');

                DB::beginTransaction();

                //find model to which medias are to be attached
                $model_to_be_attached = $model::where('business_id', $business_id)
                                        ->findOrFail($model_id);

                Media::uploadMedia($business_id, $model_to_be_attached, $request, 'file', false, $model_media_type);

                DB::commit();

                $output = [
                    'success' => true,
                    'msg' => __('lang_v1.success'),
                ];
            } catch (Exception $e) {
                DB::rollBack();

                \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = [
                    'success' => false,
                    'msg' => __('messages.something_went_wrong'),
                ];
            }

            return $output;
        }
    }

    public function getUserLocation($latlng)
    {
        $latlng_array = explode(',', $latlng);

        $response = $this->moduleUtil->getLocationFromCoordinates($latlng_array[0], $latlng_array[1]);

        return ['address' => $response];
    }

    public function optimizeClear()
    {
        Artisan::call('optimize:clear');

        return redirect()->back()->with('success', 'Caches cleared successfully!');
    }

    public function routeClear()
    {
        Artisan::call('route:clear');

        return redirect()->back()->with('success', 'Route cache cleared.');
    }
}
