<?php

namespace App\Console\Commands;

use App\Business;
use App\BusinessLocation;
use App\Product;
use App\Variation;
use App\VariationLocationDetails;
use App\Warehouse;
use App\Utils\InventoryService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class MigrateStockToWarehouses extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'stock:migrate-to-warehouses 
                            {--business_id= : Specific business ID to migrate}
                            {--dry-run : Show what would be migrated without making changes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Migrate existing location-based stock to warehouse-based stock';

    protected $inventoryService;

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->inventoryService = new InventoryService();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $business_id = $this->option('business_id');
        $dry_run = $this->option('dry-run');

        if ($dry_run) {
            $this->info('DRY RUN MODE - No changes will be made');
        }

        $query = Business::query();
        if ($business_id) {
            $query->where('id', $business_id);
        }

        $businesses = $query->get();

        if ($businesses->isEmpty()) {
            $this->error('No businesses found to migrate.');
            return 1;
        }

        $this->info("Found {$businesses->count()} business(es) to process.");

        foreach ($businesses as $business) {
            $this->info("\nProcessing Business: {$business->name} (ID: {$business->id})");
            $this->migrateBusiness($business, $dry_run);
        }

        $this->info("\nMigration completed!");
        return 0;
    }

    /**
     * Migrate stock for a single business
     *
     * @param Business $business
     * @param bool $dry_run
     * @return void
     */
    protected function migrateBusiness($business, $dry_run = false)
    {
        // Get or create default warehouse for business
        $warehouse = Warehouse::where('business_id', $business->id)
            ->where('is_active', 1)
            ->first();

        if (!$warehouse) {
            if ($dry_run) {
                $this->warn("  Would create default warehouse 'Main Warehouse'");
            } else {
                $warehouse = Warehouse::create([
                    'business_id' => $business->id,
                    'name' => 'Main Warehouse',
                    'code' => 'WH001',
                    'is_active' => 1,
                    'created_by' => 1,
                ]);

                $business->default_warehouse_id = $warehouse->id;
                $business->save();

                $this->info("  Created default warehouse: {$warehouse->name}");
            }
        } else {
            $this->info("  Using existing warehouse: {$warehouse->name}");
        }

        // Get all location-based stock
        $location_stocks = VariationLocationDetails::join('business_locations', 'variation_location_details.location_id', '=', 'business_locations.id')
            ->where('business_locations.business_id', $business->id)
            ->where('variation_location_details.qty_available', '>', 0)
            ->select('variation_location_details.*')
            ->get();

        $this->info("  Found {$location_stocks->count()} location stock records to migrate");

        $migrated = 0;
        $skipped = 0;

        foreach ($location_stocks as $location_stock) {
            // Check if product and variation still exist
            $product = Product::find($location_stock->product_id);
            $variation = Variation::find($location_stock->variation_id);

            if (!$product || !$variation) {
                $skipped++;
                continue;
            }

            // Check if stock already exists in warehouse
            $existing = DB::table('product_warehouse')
                ->where('product_id', $location_stock->product_id)
                ->where('variation_id', $location_stock->variation_id)
                ->where('warehouse_id', $warehouse->id)
                ->first();

            if ($existing) {
                // Add to existing stock
                if ($dry_run) {
                    $this->line("  Would add {$location_stock->qty_available} to existing warehouse stock for Product #{$product->id}");
                } else {
                    $this->inventoryService->addStock(
                        $location_stock->product_id,
                        $location_stock->variation_id,
                        $warehouse->id,
                        $location_stock->qty_available,
                        'opening_stock',
                        null,
                        "Migrated from location-based stock"
                    );
                }
            } else {
                // Create new warehouse stock
                if ($dry_run) {
                    $this->line("  Would create warehouse stock of {$location_stock->qty_available} for Product #{$product->id}");
                } else {
                    $this->inventoryService->addStock(
                        $location_stock->product_id,
                        $location_stock->variation_id,
                        $warehouse->id,
                        $location_stock->qty_available,
                        'opening_stock',
                        null,
                        "Migrated from location-based stock"
                    );
                }
            }

            $migrated++;
        }

        $this->info("  Migrated: {$migrated} records, Skipped: {$skipped} records");

        if (!$dry_run) {
            $this->info("  Setting default warehouse for business");
            if (!$business->default_warehouse_id) {
                $business->default_warehouse_id = $warehouse->id;
                $business->save();
            }
        }
    }
}

