<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\System;

class GenerateLicense extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'license:generate 
                            {module : Module name (e.g., businessmanagement)}
                            {username : Username for the license}
                            {email : Email for the license}
                            {--type=single : License type (single, multi, unlimited)}
                            {--expiry= : Expiry date (Y-m-d format, optional)}
                            {--code= : Custom license code (optional)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate a license for a module';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $module_name = $this->argument('module');
        $username = $this->argument('username');
        $email = $this->argument('email');
        $license_type = $this->option('type');
        $expiry_date = $this->option('expiry');
        $custom_code = $this->option('code');

        // Validate license type
        if (!in_array($license_type, ['single', 'multi', 'unlimited'])) {
            $this->error('Invalid license type. Must be: single, multi, or unlimited');
            return 1;
        }

        // Generate license code
        if ($custom_code) {
            $license_code = $custom_code;
        } else {
            $license_code = $this->generateLicenseCode($module_name, $username, $email, $license_type);
        }

        // Prepare license data
        $license_data = [
            'module_name' => $module_name,
            'username' => $username,
            'email' => $email,
            'license_code' => $license_code,
            'license_type' => $license_type,
            'expiry_date' => $expiry_date ? date('Y-m-d', strtotime($expiry_date)) : null,
            'status' => 'active',
            'created_at' => now(),
            'updated_at' => now(),
        ];

        try {
            // Store in database
            if (Schema::hasTable('module_licenses')) {
                // Check if license code already exists
                $existing = DB::table('module_licenses')
                    ->where('license_code', $license_code)
                    ->first();

                if ($existing) {
                    $this->error("License code {$license_code} already exists!");
                    return 1;
                }

                DB::table('module_licenses')->insert($license_data);
                $this->info("License stored in database.");
            } else {
                // Store in System properties as fallback
                System::addProperty('license_' . $module_name . '_' . $license_code, json_encode($license_data));
                $this->info("License stored in System properties (module_licenses table not found).");
            }

            // Display license information
            $this->info("\n=== License Generated Successfully ===");
            $this->line("Module: {$module_name}");
            $this->line("License Code: {$license_code}");
            $this->line("Username: {$username}");
            $this->line("Email: {$email}");
            $this->line("Type: {$license_type}");
            $this->line("Expiry: " . ($expiry_date ? date('Y-m-d', strtotime($expiry_date)) : 'Lifetime'));
            $this->line("Status: Active");
            $this->info("\nLicense code can now be used for module installation.");

            return 0;
        } catch (\Exception $e) {
            $this->error("Error generating license: " . $e->getMessage());
            return 1;
        }
    }

    /**
     * Generate license code
     */
    private function generateLicenseCode($module_name, $username, $email, $license_type)
    {
        $data = $module_name . $username . $email . $license_type . time() . rand(1000, 9999);
        $hash = hash('sha256', $data);
        
        // Format: BM-XXXX-XXXX-XXXX-XXXX (Business Management)
        $prefix = strtoupper(substr($module_name, 0, 2));
        $license_code = $prefix . '-' . 
                       strtoupper(substr($hash, 0, 4) . '-' . 
                                 substr($hash, 4, 4) . '-' . 
                                 substr($hash, 8, 4) . '-' . 
                                 substr($hash, 12, 4));
        
        return $license_code;
    }
}

