# Setup License Validation API Endpoint on dev.posifyme.com

## Current Status
❌ **API endpoint not found** - All tested endpoints return 404

## Solution Options

### Option 1: Set Up API Endpoint on Your Server (Recommended)

You need to create a license validation endpoint on `https://dev.posifyme.com/`.

#### Expected Endpoint Structure

**URL:** `https://dev.posifyme.com/api/license/validate` (or your preferred path)

**Method:** POST

**Request Parameters:**
```php
[
    'license_code' => 'BM-XXXX-XXXX-XXXX-XXXX',
    'username' => 'username',
    'email' => 'email@example.com',
    'product_id' => 25,
    'url' => 'http://customer-domain.com',
    'path' => '/path/to/module'
]
```

**Expected Response (JSON):**

**Valid License:**
```json
{
    "flag": "valid",
    "msg": "License is valid"
}
```

**Invalid License:**
```json
{
    "flag": "invalid",
    "msg": "Invalid license details"
}
```

#### Laravel Implementation Example

Create a route in your `routes/api.php`:
```php
Route::post('/license/validate', [LicenseController::class, 'validate']);
```

Create controller method:
```php
public function validate(Request $request)
{
    $request->validate([
        'license_code' => 'required',
        'username' => 'required',
        'product_id' => 'required',
    ]);

    // Check license in database
    $license = DB::table('module_licenses')
        ->where('license_code', $request->license_code)
        ->where('username', $request->username)
        ->where('module_name', 'businessmanagement') // or get from product_id
        ->where('status', 'active')
        ->first();

    if ($license) {
        // Check expiry
        if ($license->expiry_date && strtotime($license->expiry_date) < time()) {
            return response()->json([
                'flag' => 'invalid',
                'msg' => 'License has expired'
            ]);
        }

        return response()->json([
            'flag' => 'valid',
            'msg' => 'License is valid'
        ]);
    }

    return response()->json([
        'flag' => 'invalid',
        'msg' => 'Invalid license details'
    ]);
}
```

### Option 2: Use Local License Validation (Current Solution)

Since the API endpoint doesn't exist yet, the system **already uses local validation** as a fallback. This is working correctly!

**How it works:**
1. System first checks local `module_licenses` table
2. If license found and valid → Installation proceeds
3. If not found → Tries remote validation (will fail with 404)
4. Shows helpful error message

**Current Status:** ✅ **Local validation is working!**

## Update Configuration Once Endpoint is Ready

Once you set up the endpoint on your server:

1. **Get the endpoint URL** (e.g., `https://dev.posifyme.com/api/license/validate`)

2. **Encode it:**
   ```bash
   php -r "echo base64_encode('https://dev.posifyme.com/api/license/validate');"
   ```

3. **Update config file:**
   File: `Modules/BusinessManagement/Config/config.php`
   ```php
   'lic1' => 'BASE64_ENCODED_URL_HERE',
   ```

4. **Clear cache:**
   ```bash
   php artisan config:clear
   ```

5. **Test installation** with a license

## Current Workaround

**For now, use local license validation:**

1. ✅ Generate licenses via `/license-management`
2. ✅ Use those licenses during installation
3. ✅ System validates locally (no API needed)
4. ✅ Installation works perfectly!

The 404 error you see is expected since the API doesn't exist yet. The system will automatically fall back to local validation, which is working correctly.

## Next Steps

1. **Immediate:** Continue using local license validation (it's working!)
2. **Future:** Set up the API endpoint on your server
3. **Update config:** Once endpoint is ready, update the config file

