# UltimatePOS - Quick Reference Guide

## Permission Checks

```php
// Check permission
auth()->user()->can('permission.name')

// Check role
auth()->user()->hasRole('Admin#' . $business_id)

// Check multiple permissions
auth()->user()->hasAnyPermission(['perm1', 'perm2'])

// Check all permissions
auth()->user()->hasAllPermissions(['perm1', 'perm2'])
```

## Location Access

```php
// Get permitted locations
$locations = auth()->user()->permitted_locations();
// Returns: 'all' | [1, 2, 3]

// Check specific location
User::can_access_this_location($location_id, $business_id)

// Validate in controller
$permitted_locations = auth()->user()->permitted_locations();
if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
    abort(403, 'Unauthorized location access.');
}
```

## Stock Management

```php
// Location-based stock update
$this->productUtil->updateProductQuantity(
    $location_id,
    $product_id,
    $variation_id,
    $quantity_change,
    $old_quantity = 0,
    $new_quantity = 0,
    $type = 'purchase',
    $reference = null
);

// Warehouse-based stock update
$inventoryService = new \App\Utils\InventoryService();
$inventoryService->adjustStock($product_id, $variation_id, $warehouse_id, $quantity, $type, $reference, $notes);
```

## Common Queries

```php
// Get locations dropdown (auto-filters by permissions)
BusinessLocation::forDropdown($business_id, $show_all = false, $receipt_printer_type_attribute = false, $append_id = true, $check_permission = true);

// Get users dropdown
User::forDropdown($business_id, $prepend_none = true, $include_commission_agents = false, $prepend_all = false, $check_location_permission = false);

// Filter by permitted locations
$permitted_locations = auth()->user()->permitted_locations();
if ($permitted_locations != 'all') {
    $query->whereIn('location_id', $permitted_locations);
}
```

## Business Context

```php
// Get business ID
$business_id = request()->session()->get('user.business_id');
// or
$business_id = auth()->user()->business_id;

// Check if admin
$is_admin = $this->businessUtil->is_admin(auth()->user());
// or
$is_admin = auth()->user()->hasRole('Admin#' . $business_id);
```

## Transaction Types

- `purchase` - Purchase transaction
- `sell` - Sale transaction
- `opening_stock` - Opening stock entry
- `stock_adjustment` - Stock adjustment
- `stock_transfer` - Stock transfer between locations
- `expense` - Expense transaction
- `purchase_return` - Purchase return
- `sell_return` - Sale return

## Transaction Status

- `draft` - Draft transaction
- `final` - Finalized transaction
- `cancelled` - Cancelled transaction

## Product Types

- `single` - Single product
- `variable` - Variable product (with variations)
- `combo` - Combo product
- `modifier` - Modifier product

## Common Permissions

### Products
- `product.view`
- `product.create`
- `product.update`
- `product.delete`
- `product.opening_stock`
- `view_purchase_price`
- `edit_product_price_from_sale_screen`
- `edit_product_discount_from_sale_screen`

### Locations
- `access_all_locations`
- `location.{id}` (e.g., `location.1`)

### Transactions
- `sell.view`, `sell.create`, `sell.update`, `sell.delete`
- `purchase.view`, `purchase.create`, `purchase.update`, `purchase.delete`
- `sell.payments`, `purchase.payments`
- `view_own_sell_only`

### Reports
- `profit_loss_report.view`
- Module-specific (e.g., `accounting_reports.view`)

## Module Checks

```php
// Check if module installed
$this->moduleUtil->isModuleInstalled('ModuleName')

// Check module permission in subscription
$this->moduleUtil->hasThePermissionInSubscription($business_id, 'module_permission')
```

## Date Ranges

```php
// Get financial year
$fy = $this->businessUtil->getCurrentFinancialYear($business_id);
// Returns: ['start' => '2024-01-01', 'end' => '2024-12-31']

// Filter by date range
$query->whereBetween('transaction_date', [$start_date, $end_date]);
```

## Export Formats

```php
// Excel export
use Maatwebsite\Excel\Facades\Excel;
Excel::download($export, 'filename.xlsx');

// PDF export
use Barryvdh\DomPDF\Facade\Pdf;
return Pdf::loadView('view', $data)->download('filename.pdf');
```

## Error Handling

```php
// Permission denied
abort(403, 'Unauthorized action.');

// Location access denied
abort(403, 'Unauthorized location access.');

// Not found
abort(404, 'Resource not found.');
```

## Database Transactions

```php
DB::beginTransaction();
try {
    // Operations
    DB::commit();
} catch (\Exception $e) {
    DB::rollBack();
    throw $e;
}
```

## Eager Loading

```php
// Avoid N+1 queries
$products = Product::with(['variations', 'variations.variation_location_details', 'brand', 'category'])->get();
```

## Session Data

```php
// Get from session
$business_id = request()->session()->get('user.business_id');
$location_id = request()->session()->get('user.location_id');

// Set in session
request()->session()->put('key', 'value');
```


