# UltimatePOS - Project Architecture Overview

## Project Summary

**UltimatePOS** is a comprehensive Point of Sale (POS) system built on Laravel 9.x with a modular architecture. The system supports multi-business, multi-location inventory management, comprehensive reporting, and role-based access control.

### Key Technologies
- **Framework**: Laravel 9.51+
- **PHP**: 8.0+
- **Permission System**: Spatie Laravel Permission (v5.5)
- **Module System**: nWidart Laravel Modules (v9.0)
- **Database**: MySQL/MariaDB
- **Frontend**: Blade templates, jQuery, DataTables

---

## 1. Permission System Architecture

### 1.1 Core Components

The permission system uses **Spatie Laravel Permission** package with custom business-specific enhancements.

#### Key Models
- **User Model** (`app/User.php`): Uses `HasRoles` trait from Spatie
- **Roles**: Business-scoped (format: `RoleName#business_id`)
- **Permissions**: Global permissions with business context

#### Permission Tables
- `permissions` - Global permissions
- `roles` - Business-scoped roles (has `business_id` foreign key)
- `model_has_roles` - User-role assignments
- `model_has_permissions` - Direct user permissions
- `role_has_permissions` - Role-permission assignments

### 1.2 Location-Based Permissions

**Critical Pattern**: Location access is controlled via permissions:
- `access_all_locations` - Grants access to all locations
- `location.{location_id}` - Grants access to specific location (e.g., `location.1`, `location.2`)

#### Key Methods in User Model

```php
// Get permitted locations for user
$user->permitted_locations($business_id = null)
// Returns: 'all' | [1, 2, 3] (array of location IDs)

// Check if user can access specific location
User::can_access_this_location($location_id, $business_id = null)
// Returns: bool

// Scope to filter by permitted locations
$query->onlyPermittedLocations()
```

#### Permission Checking Patterns

**In Controllers:**
```php
// Check permission
if (!auth()->user()->can('product.view')) {
    abort(403);
}

// Check role
if (auth()->user()->hasRole('Admin#' . $business_id)) {
    // Admin access
}

// Validate location access
$permitted_locations = auth()->user()->permitted_locations();
if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
    abort(403, 'Unauthorized location access.');
}
```

**In AuthServiceProvider:**
```php
Gate::before(function ($user, $ability) {
    // Super admin bypass
    if (in_array($ability, ['backup', 'superadmin', 'manage_modules'])) {
        $administrator_list = config('constants.administrator_usernames');
        if (in_array(strtolower($user->username), explode(',', strtolower($administrator_list)))) {
            return true;
        }
    } else {
        // Business admin bypass
        if ($user->hasRole('Admin#' . $user->business_id)) {
            return true;
        }
    }
});
```

### 1.3 Common Permissions

**Product Permissions:**
- `product.view`, `product.create`, `product.update`, `product.delete`
- `product.opening_stock` - Access to opening stock management
- `view_purchase_price` - View purchase prices
- `edit_product_price_from_sale_screen` - Edit price during sale
- `edit_product_discount_from_sale_screen` - Edit discount during sale

**Location Permissions:**
- `access_all_locations` - Access all locations
- `location.{id}` - Access specific location

**Transaction Permissions:**
- `sell.view`, `sell.create`, `sell.update`, `sell.delete`
- `purchase.view`, `purchase.create`, `purchase.update`, `purchase.delete`
- `sell.payments`, `purchase.payments`
- `view_own_sell_only` - View only own sales

**Reporting Permissions:**
- `profit_loss_report.view`
- Module-specific permissions (e.g., `accounting_reports.view`)

---

## 2. Multi-Location Inventory Management

### 2.1 Core Models

#### BusinessLocation (`app/BusinessLocation.php`)
- Represents a physical store/location
- Belongs to a Business
- Has location-specific settings (printer, price groups, invoice schemes)

#### VariationLocationDetails (`app/VariationLocationDetails.php`)
- **Critical Table**: `variation_location_details`
- Stores stock quantity per variation per location
- Key fields:
  - `variation_id` - Product variation
  - `location_id` - Business location
  - `qty_available` - Available stock
  - `qty_reserved` - Reserved stock

#### StockMovement (`app/StockMovement.php`)
- Immutable ledger of all stock movements
- Tracks: purchase, sale, adjustment, transfer_in, transfer_out, return, etc.
- Links to source transaction via polymorphic relationship

### 2.2 Stock Management Flow

#### Location-Based Stock Updates

**Pattern 1: Using ProductUtil**
```php
// Update product quantity at location
$this->productUtil->updateProductQuantity(
    $location_id,
    $product_id,
    $variation_id,
    $quantity_change,
    $old_quantity = 0,
    $new_quantity = 0,
    $type = 'purchase',
    $reference = null
);
```

**Pattern 2: Using InventoryService (Warehouse-based)**
```php
$inventoryService = new \App\Utils\InventoryService();

// Add stock
$inventoryService->addStock($product_id, $variation_id, $warehouse_id, $quantity, $type, $reference, $notes);

// Remove stock
$inventoryService->removeStock($product_id, $variation_id, $warehouse_id, $quantity, $type, $reference, $notes);

// Adjust stock (with database locks)
$inventoryService->adjustStock($product_id, $variation_id, $warehouse_id, $quantity, $type, $reference, $notes);
```

### 2.3 Location Filtering Patterns

**In Queries:**
```php
// Get permitted locations
$permitted_locations = auth()->user()->permitted_locations();

// Filter by location
if ($permitted_locations != 'all') {
    $query->whereIn('location_id', $permitted_locations);
}

// Using scope
$query->whereHas('variation_location_details', function($q) use ($location_id) {
    $q->where('location_id', $location_id);
});
```

**In BusinessLocation:**
```php
// Get locations dropdown (auto-filters by permissions)
BusinessLocation::forDropdown($business_id, $show_all = false, $receipt_printer_type_attribute = false, $append_id = true, $check_permission = true);
```

### 2.4 Stock Transfer Between Locations

Stock transfers are handled via `Transaction` model with type `stock_transfer`:
- Source location: `location_id` on transaction
- Destination location: Stored in transaction details
- Stock movements created for both locations (transfer_out, transfer_in)

---

## 3. Reporting System

### 3.1 Reporting Modules

The system has multiple reporting modules:

#### Core Reporting (`app/Http/Controllers/ReportController.php`)
- Basic sales, purchase, stock reports
- Profit & Loss reports
- Tax reports

#### AccountingReports Module (`Modules/AccountingReports/`)
- **Tally-style double-entry accounting**
- 13+ financial reports:
  - Trial Balance
  - Balance Sheet
  - Profit & Loss (P&L)
  - Cash Flow
  - Receivables/Payables
  - Day Book
  - Ratio Analysis
- Features:
  - Double-entry bookkeeping (Dr = Cr)
  - FIFO costing
  - Period locking
  - Bank reconciliation
  - Multi-currency support

#### AdvancedReports Module (`Modules/AdvancedReports/`)
- 40+ professional reports
- Customer Analytics
- Sales Intelligence
- Inventory Management reports
- Financial Analysis
- Business Intelligence dashboards

### 3.2 Report Generation Patterns

**Location Filtering in Reports:**
```php
$permitted_locations = auth()->user()->permitted_locations();

if ($permitted_locations != 'all') {
    $query->whereIn('transactions.location_id', $permitted_locations);
}
```

**Date Range Filtering:**
```php
$start_date = request()->input('start_date');
$end_date = request()->input('end_date');

$query->whereBetween('transaction_date', [$start_date, $end_date]);
```

**Export Options:**
- PDF (using mPDF or DomPDF)
- Excel (using Maatwebsite Excel)
- CSV

---

## 4. Key Models and Relationships

### 4.1 Core Models

#### Business
- Top-level entity
- Has many: BusinessLocation, User, Product, Transaction

#### BusinessLocation
- Belongs to: Business
- Has many: Transactions, VariationLocationDetails

#### Product
- Belongs to: Business, Brand, Category, Unit
- Has many: ProductVariation, Variation, ProductLocation
- Types: `single`, `variable`, `combo`, `modifier`

#### Variation
- Belongs to: Product, ProductVariation
- Has many: VariationLocationDetails, TransactionSellLine, PurchaseLine
- Stores: SKU, prices, stock settings

#### VariationLocationDetails
- Belongs to: Variation, BusinessLocation
- Stores: `qty_available`, `qty_reserved` per location

#### Transaction
- Belongs to: Business, BusinessLocation, User (created_by)
- Types: `purchase`, `sell`, `opening_stock`, `stock_adjustment`, `stock_transfer`, `expense`, `purchase_return`, `sell_return`
- Status: `draft`, `final`, `cancelled`
- Has many: TransactionSellLine, TransactionPayment

#### User
- Belongs to: Business
- Uses: HasRoles trait (Spatie)
- Methods: `permitted_locations()`, `can_access_this_location()`

### 4.2 Key Relationships

```
Business
  ├── BusinessLocation (1:N)
  ├── User (1:N)
  ├── Product (1:N)
  └── Transaction (1:N)

Product
  ├── ProductVariation (1:N)
  │   └── Variation (1:N)
  │       └── VariationLocationDetails (1:N)
  │           └── BusinessLocation (N:1)
  └── ProductLocation (N:M)

Transaction
  ├── TransactionSellLine (1:N)
  │   └── Variation (N:1)
  ├── TransactionPayment (1:N)
  └── BusinessLocation (N:1)
```

---

## 5. Utility Classes

### 5.1 Core Utilities (`app/Utils/`)

#### ProductUtil
- Product creation/updates
- Variation management
- Stock quantity updates (location-based)
- Product alerts
- Barcode generation

#### TransactionUtil
- Transaction creation/updates
- Payment processing
- Invoice generation
- Stock updates on transactions

#### BusinessUtil
- Business settings
- Financial year management
- Currency handling
- Admin checks

#### ModuleUtil
- Module installation/management
- Permission checking for modules
- Subscription checks

#### InventoryService
- Warehouse-based stock management
- Stock movements tracking
- Database locking for race condition prevention

### 5.2 Common Patterns

#### Location Validation
```php
private function validateLocationAccess($location_id)
{
    if (empty($location_id)) {
        return null;
    }
    
    $permitted_locations = auth()->user()->permitted_locations();
    
    if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
        abort(403, 'Unauthorized location access.');
    }
    
    return $location_id;
}
```

#### Permission Checking
```php
// In controller constructor or methods
if (!auth()->user()->can('permission.name')) {
    abort(403, 'Unauthorized action.');
}
```

#### Business Context
```php
$business_id = request()->session()->get('user.business_id');
// or
$business_id = auth()->user()->business_id;
```

---

## 6. Module System

### 6.1 Module Structure

Modules are located in `Modules/` directory:
- Each module is self-contained
- Has: Models, Controllers, Views, Routes, Migrations, Seeders
- Uses nWidart Laravel Modules

### 6.2 Key Modules

- **Accounting** - Accounting features
- **AccountingReports** - Financial reports
- **AdvancedReports** - Advanced analytics
- **Crm** - Customer relationship management
- **Essentials** - Essential features
- **InventoryManagement** - Inventory tracking
- **Manufacturing** - Manufacturing processes
- **Project** - Project management
- **Repair** - Repair management
- **Superadmin** - Super admin features

### 6.3 Module Activation

Modules are activated via `modules_statuses.json`:
```json
{
  "Accounting": true,
  "AdvancedReports": true,
  ...
}
```

---

## 7. Database Schema Highlights

### 7.1 Key Tables

**Business & Locations:**
- `business` - Business entities
- `business_locations` - Physical locations
- `users` - System users

**Products & Inventory:**
- `products` - Product master
- `variations` - Product variations
- `variation_location_details` - Stock per location
- `product_locations` - Product-location mapping
- `stock_movements` - Stock movement ledger

**Transactions:**
- `transactions` - All transactions (sales, purchases, etc.)
- `transaction_sell_lines` - Sale line items
- `purchase_lines` - Purchase line items
- `transaction_payments` - Payment records

**Permissions:**
- `permissions` - Global permissions
- `roles` - Business-scoped roles
- `model_has_roles` - User-role assignments
- `role_has_permissions` - Role-permission assignments

### 7.2 Important Indexes

- `variation_location_details`: Indexed on `variation_id`, `location_id`
- `transactions`: Indexed on `business_id`, `location_id`, `type`, `status`, `transaction_date`
- `stock_movements`: Indexed on `business_id`, `product_id`, `variation_id`, `warehouse_id`, `type`

---

## 8. Best Practices & Patterns

### 8.1 Security

1. **Always check permissions** before allowing actions
2. **Validate location access** for location-specific operations
3. **Use business_id** from session/auth, never from user input
4. **Sanitize user input** before database queries
5. **Use transactions** for multi-step operations

### 8.2 Performance

1. **Eager load relationships** to avoid N+1 queries
2. **Use indexes** on frequently queried columns
3. **Cache permissions** (Spatie handles this automatically)
4. **Filter by permitted locations** early in queries

### 8.3 Code Organization

1. **Use Utility classes** for reusable business logic
2. **Keep controllers thin** - delegate to services/utils
3. **Use scopes** for common query patterns
4. **Follow Laravel conventions** for models, controllers, views

---

## 9. Common Tasks

### 9.1 Adding a New Permission

1. Create migration or seeder to add permission:
```php
Permission::create(['name' => 'new.permission', 'guard_name' => 'web']);
```

2. Check in controller:
```php
if (!auth()->user()->can('new.permission')) {
    abort(403);
}
```

### 9.2 Adding Location Filtering

```php
$permitted_locations = auth()->user()->permitted_locations();

$query = Model::query();

if ($permitted_locations != 'all') {
    $query->whereIn('location_id', $permitted_locations);
}
```

### 9.3 Updating Stock

```php
// Location-based
$this->productUtil->updateProductQuantity(
    $location_id,
    $product_id,
    $variation_id,
    $quantity_change,
    0,
    0,
    'purchase',
    $transaction
);

// Warehouse-based
$inventoryService->adjustStock($product_id, $variation_id, $warehouse_id, $quantity, 'purchase', $transaction);
```

---

## 10. Troubleshooting

### 10.1 Permission Issues

- Check if user has role assigned: `$user->getRoleNames()`
- Check if role has permission: `$role->hasPermissionTo('permission.name')`
- Clear permission cache: `php artisan permission:cache-reset`

### 10.2 Location Access Issues

- Verify `permitted_locations()` returns correct locations
- Check for `access_all_locations` permission
- Ensure location permissions format: `location.{id}`

### 10.3 Stock Discrepancies

- Check `stock_movements` table for audit trail
- Verify `variation_location_details` has correct quantities
- Use StockRecalculation module to recalculate stock

---

## Conclusion

This architecture supports:
- **Multi-tenant** (business-scoped)
- **Multi-location** inventory management
- **Role-based** access control with location restrictions
- **Modular** extensibility
- **Comprehensive** reporting capabilities

For specific implementation details, refer to the respective model, controller, or utility class files.


