<?php
namespace Modules\StockRecalculation\Utils\Concerns;

use Illuminate\Support\Facades\DB;

/**
 * Stock Integrity calculation trait
 * Formula: purchases + opening - finalized sells + sell_returns + stock_adjustments
 * (Transfers omitted for cross-schema compatibility.)
 */
trait StockIntegrity
{
    public function computeVldQty(int $business_id, int $variation_id, int $location_id): float
    {
        $received = (float) DB::table('purchase_lines as pl')
            ->join('transactions as t', 't.id', '=', 'pl.transaction_id')
            ->where('t.business_id', $business_id)
            ->whereIn('t.type', ['purchase', 'opening_stock'])
            ->where('t.location_id', $location_id)
            ->where('pl.variation_id', $variation_id)
            ->sum(DB::raw('(pl.quantity - IFNULL(pl.quantity_returned,0))'));

        $adjustments = (float) DB::table('stock_adjustment_lines as sal')
            ->join('transactions as t', 't.id', '=', 'sal.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('sal.variation_id', $variation_id)
            ->sum(DB::raw('IFNULL(sal.quantity,0)'));

        $sold = (float) DB::table('transaction_sell_lines as tsl')
            ->join('transactions as t', 't.id', '=', 'tsl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('tsl.variation_id', $variation_id)
            ->where('t.type', 'sell')->whereIn('t.status', ['final'])
            ->sum(DB::raw('IFNULL(tsl.quantity,0)'));

        $sell_returns = (float) DB::table('transaction_sell_lines as tsl')
            ->join('transactions as t', 't.id', '=', 'tsl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('tsl.variation_id', $variation_id)
            ->where('t.type', 'sell_return')
            ->sum(DB::raw('IFNULL(tsl.quantity,0)'));

        return (float) round(($received + $sell_returns + $adjustments) - $sold, 4);
    }

    public function syncVariationLocationQty(int $business_id, int $variation_id, int $location_id): void
    {
        $qty = $this->computeVldQty($business_id, $variation_id, $location_id);
        DB::table('variation_location_details as vld')
            ->where('vld.variation_id', $variation_id)
            ->where('vld.location_id', $location_id)
            ->update(['qty_available' => $qty]);
    }
}