# Stock Recalculation Module - Technical Documentation

## Table of Contents
1. [Architecture Overview](#architecture-overview)
2. [Database Schema](#database-schema)
3. [API Specifications](#api-specifications)
4. [Performance Optimizations](#performance-optimizations)
5. [Security Implementation](#security-implementation)
6. [Error Handling](#error-handling)
7. [Deployment Guide](#deployment-guide)
8. [Monitoring & Maintenance](#monitoring--maintenance)

## Architecture Overview

### System Architecture
The Stock Recalculation Module follows a modular Laravel architecture with the following components:

```
Modules/StockRecalculation/
├── Http/Controllers/          # Application logic layer
├── Console/Commands/          # CLI interface
├── Database/Migrations/       # Database schema management
├── Utils/Concerns/           # Reusable traits and utilities
├── Resources/views/          # Frontend interface
├── Routes/                   # API and web routing
└── Providers/               # Service registration
```

### Design Patterns Used
- **MVC Pattern**: Separation of concerns with controllers, models, and views
- **Trait Pattern**: `StockIntegrity` trait for reusable calculation logic
- **Command Pattern**: Artisan commands for CLI operations
- **Observer Pattern**: Progress tracking with real-time updates
- **Factory Pattern**: Dynamic filter and query building

## Database Schema

### Core Tables
The module interacts with existing UltimatePOS tables:

#### Primary Tables
- `variation_location_details` - Stores current stock quantities
- `transactions` - All business transactions
- `purchase_lines` - Purchase transaction details
- `transaction_sell_lines` - Sales transaction details
- `stock_adjustment_lines` - Manual stock adjustments
- `business_locations` - Business location information
- `variations` - Product variations
- `products` - Product master data

### Database Views

#### vw_vld_vs_calculated
Compares stored quantities with calculated quantities:
```sql
CREATE VIEW vw_vld_vs_calculated AS
SELECT
    bl.business_id,
    vld.location_id,
    vld.variation_id,
    vld.qty_available AS vld_qty,
    ROUND([calculated_stock_formula], 4) AS calculated_qty
FROM variation_location_details vld
JOIN business_locations bl ON bl.id = vld.location_id;
```

#### vw_stock_integrity_audit
Comprehensive audit view with product information:
```sql
CREATE VIEW vw_stock_integrity_audit AS
SELECT
    bl.business_id,
    vld.location_id,
    vld.variation_id,
    bl.name as location_name,
    p.name as product_name,
    v.name as variation_name,
    vld.qty_available AS stored_qty,
    [calculated_stock_formula] AS calculated_qty,
    [difference_calculation] AS difference
FROM variation_location_details vld
JOIN business_locations bl ON bl.id = vld.location_id
JOIN variations v ON v.id = vld.variation_id
JOIN products p ON p.id = v.product_id;
```

### Performance Indexes
Critical indexes for optimal performance:
```sql
-- Transaction processing indexes
CREATE INDEX idx_pl_variation ON purchase_lines (variation_id);
CREATE INDEX idx_t_business_type_loc ON transactions (business_id, type, location_id);
CREATE INDEX idx_tsl_variation ON transaction_sell_lines (variation_id);
CREATE INDEX idx_sal_variation ON stock_adjustment_lines (variation_id);
CREATE INDEX idx_vld_vl ON variation_location_details (variation_id, location_id);
```

## API Specifications

### RESTful Endpoints

#### Authentication & Authorization
All endpoints require:
- Valid user session authentication
- Appropriate permission levels
- CSRF token for POST requests

#### Core Endpoints

##### GET /stock-recalculation/
**Purpose**: Main dashboard interface
**Permissions**: `StockRecalculation.view`
**Response**: HTML dashboard with summary cards and controls

##### POST /stock-recalculation/verify
**Purpose**: Verify stock discrepancies
**Permissions**: `StockRecalculation.view`
**Parameters**:
```json
{
  "business_id": "integer",
  "location_id": "integer|optional",
  "product_id": "integer|optional"
}
```
**Response**: Rendered results view with discrepancy list

##### POST /stock-recalculation/backup
**Purpose**: Create data backup
**Permissions**: `StockRecalculation.backup`
**Response**:
```json
{
  "success": true,
  "backup_table": "vld_backup_1_2024_01_15_10_30_45",
  "records_backed_up": 15000,
  "message": "Backup created successfully"
}
```

##### POST /stock-recalculation/start-recalculation
**Purpose**: Initialize batch recalculation
**Permissions**: `StockRecalculation.recalculate`
**Response**:
```json
{
  "success": true,
  "message": "Recalculation started",
  "progress": {
    "status": "starting",
    "total_records": 10000,
    "processed_records": 0,
    "current_batch": 0,
    "total_batches": 100
  },
  "batch_url": "/stock-recalculation/process-batch"
}
```

##### POST /stock-recalculation/process-batch
**Purpose**: Process individual batch
**Permissions**: `StockRecalculation.recalculate`
**Response**:
```json
{
  "success": true,
  "progress": {
    "status": "running",
    "processed_records": 100,
    "updated_records": 85,
    "current_batch": 1,
    "estimated_time": "2024-01-15T10:35:00"
  }
}
```

#### Debug & Testing Endpoints

##### GET /stock-recalculation/debug-calculation
**Parameters**: `variation_id`, `location_id`
**Response**: Detailed calculation breakdown

##### GET /stock-recalculation/test-calculation
**Parameters**: `variation_id`, `location_id`
**Response**: Comparison between calculation methods

##### GET /stock-recalculation/sample-variations
**Parameters**: `location_id`
**Response**: Sample variations for testing

## Performance Optimizations

### Database Optimizations

#### Query Optimization
1. **Indexed Views**: Pre-computed calculations reduce runtime complexity
2. **Selective Indexes**: Strategic indexing on high-frequency join columns
3. **Batch Processing**: Configurable batch sizes prevent memory exhaustion
4. **Lazy Loading**: Progressive data loading for large datasets

#### Memory Management
```php
// Configuration constants
const BATCH_SIZE = 100;           // Records per batch
const MAX_EXECUTION_TIME = 25;    // Seconds per batch
const MEMORY_LIMIT = '512M';      // Memory allocation
const PROGRESS_CACHE_DURATION = 3600; // Cache duration

// Memory optimization in processing
if ($processed % 50 === 0) {
    if (function_exists('gc_collect_cycles')) {
        gc_collect_cycles();
    }
}
```

### Application-Level Optimizations

#### Caching Strategy
- **Progress Caching**: Redis/database caching for progress tracking
- **Query Result Caching**: Temporary caching of expensive calculations
- **Session Caching**: User-specific filter and preference caching

#### Asynchronous Processing
- **Batch Processing**: Non-blocking batch execution
- **Progress Polling**: Real-time updates without blocking operations
- **Cancellation Support**: Graceful termination of long-running processes

### Frontend Optimizations
- **Progressive Loading**: Incremental result display
- **AJAX Polling**: Efficient progress updates
- **DOM Optimization**: Minimal DOM manipulation for large datasets

## Security Implementation

### Authentication & Authorization

#### Permission System
```php
// Controller permission checks
if (!auth()->user()->can('StockRecalculation.view')) {
    abort(403, 'Unauthorized access');
}

// Available permissions
- StockRecalculation.view      // Dashboard and verification access
- StockRecalculation.backup    // Backup creation rights
- StockRecalculation.recalculate // Recalculation execution rights
```

#### Business Data Isolation
```php
// Automatic business_id filtering
$business_id = session()->get('user.business_id');
$query->where('business_id', $business_id);
```

### Input Validation & Sanitization

#### Request Validation
```php
// Parameter sanitization
$business_id = (int) $request->business_id;
$location_id = $request->location_id ? (int) $request->location_id : null;
$product_id = $request->product_id ? (int) $request->product_id : null;
```

#### SQL Injection Prevention
- **Parameterized Queries**: All database queries use parameter binding
- **Type Casting**: Strict type enforcement for numeric parameters
- **Query Builder**: Laravel's query builder for safe query construction

### CSRF Protection
```php
// Blade template CSRF tokens
@csrf

// JavaScript AJAX headers
headers: {
    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
}
```

## Error Handling

### Exception Management

#### Graceful Degradation
```php
try {
    $this->syncVariationLocationQty($business_id, $variation_id, $location_id);
    $updated++;
} catch (\Exception $e) {
    Log::error("Error processing variation_location_details ID {$record->id}: " . $e->getMessage());
    $processed++; // Continue processing despite individual failures
}
```

#### Error Categorization
1. **System Errors**: Database connectivity, memory limits
2. **Business Logic Errors**: Invalid data states, calculation failures
3. **User Errors**: Permission issues, invalid parameters
4. **Network Errors**: Timeout, connection failures

### Logging Strategy

#### Comprehensive Logging
```php
// Start/completion logging
Log::info('StockRecalculation: Completed by user', [
    'user_id' => auth()->id(),
    'business_id' => $business_id,
    'total_processed' => $progress['processed_records'],
    'total_updated' => $progress['updated_records'],
    'duration_seconds' => now()->diffInSeconds($progress['start_time'])
]);

// Error logging with context
Log::error('StockRecalculation Batch error: ' . $e->getMessage(), [
    'user_id' => auth()->id(),
    'business_id' => $business_id,
    'batch_number' => $currentBatch,
    'trace' => $e->getTraceAsString()
]);
```

#### Log Levels
- **INFO**: Normal operations, completion events
- **WARNING**: Non-critical issues, performance concerns
- **ERROR**: Processing failures, system errors
- **DEBUG**: Detailed calculation breakdowns (development only)

## Deployment Guide

### System Requirements

#### Server Requirements
- **PHP**: 8.0+ with required extensions
- **Database**: MySQL 5.7+ or MariaDB 10.2+
- **Memory**: Minimum 512MB PHP memory limit
- **Storage**: Adequate space for backup tables
- **Permissions**: Database DDL/DML permissions

#### Laravel Requirements
- **Framework**: Compatible with UltimatePOS Laravel version
- **Extensions**: PHP extensions (mysql, mbstring, openssl, etc.)
- **Cache**: Redis or database cache driver recommended

### Installation Process

#### 1. Module Installation
```bash
# Copy module files
cp -r StockRecalculation/ /path/to/ultimatepos/Modules/

# Set permissions
chmod -R 755 Modules/StockRecalculation/
chown -R www-data:www-data Modules/StockRecalculation/
```

#### 2. Database Migration
```bash
# Run migrations
php artisan migrate --path=Modules/StockRecalculation/Database/Migrations

# Verify views created
mysql -u user -p database_name -e "SHOW FULL TABLES LIKE 'vw_%';"
```

#### 3. Permission Configuration
```sql
-- Add permissions to role_has_permissions table
INSERT INTO permissions (name, guard_name) VALUES
('StockRecalculation.view', 'web'),
('StockRecalculation.backup', 'web'),
('StockRecalculation.recalculate', 'web');
```

#### 4. Cache Configuration
```php
// config/cache.php - ensure proper cache driver
'default' => env('CACHE_DRIVER', 'redis'),

// .env configuration
CACHE_DRIVER=redis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
```

### Configuration Files

#### Module Configuration
```php
// Config/config.php
return [
    'name' => 'StockRecalculation',
    'batch_size' => env('STOCK_RECALC_BATCH_SIZE', 100),
    'max_execution_time' => env('STOCK_RECALC_MAX_TIME', 25),
    'memory_limit' => env('STOCK_RECALC_MEMORY', '512M'),
    'progress_cache_duration' => env('STOCK_RECALC_CACHE_DURATION', 3600),
];
```

## Monitoring & Maintenance

### Performance Monitoring

#### Key Metrics
1. **Processing Speed**: Records per second
2. **Memory Usage**: Peak memory consumption
3. **Database Performance**: Query execution times
4. **Error Rates**: Failed batch processing percentage
5. **User Activity**: Module usage patterns

#### Monitoring Queries
```sql
-- Check view performance
EXPLAIN SELECT * FROM vw_stock_integrity_audit WHERE business_id = 1;

-- Monitor backup table sizes
SELECT table_name,
       ROUND(((data_length + index_length) / 1024 / 1024), 2) AS 'Size (MB)'
FROM information_schema.tables
WHERE table_name LIKE 'vld_backup_%'
ORDER BY (data_length + index_length) DESC;

-- Transaction volume analysis
SELECT DATE(created_at) as date,
       COUNT(*) as transaction_count,
       AVG(TIMESTAMPDIFF(SECOND, created_at, updated_at)) as avg_duration
FROM transactions
WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
GROUP BY DATE(created_at)
ORDER BY date DESC;
```

### Maintenance Tasks

#### Regular Maintenance
1. **Backup Cleanup**: Remove old backup tables
2. **Log Rotation**: Archive and compress log files
3. **Index Maintenance**: Rebuild fragmented indexes
4. **Cache Cleanup**: Clear expired cache entries
5. **Performance Analysis**: Review slow query logs

#### Backup Table Management
```sql
-- List backup tables
SELECT table_name, create_time,
       ROUND(((data_length + index_length) / 1024 / 1024), 2) AS 'Size (MB)'
FROM information_schema.tables
WHERE table_name LIKE 'vld_backup_%'
ORDER BY create_time DESC;

-- Cleanup old backups (older than 30 days)
SET @sql = NULL;
SELECT GROUP_CONCAT('DROP TABLE ', table_name SEPARATOR '; ')
INTO @sql
FROM information_schema.tables
WHERE table_name LIKE 'vld_backup_%'
AND table_name < CONCAT('vld_backup_', DATE_FORMAT(DATE_SUB(NOW(), INTERVAL 30 DAY), '%Y_%m_%d'));

PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;
```

### Troubleshooting Guide

#### Common Issues

##### 1. Memory Exhaustion
**Symptoms**: PHP fatal error, process termination
**Solutions**:
- Reduce batch size: `STOCK_RECALC_BATCH_SIZE=50`
- Increase memory limit: `STOCK_RECALC_MEMORY=1024M`
- Enable garbage collection optimization

##### 2. Database Lock Timeouts
**Symptoms**: Lock wait timeout errors
**Solutions**:
- Reduce batch processing frequency
- Optimize database indexes
- Consider read replicas for reporting

##### 3. Progress Tracking Issues
**Symptoms**: Progress not updating, stale data
**Solutions**:
- Clear cache: `php artisan cache:clear`
- Verify cache driver configuration
- Check Redis/cache server connectivity

##### 4. Permission Errors
**Symptoms**: 403 unauthorized errors
**Solutions**:
- Verify user role assignments
- Check permission database entries
- Validate session data integrity

### Disaster Recovery

#### Backup Strategy
1. **Database Backups**: Regular mysqldump with rotation
2. **Application Backups**: Module file versioning
3. **Configuration Backups**: Environment and config files
4. **Log Archival**: Compressed log file storage

#### Recovery Procedures
1. **Data Recovery**: Restore from backup tables if needed
2. **Service Recovery**: Restart failed processing from last checkpoint
3. **Rollback Procedures**: Restore previous stock quantities if required
4. **Audit Trail**: Maintain complete transaction history for forensics