@extends('layouts.app')
@section('title', __('stockrecalculation::lang.stock_recalculation'))

@section('css')
<style>
    /* Make widget values white */
    .small-box .inner h3 {
        color: white !important;
    }

    .small-box .inner p {
        color: rgba(255, 255, 255, 0.8) !important;
    }
</style>
@endsection

@section('content')

<!-- Content Header (Page header) -->
<section class="content-header">
    <h1 class="tw-text-xl md:tw-text-3xl tw-font-bold tw-text-black">{{
        __('stockrecalculation::lang.stock_recalculation') }}
        <small class="tw-text-sm md:tw-text-base tw-text-gray-700 tw-font-semibold">{{
            __('stockrecalculation::lang.module_description') }}</small>
    </h1>
</section>

<!-- Main content -->
<section class="content">
    <div class="row">
        <div class="col-md-12">
            <!-- Summary Cards Row -->
            <div class="row" id="summary-cards">
                <div class="col-lg-3 col-xs-6">
                    <div class="info-box">
                        <span class="info-box-icon bg-aqua"><i class="fa fa-cubes"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text"
                                style="display: block; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; width: 100%;">
                                {{ __('stockrecalculation::lang.total_products') }}
                            </span>
                            <span class="info-box-number" id="total-products"></span>
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-xs-6">
                    <div class="info-box">
                        <span class="info-box-icon bg-yellow"><i class="fa fa-exclamation-triangle"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text"
                                style="display: block; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; width: 100%;">
                                {{ __('stockrecalculation::lang.discrepancies') }}
                            </span>
                            <span class="info-box-number" id="total-discrepancies"></span>
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-xs-6">
                    <div class="info-box">
                        <span class="info-box-icon bg-red"><i class="fa fa-skull"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text"
                                style="display: block; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; width: 100%;">
                                {{ __('stockrecalculation::lang.critical_issues') }}
                            </span>
                            <span class="info-box-number" id="critical-issues"></span>
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-xs-6">
                    <div class="info-box">
                        <span class="info-box-icon bg-green"><i class="fa fa-check-circle"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text"
                                style="display: block; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; width: 100%;">
                                {{ __('stockrecalculation::lang.avg_difference') }}
                            </span>
                            <span class="info-box-number" id="avg-difference"></span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Filters Card -->
            <div class="box box-primary no-print">
                <div class="box-header with-border">
                    <h3 class="box-title">
                        <i class="fa fa-filter"></i> {{ __('stockrecalculation::lang.filters') }}
                    </h3>
                </div>
                <div class="box-body">
                    <form id="stock-filters">
                        @csrf
                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="business_id">{{ __('stockrecalculation::lang.business') }}:</label>
                                    <select class="form-control select2" id="business_id" name="business_id">
                                        @foreach($businesses as $business)
                                        <option value="{{ $business->id }}">{{ $business->name }}</option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="location_id">{{ __('stockrecalculation::lang.location') }}:</label>
                                    <select class="form-control select2" id="location_id" name="location_id">
                                        <option value="">{{ __('stockrecalculation::lang.all_locations') }}</option>
                                        @foreach($locations as $location)
                                        <option value="{{ $location->id }}">{{ $location->name }}</option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="product_id">{{ __('stockrecalculation::lang.product_id') }} ({{
                                        __('lang_v1.optional') }}):</label>
                                    <input type="number" class="form-control" id="product_id" name="product_id"
                                        placeholder="{{ __('stockrecalculation::lang.enter_product_id') }}">
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Actions Card -->
            <div class="box box-warning no-print">
                <div class="box-header with-border">
                    <h3 class="box-title">
                        <i class="fa fa-cogs"></i> {{ __('stockrecalculation::lang.actions') }}
                    </h3>
                </div>
                <div class="box-body">
                    <div class="row">
                        <div class="col-md-12">
                            <div class="alert alert-warning">
                                <strong><i class="fa fa-exclamation-triangle"></i> {{
                                    __('stockrecalculation::lang.backup_warning') }}</strong>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-3">
                            <button type="button" class="btn btn-info btn-block" id="verify-btn">
                                <i class="fa fa-search"></i> {{ __('stockrecalculation::lang.step_1') }}
                            </button>
                            <small class="text-muted">{{ __('stockrecalculation::lang.step_1_desc') }}</small>
                        </div>
                        <div class="col-md-3">
                            <button type="button" class="btn btn-primary btn-block" id="backup-btn">
                                <i class="fa fa-database"></i> {{ __('stockrecalculation::lang.step_2') }}
                            </button>
                            <small class="text-muted">{{ __('stockrecalculation::lang.step_2_desc') }}</small>
                        </div>
                        <div class="col-md-3">
                            <button type="button" class="btn btn-success btn-block" id="recalculate-btn" disabled>
                                <i class="fa fa-refresh"></i> {{ __('stockrecalculation::lang.step_3') }}
                            </button>
                            <small class="text-muted">{{ __('stockrecalculation::lang.step_3_desc') }}</small>
                        </div>
                        <div class="col-md-3">
                            <button type="button" class="btn btn-warning btn-block" id="verify-after-btn" disabled>
                                <i class="fa fa-check"></i> {{ __('stockrecalculation::lang.step_4') }}
                            </button>
                            <small class="text-muted">{{ __('stockrecalculation::lang.step_4_desc') }}</small>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Results Card -->
            <div class="box box-success" id="results-box" style="display: none;">
                <div class="box-header with-border">
                    <h3 class="box-title">
                        <i class="fa fa-list"></i> {{ __('stockrecalculation::lang.results') }}
                    </h3>
                </div>
                <div class="box-body" id="results-content">
                    <!-- Results will be loaded here -->
                </div>
            </div>

            <!-- Progress Card -->
            <div class="box box-info" id="progress-box" style="display: none;">
                <div class="box-header with-border">
                    <h3 class="box-title">
                        <i class="fa fa-spinner fa-spin"></i> <span id="progress-title">{{
                            __('stockrecalculation::lang.processing') }}</span>
                    </h3>
                    <div class="box-tools pull-right">
                        <button type="button" class="btn btn-danger btn-sm" id="cancel-btn" style="display: none;">
                            <i class="fa fa-stop"></i> {{ __('messages.cancel') }}
                        </button>
                    </div>
                </div>
                <div class="box-body">
                    <div id="progress-message">{{ __('stockrecalculation::lang.processing') }}</div>
                    <div class="progress progress-sm">
                        <div class="progress-bar progress-bar-striped active" id="progress-bar" style="width: 0%"></div>
                    </div>
                    <div class="row text-center" id="progress-stats" style="margin-top: 15px; display: none;">
                        <div class="col-md-3">
                            <strong id="processed-count">0</strong><br>
                            <small class="text-muted">Processed</small>
                        </div>
                        <div class="col-md-3">
                            <strong id="updated-count">0</strong><br>
                            <small class="text-muted">Updated</small>
                        </div>
                        <div class="col-md-3">
                            <strong id="current-batch">0</strong> / <strong id="total-batches">0</strong><br>
                            <small class="text-muted">Batches</small>
                        </div>
                        <div class="col-md-3">
                            <strong id="estimated-time">-</strong><br>
                            <small class="text-muted">ETA</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

@endsection

@section('javascript')
<script>
    $(document).ready(function() {
    let backupCreated = false;
    
    // Load summary data on page load
    loadSummary();

    // Verify discrepancies
    $('#verify-btn').click(function() {
        showProgress('{{ __('stockrecalculation::lang.checking_for_discrepancies') }}');
        
        $.post('{{ route("stock-recalculation.verify") }}', $('#stock-filters').serialize(), function(data) {
            $('#results-content').html(data);
            $('#results-box').show();
            hideProgress();
        }).fail(function(xhr) {
            swal('{{ __('stockrecalculation::lang.error') }}', xhr.responseText, 'error');
            hideProgress();
        });
    });

    // Create backup
    $('#backup-btn').click(function() {
        swal({
            title: '{{ __('stockrecalculation::lang.create_backup') }}',
            text: '{{ __('stockrecalculation::lang.create_backup_info') }}',
            icon: 'info',
            buttons: {
                cancel: '{{ __('messages.cancel') }}',
                confirm: '{{ __('messages.yes') }}'
            }
        }).then(function(confirmed) {
            if (confirmed) {
                showProgress('{{ __('stockrecalculation::lang.creating_backup') }}');
                
                $.post('{{ route("stock-recalculation.backup") }}', $('#stock-filters').serialize(), function(response) {
                    if (response.success) {
                        swal({
                            title: '{{ __('stockrecalculation::lang.backup_created') }}',
                            text: 'Table: ' + response.backup_table + ', Records: ' + (response.records_backed_up || 'All'),
                            icon: 'success'
                        });
                        backupCreated = true;
                        $('#recalculate-btn').prop('disabled', false);
                    } else {
                        swal('{{ __('stockrecalculation::lang.backup_failed') }}', response.message, 'error');
                    }
                    hideProgress();
                }).fail(function(xhr) {
                    swal('{{ __('stockrecalculation::lang.error') }}', xhr.responseText, 'error');
                    hideProgress();
                });
            }
        });
    });

    // Recalculate stocks with batch processing
    $('#recalculate-btn').click(function() {
        if (!backupCreated) {
            swal('{{ __('stockrecalculation::lang.backup_required') }}', '{{ __('stockrecalculation::lang.backup_required_info') }}', 'warning');
            return;
        }
        
        swal({
            title: '{{ __('stockrecalculation::lang.start_batch_recalculation') }}',
            text: '{{ __('stockrecalculation::lang.start_batch_recalculation_info') }}',
            icon: 'info',
            buttons: {
                cancel: '{{ __('messages.cancel') }}',
                confirm: '{{ __('messages.yes') }}'
            }
        }).then(function(confirmed) {
            if (confirmed) {
                startBatchRecalculation();
            }
        });
    });

    // Cancel recalculation
    $('#cancel-btn').click(function() {
        swal({
            title: '{{ __('stockrecalculation::lang.cancel_recalculation') }}',
            text: '{{ __('stockrecalculation::lang.cancel_recalculation_info') }}',
            icon: 'warning',
            buttons: {
                cancel: '{{ __('messages.no') }}',
                confirm: '{{ __('messages.yes') }}'
            }
        }).then(function(confirmed) {
            if (confirmed) {
                cancelRecalculation();
            }
        });
    });

    let currentProgress = null;
    let progressInterval = null;
    
    // Check for existing progress on page load
    @if(isset($currentProgress) && $currentProgress)
        currentProgress = @json($currentProgress);
        if (currentProgress && (currentProgress.status === 'running' || currentProgress.status === 'starting')) {
            showProgressWithStats('Resuming batch recalculation...', currentProgress);
            startProgressPolling();
        }
    @endif

    function startBatchRecalculation() {
        $.post('{{ route("stock-recalculation.start-recalculation") }}', $('#stock-filters').serialize(), function(response) {
            if (response.success) {
                currentProgress = response.progress;
                showProgressWithStats('Starting batch recalculation...', currentProgress);
                startProgressPolling();
                processBatch(response.batch_url);
            } else {
                swal('{{ __('stockrecalculation::lang.recalculation_failed') }}', response.message, 'error');
            }
        }).fail(function(xhr) {
            swal('{{ __('stockrecalculation::lang.error') }}', xhr.responseText, 'error');
        });
    }

    function processBatch(batchUrl) {
        if (!currentProgress || currentProgress.status === 'error' || currentProgress.status === 'completed') {
            return;
        }

        $.post(batchUrl, $('#stock-filters').serialize(), function(response) {
            if (response.success) {
                currentProgress = response.progress;
                updateProgressDisplay(currentProgress);
                
                if (currentProgress.status === 'completed') {
                    completeRecalculation();
                } else if (currentProgress.status === 'running') {
                    // Process next batch after short delay
                    setTimeout(function() {
                        processBatch(batchUrl);
                    }, 100);
                }
            } else {
                swal('{{ __('stockrecalculation::lang.batch_failed') }}', response.message, 'error');
                hideProgress();
                stopProgressPolling();
            }
        }).fail(function(xhr) {
            swal('{{ __('stockrecalculation::lang.error') }}', xhr.responseText, 'error');
            hideProgress();
            stopProgressPolling();
        });
    }

    function startProgressPolling() {
        progressInterval = setInterval(function() {
            $.get('{{ route("stock-recalculation.progress") }}', function(response) {
                if (response.progress) {
                    currentProgress = response.progress;
                    updateProgressDisplay(currentProgress);
                    
                    if (currentProgress.status === 'completed' || currentProgress.status === 'error') {
                        stopProgressPolling();
                        if (currentProgress.status === 'completed') {
                            completeRecalculation();
                        }
                    }
                }
            });
        }, 2000); // Poll every 2 seconds
    }

    function stopProgressPolling() {
        if (progressInterval) {
            clearInterval(progressInterval);
            progressInterval = null;
        }
    }

    function cancelRecalculation() {
        $.post('{{ route("stock-recalculation.cancel-recalculation") }}', function(response) {
            if (response.success) {
                hideProgress();
                stopProgressPolling();
                currentProgress = null;
                swal('{{ __('restaurant.cancelled') }}', '{{ __('stockrecalculation::lang.recalculation_cancelled') }}', 'info');
            }
        });
    }

    function completeRecalculation() {
        stopProgressPolling();
        hideProgress();
        $('#verify-after-btn').prop('disabled', false);
        loadSummary(); // Refresh summary
        
        const duration = currentProgress.completion_time ? 
            Math.round((new Date(currentProgress.completion_time) - new Date(currentProgress.start_time)) / 1000) : 0;
        
        const message = 'Records Processed: ' + currentProgress.processed_records + ', ' +
            'Records Updated: ' + currentProgress.updated_records +
            (duration > 0 ? ', Completed in ' + duration + ' seconds' : '') +
            '. You can now verify the results.';
        
        swal({
            title: '{{ __('stockrecalculation::lang.recalculation_completed_successfully') }}',
            text: message,
            icon: 'success'
        });
        
        currentProgress = null;
    }

    // Verify after recalculation
    $('#verify-after-btn').click(function() {
        $('#verify-btn').click(); // Reuse verify functionality
    });

    function showProgress(message) {
        $('#progress-message').text(message);
        $('#progress-box').show();
        $('#progress-bar').css('width', '100%');
        $('#progress-stats').hide();
        $('#cancel-btn').hide();
    }

    function showProgressWithStats(message, progress) {
        $('#progress-title').text('{{ __('stockrecalculation::lang.batch_processing') }}');
        $('#progress-message').text(message);
        $('#progress-box').show();
        $('#progress-stats').show();
        $('#cancel-btn').show();
        updateProgressDisplay(progress);
    }

    function updateProgressDisplay(progress) {
        if (!progress) return;
        
        const percentage = progress.total_records > 0 ? 
            Math.round((progress.processed_records / progress.total_records) * 100) : 0;
        
        $('#progress-bar').css('width', percentage + '%');
        $('#processed-count').text(progress.processed_records || 0);
        $('#updated-count').text(progress.updated_records || 0);
        $('#current-batch').text(progress.current_batch || 0);
        $('#total-batches').text(progress.total_batches || 0);
        
        if (progress.estimated_time) {
            const eta = new Date(progress.estimated_time);
            $('#estimated-time').text(eta.toLocaleTimeString());
        } else {
            $('#estimated-time').text('-');
        }
        
        const status = progress.status || 'unknown';
        let message = '';
        switch (status) {
            case 'starting':
                message = 'Initializing batch processing...';
                break;
            case 'running':
                message = `Processing batch ${progress.current_batch + 1} of ${progress.total_batches}...`;
                break;
            case 'completed':
                message = `Completed! Processed ${progress.processed_records} records.`;
                break;
            case 'error':
                message = 'Processing encountered an error.';
                break;
            default:
                message = '{{ __('stockrecalculation::lang.processing') }}';
        }
        $('#progress-message').text(message);
    }

    function hideProgress() {
        $('#progress-box').hide();
        $('#progress-bar').css('width', '0%');
        $('#progress-stats').hide();
        $('#cancel-btn').hide();
    }

    function loadSummary() {
        $.get('{{ route("stock-recalculation.summary") }}', function(data) {
            $('#total-products').text(data.total_products || 0);
            $('#total-discrepancies').text(data.discrepancies || 0);
            $('#critical-issues').text(data.critical_issues || 0);
            $('#avg-difference').text(data.avg_difference ? parseFloat(data.avg_difference).toFixed(2) : '0.00');
        }).fail(function() {
            console.log('Failed to load summary data');
        });
    }
});
</script>
@endsection