# Stock Recalculation Module - Product Requirements Document (PRD)

## Executive Summary

The Stock Recalculation Module is a professional-grade inventory management tool designed for UltimatePOS systems. It provides comprehensive stock integrity verification and automated recalculation capabilities to ensure accurate inventory tracking across business locations.

## Module Overview

**Module Name:** StockRecalculation
**Alias:** stockrecalculation
**Version:** 1.0
**Type:** Laravel Module for UltimatePOS
**Dependencies:** Core UltimatePOS system

## Key Features

### 1. Stock Integrity Verification
- **Discrepancy Detection**: Identifies differences between stored stock quantities and calculated quantities based on transactions
- **Audit View Integration**: Uses database views for efficient stock integrity checks
- **Real-time Analysis**: Provides instant verification results with detailed breakdown
- **Critical Issue Identification**: Highlights significant discrepancies requiring immediate attention

### 2. Batch Processing Engine
- **Scalable Processing**: Handles large datasets with configurable batch sizes (default: 100 records)
- **Memory Management**: Optimized to prevent memory exhaustion during processing
- **Progress Tracking**: Real-time progress monitoring with estimated completion times
- **Cancellation Support**: Ability to cancel long-running operations safely

### 3. Data Backup & Recovery
- **Automated Backup**: Creates timestamped backup tables before recalculation
- **Batch Backup Processing**: Handles large datasets without timeout issues
- **Recovery Support**: Maintains data integrity with comprehensive backup system
- **Safety Checks**: Enforces backup creation before allowing recalculation

### 4. Advanced Stock Calculation
- **Transaction-based Formula**: `purchases + opening_stock - finalized_sales + sell_returns + stock_adjustments`
- **Multi-transaction Support**: Handles purchases, sales, returns, adjustments, and opening stock
- **Status Filtering**: Only processes finalized transactions for accuracy
- **Cross-location Support**: Calculates stock separately for each business location

## Technical Architecture

### Core Components

#### 1. Controllers
- **StockRecalculationController**: Main application logic and API endpoints
- **InstallController**: Module installation and configuration
- **DataController**: Data processing and utilities

#### 2. Console Commands
- **RecalculateStock**: Command-line interface for batch operations
- **Signature**: `php artisan stock:recalculate {--business_id=} {--location_id=} {--variation_id=}`

#### 3. Database Components
- **Migration Files**: Creates audit views and indexes for performance
- **Views**:
  - `vw_stock_integrity_audit`: Comprehensive stock discrepancy analysis
  - `vw_vld_vs_calculated_and_indexes`: Performance optimization views

#### 4. Utility Classes
- **StockIntegrity Trait**: Core calculation logic with `computeVldQty()` method
- **Concerns**: Reusable stock calculation and synchronization methods

### API Endpoints

#### Web Routes
- `GET /stock-recalculation/` - Main dashboard interface
- `POST /stock-recalculation/verify` - Verify stock discrepancies
- `POST /stock-recalculation/backup` - Create data backup
- `POST /stock-recalculation/start-recalculation` - Initialize batch processing
- `POST /stock-recalculation/process-batch` - Process individual batches
- `GET /stock-recalculation/progress` - Monitor processing progress
- `POST /stock-recalculation/cancel-recalculation` - Cancel operations
- `GET /stock-recalculation/summary` - Dashboard statistics

#### Debug & Testing Endpoints
- `GET /stock-recalculation/debug-calculation` - Detailed calculation breakdown
- `GET /stock-recalculation/test-calculation` - Compare calculation methods
- `GET /stock-recalculation/sample-variations` - Sample data for testing

## User Interface

### Dashboard Components

#### 1. Summary Cards
- **Total Products**: Count of all tracked products
- **Discrepancies**: Number of items with stock differences
- **Critical Issues**: High-priority discrepancies requiring attention
- **Average Difference**: Statistical overview of discrepancy magnitude

#### 2. Filter Controls
- **Business Selection**: Multi-business support with dynamic filtering
- **Location Selection**: Location-specific operations
- **Product ID Filter**: Target specific products for processing

#### 3. Action Workflow
1. **Step 1 - Verify**: Check for stock discrepancies without making changes
2. **Step 2 - Backup**: Create data backup before modifications
3. **Step 3 - Recalculate**: Execute batch recalculation process
4. **Step 4 - Verify Again**: Confirm recalculation results

#### 4. Progress Monitoring
- **Real-time Progress Bar**: Visual indication of processing status
- **Batch Statistics**: Current batch, total batches, processed records
- **Time Estimation**: Estimated completion time based on processing speed
- **Cancellation Controls**: Safe operation termination

## Security & Permissions

### Permission System
- **StockRecalculation.view**: Access to verification and dashboard
- **StockRecalculation.backup**: Create data backups
- **StockRecalculation.recalculate**: Execute recalculation operations

### Security Features
- **CSRF Protection**: All forms protected against cross-site request forgery
- **Business Isolation**: Users can only access their own business data
- **Authentication Required**: All endpoints require valid user authentication
- **Authorization Checks**: Permission verification on all sensitive operations

## Performance Optimization

### Database Optimization
- **Indexed Views**: Pre-computed discrepancy calculations for fast queries
- **Batch Processing**: Configurable batch sizes to prevent timeouts
- **Memory Management**: Garbage collection and memory limit controls
- **Query Optimization**: Efficient joins and filtering for large datasets

### Configuration
```php
const BATCH_SIZE = 100;           // Records per batch
const MAX_EXECUTION_TIME = 25;    // Seconds per batch
const MEMORY_LIMIT = '512M';      // Memory allocation
const PROGRESS_CACHE_DURATION = 3600; // Progress cache duration
```

## Business Logic

### Stock Calculation Formula
The module uses a comprehensive transaction-based calculation:

```sql
calculated_stock =
  SUM(purchases.quantity - purchases.quantity_returned) +
  SUM(opening_stock.quantity) +
  SUM(sell_returns.quantity) +
  SUM(stock_adjustments.quantity) -
  SUM(finalized_sales.quantity)
```

### Transaction Types Supported
- **Purchase**: Inventory received from suppliers
- **Opening Stock**: Initial inventory quantities
- **Sales**: Customer purchases (only finalized transactions)
- **Sell Returns**: Customer returns adding back to inventory
- **Stock Adjustments**: Manual inventory corrections

### Business Rules
1. Only finalized transactions affect stock calculations
2. Returned purchase quantities are deducted from purchases
3. Each location maintains separate stock calculations
4. Backup is mandatory before recalculation
5. Progress is cached for resume capability

## Error Handling & Logging

### Error Management
- **Graceful Degradation**: Continues processing despite individual record errors
- **Detailed Logging**: Comprehensive error logging with context
- **User Feedback**: Clear error messages in user interface
- **Recovery Options**: Ability to resume interrupted operations

### Logging Events
- Recalculation start/completion with statistics
- Individual record processing errors
- Backup creation and verification
- User access and permission events

## Installation & Configuration

### Installation Steps
1. **Module Installation**: Copy module files to `Modules/StockRecalculation/`
2. **Database Migration**: Run migrations to create views and indexes
3. **Permission Setup**: Configure user permissions
4. **Cache Configuration**: Set up progress caching

### System Requirements
- **Laravel**: Compatible with UltimatePOS Laravel framework
- **Database**: MySQL/MariaDB with view support
- **Memory**: Minimum 512MB PHP memory limit
- **Permissions**: Database CREATE, INSERT, UPDATE, DELETE permissions

## Testing & Validation

### Testing Features
- **Calculation Comparison**: Compare original vs. modified calculations
- **Sample Data**: Generate test variations for validation
- **Debug Mode**: Detailed calculation breakdowns
- **Transaction Status Verification**: Audit transaction types and statuses

### Validation Tools
- **Debug Calculation**: Detailed breakdown of stock calculations
- **Test Methods**: Compare different calculation approaches
- **Sample Variations**: Pre-selected test data for verification
- **Status Checks**: Verify transaction statuses and types

## Maintenance & Support

### Regular Maintenance
- **Database View Updates**: Refresh views as needed for schema changes
- **Performance Monitoring**: Track processing times and resource usage
- **Backup Cleanup**: Regular cleanup of old backup tables
- **Log Rotation**: Manage log file sizes

### Troubleshooting
- **Debug Endpoints**: Detailed calculation analysis tools
- **Progress Recovery**: Resume interrupted operations
- **Error Analysis**: Comprehensive error logging and reporting
- **Performance Tuning**: Configurable batch sizes and timeouts

## Future Enhancements

### Planned Features
- **Scheduled Recalculation**: Automated periodic stock verification
- **Multi-business Processing**: Cross-business recalculation capabilities
- **Advanced Reporting**: Detailed discrepancy analysis reports
- **API Integration**: RESTful API for external integrations
- **Export Functionality**: CSV/Excel export of results
- **Email Notifications**: Automated completion notifications

### Scalability Considerations
- **Horizontal Scaling**: Support for distributed processing
- **Queue Integration**: Laravel queue support for background processing
- **Caching Optimization**: Advanced caching strategies
- **Database Partitioning**: Support for very large datasets

## Conclusion

The Stock Recalculation Module provides a robust, scalable solution for maintaining inventory accuracy in UltimatePOS systems. With its comprehensive verification capabilities, batch processing engine, and user-friendly interface, it ensures reliable stock management for businesses of all sizes.

The module's emphasis on data safety, performance optimization, and user experience makes it an essential tool for maintaining inventory integrity and business operations continuity.