<?php

namespace Modules\StockRecalculation\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use App\Business;
use App\BusinessLocation;
use App\Product;
use App\Variation;
use Modules\StockRecalculation\Utils\Concerns\StockIntegrity;

class StockRecalculationController extends Controller
{
    use StockIntegrity;
    // Configuration constants for batch processing
    const BATCH_SIZE = 100;           // Process 100 records at a time
    const MAX_EXECUTION_TIME = 25;    // Max execution time per batch (seconds)
    const MEMORY_LIMIT = '512M';      // Memory limit
    const PROGRESS_CACHE_DURATION = 3600; // 1 hour cache duration for progress
    /**
     * Display stock recalculation interface
     */
    public function index()
    {
        if (!auth()->user()->can('StockRecalculation.view')) {
            abort(403, 'Unauthorized access');
        }

        $business_id = session()->get('user.business_id');
        $businesses = collect([$business_id => Business::find($business_id)]);
        $locations = BusinessLocation::where('business_id', $business_id)->get();
        
        // Get current progress if any recalculation is running
        $progressKey = "stock_recalc_progress_{$business_id}";
        $currentProgress = Cache::get($progressKey);
        
        return view('stockrecalculation::index', compact('businesses', 'locations', 'currentProgress'));
    }

    /**
     * Get locations by business (AJAX endpoint)
     */
    public function getLocations(Request $request)
    {
        $locations = BusinessLocation::where('business_id', $request->business_id)->get();
        return response()->json($locations);
    }

    /**
     * Verify stock discrepancies using audit view
     */
    public function verify(Request $request)
    {
        if (!auth()->user()->can('StockRecalculation.view')) {
            abort(403, 'Unauthorized access');
        }

        try {
            set_time_limit(60); // Allow longer execution for verification
            ini_set('memory_limit', self::MEMORY_LIMIT);
            
            $business_id = session()->get('user.business_id');
            $discrepancies = $this->getDiscrepanciesInBatches($request, $business_id);
            
            return view('stockrecalculation::results', [
                'type' => 'verification',
                'data' => $discrepancies,
                'filters' => $request->all()
            ]);
        } catch (\Exception $e) {
            Log::error('StockRecalculation Verify error: ' . $e->getMessage());
            return response()->json(['error' => 'Verification failed: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Create backup of current stock data
     */
    public function backup(Request $request)
    {
        if (!auth()->user()->can('StockRecalculation.backup')) {
            return response()->json(['success' => false, 'message' => 'Unauthorized']);
        }

        try {
            set_time_limit(300); // 5 minutes for backup
            ini_set('memory_limit', self::MEMORY_LIMIT);
            
            $timestamp = now()->format('Y_m_d_H_i_s');
            $business_id = session()->get('user.business_id');
            $backupTable = "vld_backup_{$business_id}_{$timestamp}";
            
            // Create backup table structure first
            DB::statement("
                CREATE TABLE {$backupTable} LIKE variation_location_details
            ");
            
            // Copy data in batches to prevent timeout
            $offset = 0;
            $batchSize = 1000;
            $totalCopied = 0;
            
            do {
                $copied = DB::statement("
                    INSERT INTO {$backupTable} 
                    SELECT vld.* 
                    FROM variation_location_details vld
                    JOIN business_locations bl ON bl.id = vld.location_id
                    WHERE bl.business_id = {$business_id}
                    LIMIT {$batchSize} OFFSET {$offset}
                ");
                
                $totalCopied += $copied ? $batchSize : 0;
                $offset += $batchSize;
                
                // Prevent memory issues
                if (function_exists('gc_collect_cycles')) {
                    gc_collect_cycles();
                }
                
            } while ($copied && $totalCopied < 50000); // Safety limit
            
            return response()->json([
                'success' => true,
                'backup_table' => $backupTable,
                'records_backed_up' => $totalCopied,
                'message' => "Backup created: {$backupTable} ({$totalCopied} records)"
            ]);
            
        } catch (\Exception $e) {
            Log::error('StockRecalculation Backup error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Backup failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Start batch recalculation process
     */
    public function startRecalculation(Request $request)
    {
        if (!auth()->user()->can('StockRecalculation.recalculate')) {
            return response()->json(['success' => false, 'message' => 'Unauthorized']);
        }

        try {
            $business_id = session()->get('user.business_id');
            $progressKey = "stock_recalc_progress_{$business_id}";
            
            // Check if recalculation is already running
            if (Cache::has($progressKey)) {
                $progress = Cache::get($progressKey);
                if ($progress['status'] === 'running') {
                    return response()->json([
                        'success' => false,
                        'message' => 'Recalculation already in progress',
                        'progress' => $progress
                    ]);
                }
            }

            // Initialize progress tracking
            $totalRecords = $this->getTotalRecordsToProcess($request, $business_id);
            $progress = [
                'status' => 'starting',
                'total_records' => $totalRecords,
                'processed_records' => 0,
                'updated_records' => 0,
                'current_batch' => 0,
                'total_batches' => ceil($totalRecords / self::BATCH_SIZE),
                'start_time' => now()->toDateTimeString(),
                'estimated_time' => null,
                'errors' => []
            ];
            
            Cache::put($progressKey, $progress, self::PROGRESS_CACHE_DURATION);

            return response()->json([
                'success' => true,
                'message' => 'Recalculation started',
                'progress' => $progress,
                'batch_url' => route('stock-recalculation.process-batch')
            ]);

        } catch (\Exception $e) {
            Log::error('StockRecalculation Start error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to start recalculation: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Process a single batch of recalculation
     */
    public function processBatch(Request $request)
    {
        if (!auth()->user()->can('StockRecalculation.recalculate')) {
            return response()->json(['success' => false, 'message' => 'Unauthorized']);
        }

        try {
            set_time_limit(self::MAX_EXECUTION_TIME);
            ini_set('memory_limit', self::MEMORY_LIMIT);
            
            $business_id = session()->get('user.business_id');
            $progressKey = "stock_recalc_progress_{$business_id}";
            $progress = Cache::get($progressKey);

            if (!$progress) {
                return response()->json(['success' => false, 'message' => 'No active recalculation found']);
            }

            // Update status to running
            if ($progress['status'] === 'starting') {
                $progress['status'] = 'running';
            }

            $currentBatch = $progress['current_batch'];
            $offset = $currentBatch * self::BATCH_SIZE;

            // Get batch of records to process
            $batchResults = $this->processBatchOfRecords($request, $business_id, $offset, self::BATCH_SIZE);
            
            // Update progress
            $progress['processed_records'] += $batchResults['processed'];
            $progress['updated_records'] += $batchResults['updated'];
            $progress['current_batch']++;
            
            // Calculate estimated completion time
            if ($progress['processed_records'] > 0) {
                $elapsedTime = now()->diffInSeconds($progress['start_time']);
                $recordsPerSecond = $progress['processed_records'] / max($elapsedTime, 1);
                $remainingRecords = $progress['total_records'] - $progress['processed_records'];
                $estimatedSeconds = $remainingRecords / max($recordsPerSecond, 1);
                $progress['estimated_time'] = now()->addSeconds($estimatedSeconds)->toDateTimeString();
            }

            // Check if completed
            if ($progress['current_batch'] >= $progress['total_batches']) {
                $progress['status'] = 'completed';
                $progress['completion_time'] = now()->toDateTimeString();
                
                Log::info('StockRecalculation: Completed by user', [
                    'user_id' => auth()->id(),
                    'business_id' => $business_id,
                    'total_processed' => $progress['processed_records'],
                    'total_updated' => $progress['updated_records'],
                    'duration_seconds' => now()->diffInSeconds($progress['start_time'])
                ]);
            }

            // Save progress
            Cache::put($progressKey, $progress, self::PROGRESS_CACHE_DURATION);

            return response()->json([
                'success' => true,
                'progress' => $progress,
                'batch_results' => $batchResults
            ]);

        } catch (\Exception $e) {
            // Update progress with error
            if (isset($progress)) {
                $progress['status'] = 'error';
                $progress['errors'][] = $e->getMessage();
                Cache::put($progressKey, $progress, self::PROGRESS_CACHE_DURATION);
            }

            Log::error('StockRecalculation Batch error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Batch processing failed: ' . $e->getMessage(),
                'progress' => $progress ?? null
            ]);
        }
    }

    /**
     * Get current progress status
     */
    public function getProgress()
    {
        if (!auth()->user()->can('StockRecalculation.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $business_id = session()->get('user.business_id');
        $progressKey = "stock_recalc_progress_{$business_id}";
        $progress = Cache::get($progressKey);

        return response()->json([
            'progress' => $progress,
            'has_progress' => $progress !== null
        ]);
    }

    /**
     * Cancel ongoing recalculation
     */
    public function cancelRecalculation()
    {
        if (!auth()->user()->can('StockRecalculation.recalculate')) {
            return response()->json(['success' => false, 'message' => 'Unauthorized']);
        }

        $business_id = session()->get('user.business_id');
        $progressKey = "stock_recalc_progress_{$business_id}";
        
        Cache::forget($progressKey);

        return response()->json([
            'success' => true,
            'message' => 'Recalculation cancelled'
        ]);
    }

    /**
     * Get total number of records to process
     */
    private function getTotalRecordsToProcess($request, $business_id)
    {
        $filters = $this->buildFilters($request, $business_id);
        $whereClause = 'WHERE ' . implode(' AND ', $filters);
        
        $result = DB::selectOne("
            SELECT COUNT(*) as total
            FROM variation_location_details vld
            JOIN business_locations bl ON bl.id = vld.location_id
            JOIN variations v ON v.id = vld.variation_id
            {$whereClause}
        ");
        
        return $result->total ?? 0;
    }

    /**
     * Process a batch of records efficiently
     */
    private function processBatchOfRecords($request, $business_id, $offset, $limit)
    {
        $filters = $this->buildFilters($request, $business_id);
        $whereClause = 'WHERE ' . implode(' AND ', $filters);
        
        // Get batch of variation_location_details IDs to process
        $batchIds = DB::select("
            SELECT vld.id, vld.variation_id, vld.location_id
            FROM variation_location_details vld
            JOIN business_locations bl ON bl.id = vld.location_id
            JOIN variations v ON v.id = vld.variation_id
            {$whereClause}
            ORDER BY vld.id
            LIMIT {$limit} OFFSET {$offset}
        ");

        $processed = 0;
        $updated = 0;

        foreach ($batchIds as $record) {
            try {
                // Use the original stock integrity sync method
                $this->syncVariationLocationQty(
                    $business_id,
                    $record->variation_id,
                    $record->location_id
                );

                // Consider it updated (the sync method handles the actual update)
                $affectedRows = 1;

                if ($affectedRows > 0) {
                    $updated++;
                }
                $processed++;

                // Prevent memory leaks
                if ($processed % 50 === 0) {
                    if (function_exists('gc_collect_cycles')) {
                        gc_collect_cycles();
                    }
                }

            } catch (\Exception $e) {
                Log::error("Error processing variation_location_details ID {$record->id}: " . $e->getMessage());
                $processed++;
            }
        }

        return [
            'processed' => $processed,
            'updated' => $updated
        ];
    }

    /**
     * Calculate stock for a specific variation at a specific location
     * Using original stock integrity patch logic for accuracy
     */
    private function calculateStockForVariationLocation($variation_id, $location_id, $business_id)
    {
        // Use the original stock integrity calculation method
        return $this->computeVldQty($business_id, $variation_id, $location_id);
    }

    /**
     * Test the updated calculation method
     */
    public function testCalculation(Request $request)
    {
        if (!auth()->user()->can('StockRecalculation.view')) {
            abort(403, 'Unauthorized access');
        }

        $variation_id = $request->variation_id;
        $location_id = $request->location_id;
        $business_id = session()->get('user.business_id');

        if (!$variation_id || !$location_id) {
            return response()->json(['error' => 'variation_id and location_id are required']);
        }

        // Get current stored stock
        $currentStock = DB::selectOne("
            SELECT qty_available
            FROM variation_location_details
            WHERE variation_id = ? AND location_id = ?
        ", [$variation_id, $location_id]);

        // Calculate using the new method
        $calculatedStock = $this->computeVldQty($business_id, $variation_id, $location_id);

        return response()->json([
            'variation_id' => $variation_id,
            'location_id' => $location_id,
            'current_stored_stock' => $currentStock->qty_available ?? 0,
            'calculated_stock' => $calculatedStock,
            'difference' => $calculatedStock - ($currentStock->qty_available ?? 0),
            'test_passed' => true
        ]);
    }

    /**
     * Test method to compare original vs modified calculation logic
     */
    public function testCalculationMethods(Request $request)
    {
        if (!auth()->user()->can('StockRecalculation.view')) {
            abort(403, 'Unauthorized access');
        }

        $variation_id = $request->variation_id;
        $location_id = $request->location_id;
        $business_id = session()->get('user.business_id');

        if (!$variation_id || !$location_id) {
            return response()->json(['error' => 'variation_id and location_id are required']);
        }

        // Test original method
        $originalResult = $this->calculateStockOriginalMethod($variation_id, $location_id, $business_id);
        
        // Test modified method  
        $modifiedResult = $this->calculateStockForVariationLocation($variation_id, $location_id, $business_id);
        
        // Get current stored stock
        $currentStock = DB::selectOne("
            SELECT qty_available 
            FROM variation_location_details 
            WHERE variation_id = ? AND location_id = ?
        ", [$variation_id, $location_id]);

        // Get transaction details for analysis
        $transactions = DB::select("
            SELECT t.id, t.type, t.status, t.transaction_date,
                   pl.quantity as purchase_qty, pl.quantity_returned as purchase_returned,
                   tsl.quantity as sell_qty,
                   sal.quantity as adjustment_qty
            FROM transactions t
            LEFT JOIN purchase_lines pl ON pl.transaction_id = t.id AND pl.variation_id = ?
            LEFT JOIN transaction_sell_lines tsl ON tsl.transaction_id = t.id AND tsl.variation_id = ?
            LEFT JOIN stock_adjustment_lines sal ON sal.transaction_id = t.id AND sal.variation_id = ?
            WHERE t.business_id = ? AND t.location_id = ?
            AND (pl.variation_id IS NOT NULL OR tsl.variation_id IS NOT NULL OR sal.variation_id IS NOT NULL)
            ORDER BY t.transaction_date DESC
        ", [$variation_id, $variation_id, $variation_id, $business_id, $location_id]);

        return response()->json([
            'variation_id' => $variation_id,
            'location_id' => $location_id,
            'current_stored_stock' => $currentStock->qty_available ?? 0,
            'original_calculation' => $originalResult,
            'modified_calculation' => $modifiedResult,
            'difference_original_vs_stored' => $originalResult - ($currentStock->qty_available ?? 0),
            'difference_modified_vs_stored' => $modifiedResult - ($currentStock->qty_available ?? 0),
            'difference_original_vs_modified' => $originalResult - $modifiedResult,
            'transactions_found' => count($transactions),
            'transaction_details' => $transactions
        ]);
    }

    /**
     * Original calculation method (before modifications)
     */
    private function calculateStockOriginalMethod($variation_id, $location_id, $business_id)
    {
        $result = DB::selectOne("
            SELECT COALESCE(SUM(
                CASE 
                    WHEN t.type IN ('purchase','opening_stock') AND pl.variation_id = ? 
                         THEN (pl.quantity - IFNULL(pl.quantity_returned,0))
                    WHEN t.type = 'sell_return' AND tsl.variation_id = ? 
                         THEN tsl.quantity
                    WHEN t.type = 'stock_adjustment' AND sal.variation_id = ? 
                         THEN sal.quantity
                    WHEN t.type = 'sell' AND t.status = 'final' AND tsl.variation_id = ? 
                         THEN -tsl.quantity
                    ELSE 0
                END
            ), 0) as calculated_stock
            FROM transactions t
            LEFT JOIN purchase_lines pl ON pl.transaction_id = t.id
            LEFT JOIN transaction_sell_lines tsl ON tsl.transaction_id = t.id  
            LEFT JOIN stock_adjustment_lines sal ON sal.transaction_id = t.id
            WHERE t.business_id = ? AND t.location_id = ?
        ", [$variation_id, $variation_id, $variation_id, $variation_id, $business_id, $location_id]);
        
        return $result->calculated_stock ?? 0;
    }

    /**
     * Check transaction statuses in the database
     */
    public function checkTransactionStatuses(Request $request)
    {
        if (!auth()->user()->can('StockRecalculation.view')) {
            abort(403, 'Unauthorized access');
        }

        $business_id = session()->get('user.business_id');
        $variation_id = $request->variation_id ?? 1;
        
        // Check what statuses exist for each transaction type
        $statuses = DB::select("
            SELECT t.type, t.status, COUNT(*) as count
            FROM transactions t
            LEFT JOIN purchase_lines pl ON pl.transaction_id = t.id
            LEFT JOIN transaction_sell_lines tsl ON tsl.transaction_id = t.id
            LEFT JOIN stock_adjustment_lines sal ON sal.transaction_id = t.id
            WHERE t.business_id = ?
            AND (pl.variation_id = ? OR tsl.variation_id = ? OR sal.variation_id = ?)
            GROUP BY t.type, t.status
            ORDER BY t.type, count DESC
        ", [$business_id, $variation_id, $variation_id, $variation_id]);

        return response()->json([
            'variation_id' => $variation_id,
            'transaction_statuses' => $statuses
        ]);
    }

    /**
     * Get sample variations for testing
     */
    public function getSampleVariations(Request $request)
    {
        if (!auth()->user()->can('StockRecalculation.view')) {
            abort(403, 'Unauthorized access');
        }

        $business_id = session()->get('user.business_id');
        $location_id = $request->location_id ?? 1;

        // Get variations that have transactions and current stock
        $samples = DB::select("
            SELECT DISTINCT vld.variation_id, vld.location_id, vld.qty_available,
                   v.sub_sku, p.name as product_name,
                   COUNT(DISTINCT t.id) as transaction_count
            FROM variation_location_details vld
            JOIN variations v ON v.id = vld.variation_id
            JOIN products p ON p.id = v.product_id
            LEFT JOIN transactions t ON t.location_id = vld.location_id
            LEFT JOIN purchase_lines pl ON pl.transaction_id = t.id AND pl.variation_id = vld.variation_id
            LEFT JOIN transaction_sell_lines tsl ON tsl.transaction_id = t.id AND tsl.variation_id = vld.variation_id
            LEFT JOIN stock_adjustment_lines sal ON sal.transaction_id = t.id AND sal.variation_id = vld.variation_id
            WHERE vld.location_id = ?
            AND t.business_id = ?
            AND (pl.variation_id IS NOT NULL OR tsl.variation_id IS NOT NULL OR sal.variation_id IS NOT NULL)
            GROUP BY vld.variation_id, vld.location_id
            HAVING transaction_count > 0
            ORDER BY transaction_count DESC
            LIMIT 10
        ", [$location_id, $business_id]);

        return response()->json([
            'location_id' => $location_id,
            'samples' => $samples
        ]);
    }

    /**
     * Debug method to show detailed calculation breakdown for a specific variation
     */
    public function debugCalculation(Request $request)
    {
        if (!auth()->user()->can('StockRecalculation.view')) {
            abort(403, 'Unauthorized access');
        }

        $variation_id = $request->variation_id;
        $location_id = $request->location_id;
        $business_id = session()->get('user.business_id');

        if (!$variation_id || !$location_id) {
            return response()->json(['error' => 'variation_id and location_id are required']);
        }

        // Get detailed breakdown
        $purchases = DB::select("
            SELECT t.id, t.transaction_date, t.type, t.status, pl.quantity, pl.quantity_returned
            FROM transactions t
            INNER JOIN purchase_lines pl ON pl.transaction_id = t.id
            WHERE t.type IN ('purchase', 'opening_stock')
            AND pl.variation_id = ?
            AND t.business_id = ?
            AND t.location_id = ?
            ORDER BY t.transaction_date DESC
        ", [$variation_id, $business_id, $location_id]);

        $sales = DB::select("
            SELECT t.id, t.transaction_date, t.type, t.status, tsl.quantity
            FROM transactions t
            INNER JOIN transaction_sell_lines tsl ON tsl.transaction_id = t.id
            WHERE t.type = 'sell'
            AND tsl.variation_id = ?
            AND t.business_id = ?
            AND t.location_id = ?
            ORDER BY t.transaction_date DESC
        ", [$variation_id, $business_id, $location_id]);

        $calculatedStock = $this->calculateStockForVariationLocation($variation_id, $location_id, $business_id);
        
        // Get current stored stock
        $currentStock = DB::selectOne("
            SELECT qty_available 
            FROM variation_location_details 
            WHERE variation_id = ? AND location_id = ?
        ", [$variation_id, $location_id]);

        return response()->json([
            'variation_id' => $variation_id,
            'location_id' => $location_id,
            'calculated_stock' => $calculatedStock,
            'current_stored_stock' => $currentStock->qty_available ?? 0,
            'difference' => $calculatedStock - ($currentStock->qty_available ?? 0),
            'breakdown' => [
                'purchases' => $purchases,
                'sales' => $sales
            ]
        ]);
    }

    /**
     * Get discrepancies in batches to avoid memory issues
     */
    private function getDiscrepanciesInBatches($request, $business_id, $maxResults = 1000)
    {
        $locationFilter = $request->location_id ? "AND location_id = {$request->location_id}" : '';
        $productFilter = $request->product_id ? "AND v.product_id = {$request->product_id}" : '';

        $query = "
            SELECT 
                business_id,
                location_name,
                product_name,
                variation_name,
                location_id,
                variation_id,
                stored_qty as current_qty,
                calculated_qty,
                difference
            FROM vw_stock_integrity_audit
            WHERE business_id = {$business_id}
            AND ABS(difference) > 0.0001
            {$locationFilter}
            {$productFilter}
            ORDER BY ABS(difference) DESC, location_id, variation_id
            LIMIT {$maxResults}
        ";

        return DB::select($query);
    }

    /**
     * Build filters for recalculation
     */
    private function buildFilters($request, $business_id)
    {
        $filters = ["bl.business_id = {$business_id}"];
        
        if ($request->location_id) {
            $filters[] = "vld.location_id = {$request->location_id}";
        }
        
        if ($request->product_id) {
            $filters[] = "v.product_id = {$request->product_id}";
        }
        
        return $filters;
    }


    /**
     * Get dashboard summary data with optimized queries
     */
    public function getSummary()
    {
        if (!auth()->user()->can('StockRecalculation.view')) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        try {
            $business_id = session()->get('user.business_id');
            
            // Use simpler, faster queries for summary
            $summary = DB::selectOne("
                SELECT 
                    COUNT(*) as total_products,
                    SUM(CASE WHEN ABS(difference) > 0.0001 THEN 1 ELSE 0 END) as discrepancies,
                    SUM(CASE WHEN ABS(difference) > 10 THEN 1 ELSE 0 END) as critical_issues,
                    ROUND(AVG(ABS(difference)), 4) as avg_difference
                FROM vw_stock_integrity_audit
                WHERE business_id = ?
                LIMIT 1
            ", [$business_id]);

            return response()->json($summary);
        } catch (\Exception $e) {
            Log::error('StockRecalculation Summary error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to get summary'], 500);
        }
    }
}