<?php

namespace Modules\StockRecalculation\Http\Controllers;

use App\System;
use Carbon\Exceptions\Exception;
use Composer\Semver\Comparator;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Log;

class InstallController extends Controller
{
    public function __construct()
    {
        $this->module_name = 'stockrecalculation';
        $this->appVersion = config('stockrecalculation.module_version', '1.0.0');
        $this->module_display_name = 'Stock Recalculation';
    }

    /**
     * Install
     *
     * @return Response
     */
    public function index()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '512M');

        $this->installSettings();

        //Check if installed or not.
        $is_installed = System::getProperty($this->module_name . '_version');
        if (empty($is_installed)) {
            try {
                DB::beginTransaction();
                DB::statement('SET default_storage_engine=INNODB;');

                // Debug: Check before migration
                \Log::info('StockRecalculation: About to run migration');

                // Run migrations
                Artisan::call('module:migrate', ['module' => 'StockRecalculation', '--force' => true]);

                // Debug: Check after migration
                \Log::info('StockRecalculation: Migration completed');

                System::addProperty($this->module_name . '_version', $this->appVersion);

                DB::commit();

                // Create default audit view and indexes after successful installation
                $this->createStockIntegrityViews();

                // Clear all caches after successful installation
                $this->clearAllCaches();

                \Log::info('StockRecalculation: Installation successful');
            } catch (\Exception $e) {
                DB::rollBack();
                \Log::error('StockRecalculation Installation error: ' . $e->getMessage());
                throw $e;
            }
        }

        $output = [
            'success' => 1,
            'msg' => 'Stock Recalculation module installed successfully. All caches have been cleared automatically.',
        ];

        return redirect()
            ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
            ->with('status', $output);
    }

    /**
     * Initialize all install functions
     */
    private function installSettings()
    {
        config(['app.debug' => true]);
        Artisan::call('config:clear');
    }

    /**
     * Clear all Laravel caches (config, route, view, cache)
     */
    private function clearAllCaches()
    {
        try {
            // Clear configuration cache
            Artisan::call('config:clear');
            \Log::info('StockRecalculation: Configuration cache cleared');

            // Clear route cache
            Artisan::call('route:clear');
            \Log::info('StockRecalculation: Route cache cleared');

            // Clear view cache
            Artisan::call('view:clear');
            \Log::info('StockRecalculation: View cache cleared');

            // Clear application cache
            Artisan::call('cache:clear');
            \Log::info('StockRecalculation: Application cache cleared');

            // Clear compiled services and packages
            Artisan::call('clear-compiled');
            \Log::info('StockRecalculation: Compiled services cleared');

            // Optimize autoloader (optional but recommended)
            if (app()->environment('production')) {
                Artisan::call('optimize');
                \Log::info('StockRecalculation: Application optimized for production');
            }

            \Log::info('StockRecalculation: All caches cleared successfully');
        } catch (\Exception $e) {
            \Log::error('StockRecalculation: Error clearing caches: ' . $e->getMessage());
            // Don't throw exception here as cache clearing shouldn't stop installation
        }
    }

    /**
     * Create stock integrity audit views and indexes
     */
    private function createStockIntegrityViews()
    {
        try {
            // Create database indexes for performance
            $this->createPerformanceIndexes();
            
            // Create audit view for stock verification
            $this->createAuditView();

            \Log::info('StockRecalculation: Stock integrity views created');
        } catch (\Exception $e) {
            \Log::error('StockRecalculation: Error creating views: ' . $e->getMessage());
        }
    }

    /**
     * Create performance indexes
     */
    private function createPerformanceIndexes()
    {
        $indexes = [
            'CREATE INDEX idx_pl_variation_sr ON purchase_lines (variation_id)',
            'CREATE INDEX idx_t_business_type_loc_sr ON transactions (business_id, type, location_id)',
            'CREATE INDEX idx_tsl_variation_sr ON transaction_sell_lines (variation_id)',
            'CREATE INDEX idx_sal_variation_sr ON stock_adjustment_lines (variation_id)',
            'CREATE INDEX idx_vld_vl_sr ON variation_location_details (variation_id, location_id)',
        ];

        foreach ($indexes as $index) {
            try {
                DB::statement($index);
            } catch (\Throwable $e) {
                \Log::warning('StockRecalculation: Index creation warning: ' . $e->getMessage());
                // Continue with other indexes even if one fails
            }
        }
    }

    /**
     * Create audit view for stock verification
     */
    private function createAuditView()
    {
        DB::statement('DROP VIEW IF EXISTS vw_stock_integrity_audit');
        DB::statement('
            CREATE VIEW vw_stock_integrity_audit AS
            SELECT
                bl.business_id,
                vld.location_id,
                vld.variation_id,
                bl.name as location_name,
                p.name as product_name,
                v.name as variation_name,
                vld.qty_available AS stored_qty,
                ROUND(
                    COALESCE((
                        SELECT SUM((pl.quantity - IFNULL(pl.quantity_returned,0)))
                        FROM purchase_lines pl
                        JOIN transactions t ON t.id = pl.transaction_id
                        WHERE t.business_id = bl.business_id
                          AND t.location_id = vld.location_id
                          AND pl.variation_id = vld.variation_id
                          AND t.type IN (\'purchase\',\'opening_stock\')
                    ),0)
                    + COALESCE((
                        SELECT SUM(IFNULL(tsl.quantity,0))
                        FROM transaction_sell_lines tsl
                        JOIN transactions t ON t.id = tsl.transaction_id
                        WHERE t.business_id = bl.business_id
                          AND t.location_id = vld.location_id
                          AND tsl.variation_id = vld.variation_id
                          AND t.type = \'sell_return\'
                    ),0)
                    + COALESCE((
                        SELECT SUM(IFNULL(sal.quantity,0))
                        FROM stock_adjustment_lines sal
                        JOIN transactions t ON t.id = sal.transaction_id
                        WHERE t.business_id = bl.business_id
                          AND t.location_id = vld.location_id
                          AND sal.variation_id = vld.variation_id
                    ),0)
                    - COALESCE((
                        SELECT SUM(IFNULL(tsl.quantity,0))
                        FROM transaction_sell_lines tsl
                        JOIN transactions t ON t.id = tsl.transaction_id
                        WHERE t.business_id = bl.business_id
                          AND t.location_id = vld.location_id
                          AND tsl.variation_id = vld.variation_id
                          AND t.type = \'sell\'
                          AND t.status IN (\'final\')
                    ),0)
                ,4) AS calculated_qty,
                ROUND(vld.qty_available - COALESCE((
                    SELECT SUM((pl.quantity - IFNULL(pl.quantity_returned,0)))
                    FROM purchase_lines pl
                    JOIN transactions t ON t.id = pl.transaction_id
                    WHERE t.business_id = bl.business_id
                      AND t.location_id = vld.location_id
                      AND pl.variation_id = vld.variation_id
                      AND t.type IN (\'purchase\',\'opening_stock\')
                ),0) - COALESCE((
                    SELECT SUM(IFNULL(tsl.quantity,0))
                    FROM transaction_sell_lines tsl
                    JOIN transactions t ON t.id = tsl.transaction_id
                    WHERE t.business_id = bl.business_id
                      AND t.location_id = vld.location_id
                      AND tsl.variation_id = vld.variation_id
                      AND t.type = \'sell_return\'
                ),0) - COALESCE((
                    SELECT SUM(IFNULL(sal.quantity,0))
                    FROM stock_adjustment_lines sal
                    JOIN transactions t ON t.id = sal.transaction_id
                    WHERE t.business_id = bl.business_id
                      AND t.location_id = vld.location_id
                      AND sal.variation_id = vld.variation_id
                ),0) + COALESCE((
                    SELECT SUM(IFNULL(tsl.quantity,0))
                    FROM transaction_sell_lines tsl
                    JOIN transactions t ON t.id = tsl.transaction_id
                    WHERE t.business_id = bl.business_id
                      AND t.location_id = vld.location_id
                      AND tsl.variation_id = vld.variation_id
                      AND t.type = \'sell\'
                      AND t.status IN (\'final\')
                ),0), 4) AS difference
            FROM variation_location_details vld
            JOIN business_locations bl ON bl.id = vld.location_id
            JOIN variations v ON v.id = vld.variation_id
            JOIN products p ON p.id = v.product_id
        ');
    }

    // Updating
    public function update()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '512M');

        try {
            DB::beginTransaction();

            $stock_recalc_version = System::getProperty($this->module_name . '_version');

            if (Comparator::greaterThan($this->appVersion, $stock_recalc_version)) {
                DB::statement('SET default_storage_engine=INNODB;');
                Artisan::call('module:migrate', ['module' => 'StockRecalculation', '--force' => true]);

                System::setProperty($this->module_name . '_version', $this->appVersion);

                \Log::info('StockRecalculation: Update completed to version ' . $this->appVersion);
            } else {
                abort(404);
            }

            DB::commit();

            // Update audit views after transaction
            $this->createStockIntegrityViews();

            // Clear all caches after successful update
            $this->clearAllCaches();

            \Log::info('StockRecalculation: Update transaction committed successfully');
        } catch (Exception $e) {
            DB::rollBack();
            \Log::error('StockRecalculation Update error: ' . $e->getMessage());
            exit($e->getMessage());
        }

        $output = [
            'success' => 1,
            'msg' => 'Stock Recalculation module updated successfully to version ' . $this->appVersion . '! All caches have been cleared automatically.',
        ];

        return redirect()
            ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
            ->with('status', $output);
    }

    /**
     * Uninstall
     *
     * @return Response
     */
    public function uninstall()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            System::removeProperty($this->module_name . '_version');

            DB::commit();

            // Drop audit view after transaction
            DB::statement('DROP VIEW IF EXISTS vw_stock_integrity_audit');

            $output = [
                'success' => true,
                'msg' => __('stockrecalculation::lang.uninstall_success'),
            ];

            \Log::info('StockRecalculation: Uninstalled successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('StockRecalculation Uninstall error: ' . $e->getMessage());
            $output = [
                'success' => false,
                'msg' => $e->getMessage(),
            ];
        }

        return redirect()->back()->with(['status' => $output]);
    }
}