<?php

namespace Modules\StockRecalculation\Http\Controllers;

use Illuminate\Routing\Controller;
use App\Utils\ModuleUtil;
use Menu;

class DataController extends Controller
{
    /**
     * Defines module as a superadmin package.
     * @return Array
     */
    public function superadmin_package()
    {
        return [
            [
                'name' => 'stock_recalculation_module',
                'label' => __('stockrecalculation::lang.stock_recalculation_module'),
                'default' => false,
            ]
        ];
    }

    /**
     * Defines user permissions for the module.
     * @return array
     */
    public function user_permissions()
    {
        return [
            // General Access
            [
                'value' => 'StockRecalculation.view',
                'label' => __('stockrecalculation::lang.view_stock_recalculation'),
                'default' => false
            ],
            [
                'value' => 'StockRecalculation.manage',
                'label' => __('stockrecalculation::lang.manage_stock_recalculation'),
                'default' => false
            ],

            // Backup & Recalculation
            [
                'value' => 'StockRecalculation.backup',
                'label' => __('stockrecalculation::lang.backup_stock_data'),
                'default' => false
            ],
            [
                'value' => 'StockRecalculation.recalculate',
                'label' => __('stockrecalculation::lang.recalculate_stock'),
                'default' => false
            ],

            // Advanced Features
            [
                'value' => 'StockRecalculation.audit',
                'label' => __('stockrecalculation::lang.audit_stock_integrity'),
                'default' => false
            ],
            [
                'value' => 'StockRecalculation.export',
                'label' => __('stockrecalculation::lang.export_reports'),
                'default' => false
            ],
        ];
    }

    /**
     * Modify admin menu to add StockRecalculation menu items
     */
    public function modifyAdminMenu()
    {
        $business_id = session()->get('user.business_id');
        $module_util = new ModuleUtil();

        $is_stock_recalc_enabled = (bool)$module_util->hasThePermissionInSubscription($business_id, 'stock_recalculation_module', 'superadmin_package');

        if ($is_stock_recalc_enabled && auth()->user()->can('StockRecalculation.view')) {
            $menuparent = Menu::instance('admin-sidebar-menu');

            // Add as single menu item under Inventory/Stock management
            $menuparent->url(
                action([\Modules\StockRecalculation\Http\Controllers\StockRecalculationController::class, 'index']),
                __('stockrecalculation::lang.stock_recalculation'),
                [
                    'icon' => '<svg xmlns="http://www.w3.org/2000/svg" class="tw-size-5 tw-shrink-0" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                        <path stroke="none" d="M0 0h24v24H0z" fill="none"/>
                        <path d="M20 11a8.1 8.1 0 0 0 -15.5 -2m-.5 -4v4h4"/>
                        <path d="M4 13a8.1 8.1 0 0 0 15.5 2m.5 4v-4h-4"/>
                        <path d="M12 9l0 3"/>
                        <path d="M12 15l.01 0"/>
                    </svg>',
                    'active' => request()->segment(1) == 'stock-recalculation'
                ]
            )->order(85); // Position after Inventory management items

            // Or add as dropdown if you want sub-items
            /*
            $menuparent->dropdown(
                __('stockrecalculation::lang.stock_integrity'),
                function ($sub) {
                    $sub->url(
                        action([\Modules\StockRecalculation\Http\Controllers\StockRecalculationController::class, 'index']),
                        '🔧 ' . __('stockrecalculation::lang.stock_recalculation'),
                        ['active' => request()->segment(1) == 'stock-recalculation']
                    );
                    
                    if (auth()->user()->can('StockRecalculation.audit')) {
                        $sub->url(
                            action([\Modules\StockRecalculation\Http\Controllers\StockRecalculationController::class, 'getSummary']),
                            '📊 ' . __('stockrecalculation::lang.integrity_audit'),
                            ['active' => request()->segment(2) == 'audit']
                        );
                    }
                },
                [
                    'icon' => 'fa fa-database',
                    'style' => 'color: #dd4b39;'
                ]
            )->order(85);
            */
        }
    }

    /**
     * Returns details about the module
     */
    public function module_details()
    {
        return [
            'name' => 'Stock Recalculation',
            'version' => config('stockrecalculation.module_version', '1.0.0'),
            'description' => __('stockrecalculation::lang.module_description'),
            'author' => 'UPOS Development Team',
            'website' => 'https://ultimatepos.com',
            'category' => 'Inventory Management',
            'dependencies' => [],
            'features' => [
                'Automated stock quantity recalculation',
                'Stock integrity verification and audit',
                'Backup and restore functionality',
                'Discrepancy detection and reporting',
                'Performance optimized queries',
                'Business-specific data isolation'
            ]
        ];
    }

    /**
     * Module installation requirements check
     */
    public function installation_requirements()
    {
        $requirements = [
            'php_version' => '8.0',
            'laravel_version' => '10.0',
            'mysql_version' => '5.7',
            'required_permissions' => [
                'user.create', // Admin permissions needed
            ],
            'required_tables' => [
                'variation_location_details',
                'transactions',
                'purchase_lines', 
                'transaction_sell_lines',
                'stock_adjustment_lines',
                'business_locations'
            ]
        ];

        // Check if all requirements are met
        $status = [
            'php_version_ok' => version_compare(PHP_VERSION, $requirements['php_version'], '>='),
            'tables_exist' => $this->checkRequiredTables($requirements['required_tables']),
            'permissions_ok' => auth()->user() && auth()->user()->can('user.create'),
        ];

        return [
            'requirements' => $requirements,
            'status' => $status,
            'ready_to_install' => !in_array(false, $status)
        ];
    }

    /**
     * Check if required database tables exist
     */
    private function checkRequiredTables($tables)
    {
        try {
            foreach ($tables as $table) {
                if (!\Schema::hasTable($table)) {
                    return false;
                }
            }
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}