# Stock Recalculation Module - Code Documentation

## Table of Contents
1. [Code Structure Overview](#code-structure-overview)
2. [Core Classes & Methods](#core-classes--methods)
3. [Database Interactions](#database-interactions)
4. [Frontend Components](#frontend-components)
5. [Configuration & Constants](#configuration--constants)
6. [Code Examples & Usage](#code-examples--usage)
7. [Testing & Debugging](#testing--debugging)
8. [Extension Points](#extension-points)

## Code Structure Overview

### Module File Structure
```
Modules/StockRecalculation/
├── Http/Controllers/
│   ├── StockRecalculationController.php    # Main business logic
│   ├── InstallController.php               # Module installation
│   └── DataController.php                  # Data processing utilities
├── Console/Commands/
│   └── RecalculateStock.php               # CLI command interface
├── Database/
│   ├── Migrations/                         # Database schema changes
│   └── Seeders/                           # Initial data seeding
├── Utils/Concerns/
│   └── StockIntegrity.php                 # Stock calculation trait
├── Resources/
│   ├── views/                             # Blade templates
│   └── lang/                              # Localization files
├── Routes/
│   ├── web.php                            # Web routes
│   └── api.php                            # API routes
├── Providers/
│   ├── StockRecalculationServiceProvider.php
│   └── RouteServiceProvider.php
├── Config/
│   └── config.php                         # Module configuration
├── module.json                            # Module metadata
└── composer.json                          # Dependencies
```

## Core Classes & Methods

### StockRecalculationController.php

#### Class Overview
```php
namespace Modules\StockRecalculation\Http\Controllers;

class StockRecalculationController extends Controller
{
    use StockIntegrity;

    // Configuration constants
    const BATCH_SIZE = 100;
    const MAX_EXECUTION_TIME = 25;
    const MEMORY_LIMIT = '512M';
    const PROGRESS_CACHE_DURATION = 3600;
}
```

#### Key Methods

##### index()
**Purpose**: Display main dashboard interface
**Parameters**: None
**Returns**: Blade view with dashboard data
**Security**: Requires `StockRecalculation.view` permission

```php
public function index()
{
    if (!auth()->user()->can('StockRecalculation.view')) {
        abort(403, 'Unauthorized access');
    }

    $business_id = session()->get('user.business_id');
    $businesses = collect([$business_id => Business::find($business_id)]);
    $locations = BusinessLocation::where('business_id', $business_id)->get();

    // Get current progress if any recalculation is running
    $progressKey = "stock_recalc_progress_{$business_id}";
    $currentProgress = Cache::get($progressKey);

    return view('stockrecalculation::index', compact('businesses', 'locations', 'currentProgress'));
}
```

##### verify(Request $request)
**Purpose**: Verify stock discrepancies without making changes
**Parameters**: Request with filters (business_id, location_id, product_id)
**Returns**: Results view with discrepancy data
**Security**: Requires `StockRecalculation.view` permission

```php
public function verify(Request $request)
{
    if (!auth()->user()->can('StockRecalculation.view')) {
        abort(403, 'Unauthorized access');
    }

    try {
        set_time_limit(60);
        ini_set('memory_limit', self::MEMORY_LIMIT);

        $business_id = session()->get('user.business_id');
        $discrepancies = $this->getDiscrepanciesInBatches($request, $business_id);

        return view('stockrecalculation::results', [
            'type' => 'verification',
            'data' => $discrepancies,
            'filters' => $request->all()
        ]);
    } catch (\Exception $e) {
        Log::error('StockRecalculation Verify error: ' . $e->getMessage());
        return response()->json(['error' => 'Verification failed: ' . $e->getMessage()], 500);
    }
}
```

##### backup(Request $request)
**Purpose**: Create backup of current stock data
**Parameters**: Request with filter criteria
**Returns**: JSON response with backup details
**Security**: Requires `StockRecalculation.backup` permission

```php
public function backup(Request $request)
{
    if (!auth()->user()->can('StockRecalculation.backup')) {
        return response()->json(['success' => false, 'message' => 'Unauthorized']);
    }

    try {
        set_time_limit(300);
        ini_set('memory_limit', self::MEMORY_LIMIT);

        $timestamp = now()->format('Y_m_d_H_i_s');
        $business_id = session()->get('user.business_id');
        $backupTable = "vld_backup_{$business_id}_{$timestamp}";

        // Create backup table structure
        DB::statement("CREATE TABLE {$backupTable} LIKE variation_location_details");

        // Copy data in batches
        $offset = 0;
        $batchSize = 1000;
        $totalCopied = 0;

        do {
            $copied = DB::statement("
                INSERT INTO {$backupTable}
                SELECT vld.*
                FROM variation_location_details vld
                JOIN business_locations bl ON bl.id = vld.location_id
                WHERE bl.business_id = {$business_id}
                LIMIT {$batchSize} OFFSET {$offset}
            ");

            $totalCopied += $copied ? $batchSize : 0;
            $offset += $batchSize;

            if (function_exists('gc_collect_cycles')) {
                gc_collect_cycles();
            }

        } while ($copied && $totalCopied < 50000);

        return response()->json([
            'success' => true,
            'backup_table' => $backupTable,
            'records_backed_up' => $totalCopied,
            'message' => "Backup created: {$backupTable} ({$totalCopied} records)"
        ]);

    } catch (\Exception $e) {
        Log::error('StockRecalculation Backup error: ' . $e->getMessage());
        return response()->json([
            'success' => false,
            'message' => 'Backup failed: ' . $e->getMessage()
        ]);
    }
}
```

##### startRecalculation(Request $request)
**Purpose**: Initialize batch recalculation process
**Parameters**: Request with processing parameters
**Returns**: JSON response with progress tracking data
**Security**: Requires `StockRecalculation.recalculate` permission

```php
public function startRecalculation(Request $request)
{
    if (!auth()->user()->can('StockRecalculation.recalculate')) {
        return response()->json(['success' => false, 'message' => 'Unauthorized']);
    }

    try {
        $business_id = session()->get('user.business_id');
        $progressKey = "stock_recalc_progress_{$business_id}";

        // Check if recalculation is already running
        if (Cache::has($progressKey)) {
            $progress = Cache::get($progressKey);
            if ($progress['status'] === 'running') {
                return response()->json([
                    'success' => false,
                    'message' => 'Recalculation already in progress',
                    'progress' => $progress
                ]);
            }
        }

        // Initialize progress tracking
        $totalRecords = $this->getTotalRecordsToProcess($request, $business_id);
        $progress = [
            'status' => 'starting',
            'total_records' => $totalRecords,
            'processed_records' => 0,
            'updated_records' => 0,
            'current_batch' => 0,
            'total_batches' => ceil($totalRecords / self::BATCH_SIZE),
            'start_time' => now()->toDateTimeString(),
            'estimated_time' => null,
            'errors' => []
        ];

        Cache::put($progressKey, $progress, self::PROGRESS_CACHE_DURATION);

        return response()->json([
            'success' => true,
            'message' => 'Recalculation started',
            'progress' => $progress,
            'batch_url' => route('stock-recalculation.process-batch')
        ]);

    } catch (\Exception $e) {
        Log::error('StockRecalculation Start error: ' . $e->getMessage());
        return response()->json([
            'success' => false,
            'message' => 'Failed to start recalculation: ' . $e->getMessage()
        ]);
    }
}
```

##### processBatch(Request $request)
**Purpose**: Process individual batch of records
**Parameters**: Request with batch processing context
**Returns**: JSON response with batch results and updated progress
**Security**: Requires `StockRecalculation.recalculate` permission

```php
public function processBatch(Request $request)
{
    if (!auth()->user()->can('StockRecalculation.recalculate')) {
        return response()->json(['success' => false, 'message' => 'Unauthorized']);
    }

    try {
        set_time_limit(self::MAX_EXECUTION_TIME);
        ini_set('memory_limit', self::MEMORY_LIMIT);

        $business_id = session()->get('user.business_id');
        $progressKey = "stock_recalc_progress_{$business_id}";
        $progress = Cache::get($progressKey);

        if (!$progress) {
            return response()->json(['success' => false, 'message' => 'No active recalculation found']);
        }

        // Update status to running
        if ($progress['status'] === 'starting') {
            $progress['status'] = 'running';
        }

        $currentBatch = $progress['current_batch'];
        $offset = $currentBatch * self::BATCH_SIZE;

        // Process batch of records
        $batchResults = $this->processBatchOfRecords($request, $business_id, $offset, self::BATCH_SIZE);

        // Update progress
        $progress['processed_records'] += $batchResults['processed'];
        $progress['updated_records'] += $batchResults['updated'];
        $progress['current_batch']++;

        // Calculate estimated completion time
        if ($progress['processed_records'] > 0) {
            $elapsedTime = now()->diffInSeconds($progress['start_time']);
            $recordsPerSecond = $progress['processed_records'] / max($elapsedTime, 1);
            $remainingRecords = $progress['total_records'] - $progress['processed_records'];
            $estimatedSeconds = $remainingRecords / max($recordsPerSecond, 1);
            $progress['estimated_time'] = now()->addSeconds($estimatedSeconds)->toDateTimeString();
        }

        // Check if completed
        if ($progress['current_batch'] >= $progress['total_batches']) {
            $progress['status'] = 'completed';
            $progress['completion_time'] = now()->toDateTimeString();

            Log::info('StockRecalculation: Completed by user', [
                'user_id' => auth()->id(),
                'business_id' => $business_id,
                'total_processed' => $progress['processed_records'],
                'total_updated' => $progress['updated_records'],
                'duration_seconds' => now()->diffInSeconds($progress['start_time'])
            ]);
        }

        // Save progress
        Cache::put($progressKey, $progress, self::PROGRESS_CACHE_DURATION);

        return response()->json([
            'success' => true,
            'progress' => $progress,
            'batch_results' => $batchResults
        ]);

    } catch (\Exception $e) {
        // Update progress with error
        if (isset($progress)) {
            $progress['status'] = 'error';
            $progress['errors'][] = $e->getMessage();
            Cache::put($progressKey, $progress, self::PROGRESS_CACHE_DURATION);
        }

        Log::error('StockRecalculation Batch error: ' . $e->getMessage());
        return response()->json([
            'success' => false,
            'message' => 'Batch processing failed: ' . $e->getMessage(),
            'progress' => $progress ?? null
        ]);
    }
}
```

#### Helper Methods

##### processBatchOfRecords()
**Purpose**: Core batch processing logic
**Parameters**: Request, business_id, offset, limit
**Returns**: Array with processed and updated counts

```php
private function processBatchOfRecords($request, $business_id, $offset, $limit)
{
    $filters = $this->buildFilters($request, $business_id);
    $whereClause = 'WHERE ' . implode(' AND ', $filters);

    // Get batch of variation_location_details IDs to process
    $batchIds = DB::select("
        SELECT vld.id, vld.variation_id, vld.location_id
        FROM variation_location_details vld
        JOIN business_locations bl ON bl.id = vld.location_id
        JOIN variations v ON v.id = vld.variation_id
        {$whereClause}
        ORDER BY vld.id
        LIMIT {$limit} OFFSET {$offset}
    ");

    $processed = 0;
    $updated = 0;

    foreach ($batchIds as $record) {
        try {
            // Use stock integrity sync method
            $this->syncVariationLocationQty(
                $business_id,
                $record->variation_id,
                $record->location_id
            );

            $affectedRows = 1;
            if ($affectedRows > 0) {
                $updated++;
            }
            $processed++;

            // Prevent memory leaks
            if ($processed % 50 === 0) {
                if (function_exists('gc_collect_cycles')) {
                    gc_collect_cycles();
                }
            }

        } catch (\Exception $e) {
            Log::error("Error processing variation_location_details ID {$record->id}: " . $e->getMessage());
            $processed++;
        }
    }

    return [
        'processed' => $processed,
        'updated' => $updated
    ];
}
```

### StockIntegrity.php Trait

#### Core Calculation Method

##### computeVldQty()
**Purpose**: Calculate accurate stock quantity from transactions
**Parameters**: business_id, variation_id, location_id
**Returns**: Float - calculated stock quantity
**Formula**: `purchases + opening_stock - finalized_sales + sell_returns + stock_adjustments`

```php
public function computeVldQty(int $business_id, int $variation_id, int $location_id): float
{
    // Calculate received inventory (purchases + opening stock)
    $received = (float) DB::table('purchase_lines as pl')
        ->join('transactions as t', 't.id', '=', 'pl.transaction_id')
        ->where('t.business_id', $business_id)
        ->whereIn('t.type', ['purchase', 'opening_stock'])
        ->where('t.location_id', $location_id)
        ->where('pl.variation_id', $variation_id)
        ->sum(DB::raw('(pl.quantity - IFNULL(pl.quantity_returned,0))'));

    // Calculate stock adjustments
    $adjustments = (float) DB::table('stock_adjustment_lines as sal')
        ->join('transactions as t', 't.id', '=', 'sal.transaction_id')
        ->where('t.business_id', $business_id)
        ->where('t.location_id', $location_id)
        ->where('sal.variation_id', $variation_id)
        ->sum(DB::raw('IFNULL(sal.quantity,0)'));

    // Calculate sold inventory (only finalized sales)
    $sold = (float) DB::table('transaction_sell_lines as tsl')
        ->join('transactions as t', 't.id', '=', 'tsl.transaction_id')
        ->where('t.business_id', $business_id)
        ->where('t.location_id', $location_id)
        ->where('tsl.variation_id', $variation_id)
        ->where('t.type', 'sell')
        ->whereIn('t.status', ['final'])
        ->sum(DB::raw('IFNULL(tsl.quantity,0)'));

    // Calculate sell returns
    $sell_returns = (float) DB::table('transaction_sell_lines as tsl')
        ->join('transactions as t', 't.id', '=', 'tsl.transaction_id')
        ->where('t.business_id', $business_id)
        ->where('t.location_id', $location_id)
        ->where('tsl.variation_id', $variation_id)
        ->where('t.type', 'sell_return')
        ->sum(DB::raw('IFNULL(tsl.quantity,0)'));

    // Final calculation with 4 decimal precision
    return (float) round(($received + $sell_returns + $adjustments) - $sold, 4);
}
```

##### syncVariationLocationQty()
**Purpose**: Update variation_location_details with calculated quantity
**Parameters**: business_id, variation_id, location_id
**Returns**: void

```php
public function syncVariationLocationQty(int $business_id, int $variation_id, int $location_id): void
{
    $qty = $this->computeVldQty($business_id, $variation_id, $location_id);
    DB::table('variation_location_details as vld')
        ->where('vld.variation_id', $variation_id)
        ->where('vld.location_id', $location_id)
        ->update(['qty_available' => $qty]);
}
```

### RecalculateStock.php Console Command

#### Command Implementation
```php
class RecalculateStock extends Command
{
    use StockIntegrity;

    protected $signature = 'stock:recalculate {--business_id=} {--location_id=} {--variation_id=}';
    protected $description = 'Recalculate variation_location_details.qty_available from authoritative transactions.';

    public function handle(): int
    {
        $businessId = $this->option('business_id') ? (int)$this->option('business_id') : null;
        $locationId = $this->option('location_id') ? (int)$this->option('location_id') : null;
        $variationId = $this->option('variation_id') ? (int)$this->option('variation_id') : null;

        $q = DB::table('variation_location_details as vld')
            ->join('business_locations as bl', 'bl.id', '=', 'vld.location_id')
            ->select('bl.business_id', 'vld.location_id', 'vld.variation_id')
            ->when($businessId !== null, function($qq) use ($businessId){
                return $qq->where('bl.business_id', $businessId);
            })
            ->when($locationId !== null, function($qq) use ($locationId){
                return $qq->where('vld.location_id', $locationId);
            })
            ->when($variationId !== null, function($qq) use ($variationId){
                return $qq->where('vld.variation_id', $variationId);
            })
            ->orderBy('vld.location_id')->orderBy('vld.variation_id');

        $rows = $q->get();
        if($rows->isEmpty()){
            $this->info('No rows matched filters.');
            return 0;
        }

        $count = 0;
        $pu = app(ProductUtil::class);

        foreach($rows as $r){
            try{
                if(method_exists($pu,'syncVariationLocationQty')){
                    $pu->syncVariationLocationQty(
                        (int)$r->business_id,
                        (int)$r->variation_id,
                        (int)$r->location_id
                    );
                } else {
                    $qty = $this->computeVldQty(
                        (int)$r->business_id,
                        (int)$r->variation_id,
                        (int)$r->location_id
                    );
                    DB::table('variation_location_details as vld')
                        ->where('vld.variation_id',(int)$r->variation_id)
                        ->where('vld.location_id',(int)$r->location_id)
                        ->update(['qty_available'=>$qty]);
                }
                $count++;
            } catch(\Throwable $e){
                $this->error('Failed for variation_id='.(int)$r->variation_id.
                    ' location_id='.(int)$r->location_id.' : '.$e->getMessage());
            }
        }

        $this->info('Recalculated rows: '.$count);
        return 0;
    }
}
```

## Database Interactions

### Database Views Usage

#### Using vw_stock_integrity_audit
```php
// Get discrepancies with business filtering
$discrepancies = DB::select("
    SELECT
        business_id,
        location_name,
        product_name,
        variation_name,
        location_id,
        variation_id,
        stored_qty as current_qty,
        calculated_qty,
        difference
    FROM vw_stock_integrity_audit
    WHERE business_id = ?
    AND ABS(difference) > 0.0001
    ORDER BY ABS(difference) DESC
    LIMIT 1000
", [$business_id]);
```

#### Summary Statistics
```php
// Dashboard summary data
$summary = DB::selectOne("
    SELECT
        COUNT(*) as total_products,
        SUM(CASE WHEN ABS(difference) > 0.0001 THEN 1 ELSE 0 END) as discrepancies,
        SUM(CASE WHEN ABS(difference) > 10 THEN 1 ELSE 0 END) as critical_issues,
        ROUND(AVG(ABS(difference)), 4) as avg_difference
    FROM vw_stock_integrity_audit
    WHERE business_id = ?
    LIMIT 1
", [$business_id]);
```

### Query Building Patterns

#### Dynamic Filter Building
```php
private function buildFilters($request, $business_id)
{
    $filters = ["bl.business_id = {$business_id}"];

    if ($request->location_id) {
        $filters[] = "vld.location_id = {$request->location_id}";
    }

    if ($request->product_id) {
        $filters[] = "v.product_id = {$request->product_id}";
    }

    return $filters;
}
```

#### Batch Processing Query
```php
// Get batch of records for processing
$batchIds = DB::select("
    SELECT vld.id, vld.variation_id, vld.location_id
    FROM variation_location_details vld
    JOIN business_locations bl ON bl.id = vld.location_id
    JOIN variations v ON v.id = vld.variation_id
    {$whereClause}
    ORDER BY vld.id
    LIMIT {$limit} OFFSET {$offset}
");
```

## Frontend Components

### JavaScript Implementation

#### Progress Tracking
```javascript
// Real-time progress monitoring
function startProgressPolling() {
    progressInterval = setInterval(function() {
        $.get('/stock-recalculation/progress', function(response) {
            if (response.progress) {
                currentProgress = response.progress;
                updateProgressDisplay(currentProgress);

                if (currentProgress.status === 'completed' || currentProgress.status === 'error') {
                    stopProgressPolling();
                    if (currentProgress.status === 'completed') {
                        completeRecalculation();
                    }
                }
            }
        });
    }, 2000); // Poll every 2 seconds
}
```

#### Batch Processing Control
```javascript
// Process batches with automatic progression
function processBatch(batchUrl) {
    if (!currentProgress || currentProgress.status === 'error' || currentProgress.status === 'completed') {
        return;
    }

    $.post(batchUrl, $('#stock-filters').serialize(), function(response) {
        if (response.success) {
            currentProgress = response.progress;
            updateProgressDisplay(currentProgress);

            if (currentProgress.status === 'completed') {
                completeRecalculation();
            } else if (currentProgress.status === 'running') {
                // Process next batch after short delay
                setTimeout(function() {
                    processBatch(batchUrl);
                }, 100);
            }
        } else {
            swal('Batch Failed', 'Batch processing failed: ' + response.message, 'error');
            hideProgress();
            stopProgressPolling();
        }
    });
}
```

#### UI State Management
```javascript
// Update progress display with current status
function updateProgressDisplay(progress) {
    if (!progress) return;

    const percentage = progress.total_records > 0 ?
        Math.round((progress.processed_records / progress.total_records) * 100) : 0;

    $('#progress-bar').css('width', percentage + '%');
    $('#processed-count').text(progress.processed_records || 0);
    $('#updated-count').text(progress.updated_records || 0);
    $('#current-batch').text(progress.current_batch || 0);
    $('#total-batches').text(progress.total_batches || 0);

    if (progress.estimated_time) {
        const eta = new Date(progress.estimated_time);
        $('#estimated-time').text(eta.toLocaleTimeString());
    }

    // Dynamic status messages
    let message = '';
    switch (progress.status) {
        case 'starting':
            message = 'Initializing batch processing...';
            break;
        case 'running':
            message = `Processing batch ${progress.current_batch + 1} of ${progress.total_batches}...`;
            break;
        case 'completed':
            message = `Completed! Processed ${progress.processed_records} records.`;
            break;
        case 'error':
            message = 'Processing encountered an error.';
            break;
    }
    $('#progress-message').text(message);
}
```

### Blade Template Structure

#### Main Dashboard Template
```blade
{{-- resources/views/index.blade.php --}}
@extends('layouts.app')
@section('title', __('stockrecalculation::lang.stock_recalculation'))

@section('content')
<!-- Summary Cards -->
<div class="row" id="summary-cards">
    <div class="col-lg-3 col-xs-6">
        <div class="small-box bg-blue">
            <div class="inner">
                <h3 id="total-products">-</h3>
                <p>Total Products</p>
            </div>
        </div>
    </div>
    <!-- Additional summary cards -->
</div>

<!-- Filter Controls -->
<div class="box box-primary">
    <div class="box-body">
        <form id="stock-filters">
            @csrf
            <div class="row">
                <div class="col-md-4">
                    <select class="form-control" id="business_id" name="business_id">
                        @foreach($businesses as $business)
                            <option value="{{ $business->id }}">{{ $business->name }}</option>
                        @endforeach
                    </select>
                </div>
                <!-- Additional filter controls -->
            </div>
        </form>
    </div>
</div>

<!-- Action Buttons -->
<div class="box box-warning">
    <div class="box-body">
        <div class="row">
            <div class="col-md-3">
                <button type="button" class="btn btn-info btn-block" id="verify-btn">
                    <i class="fa fa-search"></i> Step 1: Verify
                </button>
            </div>
            <!-- Additional action buttons -->
        </div>
    </div>
</div>

<!-- Progress Monitoring -->
<div class="box box-info" id="progress-box" style="display: none;">
    <div class="box-body">
        <div id="progress-message">Processing...</div>
        <div class="progress progress-sm">
            <div class="progress-bar progress-bar-striped active" id="progress-bar" style="width: 0%"></div>
        </div>
        <!-- Progress statistics -->
    </div>
</div>
@endsection
```

## Configuration & Constants

### Module Configuration
```php
// Config/config.php
return [
    'name' => 'StockRecalculation',
    'description' => 'Professional Stock Recalculation Module for Ultimate POS',
    'batch_size' => env('STOCK_RECALC_BATCH_SIZE', 100),
    'max_execution_time' => env('STOCK_RECALC_MAX_TIME', 25),
    'memory_limit' => env('STOCK_RECALC_MEMORY', '512M'),
    'progress_cache_duration' => env('STOCK_RECALC_CACHE_DURATION', 3600),
    'enable_debug_endpoints' => env('STOCK_RECALC_DEBUG', false),
    'max_backup_age_days' => env('STOCK_RECALC_BACKUP_RETENTION', 30),
];
```

### Environment Variables
```bash
# .env configuration options
STOCK_RECALC_BATCH_SIZE=100
STOCK_RECALC_MAX_TIME=25
STOCK_RECALC_MEMORY=512M
STOCK_RECALC_CACHE_DURATION=3600
STOCK_RECALC_DEBUG=false
STOCK_RECALC_BACKUP_RETENTION=30
```

### Class Constants
```php
class StockRecalculationController extends Controller
{
    // Processing configuration
    const BATCH_SIZE = 100;           // Records per batch
    const MAX_EXECUTION_TIME = 25;    // Seconds per batch
    const MEMORY_LIMIT = '512M';      // Memory allocation
    const PROGRESS_CACHE_DURATION = 3600; // Progress cache duration (1 hour)

    // Safety limits
    const MAX_BACKUP_RECORDS = 50000; // Maximum records for backup
    const MEMORY_CLEANUP_INTERVAL = 50; // Records between garbage collection
    const MAX_DISCREPANCY_RESULTS = 1000; // Maximum discrepancies to display
}
```

## Code Examples & Usage

### Basic Usage Examples

#### CLI Command Usage
```bash
# Recalculate all stock for business ID 1
php artisan stock:recalculate --business_id=1

# Recalculate specific location
php artisan stock:recalculate --business_id=1 --location_id=5

# Recalculate specific variation
php artisan stock:recalculate --business_id=1 --location_id=5 --variation_id=100
```

#### Programmatic Usage
```php
// Using the trait directly
use Modules\StockRecalculation\Utils\Concerns\StockIntegrity;

class CustomController extends Controller
{
    use StockIntegrity;

    public function customRecalculation()
    {
        $business_id = 1;
        $variation_id = 100;
        $location_id = 5;

        // Calculate stock quantity
        $calculatedQty = $this->computeVldQty($business_id, $variation_id, $location_id);

        // Synchronize with database
        $this->syncVariationLocationQty($business_id, $variation_id, $location_id);

        return $calculatedQty;
    }
}
```

#### API Integration Example
```javascript
// JavaScript API integration
async function performStockRecalculation() {
    try {
        // Step 1: Create backup
        const backupResponse = await fetch('/stock-recalculation/backup', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                business_id: 1,
                location_id: 5
            })
        });

        const backupResult = await backupResponse.json();
        console.log('Backup created:', backupResult.backup_table);

        // Step 2: Start recalculation
        const recalcResponse = await fetch('/stock-recalculation/start-recalculation', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                business_id: 1,
                location_id: 5
            })
        });

        const recalcResult = await recalcResponse.json();
        console.log('Recalculation started:', recalcResult.progress);

        // Step 3: Monitor progress
        const progressInterval = setInterval(async () => {
            const progressResponse = await fetch('/stock-recalculation/progress');
            const progressResult = await progressResponse.json();

            console.log('Progress:', progressResult.progress);

            if (progressResult.progress?.status === 'completed') {
                clearInterval(progressInterval);
                console.log('Recalculation completed successfully');
            }
        }, 2000);

    } catch (error) {
        console.error('Recalculation error:', error);
    }
}
```

### Custom Extensions

#### Creating Custom Calculation Logic
```php
// Custom trait extending StockIntegrity
trait CustomStockCalculation
{
    use StockIntegrity;

    /**
     * Custom calculation including transfers
     */
    public function computeVldQtyWithTransfers(int $business_id, int $variation_id, int $location_id): float
    {
        // Get base calculation
        $baseQty = $this->computeVldQty($business_id, $variation_id, $location_id);

        // Add transfer calculations
        $transfersIn = (float) DB::table('stock_transfer_details as std')
            ->join('stock_transfers as st', 'st.id', '=', 'std.stock_transfer_id')
            ->where('st.business_id', $business_id)
            ->where('st.location_id', $location_id)
            ->where('std.variation_id', $variation_id)
            ->where('st.status', 'completed')
            ->sum('std.quantity');

        $transfersOut = (float) DB::table('stock_transfer_details as std')
            ->join('stock_transfers as st', 'st.id', '=', 'std.stock_transfer_id')
            ->where('st.business_id', $business_id)
            ->where('st.transfer_location_id', $location_id)
            ->where('std.variation_id', $variation_id)
            ->where('st.status', 'completed')
            ->sum('std.quantity');

        return (float) round($baseQty + $transfersIn - $transfersOut, 4);
    }
}
```

## Testing & Debugging

### Debug Endpoints Usage

#### Testing Calculation Methods
```php
// GET /stock-recalculation/test-calculation?variation_id=100&location_id=5
{
    "variation_id": 100,
    "location_id": 5,
    "current_stored_stock": 25.0000,
    "original_calculation": 23.5000,
    "modified_calculation": 23.5000,
    "difference_original_vs_stored": -1.5000,
    "difference_modified_vs_stored": -1.5000,
    "difference_original_vs_modified": 0.0000,
    "transactions_found": 15
}
```

#### Debug Calculation Breakdown
```php
// GET /stock-recalculation/debug-calculation?variation_id=100&location_id=5
{
    "variation_id": 100,
    "location_id": 5,
    "calculated_stock": 23.5000,
    "current_stored_stock": 25.0000,
    "difference": -1.5000,
    "breakdown": {
        "purchases": [
            {
                "id": 1001,
                "transaction_date": "2024-01-15 10:30:00",
                "type": "purchase",
                "status": "received",
                "quantity": 50.0000,
                "quantity_returned": 0.0000
            }
        ],
        "sales": [
            {
                "id": 2001,
                "transaction_date": "2024-01-16 14:20:00",
                "type": "sell",
                "status": "final",
                "quantity": 26.5000
            }
        ]
    }
}
```

### Unit Testing Examples

#### PHPUnit Test Cases
```php
// tests/Feature/StockRecalculationTest.php
class StockRecalculationTest extends TestCase
{
    use DatabaseTransactions, StockIntegrity;

    public function test_stock_calculation_accuracy()
    {
        // Create test data
        $business = Business::factory()->create();
        $location = BusinessLocation::factory()->create(['business_id' => $business->id]);
        $product = Product::factory()->create(['business_id' => $business->id]);
        $variation = Variation::factory()->create(['product_id' => $product->id]);

        // Create test transactions
        $this->createPurchaseTransaction($business, $location, $variation, 100);
        $this->createSaleTransaction($business, $location, $variation, 25);

        // Calculate stock
        $calculatedQty = $this->computeVldQty(
            $business->id,
            $variation->id,
            $location->id
        );

        // Assert expected result
        $this->assertEquals(75.0, $calculatedQty);
    }

    public function test_batch_processing_performance()
    {
        $startTime = microtime(true);

        // Process test batch
        $controller = new StockRecalculationController();
        $result = $controller->processBatchOfRecords(
            new Request(),
            1, // business_id
            0, // offset
            100 // limit
        );

        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;

        // Assert performance requirements
        $this->assertLessThan(30, $executionTime); // Should complete within 30 seconds
        $this->assertArrayHasKey('processed', $result);
        $this->assertArrayHasKey('updated', $result);
    }
}
```

### Error Testing

#### Simulating Error Conditions
```php
// Test memory limit handling
ini_set('memory_limit', '64M'); // Artificially low limit
try {
    $controller = new StockRecalculationController();
    $result = $controller->processBatch($request);
} catch (\Exception $e) {
    // Handle memory exhaustion gracefully
    Log::error('Memory limit exceeded during testing: ' . $e->getMessage());
}

// Test database connection failure
DB::disconnect();
try {
    $qty = $this->computeVldQty(1, 100, 5);
} catch (\Exception $e) {
    // Handle database connectivity issues
    Log::error('Database connection failed: ' . $e->getMessage());
}
```

## Extension Points

### Creating Custom Modules

#### Extending the Controller
```php
// Custom controller extending base functionality
class EnhancedStockRecalculationController extends StockRecalculationController
{
    /**
     * Add custom reporting functionality
     */
    public function generateDiscrepancyReport(Request $request)
    {
        $discrepancies = $this->getDiscrepanciesInBatches($request, $business_id);

        // Generate PDF report
        $pdf = PDF::loadView('reports.stock-discrepancy', compact('discrepancies'));

        return $pdf->download('stock-discrepancy-report.pdf');
    }

    /**
     * Add scheduled recalculation
     */
    public function scheduleRecalculation(Request $request)
    {
        $schedule = $request->input('schedule'); // daily, weekly, monthly

        // Create scheduled job
        ScheduledRecalculation::create([
            'business_id' => session()->get('user.business_id'),
            'schedule' => $schedule,
            'filters' => $request->input('filters', []),
            'next_run' => $this->calculateNextRun($schedule)
        ]);

        return response()->json(['success' => true]);
    }
}
```

#### Adding Custom Calculation Methods
```php
// Custom calculation trait
trait AdvancedStockCalculations
{
    use StockIntegrity;

    /**
     * Calculate stock with expiration tracking
     */
    public function computeStockWithExpiration(int $business_id, int $variation_id, int $location_id): array
    {
        $totalQty = $this->computeVldQty($business_id, $variation_id, $location_id);

        // Get expiration details
        $expirationData = DB::table('purchase_lines as pl')
            ->join('transactions as t', 't.id', '=', 'pl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('pl.variation_id', $variation_id)
            ->where('pl.exp_date', '>', now())
            ->selectRaw('
                SUM(pl.quantity - IFNULL(pl.quantity_sold, 0)) as available_qty,
                MIN(pl.exp_date) as next_expiry,
                COUNT(*) as batch_count
            ')
            ->first();

        return [
            'total_quantity' => $totalQty,
            'available_quantity' => $expirationData->available_qty ?? 0,
            'next_expiry' => $expirationData->next_expiry,
            'batch_count' => $expirationData->batch_count ?? 0
        ];
    }

    /**
     * Calculate projected stock based on sales trends
     */
    public function projectStockLevel(int $business_id, int $variation_id, int $location_id, int $days = 30): float
    {
        $currentStock = $this->computeVldQty($business_id, $variation_id, $location_id);

        // Calculate average daily sales
        $averageDailySales = DB::table('transaction_sell_lines as tsl')
            ->join('transactions as t', 't.id', '=', 'tsl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('tsl.variation_id', $variation_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->where('t.transaction_date', '>=', now()->subDays(30))
            ->avg('tsl.quantity') ?? 0;

        return max(0, $currentStock - ($averageDailySales * $days));
    }
}
```

### Integration Hooks

#### Event Listeners
```php
// Custom event listener for stock changes
class StockRecalculationEventListener
{
    /**
     * Handle successful recalculation completion
     */
    public function handleRecalculationComplete($event)
    {
        $business_id = $event->business_id;
        $statistics = $event->statistics;

        // Send notification
        $users = User::whereHas('roles', function($q) {
            $q->where('name', 'Admin');
        })->where('business_id', $business_id)->get();

        foreach ($users as $user) {
            Mail::to($user->email)->send(new StockRecalculationComplete($statistics));
        }

        // Log to audit trail
        AuditLog::create([
            'business_id' => $business_id,
            'user_id' => auth()->id(),
            'action' => 'stock_recalculation_completed',
            'details' => json_encode($statistics)
        ]);
    }
}
```

#### Middleware Integration
```php
// Custom middleware for stock recalculation operations
class StockRecalculationMiddleware
{
    public function handle($request, Closure $next)
    {
        // Check if maintenance mode
        if (Cache::has('stock_recalc_maintenance')) {
            return response()->json([
                'error' => 'Stock recalculation is temporarily unavailable due to maintenance'
            ], 503);
        }

        // Rate limiting
        $user_id = auth()->id();
        $key = "stock_recalc_rate_limit_{$user_id}";

        if (Cache::has($key)) {
            return response()->json([
                'error' => 'Rate limit exceeded. Please wait before starting another recalculation.'
            ], 429);
        }

        // Set rate limit (1 recalculation per 5 minutes)
        Cache::put($key, true, 300);

        return $next($request);
    }
}
```

This comprehensive code documentation provides detailed insights into the Stock Recalculation module's implementation, including core classes, methods, database interactions, frontend components, and extension points for customization.