<?php

namespace Modules\DailyStockHistory\Http\Controllers;

use App\Brands;
use App\Business;
use App\BusinessLocation;
use App\Category;
use App\Contact;
use App\Utils\ProductUtil;
use App\Utils\TransactionUtil;
use Datatables;
use DB;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;

class DailyStockHistoryController extends Controller
{
    protected $transactionUtil;
    protected $productUtil;

    public function __construct(TransactionUtil $transactionUtil, ProductUtil $productUtil)
    {
        $this->transactionUtil = $transactionUtil;
        $this->productUtil = $productUtil;
    }

    public function index(Request $request)
    {
        if (! auth()->user()->can('stock_report.view')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = $request->session()->get('user.business_id');
        // Get business locations (without receipt_printer_type_attribute since we don't need it for reports)
        $business_locations = BusinessLocation::forDropdown($business_id, false, false);
        $categories = Category::forDropdown($business_id, false, true, false);
        $brands = Brands::forDropdown($business_id, false, false);
        $suppliers = Contact::suppliersDropdown($business_id, false, true);
        $business = Business::find($business_id);

        if ($request->ajax()) {
            return $this->getDateWiseStockHistoryData($request);
        }

        return view('dailystockhistory::report.date_wise_stock_history')->with(compact(
            'business_locations',
            'categories',
            'brands',
            'suppliers',
            'business'
        ));
    }

    protected function getDateWiseStockHistoryData(Request $request)
    {
        $business_id = $request->session()->get('user.business_id');
        
        try {
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            $location_id = $request->input('location_id');
            $product_id = $request->input('product_id');
            $variation_id = $request->input('variation_id');
            $category_id = $request->input('category_id');
            $sub_category_id = $request->input('sub_category_id');
            $supplier_id = $request->input('supplier_id');

            $permitted_locations = auth()->user()->permitted_locations();

            // Apply common filters to all queries
            $common_filters = function ($query) use ($start_date, $end_date, $location_id, $product_id, $variation_id, $category_id, $sub_category_id, $supplier_id, $permitted_locations) {
                if ($start_date) {
                    $query->whereDate('t.transaction_date', '>=', $start_date);
                }
                if ($end_date) {
                    $query->whereDate('t.transaction_date', '<=', $end_date);
                }
                if ($location_id) {
                    $query->where('t.location_id', $location_id);
                } elseif ($permitted_locations != 'all') {
                    $query->whereIn('t.location_id', $permitted_locations);
                }
                if ($product_id) {
                    $query->where('p.id', $product_id);
                }
                if ($variation_id) {
                    $query->where('v.id', $variation_id);
                }
                if ($category_id) {
                    $query->where('p.category_id', $category_id);
                }
                if ($sub_category_id) {
                    $query->where('p.sub_category_id', $sub_category_id);
                }
                if ($supplier_id) {
                    $query->where('t.contact_id', $supplier_id);
                }
            };

            // Purchase query with selects
            $purchase_select = DB::table('transactions as t')
                ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
                ->join('variations as v', 'pl.variation_id', '=', 'v.id')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->leftjoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase')
                ->where('t.status', 'received')
                ->where('p.enable_stock', 1)
                ->whereNotNull('pl.variation_id')
                ->select(
                    DB::raw("DATE(t.transaction_date) as transaction_date"),
                    DB::raw("'Purchase' as transaction_type"),
                    't.ref_no',
                    't.invoice_no',
                    't.id as transaction_id',
                    'p.name as product_name',
                    'p.sku',
                    'v.name as variation_name',
                    'v.sub_sku',
                    DB::raw("CONCAT(p.name, IF(v.name IS NOT NULL AND v.name != 'DUMMY', CONCAT(' - ', v.name), ''), IF(COALESCE(v.sub_sku, p.sku) IS NOT NULL, CONCAT(' - ', COALESCE(v.sub_sku, p.sku)), '')) as product_display"),
                    'bl.name as location_name',
                    't.location_id',
                    DB::raw("COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0) as qty_in"),
                    DB::raw("0 as qty_out"),
                    DB::raw("COALESCE(pl.purchase_price_inc_tax, 0) as unit_cost"),
                    DB::raw("COALESCE(c.name, c.supplier_business_name, '') as contact_name"),
                    'v.id as variation_id',
                    'p.id as product_id',
                    't.transaction_date as full_transaction_date'
                );
            $common_filters($purchase_select);

            // Sale query with selects
            $sale_select = DB::table('transactions as t')
                ->join('transaction_sell_lines as sl', 't.id', '=', 'sl.transaction_id')
                ->join('variations as v', 'sl.variation_id', '=', 'v.id')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->leftjoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->where('p.enable_stock', 1)
                ->whereNotNull('sl.variation_id')
                ->select(
                    DB::raw("DATE(t.transaction_date) as transaction_date"),
                    DB::raw("'Sale' as transaction_type"),
                    't.invoice_no as ref_no',
                    't.invoice_no',
                    't.id as transaction_id',
                    'p.name as product_name',
                    'p.sku',
                    'v.name as variation_name',
                    'v.sub_sku',
                    DB::raw("CONCAT(p.name, IF(v.name IS NOT NULL AND v.name != 'DUMMY', CONCAT(' - ', v.name), ''), IF(COALESCE(v.sub_sku, p.sku) IS NOT NULL, CONCAT(' - ', COALESCE(v.sub_sku, p.sku)), '')) as product_display"),
                    'bl.name as location_name',
                    't.location_id',
                    DB::raw("0 as qty_in"),
                    DB::raw("COALESCE(sl.quantity, 0) as qty_out"),
                    DB::raw("COALESCE(sl.unit_price_inc_tax, 0) as unit_cost"),
                    DB::raw("COALESCE(c.name, c.supplier_business_name, '') as contact_name"),
                    'v.id as variation_id',
                    'p.id as product_id',
                    't.transaction_date as full_transaction_date'
                );
            $common_filters($sale_select);

            // Stock Adjustment query
            $adjustment_select = DB::table('transactions as t')
                ->join('stock_adjustment_lines as al', 't.id', '=', 'al.transaction_id')
                ->join('variations as v', 'al.variation_id', '=', 'v.id')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->leftjoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'stock_adjustment')
                ->where('p.enable_stock', 1)
                ->whereNotNull('al.variation_id')
                ->select(
                    DB::raw("DATE(t.transaction_date) as transaction_date"),
                    DB::raw("'Stock Adjustment' as transaction_type"),
                    't.ref_no',
                    't.invoice_no',
                    't.id as transaction_id',
                    'p.name as product_name',
                    'p.sku',
                    'v.name as variation_name',
                    'v.sub_sku',
                    DB::raw("CONCAT(p.name, IF(v.name IS NOT NULL AND v.name != 'DUMMY', CONCAT(' - ', v.name), ''), IF(COALESCE(v.sub_sku, p.sku) IS NOT NULL, CONCAT(' - ', COALESCE(v.sub_sku, p.sku)), '')) as product_display"),
                    'bl.name as location_name',
                    't.location_id',
                    DB::raw("0 as qty_in"),
                    DB::raw("COALESCE(al.quantity, 0) as qty_out"),
                    DB::raw("COALESCE(al.unit_price, 0) as unit_cost"),
                    DB::raw("COALESCE(c.name, c.supplier_business_name, '') as contact_name"),
                    'v.id as variation_id',
                    'p.id as product_id',
                    't.transaction_date as full_transaction_date'
                );
            $common_filters($adjustment_select);

            // Opening Stock query
            $opening_stock_select = DB::table('transactions as t')
                ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
                ->join('variations as v', 'pl.variation_id', '=', 'v.id')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->leftjoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'opening_stock')
                ->where('t.status', 'received')
                ->where('p.enable_stock', 1)
                ->whereNotNull('pl.variation_id')
                ->select(
                    DB::raw("DATE(t.transaction_date) as transaction_date"),
                    DB::raw("'Opening Stock' as transaction_type"),
                    't.ref_no',
                    't.invoice_no',
                    't.id as transaction_id',
                    'p.name as product_name',
                    'p.sku',
                    'v.name as variation_name',
                    'v.sub_sku',
                    DB::raw("CONCAT(p.name, IF(v.name IS NOT NULL AND v.name != 'DUMMY', CONCAT(' - ', v.name), ''), IF(COALESCE(v.sub_sku, p.sku) IS NOT NULL, CONCAT(' - ', COALESCE(v.sub_sku, p.sku)), '')) as product_display"),
                    'bl.name as location_name',
                    't.location_id',
                    DB::raw("COALESCE(pl.quantity, 0) as qty_in"),
                    DB::raw("0 as qty_out"),
                    DB::raw("COALESCE(pl.purchase_price_inc_tax, 0) as unit_cost"),
                    DB::raw("COALESCE(c.name, c.supplier_business_name, '') as contact_name"),
                    'v.id as variation_id',
                    'p.id as product_id',
                    't.transaction_date as full_transaction_date'
                );
            $common_filters($opening_stock_select);

            // Purchase Return query
            $purchase_return_select = DB::table('transactions as t')
                ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
                ->join('variations as v', 'pl.variation_id', '=', 'v.id')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->leftjoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase_return')
                ->where('p.enable_stock', 1)
                ->whereNotNull('pl.variation_id')
                ->whereNotNull('pl.quantity_returned')
                ->select(
                    DB::raw("DATE(t.transaction_date) as transaction_date"),
                    DB::raw("'Purchase Return' as transaction_type"),
                    't.ref_no',
                    't.invoice_no',
                    't.id as transaction_id',
                    'p.name as product_name',
                    'p.sku',
                    'v.name as variation_name',
                    'v.sub_sku',
                    DB::raw("CONCAT(p.name, IF(v.name IS NOT NULL AND v.name != 'DUMMY', CONCAT(' - ', v.name), ''), IF(COALESCE(v.sub_sku, p.sku) IS NOT NULL, CONCAT(' - ', COALESCE(v.sub_sku, p.sku)), '')) as product_display"),
                    'bl.name as location_name',
                    't.location_id',
                    DB::raw("0 as qty_in"),
                    DB::raw("COALESCE(pl.quantity_returned, 0) as qty_out"),
                    DB::raw("COALESCE(pl.purchase_price_inc_tax, 0) as unit_cost"),
                    DB::raw("COALESCE(c.name, c.supplier_business_name, '') as contact_name"),
                    'v.id as variation_id',
                    'p.id as product_id',
                    't.transaction_date as full_transaction_date'
                );
            $common_filters($purchase_return_select);

            // Sale Return query
            $sale_return_select = DB::table('transactions as t')
                ->join('transaction_sell_lines as sl', 't.id', '=', 'sl.transaction_id')
                ->join('variations as v', 'sl.variation_id', '=', 'v.id')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->leftjoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell_return')
                ->where('p.enable_stock', 1)
                ->whereNotNull('sl.variation_id')
                ->whereNotNull('sl.quantity_returned')
                ->select(
                    DB::raw("DATE(t.transaction_date) as transaction_date"),
                    DB::raw("'Sale Return' as transaction_type"),
                    't.invoice_no as ref_no',
                    't.invoice_no',
                    't.id as transaction_id',
                    'p.name as product_name',
                    'p.sku',
                    'v.name as variation_name',
                    'v.sub_sku',
                    DB::raw("CONCAT(p.name, IF(v.name IS NOT NULL AND v.name != 'DUMMY', CONCAT(' - ', v.name), ''), IF(COALESCE(v.sub_sku, p.sku) IS NOT NULL, CONCAT(' - ', COALESCE(v.sub_sku, p.sku)), '')) as product_display"),
                    'bl.name as location_name',
                    't.location_id',
                    DB::raw("COALESCE(sl.quantity_returned, 0) as qty_in"),
                    DB::raw("0 as qty_out"),
                    DB::raw("COALESCE(sl.unit_price_inc_tax, 0) as unit_cost"),
                    DB::raw("COALESCE(c.name, c.supplier_business_name, '') as contact_name"),
                    'v.id as variation_id',
                    'p.id as product_id',
                    't.transaction_date as full_transaction_date'
                );
            $common_filters($sale_return_select);

            // Transfer In query
            $transfer_in_select = DB::table('transactions as t')
                ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
                ->join('variations as v', 'pl.variation_id', '=', 'v.id')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->leftjoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase_transfer')
                ->where('t.status', 'received')
                ->where('p.enable_stock', 1)
                ->whereNotNull('pl.variation_id')
                ->select(
                    DB::raw("DATE(t.transaction_date) as transaction_date"),
                    DB::raw("'Transfer In' as transaction_type"),
                    't.ref_no',
                    't.invoice_no',
                    't.id as transaction_id',
                    'p.name as product_name',
                    'p.sku',
                    'v.name as variation_name',
                    'v.sub_sku',
                    DB::raw("CONCAT(p.name, IF(v.name IS NOT NULL AND v.name != 'DUMMY', CONCAT(' - ', v.name), ''), IF(COALESCE(v.sub_sku, p.sku) IS NOT NULL, CONCAT(' - ', COALESCE(v.sub_sku, p.sku)), '')) as product_display"),
                    'bl.name as location_name',
                    't.location_id',
                    DB::raw("COALESCE(pl.quantity, 0) as qty_in"),
                    DB::raw("0 as qty_out"),
                    DB::raw("COALESCE(pl.purchase_price_inc_tax, 0) as unit_cost"),
                    DB::raw("COALESCE(c.name, c.supplier_business_name, '') as contact_name"),
                    'v.id as variation_id',
                    'p.id as product_id',
                    't.transaction_date as full_transaction_date'
                );
            $common_filters($transfer_in_select);

            // Transfer Out query
            $transfer_out_select = DB::table('transactions as t')
                ->join('transaction_sell_lines as sl', 't.id', '=', 'sl.transaction_id')
                ->join('variations as v', 'sl.variation_id', '=', 'v.id')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->leftjoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->leftjoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell_transfer')
                ->where('t.status', 'final')
                ->where('p.enable_stock', 1)
                ->whereNotNull('sl.variation_id')
                ->select(
                    DB::raw("DATE(t.transaction_date) as transaction_date"),
                    DB::raw("'Transfer Out' as transaction_type"),
                    't.ref_no',
                    't.invoice_no',
                    't.id as transaction_id',
                    'p.name as product_name',
                    'p.sku',
                    'v.name as variation_name',
                    'v.sub_sku',
                    DB::raw("CONCAT(p.name, IF(v.name IS NOT NULL AND v.name != 'DUMMY', CONCAT(' - ', v.name), ''), IF(COALESCE(v.sub_sku, p.sku) IS NOT NULL, CONCAT(' - ', COALESCE(v.sub_sku, p.sku)), '')) as product_display"),
                    'bl.name as location_name',
                    't.location_id',
                    DB::raw("0 as qty_in"),
                    DB::raw("COALESCE(sl.quantity, 0) as qty_out"),
                    DB::raw("COALESCE(sl.unit_price_inc_tax, 0) as unit_cost"),
                    DB::raw("COALESCE(c.name, c.supplier_business_name, '') as contact_name"),
                    'v.id as variation_id',
                    'p.id as product_id',
                    't.transaction_date as full_transaction_date'
                );
            $common_filters($transfer_out_select);

            // Union all queries
            $union_query = $purchase_select
                ->union($sale_select)
                ->union($adjustment_select)
                ->union($opening_stock_select)
                ->union($purchase_return_select)
                ->union($sale_return_select)
                ->union($transfer_in_select)
                ->union($transfer_out_select);

            // Get all results and calculate running balance
            try {
                $results = $union_query
                    ->orderBy('full_transaction_date', 'asc')
                    ->orderBy('transaction_id', 'asc')
                    ->get();
                
                // Convert to array format for easier processing
                $results = $results->map(function($item) {
                    return (array) $item;
                });
            } catch (\Exception $e) {
                \Log::error('Union Query Error: ' . $e->getMessage());
                \Log::error('Union Query Trace: ' . $e->getTraceAsString());
                
                // Try alternative approach with raw SQL
                try {
                    $sql = $union_query->toSql();
                    $bindings = $union_query->getBindings();
                    
                    $raw_sql = "SELECT * FROM ({$sql}) as union_query ORDER BY full_transaction_date ASC, transaction_id ASC";
                    $results = collect(DB::select($raw_sql, $bindings))->map(function($item) {
                        return (array) $item;
                    });
                } catch (\Exception $e2) {
                    \Log::error('Raw SQL Query Error: ' . $e2->getMessage());
                    throw new \Exception('Error executing stock history query: ' . $e2->getMessage());
                }
            }

            // Get all products/variations that match filters (for finding products with only opening stock)
            $all_products_query = DB::table('variations as v')
                ->join('products as p', 'v.product_id', '=', 'p.id')
                ->leftjoin('categories as c', 'p.category_id', '=', 'c.id')
                ->leftjoin('categories as sc', 'p.sub_category_id', '=', 'sc.id')
                ->leftjoin('brands as b', 'p.brand_id', '=', 'b.id')
                ->where('p.business_id', $business_id)
                ->where('p.enable_stock', 1)
                ->select(
                    'p.id as product_id',
                    'v.id as variation_id',
                    'p.name as product_name',
                    'v.name as variation_name',
                    'v.sub_sku',
                    'p.sku'
                );

            if ($product_id) {
                $all_products_query->where('p.id', $product_id);
            }
            if ($variation_id) {
                $all_products_query->where('v.id', $variation_id);
            }
            if ($category_id) {
                $all_products_query->where('p.category_id', $category_id);
            }
            if ($sub_category_id) {
                $all_products_query->where('p.sub_category_id', $sub_category_id);
            }

            $all_products = $all_products_query->get();

            // Get list of product/variation/location combinations that already have transactions
            $products_with_transactions = [];
            foreach ($results as $row) {
                $prod_id = $row['product_id'] ?? null;
                $var_id = $row['variation_id'] ?? null;
                $loc_id = $row['location_id'] ?? null;
                $key = $prod_id . '_' . $var_id . '_' . $loc_id;
                $products_with_transactions[$key] = true;
            }

            // Find products with opening stock but no transactions in date range
            $products_with_only_opening_stock = [];
            $query_locations = [];
            if ($permitted_locations != 'all') {
                $query_locations = $permitted_locations;
            }

            foreach ($all_products as $product) {
                // Get all locations for this product
                $locations_to_check = [];
                if ($location_id) {
                    $locations_to_check[] = $location_id;
                } elseif ($query_locations) {
                    $locations_to_check = $query_locations;
                } else {
                    // Get all business locations
                    $all_locations = DB::table('business_locations')
                        ->where('business_id', $business_id)
                        ->pluck('id')
                        ->toArray();
                    $locations_to_check = $all_locations;
                }

                foreach ($locations_to_check as $loc_id) {
                    $key = $product->product_id . '_' . $product->variation_id . '_' . $loc_id;
                    
                    // Skip if this product/location already has transactions
                    if (isset($products_with_transactions[$key])) {
                        continue;
                    }

                    // Calculate opening balance
                    $opening_balance = $this->calculateProductOpeningBalance(
                        $business_id,
                        $product->product_id,
                        $product->variation_id,
                        $loc_id,
                        $start_date
                    );

                    // If opening balance > 0, add to results
                    if ($opening_balance > 0) {
                        $location_name = DB::table('business_locations')
                            ->where('id', $loc_id)
                            ->value('name');

                        $product_display = $product->product_name;
                        if ($product->variation_name && $product->variation_name != 'DUMMY') {
                            $product_display .= ' - ' . $product->variation_name;
                        }
                        if ($product->sub_sku) {
                            $product_display .= ' - ' . $product->sub_sku;
                        } elseif ($product->sku) {
                            $product_display .= ' - ' . $product->sku;
                        }

                        // Add as a special row with only opening balance
                        $products_with_only_opening_stock[] = [
                            'transaction_date' => $start_date ? date('Y-m-d', strtotime($start_date)) : date('Y-m-d'),
                            'full_transaction_date' => $start_date ? date('Y-m-d', strtotime($start_date)) : date('Y-m-d'),
                            'transaction_type' => 'Opening Balance',
                            'ref_no' => '',
                            'invoice_no' => '',
                            'transaction_id' => 0,
                            'product_name' => $product->product_name,
                            'sku' => $product->sku,
                            'variation_name' => $product->variation_name,
                            'sub_sku' => $product->sub_sku,
                            'product_display' => $product_display,
                            'location_name' => $location_name,
                            'location_id' => $loc_id,
                            'qty_in' => 0,
                            'qty_out' => 0,
                            'unit_cost' => 0,
                            'contact_name' => '',
                            'variation_id' => $product->variation_id,
                            'product_id' => $product->product_id,
                            '_opening_balance_only' => true,
                            '_opening_balance_qty' => $opening_balance
                        ];
                    }
                }
            }

            // Merge products with only opening stock into results
            if (!empty($products_with_only_opening_stock)) {
                $results = $results->merge(collect($products_with_only_opening_stock));
                // Re-sort by date
                $results = $results->sortBy(function($item) {
                    return $item['full_transaction_date'] ?? $item['transaction_date'] ?? '';
                })->values();
            }

            // Calculate opening balance for each product/location combination
            $opening_balances = [];
            foreach ($results as $row) {
                $prod_id = is_array($row) ? ($row['product_id'] ?? null) : ($row->product_id ?? null);
                $var_id = is_array($row) ? ($row['variation_id'] ?? null) : ($row->variation_id ?? null);
                $loc_id = is_array($row) ? ($row['location_id'] ?? null) : ($row->location_id ?? null);
                
                $key = $prod_id . '_' . $var_id . '_' . $loc_id;
                if (!isset($opening_balances[$key])) {
                    // If this is a product with only opening stock, use the pre-calculated value
                    if (isset($row['_opening_balance_only']) && $row['_opening_balance_only']) {
                        $opening_balances[$key] = $row['_opening_balance_qty'] ?? 0;
                    } else {
                        // Calculate opening balance up to start_date
                        $opening_balance = $this->calculateProductOpeningBalance(
                            $business_id,
                            $prod_id,
                            $var_id,
                            $loc_id,
                            $start_date
                        );
                        $opening_balances[$key] = $opening_balance;
                    }
                }
            }

            // Process results and calculate running balance with weighted average cost
            $processed_results = [];
            $running_balances = [];
            $running_cost_values = [];
            $running_cost_quantities = [];
            $total_qty_in = 0;
            $total_qty_out = 0;
            $total_opening_stock = 0;

            foreach ($results as $row) {
                $prod_id = $row['product_id'] ?? null;
                $var_id = $row['variation_id'] ?? null;
                $loc_id = $row['location_id'] ?? null;
                $key = $prod_id . '_' . $var_id . '_' . $loc_id;
                
                $opening_balance_qty = $opening_balances[$key] ?? 0;
                
                // Track if this is the first transaction for this product/location
                $is_first_transaction = !isset($running_balances[$key]);
                
                if ($is_first_transaction) {
                    $running_balances[$key] = $opening_balance_qty;
                    
                    // Calculate opening stock cost value (average cost up to previous day)
                    $opening_cost_value = 0;
                    if ($opening_balance_qty > 0 && $start_date) {
                        $opening_cost_value = $this->calculateProductOpeningCostValue(
                            $business_id,
                            $prod_id,
                            $var_id,
                            $loc_id,
                            $start_date
                        );
                    }
                    
                    // Initialize cost tracking with opening stock
                    $running_cost_values[$key] = $opening_cost_value;
                    $running_cost_quantities[$key] = $opening_balance_qty;
                    
                    // Add opening stock to total (only once per product/location)
                    if ($opening_balance_qty > 0) {
                        $total_opening_stock += $opening_balance_qty;
                    }
                }

                $qty_in = floatval($row['qty_in'] ?? 0);
                $qty_out = floatval($row['qty_out'] ?? 0);
                $transaction_unit_cost = floatval($row['unit_cost'] ?? 0);
                
                // Show opening stock only in the first row for each product/location
                $display_opening_stock = $is_first_transaction ? $opening_balance_qty : 0;
                
                // Quantity In should show ONLY the transaction quantity
                $display_qty_in = $qty_in;
                
                // Calculate weighted average cost for this transaction
                $display_unit_cost = $transaction_unit_cost;
                
                // If there's a quantity in (purchase, sale return, transfer in), calculate weighted average
                if ($qty_in > 0 && $transaction_unit_cost > 0) {
                    $current_stock_value = $running_cost_values[$key] ?? 0;
                    $current_stock_qty = $running_cost_quantities[$key] ?? 0;
                    
                    $new_transaction_value = $qty_in * $transaction_unit_cost;
                    $new_total_value = $current_stock_value + $new_transaction_value;
                    $new_total_qty = $current_stock_qty + $qty_in;
                    
                    if ($new_total_qty > 0) {
                        $weighted_avg_cost = $new_total_value / $new_total_qty;
                        
                        $running_cost_values[$key] = $new_total_value;
                        $running_cost_quantities[$key] = $new_total_qty;
                        
                        $display_unit_cost = $weighted_avg_cost;
                    }
                } elseif ($running_cost_quantities[$key] > 0) {
                    $current_stock_value = $running_cost_values[$key] ?? 0;
                    $current_stock_qty = $running_cost_quantities[$key] ?? 0;
                    $display_unit_cost = $current_stock_value / $current_stock_qty;
                }
                
                // If there's quantity out (sale, purchase return, adjustment, transfer out)
                if ($qty_out > 0) {
                    $current_stock_value = $running_cost_values[$key] ?? 0;
                    $current_stock_qty = $running_cost_quantities[$key] ?? 0;
                    
                    if ($current_stock_qty > 0) {
                        $avg_cost_at_sale = $current_stock_value / $current_stock_qty;
                        $reduction_value = $qty_out * $avg_cost_at_sale;
                        
                        $running_cost_values[$key] = max(0, $current_stock_value - $reduction_value);
                        $running_cost_quantities[$key] = max(0, $current_stock_qty - $qty_out);
                    }
                }
                
                // Update running balance
                $running_balances[$key] += ($qty_in - $qty_out);
                
                // Add to totals
                $total_qty_in += $display_qty_in;
                $total_qty_out += $qty_out;

                $processed_results[] = [
                    'date' => $row['transaction_date'] ?? '',
                    'transaction_type' => $row['transaction_type'] ?? '',
                    'ref_no' => $row['ref_no'] ?? ($row['invoice_no'] ?? '-'),
                    'product_display' => $row['product_display'] ?? '',
                    'product_name' => $row['product_name'] ?? '',
                    'sku' => $row['sku'] ?? '',
                    'variation_name' => $row['variation_name'] ?? '',
                    'sub_sku' => $row['sub_sku'] ?? '',
                    'location_name' => $row['location_name'] ?? '',
                    'opening_stock_qty' => $display_opening_stock,
                    'qty_in' => $display_qty_in,
                    'qty_out' => $qty_out,
                    'running_balance' => $this->transactionUtil->num_f($running_balances[$key], false, null, true),
                    'running_balance_raw' => $running_balances[$key],
                    'unit_cost' => $display_unit_cost,
                    'contact_name' => $row['contact_name'] ?? '',
                ];
            }

            // Calculate total closing stock: Sum of FINAL running balance for each unique product/location
            $total_running_stock = 0;
            foreach ($running_balances as $key => $final_balance) {
                $total_running_stock += $final_balance;
            }

            // Return empty result if no data
            if (empty($processed_results)) {
                return Datatables::of(collect([]))
                    ->with([
                        'total_opening_stock' => '0.00',
                        'total_qty_in' => '0.00',
                        'total_qty_out' => '0.00',
                        'total_running_stock' => '0.00',
                    ])
                    ->make(true);
            }

            return Datatables::of(collect($processed_results))
                ->editColumn('date', '{{@format_date($date)}}')
                ->editColumn('opening_stock_qty', function ($row) {
                    return $row['opening_stock_qty'] > 0 ? $this->transactionUtil->num_f($row['opening_stock_qty'], false, null, true) : '-';
                })
                ->editColumn('qty_in', function ($row) {
                    return $row['qty_in'] > 0 ? $this->transactionUtil->num_f($row['qty_in'], false, null, true) : '-';
                })
                ->editColumn('qty_out', function ($row) {
                    return $row['qty_out'] > 0 ? $this->transactionUtil->num_f($row['qty_out'], false, null, true) : '-';
                })
                ->editColumn('unit_cost', function ($row) {
                    return $row['unit_cost'] > 0 ? $this->transactionUtil->num_f($row['unit_cost'], true) : '-';
                })
                ->rawColumns(['transaction_type', 'product_display'])
                ->with([
                    'total_opening_stock' => $this->transactionUtil->num_f($total_opening_stock, false, null, true),
                    'total_qty_in' => $this->transactionUtil->num_f($total_qty_in, false, null, true),
                    'total_qty_out' => $this->transactionUtil->num_f($total_qty_out, false, null, true),
                    'total_running_stock' => $this->transactionUtil->num_f($total_running_stock, false, null, true),
                ])
                ->make(true);
        } catch (\Exception $e) {
            \Log::error('Date-wise Stock History Error: ' . $e->getMessage());
            \Log::error('Date-wise Stock History Trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'error' => 'Error loading stock history data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Calculate opening balance for a product variation at a location
     */
    protected function calculateProductOpeningBalance($business_id, $product_id, $variation_id, $location_id, $start_date)
    {
        if (!$start_date) {
            return 0;
        }

        // Calculate opening balance as the CLOSING balance of the previous day
        $previous_day = \Carbon::parse($start_date)->subDay()->format('Y-m-d');

        // Calculate all stock movements up to and including the previous day (closing balance)
        $purchase_qty = DB::table('transactions as t')
            ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('pl.product_id', $product_id)
            ->where('pl.variation_id', $variation_id)
            ->whereIn('t.type', ['purchase', 'opening_stock', 'purchase_transfer'])
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', '<=', $previous_day)
            ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));

        $sale_qty = DB::table('transactions as t')
            ->join('transaction_sell_lines as sl', 't.id', '=', 'sl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('sl.product_id', $product_id)
            ->where('sl.variation_id', $variation_id)
            ->whereIn('t.type', ['sell', 'sell_transfer'])
            ->where('t.status', 'final')
            ->whereDate('t.transaction_date', '<=', $previous_day)
            ->sum(DB::raw('COALESCE(sl.quantity, 0)'));

        // Stock adjustments always decrease stock
        $adjustment_decrease = DB::table('transactions as t')
            ->join('stock_adjustment_lines as al', 't.id', '=', 'al.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('al.product_id', $product_id)
            ->where('al.variation_id', $variation_id)
            ->where('t.type', 'stock_adjustment')
            ->whereDate('t.transaction_date', '<=', $previous_day)
            ->sum(DB::raw('COALESCE(al.quantity, 0)'));

        $purchase_return_qty = DB::table('transactions as t')
            ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('pl.product_id', $product_id)
            ->where('pl.variation_id', $variation_id)
            ->where('t.type', 'purchase_return')
            ->whereDate('t.transaction_date', '<=', $previous_day)
            ->sum(DB::raw('COALESCE(pl.quantity_returned, 0)'));

        $sale_return_qty = DB::table('transactions as t')
            ->join('transaction_sell_lines as sl', 't.id', '=', 'sl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('sl.product_id', $product_id)
            ->where('sl.variation_id', $variation_id)
            ->where('t.type', 'sell_return')
            ->whereDate('t.transaction_date', '<=', $previous_day)
            ->sum(DB::raw('COALESCE(sl.quantity_returned, 0)'));

        // Calculate closing balance of previous day = opening balance for start_date
        $opening_balance = ($purchase_qty + $sale_return_qty) 
                         - ($sale_qty + $adjustment_decrease + $purchase_return_qty);

        return $opening_balance;
    }

    /**
     * Calculate opening cost value for a product variation at a location
     */
    protected function calculateProductOpeningCostValue($business_id, $product_id, $variation_id, $location_id, $start_date)
    {
        if (!$start_date) {
            return 0;
        }

        $previous_day = \Carbon::parse($start_date)->subDay()->format('Y-m-d');
        
        // Calculate opening stock quantity
        $opening_qty = $this->calculateProductOpeningBalance($business_id, $product_id, $variation_id, $location_id, $start_date);
        
        if ($opening_qty <= 0) {
            return 0;
        }
        
        // Get average purchase price for opening stock (weighted average of all purchases up to previous day)
        $avg_purchase_price_query = DB::table('transactions as t')
            ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('pl.product_id', $product_id)
            ->where('pl.variation_id', $variation_id)
            ->whereIn('t.type', ['purchase', 'opening_stock', 'purchase_transfer'])
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', '<=', $previous_day);
        
        // Calculate weighted average: sum(quantity * price) / sum(quantity)
        $total_cost_value = $avg_purchase_price_query->sum(
            DB::raw('(COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)) * COALESCE(pl.purchase_price_inc_tax, 0)')
        );
        
        $total_purchase_qty = DB::table('transactions as t')
            ->join('purchase_lines as pl', 't.id', '=', 'pl.transaction_id')
            ->where('t.business_id', $business_id)
            ->where('t.location_id', $location_id)
            ->where('pl.product_id', $product_id)
            ->where('pl.variation_id', $variation_id)
            ->whereIn('t.type', ['purchase', 'opening_stock', 'purchase_transfer'])
            ->where('t.status', 'received')
            ->whereDate('t.transaction_date', '<=', $previous_day)
            ->sum(DB::raw('COALESCE(pl.quantity, 0) - COALESCE(pl.quantity_returned, 0)'));
        
        // Calculate weighted average price
        $avg_purchase_price = $total_purchase_qty > 0 ? ($total_cost_value / $total_purchase_qty) : 0;
        
        // Opening cost value = opening quantity * average purchase price
        $opening_cost_value = $opening_qty * $avg_purchase_price;
        
        return $opening_cost_value;
    }

    /**
     * Get products for Select2 dropdown (Optimized)
     * This method is optimized for the Date-wise Stock History report filter
     * Uses ProductUtil::filterProduct which is tested and optimized
     */
    public function getProducts(Request $request)
    {
        if (!request()->ajax()) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = $request->session()->get('user.business_id');
        $term = $request->input('term', '');
        $location_id = $request->input('location_id', '');
        $product_id = $request->input('product_id', '');
        $check_qty = $request->input('check_qty', false);
        
        try {
            // Use ProductUtil filterProduct which handles all edge cases
            // Add 'sub_sku' to search_fields automatically if 'sku' is included
            $search_fields = ['name', 'sku'];
            if (in_array('sku', $search_fields)) {
                $search_fields[] = 'sub_sku';
            }

            $products = $this->productUtil->filterProduct(
                $business_id,
                $term,
                $location_id,
                null, // not_for_selling
                null, // price_group_id
                [], // product_types
                $search_fields, // search_fields
                $check_qty, // check_qty
                'like' // search_type
            );

            // Filter by specific product if provided
            if (!empty($product_id)) {
                $products = $products->filter(function($item) use ($product_id) {
                    return $item->product_id == $product_id;
                });
            }

            // Format results for Select2 (matching ProductController format)
            $results = [];
            foreach ($products as $product) {
                // Build display text - match ProductController format
                $text = $product->name;
                if ($product->variation && $product->variation != 'DUMMY') {
                    $text .= ' - ' . $product->variation;
                }
                if ($product->sub_sku) {
                    $text .= ' (' . $product->sub_sku . ')';
                }

                $results[] = [
                    'id' => $product->variation_id,
                    'text' => $text,
                    'product_id' => $product->product_id,
                    'variation_id' => $product->variation_id,
                    'name' => $product->name,
                    'sku' => $product->sub_sku ?? '',
                    'variation' => $product->variation ?? '',
                    'sub_sku' => $product->sub_sku ?? ''
                ];
            }

            // Limit to 100 results for performance
            $results = array_slice($results, 0, 100);

            return response()->json($results);
        } catch (\Exception $e) {
            \Log::error('Get Products Error: ' . $e->getMessage());
            \Log::error('Get Products Error Trace: ' . $e->getTraceAsString());
            return response()->json(['error' => 'Error loading products: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Export Date-wise Stock History
     */
    public function exportDateWiseStockHistory(Request $request)
    {
        if (! auth()->user()->can('stock_report.view')) {
            abort(403, 'Unauthorized action.');
        }

        // For now, return a simple response
        // You can copy the export logic from ReportController::exportDateWiseStockHistory later
        return response()->json(['message' => 'Export functionality will be implemented soon'], 200);
    }
}
